/**
 * Copyright (C) 2014 4th Line GmbH, Switzerland and others
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License Version 1 or later
 * ("CDDL") (collectively, the "License"). You may not use this file
 * except in compliance with the License. See LICENSE.txt for more
 * information.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */
package org.jupnp.transport.impl.async;

import java.io.IOException;

import javax.servlet.AsyncContext;
import javax.servlet.AsyncEvent;
import javax.servlet.AsyncListener;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.jupnp.model.message.Connection;
import org.jupnp.transport.Router;
import org.jupnp.transport.impl.ServletConnection;
import org.jupnp.transport.impl.ServletStreamServerConfigurationImpl;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Http servlet implementation that uses the {@link Router}'s executor to process the current request and releases the
 * current thread(asynchronous).
 * 
 * @author Ivan Iliev
 * 
 */
public class AsyncServlet extends HttpServlet {

    /**
     * 
     */
    private static final long serialVersionUID = -5751553619541219814L;

    final private Logger log = LoggerFactory.getLogger(AsyncServlet.class);

    private final Router router;

    private int mCounter = 0;

    private final ServletStreamServerConfigurationImpl configuration;

    public AsyncServlet(Router router, ServletStreamServerConfigurationImpl configuration) {
        this.router = router;
        this.configuration = configuration;
    }

    @Override
    protected void service(HttpServletRequest req, HttpServletResponse resp) throws ServletException, IOException {

        final long startTime = System.currentTimeMillis();
        final int counter = mCounter++;
        log.info("{}", String.format("HttpServlet.service(): id: %3d, request URI: %s", counter, req.getRequestURI()));
        log.debug("Handling Servlet request asynchronously: " + req);

        AsyncContext async = req.startAsync();
        async.setTimeout(configuration.getAsyncTimeoutSeconds() * 1000);

        async.addListener(new AsyncListener() {

            @Override
            public void onTimeout(AsyncEvent arg0) throws IOException {
                long duration = System.currentTimeMillis() - startTime;
                log.debug("{}", String.format("AsyncListener.onTimeout(): id: %3d, duration: %,4d, request: %s", counter,
                        duration, arg0.getSuppliedRequest()));
            }

            @Override
            public void onStartAsync(AsyncEvent arg0) throws IOException {
                // useless
                log.debug("{}", String.format("AsyncListener.onStartAsync(): id: %3d, request: %s", counter,
                        arg0.getSuppliedRequest()));
            }

            @Override
            public void onError(AsyncEvent arg0) throws IOException {
                long duration = System.currentTimeMillis() - startTime;
                log.debug("{}", String.format("AsyncListener.onError(): id: %3d, duration: %,4d, response: %s", counter,
                        duration, arg0.getSuppliedResponse()));
            }

            @Override
            public void onComplete(AsyncEvent arg0) throws IOException {
                long duration = System.currentTimeMillis() - startTime;
                log.debug("{}", String.format("AsyncListener.onComplete(): id: %3d, duration: %,4d, response: %s", counter,
                        duration, arg0.getSuppliedResponse()));
            }

        });

        AsyncServletUpnpStream stream = new AsyncServletUpnpStream(router.getProtocolFactory(), async, req) {
            @Override
            protected Connection createConnection() {
                return new ServletConnection(getRequest());
            }
        };

        router.received(stream);
    }
}
