package com.cv.media.lib.api.okhttp;

import androidx.annotation.NonNull;

import com.cv.media.lib.common_utils.async.ThreadUtils;
import com.cv.media.lib.common_utils.utils.StringUtils;
import com.cv.media.lib.storage.mmkv.CommonPreference;
import com.franmontiel.persistentcookiejar.PersistentCookieJar;
import com.franmontiel.persistentcookiejar.cache.SetCookieCache;
import com.franmontiel.persistentcookiejar.persistence.CookiePersistor;
import com.franmontiel.persistentcookiejar.persistence.SerializableCookie;

import java.io.IOException;
import java.net.HttpCookie;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import okhttp3.Cookie;
import okhttp3.CookieJar;
import okhttp3.HttpUrl;
import okhttp3.Interceptor;
import okhttp3.Request;
import okhttp3.Response;
import okhttp3.internal.http.HttpHeaders;

/**
 * Cookie拦截器
 * 1.支持Cookies标准协议 2.Cookies添加 3.Cookies持久
 *
 * @Author Damon
 */
public abstract class BaseCookieInterceptor implements Interceptor {
    StringBuffer mBuffer = new StringBuffer(128);
    private volatile static CookieJar cookieJar;
    static {
        ThreadUtils.execute(new Runnable() {
            @Override
            public void run() {
                CookieJar cj = new PersistentCookieJar(new SetCookieCache(), new CookiePersistImpl());
                cookieJar = cj;
            }
        });
    }

    @Override
    public final Response intercept(Chain chain) throws IOException {
        String cookies = null;
        Request request = chain.request();
        List<Cookie> cookiesFromCJ = null;
        if (cookieJar != null) cookiesFromCJ = cookieJar.loadForRequest(request.url());

        synchronized (BaseCookieInterceptor.class) {
            Map<String, String> newParams = onAddCookiesValue();
            mBuffer.setLength(0);
            Set<Map.Entry<String, String>> entries = newParams.entrySet();

            if (cookiesFromCJ != null) {
                for (Cookie cookie : cookiesFromCJ) {
                    mBuffer.append(cookie.name() + "=" + cookie.value() + ";");
                }
            }

            for (Map.Entry<String, String> entry : entries) {
                if (!StringUtils.isEmpty(entry.getValue()))
                    mBuffer.append(entry.getKey()).append("=").append(entry.getValue()).append(";");
            }
            if (entries.size() > 0 || (cookiesFromCJ != null && cookiesFromCJ.size() > 0))
                mBuffer.delete(mBuffer.length() - 1, mBuffer.length());
            cookies = mBuffer.toString();
        }


        Request.Builder newRequest = request.newBuilder();
        Response response = chain.proceed(newRequest.addHeader("Cookie", cookies).build());
        if (cookieJar != null)
            HttpHeaders.receiveHeaders(cookieJar, response.request().url(), response.headers());
        return response;
    }

    abstract protected @NonNull
    Map<String, String> onAddCookiesValue();

    private static class CookiePersistImpl implements CookiePersistor {
        CommonPreference commonPreference = CommonPreference.getInstance();
        final String KEY_COOKIES = "CP_KEY_COOKIES";
        final String SPLIT = "@";
        private final HashMap<String, Cookie> keyToCookie = new HashMap<>();

        private static String createCookieKey(Cookie cookie) {
            return (cookie.secure() ? "https" : "http") + "://" + cookie.domain() + cookie.path() + "|" + cookie.name();
        }

        @Override
        public List<Cookie> loadAll() {
            keyToCookie.clear();
            List<Cookie> cookies = new ArrayList<>();
            String content = commonPreference.getString(KEY_COOKIES);
            if (!StringUtils.isEmpty(content)) {
                String[] cookiesSerial = content.split(SPLIT);
                for (String cookieSerial : cookiesSerial) {
                    Cookie cookie = new SerializableCookie().decode(cookieSerial);
                    if (cookie != null) {
                        keyToCookie.put(createCookieKey(cookie), cookie);
                        cookies.add(cookie);
                    }
                }
            }
            return cookies;
        }

        @Override
        public void saveAll(Collection<Cookie> cookies) {
            if (cookies != null && !cookies.isEmpty()) {
                for (Cookie tmp : cookies) {
                    keyToCookie.put(createCookieKey(tmp), tmp);
                }
                push(new ArrayList<>(keyToCookie.values()));
            }
        }

        @Override
        public void removeAll(Collection<Cookie> cookies) {
            if (cookies != null && !cookies.isEmpty()) {
                for (Cookie tmp : cookies) {
                    keyToCookie.remove(createCookieKey(tmp));
                }
                push(new ArrayList<>(keyToCookie.values()));
            }
        }

        private void push(List<Cookie> cookiesList) {
            if (cookiesList != null && !cookiesList.isEmpty()) {
                String[] cookiesSerial = new String[cookiesList.size()];
                for (int i = 0; i < cookiesList.size(); i++) {
                    cookiesSerial[i] = new SerializableCookie().encode(cookiesList.get(i));
                }
                commonPreference.putString(KEY_COOKIES, org.apache.commons.lang3.StringUtils.join(cookiesSerial, SPLIT));
            } else
                commonPreference.putString(KEY_COOKIES, "");
        }


        @Override
        public void clear() {
            push(null);
        }
    }
}
