package com.cv.media.lib.plugin.packing

import com.cv.media.lib.plugin.Logger
import org.gradle.api.GradleException
import org.gradle.api.tasks.Input
import org.gradle.api.tasks.Optional

import java.util.zip.ZipEntry
import java.util.zip.ZipFile

/**
 * 百度加固打包任务
 * 职责:apkprotect [-hvV] [-url serverAddress] -akey AKEY -skey SKEY -i input_file -o output_file [-type type] [-lib libs] [-shell shell_version] [-t timeout] [-nocache] [-g taskID] [-config configID]
 参数说明
 -h 打印帮助信息
 -v 打印详细日志
 -V 打印版本信息
 -url [url] 加固地址 默认:https://apkprotect.baidu.com
 -akey access_key, 登录后在加固配置处获取
 -skey security_key，登录后在加固配置处获取
 -i [input_file] 输入文件
 -o [output_file] 输出文件
 -type 加固类型：free|standard|ultimate|so|jar，分别对应：免费版|专业版|旗舰版|so|安卓SDK加固
 -lib [lib1:lib2...] 加固类型为so时，输入文件中指定需要加固so文件名（不包括路径）
 -shell 加固服务版本号，高级用户使用
 -t $timeout 设置等待超时时间, 单位秒, 默认300秒
 -nocache 禁用缓存，默认开启
 -g $taskID 用于离线加固，超时的任务在一段时间后通过-g指令获取加固结果
 -config $configID 用于指定页面配置的id进行加固，常用于安卓SDK指定配置文件
 * @author Roger
 */
class BaiduApkPackingTask extends BaseApkPackingTask {
    @Input
    @Optional
    String aKey
    @Input
    @Optional
    String sKey

    BaiduApkPackingTask() {
        setGroup("百度加固打包");
    }

    /**
     * 检测当前操作系统类型
     * @return "windows", "mac", "linux" 或抛出异常
     */
    private String detectOperatingSystem() {
        String osName = System.getProperty("os.name").toLowerCase()
        logs("检测操作系统: ${System.getProperty("os.name")}")
        if (osName.contains("win")) {
            logs("识别为Windows平台")
            return "windows"
        } else if (osName.contains("mac")) {
            logs("识别为macOS平台")
            return "mac"
        } else if (osName.contains("linux") || osName.contains("unix")) {
            logs("识别为Linux平台")
            return "linux"
        } else {
            throw new GradleException("不支持的操作系统: ${osName}")
        }
    }

    /**
     * 根据操作系统返回对应的apkprotect可执行文件名
     * @return 可执行文件名
     */
    private String getApkProtectExecutable() {
        String os = detectOperatingSystem()
        String executableName
        switch (os) {
            case "windows":
                executableName = "apkprotect.exe"
                break
            case "mac":
                executableName = "apkprotect_mac"
                break
            case "linux":
                executableName = "apkprotect_linux"
                break
            default:
                throw new GradleException("不支持的操作系统: ${os}")
        }
        logs("选择可执行文件: ${executableName} (平台: ${os})")
        return executableName
    }

    /**
     * 为Mac/Linux可执行文件设置执行权限
     * @param executableFile 可执行文件
     */
    private void ensureExecutable(File executableFile) {
        if (!executableFile.exists()) {
            throw new GradleException("可执行文件不存在: ${executableFile.path}")
        }
        String os = detectOperatingSystem()
        if (os == "mac" || os == "linux") {
            executableFile.setExecutable(true)
            logs("设置可执行权限: ${executableFile.path}")
        }
    }

    @Override
    File makeProtectedApk(File apkOriginFile) {
        logs("加固 开始...")
        Logger.startCount()
        
        // 检测BD_aKey和BD_sKey是否已配置（只在执行时检测）
        if (aKey == null || aKey.isEmpty()) {
            throw new GradleException("配置文件中缺少 BD_aKey 配置项，请在配置文件中添加 BD_aKey=xxxxxx")
        }
        if (sKey == null || sKey.isEmpty()) {
            throw new GradleException("配置文件中缺少 BD_sKey 配置项，请在配置文件中添加 BD_sKey=xxxxxxxx")
        }
        
        logs("使用配置的 BD_aKey: ${aKey.substring(0, Math.min(4, aKey.length()))}...")
        logs("使用配置的 BD_sKey: ${sKey.substring(0, Math.min(4, sKey.length()))}...")
        
        // 根据操作系统获取对应的可执行文件名
        String executableName = getApkProtectExecutable()
        File executableFile = new File(toolsDir, executableName)
        logs("可执行文件路径: ${executableFile.path}")
        
        // 为Mac/Linux版本设置可执行权限
        ensureExecutable(executableFile)
        
        // 加固语法
        String[] argArr = [
                executableFile.path,
                "-akey",
                aKey,
                "-skey",
                sKey,
                "-i",
                "${apkOriginFile.path}",
                "-o",
                "${mOutputDir.path}${File.separator}enc_unsigned.apk",
                "-type",
                "so"
        ]

        String[] javaVersion = new String[2]
        javaVersion[0] = "java"
        javaVersion[1] = "-version"
        execute(javaVersion, "检查java版本服务")
        Logger.startCount()
        execute(argArr, "百度加固服务")
        Logger.endCount("加固任务")
        logs("加固 结束...")

        List<File> apks = mOutputDir.listFiles().findAll {
            it.name.contains("enc_unsigned.apk")
        }

        if (apks.size() != 1) {
            throw new GradleException("Apk加固异常: 找到${apks.size()}个匹配apk")
        }
        File protectApkFile = new File(mOutputDir, "${apkOriginFile.name.replace(".apk", "_protected.apk")}")
        logs("更改加固后apk:${apks[0].name}为: $protectApkFile.name")
        apks[0].renameTo(protectApkFile)
        if (!protectApkFile.exists()) throw new GradleException("Apk加固失败: 加固包不存在")

        logs("开始检查加固后apk: $protectApkFile.name")
        ZipFile zipFileOfPAF = new ZipFile(protectApkFile)
        Enumeration<?> entries = zipFileOfPAF.entries()
        boolean hasBaiduSo = false
        while (entries.hasMoreElements()) {
            ZipEntry entry = (ZipEntry) entries.nextElement()
            if (entry.name.endsWith("baiduprotect.so")) {
                hasBaiduSo = true
                break
            }
        }

        if (!hasBaiduSo) {
            throw new GradleException("Apk加固异常: 未发现baiduprotect.so")
        }

        logs("加固成功: $protectApkFile.name")
        return protectApkFile
    }

    @Override
    void logs(String content) {
        println("${BaiduApkPackingTask.simpleName}--> ${content}")
    }
}
