package com.ott.trans;

import com.ott.trans.jni.J2NMain;
import com.ott.trans.utils.LogUtils;

import java.util.Map;

/**
 * Created by Joyce.wang on 2022/4/19.
 */
public class JniApiImpl {
    private static final String TAG = "OttTransApi";

    public static final int NETWORK_STATE_ETHERNET = 0;
    public static final int NETWORK_STATE_WIFI = 1;
    public static final int NETWORK_STATE_MOBILE = 2;
    public static final int NETWORK_STATE_DISCONNECTED = 3;
    public static final int NETWORK_STATE_UNKNOWN = 4;

    private boolean haveLoad = false;
    private String jniPathName = "";
    private final J2NMain jniApi;
    private static JniApiImpl INSTANCE;

    public static String seperator_comma = ",";
    public static String seperator_and = "&";

    public synchronized static void init(String bikcoreSoPath, String jniLibSoPath) {
        if (INSTANCE != null) {
            return;
        }
        INSTANCE = new JniApiImpl();
        INSTANCE.loadBikcore(bikcoreSoPath);
        INSTANCE.loadLib(jniLibSoPath);
    }

    public synchronized static void init(String soPath) {
        if (INSTANCE != null) {
            return;
        }
        INSTANCE = new JniApiImpl();
        INSTANCE.loadBikcore("");
        INSTANCE.loadLib(soPath);
    }

    public static JniApiImpl getInstance() {
        if (INSTANCE == null) {
            throw new IllegalStateException("OttTransApi is not initialized!");
        }
        return INSTANCE;
    }

    private JniApiImpl() {
        jniApi = new J2NMain();
    }

    private void loadBikcore(String path) {
//        BikCore.init(path);
    }

    private void loadLib(String path) {
        if (haveLoad) {
            return;
        }

        synchronized (J2NMain.class) {
            if (haveLoad) return;
            if (path != null && path.length() > 0) {
                try {
                    System.load(path);
                    haveLoad = true;
                    jniPathName = path;
                    return;
                } catch (Throwable t) {
                    LogUtils.e(TAG, "loadLib failed: " + path);
                }
            }

            System.loadLibrary("trans");
            haveLoad = true;
        }
    }

    public int init(String workPath, String cfgFile, Map<String, String> params) {
        if (isLoad()) {
            return jniApi.init(workPath, cfgFile, params);//错误返回-2，成功返回0
        }
        return -1;
    }

    public int uninit() {
        if (isLoad()) {
            return jniApi.destroy();//返回0
        }
        return -1;
    }

    // 直播调用
    public int startChannel(String sysParams, String channParams, Map<String, String> params) {
        if (isLoad()) {
            return jniApi.playLiveCall(sysParams, channParams, null, params);//出错返回-1，成功返回 > 0
        }
        return -1;
    }

    // 如果把mptUrl按key:value格式放在channParams，解析会有问题
    // 点播调用 startVodChannel
    public int startVodChannel(String channParams, String mptContent) {
        if (isLoad()) {
            return jniApi.playVodCall(channParams, mptContent);//出错返回-1，成功返回 > 0
        }
        return -1;
    }

    public int requestData(int channelId, long offset) {
        if (isLoad()) {
            return jniApi.requestCall(channelId, offset);//出错返回-1，成功返回 > 0
        }
        return -1;
    }

    public int requestDataByBlockName(int channelId, String blockName) {
        if (isLoad()) {
            return jniApi.requestCallByName(channelId, blockName);//出错返回-1，成功返回 > 0
        }
        return -1;
    }

    public int requestDataFromBlockName(int channelId, String blockName) {
        if (isLoad()) {
            return jniApi.requestCallFromName(channelId, blockName);//出错返回-1，成功返回 > 0
        }
        return -1;
    }

    public String getChannelM3u8(String chanHash) {
        if (isLoad()) {
            return jniApi.getCallM3u8(chanHash);
        }
        return "";
    }

    public int setChannelState(String chanHash, int downState, int cacheState) {
        if (isLoad()) {
            return jniApi.setCallState(chanHash, downState, cacheState);//出错返回-1，成功返回 > 0
        }
        return -1;
    }

    public String getLocalChannels() {
        if (isLoad()) {
            return jniApi.getLocalCalls();
        }
        return "";
    }

    public int stopChannel(int channelId) {
        if (isLoad()) {
            return jniApi.stopCall(channelId);//返回0
        }
        return -1;
    }

    public int setParam(String cfg, Object value) {
        if (isLoad()) {
            return jniApi.setCallParams(cfg, value);//返回0
        }
        return -1;
    }

    public int setNetworkState(NetworkState state, String localIp) {
        return setNetworkState(state.ordinal(), localIp);
    }

    public int setNetworkState(int state, String localIp) {
        if (isLoad()) {
            return jniApi.setNetworkMode(state, localIp);//出错返回-1，成功返回0
        }
        return -1;
    }

    public int setTransMask(int mask) {
        if (isLoad()) {
            return jniApi.setCallMask(mask);//返回0
        }
        return -1;
    }

    public void setUpnpResult(int channelId, String ipList) {
        if (isLoad()) {
            jniApi.setUpnpIpList(channelId, ipList);//返回0
        }
    }

    // period : ms
    public void setLog(boolean enable, int period) {
        if (isLoad()) {
            jniApi.enableLog(enable, period);//返回0
        }
    }

    public int flushLog() {
        if (isLoad()) {
            return jniApi.clearLog();//返回0
        }
        return -1;
    }

    public int getRand() {
        if (isLoad()) {
            return jniApi.getRsRand();//返回 > 0
        }
        return -1;
    }

    public String getVersion() {
        if (isLoad()) {
            return jniApi.getStringVersion();
        }
        return "";
    }

    public byte[] encryptByPrivateKey(byte[] content, byte[] key) {
        if (isLoad()) {
            return jniApi.encryptByPriKey(content, key);
        }
        return null;
    }

    public byte[] decryptByPrivateKey(byte[] content, byte[] key) {
        if (isLoad()) {
            return jniApi.decryptByPriKey(content, key);
        }
        return null;
    }

    public byte[] encryptByPublicKey(byte[] content, byte[] key) {
        if (isLoad()) {
            return jniApi.encryptByPubKey(content, key);
        }
        return null;
    }

    public byte[] decryptByPublicKey(byte[] content, byte[] key) {
        if (isLoad()) {
            return jniApi.decryptByPubKey(content, key);
        }
        return null;
    }

    public void notifyBuffering(int channelId, int times, int ms) {
        if (isLoad()) {
            jniApi.notifyPlayerBuffering(channelId, times, ms);
        }
    }

    public void updateChannel(int var1, String var2) {
        if (isLoad()) {
            jniApi.updateCall(var1, var2);
        }
    }

    public boolean isBlockCached(int chanId, int blockId) {
        if (isLoad()) {
            return jniApi.isBlockDownloaded(chanId, blockId);//没有返回0，有返回1
        }
        return false;
    }

    public void sendFrontLog(int chanId, String log) {
        if (isLoad()) {
            jniApi.sendAppLog(chanId, log);
        }
    }

    public int setAndGetPort(int port) {
        if (isLoad()) {
            return jniApi.getHttpProxyPort(port);
        }
        return -1;
    }

    public void initEngineProxyBuffer(int chanId, int blockLen, long resTotalLen, int solutionType, int solutionBrand, String drmKeyLink, int useBlockIdMap) {
        if (isLoad()) {
            jniApi.initHttpProxyBuffer(chanId, blockLen, resTotalLen, solutionType, solutionBrand, drmKeyLink, "", useBlockIdMap, 1, 0);
        }
    }

    public void initEngineProxyBuffer(int chanId, int blockLen, long resTotalLen, int solutionType, int solutionBrand, String drmKeyLink, String streamKey, int useBlockIdMap, int proxyPushMode, int isAbr) {
        if (isLoad()) {
            jniApi.initHttpProxyBuffer(chanId, blockLen, resTotalLen, solutionType, solutionBrand, drmKeyLink, streamKey, useBlockIdMap, proxyPushMode, isAbr);
        }
    }

    public void closeEngineProxyBuffer(int chanId) {
        if (isLoad()) {
            jniApi.closeHttpProxyBuffer(chanId);
        }
    }

    public void setBlockDecryptKey(int chanId, String key) {
        if (isLoad()) {
            jniApi.setBlockAesKey(chanId, key);
        }
    }

    private boolean isLoad() {
        if (!haveLoad) {
            LogUtils.w(TAG, "did not load trans so");
        }
        return haveLoad;
    }

    public String getJniPathName() {
        return this.jniPathName;
    }

    public int getDownSpeed(int chanId) {
        if (isLoad()) {
            return jniApi.getDownloadSpeed(chanId);
        }
        return -1;
    }

    public int requestVodInfoToPlay(String channParams, String mptContent) {
        if (isLoad()) {
            return jniApi.requestVodCallInfo(channParams, mptContent);
        }
        return -1;
    }
}
