/**
 * <h1>AWS CDK Docker Image Assets</h1>
 * <p>
 * <!--BEGIN STABILITY BANNER-->---
 * <p>
 * <img alt="cdk-constructs: Stable" src="https://img.shields.io/badge/cdk--constructs-stable-success.svg?style=for-the-badge">
 * <p>
 * <hr>
 * <p>
 * <!--END STABILITY BANNER-->
 * <p>
 * This module allows bundling Docker images as assets.
 * <p>
 * <h2>Images from Dockerfile</h2>
 * <p>
 * Images are built from a local Docker context directory (with a <code>Dockerfile</code>),
 * uploaded to Amazon Elastic Container Registry (ECR) by the CDK toolkit
 * and/or your app's CI/CD pipeline, and can be naturally referenced in your CDK app.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.ecr.assets.DockerImageAsset;
 * 
 * 
 * DockerImageAsset asset = DockerImageAsset.Builder.create(this, "MyBuildImage")
 *         .directory(join(__dirname, "my-image"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * The directory <code>my-image</code> must include a <code>Dockerfile</code>.
 * <p>
 * This will instruct the toolkit to build a Docker image from <code>my-image</code>, push it
 * to an Amazon ECR repository and wire the name of the repository as CloudFormation
 * parameters to your stack.
 * <p>
 * By default, all files in the given directory will be copied into the docker
 * <em>build context</em>. If there is a large directory that you know you definitely
 * don't need in the build context you can improve the performance by adding the
 * names of files and directories to ignore to a file called <code>.dockerignore</code>, or
 * pass them via the <code>exclude</code> property. If both are available, the patterns
 * found in <code>exclude</code> are appended to the patterns found in <code>.dockerignore</code>.
 * <p>
 * The <code>ignoreMode</code> property controls how the set of ignore patterns is
 * interpreted. The recommended setting for Docker image assets is
 * <code>IgnoreMode.DOCKER</code>. If the context flag
 * <code>&#64;aws-cdk/aws-ecr-assets:dockerIgnoreSupport</code> is set to <code>true</code> in your
 * <code>cdk.json</code> (this is by default for new projects, but must be set manually for
 * old projects) then <code>IgnoreMode.DOCKER</code> is the default and you don't need to
 * configure it on the asset itself.
 * <p>
 * Use <code>asset.imageUri</code> to reference the image. It includes both the ECR image URL
 * and tag.
 * <p>
 * You can optionally pass build args to the <code>docker build</code> command by specifying
 * the <code>buildArgs</code> property. It is recommended to skip hashing of <code>buildArgs</code> for
 * values that can change between different machines to maintain a consistent
 * asset hash.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.ecr.assets.DockerImageAsset;
 * 
 * 
 * DockerImageAsset asset = DockerImageAsset.Builder.create(this, "MyBuildImage")
 *         .directory(join(__dirname, "my-image"))
 *         .buildArgs(Map.of(
 *                 "HTTP_PROXY", "http://10.20.30.2:1234"))
 *         .invalidation(DockerImageAssetInvalidationOptions.builder()
 *                 .buildArgs(false)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * You can optionally pass a target to the <code>docker build</code> command by specifying
 * the <code>target</code> property:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.ecr.assets.DockerImageAsset;
 * 
 * 
 * DockerImageAsset asset = DockerImageAsset.Builder.create(this, "MyBuildImage")
 *         .directory(join(__dirname, "my-image"))
 *         .target("a-target")
 *         .build();
 * </pre></blockquote>
 * <p>
 * You can optionally pass networking mode to the <code>docker build</code> command by specifying
 * the <code>networkMode</code> property:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.ecr.assets.DockerImageAsset;
 * import software.amazon.awscdk.services.ecr.assets.NetworkMode;
 * 
 * 
 * DockerImageAsset asset = DockerImageAsset.Builder.create(this, "MyBuildImage")
 *         .directory(join(__dirname, "my-image"))
 *         .networkMode(NetworkMode.HOST)
 *         .build();
 * </pre></blockquote>
 * <p>
 * You can optionally pass an alternate platform to the <code>docker build</code> command by specifying
 * the <code>platform</code> property:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.ecr.assets.DockerImageAsset;
 * import software.amazon.awscdk.services.ecr.assets.Platform;
 * 
 * 
 * DockerImageAsset asset = DockerImageAsset.Builder.create(this, "MyBuildImage")
 *         .directory(join(__dirname, "my-image"))
 *         .platform(Platform.LINUX_ARM64)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Images from Tarball</h2>
 * <p>
 * Images are loaded from a local tarball, uploaded to ECR by the CDK toolkit and/or your app's CI-CD pipeline, and can be
 * naturally referenced in your CDK app.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.ecr.assets.TarballImageAsset;
 * 
 * 
 * TarballImageAsset asset = TarballImageAsset.Builder.create(this, "MyBuildImage")
 *         .tarballFile("local-image.tar")
 *         .build();
 * </pre></blockquote>
 * <p>
 * This will instruct the toolkit to add the tarball as a file asset. During deployment it will load the container image
 * from <code>local-image.tar</code>, push it to an Amazon ECR repository and wire the name of the repository as CloudFormation parameters
 * to your stack.
 * <p>
 * <h2>Publishing images to ECR repositories</h2>
 * <p>
 * <code>DockerImageAsset</code> is designed for seamless build &amp; consumption of image assets by CDK code deployed to multiple environments
 * through the CDK CLI or through CI/CD workflows. To that end, the ECR repository behind this construct is controlled by the AWS CDK.
 * The mechanics of where these images are published and how are intentionally kept as an implementation detail, and the construct
 * does not support customizations such as specifying the ECR repository name or tags.
 * <p>
 * If you are looking for a way to <em>publish</em> image assets to an ECR repository in your control, you should consider using
 * <a href="https://github.com/cdklabs/cdk-ecr-deployment">cdklabs/cdk-ecr-deployment</a>, which is able to replicate an image asset from the CDK-controlled ECR repository to a repository of
 * your choice.
 * <p>
 * Here an example from the <a href="https://github.com/cdklabs/cdk-ecr-deployment">cdklabs/cdk-ecr-deployment</a> project:
 * <p>
 * <blockquote><pre>
 * // This example available in TypeScript only
 * 
 * import { DockerImageAsset } from '&#64;aws-cdk/aws-ecr-assets';
 * import * as ecrdeploy from 'cdk-ecr-deployment';
 * 
 * const image = new DockerImageAsset(this, 'CDKDockerImage', {
 *   directory: path.join(__dirname, 'docker'),
 * });
 * 
 * new ecrdeploy.ECRDeployment(this, 'DeployDockerImage', {
 *   src: new ecrdeploy.DockerImageName(image.imageUri),
 *   dest: new ecrdeploy.DockerImageName(`${cdk.Aws.ACCOUNT_ID}.dkr.ecr.us-west-2.amazonaws.com/test:nginx`),
 * });
 * </pre></blockquote>
 * <p>
 * ⚠️ Please note that this is a 3rd-party construct library and is not officially supported by AWS.
 * You are welcome to +1 <a href="https://github.com/aws/aws-cdk/issues/12597">this GitHub issue</a> if you would like to see
 * native support for this use-case in the AWS CDK.
 * <p>
 * <h2>Pull Permissions</h2>
 * <p>
 * Depending on the consumer of your image asset, you will need to make sure
 * the principal has permissions to pull the image.
 * <p>
 * In most cases, you should use the <code>asset.repository.grantPull(principal)</code>
 * method. This will modify the IAM policy of the principal to allow it to
 * pull images from this repository.
 * <p>
 * If the pulling principal is not in the same account or is an AWS service that
 * doesn't assume a role in your account (e.g. AWS CodeBuild), pull permissions
 * must be granted on the <strong>resource policy</strong> (and not on the principal's policy).
 * To do that, you can use <code>asset.repository.addToResourcePolicy(statement)</code> to
 * grant the desired principal the following permissions: "ecr:GetDownloadUrlForLayer",
 * "ecr:BatchGetImage" and "ecr:BatchCheckLayerAvailability".
 */
@software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
package software.amazon.awscdk.services.ecr.assets;
