/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.acmpca.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Certificate revocation information used by the <a
 * href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_CreateCertificateAuthority.html"
 * >CreateCertificateAuthority</a> and <a
 * href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_UpdateCertificateAuthority.html"
 * >UpdateCertificateAuthority</a> actions. Your private certificate authority (CA) can configure Online Certificate
 * Status Protocol (OCSP) support and/or maintain a certificate revocation list (CRL). OCSP returns validation
 * information about certificates as requested by clients, and a CRL contains an updated list of certificates revoked by
 * your CA. For more information, see <a
 * href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_RevokeCertificate.html">RevokeCertificate</a> and
 * <a href="https://docs.aws.amazon.com/acm-pca/latest/userguide/revocation-setup.html">Setting up a certificate
 * revocation method</a> in the <i>Private Certificate Authority (PCA) User Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class RevocationConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<RevocationConfiguration.Builder, RevocationConfiguration> {
    private static final SdkField<CrlConfiguration> CRL_CONFIGURATION_FIELD = SdkField
            .<CrlConfiguration> builder(MarshallingType.SDK_POJO).memberName("CrlConfiguration")
            .getter(getter(RevocationConfiguration::crlConfiguration)).setter(setter(Builder::crlConfiguration))
            .constructor(CrlConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CrlConfiguration").build()).build();

    private static final SdkField<OcspConfiguration> OCSP_CONFIGURATION_FIELD = SdkField
            .<OcspConfiguration> builder(MarshallingType.SDK_POJO).memberName("OcspConfiguration")
            .getter(getter(RevocationConfiguration::ocspConfiguration)).setter(setter(Builder::ocspConfiguration))
            .constructor(OcspConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OcspConfiguration").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CRL_CONFIGURATION_FIELD,
            OCSP_CONFIGURATION_FIELD));

    private static final long serialVersionUID = 1L;

    private final CrlConfiguration crlConfiguration;

    private final OcspConfiguration ocspConfiguration;

    private RevocationConfiguration(BuilderImpl builder) {
        this.crlConfiguration = builder.crlConfiguration;
        this.ocspConfiguration = builder.ocspConfiguration;
    }

    /**
     * <p>
     * Configuration of the certificate revocation list (CRL), if any, maintained by your private CA. A CRL is typically
     * updated approximately 30 minutes after a certificate is revoked. If for any reason a CRL update fails, ACM
     * Private CA makes further attempts every 15 minutes.
     * </p>
     * 
     * @return Configuration of the certificate revocation list (CRL), if any, maintained by your private CA. A CRL is
     *         typically updated approximately 30 minutes after a certificate is revoked. If for any reason a CRL update
     *         fails, ACM Private CA makes further attempts every 15 minutes.
     */
    public final CrlConfiguration crlConfiguration() {
        return crlConfiguration;
    }

    /**
     * <p>
     * Configuration of Online Certificate Status Protocol (OCSP) support, if any, maintained by your private CA. When
     * you revoke a certificate, OCSP responses may take up to 60 minutes to reflect the new status.
     * </p>
     * 
     * @return Configuration of Online Certificate Status Protocol (OCSP) support, if any, maintained by your private
     *         CA. When you revoke a certificate, OCSP responses may take up to 60 minutes to reflect the new status.
     */
    public final OcspConfiguration ocspConfiguration() {
        return ocspConfiguration;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(crlConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(ocspConfiguration());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RevocationConfiguration)) {
            return false;
        }
        RevocationConfiguration other = (RevocationConfiguration) obj;
        return Objects.equals(crlConfiguration(), other.crlConfiguration())
                && Objects.equals(ocspConfiguration(), other.ocspConfiguration());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("RevocationConfiguration").add("CrlConfiguration", crlConfiguration())
                .add("OcspConfiguration", ocspConfiguration()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "CrlConfiguration":
            return Optional.ofNullable(clazz.cast(crlConfiguration()));
        case "OcspConfiguration":
            return Optional.ofNullable(clazz.cast(ocspConfiguration()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<RevocationConfiguration, T> g) {
        return obj -> g.apply((RevocationConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, RevocationConfiguration> {
        /**
         * <p>
         * Configuration of the certificate revocation list (CRL), if any, maintained by your private CA. A CRL is
         * typically updated approximately 30 minutes after a certificate is revoked. If for any reason a CRL update
         * fails, ACM Private CA makes further attempts every 15 minutes.
         * </p>
         * 
         * @param crlConfiguration
         *        Configuration of the certificate revocation list (CRL), if any, maintained by your private CA. A CRL
         *        is typically updated approximately 30 minutes after a certificate is revoked. If for any reason a CRL
         *        update fails, ACM Private CA makes further attempts every 15 minutes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder crlConfiguration(CrlConfiguration crlConfiguration);

        /**
         * <p>
         * Configuration of the certificate revocation list (CRL), if any, maintained by your private CA. A CRL is
         * typically updated approximately 30 minutes after a certificate is revoked. If for any reason a CRL update
         * fails, ACM Private CA makes further attempts every 15 minutes.
         * </p>
         * This is a convenience method that creates an instance of the {@link CrlConfiguration.Builder} avoiding the
         * need to create one manually via {@link CrlConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link CrlConfiguration.Builder#build()} is called immediately and its
         * result is passed to {@link #crlConfiguration(CrlConfiguration)}.
         * 
         * @param crlConfiguration
         *        a consumer that will call methods on {@link CrlConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #crlConfiguration(CrlConfiguration)
         */
        default Builder crlConfiguration(Consumer<CrlConfiguration.Builder> crlConfiguration) {
            return crlConfiguration(CrlConfiguration.builder().applyMutation(crlConfiguration).build());
        }

        /**
         * <p>
         * Configuration of Online Certificate Status Protocol (OCSP) support, if any, maintained by your private CA.
         * When you revoke a certificate, OCSP responses may take up to 60 minutes to reflect the new status.
         * </p>
         * 
         * @param ocspConfiguration
         *        Configuration of Online Certificate Status Protocol (OCSP) support, if any, maintained by your private
         *        CA. When you revoke a certificate, OCSP responses may take up to 60 minutes to reflect the new status.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ocspConfiguration(OcspConfiguration ocspConfiguration);

        /**
         * <p>
         * Configuration of Online Certificate Status Protocol (OCSP) support, if any, maintained by your private CA.
         * When you revoke a certificate, OCSP responses may take up to 60 minutes to reflect the new status.
         * </p>
         * This is a convenience method that creates an instance of the {@link OcspConfiguration.Builder} avoiding the
         * need to create one manually via {@link OcspConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link OcspConfiguration.Builder#build()} is called immediately and its
         * result is passed to {@link #ocspConfiguration(OcspConfiguration)}.
         * 
         * @param ocspConfiguration
         *        a consumer that will call methods on {@link OcspConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #ocspConfiguration(OcspConfiguration)
         */
        default Builder ocspConfiguration(Consumer<OcspConfiguration.Builder> ocspConfiguration) {
            return ocspConfiguration(OcspConfiguration.builder().applyMutation(ocspConfiguration).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private CrlConfiguration crlConfiguration;

        private OcspConfiguration ocspConfiguration;

        private BuilderImpl() {
        }

        private BuilderImpl(RevocationConfiguration model) {
            crlConfiguration(model.crlConfiguration);
            ocspConfiguration(model.ocspConfiguration);
        }

        public final CrlConfiguration.Builder getCrlConfiguration() {
            return crlConfiguration != null ? crlConfiguration.toBuilder() : null;
        }

        public final void setCrlConfiguration(CrlConfiguration.BuilderImpl crlConfiguration) {
            this.crlConfiguration = crlConfiguration != null ? crlConfiguration.build() : null;
        }

        @Override
        public final Builder crlConfiguration(CrlConfiguration crlConfiguration) {
            this.crlConfiguration = crlConfiguration;
            return this;
        }

        public final OcspConfiguration.Builder getOcspConfiguration() {
            return ocspConfiguration != null ? ocspConfiguration.toBuilder() : null;
        }

        public final void setOcspConfiguration(OcspConfiguration.BuilderImpl ocspConfiguration) {
            this.ocspConfiguration = ocspConfiguration != null ? ocspConfiguration.build() : null;
        }

        @Override
        public final Builder ocspConfiguration(OcspConfiguration ocspConfiguration) {
            this.ocspConfiguration = ocspConfiguration;
            return this;
        }

        @Override
        public RevocationConfiguration build() {
            return new RevocationConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
