/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.applicationautoscaling.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents a target tracking scaling policy configuration to use with Application Auto Scaling.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class TargetTrackingScalingPolicyConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<TargetTrackingScalingPolicyConfiguration.Builder, TargetTrackingScalingPolicyConfiguration> {
    private static final SdkField<Double> TARGET_VALUE_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .memberName("TargetValue").getter(getter(TargetTrackingScalingPolicyConfiguration::targetValue))
            .setter(setter(Builder::targetValue))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TargetValue").build()).build();

    private static final SdkField<PredefinedMetricSpecification> PREDEFINED_METRIC_SPECIFICATION_FIELD = SdkField
            .<PredefinedMetricSpecification> builder(MarshallingType.SDK_POJO)
            .memberName("PredefinedMetricSpecification")
            .getter(getter(TargetTrackingScalingPolicyConfiguration::predefinedMetricSpecification))
            .setter(setter(Builder::predefinedMetricSpecification))
            .constructor(PredefinedMetricSpecification::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PredefinedMetricSpecification")
                    .build()).build();

    private static final SdkField<CustomizedMetricSpecification> CUSTOMIZED_METRIC_SPECIFICATION_FIELD = SdkField
            .<CustomizedMetricSpecification> builder(MarshallingType.SDK_POJO)
            .memberName("CustomizedMetricSpecification")
            .getter(getter(TargetTrackingScalingPolicyConfiguration::customizedMetricSpecification))
            .setter(setter(Builder::customizedMetricSpecification))
            .constructor(CustomizedMetricSpecification::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CustomizedMetricSpecification")
                    .build()).build();

    private static final SdkField<Integer> SCALE_OUT_COOLDOWN_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("ScaleOutCooldown").getter(getter(TargetTrackingScalingPolicyConfiguration::scaleOutCooldown))
            .setter(setter(Builder::scaleOutCooldown))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ScaleOutCooldown").build()).build();

    private static final SdkField<Integer> SCALE_IN_COOLDOWN_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("ScaleInCooldown").getter(getter(TargetTrackingScalingPolicyConfiguration::scaleInCooldown))
            .setter(setter(Builder::scaleInCooldown))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ScaleInCooldown").build()).build();

    private static final SdkField<Boolean> DISABLE_SCALE_IN_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("DisableScaleIn").getter(getter(TargetTrackingScalingPolicyConfiguration::disableScaleIn))
            .setter(setter(Builder::disableScaleIn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DisableScaleIn").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TARGET_VALUE_FIELD,
            PREDEFINED_METRIC_SPECIFICATION_FIELD, CUSTOMIZED_METRIC_SPECIFICATION_FIELD, SCALE_OUT_COOLDOWN_FIELD,
            SCALE_IN_COOLDOWN_FIELD, DISABLE_SCALE_IN_FIELD));

    private static final long serialVersionUID = 1L;

    private final Double targetValue;

    private final PredefinedMetricSpecification predefinedMetricSpecification;

    private final CustomizedMetricSpecification customizedMetricSpecification;

    private final Integer scaleOutCooldown;

    private final Integer scaleInCooldown;

    private final Boolean disableScaleIn;

    private TargetTrackingScalingPolicyConfiguration(BuilderImpl builder) {
        this.targetValue = builder.targetValue;
        this.predefinedMetricSpecification = builder.predefinedMetricSpecification;
        this.customizedMetricSpecification = builder.customizedMetricSpecification;
        this.scaleOutCooldown = builder.scaleOutCooldown;
        this.scaleInCooldown = builder.scaleInCooldown;
        this.disableScaleIn = builder.disableScaleIn;
    }

    /**
     * <p>
     * The target value for the metric. Although this property accepts numbers of type Double, it won't accept values
     * that are either too small or too large. Values must be in the range of -2^360 to 2^360. The value must be a valid
     * number based on the choice of metric. For example, if the metric is CPU utilization, then the target value is a
     * percent value that represents how much of the CPU can be used before scaling out.
     * </p>
     * 
     * @return The target value for the metric. Although this property accepts numbers of type Double, it won't accept
     *         values that are either too small or too large. Values must be in the range of -2^360 to 2^360. The value
     *         must be a valid number based on the choice of metric. For example, if the metric is CPU utilization, then
     *         the target value is a percent value that represents how much of the CPU can be used before scaling out.
     */
    public final Double targetValue() {
        return targetValue;
    }

    /**
     * <p>
     * A predefined metric. You can specify either a predefined metric or a customized metric.
     * </p>
     * 
     * @return A predefined metric. You can specify either a predefined metric or a customized metric.
     */
    public final PredefinedMetricSpecification predefinedMetricSpecification() {
        return predefinedMetricSpecification;
    }

    /**
     * <p>
     * A customized metric. You can specify either a predefined metric or a customized metric.
     * </p>
     * 
     * @return A customized metric. You can specify either a predefined metric or a customized metric.
     */
    public final CustomizedMetricSpecification customizedMetricSpecification() {
        return customizedMetricSpecification;
    }

    /**
     * <p>
     * The amount of time, in seconds, to wait for a previous scale-out activity to take effect.
     * </p>
     * <p>
     * With the <i>scale-out cooldown period</i>, the intention is to continuously (but not excessively) scale out.
     * After Application Auto Scaling successfully scales out using a target tracking scaling policy, it starts to
     * calculate the cooldown time. The scaling policy won't increase the desired capacity again unless either a larger
     * scale out is triggered or the cooldown period ends. While the cooldown period is in effect, the capacity added by
     * the initiating scale-out activity is calculated as part of the desired capacity for the next scale-out activity.
     * </p>
     * <p>
     * Application Auto Scaling provides a default value of 600 for Amazon ElastiCache replication groups and a default
     * value of 300 for the following scalable targets:
     * </p>
     * <ul>
     * <li>
     * <p>
     * AppStream 2.0 fleets
     * </p>
     * </li>
     * <li>
     * <p>
     * Aurora DB clusters
     * </p>
     * </li>
     * <li>
     * <p>
     * ECS services
     * </p>
     * </li>
     * <li>
     * <p>
     * EMR clusters
     * </p>
     * </li>
     * <li>
     * <p>
     * Neptune clusters
     * </p>
     * </li>
     * <li>
     * <p>
     * SageMaker endpoint variants
     * </p>
     * </li>
     * <li>
     * <p>
     * Spot Fleets
     * </p>
     * </li>
     * <li>
     * <p>
     * Custom resources
     * </p>
     * </li>
     * </ul>
     * <p>
     * For all other scalable targets, the default value is 0:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Amazon Comprehend document classification and entity recognizer endpoints
     * </p>
     * </li>
     * <li>
     * <p>
     * DynamoDB tables and global secondary indexes
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon Keyspaces tables
     * </p>
     * </li>
     * <li>
     * <p>
     * Lambda provisioned concurrency
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon MSK broker storage
     * </p>
     * </li>
     * </ul>
     * 
     * @return The amount of time, in seconds, to wait for a previous scale-out activity to take effect.</p>
     *         <p>
     *         With the <i>scale-out cooldown period</i>, the intention is to continuously (but not excessively) scale
     *         out. After Application Auto Scaling successfully scales out using a target tracking scaling policy, it
     *         starts to calculate the cooldown time. The scaling policy won't increase the desired capacity again
     *         unless either a larger scale out is triggered or the cooldown period ends. While the cooldown period is
     *         in effect, the capacity added by the initiating scale-out activity is calculated as part of the desired
     *         capacity for the next scale-out activity.
     *         </p>
     *         <p>
     *         Application Auto Scaling provides a default value of 600 for Amazon ElastiCache replication groups and a
     *         default value of 300 for the following scalable targets:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         AppStream 2.0 fleets
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Aurora DB clusters
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         ECS services
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         EMR clusters
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Neptune clusters
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         SageMaker endpoint variants
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Spot Fleets
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Custom resources
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For all other scalable targets, the default value is 0:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Amazon Comprehend document classification and entity recognizer endpoints
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         DynamoDB tables and global secondary indexes
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Amazon Keyspaces tables
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Lambda provisioned concurrency
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Amazon MSK broker storage
     *         </p>
     *         </li>
     */
    public final Integer scaleOutCooldown() {
        return scaleOutCooldown;
    }

    /**
     * <p>
     * The amount of time, in seconds, after a scale-in activity completes before another scale-in activity can start.
     * </p>
     * <p>
     * With the <i>scale-in cooldown period</i>, the intention is to scale in conservatively to protect your
     * application’s availability, so scale-in activities are blocked until the cooldown period has expired. However, if
     * another alarm triggers a scale-out activity during the scale-in cooldown period, Application Auto Scaling scales
     * out the target immediately. In this case, the scale-in cooldown period stops and doesn't complete.
     * </p>
     * <p>
     * Application Auto Scaling provides a default value of 600 for Amazon ElastiCache replication groups and a default
     * value of 300 for the following scalable targets:
     * </p>
     * <ul>
     * <li>
     * <p>
     * AppStream 2.0 fleets
     * </p>
     * </li>
     * <li>
     * <p>
     * Aurora DB clusters
     * </p>
     * </li>
     * <li>
     * <p>
     * ECS services
     * </p>
     * </li>
     * <li>
     * <p>
     * EMR clusters
     * </p>
     * </li>
     * <li>
     * <p>
     * Neptune clusters
     * </p>
     * </li>
     * <li>
     * <p>
     * SageMaker endpoint variants
     * </p>
     * </li>
     * <li>
     * <p>
     * Spot Fleets
     * </p>
     * </li>
     * <li>
     * <p>
     * Custom resources
     * </p>
     * </li>
     * </ul>
     * <p>
     * For all other scalable targets, the default value is 0:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Amazon Comprehend document classification and entity recognizer endpoints
     * </p>
     * </li>
     * <li>
     * <p>
     * DynamoDB tables and global secondary indexes
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon Keyspaces tables
     * </p>
     * </li>
     * <li>
     * <p>
     * Lambda provisioned concurrency
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon MSK broker storage
     * </p>
     * </li>
     * </ul>
     * 
     * @return The amount of time, in seconds, after a scale-in activity completes before another scale-in activity can
     *         start.</p>
     *         <p>
     *         With the <i>scale-in cooldown period</i>, the intention is to scale in conservatively to protect your
     *         application’s availability, so scale-in activities are blocked until the cooldown period has expired.
     *         However, if another alarm triggers a scale-out activity during the scale-in cooldown period, Application
     *         Auto Scaling scales out the target immediately. In this case, the scale-in cooldown period stops and
     *         doesn't complete.
     *         </p>
     *         <p>
     *         Application Auto Scaling provides a default value of 600 for Amazon ElastiCache replication groups and a
     *         default value of 300 for the following scalable targets:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         AppStream 2.0 fleets
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Aurora DB clusters
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         ECS services
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         EMR clusters
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Neptune clusters
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         SageMaker endpoint variants
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Spot Fleets
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Custom resources
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For all other scalable targets, the default value is 0:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Amazon Comprehend document classification and entity recognizer endpoints
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         DynamoDB tables and global secondary indexes
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Amazon Keyspaces tables
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Lambda provisioned concurrency
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Amazon MSK broker storage
     *         </p>
     *         </li>
     */
    public final Integer scaleInCooldown() {
        return scaleInCooldown;
    }

    /**
     * <p>
     * Indicates whether scale in by the target tracking scaling policy is disabled. If the value is <code>true</code>,
     * scale in is disabled and the target tracking scaling policy won't remove capacity from the scalable target.
     * Otherwise, scale in is enabled and the target tracking scaling policy can remove capacity from the scalable
     * target. The default value is <code>false</code>.
     * </p>
     * 
     * @return Indicates whether scale in by the target tracking scaling policy is disabled. If the value is
     *         <code>true</code>, scale in is disabled and the target tracking scaling policy won't remove capacity from
     *         the scalable target. Otherwise, scale in is enabled and the target tracking scaling policy can remove
     *         capacity from the scalable target. The default value is <code>false</code>.
     */
    public final Boolean disableScaleIn() {
        return disableScaleIn;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(targetValue());
        hashCode = 31 * hashCode + Objects.hashCode(predefinedMetricSpecification());
        hashCode = 31 * hashCode + Objects.hashCode(customizedMetricSpecification());
        hashCode = 31 * hashCode + Objects.hashCode(scaleOutCooldown());
        hashCode = 31 * hashCode + Objects.hashCode(scaleInCooldown());
        hashCode = 31 * hashCode + Objects.hashCode(disableScaleIn());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof TargetTrackingScalingPolicyConfiguration)) {
            return false;
        }
        TargetTrackingScalingPolicyConfiguration other = (TargetTrackingScalingPolicyConfiguration) obj;
        return Objects.equals(targetValue(), other.targetValue())
                && Objects.equals(predefinedMetricSpecification(), other.predefinedMetricSpecification())
                && Objects.equals(customizedMetricSpecification(), other.customizedMetricSpecification())
                && Objects.equals(scaleOutCooldown(), other.scaleOutCooldown())
                && Objects.equals(scaleInCooldown(), other.scaleInCooldown())
                && Objects.equals(disableScaleIn(), other.disableScaleIn());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("TargetTrackingScalingPolicyConfiguration").add("TargetValue", targetValue())
                .add("PredefinedMetricSpecification", predefinedMetricSpecification())
                .add("CustomizedMetricSpecification", customizedMetricSpecification())
                .add("ScaleOutCooldown", scaleOutCooldown()).add("ScaleInCooldown", scaleInCooldown())
                .add("DisableScaleIn", disableScaleIn()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "TargetValue":
            return Optional.ofNullable(clazz.cast(targetValue()));
        case "PredefinedMetricSpecification":
            return Optional.ofNullable(clazz.cast(predefinedMetricSpecification()));
        case "CustomizedMetricSpecification":
            return Optional.ofNullable(clazz.cast(customizedMetricSpecification()));
        case "ScaleOutCooldown":
            return Optional.ofNullable(clazz.cast(scaleOutCooldown()));
        case "ScaleInCooldown":
            return Optional.ofNullable(clazz.cast(scaleInCooldown()));
        case "DisableScaleIn":
            return Optional.ofNullable(clazz.cast(disableScaleIn()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<TargetTrackingScalingPolicyConfiguration, T> g) {
        return obj -> g.apply((TargetTrackingScalingPolicyConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, TargetTrackingScalingPolicyConfiguration> {
        /**
         * <p>
         * The target value for the metric. Although this property accepts numbers of type Double, it won't accept
         * values that are either too small or too large. Values must be in the range of -2^360 to 2^360. The value must
         * be a valid number based on the choice of metric. For example, if the metric is CPU utilization, then the
         * target value is a percent value that represents how much of the CPU can be used before scaling out.
         * </p>
         * 
         * @param targetValue
         *        The target value for the metric. Although this property accepts numbers of type Double, it won't
         *        accept values that are either too small or too large. Values must be in the range of -2^360 to 2^360.
         *        The value must be a valid number based on the choice of metric. For example, if the metric is CPU
         *        utilization, then the target value is a percent value that represents how much of the CPU can be used
         *        before scaling out.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetValue(Double targetValue);

        /**
         * <p>
         * A predefined metric. You can specify either a predefined metric or a customized metric.
         * </p>
         * 
         * @param predefinedMetricSpecification
         *        A predefined metric. You can specify either a predefined metric or a customized metric.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder predefinedMetricSpecification(PredefinedMetricSpecification predefinedMetricSpecification);

        /**
         * <p>
         * A predefined metric. You can specify either a predefined metric or a customized metric.
         * </p>
         * This is a convenience method that creates an instance of the {@link PredefinedMetricSpecification.Builder}
         * avoiding the need to create one manually via {@link PredefinedMetricSpecification#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link PredefinedMetricSpecification.Builder#build()} is called
         * immediately and its result is passed to {@link #predefinedMetricSpecification(PredefinedMetricSpecification)}.
         * 
         * @param predefinedMetricSpecification
         *        a consumer that will call methods on {@link PredefinedMetricSpecification.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #predefinedMetricSpecification(PredefinedMetricSpecification)
         */
        default Builder predefinedMetricSpecification(
                Consumer<PredefinedMetricSpecification.Builder> predefinedMetricSpecification) {
            return predefinedMetricSpecification(PredefinedMetricSpecification.builder()
                    .applyMutation(predefinedMetricSpecification).build());
        }

        /**
         * <p>
         * A customized metric. You can specify either a predefined metric or a customized metric.
         * </p>
         * 
         * @param customizedMetricSpecification
         *        A customized metric. You can specify either a predefined metric or a customized metric.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder customizedMetricSpecification(CustomizedMetricSpecification customizedMetricSpecification);

        /**
         * <p>
         * A customized metric. You can specify either a predefined metric or a customized metric.
         * </p>
         * This is a convenience method that creates an instance of the {@link CustomizedMetricSpecification.Builder}
         * avoiding the need to create one manually via {@link CustomizedMetricSpecification#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link CustomizedMetricSpecification.Builder#build()} is called
         * immediately and its result is passed to {@link #customizedMetricSpecification(CustomizedMetricSpecification)}.
         * 
         * @param customizedMetricSpecification
         *        a consumer that will call methods on {@link CustomizedMetricSpecification.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #customizedMetricSpecification(CustomizedMetricSpecification)
         */
        default Builder customizedMetricSpecification(
                Consumer<CustomizedMetricSpecification.Builder> customizedMetricSpecification) {
            return customizedMetricSpecification(CustomizedMetricSpecification.builder()
                    .applyMutation(customizedMetricSpecification).build());
        }

        /**
         * <p>
         * The amount of time, in seconds, to wait for a previous scale-out activity to take effect.
         * </p>
         * <p>
         * With the <i>scale-out cooldown period</i>, the intention is to continuously (but not excessively) scale out.
         * After Application Auto Scaling successfully scales out using a target tracking scaling policy, it starts to
         * calculate the cooldown time. The scaling policy won't increase the desired capacity again unless either a
         * larger scale out is triggered or the cooldown period ends. While the cooldown period is in effect, the
         * capacity added by the initiating scale-out activity is calculated as part of the desired capacity for the
         * next scale-out activity.
         * </p>
         * <p>
         * Application Auto Scaling provides a default value of 600 for Amazon ElastiCache replication groups and a
         * default value of 300 for the following scalable targets:
         * </p>
         * <ul>
         * <li>
         * <p>
         * AppStream 2.0 fleets
         * </p>
         * </li>
         * <li>
         * <p>
         * Aurora DB clusters
         * </p>
         * </li>
         * <li>
         * <p>
         * ECS services
         * </p>
         * </li>
         * <li>
         * <p>
         * EMR clusters
         * </p>
         * </li>
         * <li>
         * <p>
         * Neptune clusters
         * </p>
         * </li>
         * <li>
         * <p>
         * SageMaker endpoint variants
         * </p>
         * </li>
         * <li>
         * <p>
         * Spot Fleets
         * </p>
         * </li>
         * <li>
         * <p>
         * Custom resources
         * </p>
         * </li>
         * </ul>
         * <p>
         * For all other scalable targets, the default value is 0:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Amazon Comprehend document classification and entity recognizer endpoints
         * </p>
         * </li>
         * <li>
         * <p>
         * DynamoDB tables and global secondary indexes
         * </p>
         * </li>
         * <li>
         * <p>
         * Amazon Keyspaces tables
         * </p>
         * </li>
         * <li>
         * <p>
         * Lambda provisioned concurrency
         * </p>
         * </li>
         * <li>
         * <p>
         * Amazon MSK broker storage
         * </p>
         * </li>
         * </ul>
         * 
         * @param scaleOutCooldown
         *        The amount of time, in seconds, to wait for a previous scale-out activity to take effect.</p>
         *        <p>
         *        With the <i>scale-out cooldown period</i>, the intention is to continuously (but not excessively)
         *        scale out. After Application Auto Scaling successfully scales out using a target tracking scaling
         *        policy, it starts to calculate the cooldown time. The scaling policy won't increase the desired
         *        capacity again unless either a larger scale out is triggered or the cooldown period ends. While the
         *        cooldown period is in effect, the capacity added by the initiating scale-out activity is calculated as
         *        part of the desired capacity for the next scale-out activity.
         *        </p>
         *        <p>
         *        Application Auto Scaling provides a default value of 600 for Amazon ElastiCache replication groups and
         *        a default value of 300 for the following scalable targets:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        AppStream 2.0 fleets
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Aurora DB clusters
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        ECS services
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        EMR clusters
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Neptune clusters
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        SageMaker endpoint variants
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Spot Fleets
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Custom resources
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For all other scalable targets, the default value is 0:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Amazon Comprehend document classification and entity recognizer endpoints
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        DynamoDB tables and global secondary indexes
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Amazon Keyspaces tables
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Lambda provisioned concurrency
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Amazon MSK broker storage
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scaleOutCooldown(Integer scaleOutCooldown);

        /**
         * <p>
         * The amount of time, in seconds, after a scale-in activity completes before another scale-in activity can
         * start.
         * </p>
         * <p>
         * With the <i>scale-in cooldown period</i>, the intention is to scale in conservatively to protect your
         * application’s availability, so scale-in activities are blocked until the cooldown period has expired.
         * However, if another alarm triggers a scale-out activity during the scale-in cooldown period, Application Auto
         * Scaling scales out the target immediately. In this case, the scale-in cooldown period stops and doesn't
         * complete.
         * </p>
         * <p>
         * Application Auto Scaling provides a default value of 600 for Amazon ElastiCache replication groups and a
         * default value of 300 for the following scalable targets:
         * </p>
         * <ul>
         * <li>
         * <p>
         * AppStream 2.0 fleets
         * </p>
         * </li>
         * <li>
         * <p>
         * Aurora DB clusters
         * </p>
         * </li>
         * <li>
         * <p>
         * ECS services
         * </p>
         * </li>
         * <li>
         * <p>
         * EMR clusters
         * </p>
         * </li>
         * <li>
         * <p>
         * Neptune clusters
         * </p>
         * </li>
         * <li>
         * <p>
         * SageMaker endpoint variants
         * </p>
         * </li>
         * <li>
         * <p>
         * Spot Fleets
         * </p>
         * </li>
         * <li>
         * <p>
         * Custom resources
         * </p>
         * </li>
         * </ul>
         * <p>
         * For all other scalable targets, the default value is 0:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Amazon Comprehend document classification and entity recognizer endpoints
         * </p>
         * </li>
         * <li>
         * <p>
         * DynamoDB tables and global secondary indexes
         * </p>
         * </li>
         * <li>
         * <p>
         * Amazon Keyspaces tables
         * </p>
         * </li>
         * <li>
         * <p>
         * Lambda provisioned concurrency
         * </p>
         * </li>
         * <li>
         * <p>
         * Amazon MSK broker storage
         * </p>
         * </li>
         * </ul>
         * 
         * @param scaleInCooldown
         *        The amount of time, in seconds, after a scale-in activity completes before another scale-in activity
         *        can start.</p>
         *        <p>
         *        With the <i>scale-in cooldown period</i>, the intention is to scale in conservatively to protect your
         *        application’s availability, so scale-in activities are blocked until the cooldown period has expired.
         *        However, if another alarm triggers a scale-out activity during the scale-in cooldown period,
         *        Application Auto Scaling scales out the target immediately. In this case, the scale-in cooldown period
         *        stops and doesn't complete.
         *        </p>
         *        <p>
         *        Application Auto Scaling provides a default value of 600 for Amazon ElastiCache replication groups and
         *        a default value of 300 for the following scalable targets:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        AppStream 2.0 fleets
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Aurora DB clusters
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        ECS services
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        EMR clusters
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Neptune clusters
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        SageMaker endpoint variants
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Spot Fleets
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Custom resources
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For all other scalable targets, the default value is 0:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Amazon Comprehend document classification and entity recognizer endpoints
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        DynamoDB tables and global secondary indexes
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Amazon Keyspaces tables
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Lambda provisioned concurrency
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Amazon MSK broker storage
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scaleInCooldown(Integer scaleInCooldown);

        /**
         * <p>
         * Indicates whether scale in by the target tracking scaling policy is disabled. If the value is
         * <code>true</code>, scale in is disabled and the target tracking scaling policy won't remove capacity from the
         * scalable target. Otherwise, scale in is enabled and the target tracking scaling policy can remove capacity
         * from the scalable target. The default value is <code>false</code>.
         * </p>
         * 
         * @param disableScaleIn
         *        Indicates whether scale in by the target tracking scaling policy is disabled. If the value is
         *        <code>true</code>, scale in is disabled and the target tracking scaling policy won't remove capacity
         *        from the scalable target. Otherwise, scale in is enabled and the target tracking scaling policy can
         *        remove capacity from the scalable target. The default value is <code>false</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder disableScaleIn(Boolean disableScaleIn);
    }

    static final class BuilderImpl implements Builder {
        private Double targetValue;

        private PredefinedMetricSpecification predefinedMetricSpecification;

        private CustomizedMetricSpecification customizedMetricSpecification;

        private Integer scaleOutCooldown;

        private Integer scaleInCooldown;

        private Boolean disableScaleIn;

        private BuilderImpl() {
        }

        private BuilderImpl(TargetTrackingScalingPolicyConfiguration model) {
            targetValue(model.targetValue);
            predefinedMetricSpecification(model.predefinedMetricSpecification);
            customizedMetricSpecification(model.customizedMetricSpecification);
            scaleOutCooldown(model.scaleOutCooldown);
            scaleInCooldown(model.scaleInCooldown);
            disableScaleIn(model.disableScaleIn);
        }

        public final Double getTargetValue() {
            return targetValue;
        }

        public final void setTargetValue(Double targetValue) {
            this.targetValue = targetValue;
        }

        @Override
        public final Builder targetValue(Double targetValue) {
            this.targetValue = targetValue;
            return this;
        }

        public final PredefinedMetricSpecification.Builder getPredefinedMetricSpecification() {
            return predefinedMetricSpecification != null ? predefinedMetricSpecification.toBuilder() : null;
        }

        public final void setPredefinedMetricSpecification(PredefinedMetricSpecification.BuilderImpl predefinedMetricSpecification) {
            this.predefinedMetricSpecification = predefinedMetricSpecification != null ? predefinedMetricSpecification.build()
                    : null;
        }

        @Override
        public final Builder predefinedMetricSpecification(PredefinedMetricSpecification predefinedMetricSpecification) {
            this.predefinedMetricSpecification = predefinedMetricSpecification;
            return this;
        }

        public final CustomizedMetricSpecification.Builder getCustomizedMetricSpecification() {
            return customizedMetricSpecification != null ? customizedMetricSpecification.toBuilder() : null;
        }

        public final void setCustomizedMetricSpecification(CustomizedMetricSpecification.BuilderImpl customizedMetricSpecification) {
            this.customizedMetricSpecification = customizedMetricSpecification != null ? customizedMetricSpecification.build()
                    : null;
        }

        @Override
        public final Builder customizedMetricSpecification(CustomizedMetricSpecification customizedMetricSpecification) {
            this.customizedMetricSpecification = customizedMetricSpecification;
            return this;
        }

        public final Integer getScaleOutCooldown() {
            return scaleOutCooldown;
        }

        public final void setScaleOutCooldown(Integer scaleOutCooldown) {
            this.scaleOutCooldown = scaleOutCooldown;
        }

        @Override
        public final Builder scaleOutCooldown(Integer scaleOutCooldown) {
            this.scaleOutCooldown = scaleOutCooldown;
            return this;
        }

        public final Integer getScaleInCooldown() {
            return scaleInCooldown;
        }

        public final void setScaleInCooldown(Integer scaleInCooldown) {
            this.scaleInCooldown = scaleInCooldown;
        }

        @Override
        public final Builder scaleInCooldown(Integer scaleInCooldown) {
            this.scaleInCooldown = scaleInCooldown;
            return this;
        }

        public final Boolean getDisableScaleIn() {
            return disableScaleIn;
        }

        public final void setDisableScaleIn(Boolean disableScaleIn) {
            this.disableScaleIn = disableScaleIn;
        }

        @Override
        public final Builder disableScaleIn(Boolean disableScaleIn) {
            this.disableScaleIn = disableScaleIn;
            return this;
        }

        @Override
        public TargetTrackingScalingPolicyConfiguration build() {
            return new TargetTrackingScalingPolicyConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
