/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.arczonalshift;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.arczonalshift.internal.ArcZonalShiftServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.arczonalshift.model.AccessDeniedException;
import software.amazon.awssdk.services.arczonalshift.model.ArcZonalShiftException;
import software.amazon.awssdk.services.arczonalshift.model.CancelZonalShiftRequest;
import software.amazon.awssdk.services.arczonalshift.model.CancelZonalShiftResponse;
import software.amazon.awssdk.services.arczonalshift.model.ConflictException;
import software.amazon.awssdk.services.arczonalshift.model.CreatePracticeRunConfigurationRequest;
import software.amazon.awssdk.services.arczonalshift.model.CreatePracticeRunConfigurationResponse;
import software.amazon.awssdk.services.arczonalshift.model.DeletePracticeRunConfigurationRequest;
import software.amazon.awssdk.services.arczonalshift.model.DeletePracticeRunConfigurationResponse;
import software.amazon.awssdk.services.arczonalshift.model.GetManagedResourceRequest;
import software.amazon.awssdk.services.arczonalshift.model.GetManagedResourceResponse;
import software.amazon.awssdk.services.arczonalshift.model.InternalServerException;
import software.amazon.awssdk.services.arczonalshift.model.ListAutoshiftsRequest;
import software.amazon.awssdk.services.arczonalshift.model.ListAutoshiftsResponse;
import software.amazon.awssdk.services.arczonalshift.model.ListManagedResourcesRequest;
import software.amazon.awssdk.services.arczonalshift.model.ListManagedResourcesResponse;
import software.amazon.awssdk.services.arczonalshift.model.ListZonalShiftsRequest;
import software.amazon.awssdk.services.arczonalshift.model.ListZonalShiftsResponse;
import software.amazon.awssdk.services.arczonalshift.model.ResourceNotFoundException;
import software.amazon.awssdk.services.arczonalshift.model.StartZonalShiftRequest;
import software.amazon.awssdk.services.arczonalshift.model.StartZonalShiftResponse;
import software.amazon.awssdk.services.arczonalshift.model.ThrottlingException;
import software.amazon.awssdk.services.arczonalshift.model.UpdatePracticeRunConfigurationRequest;
import software.amazon.awssdk.services.arczonalshift.model.UpdatePracticeRunConfigurationResponse;
import software.amazon.awssdk.services.arczonalshift.model.UpdateZonalAutoshiftConfigurationRequest;
import software.amazon.awssdk.services.arczonalshift.model.UpdateZonalAutoshiftConfigurationResponse;
import software.amazon.awssdk.services.arczonalshift.model.UpdateZonalShiftRequest;
import software.amazon.awssdk.services.arczonalshift.model.UpdateZonalShiftResponse;
import software.amazon.awssdk.services.arczonalshift.model.ValidationException;
import software.amazon.awssdk.services.arczonalshift.transform.CancelZonalShiftRequestMarshaller;
import software.amazon.awssdk.services.arczonalshift.transform.CreatePracticeRunConfigurationRequestMarshaller;
import software.amazon.awssdk.services.arczonalshift.transform.DeletePracticeRunConfigurationRequestMarshaller;
import software.amazon.awssdk.services.arczonalshift.transform.GetManagedResourceRequestMarshaller;
import software.amazon.awssdk.services.arczonalshift.transform.ListAutoshiftsRequestMarshaller;
import software.amazon.awssdk.services.arczonalshift.transform.ListManagedResourcesRequestMarshaller;
import software.amazon.awssdk.services.arczonalshift.transform.ListZonalShiftsRequestMarshaller;
import software.amazon.awssdk.services.arczonalshift.transform.StartZonalShiftRequestMarshaller;
import software.amazon.awssdk.services.arczonalshift.transform.UpdatePracticeRunConfigurationRequestMarshaller;
import software.amazon.awssdk.services.arczonalshift.transform.UpdateZonalAutoshiftConfigurationRequestMarshaller;
import software.amazon.awssdk.services.arczonalshift.transform.UpdateZonalShiftRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link ArcZonalShiftClient}.
 *
 * @see ArcZonalShiftClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultArcZonalShiftClient implements ArcZonalShiftClient {
    private static final Logger log = Logger.loggerFor(DefaultArcZonalShiftClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultArcZonalShiftClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Cancel a zonal shift in Amazon Route 53 Application Recovery Controller. To cancel the zonal shift, specify the
     * zonal shift ID.
     * </p>
     * <p>
     * A zonal shift can be one that you've started for a resource in your Amazon Web Services account in an Amazon Web
     * Services Region, or it can be a zonal shift started by a practice run with zonal autoshift.
     * </p>
     *
     * @param cancelZonalShiftRequest
     * @return Result of the CancelZonalShift operation returned by the service.
     * @throws InternalServerException
     *         There was an internal server error.
     * @throws ConflictException
     *         The request could not be processed because of conflict in the current state of the resource.
     * @throws ResourceNotFoundException
     *         The input requested a resource that was not found.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcZonalShiftException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcZonalShiftClient.CancelZonalShift
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-zonal-shift-2022-10-30/CancelZonalShift"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CancelZonalShiftResponse cancelZonalShift(CancelZonalShiftRequest cancelZonalShiftRequest)
            throws InternalServerException, ConflictException, ResourceNotFoundException, ThrottlingException,
            ValidationException, AccessDeniedException, AwsServiceException, SdkClientException, ArcZonalShiftException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CancelZonalShiftResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CancelZonalShiftResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(cancelZonalShiftRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, cancelZonalShiftRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ARC Zonal Shift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CancelZonalShift");

            return clientHandler.execute(new ClientExecutionParams<CancelZonalShiftRequest, CancelZonalShiftResponse>()
                    .withOperationName("CancelZonalShift").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(cancelZonalShiftRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CancelZonalShiftRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * A practice run configuration for zonal autoshift is required when you enable zonal autoshift. A practice run
     * configuration includes specifications for blocked dates and blocked time windows, and for Amazon CloudWatch
     * alarms that you create to use with practice runs. The alarms that you specify are an <i>outcome alarm</i>, to
     * monitor application health during practice runs and, optionally, a <i>blocking alarm</i>, to block practice runs
     * from starting.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/r53recovery/latest/dg/arc-zonal-autoshift.considerations.html"> Considerations
     * when you configure zonal autoshift</a> in the Amazon Route 53 Application Recovery Controller Developer Guide.
     * </p>
     *
     * @param createPracticeRunConfigurationRequest
     * @return Result of the CreatePracticeRunConfiguration operation returned by the service.
     * @throws InternalServerException
     *         There was an internal server error.
     * @throws ConflictException
     *         The request could not be processed because of conflict in the current state of the resource.
     * @throws ResourceNotFoundException
     *         The input requested a resource that was not found.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcZonalShiftException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcZonalShiftClient.CreatePracticeRunConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-zonal-shift-2022-10-30/CreatePracticeRunConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreatePracticeRunConfigurationResponse createPracticeRunConfiguration(
            CreatePracticeRunConfigurationRequest createPracticeRunConfigurationRequest) throws InternalServerException,
            ConflictException, ResourceNotFoundException, ThrottlingException, ValidationException, AccessDeniedException,
            AwsServiceException, SdkClientException, ArcZonalShiftException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreatePracticeRunConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreatePracticeRunConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createPracticeRunConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                createPracticeRunConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ARC Zonal Shift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreatePracticeRunConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<CreatePracticeRunConfigurationRequest, CreatePracticeRunConfigurationResponse>()
                            .withOperationName("CreatePracticeRunConfiguration").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(createPracticeRunConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreatePracticeRunConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the practice run configuration for a resource. Before you can delete a practice run configuration for a
     * resource., you must disable zonal autoshift for the resource. Practice runs must be configured for zonal
     * autoshift to be enabled.
     * </p>
     *
     * @param deletePracticeRunConfigurationRequest
     * @return Result of the DeletePracticeRunConfiguration operation returned by the service.
     * @throws InternalServerException
     *         There was an internal server error.
     * @throws ConflictException
     *         The request could not be processed because of conflict in the current state of the resource.
     * @throws ResourceNotFoundException
     *         The input requested a resource that was not found.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcZonalShiftException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcZonalShiftClient.DeletePracticeRunConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-zonal-shift-2022-10-30/DeletePracticeRunConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeletePracticeRunConfigurationResponse deletePracticeRunConfiguration(
            DeletePracticeRunConfigurationRequest deletePracticeRunConfigurationRequest) throws InternalServerException,
            ConflictException, ResourceNotFoundException, ThrottlingException, ValidationException, AccessDeniedException,
            AwsServiceException, SdkClientException, ArcZonalShiftException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeletePracticeRunConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeletePracticeRunConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deletePracticeRunConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deletePracticeRunConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ARC Zonal Shift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeletePracticeRunConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<DeletePracticeRunConfigurationRequest, DeletePracticeRunConfigurationResponse>()
                            .withOperationName("DeletePracticeRunConfiguration").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(deletePracticeRunConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeletePracticeRunConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Get information about a resource that's been registered for zonal shifts with Amazon Route 53 Application
     * Recovery Controller in this Amazon Web Services Region. Resources that are registered for zonal shifts are
     * managed resources in Route 53 ARC. You can start zonal shifts and configure zonal autoshift for managed
     * resources.
     * </p>
     * <p>
     * At this time, you can only start a zonal shift or configure zonal autoshift for Network Load Balancers and
     * Application Load Balancers with cross-zone load balancing turned off.
     * </p>
     *
     * @param getManagedResourceRequest
     * @return Result of the GetManagedResource operation returned by the service.
     * @throws InternalServerException
     *         There was an internal server error.
     * @throws ResourceNotFoundException
     *         The input requested a resource that was not found.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcZonalShiftException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcZonalShiftClient.GetManagedResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-zonal-shift-2022-10-30/GetManagedResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetManagedResourceResponse getManagedResource(GetManagedResourceRequest getManagedResourceRequest)
            throws InternalServerException, ResourceNotFoundException, ThrottlingException, ValidationException,
            AccessDeniedException, AwsServiceException, SdkClientException, ArcZonalShiftException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetManagedResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetManagedResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getManagedResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getManagedResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ARC Zonal Shift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetManagedResource");

            return clientHandler.execute(new ClientExecutionParams<GetManagedResourceRequest, GetManagedResourceResponse>()
                    .withOperationName("GetManagedResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getManagedResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetManagedResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the active autoshifts for a specified resource.
     * </p>
     *
     * @param listAutoshiftsRequest
     * @return Result of the ListAutoshifts operation returned by the service.
     * @throws InternalServerException
     *         There was an internal server error.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcZonalShiftException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcZonalShiftClient.ListAutoshifts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-zonal-shift-2022-10-30/ListAutoshifts"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListAutoshiftsResponse listAutoshifts(ListAutoshiftsRequest listAutoshiftsRequest) throws InternalServerException,
            ThrottlingException, ValidationException, AccessDeniedException, AwsServiceException, SdkClientException,
            ArcZonalShiftException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListAutoshiftsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListAutoshiftsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listAutoshiftsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAutoshiftsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ARC Zonal Shift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAutoshifts");

            return clientHandler.execute(new ClientExecutionParams<ListAutoshiftsRequest, ListAutoshiftsResponse>()
                    .withOperationName("ListAutoshifts").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listAutoshiftsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListAutoshiftsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all the resources in your Amazon Web Services account in this Amazon Web Services Region that are managed
     * for zonal shifts in Amazon Route 53 Application Recovery Controller, and information about them. The information
     * includes the zonal autoshift status for the resource, as well as the Amazon Resource Name (ARN), the Availability
     * Zones that each resource is deployed in, and the resource name.
     * </p>
     *
     * @param listManagedResourcesRequest
     * @return Result of the ListManagedResources operation returned by the service.
     * @throws InternalServerException
     *         There was an internal server error.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcZonalShiftException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcZonalShiftClient.ListManagedResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-zonal-shift-2022-10-30/ListManagedResources"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListManagedResourcesResponse listManagedResources(ListManagedResourcesRequest listManagedResourcesRequest)
            throws InternalServerException, ThrottlingException, ValidationException, AccessDeniedException, AwsServiceException,
            SdkClientException, ArcZonalShiftException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListManagedResourcesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListManagedResourcesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listManagedResourcesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listManagedResourcesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ARC Zonal Shift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListManagedResources");

            return clientHandler.execute(new ClientExecutionParams<ListManagedResourcesRequest, ListManagedResourcesResponse>()
                    .withOperationName("ListManagedResources").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listManagedResourcesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListManagedResourcesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all active and completed zonal shifts in Amazon Route 53 Application Recovery Controller in your Amazon Web
     * Services account in this Amazon Web Services Region. <code>ListZonalShifts</code> returns customer-started zonal
     * shifts, as well as practice run zonal shifts that Route 53 ARC started on your behalf for zonal autoshift.
     * </p>
     * <p>
     * The <code>ListZonalShifts</code> operation does not list autoshifts. For more information about listing
     * autoshifts, see <a
     * href="https://docs.aws.amazon.com/arc-zonal-shift/latest/api/API_ListAutoshifts.html">"&gt;ListAutoshifts</a>.
     * </p>
     *
     * @param listZonalShiftsRequest
     * @return Result of the ListZonalShifts operation returned by the service.
     * @throws InternalServerException
     *         There was an internal server error.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcZonalShiftException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcZonalShiftClient.ListZonalShifts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-zonal-shift-2022-10-30/ListZonalShifts"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListZonalShiftsResponse listZonalShifts(ListZonalShiftsRequest listZonalShiftsRequest) throws InternalServerException,
            ThrottlingException, ValidationException, AccessDeniedException, AwsServiceException, SdkClientException,
            ArcZonalShiftException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListZonalShiftsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListZonalShiftsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listZonalShiftsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listZonalShiftsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ARC Zonal Shift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListZonalShifts");

            return clientHandler.execute(new ClientExecutionParams<ListZonalShiftsRequest, ListZonalShiftsResponse>()
                    .withOperationName("ListZonalShifts").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listZonalShiftsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListZonalShiftsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * You start a zonal shift to temporarily move load balancer traffic away from an Availability Zone in an Amazon Web
     * Services Region, to help your application recover immediately, for example, from a developer's bad code
     * deployment or from an Amazon Web Services infrastructure failure in a single Availability Zone. You can start a
     * zonal shift in Route 53 ARC only for managed resources in your Amazon Web Services account in an Amazon Web
     * Services Region. Resources are automatically registered with Route 53 ARC by Amazon Web Services services.
     * </p>
     * <p>
     * At this time, you can only start a zonal shift for Network Load Balancers and Application Load Balancers with
     * cross-zone load balancing turned off.
     * </p>
     * <p>
     * When you start a zonal shift, traffic for the resource is no longer routed to the Availability Zone. The zonal
     * shift is created immediately in Route 53 ARC. However, it can take a short time, typically up to a few minutes,
     * for existing, in-progress connections in the Availability Zone to complete.
     * </p>
     * <p>
     * For more information, see <a href="https://docs.aws.amazon.com/r53recovery/latest/dg/arc-zonal-shift.html">Zonal
     * shift</a> in the Amazon Route 53 Application Recovery Controller Developer Guide.
     * </p>
     *
     * @param startZonalShiftRequest
     * @return Result of the StartZonalShift operation returned by the service.
     * @throws InternalServerException
     *         There was an internal server error.
     * @throws ConflictException
     *         The request could not be processed because of conflict in the current state of the resource.
     * @throws ResourceNotFoundException
     *         The input requested a resource that was not found.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcZonalShiftException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcZonalShiftClient.StartZonalShift
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-zonal-shift-2022-10-30/StartZonalShift"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StartZonalShiftResponse startZonalShift(StartZonalShiftRequest startZonalShiftRequest) throws InternalServerException,
            ConflictException, ResourceNotFoundException, ThrottlingException, ValidationException, AccessDeniedException,
            AwsServiceException, SdkClientException, ArcZonalShiftException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartZonalShiftResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StartZonalShiftResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startZonalShiftRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startZonalShiftRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ARC Zonal Shift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartZonalShift");

            return clientHandler.execute(new ClientExecutionParams<StartZonalShiftRequest, StartZonalShiftResponse>()
                    .withOperationName("StartZonalShift").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(startZonalShiftRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartZonalShiftRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Update a practice run configuration to change one or more of the following: add, change, or remove the blocking
     * alarm; change the outcome alarm; or add, change, or remove blocking dates or time windows.
     * </p>
     *
     * @param updatePracticeRunConfigurationRequest
     * @return Result of the UpdatePracticeRunConfiguration operation returned by the service.
     * @throws InternalServerException
     *         There was an internal server error.
     * @throws ConflictException
     *         The request could not be processed because of conflict in the current state of the resource.
     * @throws ResourceNotFoundException
     *         The input requested a resource that was not found.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcZonalShiftException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcZonalShiftClient.UpdatePracticeRunConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-zonal-shift-2022-10-30/UpdatePracticeRunConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdatePracticeRunConfigurationResponse updatePracticeRunConfiguration(
            UpdatePracticeRunConfigurationRequest updatePracticeRunConfigurationRequest) throws InternalServerException,
            ConflictException, ResourceNotFoundException, ThrottlingException, ValidationException, AccessDeniedException,
            AwsServiceException, SdkClientException, ArcZonalShiftException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdatePracticeRunConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdatePracticeRunConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updatePracticeRunConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updatePracticeRunConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ARC Zonal Shift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdatePracticeRunConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdatePracticeRunConfigurationRequest, UpdatePracticeRunConfigurationResponse>()
                            .withOperationName("UpdatePracticeRunConfiguration").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(updatePracticeRunConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdatePracticeRunConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * You can update the zonal autoshift status for a resource, to enable or disable zonal autoshift. When zonal
     * autoshift is <code>ENABLED</code>, Amazon Web Services shifts away resource traffic from an Availability Zone, on
     * your behalf, when Amazon Web Services determines that there's an issue in the Availability Zone that could
     * potentially affect customers.
     * </p>
     *
     * @param updateZonalAutoshiftConfigurationRequest
     * @return Result of the UpdateZonalAutoshiftConfiguration operation returned by the service.
     * @throws InternalServerException
     *         There was an internal server error.
     * @throws ConflictException
     *         The request could not be processed because of conflict in the current state of the resource.
     * @throws ResourceNotFoundException
     *         The input requested a resource that was not found.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcZonalShiftException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcZonalShiftClient.UpdateZonalAutoshiftConfiguration
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/arc-zonal-shift-2022-10-30/UpdateZonalAutoshiftConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateZonalAutoshiftConfigurationResponse updateZonalAutoshiftConfiguration(
            UpdateZonalAutoshiftConfigurationRequest updateZonalAutoshiftConfigurationRequest) throws InternalServerException,
            ConflictException, ResourceNotFoundException, ThrottlingException, ValidationException, AccessDeniedException,
            AwsServiceException, SdkClientException, ArcZonalShiftException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateZonalAutoshiftConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateZonalAutoshiftConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateZonalAutoshiftConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateZonalAutoshiftConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ARC Zonal Shift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateZonalAutoshiftConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateZonalAutoshiftConfigurationRequest, UpdateZonalAutoshiftConfigurationResponse>()
                            .withOperationName("UpdateZonalAutoshiftConfiguration").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(updateZonalAutoshiftConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateZonalAutoshiftConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Update an active zonal shift in Amazon Route 53 Application Recovery Controller in your Amazon Web Services
     * account. You can update a zonal shift to set a new expiration, or edit or replace the comment for the zonal
     * shift.
     * </p>
     *
     * @param updateZonalShiftRequest
     * @return Result of the UpdateZonalShift operation returned by the service.
     * @throws InternalServerException
     *         There was an internal server error.
     * @throws ConflictException
     *         The request could not be processed because of conflict in the current state of the resource.
     * @throws ResourceNotFoundException
     *         The input requested a resource that was not found.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcZonalShiftException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcZonalShiftClient.UpdateZonalShift
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-zonal-shift-2022-10-30/UpdateZonalShift"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateZonalShiftResponse updateZonalShift(UpdateZonalShiftRequest updateZonalShiftRequest)
            throws InternalServerException, ConflictException, ResourceNotFoundException, ThrottlingException,
            ValidationException, AccessDeniedException, AwsServiceException, SdkClientException, ArcZonalShiftException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateZonalShiftResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateZonalShiftResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateZonalShiftRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateZonalShiftRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ARC Zonal Shift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateZonalShift");

            return clientHandler.execute(new ClientExecutionParams<UpdateZonalShiftRequest, UpdateZonalShiftResponse>()
                    .withOperationName("UpdateZonalShift").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateZonalShiftRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateZonalShiftRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        ArcZonalShiftServiceClientConfigurationBuilder serviceConfigBuilder = new ArcZonalShiftServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(ArcZonalShiftException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build());
    }

    @Override
    public final ArcZonalShiftServiceClientConfiguration serviceClientConfiguration() {
        return new ArcZonalShiftServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
