/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.arczonalshift.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about an autoshift. Amazon Web Services starts an autoshift to temporarily move traffic for a resource
 * away from an Availability Zone in an Amazon Web Services Region when Amazon Web Services determines that there's an
 * issue in the Availability Zone that could potentially affect customers. You can configure zonal autoshift in Route 53
 * ARC for managed resources in your Amazon Web Services account in a Region. Supported Amazon Web Services resources
 * are automatically registered with Route 53 ARC.
 * </p>
 * <p>
 * Autoshifts are temporary. When the Availability Zone recovers, Amazon Web Services ends the autoshift, and traffic
 * for the resource is no longer directed to the other Availability Zones in the Region.
 * </p>
 * <p>
 * You can stop an autoshift for a resource by disabling zonal autoshift.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AutoshiftSummary implements SdkPojo, Serializable,
        ToCopyableBuilder<AutoshiftSummary.Builder, AutoshiftSummary> {
    private static final SdkField<String> AWAY_FROM_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("awayFrom").getter(getter(AutoshiftSummary::awayFrom)).setter(setter(Builder::awayFrom))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("awayFrom").build()).build();

    private static final SdkField<Instant> END_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("endTime").getter(getter(AutoshiftSummary::endTime)).setter(setter(Builder::endTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("endTime").build()).build();

    private static final SdkField<Instant> START_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("startTime").getter(getter(AutoshiftSummary::startTime)).setter(setter(Builder::startTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("startTime").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("status")
            .getter(getter(AutoshiftSummary::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(AWAY_FROM_FIELD,
            END_TIME_FIELD, START_TIME_FIELD, STATUS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String awayFrom;

    private final Instant endTime;

    private final Instant startTime;

    private final String status;

    private AutoshiftSummary(BuilderImpl builder) {
        this.awayFrom = builder.awayFrom;
        this.endTime = builder.endTime;
        this.startTime = builder.startTime;
        this.status = builder.status;
    }

    /**
     * <p>
     * The Availability Zone that traffic is shifted away from for a resource when Amazon Web Services starts an
     * autoshift. Until the autoshift ends, traffic for the resource is instead directed to other Availability Zones in
     * the Amazon Web Services Region. An autoshift can end for a resource, for example, when Amazon Web Services ends
     * the autoshift for the Availability Zone or when you disable zonal autoshift for the resource.
     * </p>
     * 
     * @return The Availability Zone that traffic is shifted away from for a resource when Amazon Web Services starts an
     *         autoshift. Until the autoshift ends, traffic for the resource is instead directed to other Availability
     *         Zones in the Amazon Web Services Region. An autoshift can end for a resource, for example, when Amazon
     *         Web Services ends the autoshift for the Availability Zone or when you disable zonal autoshift for the
     *         resource.
     */
    public final String awayFrom() {
        return awayFrom;
    }

    /**
     * <p>
     * The time (in UTC) when the autoshift ended.
     * </p>
     * 
     * @return The time (in UTC) when the autoshift ended.
     */
    public final Instant endTime() {
        return endTime;
    }

    /**
     * <p>
     * The time (in UTC) when the autoshift started.
     * </p>
     * 
     * @return The time (in UTC) when the autoshift started.
     */
    public final Instant startTime() {
        return startTime;
    }

    /**
     * <p>
     * The status for an autoshift.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link AutoshiftExecutionStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #statusAsString}.
     * </p>
     * 
     * @return The status for an autoshift.
     * @see AutoshiftExecutionStatus
     */
    public final AutoshiftExecutionStatus status() {
        return AutoshiftExecutionStatus.fromValue(status);
    }

    /**
     * <p>
     * The status for an autoshift.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link AutoshiftExecutionStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #statusAsString}.
     * </p>
     * 
     * @return The status for an autoshift.
     * @see AutoshiftExecutionStatus
     */
    public final String statusAsString() {
        return status;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(awayFrom());
        hashCode = 31 * hashCode + Objects.hashCode(endTime());
        hashCode = 31 * hashCode + Objects.hashCode(startTime());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AutoshiftSummary)) {
            return false;
        }
        AutoshiftSummary other = (AutoshiftSummary) obj;
        return Objects.equals(awayFrom(), other.awayFrom()) && Objects.equals(endTime(), other.endTime())
                && Objects.equals(startTime(), other.startTime()) && Objects.equals(statusAsString(), other.statusAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AutoshiftSummary").add("AwayFrom", awayFrom()).add("EndTime", endTime())
                .add("StartTime", startTime()).add("Status", statusAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "awayFrom":
            return Optional.ofNullable(clazz.cast(awayFrom()));
        case "endTime":
            return Optional.ofNullable(clazz.cast(endTime()));
        case "startTime":
            return Optional.ofNullable(clazz.cast(startTime()));
        case "status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<AutoshiftSummary, T> g) {
        return obj -> g.apply((AutoshiftSummary) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AutoshiftSummary> {
        /**
         * <p>
         * The Availability Zone that traffic is shifted away from for a resource when Amazon Web Services starts an
         * autoshift. Until the autoshift ends, traffic for the resource is instead directed to other Availability Zones
         * in the Amazon Web Services Region. An autoshift can end for a resource, for example, when Amazon Web Services
         * ends the autoshift for the Availability Zone or when you disable zonal autoshift for the resource.
         * </p>
         * 
         * @param awayFrom
         *        The Availability Zone that traffic is shifted away from for a resource when Amazon Web Services starts
         *        an autoshift. Until the autoshift ends, traffic for the resource is instead directed to other
         *        Availability Zones in the Amazon Web Services Region. An autoshift can end for a resource, for
         *        example, when Amazon Web Services ends the autoshift for the Availability Zone or when you disable
         *        zonal autoshift for the resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder awayFrom(String awayFrom);

        /**
         * <p>
         * The time (in UTC) when the autoshift ended.
         * </p>
         * 
         * @param endTime
         *        The time (in UTC) when the autoshift ended.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endTime(Instant endTime);

        /**
         * <p>
         * The time (in UTC) when the autoshift started.
         * </p>
         * 
         * @param startTime
         *        The time (in UTC) when the autoshift started.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startTime(Instant startTime);

        /**
         * <p>
         * The status for an autoshift.
         * </p>
         * 
         * @param status
         *        The status for an autoshift.
         * @see AutoshiftExecutionStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AutoshiftExecutionStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status for an autoshift.
         * </p>
         * 
         * @param status
         *        The status for an autoshift.
         * @see AutoshiftExecutionStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AutoshiftExecutionStatus
         */
        Builder status(AutoshiftExecutionStatus status);
    }

    static final class BuilderImpl implements Builder {
        private String awayFrom;

        private Instant endTime;

        private Instant startTime;

        private String status;

        private BuilderImpl() {
        }

        private BuilderImpl(AutoshiftSummary model) {
            awayFrom(model.awayFrom);
            endTime(model.endTime);
            startTime(model.startTime);
            status(model.status);
        }

        public final String getAwayFrom() {
            return awayFrom;
        }

        public final void setAwayFrom(String awayFrom) {
            this.awayFrom = awayFrom;
        }

        @Override
        public final Builder awayFrom(String awayFrom) {
            this.awayFrom = awayFrom;
            return this;
        }

        public final Instant getEndTime() {
            return endTime;
        }

        public final void setEndTime(Instant endTime) {
            this.endTime = endTime;
        }

        @Override
        public final Builder endTime(Instant endTime) {
            this.endTime = endTime;
            return this;
        }

        public final Instant getStartTime() {
            return startTime;
        }

        public final void setStartTime(Instant startTime) {
            this.startTime = startTime;
        }

        @Override
        public final Builder startTime(Instant startTime) {
            this.startTime = startTime;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(AutoshiftExecutionStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        @Override
        public AutoshiftSummary build() {
            return new AutoshiftSummary(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
