/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.autoscalingplans;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.autoscalingplans.internal.AutoScalingPlansServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.autoscalingplans.model.AutoScalingPlansException;
import software.amazon.awssdk.services.autoscalingplans.model.ConcurrentUpdateException;
import software.amazon.awssdk.services.autoscalingplans.model.CreateScalingPlanRequest;
import software.amazon.awssdk.services.autoscalingplans.model.CreateScalingPlanResponse;
import software.amazon.awssdk.services.autoscalingplans.model.DeleteScalingPlanRequest;
import software.amazon.awssdk.services.autoscalingplans.model.DeleteScalingPlanResponse;
import software.amazon.awssdk.services.autoscalingplans.model.DescribeScalingPlanResourcesRequest;
import software.amazon.awssdk.services.autoscalingplans.model.DescribeScalingPlanResourcesResponse;
import software.amazon.awssdk.services.autoscalingplans.model.DescribeScalingPlansRequest;
import software.amazon.awssdk.services.autoscalingplans.model.DescribeScalingPlansResponse;
import software.amazon.awssdk.services.autoscalingplans.model.GetScalingPlanResourceForecastDataRequest;
import software.amazon.awssdk.services.autoscalingplans.model.GetScalingPlanResourceForecastDataResponse;
import software.amazon.awssdk.services.autoscalingplans.model.InternalServiceException;
import software.amazon.awssdk.services.autoscalingplans.model.InvalidNextTokenException;
import software.amazon.awssdk.services.autoscalingplans.model.LimitExceededException;
import software.amazon.awssdk.services.autoscalingplans.model.ObjectNotFoundException;
import software.amazon.awssdk.services.autoscalingplans.model.UpdateScalingPlanRequest;
import software.amazon.awssdk.services.autoscalingplans.model.UpdateScalingPlanResponse;
import software.amazon.awssdk.services.autoscalingplans.model.ValidationException;
import software.amazon.awssdk.services.autoscalingplans.transform.CreateScalingPlanRequestMarshaller;
import software.amazon.awssdk.services.autoscalingplans.transform.DeleteScalingPlanRequestMarshaller;
import software.amazon.awssdk.services.autoscalingplans.transform.DescribeScalingPlanResourcesRequestMarshaller;
import software.amazon.awssdk.services.autoscalingplans.transform.DescribeScalingPlansRequestMarshaller;
import software.amazon.awssdk.services.autoscalingplans.transform.GetScalingPlanResourceForecastDataRequestMarshaller;
import software.amazon.awssdk.services.autoscalingplans.transform.UpdateScalingPlanRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link AutoScalingPlansClient}.
 *
 * @see AutoScalingPlansClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultAutoScalingPlansClient implements AutoScalingPlansClient {
    private static final Logger log = Logger.loggerFor(DefaultAutoScalingPlansClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.AWS_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultAutoScalingPlansClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates a scaling plan.
     * </p>
     *
     * @param createScalingPlanRequest
     * @return Result of the CreateScalingPlan operation returned by the service.
     * @throws ValidationException
     *         An exception was thrown for a validation issue. Review the parameters provided.
     * @throws LimitExceededException
     *         Your account exceeded a limit. This exception is thrown when a per-account resource limit is exceeded.
     * @throws ConcurrentUpdateException
     *         Concurrent updates caused an exception, for example, if you request an update to a scaling plan that
     *         already has a pending update.
     * @throws InternalServiceException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AutoScalingPlansException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AutoScalingPlansClient.CreateScalingPlan
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/autoscaling-plans-2018-01-06/CreateScalingPlan"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateScalingPlanResponse createScalingPlan(CreateScalingPlanRequest createScalingPlanRequest)
            throws ValidationException, LimitExceededException, ConcurrentUpdateException, InternalServiceException,
            AwsServiceException, SdkClientException, AutoScalingPlansException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateScalingPlanResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateScalingPlanResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createScalingPlanRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createScalingPlanRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling Plans");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateScalingPlan");

            return clientHandler.execute(new ClientExecutionParams<CreateScalingPlanRequest, CreateScalingPlanResponse>()
                    .withOperationName("CreateScalingPlan").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createScalingPlanRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateScalingPlanRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the specified scaling plan.
     * </p>
     * <p>
     * Deleting a scaling plan deletes the underlying <a>ScalingInstruction</a> for all of the scalable resources that
     * are covered by the plan.
     * </p>
     * <p>
     * If the plan has launched resources or has scaling activities in progress, you must delete those resources
     * separately.
     * </p>
     *
     * @param deleteScalingPlanRequest
     * @return Result of the DeleteScalingPlan operation returned by the service.
     * @throws ValidationException
     *         An exception was thrown for a validation issue. Review the parameters provided.
     * @throws ObjectNotFoundException
     *         The specified object could not be found.
     * @throws ConcurrentUpdateException
     *         Concurrent updates caused an exception, for example, if you request an update to a scaling plan that
     *         already has a pending update.
     * @throws InternalServiceException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AutoScalingPlansException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AutoScalingPlansClient.DeleteScalingPlan
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/autoscaling-plans-2018-01-06/DeleteScalingPlan"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteScalingPlanResponse deleteScalingPlan(DeleteScalingPlanRequest deleteScalingPlanRequest)
            throws ValidationException, ObjectNotFoundException, ConcurrentUpdateException, InternalServiceException,
            AwsServiceException, SdkClientException, AutoScalingPlansException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteScalingPlanResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteScalingPlanResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteScalingPlanRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteScalingPlanRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling Plans");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteScalingPlan");

            return clientHandler.execute(new ClientExecutionParams<DeleteScalingPlanRequest, DeleteScalingPlanResponse>()
                    .withOperationName("DeleteScalingPlan").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteScalingPlanRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteScalingPlanRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes the scalable resources in the specified scaling plan.
     * </p>
     *
     * @param describeScalingPlanResourcesRequest
     * @return Result of the DescribeScalingPlanResources operation returned by the service.
     * @throws ValidationException
     *         An exception was thrown for a validation issue. Review the parameters provided.
     * @throws InvalidNextTokenException
     *         The token provided is not valid.
     * @throws ConcurrentUpdateException
     *         Concurrent updates caused an exception, for example, if you request an update to a scaling plan that
     *         already has a pending update.
     * @throws InternalServiceException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AutoScalingPlansException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AutoScalingPlansClient.DescribeScalingPlanResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/autoscaling-plans-2018-01-06/DescribeScalingPlanResources"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeScalingPlanResourcesResponse describeScalingPlanResources(
            DescribeScalingPlanResourcesRequest describeScalingPlanResourcesRequest) throws ValidationException,
            InvalidNextTokenException, ConcurrentUpdateException, InternalServiceException, AwsServiceException,
            SdkClientException, AutoScalingPlansException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeScalingPlanResourcesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeScalingPlanResourcesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeScalingPlanResourcesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeScalingPlanResourcesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling Plans");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeScalingPlanResources");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeScalingPlanResourcesRequest, DescribeScalingPlanResourcesResponse>()
                            .withOperationName("DescribeScalingPlanResources").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(describeScalingPlanResourcesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeScalingPlanResourcesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes one or more of your scaling plans.
     * </p>
     *
     * @param describeScalingPlansRequest
     * @return Result of the DescribeScalingPlans operation returned by the service.
     * @throws ValidationException
     *         An exception was thrown for a validation issue. Review the parameters provided.
     * @throws InvalidNextTokenException
     *         The token provided is not valid.
     * @throws ConcurrentUpdateException
     *         Concurrent updates caused an exception, for example, if you request an update to a scaling plan that
     *         already has a pending update.
     * @throws InternalServiceException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AutoScalingPlansException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AutoScalingPlansClient.DescribeScalingPlans
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/autoscaling-plans-2018-01-06/DescribeScalingPlans"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeScalingPlansResponse describeScalingPlans(DescribeScalingPlansRequest describeScalingPlansRequest)
            throws ValidationException, InvalidNextTokenException, ConcurrentUpdateException, InternalServiceException,
            AwsServiceException, SdkClientException, AutoScalingPlansException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeScalingPlansResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeScalingPlansResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeScalingPlansRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeScalingPlansRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling Plans");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeScalingPlans");

            return clientHandler.execute(new ClientExecutionParams<DescribeScalingPlansRequest, DescribeScalingPlansResponse>()
                    .withOperationName("DescribeScalingPlans").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describeScalingPlansRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeScalingPlansRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the forecast data for a scalable resource.
     * </p>
     * <p>
     * Capacity forecasts are represented as predicted values, or data points, that are calculated using historical data
     * points from a specified CloudWatch load metric. Data points are available for up to 56 days.
     * </p>
     *
     * @param getScalingPlanResourceForecastDataRequest
     * @return Result of the GetScalingPlanResourceForecastData operation returned by the service.
     * @throws ValidationException
     *         An exception was thrown for a validation issue. Review the parameters provided.
     * @throws InternalServiceException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AutoScalingPlansException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AutoScalingPlansClient.GetScalingPlanResourceForecastData
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/autoscaling-plans-2018-01-06/GetScalingPlanResourceForecastData"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetScalingPlanResourceForecastDataResponse getScalingPlanResourceForecastData(
            GetScalingPlanResourceForecastDataRequest getScalingPlanResourceForecastDataRequest) throws ValidationException,
            InternalServiceException, AwsServiceException, SdkClientException, AutoScalingPlansException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetScalingPlanResourceForecastDataResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetScalingPlanResourceForecastDataResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getScalingPlanResourceForecastDataRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getScalingPlanResourceForecastDataRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling Plans");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetScalingPlanResourceForecastData");

            return clientHandler
                    .execute(new ClientExecutionParams<GetScalingPlanResourceForecastDataRequest, GetScalingPlanResourceForecastDataResponse>()
                            .withOperationName("GetScalingPlanResourceForecastData").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(getScalingPlanResourceForecastDataRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetScalingPlanResourceForecastDataRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the specified scaling plan.
     * </p>
     * <p>
     * You cannot update a scaling plan if it is in the process of being created, updated, or deleted.
     * </p>
     *
     * @param updateScalingPlanRequest
     * @return Result of the UpdateScalingPlan operation returned by the service.
     * @throws ValidationException
     *         An exception was thrown for a validation issue. Review the parameters provided.
     * @throws ConcurrentUpdateException
     *         Concurrent updates caused an exception, for example, if you request an update to a scaling plan that
     *         already has a pending update.
     * @throws InternalServiceException
     *         The service encountered an internal error.
     * @throws ObjectNotFoundException
     *         The specified object could not be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AutoScalingPlansException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AutoScalingPlansClient.UpdateScalingPlan
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/autoscaling-plans-2018-01-06/UpdateScalingPlan"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateScalingPlanResponse updateScalingPlan(UpdateScalingPlanRequest updateScalingPlanRequest)
            throws ValidationException, ConcurrentUpdateException, InternalServiceException, ObjectNotFoundException,
            AwsServiceException, SdkClientException, AutoScalingPlansException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateScalingPlanResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateScalingPlanResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateScalingPlanRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateScalingPlanRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Auto Scaling Plans");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateScalingPlan");

            return clientHandler.execute(new ClientExecutionParams<UpdateScalingPlanRequest, UpdateScalingPlanResponse>()
                    .withOperationName("UpdateScalingPlan").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateScalingPlanRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateScalingPlanRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        AutoScalingPlansServiceClientConfigurationBuilder serviceConfigBuilder = new AutoScalingPlansServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(AutoScalingPlansException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServiceException")
                                .exceptionBuilderSupplier(InternalServiceException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidNextTokenException")
                                .exceptionBuilderSupplier(InvalidNextTokenException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ObjectNotFoundException")
                                .exceptionBuilderSupplier(ObjectNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConcurrentUpdateException")
                                .exceptionBuilderSupplier(ConcurrentUpdateException::builder).httpStatusCode(400).build());
    }

    @Override
    public final AutoScalingPlansServiceClientConfiguration serviceClientConfiguration() {
        return new AutoScalingPlansServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
