/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.b2bi;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.b2bi.internal.B2BiServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.b2bi.model.AccessDeniedException;
import software.amazon.awssdk.services.b2bi.model.B2BiException;
import software.amazon.awssdk.services.b2bi.model.ConflictException;
import software.amazon.awssdk.services.b2bi.model.CreateCapabilityRequest;
import software.amazon.awssdk.services.b2bi.model.CreateCapabilityResponse;
import software.amazon.awssdk.services.b2bi.model.CreatePartnershipRequest;
import software.amazon.awssdk.services.b2bi.model.CreatePartnershipResponse;
import software.amazon.awssdk.services.b2bi.model.CreateProfileRequest;
import software.amazon.awssdk.services.b2bi.model.CreateProfileResponse;
import software.amazon.awssdk.services.b2bi.model.CreateTransformerRequest;
import software.amazon.awssdk.services.b2bi.model.CreateTransformerResponse;
import software.amazon.awssdk.services.b2bi.model.DeleteCapabilityRequest;
import software.amazon.awssdk.services.b2bi.model.DeleteCapabilityResponse;
import software.amazon.awssdk.services.b2bi.model.DeletePartnershipRequest;
import software.amazon.awssdk.services.b2bi.model.DeletePartnershipResponse;
import software.amazon.awssdk.services.b2bi.model.DeleteProfileRequest;
import software.amazon.awssdk.services.b2bi.model.DeleteProfileResponse;
import software.amazon.awssdk.services.b2bi.model.DeleteTransformerRequest;
import software.amazon.awssdk.services.b2bi.model.DeleteTransformerResponse;
import software.amazon.awssdk.services.b2bi.model.GetCapabilityRequest;
import software.amazon.awssdk.services.b2bi.model.GetCapabilityResponse;
import software.amazon.awssdk.services.b2bi.model.GetPartnershipRequest;
import software.amazon.awssdk.services.b2bi.model.GetPartnershipResponse;
import software.amazon.awssdk.services.b2bi.model.GetProfileRequest;
import software.amazon.awssdk.services.b2bi.model.GetProfileResponse;
import software.amazon.awssdk.services.b2bi.model.GetTransformerJobRequest;
import software.amazon.awssdk.services.b2bi.model.GetTransformerJobResponse;
import software.amazon.awssdk.services.b2bi.model.GetTransformerRequest;
import software.amazon.awssdk.services.b2bi.model.GetTransformerResponse;
import software.amazon.awssdk.services.b2bi.model.InternalServerException;
import software.amazon.awssdk.services.b2bi.model.ListCapabilitiesRequest;
import software.amazon.awssdk.services.b2bi.model.ListCapabilitiesResponse;
import software.amazon.awssdk.services.b2bi.model.ListPartnershipsRequest;
import software.amazon.awssdk.services.b2bi.model.ListPartnershipsResponse;
import software.amazon.awssdk.services.b2bi.model.ListProfilesRequest;
import software.amazon.awssdk.services.b2bi.model.ListProfilesResponse;
import software.amazon.awssdk.services.b2bi.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.b2bi.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.b2bi.model.ListTransformersRequest;
import software.amazon.awssdk.services.b2bi.model.ListTransformersResponse;
import software.amazon.awssdk.services.b2bi.model.ResourceNotFoundException;
import software.amazon.awssdk.services.b2bi.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.b2bi.model.StartTransformerJobRequest;
import software.amazon.awssdk.services.b2bi.model.StartTransformerJobResponse;
import software.amazon.awssdk.services.b2bi.model.TagResourceRequest;
import software.amazon.awssdk.services.b2bi.model.TagResourceResponse;
import software.amazon.awssdk.services.b2bi.model.TestMappingRequest;
import software.amazon.awssdk.services.b2bi.model.TestMappingResponse;
import software.amazon.awssdk.services.b2bi.model.TestParsingRequest;
import software.amazon.awssdk.services.b2bi.model.TestParsingResponse;
import software.amazon.awssdk.services.b2bi.model.ThrottlingException;
import software.amazon.awssdk.services.b2bi.model.UntagResourceRequest;
import software.amazon.awssdk.services.b2bi.model.UntagResourceResponse;
import software.amazon.awssdk.services.b2bi.model.UpdateCapabilityRequest;
import software.amazon.awssdk.services.b2bi.model.UpdateCapabilityResponse;
import software.amazon.awssdk.services.b2bi.model.UpdatePartnershipRequest;
import software.amazon.awssdk.services.b2bi.model.UpdatePartnershipResponse;
import software.amazon.awssdk.services.b2bi.model.UpdateProfileRequest;
import software.amazon.awssdk.services.b2bi.model.UpdateProfileResponse;
import software.amazon.awssdk.services.b2bi.model.UpdateTransformerRequest;
import software.amazon.awssdk.services.b2bi.model.UpdateTransformerResponse;
import software.amazon.awssdk.services.b2bi.model.ValidationException;
import software.amazon.awssdk.services.b2bi.transform.CreateCapabilityRequestMarshaller;
import software.amazon.awssdk.services.b2bi.transform.CreatePartnershipRequestMarshaller;
import software.amazon.awssdk.services.b2bi.transform.CreateProfileRequestMarshaller;
import software.amazon.awssdk.services.b2bi.transform.CreateTransformerRequestMarshaller;
import software.amazon.awssdk.services.b2bi.transform.DeleteCapabilityRequestMarshaller;
import software.amazon.awssdk.services.b2bi.transform.DeletePartnershipRequestMarshaller;
import software.amazon.awssdk.services.b2bi.transform.DeleteProfileRequestMarshaller;
import software.amazon.awssdk.services.b2bi.transform.DeleteTransformerRequestMarshaller;
import software.amazon.awssdk.services.b2bi.transform.GetCapabilityRequestMarshaller;
import software.amazon.awssdk.services.b2bi.transform.GetPartnershipRequestMarshaller;
import software.amazon.awssdk.services.b2bi.transform.GetProfileRequestMarshaller;
import software.amazon.awssdk.services.b2bi.transform.GetTransformerJobRequestMarshaller;
import software.amazon.awssdk.services.b2bi.transform.GetTransformerRequestMarshaller;
import software.amazon.awssdk.services.b2bi.transform.ListCapabilitiesRequestMarshaller;
import software.amazon.awssdk.services.b2bi.transform.ListPartnershipsRequestMarshaller;
import software.amazon.awssdk.services.b2bi.transform.ListProfilesRequestMarshaller;
import software.amazon.awssdk.services.b2bi.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.b2bi.transform.ListTransformersRequestMarshaller;
import software.amazon.awssdk.services.b2bi.transform.StartTransformerJobRequestMarshaller;
import software.amazon.awssdk.services.b2bi.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.b2bi.transform.TestMappingRequestMarshaller;
import software.amazon.awssdk.services.b2bi.transform.TestParsingRequestMarshaller;
import software.amazon.awssdk.services.b2bi.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.b2bi.transform.UpdateCapabilityRequestMarshaller;
import software.amazon.awssdk.services.b2bi.transform.UpdatePartnershipRequestMarshaller;
import software.amazon.awssdk.services.b2bi.transform.UpdateProfileRequestMarshaller;
import software.amazon.awssdk.services.b2bi.transform.UpdateTransformerRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link B2BiClient}.
 *
 * @see B2BiClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultB2BiClient implements B2BiClient {
    private static final Logger log = Logger.loggerFor(DefaultB2BiClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.AWS_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultB2BiClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Instantiates a capability based on the specified parameters. Capabilities contain the information necessary to
     * process incoming EDI (electronic data interchange) documents.
     * </p>
     *
     * @param createCapabilityRequest
     * @return Result of the CreateCapability operation returned by the service.
     * @throws ConflictException
     *         A conflict exception is thrown when you attempt to delete a resource (such as a profile or a capability)
     *         that is being used by other resources.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         Occurs when a B2BI object cannot be validated against a request from another object.
     * @throws ThrottlingException
     *         The request was denied due to throttling: the data speed and rendering may be limited depending on
     *         various parameters and conditions.
     * @throws ResourceNotFoundException
     *         Occurs when the requested resource does not exist, or cannot be found. In some cases, the resource exists
     *         in a region other than the region specified in the API call.
     * @throws ServiceQuotaExceededException
     *         Occurs when the calling command attempts to exceed one of the service quotas, for example trying to
     *         create a capability when you already have the maximum number of capabilities allowed.
     * @throws InternalServerException
     *         This exception is thrown when an error occurs in the Amazon Web Services B2B Data Interchange service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws B2BiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample B2BiClient.CreateCapability
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/b2bi-2022-06-23/CreateCapability" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateCapabilityResponse createCapability(CreateCapabilityRequest createCapabilityRequest) throws ConflictException,
            AccessDeniedException, ValidationException, ThrottlingException, ResourceNotFoundException,
            ServiceQuotaExceededException, InternalServerException, AwsServiceException, SdkClientException, B2BiException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateCapabilityResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateCapabilityResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createCapabilityRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createCapabilityRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "b2bi");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateCapability");

            return clientHandler.execute(new ClientExecutionParams<CreateCapabilityRequest, CreateCapabilityResponse>()
                    .withOperationName("CreateCapability").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createCapabilityRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateCapabilityRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a partnership between a customer and a trading partner, based on the supplied parameters. Partnerships
     * link trading partners with your profile and a specific transformer, so that the EDI (electronic data interchange)
     * documents that they upload to Amazon S3 can be processed according to their specifications.
     * </p>
     *
     * @param createPartnershipRequest
     * @return Result of the CreatePartnership operation returned by the service.
     * @throws ConflictException
     *         A conflict exception is thrown when you attempt to delete a resource (such as a profile or a capability)
     *         that is being used by other resources.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         Occurs when a B2BI object cannot be validated against a request from another object.
     * @throws ThrottlingException
     *         The request was denied due to throttling: the data speed and rendering may be limited depending on
     *         various parameters and conditions.
     * @throws ResourceNotFoundException
     *         Occurs when the requested resource does not exist, or cannot be found. In some cases, the resource exists
     *         in a region other than the region specified in the API call.
     * @throws ServiceQuotaExceededException
     *         Occurs when the calling command attempts to exceed one of the service quotas, for example trying to
     *         create a capability when you already have the maximum number of capabilities allowed.
     * @throws InternalServerException
     *         This exception is thrown when an error occurs in the Amazon Web Services B2B Data Interchange service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws B2BiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample B2BiClient.CreatePartnership
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/b2bi-2022-06-23/CreatePartnership" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreatePartnershipResponse createPartnership(CreatePartnershipRequest createPartnershipRequest)
            throws ConflictException, AccessDeniedException, ValidationException, ThrottlingException, ResourceNotFoundException,
            ServiceQuotaExceededException, InternalServerException, AwsServiceException, SdkClientException, B2BiException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreatePartnershipResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreatePartnershipResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createPartnershipRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createPartnershipRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "b2bi");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreatePartnership");

            return clientHandler.execute(new ClientExecutionParams<CreatePartnershipRequest, CreatePartnershipResponse>()
                    .withOperationName("CreatePartnership").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createPartnershipRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreatePartnershipRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a customer profile. You can have up to five customer profiles, each representing a distinct private
     * network. Profiles contain basic information about you and your business.
     * </p>
     *
     * @param createProfileRequest
     * @return Result of the CreateProfile operation returned by the service.
     * @throws ConflictException
     *         A conflict exception is thrown when you attempt to delete a resource (such as a profile or a capability)
     *         that is being used by other resources.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         Occurs when a B2BI object cannot be validated against a request from another object.
     * @throws ThrottlingException
     *         The request was denied due to throttling: the data speed and rendering may be limited depending on
     *         various parameters and conditions.
     * @throws ResourceNotFoundException
     *         Occurs when the requested resource does not exist, or cannot be found. In some cases, the resource exists
     *         in a region other than the region specified in the API call.
     * @throws ServiceQuotaExceededException
     *         Occurs when the calling command attempts to exceed one of the service quotas, for example trying to
     *         create a capability when you already have the maximum number of capabilities allowed.
     * @throws InternalServerException
     *         This exception is thrown when an error occurs in the Amazon Web Services B2B Data Interchange service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws B2BiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample B2BiClient.CreateProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/b2bi-2022-06-23/CreateProfile" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateProfileResponse createProfile(CreateProfileRequest createProfileRequest) throws ConflictException,
            AccessDeniedException, ValidationException, ThrottlingException, ResourceNotFoundException,
            ServiceQuotaExceededException, InternalServerException, AwsServiceException, SdkClientException, B2BiException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateProfileResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateProfileResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createProfileRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "b2bi");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateProfile");

            return clientHandler.execute(new ClientExecutionParams<CreateProfileRequest, CreateProfileResponse>()
                    .withOperationName("CreateProfile").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createProfileRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateProfileRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a transformer. Transformers describe how to process the incoming EDI (electronic data interchange)
     * documents, and extract the necessary information.
     * </p>
     *
     * @param createTransformerRequest
     * @return Result of the CreateTransformer operation returned by the service.
     * @throws ConflictException
     *         A conflict exception is thrown when you attempt to delete a resource (such as a profile or a capability)
     *         that is being used by other resources.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         Occurs when a B2BI object cannot be validated against a request from another object.
     * @throws ThrottlingException
     *         The request was denied due to throttling: the data speed and rendering may be limited depending on
     *         various parameters and conditions.
     * @throws ResourceNotFoundException
     *         Occurs when the requested resource does not exist, or cannot be found. In some cases, the resource exists
     *         in a region other than the region specified in the API call.
     * @throws ServiceQuotaExceededException
     *         Occurs when the calling command attempts to exceed one of the service quotas, for example trying to
     *         create a capability when you already have the maximum number of capabilities allowed.
     * @throws InternalServerException
     *         This exception is thrown when an error occurs in the Amazon Web Services B2B Data Interchange service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws B2BiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample B2BiClient.CreateTransformer
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/b2bi-2022-06-23/CreateTransformer" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateTransformerResponse createTransformer(CreateTransformerRequest createTransformerRequest)
            throws ConflictException, AccessDeniedException, ValidationException, ThrottlingException, ResourceNotFoundException,
            ServiceQuotaExceededException, InternalServerException, AwsServiceException, SdkClientException, B2BiException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateTransformerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateTransformerResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createTransformerRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createTransformerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "b2bi");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateTransformer");

            return clientHandler.execute(new ClientExecutionParams<CreateTransformerRequest, CreateTransformerResponse>()
                    .withOperationName("CreateTransformer").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createTransformerRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateTransformerRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the specified capability. Capabilities contain the information necessary to process incoming EDI
     * (electronic data interchange) documents.
     * </p>
     *
     * @param deleteCapabilityRequest
     * @return Result of the DeleteCapability operation returned by the service.
     * @throws ConflictException
     *         A conflict exception is thrown when you attempt to delete a resource (such as a profile or a capability)
     *         that is being used by other resources.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         Occurs when a B2BI object cannot be validated against a request from another object.
     * @throws ThrottlingException
     *         The request was denied due to throttling: the data speed and rendering may be limited depending on
     *         various parameters and conditions.
     * @throws ResourceNotFoundException
     *         Occurs when the requested resource does not exist, or cannot be found. In some cases, the resource exists
     *         in a region other than the region specified in the API call.
     * @throws InternalServerException
     *         This exception is thrown when an error occurs in the Amazon Web Services B2B Data Interchange service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws B2BiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample B2BiClient.DeleteCapability
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/b2bi-2022-06-23/DeleteCapability" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteCapabilityResponse deleteCapability(DeleteCapabilityRequest deleteCapabilityRequest) throws ConflictException,
            AccessDeniedException, ValidationException, ThrottlingException, ResourceNotFoundException, InternalServerException,
            AwsServiceException, SdkClientException, B2BiException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteCapabilityResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteCapabilityResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteCapabilityRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteCapabilityRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "b2bi");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteCapability");

            return clientHandler.execute(new ClientExecutionParams<DeleteCapabilityRequest, DeleteCapabilityResponse>()
                    .withOperationName("DeleteCapability").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteCapabilityRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteCapabilityRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the specified partnership. Partnerships link trading partners with your profile and a specific
     * transformer, so that the EDI (electronic data interchange) documents that they upload to Amazon S3 can be
     * processed according to their specifications.
     * </p>
     *
     * @param deletePartnershipRequest
     * @return Result of the DeletePartnership operation returned by the service.
     * @throws ConflictException
     *         A conflict exception is thrown when you attempt to delete a resource (such as a profile or a capability)
     *         that is being used by other resources.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         Occurs when a B2BI object cannot be validated against a request from another object.
     * @throws ThrottlingException
     *         The request was denied due to throttling: the data speed and rendering may be limited depending on
     *         various parameters and conditions.
     * @throws ResourceNotFoundException
     *         Occurs when the requested resource does not exist, or cannot be found. In some cases, the resource exists
     *         in a region other than the region specified in the API call.
     * @throws InternalServerException
     *         This exception is thrown when an error occurs in the Amazon Web Services B2B Data Interchange service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws B2BiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample B2BiClient.DeletePartnership
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/b2bi-2022-06-23/DeletePartnership" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeletePartnershipResponse deletePartnership(DeletePartnershipRequest deletePartnershipRequest)
            throws ConflictException, AccessDeniedException, ValidationException, ThrottlingException, ResourceNotFoundException,
            InternalServerException, AwsServiceException, SdkClientException, B2BiException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeletePartnershipResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeletePartnershipResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deletePartnershipRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deletePartnershipRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "b2bi");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeletePartnership");

            return clientHandler.execute(new ClientExecutionParams<DeletePartnershipRequest, DeletePartnershipResponse>()
                    .withOperationName("DeletePartnership").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deletePartnershipRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeletePartnershipRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the specified profile. Profiles contain basic information about you and your business.
     * </p>
     *
     * @param deleteProfileRequest
     * @return Result of the DeleteProfile operation returned by the service.
     * @throws ConflictException
     *         A conflict exception is thrown when you attempt to delete a resource (such as a profile or a capability)
     *         that is being used by other resources.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         Occurs when a B2BI object cannot be validated against a request from another object.
     * @throws ThrottlingException
     *         The request was denied due to throttling: the data speed and rendering may be limited depending on
     *         various parameters and conditions.
     * @throws ResourceNotFoundException
     *         Occurs when the requested resource does not exist, or cannot be found. In some cases, the resource exists
     *         in a region other than the region specified in the API call.
     * @throws InternalServerException
     *         This exception is thrown when an error occurs in the Amazon Web Services B2B Data Interchange service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws B2BiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample B2BiClient.DeleteProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/b2bi-2022-06-23/DeleteProfile" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteProfileResponse deleteProfile(DeleteProfileRequest deleteProfileRequest) throws ConflictException,
            AccessDeniedException, ValidationException, ThrottlingException, ResourceNotFoundException, InternalServerException,
            AwsServiceException, SdkClientException, B2BiException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteProfileResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteProfileResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteProfileRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "b2bi");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteProfile");

            return clientHandler.execute(new ClientExecutionParams<DeleteProfileRequest, DeleteProfileResponse>()
                    .withOperationName("DeleteProfile").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteProfileRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteProfileRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the specified transformer. Transformers describe how to process the incoming EDI (electronic data
     * interchange) documents, and extract the necessary information.
     * </p>
     *
     * @param deleteTransformerRequest
     * @return Result of the DeleteTransformer operation returned by the service.
     * @throws ConflictException
     *         A conflict exception is thrown when you attempt to delete a resource (such as a profile or a capability)
     *         that is being used by other resources.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         Occurs when a B2BI object cannot be validated against a request from another object.
     * @throws ThrottlingException
     *         The request was denied due to throttling: the data speed and rendering may be limited depending on
     *         various parameters and conditions.
     * @throws ResourceNotFoundException
     *         Occurs when the requested resource does not exist, or cannot be found. In some cases, the resource exists
     *         in a region other than the region specified in the API call.
     * @throws InternalServerException
     *         This exception is thrown when an error occurs in the Amazon Web Services B2B Data Interchange service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws B2BiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample B2BiClient.DeleteTransformer
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/b2bi-2022-06-23/DeleteTransformer" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteTransformerResponse deleteTransformer(DeleteTransformerRequest deleteTransformerRequest)
            throws ConflictException, AccessDeniedException, ValidationException, ThrottlingException, ResourceNotFoundException,
            InternalServerException, AwsServiceException, SdkClientException, B2BiException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteTransformerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteTransformerResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteTransformerRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteTransformerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "b2bi");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteTransformer");

            return clientHandler.execute(new ClientExecutionParams<DeleteTransformerRequest, DeleteTransformerResponse>()
                    .withOperationName("DeleteTransformer").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteTransformerRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteTransformerRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the details for the specified capability. Capabilities contain the information necessary to process
     * incoming EDI (electronic data interchange) documents.
     * </p>
     *
     * @param getCapabilityRequest
     * @return Result of the GetCapability operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         Occurs when a B2BI object cannot be validated against a request from another object.
     * @throws ThrottlingException
     *         The request was denied due to throttling: the data speed and rendering may be limited depending on
     *         various parameters and conditions.
     * @throws ResourceNotFoundException
     *         Occurs when the requested resource does not exist, or cannot be found. In some cases, the resource exists
     *         in a region other than the region specified in the API call.
     * @throws InternalServerException
     *         This exception is thrown when an error occurs in the Amazon Web Services B2B Data Interchange service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws B2BiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample B2BiClient.GetCapability
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/b2bi-2022-06-23/GetCapability" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetCapabilityResponse getCapability(GetCapabilityRequest getCapabilityRequest) throws AccessDeniedException,
            ValidationException, ThrottlingException, ResourceNotFoundException, InternalServerException, AwsServiceException,
            SdkClientException, B2BiException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetCapabilityResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetCapabilityResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getCapabilityRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getCapabilityRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "b2bi");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetCapability");

            return clientHandler.execute(new ClientExecutionParams<GetCapabilityRequest, GetCapabilityResponse>()
                    .withOperationName("GetCapability").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getCapabilityRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetCapabilityRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the details for a partnership, based on the partner and profile IDs specified. Partnerships link
     * trading partners with your profile and a specific transformer, so that the EDI (electronic data interchange)
     * documents that they upload to Amazon S3 can be processed according to their specifications.
     * </p>
     *
     * @param getPartnershipRequest
     * @return Result of the GetPartnership operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         Occurs when a B2BI object cannot be validated against a request from another object.
     * @throws ThrottlingException
     *         The request was denied due to throttling: the data speed and rendering may be limited depending on
     *         various parameters and conditions.
     * @throws ResourceNotFoundException
     *         Occurs when the requested resource does not exist, or cannot be found. In some cases, the resource exists
     *         in a region other than the region specified in the API call.
     * @throws InternalServerException
     *         This exception is thrown when an error occurs in the Amazon Web Services B2B Data Interchange service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws B2BiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample B2BiClient.GetPartnership
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/b2bi-2022-06-23/GetPartnership" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetPartnershipResponse getPartnership(GetPartnershipRequest getPartnershipRequest) throws AccessDeniedException,
            ValidationException, ThrottlingException, ResourceNotFoundException, InternalServerException, AwsServiceException,
            SdkClientException, B2BiException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetPartnershipResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetPartnershipResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getPartnershipRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPartnershipRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "b2bi");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPartnership");

            return clientHandler.execute(new ClientExecutionParams<GetPartnershipRequest, GetPartnershipResponse>()
                    .withOperationName("GetPartnership").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getPartnershipRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetPartnershipRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the details for the profile specified by the profile ID. Profiles contain basic information about you
     * and your business.
     * </p>
     *
     * @param getProfileRequest
     * @return Result of the GetProfile operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         Occurs when a B2BI object cannot be validated against a request from another object.
     * @throws ThrottlingException
     *         The request was denied due to throttling: the data speed and rendering may be limited depending on
     *         various parameters and conditions.
     * @throws ResourceNotFoundException
     *         Occurs when the requested resource does not exist, or cannot be found. In some cases, the resource exists
     *         in a region other than the region specified in the API call.
     * @throws InternalServerException
     *         This exception is thrown when an error occurs in the Amazon Web Services B2B Data Interchange service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws B2BiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample B2BiClient.GetProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/b2bi-2022-06-23/GetProfile" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetProfileResponse getProfile(GetProfileRequest getProfileRequest) throws AccessDeniedException, ValidationException,
            ThrottlingException, ResourceNotFoundException, InternalServerException, AwsServiceException, SdkClientException,
            B2BiException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetProfileResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetProfileResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getProfileRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "b2bi");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetProfile");

            return clientHandler.execute(new ClientExecutionParams<GetProfileRequest, GetProfileResponse>()
                    .withOperationName("GetProfile").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getProfileRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetProfileRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the details for the transformer specified by the transformer ID. Transformers describe how to process
     * the incoming EDI (electronic data interchange) documents, and extract the necessary information.
     * </p>
     *
     * @param getTransformerRequest
     * @return Result of the GetTransformer operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         Occurs when a B2BI object cannot be validated against a request from another object.
     * @throws ThrottlingException
     *         The request was denied due to throttling: the data speed and rendering may be limited depending on
     *         various parameters and conditions.
     * @throws ResourceNotFoundException
     *         Occurs when the requested resource does not exist, or cannot be found. In some cases, the resource exists
     *         in a region other than the region specified in the API call.
     * @throws InternalServerException
     *         This exception is thrown when an error occurs in the Amazon Web Services B2B Data Interchange service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws B2BiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample B2BiClient.GetTransformer
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/b2bi-2022-06-23/GetTransformer" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetTransformerResponse getTransformer(GetTransformerRequest getTransformerRequest) throws AccessDeniedException,
            ValidationException, ThrottlingException, ResourceNotFoundException, InternalServerException, AwsServiceException,
            SdkClientException, B2BiException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetTransformerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetTransformerResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getTransformerRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getTransformerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "b2bi");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetTransformer");

            return clientHandler.execute(new ClientExecutionParams<GetTransformerRequest, GetTransformerResponse>()
                    .withOperationName("GetTransformer").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getTransformerRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetTransformerRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the details of the transformer run, based on the Transformer job ID.
     * </p>
     *
     * @param getTransformerJobRequest
     * @return Result of the GetTransformerJob operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         Occurs when a B2BI object cannot be validated against a request from another object.
     * @throws ThrottlingException
     *         The request was denied due to throttling: the data speed and rendering may be limited depending on
     *         various parameters and conditions.
     * @throws ResourceNotFoundException
     *         Occurs when the requested resource does not exist, or cannot be found. In some cases, the resource exists
     *         in a region other than the region specified in the API call.
     * @throws InternalServerException
     *         This exception is thrown when an error occurs in the Amazon Web Services B2B Data Interchange service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws B2BiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample B2BiClient.GetTransformerJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/b2bi-2022-06-23/GetTransformerJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetTransformerJobResponse getTransformerJob(GetTransformerJobRequest getTransformerJobRequest)
            throws AccessDeniedException, ValidationException, ThrottlingException, ResourceNotFoundException,
            InternalServerException, AwsServiceException, SdkClientException, B2BiException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetTransformerJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetTransformerJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getTransformerJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getTransformerJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "b2bi");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetTransformerJob");

            return clientHandler.execute(new ClientExecutionParams<GetTransformerJobRequest, GetTransformerJobResponse>()
                    .withOperationName("GetTransformerJob").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getTransformerJobRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetTransformerJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the capabilities associated with your Amazon Web Services account for your current or specified region.
     * Capabilities contain the information necessary to process incoming EDI (electronic data interchange) documents.
     * </p>
     *
     * @param listCapabilitiesRequest
     * @return Result of the ListCapabilities operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws B2BiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample B2BiClient.ListCapabilities
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/b2bi-2022-06-23/ListCapabilities" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListCapabilitiesResponse listCapabilities(ListCapabilitiesRequest listCapabilitiesRequest) throws AwsServiceException,
            SdkClientException, B2BiException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListCapabilitiesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListCapabilitiesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listCapabilitiesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listCapabilitiesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "b2bi");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListCapabilities");

            return clientHandler.execute(new ClientExecutionParams<ListCapabilitiesRequest, ListCapabilitiesResponse>()
                    .withOperationName("ListCapabilities").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listCapabilitiesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListCapabilitiesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the partnerships associated with your Amazon Web Services account for your current or specified region.
     * Partnerships link trading partners with your profile and a specific transformer, so that the EDI (electronic data
     * interchange) documents that they upload to Amazon S3 can be processed according to their specifications.
     * </p>
     *
     * @param listPartnershipsRequest
     * @return Result of the ListPartnerships operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         Occurs when a B2BI object cannot be validated against a request from another object.
     * @throws ThrottlingException
     *         The request was denied due to throttling: the data speed and rendering may be limited depending on
     *         various parameters and conditions.
     * @throws ResourceNotFoundException
     *         Occurs when the requested resource does not exist, or cannot be found. In some cases, the resource exists
     *         in a region other than the region specified in the API call.
     * @throws InternalServerException
     *         This exception is thrown when an error occurs in the Amazon Web Services B2B Data Interchange service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws B2BiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample B2BiClient.ListPartnerships
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/b2bi-2022-06-23/ListPartnerships" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListPartnershipsResponse listPartnerships(ListPartnershipsRequest listPartnershipsRequest)
            throws AccessDeniedException, ValidationException, ThrottlingException, ResourceNotFoundException,
            InternalServerException, AwsServiceException, SdkClientException, B2BiException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListPartnershipsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListPartnershipsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listPartnershipsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPartnershipsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "b2bi");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPartnerships");

            return clientHandler.execute(new ClientExecutionParams<ListPartnershipsRequest, ListPartnershipsResponse>()
                    .withOperationName("ListPartnerships").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listPartnershipsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListPartnershipsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the profiles associated with your Amazon Web Services account for your current or specified region.
     * Profiles contain basic information about you and your business.
     * </p>
     *
     * @param listProfilesRequest
     * @return Result of the ListProfiles operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws B2BiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample B2BiClient.ListProfiles
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/b2bi-2022-06-23/ListProfiles" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListProfilesResponse listProfiles(ListProfilesRequest listProfilesRequest) throws AwsServiceException,
            SdkClientException, B2BiException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListProfilesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListProfilesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listProfilesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listProfilesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "b2bi");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListProfiles");

            return clientHandler.execute(new ClientExecutionParams<ListProfilesRequest, ListProfilesResponse>()
                    .withOperationName("ListProfiles").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listProfilesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListProfilesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all of the tags associated with the Amazon Resource Name (ARN) that you specify. The resource can be a
     * capability, partnership, profile, or transformer.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ValidationException
     *         Occurs when a B2BI object cannot be validated against a request from another object.
     * @throws ResourceNotFoundException
     *         Occurs when the requested resource does not exist, or cannot be found. In some cases, the resource exists
     *         in a region other than the region specified in the API call.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws B2BiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample B2BiClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/b2bi-2022-06-23/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ValidationException, ResourceNotFoundException, AwsServiceException, SdkClientException, B2BiException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "b2bi");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the available transformers. Transformers describe how to process the incoming EDI (electronic data
     * interchange) documents, and extract the necessary information.
     * </p>
     *
     * @param listTransformersRequest
     * @return Result of the ListTransformers operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws B2BiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample B2BiClient.ListTransformers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/b2bi-2022-06-23/ListTransformers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListTransformersResponse listTransformers(ListTransformersRequest listTransformersRequest) throws AwsServiceException,
            SdkClientException, B2BiException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTransformersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListTransformersResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTransformersRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTransformersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "b2bi");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTransformers");

            return clientHandler.execute(new ClientExecutionParams<ListTransformersRequest, ListTransformersResponse>()
                    .withOperationName("ListTransformers").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTransformersRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTransformersRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Runs a job, using a transformer, to parse input EDI (electronic data interchange) file into the output structures
     * used by Amazon Web Services B2BI Data Interchange.
     * </p>
     *
     * @param startTransformerJobRequest
     * @return Result of the StartTransformerJob operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         Occurs when a B2BI object cannot be validated against a request from another object.
     * @throws ThrottlingException
     *         The request was denied due to throttling: the data speed and rendering may be limited depending on
     *         various parameters and conditions.
     * @throws ResourceNotFoundException
     *         Occurs when the requested resource does not exist, or cannot be found. In some cases, the resource exists
     *         in a region other than the region specified in the API call.
     * @throws InternalServerException
     *         This exception is thrown when an error occurs in the Amazon Web Services B2B Data Interchange service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws B2BiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample B2BiClient.StartTransformerJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/b2bi-2022-06-23/StartTransformerJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public StartTransformerJobResponse startTransformerJob(StartTransformerJobRequest startTransformerJobRequest)
            throws AccessDeniedException, ValidationException, ThrottlingException, ResourceNotFoundException,
            InternalServerException, AwsServiceException, SdkClientException, B2BiException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartTransformerJobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StartTransformerJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startTransformerJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startTransformerJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "b2bi");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartTransformerJob");

            return clientHandler.execute(new ClientExecutionParams<StartTransformerJobRequest, StartTransformerJobResponse>()
                    .withOperationName("StartTransformerJob").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(startTransformerJobRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartTransformerJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Attaches a key-value pair to a resource, as identified by its Amazon Resource Name (ARN). Resources are
     * capability, partnership, profile, transformers and other entities.
     * </p>
     * <p>
     * There is no response returned from this call.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ValidationException
     *         Occurs when a B2BI object cannot be validated against a request from another object.
     * @throws ThrottlingException
     *         The request was denied due to throttling: the data speed and rendering may be limited depending on
     *         various parameters and conditions.
     * @throws ResourceNotFoundException
     *         Occurs when the requested resource does not exist, or cannot be found. In some cases, the resource exists
     *         in a region other than the region specified in the API call.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws B2BiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample B2BiClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/b2bi-2022-06-23/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ValidationException,
            ThrottlingException, ResourceNotFoundException, AwsServiceException, SdkClientException, B2BiException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "b2bi");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Maps the input file according to the provided template file.
     * </p>
     *
     * @param testMappingRequest
     * @return Result of the TestMapping operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         Occurs when a B2BI object cannot be validated against a request from another object.
     * @throws ThrottlingException
     *         The request was denied due to throttling: the data speed and rendering may be limited depending on
     *         various parameters and conditions.
     * @throws ResourceNotFoundException
     *         Occurs when the requested resource does not exist, or cannot be found. In some cases, the resource exists
     *         in a region other than the region specified in the API call.
     * @throws InternalServerException
     *         This exception is thrown when an error occurs in the Amazon Web Services B2B Data Interchange service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws B2BiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample B2BiClient.TestMapping
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/b2bi-2022-06-23/TestMapping" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TestMappingResponse testMapping(TestMappingRequest testMappingRequest) throws AccessDeniedException,
            ValidationException, ThrottlingException, ResourceNotFoundException, InternalServerException, AwsServiceException,
            SdkClientException, B2BiException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TestMappingResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TestMappingResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(testMappingRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, testMappingRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "b2bi");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TestMapping");

            return clientHandler.execute(new ClientExecutionParams<TestMappingRequest, TestMappingResponse>()
                    .withOperationName("TestMapping").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(testMappingRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TestMappingRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Parses the input EDI (electronic data interchange) file.
     * </p>
     *
     * @param testParsingRequest
     * @return Result of the TestParsing operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         Occurs when a B2BI object cannot be validated against a request from another object.
     * @throws ThrottlingException
     *         The request was denied due to throttling: the data speed and rendering may be limited depending on
     *         various parameters and conditions.
     * @throws ResourceNotFoundException
     *         Occurs when the requested resource does not exist, or cannot be found. In some cases, the resource exists
     *         in a region other than the region specified in the API call.
     * @throws InternalServerException
     *         This exception is thrown when an error occurs in the Amazon Web Services B2B Data Interchange service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws B2BiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample B2BiClient.TestParsing
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/b2bi-2022-06-23/TestParsing" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TestParsingResponse testParsing(TestParsingRequest testParsingRequest) throws AccessDeniedException,
            ValidationException, ThrottlingException, ResourceNotFoundException, InternalServerException, AwsServiceException,
            SdkClientException, B2BiException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TestParsingResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TestParsingResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(testParsingRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, testParsingRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "b2bi");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TestParsing");

            return clientHandler.execute(new ClientExecutionParams<TestParsingRequest, TestParsingResponse>()
                    .withOperationName("TestParsing").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(testParsingRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TestParsingRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Detaches a key-value pair from the specified resource, as identified by its Amazon Resource Name (ARN). Resources
     * are capability, partnership, profile, transformers and other entities.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ValidationException
     *         Occurs when a B2BI object cannot be validated against a request from another object.
     * @throws ResourceNotFoundException
     *         Occurs when the requested resource does not exist, or cannot be found. In some cases, the resource exists
     *         in a region other than the region specified in the API call.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws B2BiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample B2BiClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/b2bi-2022-06-23/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ValidationException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, B2BiException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "b2bi");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates some of the parameters for a capability, based on the specified parameters. Capabilities contain the
     * information necessary to process incoming EDI (electronic data interchange) documents.
     * </p>
     *
     * @param updateCapabilityRequest
     * @return Result of the UpdateCapability operation returned by the service.
     * @throws ConflictException
     *         A conflict exception is thrown when you attempt to delete a resource (such as a profile or a capability)
     *         that is being used by other resources.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         Occurs when a B2BI object cannot be validated against a request from another object.
     * @throws ThrottlingException
     *         The request was denied due to throttling: the data speed and rendering may be limited depending on
     *         various parameters and conditions.
     * @throws ResourceNotFoundException
     *         Occurs when the requested resource does not exist, or cannot be found. In some cases, the resource exists
     *         in a region other than the region specified in the API call.
     * @throws ServiceQuotaExceededException
     *         Occurs when the calling command attempts to exceed one of the service quotas, for example trying to
     *         create a capability when you already have the maximum number of capabilities allowed.
     * @throws InternalServerException
     *         This exception is thrown when an error occurs in the Amazon Web Services B2B Data Interchange service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws B2BiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample B2BiClient.UpdateCapability
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/b2bi-2022-06-23/UpdateCapability" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateCapabilityResponse updateCapability(UpdateCapabilityRequest updateCapabilityRequest) throws ConflictException,
            AccessDeniedException, ValidationException, ThrottlingException, ResourceNotFoundException,
            ServiceQuotaExceededException, InternalServerException, AwsServiceException, SdkClientException, B2BiException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateCapabilityResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateCapabilityResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateCapabilityRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateCapabilityRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "b2bi");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateCapability");

            return clientHandler.execute(new ClientExecutionParams<UpdateCapabilityRequest, UpdateCapabilityResponse>()
                    .withOperationName("UpdateCapability").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateCapabilityRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateCapabilityRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates some of the parameters for a partnership between a customer and trading partner. Partnerships link
     * trading partners with your profile and a specific transformer, so that the EDI (electronic data interchange)
     * documents that they upload to Amazon S3 can be processed according to their specifications.
     * </p>
     *
     * @param updatePartnershipRequest
     * @return Result of the UpdatePartnership operation returned by the service.
     * @throws ConflictException
     *         A conflict exception is thrown when you attempt to delete a resource (such as a profile or a capability)
     *         that is being used by other resources.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         Occurs when a B2BI object cannot be validated against a request from another object.
     * @throws ThrottlingException
     *         The request was denied due to throttling: the data speed and rendering may be limited depending on
     *         various parameters and conditions.
     * @throws ResourceNotFoundException
     *         Occurs when the requested resource does not exist, or cannot be found. In some cases, the resource exists
     *         in a region other than the region specified in the API call.
     * @throws ServiceQuotaExceededException
     *         Occurs when the calling command attempts to exceed one of the service quotas, for example trying to
     *         create a capability when you already have the maximum number of capabilities allowed.
     * @throws InternalServerException
     *         This exception is thrown when an error occurs in the Amazon Web Services B2B Data Interchange service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws B2BiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample B2BiClient.UpdatePartnership
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/b2bi-2022-06-23/UpdatePartnership" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdatePartnershipResponse updatePartnership(UpdatePartnershipRequest updatePartnershipRequest)
            throws ConflictException, AccessDeniedException, ValidationException, ThrottlingException, ResourceNotFoundException,
            ServiceQuotaExceededException, InternalServerException, AwsServiceException, SdkClientException, B2BiException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdatePartnershipResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdatePartnershipResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updatePartnershipRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updatePartnershipRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "b2bi");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdatePartnership");

            return clientHandler.execute(new ClientExecutionParams<UpdatePartnershipRequest, UpdatePartnershipResponse>()
                    .withOperationName("UpdatePartnership").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updatePartnershipRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdatePartnershipRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the specified parameters for a profile. Profiles contain basic information about you and your business.
     * </p>
     *
     * @param updateProfileRequest
     * @return Result of the UpdateProfile operation returned by the service.
     * @throws ConflictException
     *         A conflict exception is thrown when you attempt to delete a resource (such as a profile or a capability)
     *         that is being used by other resources.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         Occurs when a B2BI object cannot be validated against a request from another object.
     * @throws ThrottlingException
     *         The request was denied due to throttling: the data speed and rendering may be limited depending on
     *         various parameters and conditions.
     * @throws ResourceNotFoundException
     *         Occurs when the requested resource does not exist, or cannot be found. In some cases, the resource exists
     *         in a region other than the region specified in the API call.
     * @throws ServiceQuotaExceededException
     *         Occurs when the calling command attempts to exceed one of the service quotas, for example trying to
     *         create a capability when you already have the maximum number of capabilities allowed.
     * @throws InternalServerException
     *         This exception is thrown when an error occurs in the Amazon Web Services B2B Data Interchange service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws B2BiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample B2BiClient.UpdateProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/b2bi-2022-06-23/UpdateProfile" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateProfileResponse updateProfile(UpdateProfileRequest updateProfileRequest) throws ConflictException,
            AccessDeniedException, ValidationException, ThrottlingException, ResourceNotFoundException,
            ServiceQuotaExceededException, InternalServerException, AwsServiceException, SdkClientException, B2BiException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateProfileResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateProfileResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateProfileRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "b2bi");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateProfile");

            return clientHandler.execute(new ClientExecutionParams<UpdateProfileRequest, UpdateProfileResponse>()
                    .withOperationName("UpdateProfile").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateProfileRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateProfileRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the specified parameters for a transformer. Transformers describe how to process the incoming EDI
     * (electronic data interchange) documents, and extract the necessary information.
     * </p>
     *
     * @param updateTransformerRequest
     * @return Result of the UpdateTransformer operation returned by the service.
     * @throws ConflictException
     *         A conflict exception is thrown when you attempt to delete a resource (such as a profile or a capability)
     *         that is being used by other resources.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         Occurs when a B2BI object cannot be validated against a request from another object.
     * @throws ThrottlingException
     *         The request was denied due to throttling: the data speed and rendering may be limited depending on
     *         various parameters and conditions.
     * @throws ResourceNotFoundException
     *         Occurs when the requested resource does not exist, or cannot be found. In some cases, the resource exists
     *         in a region other than the region specified in the API call.
     * @throws ServiceQuotaExceededException
     *         Occurs when the calling command attempts to exceed one of the service quotas, for example trying to
     *         create a capability when you already have the maximum number of capabilities allowed.
     * @throws InternalServerException
     *         This exception is thrown when an error occurs in the Amazon Web Services B2B Data Interchange service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws B2BiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample B2BiClient.UpdateTransformer
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/b2bi-2022-06-23/UpdateTransformer" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateTransformerResponse updateTransformer(UpdateTransformerRequest updateTransformerRequest)
            throws ConflictException, AccessDeniedException, ValidationException, ThrottlingException, ResourceNotFoundException,
            ServiceQuotaExceededException, InternalServerException, AwsServiceException, SdkClientException, B2BiException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateTransformerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateTransformerResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateTransformerRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateTransformerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "b2bi");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateTransformer");

            return clientHandler.execute(new ClientExecutionParams<UpdateTransformerRequest, UpdateTransformerResponse>()
                    .withOperationName("UpdateTransformer").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateTransformerRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateTransformerRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        B2BiServiceClientConfigurationBuilder serviceConfigBuilder = new B2BiServiceClientConfigurationBuilder(configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(B2BiException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.0")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build());
    }

    @Override
    public final B2BiServiceClientConfiguration serviceClientConfiguration() {
        return new B2BiServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
