/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.backupgateway;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.backupgateway.internal.BackupGatewayServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.backupgateway.model.AccessDeniedException;
import software.amazon.awssdk.services.backupgateway.model.AssociateGatewayToServerRequest;
import software.amazon.awssdk.services.backupgateway.model.AssociateGatewayToServerResponse;
import software.amazon.awssdk.services.backupgateway.model.BackupGatewayException;
import software.amazon.awssdk.services.backupgateway.model.ConflictException;
import software.amazon.awssdk.services.backupgateway.model.CreateGatewayRequest;
import software.amazon.awssdk.services.backupgateway.model.CreateGatewayResponse;
import software.amazon.awssdk.services.backupgateway.model.DeleteGatewayRequest;
import software.amazon.awssdk.services.backupgateway.model.DeleteGatewayResponse;
import software.amazon.awssdk.services.backupgateway.model.DeleteHypervisorRequest;
import software.amazon.awssdk.services.backupgateway.model.DeleteHypervisorResponse;
import software.amazon.awssdk.services.backupgateway.model.DisassociateGatewayFromServerRequest;
import software.amazon.awssdk.services.backupgateway.model.DisassociateGatewayFromServerResponse;
import software.amazon.awssdk.services.backupgateway.model.GetBandwidthRateLimitScheduleRequest;
import software.amazon.awssdk.services.backupgateway.model.GetBandwidthRateLimitScheduleResponse;
import software.amazon.awssdk.services.backupgateway.model.GetGatewayRequest;
import software.amazon.awssdk.services.backupgateway.model.GetGatewayResponse;
import software.amazon.awssdk.services.backupgateway.model.GetHypervisorPropertyMappingsRequest;
import software.amazon.awssdk.services.backupgateway.model.GetHypervisorPropertyMappingsResponse;
import software.amazon.awssdk.services.backupgateway.model.GetHypervisorRequest;
import software.amazon.awssdk.services.backupgateway.model.GetHypervisorResponse;
import software.amazon.awssdk.services.backupgateway.model.GetVirtualMachineRequest;
import software.amazon.awssdk.services.backupgateway.model.GetVirtualMachineResponse;
import software.amazon.awssdk.services.backupgateway.model.ImportHypervisorConfigurationRequest;
import software.amazon.awssdk.services.backupgateway.model.ImportHypervisorConfigurationResponse;
import software.amazon.awssdk.services.backupgateway.model.InternalServerException;
import software.amazon.awssdk.services.backupgateway.model.ListGatewaysRequest;
import software.amazon.awssdk.services.backupgateway.model.ListGatewaysResponse;
import software.amazon.awssdk.services.backupgateway.model.ListHypervisorsRequest;
import software.amazon.awssdk.services.backupgateway.model.ListHypervisorsResponse;
import software.amazon.awssdk.services.backupgateway.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.backupgateway.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.backupgateway.model.ListVirtualMachinesRequest;
import software.amazon.awssdk.services.backupgateway.model.ListVirtualMachinesResponse;
import software.amazon.awssdk.services.backupgateway.model.PutBandwidthRateLimitScheduleRequest;
import software.amazon.awssdk.services.backupgateway.model.PutBandwidthRateLimitScheduleResponse;
import software.amazon.awssdk.services.backupgateway.model.PutHypervisorPropertyMappingsRequest;
import software.amazon.awssdk.services.backupgateway.model.PutHypervisorPropertyMappingsResponse;
import software.amazon.awssdk.services.backupgateway.model.PutMaintenanceStartTimeRequest;
import software.amazon.awssdk.services.backupgateway.model.PutMaintenanceStartTimeResponse;
import software.amazon.awssdk.services.backupgateway.model.ResourceNotFoundException;
import software.amazon.awssdk.services.backupgateway.model.StartVirtualMachinesMetadataSyncRequest;
import software.amazon.awssdk.services.backupgateway.model.StartVirtualMachinesMetadataSyncResponse;
import software.amazon.awssdk.services.backupgateway.model.TagResourceRequest;
import software.amazon.awssdk.services.backupgateway.model.TagResourceResponse;
import software.amazon.awssdk.services.backupgateway.model.TestHypervisorConfigurationRequest;
import software.amazon.awssdk.services.backupgateway.model.TestHypervisorConfigurationResponse;
import software.amazon.awssdk.services.backupgateway.model.ThrottlingException;
import software.amazon.awssdk.services.backupgateway.model.UntagResourceRequest;
import software.amazon.awssdk.services.backupgateway.model.UntagResourceResponse;
import software.amazon.awssdk.services.backupgateway.model.UpdateGatewayInformationRequest;
import software.amazon.awssdk.services.backupgateway.model.UpdateGatewayInformationResponse;
import software.amazon.awssdk.services.backupgateway.model.UpdateGatewaySoftwareNowRequest;
import software.amazon.awssdk.services.backupgateway.model.UpdateGatewaySoftwareNowResponse;
import software.amazon.awssdk.services.backupgateway.model.UpdateHypervisorRequest;
import software.amazon.awssdk.services.backupgateway.model.UpdateHypervisorResponse;
import software.amazon.awssdk.services.backupgateway.model.ValidationException;
import software.amazon.awssdk.services.backupgateway.transform.AssociateGatewayToServerRequestMarshaller;
import software.amazon.awssdk.services.backupgateway.transform.CreateGatewayRequestMarshaller;
import software.amazon.awssdk.services.backupgateway.transform.DeleteGatewayRequestMarshaller;
import software.amazon.awssdk.services.backupgateway.transform.DeleteHypervisorRequestMarshaller;
import software.amazon.awssdk.services.backupgateway.transform.DisassociateGatewayFromServerRequestMarshaller;
import software.amazon.awssdk.services.backupgateway.transform.GetBandwidthRateLimitScheduleRequestMarshaller;
import software.amazon.awssdk.services.backupgateway.transform.GetGatewayRequestMarshaller;
import software.amazon.awssdk.services.backupgateway.transform.GetHypervisorPropertyMappingsRequestMarshaller;
import software.amazon.awssdk.services.backupgateway.transform.GetHypervisorRequestMarshaller;
import software.amazon.awssdk.services.backupgateway.transform.GetVirtualMachineRequestMarshaller;
import software.amazon.awssdk.services.backupgateway.transform.ImportHypervisorConfigurationRequestMarshaller;
import software.amazon.awssdk.services.backupgateway.transform.ListGatewaysRequestMarshaller;
import software.amazon.awssdk.services.backupgateway.transform.ListHypervisorsRequestMarshaller;
import software.amazon.awssdk.services.backupgateway.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.backupgateway.transform.ListVirtualMachinesRequestMarshaller;
import software.amazon.awssdk.services.backupgateway.transform.PutBandwidthRateLimitScheduleRequestMarshaller;
import software.amazon.awssdk.services.backupgateway.transform.PutHypervisorPropertyMappingsRequestMarshaller;
import software.amazon.awssdk.services.backupgateway.transform.PutMaintenanceStartTimeRequestMarshaller;
import software.amazon.awssdk.services.backupgateway.transform.StartVirtualMachinesMetadataSyncRequestMarshaller;
import software.amazon.awssdk.services.backupgateway.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.backupgateway.transform.TestHypervisorConfigurationRequestMarshaller;
import software.amazon.awssdk.services.backupgateway.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.backupgateway.transform.UpdateGatewayInformationRequestMarshaller;
import software.amazon.awssdk.services.backupgateway.transform.UpdateGatewaySoftwareNowRequestMarshaller;
import software.amazon.awssdk.services.backupgateway.transform.UpdateHypervisorRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link BackupGatewayAsyncClient}.
 *
 * @see BackupGatewayAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultBackupGatewayAsyncClient implements BackupGatewayAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultBackupGatewayAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.AWS_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultBackupGatewayAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Associates a backup gateway with your server. After you complete the association process, you can back up and
     * restore your VMs through the gateway.
     * </p>
     *
     * @param associateGatewayToServerRequest
     * @return A Java Future containing the result of the AssociateGatewayToServer operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The operation did not succeed because a validation error occurred.</li>
     *         <li>ConflictException The operation cannot proceed because it is not supported.</li>
     *         <li>InternalServerException The operation did not succeed because an internal error occurred. Try again
     *         later.</li>
     *         <li>ThrottlingException TPS has been limited to protect against intentional or unintentional high request
     *         volumes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupGatewayException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupGatewayAsyncClient.AssociateGatewayToServer
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-gateway-2021-01-01/AssociateGatewayToServer"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<AssociateGatewayToServerResponse> associateGatewayToServer(
            AssociateGatewayToServerRequest associateGatewayToServerRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(associateGatewayToServerRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, associateGatewayToServerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup Gateway");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateGatewayToServer");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AssociateGatewayToServerResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AssociateGatewayToServerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AssociateGatewayToServerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AssociateGatewayToServerRequest, AssociateGatewayToServerResponse>()
                            .withOperationName("AssociateGatewayToServer").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new AssociateGatewayToServerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(associateGatewayToServerRequest));
            CompletableFuture<AssociateGatewayToServerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a backup gateway. After you create a gateway, you can associate it with a server using the
     * <code>AssociateGatewayToServer</code> operation.
     * </p>
     *
     * @param createGatewayRequest
     * @return A Java Future containing the result of the CreateGateway operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The operation did not succeed because a validation error occurred.</li>
     *         <li>InternalServerException The operation did not succeed because an internal error occurred. Try again
     *         later.</li>
     *         <li>ThrottlingException TPS has been limited to protect against intentional or unintentional high request
     *         volumes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupGatewayException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupGatewayAsyncClient.CreateGateway
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-gateway-2021-01-01/CreateGateway" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateGatewayResponse> createGateway(CreateGatewayRequest createGatewayRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createGatewayRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createGatewayRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup Gateway");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateGateway");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateGatewayResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateGatewayResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateGatewayResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateGatewayRequest, CreateGatewayResponse>()
                            .withOperationName("CreateGateway").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateGatewayRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createGatewayRequest));
            CompletableFuture<CreateGatewayResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a backup gateway.
     * </p>
     *
     * @param deleteGatewayRequest
     * @return A Java Future containing the result of the DeleteGateway operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The operation did not succeed because a validation error occurred.</li>
     *         <li>InternalServerException The operation did not succeed because an internal error occurred. Try again
     *         later.</li>
     *         <li>ResourceNotFoundException A resource that is required for the action wasn't found.</li>
     *         <li>ThrottlingException TPS has been limited to protect against intentional or unintentional high request
     *         volumes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupGatewayException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupGatewayAsyncClient.DeleteGateway
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-gateway-2021-01-01/DeleteGateway" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteGatewayResponse> deleteGateway(DeleteGatewayRequest deleteGatewayRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteGatewayRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteGatewayRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup Gateway");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteGateway");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteGatewayResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteGatewayResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteGatewayResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteGatewayRequest, DeleteGatewayResponse>()
                            .withOperationName("DeleteGateway").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteGatewayRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteGatewayRequest));
            CompletableFuture<DeleteGatewayResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a hypervisor.
     * </p>
     *
     * @param deleteHypervisorRequest
     * @return A Java Future containing the result of the DeleteHypervisor operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The operation did not succeed because a validation error occurred.</li>
     *         <li>ConflictException The operation cannot proceed because it is not supported.</li>
     *         <li>InternalServerException The operation did not succeed because an internal error occurred. Try again
     *         later.</li>
     *         <li>AccessDeniedException The operation cannot proceed because you have insufficient permissions.</li>
     *         <li>ResourceNotFoundException A resource that is required for the action wasn't found.</li>
     *         <li>ThrottlingException TPS has been limited to protect against intentional or unintentional high request
     *         volumes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupGatewayException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupGatewayAsyncClient.DeleteHypervisor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-gateway-2021-01-01/DeleteHypervisor"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteHypervisorResponse> deleteHypervisor(DeleteHypervisorRequest deleteHypervisorRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteHypervisorRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteHypervisorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup Gateway");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteHypervisor");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteHypervisorResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteHypervisorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteHypervisorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteHypervisorRequest, DeleteHypervisorResponse>()
                            .withOperationName("DeleteHypervisor").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteHypervisorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteHypervisorRequest));
            CompletableFuture<DeleteHypervisorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Disassociates a backup gateway from the specified server. After the disassociation process finishes, the gateway
     * can no longer access the virtual machines on the server.
     * </p>
     *
     * @param disassociateGatewayFromServerRequest
     * @return A Java Future containing the result of the DisassociateGatewayFromServer operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The operation did not succeed because a validation error occurred.</li>
     *         <li>ConflictException The operation cannot proceed because it is not supported.</li>
     *         <li>InternalServerException The operation did not succeed because an internal error occurred. Try again
     *         later.</li>
     *         <li>ResourceNotFoundException A resource that is required for the action wasn't found.</li>
     *         <li>ThrottlingException TPS has been limited to protect against intentional or unintentional high request
     *         volumes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupGatewayException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupGatewayAsyncClient.DisassociateGatewayFromServer
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-gateway-2021-01-01/DisassociateGatewayFromServer"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DisassociateGatewayFromServerResponse> disassociateGatewayFromServer(
            DisassociateGatewayFromServerRequest disassociateGatewayFromServerRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(disassociateGatewayFromServerRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                disassociateGatewayFromServerRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup Gateway");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateGatewayFromServer");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisassociateGatewayFromServerResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DisassociateGatewayFromServerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisassociateGatewayFromServerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisassociateGatewayFromServerRequest, DisassociateGatewayFromServerResponse>()
                            .withOperationName("DisassociateGatewayFromServer").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DisassociateGatewayFromServerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(disassociateGatewayFromServerRequest));
            CompletableFuture<DisassociateGatewayFromServerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the bandwidth rate limit schedule for a specified gateway. By default, gateways do not have bandwidth
     * rate limit schedules, which means no bandwidth rate limiting is in effect. Use this to get a gateway's bandwidth
     * rate limit schedule.
     * </p>
     *
     * @param getBandwidthRateLimitScheduleRequest
     * @return A Java Future containing the result of the GetBandwidthRateLimitSchedule operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The operation did not succeed because a validation error occurred.</li>
     *         <li>InternalServerException The operation did not succeed because an internal error occurred. Try again
     *         later.</li>
     *         <li>ResourceNotFoundException A resource that is required for the action wasn't found.</li>
     *         <li>ThrottlingException TPS has been limited to protect against intentional or unintentional high request
     *         volumes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupGatewayException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupGatewayAsyncClient.GetBandwidthRateLimitSchedule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-gateway-2021-01-01/GetBandwidthRateLimitSchedule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetBandwidthRateLimitScheduleResponse> getBandwidthRateLimitSchedule(
            GetBandwidthRateLimitScheduleRequest getBandwidthRateLimitScheduleRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getBandwidthRateLimitScheduleRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getBandwidthRateLimitScheduleRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup Gateway");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetBandwidthRateLimitSchedule");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetBandwidthRateLimitScheduleResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetBandwidthRateLimitScheduleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetBandwidthRateLimitScheduleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetBandwidthRateLimitScheduleRequest, GetBandwidthRateLimitScheduleResponse>()
                            .withOperationName("GetBandwidthRateLimitSchedule").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetBandwidthRateLimitScheduleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getBandwidthRateLimitScheduleRequest));
            CompletableFuture<GetBandwidthRateLimitScheduleResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * By providing the ARN (Amazon Resource Name), this API returns the gateway.
     * </p>
     *
     * @param getGatewayRequest
     * @return A Java Future containing the result of the GetGateway operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The operation did not succeed because a validation error occurred.</li>
     *         <li>InternalServerException The operation did not succeed because an internal error occurred. Try again
     *         later.</li>
     *         <li>ResourceNotFoundException A resource that is required for the action wasn't found.</li>
     *         <li>ThrottlingException TPS has been limited to protect against intentional or unintentional high request
     *         volumes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupGatewayException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupGatewayAsyncClient.GetGateway
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-gateway-2021-01-01/GetGateway" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetGatewayResponse> getGateway(GetGatewayRequest getGatewayRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getGatewayRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getGatewayRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup Gateway");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetGateway");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetGatewayResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetGatewayResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetGatewayResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetGatewayRequest, GetGatewayResponse>().withOperationName("GetGateway")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetGatewayRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getGatewayRequest));
            CompletableFuture<GetGatewayResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This action requests information about the specified hypervisor to which the gateway will connect. A hypervisor
     * is hardware, software, or firmware that creates and manages virtual machines, and allocates resources to them.
     * </p>
     *
     * @param getHypervisorRequest
     * @return A Java Future containing the result of the GetHypervisor operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The operation did not succeed because a validation error occurred.</li>
     *         <li>InternalServerException The operation did not succeed because an internal error occurred. Try again
     *         later.</li>
     *         <li>ResourceNotFoundException A resource that is required for the action wasn't found.</li>
     *         <li>ThrottlingException TPS has been limited to protect against intentional or unintentional high request
     *         volumes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupGatewayException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupGatewayAsyncClient.GetHypervisor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-gateway-2021-01-01/GetHypervisor" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetHypervisorResponse> getHypervisor(GetHypervisorRequest getHypervisorRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getHypervisorRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getHypervisorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup Gateway");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetHypervisor");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetHypervisorResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetHypervisorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetHypervisorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetHypervisorRequest, GetHypervisorResponse>()
                            .withOperationName("GetHypervisor").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetHypervisorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getHypervisorRequest));
            CompletableFuture<GetHypervisorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This action retrieves the property mappings for the specified hypervisor. A hypervisor property mapping displays
     * the relationship of entity properties available from the on-premises hypervisor to the properties available in
     * Amazon Web Services.
     * </p>
     *
     * @param getHypervisorPropertyMappingsRequest
     * @return A Java Future containing the result of the GetHypervisorPropertyMappings operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The operation did not succeed because a validation error occurred.</li>
     *         <li>InternalServerException The operation did not succeed because an internal error occurred. Try again
     *         later.</li>
     *         <li>ResourceNotFoundException A resource that is required for the action wasn't found.</li>
     *         <li>ThrottlingException TPS has been limited to protect against intentional or unintentional high request
     *         volumes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupGatewayException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupGatewayAsyncClient.GetHypervisorPropertyMappings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-gateway-2021-01-01/GetHypervisorPropertyMappings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetHypervisorPropertyMappingsResponse> getHypervisorPropertyMappings(
            GetHypervisorPropertyMappingsRequest getHypervisorPropertyMappingsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getHypervisorPropertyMappingsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getHypervisorPropertyMappingsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup Gateway");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetHypervisorPropertyMappings");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetHypervisorPropertyMappingsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetHypervisorPropertyMappingsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetHypervisorPropertyMappingsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetHypervisorPropertyMappingsRequest, GetHypervisorPropertyMappingsResponse>()
                            .withOperationName("GetHypervisorPropertyMappings").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetHypervisorPropertyMappingsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getHypervisorPropertyMappingsRequest));
            CompletableFuture<GetHypervisorPropertyMappingsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * By providing the ARN (Amazon Resource Name), this API returns the virtual machine.
     * </p>
     *
     * @param getVirtualMachineRequest
     * @return A Java Future containing the result of the GetVirtualMachine operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The operation did not succeed because a validation error occurred.</li>
     *         <li>InternalServerException The operation did not succeed because an internal error occurred. Try again
     *         later.</li>
     *         <li>ResourceNotFoundException A resource that is required for the action wasn't found.</li>
     *         <li>ThrottlingException TPS has been limited to protect against intentional or unintentional high request
     *         volumes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupGatewayException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupGatewayAsyncClient.GetVirtualMachine
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-gateway-2021-01-01/GetVirtualMachine"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetVirtualMachineResponse> getVirtualMachine(GetVirtualMachineRequest getVirtualMachineRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getVirtualMachineRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getVirtualMachineRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup Gateway");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetVirtualMachine");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetVirtualMachineResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetVirtualMachineResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetVirtualMachineResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetVirtualMachineRequest, GetVirtualMachineResponse>()
                            .withOperationName("GetVirtualMachine").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetVirtualMachineRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getVirtualMachineRequest));
            CompletableFuture<GetVirtualMachineResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Connect to a hypervisor by importing its configuration.
     * </p>
     *
     * @param importHypervisorConfigurationRequest
     * @return A Java Future containing the result of the ImportHypervisorConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The operation did not succeed because a validation error occurred.</li>
     *         <li>ConflictException The operation cannot proceed because it is not supported.</li>
     *         <li>InternalServerException The operation did not succeed because an internal error occurred. Try again
     *         later.</li>
     *         <li>AccessDeniedException The operation cannot proceed because you have insufficient permissions.</li>
     *         <li>ThrottlingException TPS has been limited to protect against intentional or unintentional high request
     *         volumes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupGatewayException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupGatewayAsyncClient.ImportHypervisorConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-gateway-2021-01-01/ImportHypervisorConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ImportHypervisorConfigurationResponse> importHypervisorConfiguration(
            ImportHypervisorConfigurationRequest importHypervisorConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(importHypervisorConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                importHypervisorConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup Gateway");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ImportHypervisorConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ImportHypervisorConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ImportHypervisorConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ImportHypervisorConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ImportHypervisorConfigurationRequest, ImportHypervisorConfigurationResponse>()
                            .withOperationName("ImportHypervisorConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ImportHypervisorConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(importHypervisorConfigurationRequest));
            CompletableFuture<ImportHypervisorConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists backup gateways owned by an Amazon Web Services account in an Amazon Web Services Region. The returned list
     * is ordered by gateway Amazon Resource Name (ARN).
     * </p>
     *
     * @param listGatewaysRequest
     * @return A Java Future containing the result of the ListGateways operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The operation did not succeed because a validation error occurred.</li>
     *         <li>InternalServerException The operation did not succeed because an internal error occurred. Try again
     *         later.</li>
     *         <li>ThrottlingException TPS has been limited to protect against intentional or unintentional high request
     *         volumes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupGatewayException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupGatewayAsyncClient.ListGateways
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-gateway-2021-01-01/ListGateways" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListGatewaysResponse> listGateways(ListGatewaysRequest listGatewaysRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listGatewaysRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listGatewaysRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup Gateway");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListGateways");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListGatewaysResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListGatewaysResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListGatewaysResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListGatewaysRequest, ListGatewaysResponse>()
                            .withOperationName("ListGateways").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListGatewaysRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listGatewaysRequest));
            CompletableFuture<ListGatewaysResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists your hypervisors.
     * </p>
     *
     * @param listHypervisorsRequest
     * @return A Java Future containing the result of the ListHypervisors operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The operation did not succeed because a validation error occurred.</li>
     *         <li>InternalServerException The operation did not succeed because an internal error occurred. Try again
     *         later.</li>
     *         <li>ThrottlingException TPS has been limited to protect against intentional or unintentional high request
     *         volumes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupGatewayException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupGatewayAsyncClient.ListHypervisors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-gateway-2021-01-01/ListHypervisors"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListHypervisorsResponse> listHypervisors(ListHypervisorsRequest listHypervisorsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listHypervisorsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listHypervisorsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup Gateway");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListHypervisors");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListHypervisorsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListHypervisorsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListHypervisorsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListHypervisorsRequest, ListHypervisorsResponse>()
                            .withOperationName("ListHypervisors").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListHypervisorsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listHypervisorsRequest));
            CompletableFuture<ListHypervisorsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the tags applied to the resource identified by its Amazon Resource Name (ARN).
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The operation did not succeed because a validation error occurred.</li>
     *         <li>InternalServerException The operation did not succeed because an internal error occurred. Try again
     *         later.</li>
     *         <li>ResourceNotFoundException A resource that is required for the action wasn't found.</li>
     *         <li>ThrottlingException TPS has been limited to protect against intentional or unintentional high request
     *         volumes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupGatewayException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupGatewayAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-gateway-2021-01-01/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup Gateway");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists your virtual machines.
     * </p>
     *
     * @param listVirtualMachinesRequest
     * @return A Java Future containing the result of the ListVirtualMachines operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The operation did not succeed because a validation error occurred.</li>
     *         <li>InternalServerException The operation did not succeed because an internal error occurred. Try again
     *         later.</li>
     *         <li>ThrottlingException TPS has been limited to protect against intentional or unintentional high request
     *         volumes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupGatewayException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupGatewayAsyncClient.ListVirtualMachines
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-gateway-2021-01-01/ListVirtualMachines"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListVirtualMachinesResponse> listVirtualMachines(
            ListVirtualMachinesRequest listVirtualMachinesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listVirtualMachinesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listVirtualMachinesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup Gateway");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListVirtualMachines");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListVirtualMachinesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListVirtualMachinesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListVirtualMachinesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListVirtualMachinesRequest, ListVirtualMachinesResponse>()
                            .withOperationName("ListVirtualMachines").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListVirtualMachinesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listVirtualMachinesRequest));
            CompletableFuture<ListVirtualMachinesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This action sets the bandwidth rate limit schedule for a specified gateway. By default, gateways do not have a
     * bandwidth rate limit schedule, which means no bandwidth rate limiting is in effect. Use this to initiate a
     * gateway's bandwidth rate limit schedule.
     * </p>
     *
     * @param putBandwidthRateLimitScheduleRequest
     * @return A Java Future containing the result of the PutBandwidthRateLimitSchedule operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The operation did not succeed because a validation error occurred.</li>
     *         <li>InternalServerException The operation did not succeed because an internal error occurred. Try again
     *         later.</li>
     *         <li>ResourceNotFoundException A resource that is required for the action wasn't found.</li>
     *         <li>ThrottlingException TPS has been limited to protect against intentional or unintentional high request
     *         volumes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupGatewayException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupGatewayAsyncClient.PutBandwidthRateLimitSchedule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-gateway-2021-01-01/PutBandwidthRateLimitSchedule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutBandwidthRateLimitScheduleResponse> putBandwidthRateLimitSchedule(
            PutBandwidthRateLimitScheduleRequest putBandwidthRateLimitScheduleRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putBandwidthRateLimitScheduleRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                putBandwidthRateLimitScheduleRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup Gateway");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutBandwidthRateLimitSchedule");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutBandwidthRateLimitScheduleResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutBandwidthRateLimitScheduleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutBandwidthRateLimitScheduleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutBandwidthRateLimitScheduleRequest, PutBandwidthRateLimitScheduleResponse>()
                            .withOperationName("PutBandwidthRateLimitSchedule").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PutBandwidthRateLimitScheduleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putBandwidthRateLimitScheduleRequest));
            CompletableFuture<PutBandwidthRateLimitScheduleResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This action sets the property mappings for the specified hypervisor. A hypervisor property mapping displays the
     * relationship of entity properties available from the on-premises hypervisor to the properties available in Amazon
     * Web Services.
     * </p>
     *
     * @param putHypervisorPropertyMappingsRequest
     * @return A Java Future containing the result of the PutHypervisorPropertyMappings operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The operation did not succeed because a validation error occurred.</li>
     *         <li>ConflictException The operation cannot proceed because it is not supported.</li>
     *         <li>InternalServerException The operation did not succeed because an internal error occurred. Try again
     *         later.</li>
     *         <li>AccessDeniedException The operation cannot proceed because you have insufficient permissions.</li>
     *         <li>ResourceNotFoundException A resource that is required for the action wasn't found.</li>
     *         <li>ThrottlingException TPS has been limited to protect against intentional or unintentional high request
     *         volumes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupGatewayException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupGatewayAsyncClient.PutHypervisorPropertyMappings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-gateway-2021-01-01/PutHypervisorPropertyMappings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutHypervisorPropertyMappingsResponse> putHypervisorPropertyMappings(
            PutHypervisorPropertyMappingsRequest putHypervisorPropertyMappingsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putHypervisorPropertyMappingsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                putHypervisorPropertyMappingsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup Gateway");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutHypervisorPropertyMappings");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutHypervisorPropertyMappingsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutHypervisorPropertyMappingsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutHypervisorPropertyMappingsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutHypervisorPropertyMappingsRequest, PutHypervisorPropertyMappingsResponse>()
                            .withOperationName("PutHypervisorPropertyMappings").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PutHypervisorPropertyMappingsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putHypervisorPropertyMappingsRequest));
            CompletableFuture<PutHypervisorPropertyMappingsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Set the maintenance start time for a gateway.
     * </p>
     *
     * @param putMaintenanceStartTimeRequest
     * @return A Java Future containing the result of the PutMaintenanceStartTime operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The operation did not succeed because a validation error occurred.</li>
     *         <li>ConflictException The operation cannot proceed because it is not supported.</li>
     *         <li>InternalServerException The operation did not succeed because an internal error occurred. Try again
     *         later.</li>
     *         <li>ResourceNotFoundException A resource that is required for the action wasn't found.</li>
     *         <li>ThrottlingException TPS has been limited to protect against intentional or unintentional high request
     *         volumes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupGatewayException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupGatewayAsyncClient.PutMaintenanceStartTime
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-gateway-2021-01-01/PutMaintenanceStartTime"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutMaintenanceStartTimeResponse> putMaintenanceStartTime(
            PutMaintenanceStartTimeRequest putMaintenanceStartTimeRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putMaintenanceStartTimeRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putMaintenanceStartTimeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup Gateway");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutMaintenanceStartTime");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutMaintenanceStartTimeResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutMaintenanceStartTimeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutMaintenanceStartTimeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutMaintenanceStartTimeRequest, PutMaintenanceStartTimeResponse>()
                            .withOperationName("PutMaintenanceStartTime").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PutMaintenanceStartTimeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putMaintenanceStartTimeRequest));
            CompletableFuture<PutMaintenanceStartTimeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This action sends a request to sync metadata across the specified virtual machines.
     * </p>
     *
     * @param startVirtualMachinesMetadataSyncRequest
     * @return A Java Future containing the result of the StartVirtualMachinesMetadataSync operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The operation did not succeed because a validation error occurred.</li>
     *         <li>InternalServerException The operation did not succeed because an internal error occurred. Try again
     *         later.</li>
     *         <li>AccessDeniedException The operation cannot proceed because you have insufficient permissions.</li>
     *         <li>ResourceNotFoundException A resource that is required for the action wasn't found.</li>
     *         <li>ThrottlingException TPS has been limited to protect against intentional or unintentional high request
     *         volumes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupGatewayException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupGatewayAsyncClient.StartVirtualMachinesMetadataSync
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-gateway-2021-01-01/StartVirtualMachinesMetadataSync"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartVirtualMachinesMetadataSyncResponse> startVirtualMachinesMetadataSync(
            StartVirtualMachinesMetadataSyncRequest startVirtualMachinesMetadataSyncRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startVirtualMachinesMetadataSyncRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                startVirtualMachinesMetadataSyncRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup Gateway");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartVirtualMachinesMetadataSync");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartVirtualMachinesMetadataSyncResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, StartVirtualMachinesMetadataSyncResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartVirtualMachinesMetadataSyncResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartVirtualMachinesMetadataSyncRequest, StartVirtualMachinesMetadataSyncResponse>()
                            .withOperationName("StartVirtualMachinesMetadataSync").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StartVirtualMachinesMetadataSyncRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(startVirtualMachinesMetadataSyncRequest));
            CompletableFuture<StartVirtualMachinesMetadataSyncResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Tag the resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The operation did not succeed because a validation error occurred.</li>
     *         <li>InternalServerException The operation did not succeed because an internal error occurred. Try again
     *         later.</li>
     *         <li>ResourceNotFoundException A resource that is required for the action wasn't found.</li>
     *         <li>ThrottlingException TPS has been limited to protect against intentional or unintentional high request
     *         volumes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupGatewayException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupGatewayAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-gateway-2021-01-01/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup Gateway");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Tests your hypervisor configuration to validate that backup gateway can connect with the hypervisor and its
     * resources.
     * </p>
     *
     * @param testHypervisorConfigurationRequest
     * @return A Java Future containing the result of the TestHypervisorConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The operation did not succeed because a validation error occurred.</li>
     *         <li>ConflictException The operation cannot proceed because it is not supported.</li>
     *         <li>InternalServerException The operation did not succeed because an internal error occurred. Try again
     *         later.</li>
     *         <li>ResourceNotFoundException A resource that is required for the action wasn't found.</li>
     *         <li>ThrottlingException TPS has been limited to protect against intentional or unintentional high request
     *         volumes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupGatewayException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupGatewayAsyncClient.TestHypervisorConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-gateway-2021-01-01/TestHypervisorConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<TestHypervisorConfigurationResponse> testHypervisorConfiguration(
            TestHypervisorConfigurationRequest testHypervisorConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(testHypervisorConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, testHypervisorConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup Gateway");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TestHypervisorConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TestHypervisorConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, TestHypervisorConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TestHypervisorConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TestHypervisorConfigurationRequest, TestHypervisorConfigurationResponse>()
                            .withOperationName("TestHypervisorConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TestHypervisorConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(testHypervisorConfigurationRequest));
            CompletableFuture<TestHypervisorConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes tags from the resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The operation did not succeed because a validation error occurred.</li>
     *         <li>InternalServerException The operation did not succeed because an internal error occurred. Try again
     *         later.</li>
     *         <li>ResourceNotFoundException A resource that is required for the action wasn't found.</li>
     *         <li>ThrottlingException TPS has been limited to protect against intentional or unintentional high request
     *         volumes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupGatewayException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupGatewayAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-gateway-2021-01-01/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup Gateway");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a gateway's name. Specify which gateway to update using the Amazon Resource Name (ARN) of the gateway in
     * your request.
     * </p>
     *
     * @param updateGatewayInformationRequest
     * @return A Java Future containing the result of the UpdateGatewayInformation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The operation did not succeed because a validation error occurred.</li>
     *         <li>ConflictException The operation cannot proceed because it is not supported.</li>
     *         <li>InternalServerException The operation did not succeed because an internal error occurred. Try again
     *         later.</li>
     *         <li>ResourceNotFoundException A resource that is required for the action wasn't found.</li>
     *         <li>ThrottlingException TPS has been limited to protect against intentional or unintentional high request
     *         volumes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupGatewayException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupGatewayAsyncClient.UpdateGatewayInformation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-gateway-2021-01-01/UpdateGatewayInformation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateGatewayInformationResponse> updateGatewayInformation(
            UpdateGatewayInformationRequest updateGatewayInformationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateGatewayInformationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateGatewayInformationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup Gateway");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateGatewayInformation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateGatewayInformationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateGatewayInformationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateGatewayInformationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateGatewayInformationRequest, UpdateGatewayInformationResponse>()
                            .withOperationName("UpdateGatewayInformation").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateGatewayInformationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateGatewayInformationRequest));
            CompletableFuture<UpdateGatewayInformationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the gateway virtual machine (VM) software. The request immediately triggers the software update.
     * </p>
     * <note>
     * <p>
     * When you make this request, you get a <code>200 OK</code> success response immediately. However, it might take
     * some time for the update to complete.
     * </p>
     * </note>
     *
     * @param updateGatewaySoftwareNowRequest
     * @return A Java Future containing the result of the UpdateGatewaySoftwareNow operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The operation did not succeed because a validation error occurred.</li>
     *         <li>InternalServerException The operation did not succeed because an internal error occurred. Try again
     *         later.</li>
     *         <li>ResourceNotFoundException A resource that is required for the action wasn't found.</li>
     *         <li>ThrottlingException TPS has been limited to protect against intentional or unintentional high request
     *         volumes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupGatewayException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupGatewayAsyncClient.UpdateGatewaySoftwareNow
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-gateway-2021-01-01/UpdateGatewaySoftwareNow"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateGatewaySoftwareNowResponse> updateGatewaySoftwareNow(
            UpdateGatewaySoftwareNowRequest updateGatewaySoftwareNowRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateGatewaySoftwareNowRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateGatewaySoftwareNowRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup Gateway");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateGatewaySoftwareNow");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateGatewaySoftwareNowResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateGatewaySoftwareNowResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateGatewaySoftwareNowResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateGatewaySoftwareNowRequest, UpdateGatewaySoftwareNowResponse>()
                            .withOperationName("UpdateGatewaySoftwareNow").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateGatewaySoftwareNowRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateGatewaySoftwareNowRequest));
            CompletableFuture<UpdateGatewaySoftwareNowResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a hypervisor metadata, including its host, username, and password. Specify which hypervisor to update
     * using the Amazon Resource Name (ARN) of the hypervisor in your request.
     * </p>
     *
     * @param updateHypervisorRequest
     * @return A Java Future containing the result of the UpdateHypervisor operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The operation did not succeed because a validation error occurred.</li>
     *         <li>ConflictException The operation cannot proceed because it is not supported.</li>
     *         <li>InternalServerException The operation did not succeed because an internal error occurred. Try again
     *         later.</li>
     *         <li>AccessDeniedException The operation cannot proceed because you have insufficient permissions.</li>
     *         <li>ResourceNotFoundException A resource that is required for the action wasn't found.</li>
     *         <li>ThrottlingException TPS has been limited to protect against intentional or unintentional high request
     *         volumes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupGatewayException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupGatewayAsyncClient.UpdateHypervisor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-gateway-2021-01-01/UpdateHypervisor"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateHypervisorResponse> updateHypervisor(UpdateHypervisorRequest updateHypervisorRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateHypervisorRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateHypervisorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup Gateway");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateHypervisor");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateHypervisorResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateHypervisorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateHypervisorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateHypervisorRequest, UpdateHypervisorResponse>()
                            .withOperationName("UpdateHypervisor").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateHypervisorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateHypervisorRequest));
            CompletableFuture<UpdateHypervisorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final BackupGatewayServiceClientConfiguration serviceClientConfiguration() {
        return new BackupGatewayServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(BackupGatewayException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.0")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        BackupGatewayServiceClientConfigurationBuilder serviceConfigBuilder = new BackupGatewayServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
