/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.backupstorage;

import java.nio.file.Path;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsClient;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ResponseBytes;
import software.amazon.awssdk.core.ResponseInputStream;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.sync.RequestBody;
import software.amazon.awssdk.core.sync.ResponseTransformer;
import software.amazon.awssdk.regions.ServiceMetadata;
import software.amazon.awssdk.services.backupstorage.model.AccessDeniedException;
import software.amazon.awssdk.services.backupstorage.model.BackupStorageException;
import software.amazon.awssdk.services.backupstorage.model.DataAlreadyExistsException;
import software.amazon.awssdk.services.backupstorage.model.DeleteObjectRequest;
import software.amazon.awssdk.services.backupstorage.model.DeleteObjectResponse;
import software.amazon.awssdk.services.backupstorage.model.GetChunkRequest;
import software.amazon.awssdk.services.backupstorage.model.GetChunkResponse;
import software.amazon.awssdk.services.backupstorage.model.GetObjectMetadataRequest;
import software.amazon.awssdk.services.backupstorage.model.GetObjectMetadataResponse;
import software.amazon.awssdk.services.backupstorage.model.IllegalArgumentException;
import software.amazon.awssdk.services.backupstorage.model.KmsInvalidKeyUsageException;
import software.amazon.awssdk.services.backupstorage.model.ListChunksRequest;
import software.amazon.awssdk.services.backupstorage.model.ListChunksResponse;
import software.amazon.awssdk.services.backupstorage.model.ListObjectsRequest;
import software.amazon.awssdk.services.backupstorage.model.ListObjectsResponse;
import software.amazon.awssdk.services.backupstorage.model.NotReadableInputStreamException;
import software.amazon.awssdk.services.backupstorage.model.NotifyObjectCompleteRequest;
import software.amazon.awssdk.services.backupstorage.model.NotifyObjectCompleteResponse;
import software.amazon.awssdk.services.backupstorage.model.PutChunkRequest;
import software.amazon.awssdk.services.backupstorage.model.PutChunkResponse;
import software.amazon.awssdk.services.backupstorage.model.PutObjectRequest;
import software.amazon.awssdk.services.backupstorage.model.PutObjectResponse;
import software.amazon.awssdk.services.backupstorage.model.ResourceNotFoundException;
import software.amazon.awssdk.services.backupstorage.model.RetryableException;
import software.amazon.awssdk.services.backupstorage.model.ServiceInternalException;
import software.amazon.awssdk.services.backupstorage.model.ServiceUnavailableException;
import software.amazon.awssdk.services.backupstorage.model.StartObjectRequest;
import software.amazon.awssdk.services.backupstorage.model.StartObjectResponse;
import software.amazon.awssdk.services.backupstorage.model.ThrottlingException;
import software.amazon.awssdk.services.backupstorage.paginators.ListChunksIterable;
import software.amazon.awssdk.services.backupstorage.paginators.ListObjectsIterable;

/**
 * Service client for accessing AWS Backup Storage. This can be created using the static {@link #builder()} method.
 *
 * The frontend service for Cryo Storage.
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
public interface BackupStorageClient extends AwsClient {
    String SERVICE_NAME = "backup-storage";

    /**
     * Value for looking up the service's metadata from the
     * {@link software.amazon.awssdk.regions.ServiceMetadataProvider}.
     */
    String SERVICE_METADATA_ID = "backupstorage";

    /**
     * Delete Object from the incremental base Backup.
     *
     * @param deleteObjectRequest
     * @return Result of the DeleteObject operation returned by the service.
     * @throws ServiceUnavailableException
     *         Retryable exception, indicates internal server error.
     * @throws ServiceInternalException
     *         Deprecated. To be removed from the model.
     * @throws RetryableException
     *         Retryable exception. In general indicates internal failure that can be fixed by retry.
     * @throws IllegalArgumentException
     *         Non-retryable exception, indicates client error (wrong argument passed to API). See exception message for
     *         details.
     * @throws ResourceNotFoundException
     *         Non-retryable exception. Attempted to make an operation on non-existing or expired resource.
     * @throws ThrottlingException
     *         Increased rate over throttling limits. Can be retried with exponential backoff.
     * @throws AccessDeniedException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupStorageException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupStorageClient.DeleteObject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/DeleteObject" target="_top">AWS
     *      API Documentation</a>
     */
    default DeleteObjectResponse deleteObject(DeleteObjectRequest deleteObjectRequest) throws ServiceUnavailableException,
            ServiceInternalException, RetryableException, IllegalArgumentException, ResourceNotFoundException,
            ThrottlingException, AccessDeniedException, AwsServiceException, SdkClientException, BackupStorageException {
        throw new UnsupportedOperationException();
    }

    /**
     * Delete Object from the incremental base Backup.<br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteObjectRequest.Builder} avoiding the need to
     * create one manually via {@link DeleteObjectRequest#builder()}
     * </p>
     *
     * @param deleteObjectRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.backupstorage.model.DeleteObjectRequest.Builder} to create a
     *        request.
     * @return Result of the DeleteObject operation returned by the service.
     * @throws ServiceUnavailableException
     *         Retryable exception, indicates internal server error.
     * @throws ServiceInternalException
     *         Deprecated. To be removed from the model.
     * @throws RetryableException
     *         Retryable exception. In general indicates internal failure that can be fixed by retry.
     * @throws IllegalArgumentException
     *         Non-retryable exception, indicates client error (wrong argument passed to API). See exception message for
     *         details.
     * @throws ResourceNotFoundException
     *         Non-retryable exception. Attempted to make an operation on non-existing or expired resource.
     * @throws ThrottlingException
     *         Increased rate over throttling limits. Can be retried with exponential backoff.
     * @throws AccessDeniedException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupStorageException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupStorageClient.DeleteObject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/DeleteObject" target="_top">AWS
     *      API Documentation</a>
     */
    default DeleteObjectResponse deleteObject(Consumer<DeleteObjectRequest.Builder> deleteObjectRequest)
            throws ServiceUnavailableException, ServiceInternalException, RetryableException, IllegalArgumentException,
            ResourceNotFoundException, ThrottlingException, AccessDeniedException, AwsServiceException, SdkClientException,
            BackupStorageException {
        return deleteObject(DeleteObjectRequest.builder().applyMutation(deleteObjectRequest).build());
    }

    /**
     * Gets the specified object's chunk.
     *
     * @param getChunkRequest
     * @param responseTransformer
     *        Functional interface for processing the streamed response content. The unmarshalled GetChunkResponse and
     *        an InputStream to the response content are provided as parameters to the callback. The callback may return
     *        a transformed type which will be the return value of this method. See
     *        {@link software.amazon.awssdk.core.sync.ResponseTransformer} for details on implementing this interface
     *        and for links to pre-canned implementations for common scenarios like downloading to a file. The service
     *        documentation for the response content is as follows 'Chunk data'.
     * @return The transformed result of the ResponseTransformer.
     * @throws IllegalArgumentException
     *         Non-retryable exception, indicates client error (wrong argument passed to API). See exception message for
     *         details.
     * @throws RetryableException
     *         Retryable exception. In general indicates internal failure that can be fixed by retry.
     * @throws ResourceNotFoundException
     *         Non-retryable exception. Attempted to make an operation on non-existing or expired resource.
     * @throws ServiceInternalException
     *         Deprecated. To be removed from the model.
     * @throws ThrottlingException
     *         Increased rate over throttling limits. Can be retried with exponential backoff.
     * @throws KmsInvalidKeyUsageException
     *         Non-retryable exception. Indicates the KMS key usage is incorrect. See exception message for details.
     * @throws AccessDeniedException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupStorageException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupStorageClient.GetChunk
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/GetChunk" target="_top">AWS API
     *      Documentation</a>
     */
    default <ReturnT> ReturnT getChunk(GetChunkRequest getChunkRequest,
            ResponseTransformer<GetChunkResponse, ReturnT> responseTransformer) throws IllegalArgumentException,
            RetryableException, ResourceNotFoundException, ServiceInternalException, ThrottlingException,
            KmsInvalidKeyUsageException, AccessDeniedException, AwsServiceException, SdkClientException, BackupStorageException {
        throw new UnsupportedOperationException();
    }

    /**
     * Gets the specified object's chunk.<br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetChunkRequest.Builder} avoiding the need to
     * create one manually via {@link GetChunkRequest#builder()}
     * </p>
     *
     * @param getChunkRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.backupstorage.model.GetChunkRequest.Builder} to create a request.
     * @param responseTransformer
     *        Functional interface for processing the streamed response content. The unmarshalled GetChunkResponse and
     *        an InputStream to the response content are provided as parameters to the callback. The callback may return
     *        a transformed type which will be the return value of this method. See
     *        {@link software.amazon.awssdk.core.sync.ResponseTransformer} for details on implementing this interface
     *        and for links to pre-canned implementations for common scenarios like downloading to a file. The service
     *        documentation for the response content is as follows 'Chunk data'.
     * @return The transformed result of the ResponseTransformer.
     * @throws IllegalArgumentException
     *         Non-retryable exception, indicates client error (wrong argument passed to API). See exception message for
     *         details.
     * @throws RetryableException
     *         Retryable exception. In general indicates internal failure that can be fixed by retry.
     * @throws ResourceNotFoundException
     *         Non-retryable exception. Attempted to make an operation on non-existing or expired resource.
     * @throws ServiceInternalException
     *         Deprecated. To be removed from the model.
     * @throws ThrottlingException
     *         Increased rate over throttling limits. Can be retried with exponential backoff.
     * @throws KmsInvalidKeyUsageException
     *         Non-retryable exception. Indicates the KMS key usage is incorrect. See exception message for details.
     * @throws AccessDeniedException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupStorageException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupStorageClient.GetChunk
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/GetChunk" target="_top">AWS API
     *      Documentation</a>
     */
    default <ReturnT> ReturnT getChunk(Consumer<GetChunkRequest.Builder> getChunkRequest,
            ResponseTransformer<GetChunkResponse, ReturnT> responseTransformer) throws IllegalArgumentException,
            RetryableException, ResourceNotFoundException, ServiceInternalException, ThrottlingException,
            KmsInvalidKeyUsageException, AccessDeniedException, AwsServiceException, SdkClientException, BackupStorageException {
        return getChunk(GetChunkRequest.builder().applyMutation(getChunkRequest).build(), responseTransformer);
    }

    /**
     * Gets the specified object's chunk.
     *
     * @param getChunkRequest
     * @param destinationPath
     *        {@link Path} to file that response contents will be written to. The file must not exist or this method
     *        will throw an exception. If the file is not writable by the current user then an exception will be thrown.
     *        The service documentation for the response content is as follows 'Chunk data'.
     * @return The transformed result of the ResponseTransformer.
     * @throws IllegalArgumentException
     *         Non-retryable exception, indicates client error (wrong argument passed to API). See exception message for
     *         details.
     * @throws RetryableException
     *         Retryable exception. In general indicates internal failure that can be fixed by retry.
     * @throws ResourceNotFoundException
     *         Non-retryable exception. Attempted to make an operation on non-existing or expired resource.
     * @throws ServiceInternalException
     *         Deprecated. To be removed from the model.
     * @throws ThrottlingException
     *         Increased rate over throttling limits. Can be retried with exponential backoff.
     * @throws KmsInvalidKeyUsageException
     *         Non-retryable exception. Indicates the KMS key usage is incorrect. See exception message for details.
     * @throws AccessDeniedException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupStorageException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupStorageClient.GetChunk
     * @see #getChunk(GetChunkRequest, ResponseTransformer)
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/GetChunk" target="_top">AWS API
     *      Documentation</a>
     */
    default GetChunkResponse getChunk(GetChunkRequest getChunkRequest, Path destinationPath) throws IllegalArgumentException,
            RetryableException, ResourceNotFoundException, ServiceInternalException, ThrottlingException,
            KmsInvalidKeyUsageException, AccessDeniedException, AwsServiceException, SdkClientException, BackupStorageException {
        return getChunk(getChunkRequest, ResponseTransformer.toFile(destinationPath));
    }

    /**
     * Gets the specified object's chunk.<br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetChunkRequest.Builder} avoiding the need to
     * create one manually via {@link GetChunkRequest#builder()}
     * </p>
     *
     * @param getChunkRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.backupstorage.model.GetChunkRequest.Builder} to create a request.
     * @param destinationPath
     *        {@link Path} to file that response contents will be written to. The file must not exist or this method
     *        will throw an exception. If the file is not writable by the current user then an exception will be thrown.
     *        The service documentation for the response content is as follows 'Chunk data'.
     * @return The transformed result of the ResponseTransformer.
     * @throws IllegalArgumentException
     *         Non-retryable exception, indicates client error (wrong argument passed to API). See exception message for
     *         details.
     * @throws RetryableException
     *         Retryable exception. In general indicates internal failure that can be fixed by retry.
     * @throws ResourceNotFoundException
     *         Non-retryable exception. Attempted to make an operation on non-existing or expired resource.
     * @throws ServiceInternalException
     *         Deprecated. To be removed from the model.
     * @throws ThrottlingException
     *         Increased rate over throttling limits. Can be retried with exponential backoff.
     * @throws KmsInvalidKeyUsageException
     *         Non-retryable exception. Indicates the KMS key usage is incorrect. See exception message for details.
     * @throws AccessDeniedException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupStorageException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupStorageClient.GetChunk
     * @see #getChunk(GetChunkRequest, ResponseTransformer)
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/GetChunk" target="_top">AWS API
     *      Documentation</a>
     */
    default GetChunkResponse getChunk(Consumer<GetChunkRequest.Builder> getChunkRequest, Path destinationPath)
            throws IllegalArgumentException, RetryableException, ResourceNotFoundException, ServiceInternalException,
            ThrottlingException, KmsInvalidKeyUsageException, AccessDeniedException, AwsServiceException, SdkClientException,
            BackupStorageException {
        return getChunk(GetChunkRequest.builder().applyMutation(getChunkRequest).build(), destinationPath);
    }

    /**
     * Gets the specified object's chunk.
     *
     * @param getChunkRequest
     * @return A {@link ResponseInputStream} containing data streamed from service. Note that this is an unmanaged
     *         reference to the underlying HTTP connection so great care must be taken to ensure all data if fully read
     *         from the input stream and that it is properly closed. Failure to do so may result in sub-optimal behavior
     *         and exhausting connections in the connection pool. The unmarshalled response object can be obtained via
     *         {@link ResponseInputStream#response()}. The service documentation for the response content is as follows
     *         'Chunk data'.
     * @throws IllegalArgumentException
     *         Non-retryable exception, indicates client error (wrong argument passed to API). See exception message for
     *         details.
     * @throws RetryableException
     *         Retryable exception. In general indicates internal failure that can be fixed by retry.
     * @throws ResourceNotFoundException
     *         Non-retryable exception. Attempted to make an operation on non-existing or expired resource.
     * @throws ServiceInternalException
     *         Deprecated. To be removed from the model.
     * @throws ThrottlingException
     *         Increased rate over throttling limits. Can be retried with exponential backoff.
     * @throws KmsInvalidKeyUsageException
     *         Non-retryable exception. Indicates the KMS key usage is incorrect. See exception message for details.
     * @throws AccessDeniedException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupStorageException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupStorageClient.GetChunk
     * @see #getObject(getChunk, ResponseTransformer)
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/GetChunk" target="_top">AWS API
     *      Documentation</a>
     */
    default ResponseInputStream<GetChunkResponse> getChunk(GetChunkRequest getChunkRequest) throws IllegalArgumentException,
            RetryableException, ResourceNotFoundException, ServiceInternalException, ThrottlingException,
            KmsInvalidKeyUsageException, AccessDeniedException, AwsServiceException, SdkClientException, BackupStorageException {
        return getChunk(getChunkRequest, ResponseTransformer.toInputStream());
    }

    /**
     * Gets the specified object's chunk.<br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetChunkRequest.Builder} avoiding the need to
     * create one manually via {@link GetChunkRequest#builder()}
     * </p>
     *
     * @param getChunkRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.backupstorage.model.GetChunkRequest.Builder} to create a request.
     * @return A {@link ResponseInputStream} containing data streamed from service. Note that this is an unmanaged
     *         reference to the underlying HTTP connection so great care must be taken to ensure all data if fully read
     *         from the input stream and that it is properly closed. Failure to do so may result in sub-optimal behavior
     *         and exhausting connections in the connection pool. The unmarshalled response object can be obtained via
     *         {@link ResponseInputStream#response()}. The service documentation for the response content is as follows
     *         'Chunk data'.
     * @throws IllegalArgumentException
     *         Non-retryable exception, indicates client error (wrong argument passed to API). See exception message for
     *         details.
     * @throws RetryableException
     *         Retryable exception. In general indicates internal failure that can be fixed by retry.
     * @throws ResourceNotFoundException
     *         Non-retryable exception. Attempted to make an operation on non-existing or expired resource.
     * @throws ServiceInternalException
     *         Deprecated. To be removed from the model.
     * @throws ThrottlingException
     *         Increased rate over throttling limits. Can be retried with exponential backoff.
     * @throws KmsInvalidKeyUsageException
     *         Non-retryable exception. Indicates the KMS key usage is incorrect. See exception message for details.
     * @throws AccessDeniedException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupStorageException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupStorageClient.GetChunk
     * @see #getObject(getChunk, ResponseTransformer)
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/GetChunk" target="_top">AWS API
     *      Documentation</a>
     */
    default ResponseInputStream<GetChunkResponse> getChunk(Consumer<GetChunkRequest.Builder> getChunkRequest)
            throws IllegalArgumentException, RetryableException, ResourceNotFoundException, ServiceInternalException,
            ThrottlingException, KmsInvalidKeyUsageException, AccessDeniedException, AwsServiceException, SdkClientException,
            BackupStorageException {
        return getChunk(GetChunkRequest.builder().applyMutation(getChunkRequest).build());
    }

    /**
     * Gets the specified object's chunk.
     *
     * @param getChunkRequest
     * @return A {@link ResponseBytes} that loads the data streamed from the service into memory and exposes it in
     *         convenient in-memory representations like a byte buffer or string. The unmarshalled response object can
     *         be obtained via {@link ResponseBytes#response()}. The service documentation for the response content is
     *         as follows 'Chunk data'.
     * @throws IllegalArgumentException
     *         Non-retryable exception, indicates client error (wrong argument passed to API). See exception message for
     *         details.
     * @throws RetryableException
     *         Retryable exception. In general indicates internal failure that can be fixed by retry.
     * @throws ResourceNotFoundException
     *         Non-retryable exception. Attempted to make an operation on non-existing or expired resource.
     * @throws ServiceInternalException
     *         Deprecated. To be removed from the model.
     * @throws ThrottlingException
     *         Increased rate over throttling limits. Can be retried with exponential backoff.
     * @throws KmsInvalidKeyUsageException
     *         Non-retryable exception. Indicates the KMS key usage is incorrect. See exception message for details.
     * @throws AccessDeniedException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupStorageException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupStorageClient.GetChunk
     * @see #getObject(getChunk, ResponseTransformer)
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/GetChunk" target="_top">AWS API
     *      Documentation</a>
     */
    default ResponseBytes<GetChunkResponse> getChunkAsBytes(GetChunkRequest getChunkRequest) throws IllegalArgumentException,
            RetryableException, ResourceNotFoundException, ServiceInternalException, ThrottlingException,
            KmsInvalidKeyUsageException, AccessDeniedException, AwsServiceException, SdkClientException, BackupStorageException {
        return getChunk(getChunkRequest, ResponseTransformer.toBytes());
    }

    /**
     * Gets the specified object's chunk.<br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetChunkRequest.Builder} avoiding the need to
     * create one manually via {@link GetChunkRequest#builder()}
     * </p>
     *
     * @param getChunkRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.backupstorage.model.GetChunkRequest.Builder} to create a request.
     * @return A {@link ResponseBytes} that loads the data streamed from the service into memory and exposes it in
     *         convenient in-memory representations like a byte buffer or string. The unmarshalled response object can
     *         be obtained via {@link ResponseBytes#response()}. The service documentation for the response content is
     *         as follows 'Chunk data'.
     * @throws IllegalArgumentException
     *         Non-retryable exception, indicates client error (wrong argument passed to API). See exception message for
     *         details.
     * @throws RetryableException
     *         Retryable exception. In general indicates internal failure that can be fixed by retry.
     * @throws ResourceNotFoundException
     *         Non-retryable exception. Attempted to make an operation on non-existing or expired resource.
     * @throws ServiceInternalException
     *         Deprecated. To be removed from the model.
     * @throws ThrottlingException
     *         Increased rate over throttling limits. Can be retried with exponential backoff.
     * @throws KmsInvalidKeyUsageException
     *         Non-retryable exception. Indicates the KMS key usage is incorrect. See exception message for details.
     * @throws AccessDeniedException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupStorageException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupStorageClient.GetChunk
     * @see #getObject(getChunk, ResponseTransformer)
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/GetChunk" target="_top">AWS API
     *      Documentation</a>
     */
    default ResponseBytes<GetChunkResponse> getChunkAsBytes(Consumer<GetChunkRequest.Builder> getChunkRequest)
            throws IllegalArgumentException, RetryableException, ResourceNotFoundException, ServiceInternalException,
            ThrottlingException, KmsInvalidKeyUsageException, AccessDeniedException, AwsServiceException, SdkClientException,
            BackupStorageException {
        return getChunkAsBytes(GetChunkRequest.builder().applyMutation(getChunkRequest).build());
    }

    /**
     * Get metadata associated with an Object.
     *
     * @param getObjectMetadataRequest
     * @param responseTransformer
     *        Functional interface for processing the streamed response content. The unmarshalled
     *        GetObjectMetadataResponse and an InputStream to the response content are provided as parameters to the
     *        callback. The callback may return a transformed type which will be the return value of this method. See
     *        {@link software.amazon.awssdk.core.sync.ResponseTransformer} for details on implementing this interface
     *        and for links to pre-canned implementations for common scenarios like downloading to a file. The service
     *        documentation for the response content is as follows 'Metadata blob.'.
     * @return The transformed result of the ResponseTransformer.
     * @throws ServiceUnavailableException
     *         Retryable exception, indicates internal server error.
     * @throws ServiceInternalException
     *         Deprecated. To be removed from the model.
     * @throws ResourceNotFoundException
     *         Non-retryable exception. Attempted to make an operation on non-existing or expired resource.
     * @throws RetryableException
     *         Retryable exception. In general indicates internal failure that can be fixed by retry.
     * @throws IllegalArgumentException
     *         Non-retryable exception, indicates client error (wrong argument passed to API). See exception message for
     *         details.
     * @throws ThrottlingException
     *         Increased rate over throttling limits. Can be retried with exponential backoff.
     * @throws KmsInvalidKeyUsageException
     *         Non-retryable exception. Indicates the KMS key usage is incorrect. See exception message for details.
     * @throws AccessDeniedException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupStorageException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupStorageClient.GetObjectMetadata
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/GetObjectMetadata"
     *      target="_top">AWS API Documentation</a>
     */
    default <ReturnT> ReturnT getObjectMetadata(GetObjectMetadataRequest getObjectMetadataRequest,
            ResponseTransformer<GetObjectMetadataResponse, ReturnT> responseTransformer) throws ServiceUnavailableException,
            ServiceInternalException, ResourceNotFoundException, RetryableException, IllegalArgumentException,
            ThrottlingException, KmsInvalidKeyUsageException, AccessDeniedException, AwsServiceException, SdkClientException,
            BackupStorageException {
        throw new UnsupportedOperationException();
    }

    /**
     * Get metadata associated with an Object.<br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetObjectMetadataRequest.Builder} avoiding the need
     * to create one manually via {@link GetObjectMetadataRequest#builder()}
     * </p>
     *
     * @param getObjectMetadataRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.backupstorage.model.GetObjectMetadataRequest.Builder} to create a
     *        request.
     * @param responseTransformer
     *        Functional interface for processing the streamed response content. The unmarshalled
     *        GetObjectMetadataResponse and an InputStream to the response content are provided as parameters to the
     *        callback. The callback may return a transformed type which will be the return value of this method. See
     *        {@link software.amazon.awssdk.core.sync.ResponseTransformer} for details on implementing this interface
     *        and for links to pre-canned implementations for common scenarios like downloading to a file. The service
     *        documentation for the response content is as follows 'Metadata blob.'.
     * @return The transformed result of the ResponseTransformer.
     * @throws ServiceUnavailableException
     *         Retryable exception, indicates internal server error.
     * @throws ServiceInternalException
     *         Deprecated. To be removed from the model.
     * @throws ResourceNotFoundException
     *         Non-retryable exception. Attempted to make an operation on non-existing or expired resource.
     * @throws RetryableException
     *         Retryable exception. In general indicates internal failure that can be fixed by retry.
     * @throws IllegalArgumentException
     *         Non-retryable exception, indicates client error (wrong argument passed to API). See exception message for
     *         details.
     * @throws ThrottlingException
     *         Increased rate over throttling limits. Can be retried with exponential backoff.
     * @throws KmsInvalidKeyUsageException
     *         Non-retryable exception. Indicates the KMS key usage is incorrect. See exception message for details.
     * @throws AccessDeniedException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupStorageException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupStorageClient.GetObjectMetadata
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/GetObjectMetadata"
     *      target="_top">AWS API Documentation</a>
     */
    default <ReturnT> ReturnT getObjectMetadata(Consumer<GetObjectMetadataRequest.Builder> getObjectMetadataRequest,
            ResponseTransformer<GetObjectMetadataResponse, ReturnT> responseTransformer) throws ServiceUnavailableException,
            ServiceInternalException, ResourceNotFoundException, RetryableException, IllegalArgumentException,
            ThrottlingException, KmsInvalidKeyUsageException, AccessDeniedException, AwsServiceException, SdkClientException,
            BackupStorageException {
        return getObjectMetadata(GetObjectMetadataRequest.builder().applyMutation(getObjectMetadataRequest).build(),
                responseTransformer);
    }

    /**
     * Get metadata associated with an Object.
     *
     * @param getObjectMetadataRequest
     * @param destinationPath
     *        {@link Path} to file that response contents will be written to. The file must not exist or this method
     *        will throw an exception. If the file is not writable by the current user then an exception will be thrown.
     *        The service documentation for the response content is as follows 'Metadata blob.'.
     * @return The transformed result of the ResponseTransformer.
     * @throws ServiceUnavailableException
     *         Retryable exception, indicates internal server error.
     * @throws ServiceInternalException
     *         Deprecated. To be removed from the model.
     * @throws ResourceNotFoundException
     *         Non-retryable exception. Attempted to make an operation on non-existing or expired resource.
     * @throws RetryableException
     *         Retryable exception. In general indicates internal failure that can be fixed by retry.
     * @throws IllegalArgumentException
     *         Non-retryable exception, indicates client error (wrong argument passed to API). See exception message for
     *         details.
     * @throws ThrottlingException
     *         Increased rate over throttling limits. Can be retried with exponential backoff.
     * @throws KmsInvalidKeyUsageException
     *         Non-retryable exception. Indicates the KMS key usage is incorrect. See exception message for details.
     * @throws AccessDeniedException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupStorageException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupStorageClient.GetObjectMetadata
     * @see #getObjectMetadata(GetObjectMetadataRequest, ResponseTransformer)
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/GetObjectMetadata"
     *      target="_top">AWS API Documentation</a>
     */
    default GetObjectMetadataResponse getObjectMetadata(GetObjectMetadataRequest getObjectMetadataRequest, Path destinationPath)
            throws ServiceUnavailableException, ServiceInternalException, ResourceNotFoundException, RetryableException,
            IllegalArgumentException, ThrottlingException, KmsInvalidKeyUsageException, AccessDeniedException,
            AwsServiceException, SdkClientException, BackupStorageException {
        return getObjectMetadata(getObjectMetadataRequest, ResponseTransformer.toFile(destinationPath));
    }

    /**
     * Get metadata associated with an Object.<br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetObjectMetadataRequest.Builder} avoiding the need
     * to create one manually via {@link GetObjectMetadataRequest#builder()}
     * </p>
     *
     * @param getObjectMetadataRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.backupstorage.model.GetObjectMetadataRequest.Builder} to create a
     *        request.
     * @param destinationPath
     *        {@link Path} to file that response contents will be written to. The file must not exist or this method
     *        will throw an exception. If the file is not writable by the current user then an exception will be thrown.
     *        The service documentation for the response content is as follows 'Metadata blob.'.
     * @return The transformed result of the ResponseTransformer.
     * @throws ServiceUnavailableException
     *         Retryable exception, indicates internal server error.
     * @throws ServiceInternalException
     *         Deprecated. To be removed from the model.
     * @throws ResourceNotFoundException
     *         Non-retryable exception. Attempted to make an operation on non-existing or expired resource.
     * @throws RetryableException
     *         Retryable exception. In general indicates internal failure that can be fixed by retry.
     * @throws IllegalArgumentException
     *         Non-retryable exception, indicates client error (wrong argument passed to API). See exception message for
     *         details.
     * @throws ThrottlingException
     *         Increased rate over throttling limits. Can be retried with exponential backoff.
     * @throws KmsInvalidKeyUsageException
     *         Non-retryable exception. Indicates the KMS key usage is incorrect. See exception message for details.
     * @throws AccessDeniedException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupStorageException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupStorageClient.GetObjectMetadata
     * @see #getObjectMetadata(GetObjectMetadataRequest, ResponseTransformer)
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/GetObjectMetadata"
     *      target="_top">AWS API Documentation</a>
     */
    default GetObjectMetadataResponse getObjectMetadata(Consumer<GetObjectMetadataRequest.Builder> getObjectMetadataRequest,
            Path destinationPath) throws ServiceUnavailableException, ServiceInternalException, ResourceNotFoundException,
            RetryableException, IllegalArgumentException, ThrottlingException, KmsInvalidKeyUsageException,
            AccessDeniedException, AwsServiceException, SdkClientException, BackupStorageException {
        return getObjectMetadata(GetObjectMetadataRequest.builder().applyMutation(getObjectMetadataRequest).build(),
                destinationPath);
    }

    /**
     * Get metadata associated with an Object.
     *
     * @param getObjectMetadataRequest
     * @return A {@link ResponseInputStream} containing data streamed from service. Note that this is an unmanaged
     *         reference to the underlying HTTP connection so great care must be taken to ensure all data if fully read
     *         from the input stream and that it is properly closed. Failure to do so may result in sub-optimal behavior
     *         and exhausting connections in the connection pool. The unmarshalled response object can be obtained via
     *         {@link ResponseInputStream#response()}. The service documentation for the response content is as follows
     *         'Metadata blob.'.
     * @throws ServiceUnavailableException
     *         Retryable exception, indicates internal server error.
     * @throws ServiceInternalException
     *         Deprecated. To be removed from the model.
     * @throws ResourceNotFoundException
     *         Non-retryable exception. Attempted to make an operation on non-existing or expired resource.
     * @throws RetryableException
     *         Retryable exception. In general indicates internal failure that can be fixed by retry.
     * @throws IllegalArgumentException
     *         Non-retryable exception, indicates client error (wrong argument passed to API). See exception message for
     *         details.
     * @throws ThrottlingException
     *         Increased rate over throttling limits. Can be retried with exponential backoff.
     * @throws KmsInvalidKeyUsageException
     *         Non-retryable exception. Indicates the KMS key usage is incorrect. See exception message for details.
     * @throws AccessDeniedException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupStorageException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupStorageClient.GetObjectMetadata
     * @see #getObject(getObjectMetadata, ResponseTransformer)
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/GetObjectMetadata"
     *      target="_top">AWS API Documentation</a>
     */
    default ResponseInputStream<GetObjectMetadataResponse> getObjectMetadata(GetObjectMetadataRequest getObjectMetadataRequest)
            throws ServiceUnavailableException, ServiceInternalException, ResourceNotFoundException, RetryableException,
            IllegalArgumentException, ThrottlingException, KmsInvalidKeyUsageException, AccessDeniedException,
            AwsServiceException, SdkClientException, BackupStorageException {
        return getObjectMetadata(getObjectMetadataRequest, ResponseTransformer.toInputStream());
    }

    /**
     * Get metadata associated with an Object.<br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetObjectMetadataRequest.Builder} avoiding the need
     * to create one manually via {@link GetObjectMetadataRequest#builder()}
     * </p>
     *
     * @param getObjectMetadataRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.backupstorage.model.GetObjectMetadataRequest.Builder} to create a
     *        request.
     * @return A {@link ResponseInputStream} containing data streamed from service. Note that this is an unmanaged
     *         reference to the underlying HTTP connection so great care must be taken to ensure all data if fully read
     *         from the input stream and that it is properly closed. Failure to do so may result in sub-optimal behavior
     *         and exhausting connections in the connection pool. The unmarshalled response object can be obtained via
     *         {@link ResponseInputStream#response()}. The service documentation for the response content is as follows
     *         'Metadata blob.'.
     * @throws ServiceUnavailableException
     *         Retryable exception, indicates internal server error.
     * @throws ServiceInternalException
     *         Deprecated. To be removed from the model.
     * @throws ResourceNotFoundException
     *         Non-retryable exception. Attempted to make an operation on non-existing or expired resource.
     * @throws RetryableException
     *         Retryable exception. In general indicates internal failure that can be fixed by retry.
     * @throws IllegalArgumentException
     *         Non-retryable exception, indicates client error (wrong argument passed to API). See exception message for
     *         details.
     * @throws ThrottlingException
     *         Increased rate over throttling limits. Can be retried with exponential backoff.
     * @throws KmsInvalidKeyUsageException
     *         Non-retryable exception. Indicates the KMS key usage is incorrect. See exception message for details.
     * @throws AccessDeniedException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupStorageException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupStorageClient.GetObjectMetadata
     * @see #getObject(getObjectMetadata, ResponseTransformer)
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/GetObjectMetadata"
     *      target="_top">AWS API Documentation</a>
     */
    default ResponseInputStream<GetObjectMetadataResponse> getObjectMetadata(
            Consumer<GetObjectMetadataRequest.Builder> getObjectMetadataRequest) throws ServiceUnavailableException,
            ServiceInternalException, ResourceNotFoundException, RetryableException, IllegalArgumentException,
            ThrottlingException, KmsInvalidKeyUsageException, AccessDeniedException, AwsServiceException, SdkClientException,
            BackupStorageException {
        return getObjectMetadata(GetObjectMetadataRequest.builder().applyMutation(getObjectMetadataRequest).build());
    }

    /**
     * Get metadata associated with an Object.
     *
     * @param getObjectMetadataRequest
     * @return A {@link ResponseBytes} that loads the data streamed from the service into memory and exposes it in
     *         convenient in-memory representations like a byte buffer or string. The unmarshalled response object can
     *         be obtained via {@link ResponseBytes#response()}. The service documentation for the response content is
     *         as follows 'Metadata blob.'.
     * @throws ServiceUnavailableException
     *         Retryable exception, indicates internal server error.
     * @throws ServiceInternalException
     *         Deprecated. To be removed from the model.
     * @throws ResourceNotFoundException
     *         Non-retryable exception. Attempted to make an operation on non-existing or expired resource.
     * @throws RetryableException
     *         Retryable exception. In general indicates internal failure that can be fixed by retry.
     * @throws IllegalArgumentException
     *         Non-retryable exception, indicates client error (wrong argument passed to API). See exception message for
     *         details.
     * @throws ThrottlingException
     *         Increased rate over throttling limits. Can be retried with exponential backoff.
     * @throws KmsInvalidKeyUsageException
     *         Non-retryable exception. Indicates the KMS key usage is incorrect. See exception message for details.
     * @throws AccessDeniedException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupStorageException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupStorageClient.GetObjectMetadata
     * @see #getObject(getObjectMetadata, ResponseTransformer)
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/GetObjectMetadata"
     *      target="_top">AWS API Documentation</a>
     */
    default ResponseBytes<GetObjectMetadataResponse> getObjectMetadataAsBytes(GetObjectMetadataRequest getObjectMetadataRequest)
            throws ServiceUnavailableException, ServiceInternalException, ResourceNotFoundException, RetryableException,
            IllegalArgumentException, ThrottlingException, KmsInvalidKeyUsageException, AccessDeniedException,
            AwsServiceException, SdkClientException, BackupStorageException {
        return getObjectMetadata(getObjectMetadataRequest, ResponseTransformer.toBytes());
    }

    /**
     * Get metadata associated with an Object.<br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetObjectMetadataRequest.Builder} avoiding the need
     * to create one manually via {@link GetObjectMetadataRequest#builder()}
     * </p>
     *
     * @param getObjectMetadataRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.backupstorage.model.GetObjectMetadataRequest.Builder} to create a
     *        request.
     * @return A {@link ResponseBytes} that loads the data streamed from the service into memory and exposes it in
     *         convenient in-memory representations like a byte buffer or string. The unmarshalled response object can
     *         be obtained via {@link ResponseBytes#response()}. The service documentation for the response content is
     *         as follows 'Metadata blob.'.
     * @throws ServiceUnavailableException
     *         Retryable exception, indicates internal server error.
     * @throws ServiceInternalException
     *         Deprecated. To be removed from the model.
     * @throws ResourceNotFoundException
     *         Non-retryable exception. Attempted to make an operation on non-existing or expired resource.
     * @throws RetryableException
     *         Retryable exception. In general indicates internal failure that can be fixed by retry.
     * @throws IllegalArgumentException
     *         Non-retryable exception, indicates client error (wrong argument passed to API). See exception message for
     *         details.
     * @throws ThrottlingException
     *         Increased rate over throttling limits. Can be retried with exponential backoff.
     * @throws KmsInvalidKeyUsageException
     *         Non-retryable exception. Indicates the KMS key usage is incorrect. See exception message for details.
     * @throws AccessDeniedException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupStorageException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupStorageClient.GetObjectMetadata
     * @see #getObject(getObjectMetadata, ResponseTransformer)
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/GetObjectMetadata"
     *      target="_top">AWS API Documentation</a>
     */
    default ResponseBytes<GetObjectMetadataResponse> getObjectMetadataAsBytes(
            Consumer<GetObjectMetadataRequest.Builder> getObjectMetadataRequest) throws ServiceUnavailableException,
            ServiceInternalException, ResourceNotFoundException, RetryableException, IllegalArgumentException,
            ThrottlingException, KmsInvalidKeyUsageException, AccessDeniedException, AwsServiceException, SdkClientException,
            BackupStorageException {
        return getObjectMetadataAsBytes(GetObjectMetadataRequest.builder().applyMutation(getObjectMetadataRequest).build());
    }

    /**
     * List chunks in a given Object
     *
     * @param listChunksRequest
     * @return Result of the ListChunks operation returned by the service.
     * @throws ServiceUnavailableException
     *         Retryable exception, indicates internal server error.
     * @throws ResourceNotFoundException
     *         Non-retryable exception. Attempted to make an operation on non-existing or expired resource.
     * @throws ServiceInternalException
     *         Deprecated. To be removed from the model.
     * @throws RetryableException
     *         Retryable exception. In general indicates internal failure that can be fixed by retry.
     * @throws IllegalArgumentException
     *         Non-retryable exception, indicates client error (wrong argument passed to API). See exception message for
     *         details.
     * @throws AccessDeniedException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupStorageException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupStorageClient.ListChunks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/ListChunks" target="_top">AWS API
     *      Documentation</a>
     */
    default ListChunksResponse listChunks(ListChunksRequest listChunksRequest) throws ServiceUnavailableException,
            ResourceNotFoundException, ServiceInternalException, RetryableException, IllegalArgumentException,
            AccessDeniedException, AwsServiceException, SdkClientException, BackupStorageException {
        throw new UnsupportedOperationException();
    }

    /**
     * List chunks in a given Object<br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListChunksRequest.Builder} avoiding the need to
     * create one manually via {@link ListChunksRequest#builder()}
     * </p>
     *
     * @param listChunksRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.backupstorage.model.ListChunksRequest.Builder} to create a request.
     * @return Result of the ListChunks operation returned by the service.
     * @throws ServiceUnavailableException
     *         Retryable exception, indicates internal server error.
     * @throws ResourceNotFoundException
     *         Non-retryable exception. Attempted to make an operation on non-existing or expired resource.
     * @throws ServiceInternalException
     *         Deprecated. To be removed from the model.
     * @throws RetryableException
     *         Retryable exception. In general indicates internal failure that can be fixed by retry.
     * @throws IllegalArgumentException
     *         Non-retryable exception, indicates client error (wrong argument passed to API). See exception message for
     *         details.
     * @throws AccessDeniedException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupStorageException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupStorageClient.ListChunks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/ListChunks" target="_top">AWS API
     *      Documentation</a>
     */
    default ListChunksResponse listChunks(Consumer<ListChunksRequest.Builder> listChunksRequest)
            throws ServiceUnavailableException, ResourceNotFoundException, ServiceInternalException, RetryableException,
            IllegalArgumentException, AccessDeniedException, AwsServiceException, SdkClientException, BackupStorageException {
        return listChunks(ListChunksRequest.builder().applyMutation(listChunksRequest).build());
    }

    /**
     * List chunks in a given Object<br/>
     * <p>
     * This is a variant of {@link #listChunks(software.amazon.awssdk.services.backupstorage.model.ListChunksRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.backupstorage.paginators.ListChunksIterable responses = client.listChunksPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.backupstorage.paginators.ListChunksIterable responses = client.listChunksPaginator(request);
     *     for (software.amazon.awssdk.services.backupstorage.model.ListChunksResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.backupstorage.paginators.ListChunksIterable responses = client.listChunksPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listChunks(software.amazon.awssdk.services.backupstorage.model.ListChunksRequest)} operation.</b>
     * </p>
     *
     * @param listChunksRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ServiceUnavailableException
     *         Retryable exception, indicates internal server error.
     * @throws ResourceNotFoundException
     *         Non-retryable exception. Attempted to make an operation on non-existing or expired resource.
     * @throws ServiceInternalException
     *         Deprecated. To be removed from the model.
     * @throws RetryableException
     *         Retryable exception. In general indicates internal failure that can be fixed by retry.
     * @throws IllegalArgumentException
     *         Non-retryable exception, indicates client error (wrong argument passed to API). See exception message for
     *         details.
     * @throws AccessDeniedException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupStorageException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupStorageClient.ListChunks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/ListChunks" target="_top">AWS API
     *      Documentation</a>
     */
    default ListChunksIterable listChunksPaginator(ListChunksRequest listChunksRequest) throws ServiceUnavailableException,
            ResourceNotFoundException, ServiceInternalException, RetryableException, IllegalArgumentException,
            AccessDeniedException, AwsServiceException, SdkClientException, BackupStorageException {
        return new ListChunksIterable(this, listChunksRequest);
    }

    /**
     * List chunks in a given Object<br/>
     * <p>
     * This is a variant of {@link #listChunks(software.amazon.awssdk.services.backupstorage.model.ListChunksRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.backupstorage.paginators.ListChunksIterable responses = client.listChunksPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.backupstorage.paginators.ListChunksIterable responses = client.listChunksPaginator(request);
     *     for (software.amazon.awssdk.services.backupstorage.model.ListChunksResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.backupstorage.paginators.ListChunksIterable responses = client.listChunksPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listChunks(software.amazon.awssdk.services.backupstorage.model.ListChunksRequest)} operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link ListChunksRequest.Builder} avoiding the need to
     * create one manually via {@link ListChunksRequest#builder()}
     * </p>
     *
     * @param listChunksRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.backupstorage.model.ListChunksRequest.Builder} to create a request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ServiceUnavailableException
     *         Retryable exception, indicates internal server error.
     * @throws ResourceNotFoundException
     *         Non-retryable exception. Attempted to make an operation on non-existing or expired resource.
     * @throws ServiceInternalException
     *         Deprecated. To be removed from the model.
     * @throws RetryableException
     *         Retryable exception. In general indicates internal failure that can be fixed by retry.
     * @throws IllegalArgumentException
     *         Non-retryable exception, indicates client error (wrong argument passed to API). See exception message for
     *         details.
     * @throws AccessDeniedException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupStorageException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupStorageClient.ListChunks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/ListChunks" target="_top">AWS API
     *      Documentation</a>
     */
    default ListChunksIterable listChunksPaginator(Consumer<ListChunksRequest.Builder> listChunksRequest)
            throws ServiceUnavailableException, ResourceNotFoundException, ServiceInternalException, RetryableException,
            IllegalArgumentException, AccessDeniedException, AwsServiceException, SdkClientException, BackupStorageException {
        return listChunksPaginator(ListChunksRequest.builder().applyMutation(listChunksRequest).build());
    }

    /**
     * List all Objects in a given Backup.
     *
     * @param listObjectsRequest
     * @return Result of the ListObjects operation returned by the service.
     * @throws ServiceUnavailableException
     *         Retryable exception, indicates internal server error.
     * @throws ServiceInternalException
     *         Deprecated. To be removed from the model.
     * @throws RetryableException
     *         Retryable exception. In general indicates internal failure that can be fixed by retry.
     * @throws IllegalArgumentException
     *         Non-retryable exception, indicates client error (wrong argument passed to API). See exception message for
     *         details.
     * @throws ThrottlingException
     *         Increased rate over throttling limits. Can be retried with exponential backoff.
     * @throws ResourceNotFoundException
     *         Non-retryable exception. Attempted to make an operation on non-existing or expired resource.
     * @throws KmsInvalidKeyUsageException
     *         Non-retryable exception. Indicates the KMS key usage is incorrect. See exception message for details.
     * @throws AccessDeniedException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupStorageException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupStorageClient.ListObjects
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/ListObjects" target="_top">AWS API
     *      Documentation</a>
     */
    default ListObjectsResponse listObjects(ListObjectsRequest listObjectsRequest) throws ServiceUnavailableException,
            ServiceInternalException, RetryableException, IllegalArgumentException, ThrottlingException,
            ResourceNotFoundException, KmsInvalidKeyUsageException, AccessDeniedException, AwsServiceException,
            SdkClientException, BackupStorageException {
        throw new UnsupportedOperationException();
    }

    /**
     * List all Objects in a given Backup.<br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListObjectsRequest.Builder} avoiding the need to
     * create one manually via {@link ListObjectsRequest#builder()}
     * </p>
     *
     * @param listObjectsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.backupstorage.model.ListObjectsRequest.Builder} to create a
     *        request.
     * @return Result of the ListObjects operation returned by the service.
     * @throws ServiceUnavailableException
     *         Retryable exception, indicates internal server error.
     * @throws ServiceInternalException
     *         Deprecated. To be removed from the model.
     * @throws RetryableException
     *         Retryable exception. In general indicates internal failure that can be fixed by retry.
     * @throws IllegalArgumentException
     *         Non-retryable exception, indicates client error (wrong argument passed to API). See exception message for
     *         details.
     * @throws ThrottlingException
     *         Increased rate over throttling limits. Can be retried with exponential backoff.
     * @throws ResourceNotFoundException
     *         Non-retryable exception. Attempted to make an operation on non-existing or expired resource.
     * @throws KmsInvalidKeyUsageException
     *         Non-retryable exception. Indicates the KMS key usage is incorrect. See exception message for details.
     * @throws AccessDeniedException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupStorageException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupStorageClient.ListObjects
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/ListObjects" target="_top">AWS API
     *      Documentation</a>
     */
    default ListObjectsResponse listObjects(Consumer<ListObjectsRequest.Builder> listObjectsRequest)
            throws ServiceUnavailableException, ServiceInternalException, RetryableException, IllegalArgumentException,
            ThrottlingException, ResourceNotFoundException, KmsInvalidKeyUsageException, AccessDeniedException,
            AwsServiceException, SdkClientException, BackupStorageException {
        return listObjects(ListObjectsRequest.builder().applyMutation(listObjectsRequest).build());
    }

    /**
     * List all Objects in a given Backup.<br/>
     * <p>
     * This is a variant of {@link #listObjects(software.amazon.awssdk.services.backupstorage.model.ListObjectsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.backupstorage.paginators.ListObjectsIterable responses = client.listObjectsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.backupstorage.paginators.ListObjectsIterable responses = client.listObjectsPaginator(request);
     *     for (software.amazon.awssdk.services.backupstorage.model.ListObjectsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.backupstorage.paginators.ListObjectsIterable responses = client.listObjectsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listObjects(software.amazon.awssdk.services.backupstorage.model.ListObjectsRequest)} operation.</b>
     * </p>
     *
     * @param listObjectsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ServiceUnavailableException
     *         Retryable exception, indicates internal server error.
     * @throws ServiceInternalException
     *         Deprecated. To be removed from the model.
     * @throws RetryableException
     *         Retryable exception. In general indicates internal failure that can be fixed by retry.
     * @throws IllegalArgumentException
     *         Non-retryable exception, indicates client error (wrong argument passed to API). See exception message for
     *         details.
     * @throws ThrottlingException
     *         Increased rate over throttling limits. Can be retried with exponential backoff.
     * @throws ResourceNotFoundException
     *         Non-retryable exception. Attempted to make an operation on non-existing or expired resource.
     * @throws KmsInvalidKeyUsageException
     *         Non-retryable exception. Indicates the KMS key usage is incorrect. See exception message for details.
     * @throws AccessDeniedException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupStorageException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupStorageClient.ListObjects
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/ListObjects" target="_top">AWS API
     *      Documentation</a>
     */
    default ListObjectsIterable listObjectsPaginator(ListObjectsRequest listObjectsRequest) throws ServiceUnavailableException,
            ServiceInternalException, RetryableException, IllegalArgumentException, ThrottlingException,
            ResourceNotFoundException, KmsInvalidKeyUsageException, AccessDeniedException, AwsServiceException,
            SdkClientException, BackupStorageException {
        return new ListObjectsIterable(this, listObjectsRequest);
    }

    /**
     * List all Objects in a given Backup.<br/>
     * <p>
     * This is a variant of {@link #listObjects(software.amazon.awssdk.services.backupstorage.model.ListObjectsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.backupstorage.paginators.ListObjectsIterable responses = client.listObjectsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.backupstorage.paginators.ListObjectsIterable responses = client.listObjectsPaginator(request);
     *     for (software.amazon.awssdk.services.backupstorage.model.ListObjectsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.backupstorage.paginators.ListObjectsIterable responses = client.listObjectsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listObjects(software.amazon.awssdk.services.backupstorage.model.ListObjectsRequest)} operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link ListObjectsRequest.Builder} avoiding the need to
     * create one manually via {@link ListObjectsRequest#builder()}
     * </p>
     *
     * @param listObjectsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.backupstorage.model.ListObjectsRequest.Builder} to create a
     *        request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ServiceUnavailableException
     *         Retryable exception, indicates internal server error.
     * @throws ServiceInternalException
     *         Deprecated. To be removed from the model.
     * @throws RetryableException
     *         Retryable exception. In general indicates internal failure that can be fixed by retry.
     * @throws IllegalArgumentException
     *         Non-retryable exception, indicates client error (wrong argument passed to API). See exception message for
     *         details.
     * @throws ThrottlingException
     *         Increased rate over throttling limits. Can be retried with exponential backoff.
     * @throws ResourceNotFoundException
     *         Non-retryable exception. Attempted to make an operation on non-existing or expired resource.
     * @throws KmsInvalidKeyUsageException
     *         Non-retryable exception. Indicates the KMS key usage is incorrect. See exception message for details.
     * @throws AccessDeniedException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupStorageException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupStorageClient.ListObjects
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/ListObjects" target="_top">AWS API
     *      Documentation</a>
     */
    default ListObjectsIterable listObjectsPaginator(Consumer<ListObjectsRequest.Builder> listObjectsRequest)
            throws ServiceUnavailableException, ServiceInternalException, RetryableException, IllegalArgumentException,
            ThrottlingException, ResourceNotFoundException, KmsInvalidKeyUsageException, AccessDeniedException,
            AwsServiceException, SdkClientException, BackupStorageException {
        return listObjectsPaginator(ListObjectsRequest.builder().applyMutation(listObjectsRequest).build());
    }

    /**
     * Complete upload
     *
     * @param notifyObjectCompleteRequest
     * @param requestBody
     *        The content to send to the service. A {@link RequestBody} can be created using one of several factory
     *        methods for various sources of data. For example, to create a request body from a file you can do the
     *        following.
     * 
     *        <pre>
     * {@code RequestBody.fromFile(new File("myfile.txt"))}
     * </pre>
     * 
     *        See documentation in {@link RequestBody} for additional details and which sources of data are supported.
     *        The service documentation for the request content is as follows 'Optional metadata associated with an
     *        Object. Maximum length is 4MB.'
     * @return Result of the NotifyObjectComplete operation returned by the service.
     * @throws ServiceUnavailableException
     *         Retryable exception, indicates internal server error.
     * @throws ServiceInternalException
     *         Deprecated. To be removed from the model.
     * @throws NotReadableInputStreamException
     *         Retryalble exception. Indicated issues while reading an input stream due to the networking issues or
     *         connection drop on the client side.
     * @throws RetryableException
     *         Retryable exception. In general indicates internal failure that can be fixed by retry.
     * @throws IllegalArgumentException
     *         Non-retryable exception, indicates client error (wrong argument passed to API). See exception message for
     *         details.
     * @throws ThrottlingException
     *         Increased rate over throttling limits. Can be retried with exponential backoff.
     * @throws KmsInvalidKeyUsageException
     *         Non-retryable exception. Indicates the KMS key usage is incorrect. See exception message for details.
     * @throws AccessDeniedException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupStorageException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupStorageClient.NotifyObjectComplete
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/NotifyObjectComplete"
     *      target="_top">AWS API Documentation</a>
     */
    default NotifyObjectCompleteResponse notifyObjectComplete(NotifyObjectCompleteRequest notifyObjectCompleteRequest,
            RequestBody requestBody) throws ServiceUnavailableException, ServiceInternalException,
            NotReadableInputStreamException, RetryableException, IllegalArgumentException, ThrottlingException,
            KmsInvalidKeyUsageException, AccessDeniedException, AwsServiceException, SdkClientException, BackupStorageException {
        throw new UnsupportedOperationException();
    }

    /**
     * Complete upload<br/>
     * <p>
     * This is a convenience which creates an instance of the {@link NotifyObjectCompleteRequest.Builder} avoiding the
     * need to create one manually via {@link NotifyObjectCompleteRequest#builder()}
     * </p>
     *
     * @param notifyObjectCompleteRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.backupstorage.model.NotifyObjectCompleteRequest.Builder} to create
     *        a request.
     * @param requestBody
     *        The content to send to the service. A {@link RequestBody} can be created using one of several factory
     *        methods for various sources of data. For example, to create a request body from a file you can do the
     *        following.
     * 
     *        <pre>
     * {@code RequestBody.fromFile(new File("myfile.txt"))}
     * </pre>
     * 
     *        See documentation in {@link RequestBody} for additional details and which sources of data are supported.
     *        The service documentation for the request content is as follows 'Optional metadata associated with an
     *        Object. Maximum length is 4MB.'
     * @return Result of the NotifyObjectComplete operation returned by the service.
     * @throws ServiceUnavailableException
     *         Retryable exception, indicates internal server error.
     * @throws ServiceInternalException
     *         Deprecated. To be removed from the model.
     * @throws NotReadableInputStreamException
     *         Retryalble exception. Indicated issues while reading an input stream due to the networking issues or
     *         connection drop on the client side.
     * @throws RetryableException
     *         Retryable exception. In general indicates internal failure that can be fixed by retry.
     * @throws IllegalArgumentException
     *         Non-retryable exception, indicates client error (wrong argument passed to API). See exception message for
     *         details.
     * @throws ThrottlingException
     *         Increased rate over throttling limits. Can be retried with exponential backoff.
     * @throws KmsInvalidKeyUsageException
     *         Non-retryable exception. Indicates the KMS key usage is incorrect. See exception message for details.
     * @throws AccessDeniedException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupStorageException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupStorageClient.NotifyObjectComplete
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/NotifyObjectComplete"
     *      target="_top">AWS API Documentation</a>
     */
    default NotifyObjectCompleteResponse notifyObjectComplete(
            Consumer<NotifyObjectCompleteRequest.Builder> notifyObjectCompleteRequest, RequestBody requestBody)
            throws ServiceUnavailableException, ServiceInternalException, NotReadableInputStreamException, RetryableException,
            IllegalArgumentException, ThrottlingException, KmsInvalidKeyUsageException, AccessDeniedException,
            AwsServiceException, SdkClientException, BackupStorageException {
        return notifyObjectComplete(NotifyObjectCompleteRequest.builder().applyMutation(notifyObjectCompleteRequest).build(),
                requestBody);
    }

    /**
     * Complete upload
     *
     * @param notifyObjectCompleteRequest
     * @param sourcePath
     *        {@link Path} to file containing data to send to the service. File will be read entirely and may be read
     *        multiple times in the event of a retry. If the file does not exist or the current user does not have
     *        access to read it then an exception will be thrown. The service documentation for the request content is
     *        as follows 'Optional metadata associated with an Object. Maximum length is 4MB.'
     * @return Result of the NotifyObjectComplete operation returned by the service.
     * @throws ServiceUnavailableException
     *         Retryable exception, indicates internal server error.
     * @throws ServiceInternalException
     *         Deprecated. To be removed from the model.
     * @throws NotReadableInputStreamException
     *         Retryalble exception. Indicated issues while reading an input stream due to the networking issues or
     *         connection drop on the client side.
     * @throws RetryableException
     *         Retryable exception. In general indicates internal failure that can be fixed by retry.
     * @throws IllegalArgumentException
     *         Non-retryable exception, indicates client error (wrong argument passed to API). See exception message for
     *         details.
     * @throws ThrottlingException
     *         Increased rate over throttling limits. Can be retried with exponential backoff.
     * @throws KmsInvalidKeyUsageException
     *         Non-retryable exception. Indicates the KMS key usage is incorrect. See exception message for details.
     * @throws AccessDeniedException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupStorageException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupStorageClient.NotifyObjectComplete
     * @see #notifyObjectComplete(NotifyObjectCompleteRequest, RequestBody)
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/NotifyObjectComplete"
     *      target="_top">AWS API Documentation</a>
     */
    default NotifyObjectCompleteResponse notifyObjectComplete(NotifyObjectCompleteRequest notifyObjectCompleteRequest,
            Path sourcePath) throws ServiceUnavailableException, ServiceInternalException, NotReadableInputStreamException,
            RetryableException, IllegalArgumentException, ThrottlingException, KmsInvalidKeyUsageException,
            AccessDeniedException, AwsServiceException, SdkClientException, BackupStorageException {
        return notifyObjectComplete(notifyObjectCompleteRequest, RequestBody.fromFile(sourcePath));
    }

    /**
     * Complete upload<br/>
     * <p>
     * This is a convenience which creates an instance of the {@link NotifyObjectCompleteRequest.Builder} avoiding the
     * need to create one manually via {@link NotifyObjectCompleteRequest#builder()}
     * </p>
     *
     * @param notifyObjectCompleteRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.backupstorage.model.NotifyObjectCompleteRequest.Builder} to create
     *        a request.
     * @param sourcePath
     *        {@link Path} to file containing data to send to the service. File will be read entirely and may be read
     *        multiple times in the event of a retry. If the file does not exist or the current user does not have
     *        access to read it then an exception will be thrown. The service documentation for the request content is
     *        as follows 'Optional metadata associated with an Object. Maximum length is 4MB.'
     * @return Result of the NotifyObjectComplete operation returned by the service.
     * @throws ServiceUnavailableException
     *         Retryable exception, indicates internal server error.
     * @throws ServiceInternalException
     *         Deprecated. To be removed from the model.
     * @throws NotReadableInputStreamException
     *         Retryalble exception. Indicated issues while reading an input stream due to the networking issues or
     *         connection drop on the client side.
     * @throws RetryableException
     *         Retryable exception. In general indicates internal failure that can be fixed by retry.
     * @throws IllegalArgumentException
     *         Non-retryable exception, indicates client error (wrong argument passed to API). See exception message for
     *         details.
     * @throws ThrottlingException
     *         Increased rate over throttling limits. Can be retried with exponential backoff.
     * @throws KmsInvalidKeyUsageException
     *         Non-retryable exception. Indicates the KMS key usage is incorrect. See exception message for details.
     * @throws AccessDeniedException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupStorageException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupStorageClient.NotifyObjectComplete
     * @see #notifyObjectComplete(NotifyObjectCompleteRequest, RequestBody)
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/NotifyObjectComplete"
     *      target="_top">AWS API Documentation</a>
     */
    default NotifyObjectCompleteResponse notifyObjectComplete(
            Consumer<NotifyObjectCompleteRequest.Builder> notifyObjectCompleteRequest, Path sourcePath)
            throws ServiceUnavailableException, ServiceInternalException, NotReadableInputStreamException, RetryableException,
            IllegalArgumentException, ThrottlingException, KmsInvalidKeyUsageException, AccessDeniedException,
            AwsServiceException, SdkClientException, BackupStorageException {
        return notifyObjectComplete(NotifyObjectCompleteRequest.builder().applyMutation(notifyObjectCompleteRequest).build(),
                sourcePath);
    }

    /**
     * Upload chunk.
     *
     * @param putChunkRequest
     * @param requestBody
     *        The content to send to the service. A {@link RequestBody} can be created using one of several factory
     *        methods for various sources of data. For example, to create a request body from a file you can do the
     *        following.
     * 
     *        <pre>
     * {@code RequestBody.fromFile(new File("myfile.txt"))}
     * </pre>
     * 
     *        See documentation in {@link RequestBody} for additional details and which sources of data are supported.
     *        The service documentation for the request content is as follows 'Data to be uploaded'
     * @return Result of the PutChunk operation returned by the service.
     * @throws ServiceUnavailableException
     *         Retryable exception, indicates internal server error.
     * @throws ServiceInternalException
     *         Deprecated. To be removed from the model.
     * @throws NotReadableInputStreamException
     *         Retryalble exception. Indicated issues while reading an input stream due to the networking issues or
     *         connection drop on the client side.
     * @throws RetryableException
     *         Retryable exception. In general indicates internal failure that can be fixed by retry.
     * @throws IllegalArgumentException
     *         Non-retryable exception, indicates client error (wrong argument passed to API). See exception message for
     *         details.
     * @throws ThrottlingException
     *         Increased rate over throttling limits. Can be retried with exponential backoff.
     * @throws KmsInvalidKeyUsageException
     *         Non-retryable exception. Indicates the KMS key usage is incorrect. See exception message for details.
     * @throws AccessDeniedException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupStorageException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupStorageClient.PutChunk
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/PutChunk" target="_top">AWS API
     *      Documentation</a>
     */
    default PutChunkResponse putChunk(PutChunkRequest putChunkRequest, RequestBody requestBody)
            throws ServiceUnavailableException, ServiceInternalException, NotReadableInputStreamException, RetryableException,
            IllegalArgumentException, ThrottlingException, KmsInvalidKeyUsageException, AccessDeniedException,
            AwsServiceException, SdkClientException, BackupStorageException {
        throw new UnsupportedOperationException();
    }

    /**
     * Upload chunk.<br/>
     * <p>
     * This is a convenience which creates an instance of the {@link PutChunkRequest.Builder} avoiding the need to
     * create one manually via {@link PutChunkRequest#builder()}
     * </p>
     *
     * @param putChunkRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.backupstorage.model.PutChunkRequest.Builder} to create a request.
     * @param requestBody
     *        The content to send to the service. A {@link RequestBody} can be created using one of several factory
     *        methods for various sources of data. For example, to create a request body from a file you can do the
     *        following.
     * 
     *        <pre>
     * {@code RequestBody.fromFile(new File("myfile.txt"))}
     * </pre>
     * 
     *        See documentation in {@link RequestBody} for additional details and which sources of data are supported.
     *        The service documentation for the request content is as follows 'Data to be uploaded'
     * @return Result of the PutChunk operation returned by the service.
     * @throws ServiceUnavailableException
     *         Retryable exception, indicates internal server error.
     * @throws ServiceInternalException
     *         Deprecated. To be removed from the model.
     * @throws NotReadableInputStreamException
     *         Retryalble exception. Indicated issues while reading an input stream due to the networking issues or
     *         connection drop on the client side.
     * @throws RetryableException
     *         Retryable exception. In general indicates internal failure that can be fixed by retry.
     * @throws IllegalArgumentException
     *         Non-retryable exception, indicates client error (wrong argument passed to API). See exception message for
     *         details.
     * @throws ThrottlingException
     *         Increased rate over throttling limits. Can be retried with exponential backoff.
     * @throws KmsInvalidKeyUsageException
     *         Non-retryable exception. Indicates the KMS key usage is incorrect. See exception message for details.
     * @throws AccessDeniedException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupStorageException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupStorageClient.PutChunk
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/PutChunk" target="_top">AWS API
     *      Documentation</a>
     */
    default PutChunkResponse putChunk(Consumer<PutChunkRequest.Builder> putChunkRequest, RequestBody requestBody)
            throws ServiceUnavailableException, ServiceInternalException, NotReadableInputStreamException, RetryableException,
            IllegalArgumentException, ThrottlingException, KmsInvalidKeyUsageException, AccessDeniedException,
            AwsServiceException, SdkClientException, BackupStorageException {
        return putChunk(PutChunkRequest.builder().applyMutation(putChunkRequest).build(), requestBody);
    }

    /**
     * Upload chunk.
     *
     * @param putChunkRequest
     * @param sourcePath
     *        {@link Path} to file containing data to send to the service. File will be read entirely and may be read
     *        multiple times in the event of a retry. If the file does not exist or the current user does not have
     *        access to read it then an exception will be thrown. The service documentation for the request content is
     *        as follows 'Data to be uploaded'
     * @return Result of the PutChunk operation returned by the service.
     * @throws ServiceUnavailableException
     *         Retryable exception, indicates internal server error.
     * @throws ServiceInternalException
     *         Deprecated. To be removed from the model.
     * @throws NotReadableInputStreamException
     *         Retryalble exception. Indicated issues while reading an input stream due to the networking issues or
     *         connection drop on the client side.
     * @throws RetryableException
     *         Retryable exception. In general indicates internal failure that can be fixed by retry.
     * @throws IllegalArgumentException
     *         Non-retryable exception, indicates client error (wrong argument passed to API). See exception message for
     *         details.
     * @throws ThrottlingException
     *         Increased rate over throttling limits. Can be retried with exponential backoff.
     * @throws KmsInvalidKeyUsageException
     *         Non-retryable exception. Indicates the KMS key usage is incorrect. See exception message for details.
     * @throws AccessDeniedException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupStorageException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupStorageClient.PutChunk
     * @see #putChunk(PutChunkRequest, RequestBody)
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/PutChunk" target="_top">AWS API
     *      Documentation</a>
     */
    default PutChunkResponse putChunk(PutChunkRequest putChunkRequest, Path sourcePath) throws ServiceUnavailableException,
            ServiceInternalException, NotReadableInputStreamException, RetryableException, IllegalArgumentException,
            ThrottlingException, KmsInvalidKeyUsageException, AccessDeniedException, AwsServiceException, SdkClientException,
            BackupStorageException {
        return putChunk(putChunkRequest, RequestBody.fromFile(sourcePath));
    }

    /**
     * Upload chunk.<br/>
     * <p>
     * This is a convenience which creates an instance of the {@link PutChunkRequest.Builder} avoiding the need to
     * create one manually via {@link PutChunkRequest#builder()}
     * </p>
     *
     * @param putChunkRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.backupstorage.model.PutChunkRequest.Builder} to create a request.
     * @param sourcePath
     *        {@link Path} to file containing data to send to the service. File will be read entirely and may be read
     *        multiple times in the event of a retry. If the file does not exist or the current user does not have
     *        access to read it then an exception will be thrown. The service documentation for the request content is
     *        as follows 'Data to be uploaded'
     * @return Result of the PutChunk operation returned by the service.
     * @throws ServiceUnavailableException
     *         Retryable exception, indicates internal server error.
     * @throws ServiceInternalException
     *         Deprecated. To be removed from the model.
     * @throws NotReadableInputStreamException
     *         Retryalble exception. Indicated issues while reading an input stream due to the networking issues or
     *         connection drop on the client side.
     * @throws RetryableException
     *         Retryable exception. In general indicates internal failure that can be fixed by retry.
     * @throws IllegalArgumentException
     *         Non-retryable exception, indicates client error (wrong argument passed to API). See exception message for
     *         details.
     * @throws ThrottlingException
     *         Increased rate over throttling limits. Can be retried with exponential backoff.
     * @throws KmsInvalidKeyUsageException
     *         Non-retryable exception. Indicates the KMS key usage is incorrect. See exception message for details.
     * @throws AccessDeniedException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupStorageException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupStorageClient.PutChunk
     * @see #putChunk(PutChunkRequest, RequestBody)
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/PutChunk" target="_top">AWS API
     *      Documentation</a>
     */
    default PutChunkResponse putChunk(Consumer<PutChunkRequest.Builder> putChunkRequest, Path sourcePath)
            throws ServiceUnavailableException, ServiceInternalException, NotReadableInputStreamException, RetryableException,
            IllegalArgumentException, ThrottlingException, KmsInvalidKeyUsageException, AccessDeniedException,
            AwsServiceException, SdkClientException, BackupStorageException {
        return putChunk(PutChunkRequest.builder().applyMutation(putChunkRequest).build(), sourcePath);
    }

    /**
     * Upload object that can store object metadata String and data blob in single API call using inline chunk field.
     *
     * @param putObjectRequest
     * @param requestBody
     *        The content to send to the service. A {@link RequestBody} can be created using one of several factory
     *        methods for various sources of data. For example, to create a request body from a file you can do the
     *        following.
     * 
     *        <pre>
     * {@code RequestBody.fromFile(new File("myfile.txt"))}
     * </pre>
     * 
     *        See documentation in {@link RequestBody} for additional details and which sources of data are supported.
     *        The service documentation for the request content is as follows 'Inline chunk data to be uploaded.'
     * @return Result of the PutObject operation returned by the service.
     * @throws ServiceUnavailableException
     *         Retryable exception, indicates internal server error.
     * @throws ServiceInternalException
     *         Deprecated. To be removed from the model.
     * @throws NotReadableInputStreamException
     *         Retryalble exception. Indicated issues while reading an input stream due to the networking issues or
     *         connection drop on the client side.
     * @throws RetryableException
     *         Retryable exception. In general indicates internal failure that can be fixed by retry.
     * @throws IllegalArgumentException
     *         Non-retryable exception, indicates client error (wrong argument passed to API). See exception message for
     *         details.
     * @throws ThrottlingException
     *         Increased rate over throttling limits. Can be retried with exponential backoff.
     * @throws KmsInvalidKeyUsageException
     *         Non-retryable exception. Indicates the KMS key usage is incorrect. See exception message for details.
     * @throws AccessDeniedException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupStorageException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupStorageClient.PutObject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/PutObject" target="_top">AWS API
     *      Documentation</a>
     */
    default PutObjectResponse putObject(PutObjectRequest putObjectRequest, RequestBody requestBody)
            throws ServiceUnavailableException, ServiceInternalException, NotReadableInputStreamException, RetryableException,
            IllegalArgumentException, ThrottlingException, KmsInvalidKeyUsageException, AccessDeniedException,
            AwsServiceException, SdkClientException, BackupStorageException {
        throw new UnsupportedOperationException();
    }

    /**
     * Upload object that can store object metadata String and data blob in single API call using inline chunk field.<br/>
     * <p>
     * This is a convenience which creates an instance of the {@link PutObjectRequest.Builder} avoiding the need to
     * create one manually via {@link PutObjectRequest#builder()}
     * </p>
     *
     * @param putObjectRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.backupstorage.model.PutObjectRequest.Builder} to create a request.
     * @param requestBody
     *        The content to send to the service. A {@link RequestBody} can be created using one of several factory
     *        methods for various sources of data. For example, to create a request body from a file you can do the
     *        following.
     * 
     *        <pre>
     * {@code RequestBody.fromFile(new File("myfile.txt"))}
     * </pre>
     * 
     *        See documentation in {@link RequestBody} for additional details and which sources of data are supported.
     *        The service documentation for the request content is as follows 'Inline chunk data to be uploaded.'
     * @return Result of the PutObject operation returned by the service.
     * @throws ServiceUnavailableException
     *         Retryable exception, indicates internal server error.
     * @throws ServiceInternalException
     *         Deprecated. To be removed from the model.
     * @throws NotReadableInputStreamException
     *         Retryalble exception. Indicated issues while reading an input stream due to the networking issues or
     *         connection drop on the client side.
     * @throws RetryableException
     *         Retryable exception. In general indicates internal failure that can be fixed by retry.
     * @throws IllegalArgumentException
     *         Non-retryable exception, indicates client error (wrong argument passed to API). See exception message for
     *         details.
     * @throws ThrottlingException
     *         Increased rate over throttling limits. Can be retried with exponential backoff.
     * @throws KmsInvalidKeyUsageException
     *         Non-retryable exception. Indicates the KMS key usage is incorrect. See exception message for details.
     * @throws AccessDeniedException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupStorageException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupStorageClient.PutObject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/PutObject" target="_top">AWS API
     *      Documentation</a>
     */
    default PutObjectResponse putObject(Consumer<PutObjectRequest.Builder> putObjectRequest, RequestBody requestBody)
            throws ServiceUnavailableException, ServiceInternalException, NotReadableInputStreamException, RetryableException,
            IllegalArgumentException, ThrottlingException, KmsInvalidKeyUsageException, AccessDeniedException,
            AwsServiceException, SdkClientException, BackupStorageException {
        return putObject(PutObjectRequest.builder().applyMutation(putObjectRequest).build(), requestBody);
    }

    /**
     * Upload object that can store object metadata String and data blob in single API call using inline chunk field.
     *
     * @param putObjectRequest
     * @param sourcePath
     *        {@link Path} to file containing data to send to the service. File will be read entirely and may be read
     *        multiple times in the event of a retry. If the file does not exist or the current user does not have
     *        access to read it then an exception will be thrown. The service documentation for the request content is
     *        as follows 'Inline chunk data to be uploaded.'
     * @return Result of the PutObject operation returned by the service.
     * @throws ServiceUnavailableException
     *         Retryable exception, indicates internal server error.
     * @throws ServiceInternalException
     *         Deprecated. To be removed from the model.
     * @throws NotReadableInputStreamException
     *         Retryalble exception. Indicated issues while reading an input stream due to the networking issues or
     *         connection drop on the client side.
     * @throws RetryableException
     *         Retryable exception. In general indicates internal failure that can be fixed by retry.
     * @throws IllegalArgumentException
     *         Non-retryable exception, indicates client error (wrong argument passed to API). See exception message for
     *         details.
     * @throws ThrottlingException
     *         Increased rate over throttling limits. Can be retried with exponential backoff.
     * @throws KmsInvalidKeyUsageException
     *         Non-retryable exception. Indicates the KMS key usage is incorrect. See exception message for details.
     * @throws AccessDeniedException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupStorageException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupStorageClient.PutObject
     * @see #putObject(PutObjectRequest, RequestBody)
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/PutObject" target="_top">AWS API
     *      Documentation</a>
     */
    default PutObjectResponse putObject(PutObjectRequest putObjectRequest, Path sourcePath) throws ServiceUnavailableException,
            ServiceInternalException, NotReadableInputStreamException, RetryableException, IllegalArgumentException,
            ThrottlingException, KmsInvalidKeyUsageException, AccessDeniedException, AwsServiceException, SdkClientException,
            BackupStorageException {
        return putObject(putObjectRequest, RequestBody.fromFile(sourcePath));
    }

    /**
     * Upload object that can store object metadata String and data blob in single API call using inline chunk field.<br/>
     * <p>
     * This is a convenience which creates an instance of the {@link PutObjectRequest.Builder} avoiding the need to
     * create one manually via {@link PutObjectRequest#builder()}
     * </p>
     *
     * @param putObjectRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.backupstorage.model.PutObjectRequest.Builder} to create a request.
     * @param sourcePath
     *        {@link Path} to file containing data to send to the service. File will be read entirely and may be read
     *        multiple times in the event of a retry. If the file does not exist or the current user does not have
     *        access to read it then an exception will be thrown. The service documentation for the request content is
     *        as follows 'Inline chunk data to be uploaded.'
     * @return Result of the PutObject operation returned by the service.
     * @throws ServiceUnavailableException
     *         Retryable exception, indicates internal server error.
     * @throws ServiceInternalException
     *         Deprecated. To be removed from the model.
     * @throws NotReadableInputStreamException
     *         Retryalble exception. Indicated issues while reading an input stream due to the networking issues or
     *         connection drop on the client side.
     * @throws RetryableException
     *         Retryable exception. In general indicates internal failure that can be fixed by retry.
     * @throws IllegalArgumentException
     *         Non-retryable exception, indicates client error (wrong argument passed to API). See exception message for
     *         details.
     * @throws ThrottlingException
     *         Increased rate over throttling limits. Can be retried with exponential backoff.
     * @throws KmsInvalidKeyUsageException
     *         Non-retryable exception. Indicates the KMS key usage is incorrect. See exception message for details.
     * @throws AccessDeniedException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupStorageException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupStorageClient.PutObject
     * @see #putObject(PutObjectRequest, RequestBody)
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/PutObject" target="_top">AWS API
     *      Documentation</a>
     */
    default PutObjectResponse putObject(Consumer<PutObjectRequest.Builder> putObjectRequest, Path sourcePath)
            throws ServiceUnavailableException, ServiceInternalException, NotReadableInputStreamException, RetryableException,
            IllegalArgumentException, ThrottlingException, KmsInvalidKeyUsageException, AccessDeniedException,
            AwsServiceException, SdkClientException, BackupStorageException {
        return putObject(PutObjectRequest.builder().applyMutation(putObjectRequest).build(), sourcePath);
    }

    /**
     * Start upload containing one or many chunks.
     *
     * @param startObjectRequest
     * @return Result of the StartObject operation returned by the service.
     * @throws ServiceUnavailableException
     *         Retryable exception, indicates internal server error.
     * @throws ServiceInternalException
     *         Deprecated. To be removed from the model.
     * @throws RetryableException
     *         Retryable exception. In general indicates internal failure that can be fixed by retry.
     * @throws IllegalArgumentException
     *         Non-retryable exception, indicates client error (wrong argument passed to API). See exception message for
     *         details.
     * @throws ResourceNotFoundException
     *         Non-retryable exception. Attempted to make an operation on non-existing or expired resource.
     * @throws DataAlreadyExistsException
     *         Non-retryable exception. Attempted to create already existing object or chunk. This message contains a
     *         checksum of already presented data.
     * @throws ThrottlingException
     *         Increased rate over throttling limits. Can be retried with exponential backoff.
     * @throws AccessDeniedException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupStorageException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupStorageClient.StartObject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/StartObject" target="_top">AWS API
     *      Documentation</a>
     */
    default StartObjectResponse startObject(StartObjectRequest startObjectRequest) throws ServiceUnavailableException,
            ServiceInternalException, RetryableException, IllegalArgumentException, ResourceNotFoundException,
            DataAlreadyExistsException, ThrottlingException, AccessDeniedException, AwsServiceException, SdkClientException,
            BackupStorageException {
        throw new UnsupportedOperationException();
    }

    /**
     * Start upload containing one or many chunks.<br/>
     * <p>
     * This is a convenience which creates an instance of the {@link StartObjectRequest.Builder} avoiding the need to
     * create one manually via {@link StartObjectRequest#builder()}
     * </p>
     *
     * @param startObjectRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.backupstorage.model.StartObjectRequest.Builder} to create a
     *        request.
     * @return Result of the StartObject operation returned by the service.
     * @throws ServiceUnavailableException
     *         Retryable exception, indicates internal server error.
     * @throws ServiceInternalException
     *         Deprecated. To be removed from the model.
     * @throws RetryableException
     *         Retryable exception. In general indicates internal failure that can be fixed by retry.
     * @throws IllegalArgumentException
     *         Non-retryable exception, indicates client error (wrong argument passed to API). See exception message for
     *         details.
     * @throws ResourceNotFoundException
     *         Non-retryable exception. Attempted to make an operation on non-existing or expired resource.
     * @throws DataAlreadyExistsException
     *         Non-retryable exception. Attempted to create already existing object or chunk. This message contains a
     *         checksum of already presented data.
     * @throws ThrottlingException
     *         Increased rate over throttling limits. Can be retried with exponential backoff.
     * @throws AccessDeniedException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupStorageException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupStorageClient.StartObject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/StartObject" target="_top">AWS API
     *      Documentation</a>
     */
    default StartObjectResponse startObject(Consumer<StartObjectRequest.Builder> startObjectRequest)
            throws ServiceUnavailableException, ServiceInternalException, RetryableException, IllegalArgumentException,
            ResourceNotFoundException, DataAlreadyExistsException, ThrottlingException, AccessDeniedException,
            AwsServiceException, SdkClientException, BackupStorageException {
        return startObject(StartObjectRequest.builder().applyMutation(startObjectRequest).build());
    }

    /**
     * Create a {@link BackupStorageClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static BackupStorageClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link BackupStorageClient}.
     */
    static BackupStorageClientBuilder builder() {
        return new DefaultBackupStorageClientBuilder();
    }

    static ServiceMetadata serviceMetadata() {
        return ServiceMetadata.of(SERVICE_METADATA_ID);
    }

    @Override
    default BackupStorageServiceClientConfiguration serviceClientConfiguration() {
        throw new UnsupportedOperationException();
    }
}
