/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.chimesdkmeetings;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.chimesdkmeetings.internal.ChimeSdkMeetingsServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.chimesdkmeetings.model.BadRequestException;
import software.amazon.awssdk.services.chimesdkmeetings.model.BatchCreateAttendeeRequest;
import software.amazon.awssdk.services.chimesdkmeetings.model.BatchCreateAttendeeResponse;
import software.amazon.awssdk.services.chimesdkmeetings.model.BatchUpdateAttendeeCapabilitiesExceptRequest;
import software.amazon.awssdk.services.chimesdkmeetings.model.BatchUpdateAttendeeCapabilitiesExceptResponse;
import software.amazon.awssdk.services.chimesdkmeetings.model.ChimeSdkMeetingsException;
import software.amazon.awssdk.services.chimesdkmeetings.model.ConflictException;
import software.amazon.awssdk.services.chimesdkmeetings.model.CreateAttendeeRequest;
import software.amazon.awssdk.services.chimesdkmeetings.model.CreateAttendeeResponse;
import software.amazon.awssdk.services.chimesdkmeetings.model.CreateMeetingRequest;
import software.amazon.awssdk.services.chimesdkmeetings.model.CreateMeetingResponse;
import software.amazon.awssdk.services.chimesdkmeetings.model.CreateMeetingWithAttendeesRequest;
import software.amazon.awssdk.services.chimesdkmeetings.model.CreateMeetingWithAttendeesResponse;
import software.amazon.awssdk.services.chimesdkmeetings.model.DeleteAttendeeRequest;
import software.amazon.awssdk.services.chimesdkmeetings.model.DeleteAttendeeResponse;
import software.amazon.awssdk.services.chimesdkmeetings.model.DeleteMeetingRequest;
import software.amazon.awssdk.services.chimesdkmeetings.model.DeleteMeetingResponse;
import software.amazon.awssdk.services.chimesdkmeetings.model.ForbiddenException;
import software.amazon.awssdk.services.chimesdkmeetings.model.GetAttendeeRequest;
import software.amazon.awssdk.services.chimesdkmeetings.model.GetAttendeeResponse;
import software.amazon.awssdk.services.chimesdkmeetings.model.GetMeetingRequest;
import software.amazon.awssdk.services.chimesdkmeetings.model.GetMeetingResponse;
import software.amazon.awssdk.services.chimesdkmeetings.model.LimitExceededException;
import software.amazon.awssdk.services.chimesdkmeetings.model.ListAttendeesRequest;
import software.amazon.awssdk.services.chimesdkmeetings.model.ListAttendeesResponse;
import software.amazon.awssdk.services.chimesdkmeetings.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.chimesdkmeetings.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.chimesdkmeetings.model.NotFoundException;
import software.amazon.awssdk.services.chimesdkmeetings.model.ResourceNotFoundException;
import software.amazon.awssdk.services.chimesdkmeetings.model.ServiceFailureException;
import software.amazon.awssdk.services.chimesdkmeetings.model.ServiceUnavailableException;
import software.amazon.awssdk.services.chimesdkmeetings.model.StartMeetingTranscriptionRequest;
import software.amazon.awssdk.services.chimesdkmeetings.model.StartMeetingTranscriptionResponse;
import software.amazon.awssdk.services.chimesdkmeetings.model.StopMeetingTranscriptionRequest;
import software.amazon.awssdk.services.chimesdkmeetings.model.StopMeetingTranscriptionResponse;
import software.amazon.awssdk.services.chimesdkmeetings.model.TagResourceRequest;
import software.amazon.awssdk.services.chimesdkmeetings.model.TagResourceResponse;
import software.amazon.awssdk.services.chimesdkmeetings.model.ThrottlingException;
import software.amazon.awssdk.services.chimesdkmeetings.model.TooManyTagsException;
import software.amazon.awssdk.services.chimesdkmeetings.model.UnauthorizedException;
import software.amazon.awssdk.services.chimesdkmeetings.model.UnprocessableEntityException;
import software.amazon.awssdk.services.chimesdkmeetings.model.UntagResourceRequest;
import software.amazon.awssdk.services.chimesdkmeetings.model.UntagResourceResponse;
import software.amazon.awssdk.services.chimesdkmeetings.model.UpdateAttendeeCapabilitiesRequest;
import software.amazon.awssdk.services.chimesdkmeetings.model.UpdateAttendeeCapabilitiesResponse;
import software.amazon.awssdk.services.chimesdkmeetings.transform.BatchCreateAttendeeRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmeetings.transform.BatchUpdateAttendeeCapabilitiesExceptRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmeetings.transform.CreateAttendeeRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmeetings.transform.CreateMeetingRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmeetings.transform.CreateMeetingWithAttendeesRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmeetings.transform.DeleteAttendeeRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmeetings.transform.DeleteMeetingRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmeetings.transform.GetAttendeeRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmeetings.transform.GetMeetingRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmeetings.transform.ListAttendeesRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmeetings.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmeetings.transform.StartMeetingTranscriptionRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmeetings.transform.StopMeetingTranscriptionRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmeetings.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmeetings.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmeetings.transform.UpdateAttendeeCapabilitiesRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link ChimeSdkMeetingsAsyncClient}.
 *
 * @see ChimeSdkMeetingsAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultChimeSdkMeetingsAsyncClient implements ChimeSdkMeetingsAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultChimeSdkMeetingsAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultChimeSdkMeetingsAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates up to 100 attendees for an active Amazon Chime SDK meeting. For more information about the Amazon Chime
     * SDK, see <a href="https://docs.aws.amazon.com/chime/latest/dg/meetings-sdk.html">Using the Amazon Chime SDK</a>
     * in the <i>Amazon Chime Developer Guide</i>.
     * </p>
     *
     * @param batchCreateAttendeeRequest
     * @return A Java Future containing the result of the BatchCreateAttendee operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>UnauthorizedException The user isn't authorized to request a resource.</li>
     *         <li>UnprocessableEntityException The request was well-formed but was unable to be followed due to
     *         semantic errors.</li>
     *         <li>LimitExceededException The request exceeds the resource limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>ThrottlingException The number of customer requests exceeds the request rate limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMeetingsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMeetingsAsyncClient.BatchCreateAttendee
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-meetings-2021-07-15/BatchCreateAttendee"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchCreateAttendeeResponse> batchCreateAttendee(
            BatchCreateAttendeeRequest batchCreateAttendeeRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(batchCreateAttendeeRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchCreateAttendeeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Meetings");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchCreateAttendee");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchCreateAttendeeResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchCreateAttendeeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BatchCreateAttendeeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchCreateAttendeeRequest, BatchCreateAttendeeResponse>()
                            .withOperationName("BatchCreateAttendee").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new BatchCreateAttendeeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(batchCreateAttendeeRequest));
            CompletableFuture<BatchCreateAttendeeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates <code>AttendeeCapabilities</code> except the capabilities listed in an <code>ExcludedAttendeeIds</code>
     * table.
     * </p>
     * <note>
     * <p>
     * You use the capabilities with a set of values that control what the capabilities can do, such as
     * <code>SendReceive</code> data. For more information about those values, see .
     * </p>
     * </note>
     * <p>
     * When using capabilities, be aware of these corner cases:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If you specify <code>MeetingFeatures:Video:MaxResolution:None</code> when you create a meeting, all API requests
     * that include <code>SendReceive</code>, <code>Send</code>, or <code>Receive</code> for
     * <code>AttendeeCapabilities:Video</code> will be rejected with <code>ValidationError 400</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you specify <code>MeetingFeatures:Content:MaxResolution:None</code> when you create a meeting, all API
     * requests that include <code>SendReceive</code>, <code>Send</code>, or <code>Receive</code> for
     * <code>AttendeeCapabilities:Content</code> will be rejected with <code>ValidationError 400</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * You can't set <code>content</code> capabilities to <code>SendReceive</code> or <code>Receive</code> unless you
     * also set <code>video</code> capabilities to <code>SendReceive</code> or <code>Receive</code>. If you don't set
     * the <code>video</code> capability to receive, the response will contain an HTTP 400 Bad Request status code.
     * However, you can set your <code>video</code> capability to receive and you set your <code>content</code>
     * capability to not receive.
     * </p>
     * </li>
     * <li>
     * <p>
     * When you change an <code>audio</code> capability from <code>None</code> or <code>Receive</code> to
     * <code>Send</code> or <code>SendReceive</code> , and if the attendee left their microphone unmuted, audio will
     * flow from the attendee to the other meeting participants.
     * </p>
     * </li>
     * <li>
     * <p>
     * When you change a <code>video</code> or <code>content</code> capability from <code>None</code> or
     * <code>Receive</code> to <code>Send</code> or <code>SendReceive</code> , and if the attendee turned on their video
     * or content streams, remote attendees can receive those streams, but only after media renegotiation between the
     * client and the Amazon Chime back-end server.
     * </p>
     * </li>
     * </ul>
     *
     * @param batchUpdateAttendeeCapabilitiesExceptRequest
     * @return A Java Future containing the result of the BatchUpdateAttendeeCapabilitiesExcept operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ConflictException Multiple instances of the same request have been made simultaneously.</li>
     *         <li>UnauthorizedException The user isn't authorized to request a resource.</li>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>ThrottlingException The number of customer requests exceeds the request rate limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMeetingsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMeetingsAsyncClient.BatchUpdateAttendeeCapabilitiesExcept
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-meetings-2021-07-15/BatchUpdateAttendeeCapabilitiesExcept"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchUpdateAttendeeCapabilitiesExceptResponse> batchUpdateAttendeeCapabilitiesExcept(
            BatchUpdateAttendeeCapabilitiesExceptRequest batchUpdateAttendeeCapabilitiesExceptRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(batchUpdateAttendeeCapabilitiesExceptRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                batchUpdateAttendeeCapabilitiesExceptRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Meetings");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchUpdateAttendeeCapabilitiesExcept");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchUpdateAttendeeCapabilitiesExceptResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, BatchUpdateAttendeeCapabilitiesExceptResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BatchUpdateAttendeeCapabilitiesExceptResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchUpdateAttendeeCapabilitiesExceptRequest, BatchUpdateAttendeeCapabilitiesExceptResponse>()
                            .withOperationName("BatchUpdateAttendeeCapabilitiesExcept").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new BatchUpdateAttendeeCapabilitiesExceptRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(batchUpdateAttendeeCapabilitiesExceptRequest));
            CompletableFuture<BatchUpdateAttendeeCapabilitiesExceptResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new attendee for an active Amazon Chime SDK meeting. For more information about the Amazon Chime SDK,
     * see <a href="https://docs.aws.amazon.com/chime/latest/dg/meetings-sdk.html">Using the Amazon Chime SDK</a> in the
     * <i>Amazon Chime Developer Guide</i>.
     * </p>
     *
     * @param createAttendeeRequest
     * @return A Java Future containing the result of the CreateAttendee operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>UnauthorizedException The user isn't authorized to request a resource.</li>
     *         <li>UnprocessableEntityException The request was well-formed but was unable to be followed due to
     *         semantic errors.</li>
     *         <li>LimitExceededException The request exceeds the resource limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>ThrottlingException The number of customer requests exceeds the request rate limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMeetingsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMeetingsAsyncClient.CreateAttendee
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-meetings-2021-07-15/CreateAttendee"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateAttendeeResponse> createAttendee(CreateAttendeeRequest createAttendeeRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createAttendeeRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createAttendeeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Meetings");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateAttendee");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateAttendeeResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateAttendeeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateAttendeeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateAttendeeRequest, CreateAttendeeResponse>()
                            .withOperationName("CreateAttendee").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateAttendeeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createAttendeeRequest));
            CompletableFuture<CreateAttendeeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new Amazon Chime SDK meeting in the specified media Region with no initial attendees. For more
     * information about specifying media Regions, see <a
     * href="https://docs.aws.amazon.com/chime/latest/dg/chime-sdk-meetings-regions.html">Amazon Chime SDK Media
     * Regions</a> in the <i>Amazon Chime Developer Guide</i>. For more information about the Amazon Chime SDK, see <a
     * href="https://docs.aws.amazon.com/chime/latest/dg/meetings-sdk.html">Using the Amazon Chime SDK</a> in the
     * <i>Amazon Chime Developer Guide</i>.
     * </p>
     *
     * @param createMeetingRequest
     * @return A Java Future containing the result of the CreateMeeting operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ConflictException Multiple instances of the same request have been made simultaneously.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedException The user isn't authorized to request a resource.</li>
     *         <li>ThrottlingException The number of customer requests exceeds the request rate limit.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>LimitExceededException The request exceeds the resource limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMeetingsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMeetingsAsyncClient.CreateMeeting
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-meetings-2021-07-15/CreateMeeting"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateMeetingResponse> createMeeting(CreateMeetingRequest createMeetingRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createMeetingRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createMeetingRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Meetings");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateMeeting");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateMeetingResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateMeetingResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateMeetingResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateMeetingRequest, CreateMeetingResponse>()
                            .withOperationName("CreateMeeting").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateMeetingRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createMeetingRequest));
            CompletableFuture<CreateMeetingResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new Amazon Chime SDK meeting in the specified media Region, with attendees. For more information about
     * specifying media Regions, see <a
     * href="https://docs.aws.amazon.com/chime/latest/dg/chime-sdk-meetings-regions.html">Amazon Chime SDK Media
     * Regions</a> in the <i>Amazon Chime Developer Guide</i>. For more information about the Amazon Chime SDK, see <a
     * href="https://docs.aws.amazon.com/chime/latest/dg/meetings-sdk.html">Using the Amazon Chime SDK</a> in the
     * <i>Amazon Chime Developer Guide</i>.
     * </p>
     *
     * @param createMeetingWithAttendeesRequest
     * @return A Java Future containing the result of the CreateMeetingWithAttendees operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ConflictException Multiple instances of the same request have been made simultaneously.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedException The user isn't authorized to request a resource.</li>
     *         <li>ThrottlingException The number of customer requests exceeds the request rate limit.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>LimitExceededException The request exceeds the resource limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMeetingsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMeetingsAsyncClient.CreateMeetingWithAttendees
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-meetings-2021-07-15/CreateMeetingWithAttendees"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateMeetingWithAttendeesResponse> createMeetingWithAttendees(
            CreateMeetingWithAttendeesRequest createMeetingWithAttendeesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createMeetingWithAttendeesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createMeetingWithAttendeesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Meetings");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateMeetingWithAttendees");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateMeetingWithAttendeesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateMeetingWithAttendeesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateMeetingWithAttendeesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateMeetingWithAttendeesRequest, CreateMeetingWithAttendeesResponse>()
                            .withOperationName("CreateMeetingWithAttendees").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateMeetingWithAttendeesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createMeetingWithAttendeesRequest));
            CompletableFuture<CreateMeetingWithAttendeesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an attendee from the specified Amazon Chime SDK meeting and deletes their <code>JoinToken</code>.
     * Attendees are automatically deleted when a Amazon Chime SDK meeting is deleted. For more information about the
     * Amazon Chime SDK, see <a href="https://docs.aws.amazon.com/chime/latest/dg/meetings-sdk.html">Using the Amazon
     * Chime SDK</a> in the <i>Amazon Chime Developer Guide</i>.
     * </p>
     *
     * @param deleteAttendeeRequest
     * @return A Java Future containing the result of the DeleteAttendee operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>UnauthorizedException The user isn't authorized to request a resource.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>ThrottlingException The number of customer requests exceeds the request rate limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMeetingsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMeetingsAsyncClient.DeleteAttendee
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-meetings-2021-07-15/DeleteAttendee"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteAttendeeResponse> deleteAttendee(DeleteAttendeeRequest deleteAttendeeRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteAttendeeRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAttendeeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Meetings");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAttendee");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteAttendeeResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteAttendeeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteAttendeeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteAttendeeRequest, DeleteAttendeeResponse>()
                            .withOperationName("DeleteAttendee").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteAttendeeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteAttendeeRequest));
            CompletableFuture<DeleteAttendeeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified Amazon Chime SDK meeting. The operation deletes all attendees, disconnects all clients, and
     * prevents new clients from joining the meeting. For more information about the Amazon Chime SDK, see <a
     * href="https://docs.aws.amazon.com/chime/latest/dg/meetings-sdk.html">Using the Amazon Chime SDK</a> in the
     * <i>Amazon Chime Developer Guide</i>.
     * </p>
     *
     * @param deleteMeetingRequest
     * @return A Java Future containing the result of the DeleteMeeting operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedException The user isn't authorized to request a resource.</li>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>ThrottlingException The number of customer requests exceeds the request rate limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMeetingsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMeetingsAsyncClient.DeleteMeeting
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-meetings-2021-07-15/DeleteMeeting"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteMeetingResponse> deleteMeeting(DeleteMeetingRequest deleteMeetingRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteMeetingRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteMeetingRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Meetings");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteMeeting");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteMeetingResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteMeetingResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteMeetingResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteMeetingRequest, DeleteMeetingResponse>()
                            .withOperationName("DeleteMeeting").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteMeetingRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteMeetingRequest));
            CompletableFuture<DeleteMeetingResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the Amazon Chime SDK attendee details for a specified meeting ID and attendee ID. For more information about
     * the Amazon Chime SDK, see <a href="https://docs.aws.amazon.com/chime/latest/dg/meetings-sdk.html">Using the
     * Amazon Chime SDK</a> in the <i>Amazon Chime Developer Guide</i>.
     * </p>
     *
     * @param getAttendeeRequest
     * @return A Java Future containing the result of the GetAttendee operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>UnauthorizedException The user isn't authorized to request a resource.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>ThrottlingException The number of customer requests exceeds the request rate limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMeetingsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMeetingsAsyncClient.GetAttendee
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-meetings-2021-07-15/GetAttendee"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetAttendeeResponse> getAttendee(GetAttendeeRequest getAttendeeRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getAttendeeRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAttendeeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Meetings");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAttendee");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetAttendeeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetAttendeeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetAttendeeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetAttendeeRequest, GetAttendeeResponse>()
                            .withOperationName("GetAttendee").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetAttendeeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getAttendeeRequest));
            CompletableFuture<GetAttendeeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the Amazon Chime SDK meeting details for the specified meeting ID. For more information about the Amazon
     * Chime SDK, see <a href="https://docs.aws.amazon.com/chime/latest/dg/meetings-sdk.html">Using the Amazon Chime
     * SDK</a> in the <i>Amazon Chime Developer Guide</i>.
     * </p>
     *
     * @param getMeetingRequest
     * @return A Java Future containing the result of the GetMeeting operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedException The user isn't authorized to request a resource.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>ThrottlingException The number of customer requests exceeds the request rate limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMeetingsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMeetingsAsyncClient.GetMeeting
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-meetings-2021-07-15/GetMeeting" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetMeetingResponse> getMeeting(GetMeetingRequest getMeetingRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getMeetingRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMeetingRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Meetings");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMeeting");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetMeetingResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetMeetingResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetMeetingResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetMeetingRequest, GetMeetingResponse>().withOperationName("GetMeeting")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetMeetingRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getMeetingRequest));
            CompletableFuture<GetMeetingResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the attendees for the specified Amazon Chime SDK meeting. For more information about the Amazon Chime SDK,
     * see <a href="https://docs.aws.amazon.com/chime/latest/dg/meetings-sdk.html">Using the Amazon Chime SDK</a> in the
     * <i>Amazon Chime Developer Guide</i>.
     * </p>
     *
     * @param listAttendeesRequest
     * @return A Java Future containing the result of the ListAttendees operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>UnauthorizedException The user isn't authorized to request a resource.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>ThrottlingException The number of customer requests exceeds the request rate limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMeetingsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMeetingsAsyncClient.ListAttendees
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-meetings-2021-07-15/ListAttendees"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListAttendeesResponse> listAttendees(ListAttendeesRequest listAttendeesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listAttendeesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAttendeesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Meetings");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAttendees");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAttendeesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListAttendeesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListAttendeesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAttendeesRequest, ListAttendeesResponse>()
                            .withOperationName("ListAttendees").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListAttendeesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listAttendeesRequest));
            CompletableFuture<ListAttendeesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of the tags available for the specified resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedException The user isn't authorized to request a resource.</li>
     *         <li>LimitExceededException The request exceeds the resource limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>ThrottlingException The number of customer requests exceeds the request rate limit.</li>
     *         <li>ResourceNotFoundException The resource that you want to tag couldn't be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMeetingsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMeetingsAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-meetings-2021-07-15/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Meetings");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts transcription for the specified <code>meetingId</code>. For more information, refer to <a
     * href="https://docs.aws.amazon.com/chime-sdk/latest/dg/meeting-transcription.html"> Using Amazon Chime SDK live
     * transcription </a> in the <i>Amazon Chime SDK Developer Guide</i>.
     * </p>
     * <p>
     * If you specify an invalid configuration, a <code>TranscriptFailed</code> event will be sent with the contents of
     * the <code>BadRequestException</code> generated by Amazon Transcribe. For more information on each parameter and
     * which combinations are valid, refer to the <a
     * href="https://docs.aws.amazon.com/transcribe/latest/APIReference/API_streaming_StartStreamTranscription.html"
     * >StartStreamTranscription</a> API in the <i>Amazon Transcribe Developer Guide</i>.
     * </p>
     * <note>
     * <p>
     * By default, Amazon Transcribe may use and store audio content processed by the service to develop and improve
     * Amazon Web Services AI/ML services as further described in section 50 of the <a
     * href="https://aws.amazon.com/service-terms/">Amazon Web Services Service Terms</a>. Using Amazon Transcribe may
     * be subject to federal and state laws or regulations regarding the recording or interception of electronic
     * communications. It is your and your end users’ responsibility to comply with all applicable laws regarding the
     * recording, including properly notifying all participants in a recorded session or communication that the session
     * or communication is being recorded, and obtaining all necessary consents. You can opt out from Amazon Web
     * Services using audio content to develop and improve AWS AI/ML services by configuring an AI services opt out
     * policy using Amazon Web Services Organizations.
     * </p>
     * </note>
     *
     * @param startMeetingTranscriptionRequest
     * @return A Java Future containing the result of the StartMeetingTranscription operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>UnauthorizedException The user isn't authorized to request a resource.</li>
     *         <li>LimitExceededException The request exceeds the resource limit.</li>
     *         <li>UnprocessableEntityException The request was well-formed but was unable to be followed due to
     *         semantic errors.</li>
     *         <li>ThrottlingException The number of customer requests exceeds the request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMeetingsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMeetingsAsyncClient.StartMeetingTranscription
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-meetings-2021-07-15/StartMeetingTranscription"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartMeetingTranscriptionResponse> startMeetingTranscription(
            StartMeetingTranscriptionRequest startMeetingTranscriptionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startMeetingTranscriptionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startMeetingTranscriptionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Meetings");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartMeetingTranscription");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartMeetingTranscriptionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartMeetingTranscriptionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartMeetingTranscriptionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartMeetingTranscriptionRequest, StartMeetingTranscriptionResponse>()
                            .withOperationName("StartMeetingTranscription").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StartMeetingTranscriptionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(startMeetingTranscriptionRequest));
            CompletableFuture<StartMeetingTranscriptionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Stops transcription for the specified <code>meetingId</code>. For more information, refer to <a
     * href="https://docs.aws.amazon.com/chime-sdk/latest/dg/meeting-transcription.html"> Using Amazon Chime SDK live
     * transcription </a> in the <i>Amazon Chime SDK Developer Guide</i>.
     * </p>
     * <important>
     * <p>
     * By default, Amazon Transcribe may use and store audio content processed by the service to develop and improve
     * Amazon Web Services AI/ML services as further described in section 50 of the <a
     * href="https://aws.amazon.com/service-terms/">Amazon Web Services Service Terms</a>. Using Amazon Transcribe may
     * be subject to federal and state laws or regulations regarding the recording or interception of electronic
     * communications. It is your and your end users’ responsibility to comply with all applicable laws regarding the
     * recording, including properly notifying all participants in a recorded session or communication that the session
     * or communication is being recorded, and obtaining all necessary consents. You can opt out from Amazon Web
     * Services using audio content to develop and improve Amazon Web Services AI/ML services by configuring an AI
     * services opt out policy using Amazon Web Services Organizations.
     * </p>
     * </important>
     *
     * @param stopMeetingTranscriptionRequest
     * @return A Java Future containing the result of the StopMeetingTranscription operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>UnauthorizedException The user isn't authorized to request a resource.</li>
     *         <li>UnprocessableEntityException The request was well-formed but was unable to be followed due to
     *         semantic errors.</li>
     *         <li>ThrottlingException The number of customer requests exceeds the request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMeetingsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMeetingsAsyncClient.StopMeetingTranscription
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-meetings-2021-07-15/StopMeetingTranscription"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StopMeetingTranscriptionResponse> stopMeetingTranscription(
            StopMeetingTranscriptionRequest stopMeetingTranscriptionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(stopMeetingTranscriptionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopMeetingTranscriptionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Meetings");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopMeetingTranscription");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StopMeetingTranscriptionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StopMeetingTranscriptionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StopMeetingTranscriptionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StopMeetingTranscriptionRequest, StopMeetingTranscriptionResponse>()
                            .withOperationName("StopMeetingTranscription").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StopMeetingTranscriptionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(stopMeetingTranscriptionRequest));
            CompletableFuture<StopMeetingTranscriptionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The resource that supports tags.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedException The user isn't authorized to request a resource.</li>
     *         <li>LimitExceededException The request exceeds the resource limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>ThrottlingException The number of customer requests exceeds the request rate limit.</li>
     *         <li>ResourceNotFoundException The resource that you want to tag couldn't be found.</li>
     *         <li>TooManyTagsException Too many tags were added to the specified resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMeetingsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMeetingsAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-meetings-2021-07-15/TagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Meetings");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes the specified tags from the specified resources. When you specify a tag key, the action removes both that
     * key and its associated value. The operation succeeds even if you attempt to remove tags from a resource that were
     * already removed. Note the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * To remove tags from a resource, you need the necessary permissions for the service that the resource belongs to
     * as well as permissions for removing tags. For more information, see the documentation for the service whose
     * resource you want to untag.
     * </p>
     * </li>
     * <li>
     * <p>
     * You can only tag resources that are located in the specified Amazon Web Services Region for the calling Amazon
     * Web Services account.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>Minimum permissions</b>
     * </p>
     * <p>
     * In addition to the <code>tag:UntagResources</code> permission required by this operation, you must also have the
     * remove tags permission defined by the service that created the resource. For example, to remove the tags from an
     * Amazon EC2 instance using the <code>UntagResources</code> operation, you must have both of the following
     * permissions:
     * </p>
     * <p>
     * <code>tag:UntagResource</code>
     * </p>
     * <p>
     * <code>ChimeSDKMeetings:DeleteTags</code>
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedException The user isn't authorized to request a resource.</li>
     *         <li>LimitExceededException The request exceeds the resource limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>ThrottlingException The number of customer requests exceeds the request rate limit.</li>
     *         <li>ResourceNotFoundException The resource that you want to tag couldn't be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMeetingsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMeetingsAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-meetings-2021-07-15/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Meetings");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The capabilities that you want to update.
     * </p>
     * <note>
     * <p>
     * You use the capabilities with a set of values that control what the capabilities can do, such as
     * <code>SendReceive</code> data. For more information about those values, see .
     * </p>
     * </note>
     * <p>
     * When using capabilities, be aware of these corner cases:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If you specify <code>MeetingFeatures:Video:MaxResolution:None</code> when you create a meeting, all API requests
     * that include <code>SendReceive</code>, <code>Send</code>, or <code>Receive</code> for
     * <code>AttendeeCapabilities:Video</code> will be rejected with <code>ValidationError 400</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you specify <code>MeetingFeatures:Content:MaxResolution:None</code> when you create a meeting, all API
     * requests that include <code>SendReceive</code>, <code>Send</code>, or <code>Receive</code> for
     * <code>AttendeeCapabilities:Content</code> will be rejected with <code>ValidationError 400</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * You can't set <code>content</code> capabilities to <code>SendReceive</code> or <code>Receive</code> unless you
     * also set <code>video</code> capabilities to <code>SendReceive</code> or <code>Receive</code>. If you don't set
     * the <code>video</code> capability to receive, the response will contain an HTTP 400 Bad Request status code.
     * However, you can set your <code>video</code> capability to receive and you set your <code>content</code>
     * capability to not receive.
     * </p>
     * </li>
     * <li>
     * <p>
     * When you change an <code>audio</code> capability from <code>None</code> or <code>Receive</code> to
     * <code>Send</code> or <code>SendReceive</code> , and if the attendee left their microphone unmuted, audio will
     * flow from the attendee to the other meeting participants.
     * </p>
     * </li>
     * <li>
     * <p>
     * When you change a <code>video</code> or <code>content</code> capability from <code>None</code> or
     * <code>Receive</code> to <code>Send</code> or <code>SendReceive</code> , and if the attendee turned on their video
     * or content streams, remote attendees can receive those streams, but only after media renegotiation between the
     * client and the Amazon Chime back-end server.
     * </p>
     * </li>
     * </ul>
     *
     * @param updateAttendeeCapabilitiesRequest
     * @return A Java Future containing the result of the UpdateAttendeeCapabilities operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ConflictException Multiple instances of the same request have been made simultaneously.</li>
     *         <li>UnauthorizedException The user isn't authorized to request a resource.</li>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>ThrottlingException The number of customer requests exceeds the request rate limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMeetingsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMeetingsAsyncClient.UpdateAttendeeCapabilities
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-meetings-2021-07-15/UpdateAttendeeCapabilities"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateAttendeeCapabilitiesResponse> updateAttendeeCapabilities(
            UpdateAttendeeCapabilitiesRequest updateAttendeeCapabilitiesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateAttendeeCapabilitiesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateAttendeeCapabilitiesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Meetings");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateAttendeeCapabilities");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateAttendeeCapabilitiesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateAttendeeCapabilitiesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateAttendeeCapabilitiesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateAttendeeCapabilitiesRequest, UpdateAttendeeCapabilitiesResponse>()
                            .withOperationName("UpdateAttendeeCapabilities").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateAttendeeCapabilitiesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateAttendeeCapabilitiesRequest));
            CompletableFuture<UpdateAttendeeCapabilitiesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final ChimeSdkMeetingsServiceClientConfiguration serviceClientConfiguration() {
        return new ChimeSdkMeetingsServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(ChimeSdkMeetingsException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyTagsException")
                                .exceptionBuilderSupplier(TooManyTagsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnprocessableEntityException")
                                .exceptionBuilderSupplier(UnprocessableEntityException::builder).httpStatusCode(422).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotFoundException")
                                .exceptionBuilderSupplier(NotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).httpStatusCode(503).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnauthorizedException")
                                .exceptionBuilderSupplier(UnauthorizedException::builder).httpStatusCode(401).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ForbiddenException")
                                .exceptionBuilderSupplier(ForbiddenException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequestException")
                                .exceptionBuilderSupplier(BadRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceFailureException")
                                .exceptionBuilderSupplier(ServiceFailureException::builder).httpStatusCode(500).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        ChimeSdkMeetingsServiceClientConfigurationBuilder serviceConfigBuilder = new ChimeSdkMeetingsServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
