/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloud9;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.cloud9.model.BadRequestException;
import software.amazon.awssdk.services.cloud9.model.Cloud9Exception;
import software.amazon.awssdk.services.cloud9.model.Cloud9Request;
import software.amazon.awssdk.services.cloud9.model.ConcurrentAccessException;
import software.amazon.awssdk.services.cloud9.model.ConflictException;
import software.amazon.awssdk.services.cloud9.model.CreateEnvironmentEc2Request;
import software.amazon.awssdk.services.cloud9.model.CreateEnvironmentEc2Response;
import software.amazon.awssdk.services.cloud9.model.CreateEnvironmentMembershipRequest;
import software.amazon.awssdk.services.cloud9.model.CreateEnvironmentMembershipResponse;
import software.amazon.awssdk.services.cloud9.model.DeleteEnvironmentMembershipRequest;
import software.amazon.awssdk.services.cloud9.model.DeleteEnvironmentMembershipResponse;
import software.amazon.awssdk.services.cloud9.model.DeleteEnvironmentRequest;
import software.amazon.awssdk.services.cloud9.model.DeleteEnvironmentResponse;
import software.amazon.awssdk.services.cloud9.model.DescribeEnvironmentMembershipsRequest;
import software.amazon.awssdk.services.cloud9.model.DescribeEnvironmentMembershipsResponse;
import software.amazon.awssdk.services.cloud9.model.DescribeEnvironmentStatusRequest;
import software.amazon.awssdk.services.cloud9.model.DescribeEnvironmentStatusResponse;
import software.amazon.awssdk.services.cloud9.model.DescribeEnvironmentsRequest;
import software.amazon.awssdk.services.cloud9.model.DescribeEnvironmentsResponse;
import software.amazon.awssdk.services.cloud9.model.ForbiddenException;
import software.amazon.awssdk.services.cloud9.model.InternalServerErrorException;
import software.amazon.awssdk.services.cloud9.model.LimitExceededException;
import software.amazon.awssdk.services.cloud9.model.ListEnvironmentsRequest;
import software.amazon.awssdk.services.cloud9.model.ListEnvironmentsResponse;
import software.amazon.awssdk.services.cloud9.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.cloud9.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.cloud9.model.NotFoundException;
import software.amazon.awssdk.services.cloud9.model.TagResourceRequest;
import software.amazon.awssdk.services.cloud9.model.TagResourceResponse;
import software.amazon.awssdk.services.cloud9.model.TooManyRequestsException;
import software.amazon.awssdk.services.cloud9.model.UntagResourceRequest;
import software.amazon.awssdk.services.cloud9.model.UntagResourceResponse;
import software.amazon.awssdk.services.cloud9.model.UpdateEnvironmentMembershipRequest;
import software.amazon.awssdk.services.cloud9.model.UpdateEnvironmentMembershipResponse;
import software.amazon.awssdk.services.cloud9.model.UpdateEnvironmentRequest;
import software.amazon.awssdk.services.cloud9.model.UpdateEnvironmentResponse;
import software.amazon.awssdk.services.cloud9.paginators.DescribeEnvironmentMembershipsIterable;
import software.amazon.awssdk.services.cloud9.paginators.ListEnvironmentsIterable;
import software.amazon.awssdk.services.cloud9.transform.CreateEnvironmentEc2RequestMarshaller;
import software.amazon.awssdk.services.cloud9.transform.CreateEnvironmentMembershipRequestMarshaller;
import software.amazon.awssdk.services.cloud9.transform.DeleteEnvironmentMembershipRequestMarshaller;
import software.amazon.awssdk.services.cloud9.transform.DeleteEnvironmentRequestMarshaller;
import software.amazon.awssdk.services.cloud9.transform.DescribeEnvironmentMembershipsRequestMarshaller;
import software.amazon.awssdk.services.cloud9.transform.DescribeEnvironmentStatusRequestMarshaller;
import software.amazon.awssdk.services.cloud9.transform.DescribeEnvironmentsRequestMarshaller;
import software.amazon.awssdk.services.cloud9.transform.ListEnvironmentsRequestMarshaller;
import software.amazon.awssdk.services.cloud9.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.cloud9.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.cloud9.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.cloud9.transform.UpdateEnvironmentMembershipRequestMarshaller;
import software.amazon.awssdk.services.cloud9.transform.UpdateEnvironmentRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link Cloud9Client}.
 *
 * @see Cloud9Client#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultCloud9Client implements Cloud9Client {
    private static final Logger log = Logger.loggerFor(DefaultCloud9Client.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultCloud9Client(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Creates an Cloud9 development environment, launches an Amazon Elastic Compute Cloud (Amazon EC2) instance, and
     * then connects from the instance to the environment.
     * </p>
     *
     * @param createEnvironmentEc2Request
     * @return Result of the CreateEnvironmentEC2 operation returned by the service.
     * @throws BadRequestException
     *         The target request is invalid.
     * @throws ConflictException
     *         A conflict occurred.
     * @throws NotFoundException
     *         The target resource cannot be found.
     * @throws ForbiddenException
     *         An access permissions issue occurred.
     * @throws TooManyRequestsException
     *         Too many service requests were made over the given time period.
     * @throws LimitExceededException
     *         A service limit was exceeded.
     * @throws InternalServerErrorException
     *         An internal server error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Cloud9Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Cloud9Client.CreateEnvironmentEC2
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloud9-2017-09-23/CreateEnvironmentEC2" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateEnvironmentEc2Response createEnvironmentEC2(CreateEnvironmentEc2Request createEnvironmentEc2Request)
            throws BadRequestException, ConflictException, NotFoundException, ForbiddenException, TooManyRequestsException,
            LimitExceededException, InternalServerErrorException, AwsServiceException, SdkClientException, Cloud9Exception {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateEnvironmentEc2Response> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateEnvironmentEc2Response::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createEnvironmentEc2Request
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cloud9");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateEnvironmentEC2");

            return clientHandler.execute(new ClientExecutionParams<CreateEnvironmentEc2Request, CreateEnvironmentEc2Response>()
                    .withOperationName("CreateEnvironmentEC2").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createEnvironmentEc2Request)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateEnvironmentEc2RequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds an environment member to an Cloud9 development environment.
     * </p>
     *
     * @param createEnvironmentMembershipRequest
     * @return Result of the CreateEnvironmentMembership operation returned by the service.
     * @throws BadRequestException
     *         The target request is invalid.
     * @throws ConflictException
     *         A conflict occurred.
     * @throws NotFoundException
     *         The target resource cannot be found.
     * @throws ForbiddenException
     *         An access permissions issue occurred.
     * @throws TooManyRequestsException
     *         Too many service requests were made over the given time period.
     * @throws LimitExceededException
     *         A service limit was exceeded.
     * @throws InternalServerErrorException
     *         An internal server error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Cloud9Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Cloud9Client.CreateEnvironmentMembership
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloud9-2017-09-23/CreateEnvironmentMembership"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateEnvironmentMembershipResponse createEnvironmentMembership(
            CreateEnvironmentMembershipRequest createEnvironmentMembershipRequest) throws BadRequestException, ConflictException,
            NotFoundException, ForbiddenException, TooManyRequestsException, LimitExceededException,
            InternalServerErrorException, AwsServiceException, SdkClientException, Cloud9Exception {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateEnvironmentMembershipResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateEnvironmentMembershipResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createEnvironmentMembershipRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cloud9");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateEnvironmentMembership");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateEnvironmentMembershipRequest, CreateEnvironmentMembershipResponse>()
                            .withOperationName("CreateEnvironmentMembership").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(createEnvironmentMembershipRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateEnvironmentMembershipRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes an Cloud9 development environment. If an Amazon EC2 instance is connected to the environment, also
     * terminates the instance.
     * </p>
     *
     * @param deleteEnvironmentRequest
     * @return Result of the DeleteEnvironment operation returned by the service.
     * @throws BadRequestException
     *         The target request is invalid.
     * @throws ConflictException
     *         A conflict occurred.
     * @throws NotFoundException
     *         The target resource cannot be found.
     * @throws ForbiddenException
     *         An access permissions issue occurred.
     * @throws TooManyRequestsException
     *         Too many service requests were made over the given time period.
     * @throws LimitExceededException
     *         A service limit was exceeded.
     * @throws InternalServerErrorException
     *         An internal server error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Cloud9Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Cloud9Client.DeleteEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloud9-2017-09-23/DeleteEnvironment" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteEnvironmentResponse deleteEnvironment(DeleteEnvironmentRequest deleteEnvironmentRequest)
            throws BadRequestException, ConflictException, NotFoundException, ForbiddenException, TooManyRequestsException,
            LimitExceededException, InternalServerErrorException, AwsServiceException, SdkClientException, Cloud9Exception {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteEnvironmentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteEnvironmentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteEnvironmentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cloud9");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteEnvironment");

            return clientHandler.execute(new ClientExecutionParams<DeleteEnvironmentRequest, DeleteEnvironmentResponse>()
                    .withOperationName("DeleteEnvironment").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteEnvironmentRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteEnvironmentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes an environment member from a development environment.
     * </p>
     *
     * @param deleteEnvironmentMembershipRequest
     * @return Result of the DeleteEnvironmentMembership operation returned by the service.
     * @throws BadRequestException
     *         The target request is invalid.
     * @throws ConflictException
     *         A conflict occurred.
     * @throws NotFoundException
     *         The target resource cannot be found.
     * @throws ForbiddenException
     *         An access permissions issue occurred.
     * @throws TooManyRequestsException
     *         Too many service requests were made over the given time period.
     * @throws LimitExceededException
     *         A service limit was exceeded.
     * @throws InternalServerErrorException
     *         An internal server error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Cloud9Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Cloud9Client.DeleteEnvironmentMembership
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloud9-2017-09-23/DeleteEnvironmentMembership"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteEnvironmentMembershipResponse deleteEnvironmentMembership(
            DeleteEnvironmentMembershipRequest deleteEnvironmentMembershipRequest) throws BadRequestException, ConflictException,
            NotFoundException, ForbiddenException, TooManyRequestsException, LimitExceededException,
            InternalServerErrorException, AwsServiceException, SdkClientException, Cloud9Exception {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteEnvironmentMembershipResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteEnvironmentMembershipResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteEnvironmentMembershipRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cloud9");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteEnvironmentMembership");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteEnvironmentMembershipRequest, DeleteEnvironmentMembershipResponse>()
                            .withOperationName("DeleteEnvironmentMembership").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(deleteEnvironmentMembershipRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteEnvironmentMembershipRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets information about environment members for an Cloud9 development environment.
     * </p>
     *
     * @param describeEnvironmentMembershipsRequest
     * @return Result of the DescribeEnvironmentMemberships operation returned by the service.
     * @throws BadRequestException
     *         The target request is invalid.
     * @throws ConflictException
     *         A conflict occurred.
     * @throws NotFoundException
     *         The target resource cannot be found.
     * @throws ForbiddenException
     *         An access permissions issue occurred.
     * @throws TooManyRequestsException
     *         Too many service requests were made over the given time period.
     * @throws LimitExceededException
     *         A service limit was exceeded.
     * @throws InternalServerErrorException
     *         An internal server error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Cloud9Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Cloud9Client.DescribeEnvironmentMemberships
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloud9-2017-09-23/DescribeEnvironmentMemberships"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeEnvironmentMembershipsResponse describeEnvironmentMemberships(
            DescribeEnvironmentMembershipsRequest describeEnvironmentMembershipsRequest) throws BadRequestException,
            ConflictException, NotFoundException, ForbiddenException, TooManyRequestsException, LimitExceededException,
            InternalServerErrorException, AwsServiceException, SdkClientException, Cloud9Exception {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeEnvironmentMembershipsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeEnvironmentMembershipsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeEnvironmentMembershipsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cloud9");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeEnvironmentMemberships");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeEnvironmentMembershipsRequest, DescribeEnvironmentMembershipsResponse>()
                            .withOperationName("DescribeEnvironmentMemberships").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeEnvironmentMembershipsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeEnvironmentMembershipsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets information about environment members for an Cloud9 development environment.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeEnvironmentMemberships(software.amazon.awssdk.services.cloud9.model.DescribeEnvironmentMembershipsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloud9.paginators.DescribeEnvironmentMembershipsIterable responses = client.describeEnvironmentMembershipsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.cloud9.paginators.DescribeEnvironmentMembershipsIterable responses = client
     *             .describeEnvironmentMembershipsPaginator(request);
     *     for (software.amazon.awssdk.services.cloud9.model.DescribeEnvironmentMembershipsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloud9.paginators.DescribeEnvironmentMembershipsIterable responses = client.describeEnvironmentMembershipsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeEnvironmentMemberships(software.amazon.awssdk.services.cloud9.model.DescribeEnvironmentMembershipsRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeEnvironmentMembershipsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws BadRequestException
     *         The target request is invalid.
     * @throws ConflictException
     *         A conflict occurred.
     * @throws NotFoundException
     *         The target resource cannot be found.
     * @throws ForbiddenException
     *         An access permissions issue occurred.
     * @throws TooManyRequestsException
     *         Too many service requests were made over the given time period.
     * @throws LimitExceededException
     *         A service limit was exceeded.
     * @throws InternalServerErrorException
     *         An internal server error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Cloud9Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Cloud9Client.DescribeEnvironmentMemberships
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloud9-2017-09-23/DescribeEnvironmentMemberships"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeEnvironmentMembershipsIterable describeEnvironmentMembershipsPaginator(
            DescribeEnvironmentMembershipsRequest describeEnvironmentMembershipsRequest) throws BadRequestException,
            ConflictException, NotFoundException, ForbiddenException, TooManyRequestsException, LimitExceededException,
            InternalServerErrorException, AwsServiceException, SdkClientException, Cloud9Exception {
        return new DescribeEnvironmentMembershipsIterable(this, applyPaginatorUserAgent(describeEnvironmentMembershipsRequest));
    }

    /**
     * <p>
     * Gets status information for an Cloud9 development environment.
     * </p>
     *
     * @param describeEnvironmentStatusRequest
     * @return Result of the DescribeEnvironmentStatus operation returned by the service.
     * @throws BadRequestException
     *         The target request is invalid.
     * @throws ConflictException
     *         A conflict occurred.
     * @throws NotFoundException
     *         The target resource cannot be found.
     * @throws ForbiddenException
     *         An access permissions issue occurred.
     * @throws TooManyRequestsException
     *         Too many service requests were made over the given time period.
     * @throws LimitExceededException
     *         A service limit was exceeded.
     * @throws InternalServerErrorException
     *         An internal server error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Cloud9Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Cloud9Client.DescribeEnvironmentStatus
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloud9-2017-09-23/DescribeEnvironmentStatus"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeEnvironmentStatusResponse describeEnvironmentStatus(
            DescribeEnvironmentStatusRequest describeEnvironmentStatusRequest) throws BadRequestException, ConflictException,
            NotFoundException, ForbiddenException, TooManyRequestsException, LimitExceededException,
            InternalServerErrorException, AwsServiceException, SdkClientException, Cloud9Exception {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeEnvironmentStatusResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeEnvironmentStatusResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeEnvironmentStatusRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cloud9");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeEnvironmentStatus");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeEnvironmentStatusRequest, DescribeEnvironmentStatusResponse>()
                            .withOperationName("DescribeEnvironmentStatus").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeEnvironmentStatusRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeEnvironmentStatusRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets information about Cloud9 development environments.
     * </p>
     *
     * @param describeEnvironmentsRequest
     * @return Result of the DescribeEnvironments operation returned by the service.
     * @throws BadRequestException
     *         The target request is invalid.
     * @throws ConflictException
     *         A conflict occurred.
     * @throws NotFoundException
     *         The target resource cannot be found.
     * @throws ForbiddenException
     *         An access permissions issue occurred.
     * @throws TooManyRequestsException
     *         Too many service requests were made over the given time period.
     * @throws LimitExceededException
     *         A service limit was exceeded.
     * @throws InternalServerErrorException
     *         An internal server error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Cloud9Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Cloud9Client.DescribeEnvironments
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloud9-2017-09-23/DescribeEnvironments" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeEnvironmentsResponse describeEnvironments(DescribeEnvironmentsRequest describeEnvironmentsRequest)
            throws BadRequestException, ConflictException, NotFoundException, ForbiddenException, TooManyRequestsException,
            LimitExceededException, InternalServerErrorException, AwsServiceException, SdkClientException, Cloud9Exception {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeEnvironmentsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeEnvironmentsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeEnvironmentsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cloud9");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeEnvironments");

            return clientHandler.execute(new ClientExecutionParams<DescribeEnvironmentsRequest, DescribeEnvironmentsResponse>()
                    .withOperationName("DescribeEnvironments").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeEnvironmentsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeEnvironmentsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets a list of Cloud9 development environment identifiers.
     * </p>
     *
     * @param listEnvironmentsRequest
     * @return Result of the ListEnvironments operation returned by the service.
     * @throws BadRequestException
     *         The target request is invalid.
     * @throws ConflictException
     *         A conflict occurred.
     * @throws NotFoundException
     *         The target resource cannot be found.
     * @throws ForbiddenException
     *         An access permissions issue occurred.
     * @throws TooManyRequestsException
     *         Too many service requests were made over the given time period.
     * @throws LimitExceededException
     *         A service limit was exceeded.
     * @throws InternalServerErrorException
     *         An internal server error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Cloud9Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Cloud9Client.ListEnvironments
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloud9-2017-09-23/ListEnvironments" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListEnvironmentsResponse listEnvironments(ListEnvironmentsRequest listEnvironmentsRequest) throws BadRequestException,
            ConflictException, NotFoundException, ForbiddenException, TooManyRequestsException, LimitExceededException,
            InternalServerErrorException, AwsServiceException, SdkClientException, Cloud9Exception {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListEnvironmentsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListEnvironmentsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listEnvironmentsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cloud9");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListEnvironments");

            return clientHandler.execute(new ClientExecutionParams<ListEnvironmentsRequest, ListEnvironmentsResponse>()
                    .withOperationName("ListEnvironments").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listEnvironmentsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListEnvironmentsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets a list of Cloud9 development environment identifiers.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listEnvironments(software.amazon.awssdk.services.cloud9.model.ListEnvironmentsRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloud9.paginators.ListEnvironmentsIterable responses = client.listEnvironmentsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.cloud9.paginators.ListEnvironmentsIterable responses = client
     *             .listEnvironmentsPaginator(request);
     *     for (software.amazon.awssdk.services.cloud9.model.ListEnvironmentsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloud9.paginators.ListEnvironmentsIterable responses = client.listEnvironmentsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listEnvironments(software.amazon.awssdk.services.cloud9.model.ListEnvironmentsRequest)} operation.</b>
     * </p>
     *
     * @param listEnvironmentsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws BadRequestException
     *         The target request is invalid.
     * @throws ConflictException
     *         A conflict occurred.
     * @throws NotFoundException
     *         The target resource cannot be found.
     * @throws ForbiddenException
     *         An access permissions issue occurred.
     * @throws TooManyRequestsException
     *         Too many service requests were made over the given time period.
     * @throws LimitExceededException
     *         A service limit was exceeded.
     * @throws InternalServerErrorException
     *         An internal server error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Cloud9Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Cloud9Client.ListEnvironments
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloud9-2017-09-23/ListEnvironments" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListEnvironmentsIterable listEnvironmentsPaginator(ListEnvironmentsRequest listEnvironmentsRequest)
            throws BadRequestException, ConflictException, NotFoundException, ForbiddenException, TooManyRequestsException,
            LimitExceededException, InternalServerErrorException, AwsServiceException, SdkClientException, Cloud9Exception {
        return new ListEnvironmentsIterable(this, applyPaginatorUserAgent(listEnvironmentsRequest));
    }

    /**
     * <p>
     * Gets a list of the tags associated with an Cloud9 development environment.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws NotFoundException
     *         The target resource cannot be found.
     * @throws InternalServerErrorException
     *         An internal server error occurred.
     * @throws BadRequestException
     *         The target request is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Cloud9Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Cloud9Client.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloud9-2017-09-23/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws NotFoundException, InternalServerErrorException, BadRequestException, AwsServiceException, SdkClientException,
            Cloud9Exception {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cloud9");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds tags to an Cloud9 development environment.
     * </p>
     * <important>
     * <p>
     * Tags that you add to an Cloud9 environment by using this method will NOT be automatically propagated to
     * underlying resources.
     * </p>
     * </important>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws NotFoundException
     *         The target resource cannot be found.
     * @throws InternalServerErrorException
     *         An internal server error occurred.
     * @throws BadRequestException
     *         The target request is invalid.
     * @throws ConcurrentAccessException
     *         A concurrent access issue occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Cloud9Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Cloud9Client.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloud9-2017-09-23/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws NotFoundException,
            InternalServerErrorException, BadRequestException, ConcurrentAccessException, AwsServiceException,
            SdkClientException, Cloud9Exception {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cloud9");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(tagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes tags from an Cloud9 development environment.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws NotFoundException
     *         The target resource cannot be found.
     * @throws InternalServerErrorException
     *         An internal server error occurred.
     * @throws BadRequestException
     *         The target request is invalid.
     * @throws ConcurrentAccessException
     *         A concurrent access issue occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Cloud9Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Cloud9Client.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloud9-2017-09-23/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws NotFoundException,
            InternalServerErrorException, BadRequestException, ConcurrentAccessException, AwsServiceException,
            SdkClientException, Cloud9Exception {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cloud9");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Changes the settings of an existing Cloud9 development environment.
     * </p>
     *
     * @param updateEnvironmentRequest
     * @return Result of the UpdateEnvironment operation returned by the service.
     * @throws BadRequestException
     *         The target request is invalid.
     * @throws ConflictException
     *         A conflict occurred.
     * @throws NotFoundException
     *         The target resource cannot be found.
     * @throws ForbiddenException
     *         An access permissions issue occurred.
     * @throws TooManyRequestsException
     *         Too many service requests were made over the given time period.
     * @throws LimitExceededException
     *         A service limit was exceeded.
     * @throws InternalServerErrorException
     *         An internal server error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Cloud9Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Cloud9Client.UpdateEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloud9-2017-09-23/UpdateEnvironment" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateEnvironmentResponse updateEnvironment(UpdateEnvironmentRequest updateEnvironmentRequest)
            throws BadRequestException, ConflictException, NotFoundException, ForbiddenException, TooManyRequestsException,
            LimitExceededException, InternalServerErrorException, AwsServiceException, SdkClientException, Cloud9Exception {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateEnvironmentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateEnvironmentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateEnvironmentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cloud9");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateEnvironment");

            return clientHandler.execute(new ClientExecutionParams<UpdateEnvironmentRequest, UpdateEnvironmentResponse>()
                    .withOperationName("UpdateEnvironment").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateEnvironmentRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateEnvironmentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Changes the settings of an existing environment member for an Cloud9 development environment.
     * </p>
     *
     * @param updateEnvironmentMembershipRequest
     * @return Result of the UpdateEnvironmentMembership operation returned by the service.
     * @throws BadRequestException
     *         The target request is invalid.
     * @throws ConflictException
     *         A conflict occurred.
     * @throws NotFoundException
     *         The target resource cannot be found.
     * @throws ForbiddenException
     *         An access permissions issue occurred.
     * @throws TooManyRequestsException
     *         Too many service requests were made over the given time period.
     * @throws LimitExceededException
     *         A service limit was exceeded.
     * @throws InternalServerErrorException
     *         An internal server error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Cloud9Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Cloud9Client.UpdateEnvironmentMembership
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloud9-2017-09-23/UpdateEnvironmentMembership"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateEnvironmentMembershipResponse updateEnvironmentMembership(
            UpdateEnvironmentMembershipRequest updateEnvironmentMembershipRequest) throws BadRequestException, ConflictException,
            NotFoundException, ForbiddenException, TooManyRequestsException, LimitExceededException,
            InternalServerErrorException, AwsServiceException, SdkClientException, Cloud9Exception {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateEnvironmentMembershipResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateEnvironmentMembershipResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateEnvironmentMembershipRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cloud9");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateEnvironmentMembership");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateEnvironmentMembershipRequest, UpdateEnvironmentMembershipResponse>()
                            .withOperationName("UpdateEnvironmentMembership").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(updateEnvironmentMembershipRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateEnvironmentMembershipRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(Cloud9Exception::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotFoundException")
                                .exceptionBuilderSupplier(NotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ForbiddenException")
                                .exceptionBuilderSupplier(ForbiddenException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyRequestsException")
                                .exceptionBuilderSupplier(TooManyRequestsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequestException")
                                .exceptionBuilderSupplier(BadRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerErrorException")
                                .exceptionBuilderSupplier(InternalServerErrorException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConcurrentAccessException")
                                .exceptionBuilderSupplier(ConcurrentAccessException::builder).httpStatusCode(400).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends Cloud9Request> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
