/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codebuild.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about a build.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Build implements SdkPojo, Serializable, ToCopyableBuilder<Build.Builder, Build> {
    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("id")
            .getter(getter(Build::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("id").build()).build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("arn")
            .getter(getter(Build::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("arn").build()).build();

    private static final SdkField<Long> BUILD_NUMBER_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("buildNumber").getter(getter(Build::buildNumber)).setter(setter(Builder::buildNumber))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("buildNumber").build()).build();

    private static final SdkField<Instant> START_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("startTime").getter(getter(Build::startTime)).setter(setter(Builder::startTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("startTime").build()).build();

    private static final SdkField<Instant> END_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("endTime").getter(getter(Build::endTime)).setter(setter(Builder::endTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("endTime").build()).build();

    private static final SdkField<String> CURRENT_PHASE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("currentPhase").getter(getter(Build::currentPhase)).setter(setter(Builder::currentPhase))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("currentPhase").build()).build();

    private static final SdkField<String> BUILD_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("buildStatus").getter(getter(Build::buildStatusAsString)).setter(setter(Builder::buildStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("buildStatus").build()).build();

    private static final SdkField<String> SOURCE_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("sourceVersion").getter(getter(Build::sourceVersion)).setter(setter(Builder::sourceVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("sourceVersion").build()).build();

    private static final SdkField<String> RESOLVED_SOURCE_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("resolvedSourceVersion").getter(getter(Build::resolvedSourceVersion))
            .setter(setter(Builder::resolvedSourceVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("resolvedSourceVersion").build())
            .build();

    private static final SdkField<String> PROJECT_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("projectName").getter(getter(Build::projectName)).setter(setter(Builder::projectName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("projectName").build()).build();

    private static final SdkField<List<BuildPhase>> PHASES_FIELD = SdkField
            .<List<BuildPhase>> builder(MarshallingType.LIST)
            .memberName("phases")
            .getter(getter(Build::phases))
            .setter(setter(Builder::phases))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("phases").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<BuildPhase> builder(MarshallingType.SDK_POJO)
                                            .constructor(BuildPhase::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<ProjectSource> SOURCE_FIELD = SdkField.<ProjectSource> builder(MarshallingType.SDK_POJO)
            .memberName("source").getter(getter(Build::source)).setter(setter(Builder::source))
            .constructor(ProjectSource::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("source").build()).build();

    private static final SdkField<List<ProjectSource>> SECONDARY_SOURCES_FIELD = SdkField
            .<List<ProjectSource>> builder(MarshallingType.LIST)
            .memberName("secondarySources")
            .getter(getter(Build::secondarySources))
            .setter(setter(Builder::secondarySources))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("secondarySources").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ProjectSource> builder(MarshallingType.SDK_POJO)
                                            .constructor(ProjectSource::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<ProjectSourceVersion>> SECONDARY_SOURCE_VERSIONS_FIELD = SdkField
            .<List<ProjectSourceVersion>> builder(MarshallingType.LIST)
            .memberName("secondarySourceVersions")
            .getter(getter(Build::secondarySourceVersions))
            .setter(setter(Builder::secondarySourceVersions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("secondarySourceVersions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ProjectSourceVersion> builder(MarshallingType.SDK_POJO)
                                            .constructor(ProjectSourceVersion::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<BuildArtifacts> ARTIFACTS_FIELD = SdkField.<BuildArtifacts> builder(MarshallingType.SDK_POJO)
            .memberName("artifacts").getter(getter(Build::artifacts)).setter(setter(Builder::artifacts))
            .constructor(BuildArtifacts::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("artifacts").build()).build();

    private static final SdkField<List<BuildArtifacts>> SECONDARY_ARTIFACTS_FIELD = SdkField
            .<List<BuildArtifacts>> builder(MarshallingType.LIST)
            .memberName("secondaryArtifacts")
            .getter(getter(Build::secondaryArtifacts))
            .setter(setter(Builder::secondaryArtifacts))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("secondaryArtifacts").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<BuildArtifacts> builder(MarshallingType.SDK_POJO)
                                            .constructor(BuildArtifacts::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<ProjectCache> CACHE_FIELD = SdkField.<ProjectCache> builder(MarshallingType.SDK_POJO)
            .memberName("cache").getter(getter(Build::cache)).setter(setter(Builder::cache)).constructor(ProjectCache::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("cache").build()).build();

    private static final SdkField<ProjectEnvironment> ENVIRONMENT_FIELD = SdkField
            .<ProjectEnvironment> builder(MarshallingType.SDK_POJO).memberName("environment").getter(getter(Build::environment))
            .setter(setter(Builder::environment)).constructor(ProjectEnvironment::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("environment").build()).build();

    private static final SdkField<String> SERVICE_ROLE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("serviceRole").getter(getter(Build::serviceRole)).setter(setter(Builder::serviceRole))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("serviceRole").build()).build();

    private static final SdkField<LogsLocation> LOGS_FIELD = SdkField.<LogsLocation> builder(MarshallingType.SDK_POJO)
            .memberName("logs").getter(getter(Build::logs)).setter(setter(Builder::logs)).constructor(LogsLocation::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("logs").build()).build();

    private static final SdkField<Integer> TIMEOUT_IN_MINUTES_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("timeoutInMinutes").getter(getter(Build::timeoutInMinutes)).setter(setter(Builder::timeoutInMinutes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("timeoutInMinutes").build()).build();

    private static final SdkField<Integer> QUEUED_TIMEOUT_IN_MINUTES_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("queuedTimeoutInMinutes").getter(getter(Build::queuedTimeoutInMinutes))
            .setter(setter(Builder::queuedTimeoutInMinutes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("queuedTimeoutInMinutes").build())
            .build();

    private static final SdkField<Boolean> BUILD_COMPLETE_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("buildComplete").getter(getter(Build::buildComplete)).setter(setter(Builder::buildComplete))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("buildComplete").build()).build();

    private static final SdkField<String> INITIATOR_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("initiator").getter(getter(Build::initiator)).setter(setter(Builder::initiator))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("initiator").build()).build();

    private static final SdkField<VpcConfig> VPC_CONFIG_FIELD = SdkField.<VpcConfig> builder(MarshallingType.SDK_POJO)
            .memberName("vpcConfig").getter(getter(Build::vpcConfig)).setter(setter(Builder::vpcConfig))
            .constructor(VpcConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("vpcConfig").build()).build();

    private static final SdkField<NetworkInterface> NETWORK_INTERFACE_FIELD = SdkField
            .<NetworkInterface> builder(MarshallingType.SDK_POJO).memberName("networkInterface")
            .getter(getter(Build::networkInterface)).setter(setter(Builder::networkInterface))
            .constructor(NetworkInterface::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("networkInterface").build()).build();

    private static final SdkField<String> ENCRYPTION_KEY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("encryptionKey").getter(getter(Build::encryptionKey)).setter(setter(Builder::encryptionKey))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("encryptionKey").build()).build();

    private static final SdkField<List<ExportedEnvironmentVariable>> EXPORTED_ENVIRONMENT_VARIABLES_FIELD = SdkField
            .<List<ExportedEnvironmentVariable>> builder(MarshallingType.LIST)
            .memberName("exportedEnvironmentVariables")
            .getter(getter(Build::exportedEnvironmentVariables))
            .setter(setter(Builder::exportedEnvironmentVariables))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("exportedEnvironmentVariables")
                    .build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ExportedEnvironmentVariable> builder(MarshallingType.SDK_POJO)
                                            .constructor(ExportedEnvironmentVariable::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> REPORT_ARNS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("reportArns")
            .getter(getter(Build::reportArns))
            .setter(setter(Builder::reportArns))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("reportArns").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<ProjectFileSystemLocation>> FILE_SYSTEM_LOCATIONS_FIELD = SdkField
            .<List<ProjectFileSystemLocation>> builder(MarshallingType.LIST)
            .memberName("fileSystemLocations")
            .getter(getter(Build::fileSystemLocations))
            .setter(setter(Builder::fileSystemLocations))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("fileSystemLocations").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ProjectFileSystemLocation> builder(MarshallingType.SDK_POJO)
                                            .constructor(ProjectFileSystemLocation::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<DebugSession> DEBUG_SESSION_FIELD = SdkField.<DebugSession> builder(MarshallingType.SDK_POJO)
            .memberName("debugSession").getter(getter(Build::debugSession)).setter(setter(Builder::debugSession))
            .constructor(DebugSession::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("debugSession").build()).build();

    private static final SdkField<String> BUILD_BATCH_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("buildBatchArn").getter(getter(Build::buildBatchArn)).setter(setter(Builder::buildBatchArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("buildBatchArn").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ID_FIELD, ARN_FIELD,
            BUILD_NUMBER_FIELD, START_TIME_FIELD, END_TIME_FIELD, CURRENT_PHASE_FIELD, BUILD_STATUS_FIELD, SOURCE_VERSION_FIELD,
            RESOLVED_SOURCE_VERSION_FIELD, PROJECT_NAME_FIELD, PHASES_FIELD, SOURCE_FIELD, SECONDARY_SOURCES_FIELD,
            SECONDARY_SOURCE_VERSIONS_FIELD, ARTIFACTS_FIELD, SECONDARY_ARTIFACTS_FIELD, CACHE_FIELD, ENVIRONMENT_FIELD,
            SERVICE_ROLE_FIELD, LOGS_FIELD, TIMEOUT_IN_MINUTES_FIELD, QUEUED_TIMEOUT_IN_MINUTES_FIELD, BUILD_COMPLETE_FIELD,
            INITIATOR_FIELD, VPC_CONFIG_FIELD, NETWORK_INTERFACE_FIELD, ENCRYPTION_KEY_FIELD,
            EXPORTED_ENVIRONMENT_VARIABLES_FIELD, REPORT_ARNS_FIELD, FILE_SYSTEM_LOCATIONS_FIELD, DEBUG_SESSION_FIELD,
            BUILD_BATCH_ARN_FIELD));

    private static final long serialVersionUID = 1L;

    private final String id;

    private final String arn;

    private final Long buildNumber;

    private final Instant startTime;

    private final Instant endTime;

    private final String currentPhase;

    private final String buildStatus;

    private final String sourceVersion;

    private final String resolvedSourceVersion;

    private final String projectName;

    private final List<BuildPhase> phases;

    private final ProjectSource source;

    private final List<ProjectSource> secondarySources;

    private final List<ProjectSourceVersion> secondarySourceVersions;

    private final BuildArtifacts artifacts;

    private final List<BuildArtifacts> secondaryArtifacts;

    private final ProjectCache cache;

    private final ProjectEnvironment environment;

    private final String serviceRole;

    private final LogsLocation logs;

    private final Integer timeoutInMinutes;

    private final Integer queuedTimeoutInMinutes;

    private final Boolean buildComplete;

    private final String initiator;

    private final VpcConfig vpcConfig;

    private final NetworkInterface networkInterface;

    private final String encryptionKey;

    private final List<ExportedEnvironmentVariable> exportedEnvironmentVariables;

    private final List<String> reportArns;

    private final List<ProjectFileSystemLocation> fileSystemLocations;

    private final DebugSession debugSession;

    private final String buildBatchArn;

    private Build(BuilderImpl builder) {
        this.id = builder.id;
        this.arn = builder.arn;
        this.buildNumber = builder.buildNumber;
        this.startTime = builder.startTime;
        this.endTime = builder.endTime;
        this.currentPhase = builder.currentPhase;
        this.buildStatus = builder.buildStatus;
        this.sourceVersion = builder.sourceVersion;
        this.resolvedSourceVersion = builder.resolvedSourceVersion;
        this.projectName = builder.projectName;
        this.phases = builder.phases;
        this.source = builder.source;
        this.secondarySources = builder.secondarySources;
        this.secondarySourceVersions = builder.secondarySourceVersions;
        this.artifacts = builder.artifacts;
        this.secondaryArtifacts = builder.secondaryArtifacts;
        this.cache = builder.cache;
        this.environment = builder.environment;
        this.serviceRole = builder.serviceRole;
        this.logs = builder.logs;
        this.timeoutInMinutes = builder.timeoutInMinutes;
        this.queuedTimeoutInMinutes = builder.queuedTimeoutInMinutes;
        this.buildComplete = builder.buildComplete;
        this.initiator = builder.initiator;
        this.vpcConfig = builder.vpcConfig;
        this.networkInterface = builder.networkInterface;
        this.encryptionKey = builder.encryptionKey;
        this.exportedEnvironmentVariables = builder.exportedEnvironmentVariables;
        this.reportArns = builder.reportArns;
        this.fileSystemLocations = builder.fileSystemLocations;
        this.debugSession = builder.debugSession;
        this.buildBatchArn = builder.buildBatchArn;
    }

    /**
     * <p>
     * The unique ID for the build.
     * </p>
     * 
     * @return The unique ID for the build.
     */
    public final String id() {
        return id;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the build.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the build.
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * The number of the build. For each project, the <code>buildNumber</code> of its first build is <code>1</code>. The
     * <code>buildNumber</code> of each subsequent build is incremented by <code>1</code>. If a build is deleted, the
     * <code>buildNumber</code> of other builds does not change.
     * </p>
     * 
     * @return The number of the build. For each project, the <code>buildNumber</code> of its first build is
     *         <code>1</code>. The <code>buildNumber</code> of each subsequent build is incremented by <code>1</code>.
     *         If a build is deleted, the <code>buildNumber</code> of other builds does not change.
     */
    public final Long buildNumber() {
        return buildNumber;
    }

    /**
     * <p>
     * When the build process started, expressed in Unix time format.
     * </p>
     * 
     * @return When the build process started, expressed in Unix time format.
     */
    public final Instant startTime() {
        return startTime;
    }

    /**
     * <p>
     * When the build process ended, expressed in Unix time format.
     * </p>
     * 
     * @return When the build process ended, expressed in Unix time format.
     */
    public final Instant endTime() {
        return endTime;
    }

    /**
     * <p>
     * The current build phase.
     * </p>
     * 
     * @return The current build phase.
     */
    public final String currentPhase() {
        return currentPhase;
    }

    /**
     * <p>
     * The current status of the build. Valid values include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>FAILED</code>: The build failed.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FAULT</code>: The build faulted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>IN_PROGRESS</code>: The build is still in progress.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>STOPPED</code>: The build stopped.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SUCCEEDED</code>: The build succeeded.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>TIMED_OUT</code>: The build timed out.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #buildStatus} will
     * return {@link StatusType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #buildStatusAsString}.
     * </p>
     * 
     * @return The current status of the build. Valid values include:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>FAILED</code>: The build failed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FAULT</code>: The build faulted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>IN_PROGRESS</code>: The build is still in progress.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>STOPPED</code>: The build stopped.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SUCCEEDED</code>: The build succeeded.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>TIMED_OUT</code>: The build timed out.
     *         </p>
     *         </li>
     * @see StatusType
     */
    public final StatusType buildStatus() {
        return StatusType.fromValue(buildStatus);
    }

    /**
     * <p>
     * The current status of the build. Valid values include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>FAILED</code>: The build failed.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FAULT</code>: The build faulted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>IN_PROGRESS</code>: The build is still in progress.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>STOPPED</code>: The build stopped.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SUCCEEDED</code>: The build succeeded.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>TIMED_OUT</code>: The build timed out.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #buildStatus} will
     * return {@link StatusType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #buildStatusAsString}.
     * </p>
     * 
     * @return The current status of the build. Valid values include:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>FAILED</code>: The build failed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FAULT</code>: The build faulted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>IN_PROGRESS</code>: The build is still in progress.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>STOPPED</code>: The build stopped.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SUCCEEDED</code>: The build succeeded.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>TIMED_OUT</code>: The build timed out.
     *         </p>
     *         </li>
     * @see StatusType
     */
    public final String buildStatusAsString() {
        return buildStatus;
    }

    /**
     * <p>
     * Any version identifier for the version of the source code to be built. If <code>sourceVersion</code> is specified
     * at the project level, then this <code>sourceVersion</code> (at the build level) takes precedence.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/codebuild/latest/userguide/sample-source-version.html">Source Version Sample
     * with CodeBuild</a> in the <i>CodeBuild User Guide</i>.
     * </p>
     * 
     * @return Any version identifier for the version of the source code to be built. If <code>sourceVersion</code> is
     *         specified at the project level, then this <code>sourceVersion</code> (at the build level) takes
     *         precedence. </p>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/codebuild/latest/userguide/sample-source-version.html">Source Version
     *         Sample with CodeBuild</a> in the <i>CodeBuild User Guide</i>.
     */
    public final String sourceVersion() {
        return sourceVersion;
    }

    /**
     * <p>
     * An identifier for the version of this build's source code.
     * </p>
     * <ul>
     * <li>
     * <p>
     * For CodeCommit, GitHub, GitHub Enterprise, and BitBucket, the commit ID.
     * </p>
     * </li>
     * <li>
     * <p>
     * For CodePipeline, the source revision provided by CodePipeline.
     * </p>
     * </li>
     * <li>
     * <p>
     * For Amazon S3, this does not apply.
     * </p>
     * </li>
     * </ul>
     * 
     * @return An identifier for the version of this build's source code. </p>
     *         <ul>
     *         <li>
     *         <p>
     *         For CodeCommit, GitHub, GitHub Enterprise, and BitBucket, the commit ID.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For CodePipeline, the source revision provided by CodePipeline.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For Amazon S3, this does not apply.
     *         </p>
     *         </li>
     */
    public final String resolvedSourceVersion() {
        return resolvedSourceVersion;
    }

    /**
     * <p>
     * The name of the CodeBuild project.
     * </p>
     * 
     * @return The name of the CodeBuild project.
     */
    public final String projectName() {
        return projectName;
    }

    /**
     * For responses, this returns true if the service returned a value for the Phases property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasPhases() {
        return phases != null && !(phases instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Information about all previous build phases that are complete and information about any current build phase that
     * is not yet complete.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPhases} method.
     * </p>
     * 
     * @return Information about all previous build phases that are complete and information about any current build
     *         phase that is not yet complete.
     */
    public final List<BuildPhase> phases() {
        return phases;
    }

    /**
     * <p>
     * Information about the source code to be built.
     * </p>
     * 
     * @return Information about the source code to be built.
     */
    public final ProjectSource source() {
        return source;
    }

    /**
     * For responses, this returns true if the service returned a value for the SecondarySources property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasSecondarySources() {
        return secondarySources != null && !(secondarySources instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of <code>ProjectSource</code> objects.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSecondarySources} method.
     * </p>
     * 
     * @return An array of <code>ProjectSource</code> objects.
     */
    public final List<ProjectSource> secondarySources() {
        return secondarySources;
    }

    /**
     * For responses, this returns true if the service returned a value for the SecondarySourceVersions property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasSecondarySourceVersions() {
        return secondarySourceVersions != null && !(secondarySourceVersions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of <code>ProjectSourceVersion</code> objects. Each <code>ProjectSourceVersion</code> must be one of:
     * </p>
     * <ul>
     * <li>
     * <p>
     * For CodeCommit: the commit ID, branch, or Git tag to use.
     * </p>
     * </li>
     * <li>
     * <p>
     * For GitHub: the commit ID, pull request ID, branch name, or tag name that corresponds to the version of the
     * source code you want to build. If a pull request ID is specified, it must use the format
     * <code>pr/pull-request-ID</code> (for example, <code>pr/25</code>). If a branch name is specified, the branch's
     * HEAD commit ID is used. If not specified, the default branch's HEAD commit ID is used.
     * </p>
     * </li>
     * <li>
     * <p>
     * For Bitbucket: the commit ID, branch name, or tag name that corresponds to the version of the source code you
     * want to build. If a branch name is specified, the branch's HEAD commit ID is used. If not specified, the default
     * branch's HEAD commit ID is used.
     * </p>
     * </li>
     * <li>
     * <p>
     * For Amazon S3: the version ID of the object that represents the build input ZIP file to use.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSecondarySourceVersions} method.
     * </p>
     * 
     * @return An array of <code>ProjectSourceVersion</code> objects. Each <code>ProjectSourceVersion</code> must be one
     *         of: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         For CodeCommit: the commit ID, branch, or Git tag to use.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For GitHub: the commit ID, pull request ID, branch name, or tag name that corresponds to the version of
     *         the source code you want to build. If a pull request ID is specified, it must use the format
     *         <code>pr/pull-request-ID</code> (for example, <code>pr/25</code>). If a branch name is specified, the
     *         branch's HEAD commit ID is used. If not specified, the default branch's HEAD commit ID is used.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For Bitbucket: the commit ID, branch name, or tag name that corresponds to the version of the source code
     *         you want to build. If a branch name is specified, the branch's HEAD commit ID is used. If not specified,
     *         the default branch's HEAD commit ID is used.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For Amazon S3: the version ID of the object that represents the build input ZIP file to use.
     *         </p>
     *         </li>
     */
    public final List<ProjectSourceVersion> secondarySourceVersions() {
        return secondarySourceVersions;
    }

    /**
     * <p>
     * Information about the output artifacts for the build.
     * </p>
     * 
     * @return Information about the output artifacts for the build.
     */
    public final BuildArtifacts artifacts() {
        return artifacts;
    }

    /**
     * For responses, this returns true if the service returned a value for the SecondaryArtifacts property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasSecondaryArtifacts() {
        return secondaryArtifacts != null && !(secondaryArtifacts instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of <code>ProjectArtifacts</code> objects.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSecondaryArtifacts} method.
     * </p>
     * 
     * @return An array of <code>ProjectArtifacts</code> objects.
     */
    public final List<BuildArtifacts> secondaryArtifacts() {
        return secondaryArtifacts;
    }

    /**
     * <p>
     * Information about the cache for the build.
     * </p>
     * 
     * @return Information about the cache for the build.
     */
    public final ProjectCache cache() {
        return cache;
    }

    /**
     * <p>
     * Information about the build environment for this build.
     * </p>
     * 
     * @return Information about the build environment for this build.
     */
    public final ProjectEnvironment environment() {
        return environment;
    }

    /**
     * <p>
     * The name of a service role used for this build.
     * </p>
     * 
     * @return The name of a service role used for this build.
     */
    public final String serviceRole() {
        return serviceRole;
    }

    /**
     * <p>
     * Information about the build's logs in CloudWatch Logs.
     * </p>
     * 
     * @return Information about the build's logs in CloudWatch Logs.
     */
    public final LogsLocation logs() {
        return logs;
    }

    /**
     * <p>
     * How long, in minutes, for CodeBuild to wait before timing out this build if it does not get marked as completed.
     * </p>
     * 
     * @return How long, in minutes, for CodeBuild to wait before timing out this build if it does not get marked as
     *         completed.
     */
    public final Integer timeoutInMinutes() {
        return timeoutInMinutes;
    }

    /**
     * <p>
     * The number of minutes a build is allowed to be queued before it times out.
     * </p>
     * 
     * @return The number of minutes a build is allowed to be queued before it times out.
     */
    public final Integer queuedTimeoutInMinutes() {
        return queuedTimeoutInMinutes;
    }

    /**
     * <p>
     * Whether the build is complete. True if complete; otherwise, false.
     * </p>
     * 
     * @return Whether the build is complete. True if complete; otherwise, false.
     */
    public final Boolean buildComplete() {
        return buildComplete;
    }

    /**
     * <p>
     * The entity that started the build. Valid values include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If CodePipeline started the build, the pipeline's name (for example, <code>codepipeline/my-demo-pipeline</code>).
     * </p>
     * </li>
     * <li>
     * <p>
     * If a user started the build, the user's name (for example, <code>MyUserName</code>).
     * </p>
     * </li>
     * <li>
     * <p>
     * If the Jenkins plugin for CodeBuild started the build, the string <code>CodeBuild-Jenkins-Plugin</code>.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The entity that started the build. Valid values include:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         If CodePipeline started the build, the pipeline's name (for example,
     *         <code>codepipeline/my-demo-pipeline</code>).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If a user started the build, the user's name (for example, <code>MyUserName</code>).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If the Jenkins plugin for CodeBuild started the build, the string <code>CodeBuild-Jenkins-Plugin</code>.
     *         </p>
     *         </li>
     */
    public final String initiator() {
        return initiator;
    }

    /**
     * <p>
     * If your CodeBuild project accesses resources in an Amazon VPC, you provide this parameter that identifies the VPC
     * ID and the list of security group IDs and subnet IDs. The security groups and subnets must belong to the same
     * VPC. You must provide at least one security group and one subnet ID.
     * </p>
     * 
     * @return If your CodeBuild project accesses resources in an Amazon VPC, you provide this parameter that identifies
     *         the VPC ID and the list of security group IDs and subnet IDs. The security groups and subnets must belong
     *         to the same VPC. You must provide at least one security group and one subnet ID.
     */
    public final VpcConfig vpcConfig() {
        return vpcConfig;
    }

    /**
     * <p>
     * Describes a network interface.
     * </p>
     * 
     * @return Describes a network interface.
     */
    public final NetworkInterface networkInterface() {
        return networkInterface;
    }

    /**
     * <p>
     * The Key Management Service customer master key (CMK) to be used for encrypting the build output artifacts.
     * </p>
     * <note>
     * <p>
     * You can use a cross-account KMS key to encrypt the build output artifacts if your service role has permission to
     * that key.
     * </p>
     * </note>
     * <p>
     * You can specify either the Amazon Resource Name (ARN) of the CMK or, if available, the CMK's alias (using the
     * format <code>alias/&lt;alias-name&gt;</code>).
     * </p>
     * 
     * @return The Key Management Service customer master key (CMK) to be used for encrypting the build output
     *         artifacts.</p> <note>
     *         <p>
     *         You can use a cross-account KMS key to encrypt the build output artifacts if your service role has
     *         permission to that key.
     *         </p>
     *         </note>
     *         <p>
     *         You can specify either the Amazon Resource Name (ARN) of the CMK or, if available, the CMK's alias (using
     *         the format <code>alias/&lt;alias-name&gt;</code>).
     */
    public final String encryptionKey() {
        return encryptionKey;
    }

    /**
     * For responses, this returns true if the service returned a value for the ExportedEnvironmentVariables property.
     * This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasExportedEnvironmentVariables() {
        return exportedEnvironmentVariables != null && !(exportedEnvironmentVariables instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of exported environment variables for this build.
     * </p>
     * <p>
     * Exported environment variables are used in conjunction with CodePipeline to export environment variables from the
     * current build stage to subsequent stages in the pipeline. For more information, see <a
     * href="https://docs.aws.amazon.com/codepipeline/latest/userguide/actions-variables.html">Working with
     * variables</a> in the <i>CodePipeline User Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasExportedEnvironmentVariables} method.
     * </p>
     * 
     * @return A list of exported environment variables for this build.</p>
     *         <p>
     *         Exported environment variables are used in conjunction with CodePipeline to export environment variables
     *         from the current build stage to subsequent stages in the pipeline. For more information, see <a
     *         href="https://docs.aws.amazon.com/codepipeline/latest/userguide/actions-variables.html">Working with
     *         variables</a> in the <i>CodePipeline User Guide</i>.
     */
    public final List<ExportedEnvironmentVariable> exportedEnvironmentVariables() {
        return exportedEnvironmentVariables;
    }

    /**
     * For responses, this returns true if the service returned a value for the ReportArns property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasReportArns() {
        return reportArns != null && !(reportArns instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of the ARNs associated with this build's reports.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasReportArns} method.
     * </p>
     * 
     * @return An array of the ARNs associated with this build's reports.
     */
    public final List<String> reportArns() {
        return reportArns;
    }

    /**
     * For responses, this returns true if the service returned a value for the FileSystemLocations property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasFileSystemLocations() {
        return fileSystemLocations != null && !(fileSystemLocations instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of <code>ProjectFileSystemLocation</code> objects for a CodeBuild build project. A
     * <code>ProjectFileSystemLocation</code> object specifies the <code>identifier</code>, <code>location</code>,
     * <code>mountOptions</code>, <code>mountPoint</code>, and <code>type</code> of a file system created using Amazon
     * Elastic File System.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasFileSystemLocations} method.
     * </p>
     * 
     * @return An array of <code>ProjectFileSystemLocation</code> objects for a CodeBuild build project. A
     *         <code>ProjectFileSystemLocation</code> object specifies the <code>identifier</code>,
     *         <code>location</code>, <code>mountOptions</code>, <code>mountPoint</code>, and <code>type</code> of a
     *         file system created using Amazon Elastic File System.
     */
    public final List<ProjectFileSystemLocation> fileSystemLocations() {
        return fileSystemLocations;
    }

    /**
     * <p>
     * Contains information about the debug session for this build.
     * </p>
     * 
     * @return Contains information about the debug session for this build.
     */
    public final DebugSession debugSession() {
        return debugSession;
    }

    /**
     * <p>
     * The ARN of the batch build that this build is a member of, if applicable.
     * </p>
     * 
     * @return The ARN of the batch build that this build is a member of, if applicable.
     */
    public final String buildBatchArn() {
        return buildBatchArn;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(buildNumber());
        hashCode = 31 * hashCode + Objects.hashCode(startTime());
        hashCode = 31 * hashCode + Objects.hashCode(endTime());
        hashCode = 31 * hashCode + Objects.hashCode(currentPhase());
        hashCode = 31 * hashCode + Objects.hashCode(buildStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(sourceVersion());
        hashCode = 31 * hashCode + Objects.hashCode(resolvedSourceVersion());
        hashCode = 31 * hashCode + Objects.hashCode(projectName());
        hashCode = 31 * hashCode + Objects.hashCode(hasPhases() ? phases() : null);
        hashCode = 31 * hashCode + Objects.hashCode(source());
        hashCode = 31 * hashCode + Objects.hashCode(hasSecondarySources() ? secondarySources() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasSecondarySourceVersions() ? secondarySourceVersions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(artifacts());
        hashCode = 31 * hashCode + Objects.hashCode(hasSecondaryArtifacts() ? secondaryArtifacts() : null);
        hashCode = 31 * hashCode + Objects.hashCode(cache());
        hashCode = 31 * hashCode + Objects.hashCode(environment());
        hashCode = 31 * hashCode + Objects.hashCode(serviceRole());
        hashCode = 31 * hashCode + Objects.hashCode(logs());
        hashCode = 31 * hashCode + Objects.hashCode(timeoutInMinutes());
        hashCode = 31 * hashCode + Objects.hashCode(queuedTimeoutInMinutes());
        hashCode = 31 * hashCode + Objects.hashCode(buildComplete());
        hashCode = 31 * hashCode + Objects.hashCode(initiator());
        hashCode = 31 * hashCode + Objects.hashCode(vpcConfig());
        hashCode = 31 * hashCode + Objects.hashCode(networkInterface());
        hashCode = 31 * hashCode + Objects.hashCode(encryptionKey());
        hashCode = 31 * hashCode + Objects.hashCode(hasExportedEnvironmentVariables() ? exportedEnvironmentVariables() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasReportArns() ? reportArns() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasFileSystemLocations() ? fileSystemLocations() : null);
        hashCode = 31 * hashCode + Objects.hashCode(debugSession());
        hashCode = 31 * hashCode + Objects.hashCode(buildBatchArn());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Build)) {
            return false;
        }
        Build other = (Build) obj;
        return Objects.equals(id(), other.id()) && Objects.equals(arn(), other.arn())
                && Objects.equals(buildNumber(), other.buildNumber()) && Objects.equals(startTime(), other.startTime())
                && Objects.equals(endTime(), other.endTime()) && Objects.equals(currentPhase(), other.currentPhase())
                && Objects.equals(buildStatusAsString(), other.buildStatusAsString())
                && Objects.equals(sourceVersion(), other.sourceVersion())
                && Objects.equals(resolvedSourceVersion(), other.resolvedSourceVersion())
                && Objects.equals(projectName(), other.projectName()) && hasPhases() == other.hasPhases()
                && Objects.equals(phases(), other.phases()) && Objects.equals(source(), other.source())
                && hasSecondarySources() == other.hasSecondarySources()
                && Objects.equals(secondarySources(), other.secondarySources())
                && hasSecondarySourceVersions() == other.hasSecondarySourceVersions()
                && Objects.equals(secondarySourceVersions(), other.secondarySourceVersions())
                && Objects.equals(artifacts(), other.artifacts()) && hasSecondaryArtifacts() == other.hasSecondaryArtifacts()
                && Objects.equals(secondaryArtifacts(), other.secondaryArtifacts()) && Objects.equals(cache(), other.cache())
                && Objects.equals(environment(), other.environment()) && Objects.equals(serviceRole(), other.serviceRole())
                && Objects.equals(logs(), other.logs()) && Objects.equals(timeoutInMinutes(), other.timeoutInMinutes())
                && Objects.equals(queuedTimeoutInMinutes(), other.queuedTimeoutInMinutes())
                && Objects.equals(buildComplete(), other.buildComplete()) && Objects.equals(initiator(), other.initiator())
                && Objects.equals(vpcConfig(), other.vpcConfig()) && Objects.equals(networkInterface(), other.networkInterface())
                && Objects.equals(encryptionKey(), other.encryptionKey())
                && hasExportedEnvironmentVariables() == other.hasExportedEnvironmentVariables()
                && Objects.equals(exportedEnvironmentVariables(), other.exportedEnvironmentVariables())
                && hasReportArns() == other.hasReportArns() && Objects.equals(reportArns(), other.reportArns())
                && hasFileSystemLocations() == other.hasFileSystemLocations()
                && Objects.equals(fileSystemLocations(), other.fileSystemLocations())
                && Objects.equals(debugSession(), other.debugSession()) && Objects.equals(buildBatchArn(), other.buildBatchArn());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Build").add("Id", id()).add("Arn", arn()).add("BuildNumber", buildNumber())
                .add("StartTime", startTime()).add("EndTime", endTime()).add("CurrentPhase", currentPhase())
                .add("BuildStatus", buildStatusAsString()).add("SourceVersion", sourceVersion())
                .add("ResolvedSourceVersion", resolvedSourceVersion()).add("ProjectName", projectName())
                .add("Phases", hasPhases() ? phases() : null).add("Source", source())
                .add("SecondarySources", hasSecondarySources() ? secondarySources() : null)
                .add("SecondarySourceVersions", hasSecondarySourceVersions() ? secondarySourceVersions() : null)
                .add("Artifacts", artifacts()).add("SecondaryArtifacts", hasSecondaryArtifacts() ? secondaryArtifacts() : null)
                .add("Cache", cache()).add("Environment", environment()).add("ServiceRole", serviceRole()).add("Logs", logs())
                .add("TimeoutInMinutes", timeoutInMinutes()).add("QueuedTimeoutInMinutes", queuedTimeoutInMinutes())
                .add("BuildComplete", buildComplete()).add("Initiator", initiator()).add("VpcConfig", vpcConfig())
                .add("NetworkInterface", networkInterface()).add("EncryptionKey", encryptionKey())
                .add("ExportedEnvironmentVariables", hasExportedEnvironmentVariables() ? exportedEnvironmentVariables() : null)
                .add("ReportArns", hasReportArns() ? reportArns() : null)
                .add("FileSystemLocations", hasFileSystemLocations() ? fileSystemLocations() : null)
                .add("DebugSession", debugSession()).add("BuildBatchArn", buildBatchArn()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "id":
            return Optional.ofNullable(clazz.cast(id()));
        case "arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "buildNumber":
            return Optional.ofNullable(clazz.cast(buildNumber()));
        case "startTime":
            return Optional.ofNullable(clazz.cast(startTime()));
        case "endTime":
            return Optional.ofNullable(clazz.cast(endTime()));
        case "currentPhase":
            return Optional.ofNullable(clazz.cast(currentPhase()));
        case "buildStatus":
            return Optional.ofNullable(clazz.cast(buildStatusAsString()));
        case "sourceVersion":
            return Optional.ofNullable(clazz.cast(sourceVersion()));
        case "resolvedSourceVersion":
            return Optional.ofNullable(clazz.cast(resolvedSourceVersion()));
        case "projectName":
            return Optional.ofNullable(clazz.cast(projectName()));
        case "phases":
            return Optional.ofNullable(clazz.cast(phases()));
        case "source":
            return Optional.ofNullable(clazz.cast(source()));
        case "secondarySources":
            return Optional.ofNullable(clazz.cast(secondarySources()));
        case "secondarySourceVersions":
            return Optional.ofNullable(clazz.cast(secondarySourceVersions()));
        case "artifacts":
            return Optional.ofNullable(clazz.cast(artifacts()));
        case "secondaryArtifacts":
            return Optional.ofNullable(clazz.cast(secondaryArtifacts()));
        case "cache":
            return Optional.ofNullable(clazz.cast(cache()));
        case "environment":
            return Optional.ofNullable(clazz.cast(environment()));
        case "serviceRole":
            return Optional.ofNullable(clazz.cast(serviceRole()));
        case "logs":
            return Optional.ofNullable(clazz.cast(logs()));
        case "timeoutInMinutes":
            return Optional.ofNullable(clazz.cast(timeoutInMinutes()));
        case "queuedTimeoutInMinutes":
            return Optional.ofNullable(clazz.cast(queuedTimeoutInMinutes()));
        case "buildComplete":
            return Optional.ofNullable(clazz.cast(buildComplete()));
        case "initiator":
            return Optional.ofNullable(clazz.cast(initiator()));
        case "vpcConfig":
            return Optional.ofNullable(clazz.cast(vpcConfig()));
        case "networkInterface":
            return Optional.ofNullable(clazz.cast(networkInterface()));
        case "encryptionKey":
            return Optional.ofNullable(clazz.cast(encryptionKey()));
        case "exportedEnvironmentVariables":
            return Optional.ofNullable(clazz.cast(exportedEnvironmentVariables()));
        case "reportArns":
            return Optional.ofNullable(clazz.cast(reportArns()));
        case "fileSystemLocations":
            return Optional.ofNullable(clazz.cast(fileSystemLocations()));
        case "debugSession":
            return Optional.ofNullable(clazz.cast(debugSession()));
        case "buildBatchArn":
            return Optional.ofNullable(clazz.cast(buildBatchArn()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Build, T> g) {
        return obj -> g.apply((Build) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Build> {
        /**
         * <p>
         * The unique ID for the build.
         * </p>
         * 
         * @param id
         *        The unique ID for the build.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the build.
         * </p>
         * 
         * @param arn
         *        The Amazon Resource Name (ARN) of the build.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The number of the build. For each project, the <code>buildNumber</code> of its first build is <code>1</code>.
         * The <code>buildNumber</code> of each subsequent build is incremented by <code>1</code>. If a build is
         * deleted, the <code>buildNumber</code> of other builds does not change.
         * </p>
         * 
         * @param buildNumber
         *        The number of the build. For each project, the <code>buildNumber</code> of its first build is
         *        <code>1</code>. The <code>buildNumber</code> of each subsequent build is incremented by <code>1</code>
         *        . If a build is deleted, the <code>buildNumber</code> of other builds does not change.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder buildNumber(Long buildNumber);

        /**
         * <p>
         * When the build process started, expressed in Unix time format.
         * </p>
         * 
         * @param startTime
         *        When the build process started, expressed in Unix time format.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startTime(Instant startTime);

        /**
         * <p>
         * When the build process ended, expressed in Unix time format.
         * </p>
         * 
         * @param endTime
         *        When the build process ended, expressed in Unix time format.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endTime(Instant endTime);

        /**
         * <p>
         * The current build phase.
         * </p>
         * 
         * @param currentPhase
         *        The current build phase.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder currentPhase(String currentPhase);

        /**
         * <p>
         * The current status of the build. Valid values include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>FAILED</code>: The build failed.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FAULT</code>: The build faulted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IN_PROGRESS</code>: The build is still in progress.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>STOPPED</code>: The build stopped.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SUCCEEDED</code>: The build succeeded.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>TIMED_OUT</code>: The build timed out.
         * </p>
         * </li>
         * </ul>
         * 
         * @param buildStatus
         *        The current status of the build. Valid values include:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>FAILED</code>: The build failed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FAULT</code>: The build faulted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>IN_PROGRESS</code>: The build is still in progress.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>STOPPED</code>: The build stopped.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SUCCEEDED</code>: The build succeeded.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>TIMED_OUT</code>: The build timed out.
         *        </p>
         *        </li>
         * @see StatusType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StatusType
         */
        Builder buildStatus(String buildStatus);

        /**
         * <p>
         * The current status of the build. Valid values include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>FAILED</code>: The build failed.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FAULT</code>: The build faulted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IN_PROGRESS</code>: The build is still in progress.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>STOPPED</code>: The build stopped.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SUCCEEDED</code>: The build succeeded.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>TIMED_OUT</code>: The build timed out.
         * </p>
         * </li>
         * </ul>
         * 
         * @param buildStatus
         *        The current status of the build. Valid values include:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>FAILED</code>: The build failed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FAULT</code>: The build faulted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>IN_PROGRESS</code>: The build is still in progress.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>STOPPED</code>: The build stopped.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SUCCEEDED</code>: The build succeeded.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>TIMED_OUT</code>: The build timed out.
         *        </p>
         *        </li>
         * @see StatusType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StatusType
         */
        Builder buildStatus(StatusType buildStatus);

        /**
         * <p>
         * Any version identifier for the version of the source code to be built. If <code>sourceVersion</code> is
         * specified at the project level, then this <code>sourceVersion</code> (at the build level) takes precedence.
         * </p>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/codebuild/latest/userguide/sample-source-version.html">Source Version
         * Sample with CodeBuild</a> in the <i>CodeBuild User Guide</i>.
         * </p>
         * 
         * @param sourceVersion
         *        Any version identifier for the version of the source code to be built. If <code>sourceVersion</code>
         *        is specified at the project level, then this <code>sourceVersion</code> (at the build level) takes
         *        precedence. </p>
         *        <p>
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/codebuild/latest/userguide/sample-source-version.html">Source
         *        Version Sample with CodeBuild</a> in the <i>CodeBuild User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceVersion(String sourceVersion);

        /**
         * <p>
         * An identifier for the version of this build's source code.
         * </p>
         * <ul>
         * <li>
         * <p>
         * For CodeCommit, GitHub, GitHub Enterprise, and BitBucket, the commit ID.
         * </p>
         * </li>
         * <li>
         * <p>
         * For CodePipeline, the source revision provided by CodePipeline.
         * </p>
         * </li>
         * <li>
         * <p>
         * For Amazon S3, this does not apply.
         * </p>
         * </li>
         * </ul>
         * 
         * @param resolvedSourceVersion
         *        An identifier for the version of this build's source code. </p>
         *        <ul>
         *        <li>
         *        <p>
         *        For CodeCommit, GitHub, GitHub Enterprise, and BitBucket, the commit ID.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For CodePipeline, the source revision provided by CodePipeline.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For Amazon S3, this does not apply.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resolvedSourceVersion(String resolvedSourceVersion);

        /**
         * <p>
         * The name of the CodeBuild project.
         * </p>
         * 
         * @param projectName
         *        The name of the CodeBuild project.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder projectName(String projectName);

        /**
         * <p>
         * Information about all previous build phases that are complete and information about any current build phase
         * that is not yet complete.
         * </p>
         * 
         * @param phases
         *        Information about all previous build phases that are complete and information about any current build
         *        phase that is not yet complete.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder phases(Collection<BuildPhase> phases);

        /**
         * <p>
         * Information about all previous build phases that are complete and information about any current build phase
         * that is not yet complete.
         * </p>
         * 
         * @param phases
         *        Information about all previous build phases that are complete and information about any current build
         *        phase that is not yet complete.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder phases(BuildPhase... phases);

        /**
         * <p>
         * Information about all previous build phases that are complete and information about any current build phase
         * that is not yet complete.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.codebuild.model.BuildPhase.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.codebuild.model.BuildPhase#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.codebuild.model.BuildPhase.Builder#build()} is called immediately and
         * its result is passed to {@link #phases(List<BuildPhase>)}.
         * 
         * @param phases
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.codebuild.model.BuildPhase.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #phases(java.util.Collection<BuildPhase>)
         */
        Builder phases(Consumer<BuildPhase.Builder>... phases);

        /**
         * <p>
         * Information about the source code to be built.
         * </p>
         * 
         * @param source
         *        Information about the source code to be built.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder source(ProjectSource source);

        /**
         * <p>
         * Information about the source code to be built.
         * </p>
         * This is a convenience method that creates an instance of the {@link ProjectSource.Builder} avoiding the need
         * to create one manually via {@link ProjectSource#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ProjectSource.Builder#build()} is called immediately and its
         * result is passed to {@link #source(ProjectSource)}.
         * 
         * @param source
         *        a consumer that will call methods on {@link ProjectSource.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #source(ProjectSource)
         */
        default Builder source(Consumer<ProjectSource.Builder> source) {
            return source(ProjectSource.builder().applyMutation(source).build());
        }

        /**
         * <p>
         * An array of <code>ProjectSource</code> objects.
         * </p>
         * 
         * @param secondarySources
         *        An array of <code>ProjectSource</code> objects.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secondarySources(Collection<ProjectSource> secondarySources);

        /**
         * <p>
         * An array of <code>ProjectSource</code> objects.
         * </p>
         * 
         * @param secondarySources
         *        An array of <code>ProjectSource</code> objects.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secondarySources(ProjectSource... secondarySources);

        /**
         * <p>
         * An array of <code>ProjectSource</code> objects.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.codebuild.model.ProjectSource.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.codebuild.model.ProjectSource#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.codebuild.model.ProjectSource.Builder#build()} is called immediately
         * and its result is passed to {@link #secondarySources(List<ProjectSource>)}.
         * 
         * @param secondarySources
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.codebuild.model.ProjectSource.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #secondarySources(java.util.Collection<ProjectSource>)
         */
        Builder secondarySources(Consumer<ProjectSource.Builder>... secondarySources);

        /**
         * <p>
         * An array of <code>ProjectSourceVersion</code> objects. Each <code>ProjectSourceVersion</code> must be one of:
         * </p>
         * <ul>
         * <li>
         * <p>
         * For CodeCommit: the commit ID, branch, or Git tag to use.
         * </p>
         * </li>
         * <li>
         * <p>
         * For GitHub: the commit ID, pull request ID, branch name, or tag name that corresponds to the version of the
         * source code you want to build. If a pull request ID is specified, it must use the format
         * <code>pr/pull-request-ID</code> (for example, <code>pr/25</code>). If a branch name is specified, the
         * branch's HEAD commit ID is used. If not specified, the default branch's HEAD commit ID is used.
         * </p>
         * </li>
         * <li>
         * <p>
         * For Bitbucket: the commit ID, branch name, or tag name that corresponds to the version of the source code you
         * want to build. If a branch name is specified, the branch's HEAD commit ID is used. If not specified, the
         * default branch's HEAD commit ID is used.
         * </p>
         * </li>
         * <li>
         * <p>
         * For Amazon S3: the version ID of the object that represents the build input ZIP file to use.
         * </p>
         * </li>
         * </ul>
         * 
         * @param secondarySourceVersions
         *        An array of <code>ProjectSourceVersion</code> objects. Each <code>ProjectSourceVersion</code> must be
         *        one of: </p>
         *        <ul>
         *        <li>
         *        <p>
         *        For CodeCommit: the commit ID, branch, or Git tag to use.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For GitHub: the commit ID, pull request ID, branch name, or tag name that corresponds to the version
         *        of the source code you want to build. If a pull request ID is specified, it must use the format
         *        <code>pr/pull-request-ID</code> (for example, <code>pr/25</code>). If a branch name is specified, the
         *        branch's HEAD commit ID is used. If not specified, the default branch's HEAD commit ID is used.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For Bitbucket: the commit ID, branch name, or tag name that corresponds to the version of the source
         *        code you want to build. If a branch name is specified, the branch's HEAD commit ID is used. If not
         *        specified, the default branch's HEAD commit ID is used.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For Amazon S3: the version ID of the object that represents the build input ZIP file to use.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secondarySourceVersions(Collection<ProjectSourceVersion> secondarySourceVersions);

        /**
         * <p>
         * An array of <code>ProjectSourceVersion</code> objects. Each <code>ProjectSourceVersion</code> must be one of:
         * </p>
         * <ul>
         * <li>
         * <p>
         * For CodeCommit: the commit ID, branch, or Git tag to use.
         * </p>
         * </li>
         * <li>
         * <p>
         * For GitHub: the commit ID, pull request ID, branch name, or tag name that corresponds to the version of the
         * source code you want to build. If a pull request ID is specified, it must use the format
         * <code>pr/pull-request-ID</code> (for example, <code>pr/25</code>). If a branch name is specified, the
         * branch's HEAD commit ID is used. If not specified, the default branch's HEAD commit ID is used.
         * </p>
         * </li>
         * <li>
         * <p>
         * For Bitbucket: the commit ID, branch name, or tag name that corresponds to the version of the source code you
         * want to build. If a branch name is specified, the branch's HEAD commit ID is used. If not specified, the
         * default branch's HEAD commit ID is used.
         * </p>
         * </li>
         * <li>
         * <p>
         * For Amazon S3: the version ID of the object that represents the build input ZIP file to use.
         * </p>
         * </li>
         * </ul>
         * 
         * @param secondarySourceVersions
         *        An array of <code>ProjectSourceVersion</code> objects. Each <code>ProjectSourceVersion</code> must be
         *        one of: </p>
         *        <ul>
         *        <li>
         *        <p>
         *        For CodeCommit: the commit ID, branch, or Git tag to use.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For GitHub: the commit ID, pull request ID, branch name, or tag name that corresponds to the version
         *        of the source code you want to build. If a pull request ID is specified, it must use the format
         *        <code>pr/pull-request-ID</code> (for example, <code>pr/25</code>). If a branch name is specified, the
         *        branch's HEAD commit ID is used. If not specified, the default branch's HEAD commit ID is used.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For Bitbucket: the commit ID, branch name, or tag name that corresponds to the version of the source
         *        code you want to build. If a branch name is specified, the branch's HEAD commit ID is used. If not
         *        specified, the default branch's HEAD commit ID is used.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For Amazon S3: the version ID of the object that represents the build input ZIP file to use.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secondarySourceVersions(ProjectSourceVersion... secondarySourceVersions);

        /**
         * <p>
         * An array of <code>ProjectSourceVersion</code> objects. Each <code>ProjectSourceVersion</code> must be one of:
         * </p>
         * <ul>
         * <li>
         * <p>
         * For CodeCommit: the commit ID, branch, or Git tag to use.
         * </p>
         * </li>
         * <li>
         * <p>
         * For GitHub: the commit ID, pull request ID, branch name, or tag name that corresponds to the version of the
         * source code you want to build. If a pull request ID is specified, it must use the format
         * <code>pr/pull-request-ID</code> (for example, <code>pr/25</code>). If a branch name is specified, the
         * branch's HEAD commit ID is used. If not specified, the default branch's HEAD commit ID is used.
         * </p>
         * </li>
         * <li>
         * <p>
         * For Bitbucket: the commit ID, branch name, or tag name that corresponds to the version of the source code you
         * want to build. If a branch name is specified, the branch's HEAD commit ID is used. If not specified, the
         * default branch's HEAD commit ID is used.
         * </p>
         * </li>
         * <li>
         * <p>
         * For Amazon S3: the version ID of the object that represents the build input ZIP file to use.
         * </p>
         * </li>
         * </ul>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.codebuild.model.ProjectSourceVersion.Builder} avoiding the need to
         * create one manually via
         * {@link software.amazon.awssdk.services.codebuild.model.ProjectSourceVersion#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.codebuild.model.ProjectSourceVersion.Builder#build()} is called
         * immediately and its result is passed to {@link #secondarySourceVersions(List<ProjectSourceVersion>)}.
         * 
         * @param secondarySourceVersions
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.codebuild.model.ProjectSourceVersion.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #secondarySourceVersions(java.util.Collection<ProjectSourceVersion>)
         */
        Builder secondarySourceVersions(Consumer<ProjectSourceVersion.Builder>... secondarySourceVersions);

        /**
         * <p>
         * Information about the output artifacts for the build.
         * </p>
         * 
         * @param artifacts
         *        Information about the output artifacts for the build.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder artifacts(BuildArtifacts artifacts);

        /**
         * <p>
         * Information about the output artifacts for the build.
         * </p>
         * This is a convenience method that creates an instance of the {@link BuildArtifacts.Builder} avoiding the need
         * to create one manually via {@link BuildArtifacts#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link BuildArtifacts.Builder#build()} is called immediately and its
         * result is passed to {@link #artifacts(BuildArtifacts)}.
         * 
         * @param artifacts
         *        a consumer that will call methods on {@link BuildArtifacts.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #artifacts(BuildArtifacts)
         */
        default Builder artifacts(Consumer<BuildArtifacts.Builder> artifacts) {
            return artifacts(BuildArtifacts.builder().applyMutation(artifacts).build());
        }

        /**
         * <p>
         * An array of <code>ProjectArtifacts</code> objects.
         * </p>
         * 
         * @param secondaryArtifacts
         *        An array of <code>ProjectArtifacts</code> objects.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secondaryArtifacts(Collection<BuildArtifacts> secondaryArtifacts);

        /**
         * <p>
         * An array of <code>ProjectArtifacts</code> objects.
         * </p>
         * 
         * @param secondaryArtifacts
         *        An array of <code>ProjectArtifacts</code> objects.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secondaryArtifacts(BuildArtifacts... secondaryArtifacts);

        /**
         * <p>
         * An array of <code>ProjectArtifacts</code> objects.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.codebuild.model.BuildArtifacts.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.codebuild.model.BuildArtifacts#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.codebuild.model.BuildArtifacts.Builder#build()} is called immediately
         * and its result is passed to {@link #secondaryArtifacts(List<BuildArtifacts>)}.
         * 
         * @param secondaryArtifacts
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.codebuild.model.BuildArtifacts.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #secondaryArtifacts(java.util.Collection<BuildArtifacts>)
         */
        Builder secondaryArtifacts(Consumer<BuildArtifacts.Builder>... secondaryArtifacts);

        /**
         * <p>
         * Information about the cache for the build.
         * </p>
         * 
         * @param cache
         *        Information about the cache for the build.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cache(ProjectCache cache);

        /**
         * <p>
         * Information about the cache for the build.
         * </p>
         * This is a convenience method that creates an instance of the {@link ProjectCache.Builder} avoiding the need
         * to create one manually via {@link ProjectCache#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ProjectCache.Builder#build()} is called immediately and its
         * result is passed to {@link #cache(ProjectCache)}.
         * 
         * @param cache
         *        a consumer that will call methods on {@link ProjectCache.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #cache(ProjectCache)
         */
        default Builder cache(Consumer<ProjectCache.Builder> cache) {
            return cache(ProjectCache.builder().applyMutation(cache).build());
        }

        /**
         * <p>
         * Information about the build environment for this build.
         * </p>
         * 
         * @param environment
         *        Information about the build environment for this build.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder environment(ProjectEnvironment environment);

        /**
         * <p>
         * Information about the build environment for this build.
         * </p>
         * This is a convenience method that creates an instance of the {@link ProjectEnvironment.Builder} avoiding the
         * need to create one manually via {@link ProjectEnvironment#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ProjectEnvironment.Builder#build()} is called immediately and its
         * result is passed to {@link #environment(ProjectEnvironment)}.
         * 
         * @param environment
         *        a consumer that will call methods on {@link ProjectEnvironment.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #environment(ProjectEnvironment)
         */
        default Builder environment(Consumer<ProjectEnvironment.Builder> environment) {
            return environment(ProjectEnvironment.builder().applyMutation(environment).build());
        }

        /**
         * <p>
         * The name of a service role used for this build.
         * </p>
         * 
         * @param serviceRole
         *        The name of a service role used for this build.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serviceRole(String serviceRole);

        /**
         * <p>
         * Information about the build's logs in CloudWatch Logs.
         * </p>
         * 
         * @param logs
         *        Information about the build's logs in CloudWatch Logs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder logs(LogsLocation logs);

        /**
         * <p>
         * Information about the build's logs in CloudWatch Logs.
         * </p>
         * This is a convenience method that creates an instance of the {@link LogsLocation.Builder} avoiding the need
         * to create one manually via {@link LogsLocation#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link LogsLocation.Builder#build()} is called immediately and its
         * result is passed to {@link #logs(LogsLocation)}.
         * 
         * @param logs
         *        a consumer that will call methods on {@link LogsLocation.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #logs(LogsLocation)
         */
        default Builder logs(Consumer<LogsLocation.Builder> logs) {
            return logs(LogsLocation.builder().applyMutation(logs).build());
        }

        /**
         * <p>
         * How long, in minutes, for CodeBuild to wait before timing out this build if it does not get marked as
         * completed.
         * </p>
         * 
         * @param timeoutInMinutes
         *        How long, in minutes, for CodeBuild to wait before timing out this build if it does not get marked as
         *        completed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder timeoutInMinutes(Integer timeoutInMinutes);

        /**
         * <p>
         * The number of minutes a build is allowed to be queued before it times out.
         * </p>
         * 
         * @param queuedTimeoutInMinutes
         *        The number of minutes a build is allowed to be queued before it times out.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder queuedTimeoutInMinutes(Integer queuedTimeoutInMinutes);

        /**
         * <p>
         * Whether the build is complete. True if complete; otherwise, false.
         * </p>
         * 
         * @param buildComplete
         *        Whether the build is complete. True if complete; otherwise, false.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder buildComplete(Boolean buildComplete);

        /**
         * <p>
         * The entity that started the build. Valid values include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * If CodePipeline started the build, the pipeline's name (for example,
         * <code>codepipeline/my-demo-pipeline</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * If a user started the build, the user's name (for example, <code>MyUserName</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * If the Jenkins plugin for CodeBuild started the build, the string <code>CodeBuild-Jenkins-Plugin</code>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param initiator
         *        The entity that started the build. Valid values include:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        If CodePipeline started the build, the pipeline's name (for example,
         *        <code>codepipeline/my-demo-pipeline</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If a user started the build, the user's name (for example, <code>MyUserName</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If the Jenkins plugin for CodeBuild started the build, the string
         *        <code>CodeBuild-Jenkins-Plugin</code>.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder initiator(String initiator);

        /**
         * <p>
         * If your CodeBuild project accesses resources in an Amazon VPC, you provide this parameter that identifies the
         * VPC ID and the list of security group IDs and subnet IDs. The security groups and subnets must belong to the
         * same VPC. You must provide at least one security group and one subnet ID.
         * </p>
         * 
         * @param vpcConfig
         *        If your CodeBuild project accesses resources in an Amazon VPC, you provide this parameter that
         *        identifies the VPC ID and the list of security group IDs and subnet IDs. The security groups and
         *        subnets must belong to the same VPC. You must provide at least one security group and one subnet ID.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcConfig(VpcConfig vpcConfig);

        /**
         * <p>
         * If your CodeBuild project accesses resources in an Amazon VPC, you provide this parameter that identifies the
         * VPC ID and the list of security group IDs and subnet IDs. The security groups and subnets must belong to the
         * same VPC. You must provide at least one security group and one subnet ID.
         * </p>
         * This is a convenience method that creates an instance of the {@link VpcConfig.Builder} avoiding the need to
         * create one manually via {@link VpcConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link VpcConfig.Builder#build()} is called immediately and its result
         * is passed to {@link #vpcConfig(VpcConfig)}.
         * 
         * @param vpcConfig
         *        a consumer that will call methods on {@link VpcConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #vpcConfig(VpcConfig)
         */
        default Builder vpcConfig(Consumer<VpcConfig.Builder> vpcConfig) {
            return vpcConfig(VpcConfig.builder().applyMutation(vpcConfig).build());
        }

        /**
         * <p>
         * Describes a network interface.
         * </p>
         * 
         * @param networkInterface
         *        Describes a network interface.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder networkInterface(NetworkInterface networkInterface);

        /**
         * <p>
         * Describes a network interface.
         * </p>
         * This is a convenience method that creates an instance of the {@link NetworkInterface.Builder} avoiding the
         * need to create one manually via {@link NetworkInterface#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link NetworkInterface.Builder#build()} is called immediately and its
         * result is passed to {@link #networkInterface(NetworkInterface)}.
         * 
         * @param networkInterface
         *        a consumer that will call methods on {@link NetworkInterface.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #networkInterface(NetworkInterface)
         */
        default Builder networkInterface(Consumer<NetworkInterface.Builder> networkInterface) {
            return networkInterface(NetworkInterface.builder().applyMutation(networkInterface).build());
        }

        /**
         * <p>
         * The Key Management Service customer master key (CMK) to be used for encrypting the build output artifacts.
         * </p>
         * <note>
         * <p>
         * You can use a cross-account KMS key to encrypt the build output artifacts if your service role has permission
         * to that key.
         * </p>
         * </note>
         * <p>
         * You can specify either the Amazon Resource Name (ARN) of the CMK or, if available, the CMK's alias (using the
         * format <code>alias/&lt;alias-name&gt;</code>).
         * </p>
         * 
         * @param encryptionKey
         *        The Key Management Service customer master key (CMK) to be used for encrypting the build output
         *        artifacts.</p> <note>
         *        <p>
         *        You can use a cross-account KMS key to encrypt the build output artifacts if your service role has
         *        permission to that key.
         *        </p>
         *        </note>
         *        <p>
         *        You can specify either the Amazon Resource Name (ARN) of the CMK or, if available, the CMK's alias
         *        (using the format <code>alias/&lt;alias-name&gt;</code>).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder encryptionKey(String encryptionKey);

        /**
         * <p>
         * A list of exported environment variables for this build.
         * </p>
         * <p>
         * Exported environment variables are used in conjunction with CodePipeline to export environment variables from
         * the current build stage to subsequent stages in the pipeline. For more information, see <a
         * href="https://docs.aws.amazon.com/codepipeline/latest/userguide/actions-variables.html">Working with
         * variables</a> in the <i>CodePipeline User Guide</i>.
         * </p>
         * 
         * @param exportedEnvironmentVariables
         *        A list of exported environment variables for this build.</p>
         *        <p>
         *        Exported environment variables are used in conjunction with CodePipeline to export environment
         *        variables from the current build stage to subsequent stages in the pipeline. For more information, see
         *        <a href="https://docs.aws.amazon.com/codepipeline/latest/userguide/actions-variables.html">Working
         *        with variables</a> in the <i>CodePipeline User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder exportedEnvironmentVariables(Collection<ExportedEnvironmentVariable> exportedEnvironmentVariables);

        /**
         * <p>
         * A list of exported environment variables for this build.
         * </p>
         * <p>
         * Exported environment variables are used in conjunction with CodePipeline to export environment variables from
         * the current build stage to subsequent stages in the pipeline. For more information, see <a
         * href="https://docs.aws.amazon.com/codepipeline/latest/userguide/actions-variables.html">Working with
         * variables</a> in the <i>CodePipeline User Guide</i>.
         * </p>
         * 
         * @param exportedEnvironmentVariables
         *        A list of exported environment variables for this build.</p>
         *        <p>
         *        Exported environment variables are used in conjunction with CodePipeline to export environment
         *        variables from the current build stage to subsequent stages in the pipeline. For more information, see
         *        <a href="https://docs.aws.amazon.com/codepipeline/latest/userguide/actions-variables.html">Working
         *        with variables</a> in the <i>CodePipeline User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder exportedEnvironmentVariables(ExportedEnvironmentVariable... exportedEnvironmentVariables);

        /**
         * <p>
         * A list of exported environment variables for this build.
         * </p>
         * <p>
         * Exported environment variables are used in conjunction with CodePipeline to export environment variables from
         * the current build stage to subsequent stages in the pipeline. For more information, see <a
         * href="https://docs.aws.amazon.com/codepipeline/latest/userguide/actions-variables.html">Working with
         * variables</a> in the <i>CodePipeline User Guide</i>.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.codebuild.model.ExportedEnvironmentVariable.Builder} avoiding the need
         * to create one manually via
         * {@link software.amazon.awssdk.services.codebuild.model.ExportedEnvironmentVariable#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.codebuild.model.ExportedEnvironmentVariable.Builder#build()} is called
         * immediately and its result is passed to {@link
         * #exportedEnvironmentVariables(List<ExportedEnvironmentVariable>)}.
         * 
         * @param exportedEnvironmentVariables
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.codebuild.model.ExportedEnvironmentVariable.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #exportedEnvironmentVariables(java.util.Collection<ExportedEnvironmentVariable>)
         */
        Builder exportedEnvironmentVariables(Consumer<ExportedEnvironmentVariable.Builder>... exportedEnvironmentVariables);

        /**
         * <p>
         * An array of the ARNs associated with this build's reports.
         * </p>
         * 
         * @param reportArns
         *        An array of the ARNs associated with this build's reports.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder reportArns(Collection<String> reportArns);

        /**
         * <p>
         * An array of the ARNs associated with this build's reports.
         * </p>
         * 
         * @param reportArns
         *        An array of the ARNs associated with this build's reports.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder reportArns(String... reportArns);

        /**
         * <p>
         * An array of <code>ProjectFileSystemLocation</code> objects for a CodeBuild build project. A
         * <code>ProjectFileSystemLocation</code> object specifies the <code>identifier</code>, <code>location</code>,
         * <code>mountOptions</code>, <code>mountPoint</code>, and <code>type</code> of a file system created using
         * Amazon Elastic File System.
         * </p>
         * 
         * @param fileSystemLocations
         *        An array of <code>ProjectFileSystemLocation</code> objects for a CodeBuild build project. A
         *        <code>ProjectFileSystemLocation</code> object specifies the <code>identifier</code>,
         *        <code>location</code>, <code>mountOptions</code>, <code>mountPoint</code>, and <code>type</code> of a
         *        file system created using Amazon Elastic File System.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fileSystemLocations(Collection<ProjectFileSystemLocation> fileSystemLocations);

        /**
         * <p>
         * An array of <code>ProjectFileSystemLocation</code> objects for a CodeBuild build project. A
         * <code>ProjectFileSystemLocation</code> object specifies the <code>identifier</code>, <code>location</code>,
         * <code>mountOptions</code>, <code>mountPoint</code>, and <code>type</code> of a file system created using
         * Amazon Elastic File System.
         * </p>
         * 
         * @param fileSystemLocations
         *        An array of <code>ProjectFileSystemLocation</code> objects for a CodeBuild build project. A
         *        <code>ProjectFileSystemLocation</code> object specifies the <code>identifier</code>,
         *        <code>location</code>, <code>mountOptions</code>, <code>mountPoint</code>, and <code>type</code> of a
         *        file system created using Amazon Elastic File System.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fileSystemLocations(ProjectFileSystemLocation... fileSystemLocations);

        /**
         * <p>
         * An array of <code>ProjectFileSystemLocation</code> objects for a CodeBuild build project. A
         * <code>ProjectFileSystemLocation</code> object specifies the <code>identifier</code>, <code>location</code>,
         * <code>mountOptions</code>, <code>mountPoint</code>, and <code>type</code> of a file system created using
         * Amazon Elastic File System.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.codebuild.model.ProjectFileSystemLocation.Builder} avoiding the need
         * to create one manually via
         * {@link software.amazon.awssdk.services.codebuild.model.ProjectFileSystemLocation#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.codebuild.model.ProjectFileSystemLocation.Builder#build()} is called
         * immediately and its result is passed to {@link #fileSystemLocations(List<ProjectFileSystemLocation>)}.
         * 
         * @param fileSystemLocations
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.codebuild.model.ProjectFileSystemLocation.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #fileSystemLocations(java.util.Collection<ProjectFileSystemLocation>)
         */
        Builder fileSystemLocations(Consumer<ProjectFileSystemLocation.Builder>... fileSystemLocations);

        /**
         * <p>
         * Contains information about the debug session for this build.
         * </p>
         * 
         * @param debugSession
         *        Contains information about the debug session for this build.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder debugSession(DebugSession debugSession);

        /**
         * <p>
         * Contains information about the debug session for this build.
         * </p>
         * This is a convenience method that creates an instance of the {@link DebugSession.Builder} avoiding the need
         * to create one manually via {@link DebugSession#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link DebugSession.Builder#build()} is called immediately and its
         * result is passed to {@link #debugSession(DebugSession)}.
         * 
         * @param debugSession
         *        a consumer that will call methods on {@link DebugSession.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #debugSession(DebugSession)
         */
        default Builder debugSession(Consumer<DebugSession.Builder> debugSession) {
            return debugSession(DebugSession.builder().applyMutation(debugSession).build());
        }

        /**
         * <p>
         * The ARN of the batch build that this build is a member of, if applicable.
         * </p>
         * 
         * @param buildBatchArn
         *        The ARN of the batch build that this build is a member of, if applicable.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder buildBatchArn(String buildBatchArn);
    }

    static final class BuilderImpl implements Builder {
        private String id;

        private String arn;

        private Long buildNumber;

        private Instant startTime;

        private Instant endTime;

        private String currentPhase;

        private String buildStatus;

        private String sourceVersion;

        private String resolvedSourceVersion;

        private String projectName;

        private List<BuildPhase> phases = DefaultSdkAutoConstructList.getInstance();

        private ProjectSource source;

        private List<ProjectSource> secondarySources = DefaultSdkAutoConstructList.getInstance();

        private List<ProjectSourceVersion> secondarySourceVersions = DefaultSdkAutoConstructList.getInstance();

        private BuildArtifacts artifacts;

        private List<BuildArtifacts> secondaryArtifacts = DefaultSdkAutoConstructList.getInstance();

        private ProjectCache cache;

        private ProjectEnvironment environment;

        private String serviceRole;

        private LogsLocation logs;

        private Integer timeoutInMinutes;

        private Integer queuedTimeoutInMinutes;

        private Boolean buildComplete;

        private String initiator;

        private VpcConfig vpcConfig;

        private NetworkInterface networkInterface;

        private String encryptionKey;

        private List<ExportedEnvironmentVariable> exportedEnvironmentVariables = DefaultSdkAutoConstructList.getInstance();

        private List<String> reportArns = DefaultSdkAutoConstructList.getInstance();

        private List<ProjectFileSystemLocation> fileSystemLocations = DefaultSdkAutoConstructList.getInstance();

        private DebugSession debugSession;

        private String buildBatchArn;

        private BuilderImpl() {
        }

        private BuilderImpl(Build model) {
            id(model.id);
            arn(model.arn);
            buildNumber(model.buildNumber);
            startTime(model.startTime);
            endTime(model.endTime);
            currentPhase(model.currentPhase);
            buildStatus(model.buildStatus);
            sourceVersion(model.sourceVersion);
            resolvedSourceVersion(model.resolvedSourceVersion);
            projectName(model.projectName);
            phases(model.phases);
            source(model.source);
            secondarySources(model.secondarySources);
            secondarySourceVersions(model.secondarySourceVersions);
            artifacts(model.artifacts);
            secondaryArtifacts(model.secondaryArtifacts);
            cache(model.cache);
            environment(model.environment);
            serviceRole(model.serviceRole);
            logs(model.logs);
            timeoutInMinutes(model.timeoutInMinutes);
            queuedTimeoutInMinutes(model.queuedTimeoutInMinutes);
            buildComplete(model.buildComplete);
            initiator(model.initiator);
            vpcConfig(model.vpcConfig);
            networkInterface(model.networkInterface);
            encryptionKey(model.encryptionKey);
            exportedEnvironmentVariables(model.exportedEnvironmentVariables);
            reportArns(model.reportArns);
            fileSystemLocations(model.fileSystemLocations);
            debugSession(model.debugSession);
            buildBatchArn(model.buildBatchArn);
        }

        public final String getId() {
            return id;
        }

        public final void setId(String id) {
            this.id = id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final Long getBuildNumber() {
            return buildNumber;
        }

        public final void setBuildNumber(Long buildNumber) {
            this.buildNumber = buildNumber;
        }

        @Override
        public final Builder buildNumber(Long buildNumber) {
            this.buildNumber = buildNumber;
            return this;
        }

        public final Instant getStartTime() {
            return startTime;
        }

        public final void setStartTime(Instant startTime) {
            this.startTime = startTime;
        }

        @Override
        public final Builder startTime(Instant startTime) {
            this.startTime = startTime;
            return this;
        }

        public final Instant getEndTime() {
            return endTime;
        }

        public final void setEndTime(Instant endTime) {
            this.endTime = endTime;
        }

        @Override
        public final Builder endTime(Instant endTime) {
            this.endTime = endTime;
            return this;
        }

        public final String getCurrentPhase() {
            return currentPhase;
        }

        public final void setCurrentPhase(String currentPhase) {
            this.currentPhase = currentPhase;
        }

        @Override
        public final Builder currentPhase(String currentPhase) {
            this.currentPhase = currentPhase;
            return this;
        }

        public final String getBuildStatus() {
            return buildStatus;
        }

        public final void setBuildStatus(String buildStatus) {
            this.buildStatus = buildStatus;
        }

        @Override
        public final Builder buildStatus(String buildStatus) {
            this.buildStatus = buildStatus;
            return this;
        }

        @Override
        public final Builder buildStatus(StatusType buildStatus) {
            this.buildStatus(buildStatus == null ? null : buildStatus.toString());
            return this;
        }

        public final String getSourceVersion() {
            return sourceVersion;
        }

        public final void setSourceVersion(String sourceVersion) {
            this.sourceVersion = sourceVersion;
        }

        @Override
        public final Builder sourceVersion(String sourceVersion) {
            this.sourceVersion = sourceVersion;
            return this;
        }

        public final String getResolvedSourceVersion() {
            return resolvedSourceVersion;
        }

        public final void setResolvedSourceVersion(String resolvedSourceVersion) {
            this.resolvedSourceVersion = resolvedSourceVersion;
        }

        @Override
        public final Builder resolvedSourceVersion(String resolvedSourceVersion) {
            this.resolvedSourceVersion = resolvedSourceVersion;
            return this;
        }

        public final String getProjectName() {
            return projectName;
        }

        public final void setProjectName(String projectName) {
            this.projectName = projectName;
        }

        @Override
        public final Builder projectName(String projectName) {
            this.projectName = projectName;
            return this;
        }

        public final List<BuildPhase.Builder> getPhases() {
            List<BuildPhase.Builder> result = BuildPhasesCopier.copyToBuilder(this.phases);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setPhases(Collection<BuildPhase.BuilderImpl> phases) {
            this.phases = BuildPhasesCopier.copyFromBuilder(phases);
        }

        @Override
        public final Builder phases(Collection<BuildPhase> phases) {
            this.phases = BuildPhasesCopier.copy(phases);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder phases(BuildPhase... phases) {
            phases(Arrays.asList(phases));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder phases(Consumer<BuildPhase.Builder>... phases) {
            phases(Stream.of(phases).map(c -> BuildPhase.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final ProjectSource.Builder getSource() {
            return source != null ? source.toBuilder() : null;
        }

        public final void setSource(ProjectSource.BuilderImpl source) {
            this.source = source != null ? source.build() : null;
        }

        @Override
        public final Builder source(ProjectSource source) {
            this.source = source;
            return this;
        }

        public final List<ProjectSource.Builder> getSecondarySources() {
            List<ProjectSource.Builder> result = ProjectSourcesCopier.copyToBuilder(this.secondarySources);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setSecondarySources(Collection<ProjectSource.BuilderImpl> secondarySources) {
            this.secondarySources = ProjectSourcesCopier.copyFromBuilder(secondarySources);
        }

        @Override
        public final Builder secondarySources(Collection<ProjectSource> secondarySources) {
            this.secondarySources = ProjectSourcesCopier.copy(secondarySources);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder secondarySources(ProjectSource... secondarySources) {
            secondarySources(Arrays.asList(secondarySources));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder secondarySources(Consumer<ProjectSource.Builder>... secondarySources) {
            secondarySources(Stream.of(secondarySources).map(c -> ProjectSource.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final List<ProjectSourceVersion.Builder> getSecondarySourceVersions() {
            List<ProjectSourceVersion.Builder> result = ProjectSecondarySourceVersionsCopier
                    .copyToBuilder(this.secondarySourceVersions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setSecondarySourceVersions(Collection<ProjectSourceVersion.BuilderImpl> secondarySourceVersions) {
            this.secondarySourceVersions = ProjectSecondarySourceVersionsCopier.copyFromBuilder(secondarySourceVersions);
        }

        @Override
        public final Builder secondarySourceVersions(Collection<ProjectSourceVersion> secondarySourceVersions) {
            this.secondarySourceVersions = ProjectSecondarySourceVersionsCopier.copy(secondarySourceVersions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder secondarySourceVersions(ProjectSourceVersion... secondarySourceVersions) {
            secondarySourceVersions(Arrays.asList(secondarySourceVersions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder secondarySourceVersions(Consumer<ProjectSourceVersion.Builder>... secondarySourceVersions) {
            secondarySourceVersions(Stream.of(secondarySourceVersions)
                    .map(c -> ProjectSourceVersion.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final BuildArtifacts.Builder getArtifacts() {
            return artifacts != null ? artifacts.toBuilder() : null;
        }

        public final void setArtifacts(BuildArtifacts.BuilderImpl artifacts) {
            this.artifacts = artifacts != null ? artifacts.build() : null;
        }

        @Override
        public final Builder artifacts(BuildArtifacts artifacts) {
            this.artifacts = artifacts;
            return this;
        }

        public final List<BuildArtifacts.Builder> getSecondaryArtifacts() {
            List<BuildArtifacts.Builder> result = BuildArtifactsListCopier.copyToBuilder(this.secondaryArtifacts);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setSecondaryArtifacts(Collection<BuildArtifacts.BuilderImpl> secondaryArtifacts) {
            this.secondaryArtifacts = BuildArtifactsListCopier.copyFromBuilder(secondaryArtifacts);
        }

        @Override
        public final Builder secondaryArtifacts(Collection<BuildArtifacts> secondaryArtifacts) {
            this.secondaryArtifacts = BuildArtifactsListCopier.copy(secondaryArtifacts);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder secondaryArtifacts(BuildArtifacts... secondaryArtifacts) {
            secondaryArtifacts(Arrays.asList(secondaryArtifacts));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder secondaryArtifacts(Consumer<BuildArtifacts.Builder>... secondaryArtifacts) {
            secondaryArtifacts(Stream.of(secondaryArtifacts).map(c -> BuildArtifacts.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final ProjectCache.Builder getCache() {
            return cache != null ? cache.toBuilder() : null;
        }

        public final void setCache(ProjectCache.BuilderImpl cache) {
            this.cache = cache != null ? cache.build() : null;
        }

        @Override
        public final Builder cache(ProjectCache cache) {
            this.cache = cache;
            return this;
        }

        public final ProjectEnvironment.Builder getEnvironment() {
            return environment != null ? environment.toBuilder() : null;
        }

        public final void setEnvironment(ProjectEnvironment.BuilderImpl environment) {
            this.environment = environment != null ? environment.build() : null;
        }

        @Override
        public final Builder environment(ProjectEnvironment environment) {
            this.environment = environment;
            return this;
        }

        public final String getServiceRole() {
            return serviceRole;
        }

        public final void setServiceRole(String serviceRole) {
            this.serviceRole = serviceRole;
        }

        @Override
        public final Builder serviceRole(String serviceRole) {
            this.serviceRole = serviceRole;
            return this;
        }

        public final LogsLocation.Builder getLogs() {
            return logs != null ? logs.toBuilder() : null;
        }

        public final void setLogs(LogsLocation.BuilderImpl logs) {
            this.logs = logs != null ? logs.build() : null;
        }

        @Override
        public final Builder logs(LogsLocation logs) {
            this.logs = logs;
            return this;
        }

        public final Integer getTimeoutInMinutes() {
            return timeoutInMinutes;
        }

        public final void setTimeoutInMinutes(Integer timeoutInMinutes) {
            this.timeoutInMinutes = timeoutInMinutes;
        }

        @Override
        public final Builder timeoutInMinutes(Integer timeoutInMinutes) {
            this.timeoutInMinutes = timeoutInMinutes;
            return this;
        }

        public final Integer getQueuedTimeoutInMinutes() {
            return queuedTimeoutInMinutes;
        }

        public final void setQueuedTimeoutInMinutes(Integer queuedTimeoutInMinutes) {
            this.queuedTimeoutInMinutes = queuedTimeoutInMinutes;
        }

        @Override
        public final Builder queuedTimeoutInMinutes(Integer queuedTimeoutInMinutes) {
            this.queuedTimeoutInMinutes = queuedTimeoutInMinutes;
            return this;
        }

        public final Boolean getBuildComplete() {
            return buildComplete;
        }

        public final void setBuildComplete(Boolean buildComplete) {
            this.buildComplete = buildComplete;
        }

        @Override
        public final Builder buildComplete(Boolean buildComplete) {
            this.buildComplete = buildComplete;
            return this;
        }

        public final String getInitiator() {
            return initiator;
        }

        public final void setInitiator(String initiator) {
            this.initiator = initiator;
        }

        @Override
        public final Builder initiator(String initiator) {
            this.initiator = initiator;
            return this;
        }

        public final VpcConfig.Builder getVpcConfig() {
            return vpcConfig != null ? vpcConfig.toBuilder() : null;
        }

        public final void setVpcConfig(VpcConfig.BuilderImpl vpcConfig) {
            this.vpcConfig = vpcConfig != null ? vpcConfig.build() : null;
        }

        @Override
        public final Builder vpcConfig(VpcConfig vpcConfig) {
            this.vpcConfig = vpcConfig;
            return this;
        }

        public final NetworkInterface.Builder getNetworkInterface() {
            return networkInterface != null ? networkInterface.toBuilder() : null;
        }

        public final void setNetworkInterface(NetworkInterface.BuilderImpl networkInterface) {
            this.networkInterface = networkInterface != null ? networkInterface.build() : null;
        }

        @Override
        public final Builder networkInterface(NetworkInterface networkInterface) {
            this.networkInterface = networkInterface;
            return this;
        }

        public final String getEncryptionKey() {
            return encryptionKey;
        }

        public final void setEncryptionKey(String encryptionKey) {
            this.encryptionKey = encryptionKey;
        }

        @Override
        public final Builder encryptionKey(String encryptionKey) {
            this.encryptionKey = encryptionKey;
            return this;
        }

        public final List<ExportedEnvironmentVariable.Builder> getExportedEnvironmentVariables() {
            List<ExportedEnvironmentVariable.Builder> result = ExportedEnvironmentVariablesCopier
                    .copyToBuilder(this.exportedEnvironmentVariables);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setExportedEnvironmentVariables(
                Collection<ExportedEnvironmentVariable.BuilderImpl> exportedEnvironmentVariables) {
            this.exportedEnvironmentVariables = ExportedEnvironmentVariablesCopier.copyFromBuilder(exportedEnvironmentVariables);
        }

        @Override
        public final Builder exportedEnvironmentVariables(Collection<ExportedEnvironmentVariable> exportedEnvironmentVariables) {
            this.exportedEnvironmentVariables = ExportedEnvironmentVariablesCopier.copy(exportedEnvironmentVariables);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder exportedEnvironmentVariables(ExportedEnvironmentVariable... exportedEnvironmentVariables) {
            exportedEnvironmentVariables(Arrays.asList(exportedEnvironmentVariables));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder exportedEnvironmentVariables(
                Consumer<ExportedEnvironmentVariable.Builder>... exportedEnvironmentVariables) {
            exportedEnvironmentVariables(Stream.of(exportedEnvironmentVariables)
                    .map(c -> ExportedEnvironmentVariable.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final Collection<String> getReportArns() {
            if (reportArns instanceof SdkAutoConstructList) {
                return null;
            }
            return reportArns;
        }

        public final void setReportArns(Collection<String> reportArns) {
            this.reportArns = BuildReportArnsCopier.copy(reportArns);
        }

        @Override
        public final Builder reportArns(Collection<String> reportArns) {
            this.reportArns = BuildReportArnsCopier.copy(reportArns);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder reportArns(String... reportArns) {
            reportArns(Arrays.asList(reportArns));
            return this;
        }

        public final List<ProjectFileSystemLocation.Builder> getFileSystemLocations() {
            List<ProjectFileSystemLocation.Builder> result = ProjectFileSystemLocationsCopier
                    .copyToBuilder(this.fileSystemLocations);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setFileSystemLocations(Collection<ProjectFileSystemLocation.BuilderImpl> fileSystemLocations) {
            this.fileSystemLocations = ProjectFileSystemLocationsCopier.copyFromBuilder(fileSystemLocations);
        }

        @Override
        public final Builder fileSystemLocations(Collection<ProjectFileSystemLocation> fileSystemLocations) {
            this.fileSystemLocations = ProjectFileSystemLocationsCopier.copy(fileSystemLocations);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder fileSystemLocations(ProjectFileSystemLocation... fileSystemLocations) {
            fileSystemLocations(Arrays.asList(fileSystemLocations));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder fileSystemLocations(Consumer<ProjectFileSystemLocation.Builder>... fileSystemLocations) {
            fileSystemLocations(Stream.of(fileSystemLocations)
                    .map(c -> ProjectFileSystemLocation.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final DebugSession.Builder getDebugSession() {
            return debugSession != null ? debugSession.toBuilder() : null;
        }

        public final void setDebugSession(DebugSession.BuilderImpl debugSession) {
            this.debugSession = debugSession != null ? debugSession.build() : null;
        }

        @Override
        public final Builder debugSession(DebugSession debugSession) {
            this.debugSession = debugSession;
            return this;
        }

        public final String getBuildBatchArn() {
            return buildBatchArn;
        }

        public final void setBuildBatchArn(String buildBatchArn) {
            this.buildBatchArn = buildBatchArn;
        }

        @Override
        public final Builder buildBatchArn(String buildBatchArn) {
            this.buildBatchArn = buildBatchArn;
            return this;
        }

        @Override
        public Build build() {
            return new Build(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
