/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codepipeline.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The action engine, or executor, for an action type created for a provider, where the action is to be used by
 * customers of the provider. The action engine is associated with the model used to create and update the action, such
 * as the Lambda integration model.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ActionTypeExecutor implements SdkPojo, Serializable,
        ToCopyableBuilder<ActionTypeExecutor.Builder, ActionTypeExecutor> {
    private static final SdkField<ExecutorConfiguration> CONFIGURATION_FIELD = SdkField
            .<ExecutorConfiguration> builder(MarshallingType.SDK_POJO).memberName("configuration")
            .getter(getter(ActionTypeExecutor::configuration)).setter(setter(Builder::configuration))
            .constructor(ExecutorConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("configuration").build()).build();

    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("type")
            .getter(getter(ActionTypeExecutor::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("type").build()).build();

    private static final SdkField<String> POLICY_STATEMENTS_TEMPLATE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("policyStatementsTemplate").getter(getter(ActionTypeExecutor::policyStatementsTemplate))
            .setter(setter(Builder::policyStatementsTemplate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("policyStatementsTemplate").build())
            .build();

    private static final SdkField<Integer> JOB_TIMEOUT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("jobTimeout").getter(getter(ActionTypeExecutor::jobTimeout)).setter(setter(Builder::jobTimeout))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("jobTimeout").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CONFIGURATION_FIELD,
            TYPE_FIELD, POLICY_STATEMENTS_TEMPLATE_FIELD, JOB_TIMEOUT_FIELD));

    private static final long serialVersionUID = 1L;

    private final ExecutorConfiguration configuration;

    private final String type;

    private final String policyStatementsTemplate;

    private final Integer jobTimeout;

    private ActionTypeExecutor(BuilderImpl builder) {
        this.configuration = builder.configuration;
        this.type = builder.type;
        this.policyStatementsTemplate = builder.policyStatementsTemplate;
        this.jobTimeout = builder.jobTimeout;
    }

    /**
     * <p>
     * The action configuration properties for the action type. These properties are specified in the action definition
     * when the action type is created.
     * </p>
     * 
     * @return The action configuration properties for the action type. These properties are specified in the action
     *         definition when the action type is created.
     */
    public final ExecutorConfiguration configuration() {
        return configuration;
    }

    /**
     * <p>
     * The integration model used to create and update the action type, <code>Lambda</code> or <code>JobWorker</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link ExecutorType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The integration model used to create and update the action type, <code>Lambda</code> or
     *         <code>JobWorker</code>.
     * @see ExecutorType
     */
    public final ExecutorType type() {
        return ExecutorType.fromValue(type);
    }

    /**
     * <p>
     * The integration model used to create and update the action type, <code>Lambda</code> or <code>JobWorker</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link ExecutorType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The integration model used to create and update the action type, <code>Lambda</code> or
     *         <code>JobWorker</code>.
     * @see ExecutorType
     */
    public final String typeAsString() {
        return type;
    }

    /**
     * <p>
     * The policy statement that specifies the permissions in the CodePipeline customer’s account that are needed to
     * successfully run an action.
     * </p>
     * <p>
     * To grant permission to another account, specify the account ID as the Principal, a domain-style identifier
     * defined by the service, for example <code>codepipeline.amazonaws.com</code>.
     * </p>
     * <note>
     * <p>
     * The size of the passed JSON policy document cannot exceed 2048 characters.
     * </p>
     * </note>
     * 
     * @return The policy statement that specifies the permissions in the CodePipeline customer’s account that are
     *         needed to successfully run an action.</p>
     *         <p>
     *         To grant permission to another account, specify the account ID as the Principal, a domain-style
     *         identifier defined by the service, for example <code>codepipeline.amazonaws.com</code>.
     *         </p>
     *         <note>
     *         <p>
     *         The size of the passed JSON policy document cannot exceed 2048 characters.
     *         </p>
     */
    public final String policyStatementsTemplate() {
        return policyStatementsTemplate;
    }

    /**
     * <p>
     * The timeout in seconds for the job. An action execution can have multiple jobs. This is the timeout for a single
     * job, not the entire action execution.
     * </p>
     * 
     * @return The timeout in seconds for the job. An action execution can have multiple jobs. This is the timeout for a
     *         single job, not the entire action execution.
     */
    public final Integer jobTimeout() {
        return jobTimeout;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(configuration());
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(policyStatementsTemplate());
        hashCode = 31 * hashCode + Objects.hashCode(jobTimeout());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ActionTypeExecutor)) {
            return false;
        }
        ActionTypeExecutor other = (ActionTypeExecutor) obj;
        return Objects.equals(configuration(), other.configuration()) && Objects.equals(typeAsString(), other.typeAsString())
                && Objects.equals(policyStatementsTemplate(), other.policyStatementsTemplate())
                && Objects.equals(jobTimeout(), other.jobTimeout());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ActionTypeExecutor").add("Configuration", configuration()).add("Type", typeAsString())
                .add("PolicyStatementsTemplate", policyStatementsTemplate()).add("JobTimeout", jobTimeout()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "configuration":
            return Optional.ofNullable(clazz.cast(configuration()));
        case "type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "policyStatementsTemplate":
            return Optional.ofNullable(clazz.cast(policyStatementsTemplate()));
        case "jobTimeout":
            return Optional.ofNullable(clazz.cast(jobTimeout()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ActionTypeExecutor, T> g) {
        return obj -> g.apply((ActionTypeExecutor) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ActionTypeExecutor> {
        /**
         * <p>
         * The action configuration properties for the action type. These properties are specified in the action
         * definition when the action type is created.
         * </p>
         * 
         * @param configuration
         *        The action configuration properties for the action type. These properties are specified in the action
         *        definition when the action type is created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configuration(ExecutorConfiguration configuration);

        /**
         * <p>
         * The action configuration properties for the action type. These properties are specified in the action
         * definition when the action type is created.
         * </p>
         * This is a convenience method that creates an instance of the {@link ExecutorConfiguration.Builder} avoiding
         * the need to create one manually via {@link ExecutorConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ExecutorConfiguration.Builder#build()} is called immediately and
         * its result is passed to {@link #configuration(ExecutorConfiguration)}.
         * 
         * @param configuration
         *        a consumer that will call methods on {@link ExecutorConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #configuration(ExecutorConfiguration)
         */
        default Builder configuration(Consumer<ExecutorConfiguration.Builder> configuration) {
            return configuration(ExecutorConfiguration.builder().applyMutation(configuration).build());
        }

        /**
         * <p>
         * The integration model used to create and update the action type, <code>Lambda</code> or
         * <code>JobWorker</code>.
         * </p>
         * 
         * @param type
         *        The integration model used to create and update the action type, <code>Lambda</code> or
         *        <code>JobWorker</code>.
         * @see ExecutorType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ExecutorType
         */
        Builder type(String type);

        /**
         * <p>
         * The integration model used to create and update the action type, <code>Lambda</code> or
         * <code>JobWorker</code>.
         * </p>
         * 
         * @param type
         *        The integration model used to create and update the action type, <code>Lambda</code> or
         *        <code>JobWorker</code>.
         * @see ExecutorType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ExecutorType
         */
        Builder type(ExecutorType type);

        /**
         * <p>
         * The policy statement that specifies the permissions in the CodePipeline customer’s account that are needed to
         * successfully run an action.
         * </p>
         * <p>
         * To grant permission to another account, specify the account ID as the Principal, a domain-style identifier
         * defined by the service, for example <code>codepipeline.amazonaws.com</code>.
         * </p>
         * <note>
         * <p>
         * The size of the passed JSON policy document cannot exceed 2048 characters.
         * </p>
         * </note>
         * 
         * @param policyStatementsTemplate
         *        The policy statement that specifies the permissions in the CodePipeline customer’s account that are
         *        needed to successfully run an action.</p>
         *        <p>
         *        To grant permission to another account, specify the account ID as the Principal, a domain-style
         *        identifier defined by the service, for example <code>codepipeline.amazonaws.com</code>.
         *        </p>
         *        <note>
         *        <p>
         *        The size of the passed JSON policy document cannot exceed 2048 characters.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder policyStatementsTemplate(String policyStatementsTemplate);

        /**
         * <p>
         * The timeout in seconds for the job. An action execution can have multiple jobs. This is the timeout for a
         * single job, not the entire action execution.
         * </p>
         * 
         * @param jobTimeout
         *        The timeout in seconds for the job. An action execution can have multiple jobs. This is the timeout
         *        for a single job, not the entire action execution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jobTimeout(Integer jobTimeout);
    }

    static final class BuilderImpl implements Builder {
        private ExecutorConfiguration configuration;

        private String type;

        private String policyStatementsTemplate;

        private Integer jobTimeout;

        private BuilderImpl() {
        }

        private BuilderImpl(ActionTypeExecutor model) {
            configuration(model.configuration);
            type(model.type);
            policyStatementsTemplate(model.policyStatementsTemplate);
            jobTimeout(model.jobTimeout);
        }

        public final ExecutorConfiguration.Builder getConfiguration() {
            return configuration != null ? configuration.toBuilder() : null;
        }

        public final void setConfiguration(ExecutorConfiguration.BuilderImpl configuration) {
            this.configuration = configuration != null ? configuration.build() : null;
        }

        @Override
        public final Builder configuration(ExecutorConfiguration configuration) {
            this.configuration = configuration;
            return this;
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(ExecutorType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final String getPolicyStatementsTemplate() {
            return policyStatementsTemplate;
        }

        public final void setPolicyStatementsTemplate(String policyStatementsTemplate) {
            this.policyStatementsTemplate = policyStatementsTemplate;
        }

        @Override
        public final Builder policyStatementsTemplate(String policyStatementsTemplate) {
            this.policyStatementsTemplate = policyStatementsTemplate;
            return this;
        }

        public final Integer getJobTimeout() {
            return jobTimeout;
        }

        public final void setJobTimeout(Integer jobTimeout) {
            this.jobTimeout = jobTimeout;
        }

        @Override
        public final Builder jobTimeout(Integer jobTimeout) {
            this.jobTimeout = jobTimeout;
            return this;
        }

        @Override
        public ActionTypeExecutor build() {
            return new ActionTypeExecutor(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
