/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.computeoptimizer.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a filter that returns a more specific list of recommendations. Use this filter with the
 * <a>GetAutoScalingGroupRecommendations</a> and <a>GetEC2InstanceRecommendations</a> actions.
 * </p>
 * <p>
 * You can use <code>EBSFilter</code> with the <a>GetEBSVolumeRecommendations</a> action,
 * <code>LambdaFunctionRecommendationFilter</code> with the <a>GetLambdaFunctionRecommendations</a> action, and
 * <code>JobFilter</code> with the <a>DescribeRecommendationExportJobs</a> action.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Filter implements SdkPojo, Serializable, ToCopyableBuilder<Filter.Builder, Filter> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(Filter::nameAsString)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<List<String>> VALUES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("values")
            .getter(getter(Filter::values))
            .setter(setter(Builder::values))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("values").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, VALUES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final List<String> values;

    private Filter(BuilderImpl builder) {
        this.name = builder.name;
        this.values = builder.values;
    }

    /**
     * <p>
     * The name of the filter.
     * </p>
     * <p>
     * Specify <code>Finding</code> to return recommendations with a specific finding classification (for example,
     * <code>Underprovisioned</code>).
     * </p>
     * <p>
     * Specify <code>RecommendationSourceType</code> to return recommendations of a specific resource type (for example,
     * <code>Ec2Instance</code>).
     * </p>
     * <p>
     * Specify <code>FindingReasonCodes</code> to return recommendations with a specific finding reason code (for
     * example, <code>CPUUnderprovisioned</code>).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #name} will return
     * {@link FilterName#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #nameAsString}.
     * </p>
     * 
     * @return The name of the filter.</p>
     *         <p>
     *         Specify <code>Finding</code> to return recommendations with a specific finding classification (for
     *         example, <code>Underprovisioned</code>).
     *         </p>
     *         <p>
     *         Specify <code>RecommendationSourceType</code> to return recommendations of a specific resource type (for
     *         example, <code>Ec2Instance</code>).
     *         </p>
     *         <p>
     *         Specify <code>FindingReasonCodes</code> to return recommendations with a specific finding reason code
     *         (for example, <code>CPUUnderprovisioned</code>).
     * @see FilterName
     */
    public final FilterName name() {
        return FilterName.fromValue(name);
    }

    /**
     * <p>
     * The name of the filter.
     * </p>
     * <p>
     * Specify <code>Finding</code> to return recommendations with a specific finding classification (for example,
     * <code>Underprovisioned</code>).
     * </p>
     * <p>
     * Specify <code>RecommendationSourceType</code> to return recommendations of a specific resource type (for example,
     * <code>Ec2Instance</code>).
     * </p>
     * <p>
     * Specify <code>FindingReasonCodes</code> to return recommendations with a specific finding reason code (for
     * example, <code>CPUUnderprovisioned</code>).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #name} will return
     * {@link FilterName#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #nameAsString}.
     * </p>
     * 
     * @return The name of the filter.</p>
     *         <p>
     *         Specify <code>Finding</code> to return recommendations with a specific finding classification (for
     *         example, <code>Underprovisioned</code>).
     *         </p>
     *         <p>
     *         Specify <code>RecommendationSourceType</code> to return recommendations of a specific resource type (for
     *         example, <code>Ec2Instance</code>).
     *         </p>
     *         <p>
     *         Specify <code>FindingReasonCodes</code> to return recommendations with a specific finding reason code
     *         (for example, <code>CPUUnderprovisioned</code>).
     * @see FilterName
     */
    public final String nameAsString() {
        return name;
    }

    /**
     * For responses, this returns true if the service returned a value for the Values property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasValues() {
        return values != null && !(values instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The value of the filter.
     * </p>
     * <p>
     * The valid values for this parameter are as follows, depending on what you specify for the <code>name</code>
     * parameter and the resource type that you wish to filter results for:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Specify <code>Optimized</code> or <code>NotOptimized</code> if you specify the <code>name</code> parameter as
     * <code>Finding</code> and you want to filter results for Auto Scaling groups.
     * </p>
     * </li>
     * <li>
     * <p>
     * Specify <code>Underprovisioned</code>, <code>Overprovisioned</code>, or <code>Optimized</code> if you specify the
     * <code>name</code> parameter as <code>Finding</code> and you want to filter results for EC2 instances.
     * </p>
     * </li>
     * <li>
     * <p>
     * Specify <code>Ec2Instance</code> or <code>AutoScalingGroup</code> if you specify the <code>name</code> parameter
     * as <code>RecommendationSourceType</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Specify one of the following options if you specify the <code>name</code> parameter as
     * <code>FindingReasonCodes</code>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b> <code>CPUOverprovisioned</code> </b> — The instance’s CPU configuration can be sized down while still meeting
     * the performance requirements of your workload.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>CPUUnderprovisioned</code> </b> — The instance’s CPU configuration doesn't meet the performance
     * requirements of your workload and there is an alternative instance type that provides better CPU performance.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>MemoryOverprovisioned</code> </b> — The instance’s memory configuration can be sized down while still
     * meeting the performance requirements of your workload.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>MemoryUnderprovisioned</code> </b> — The instance’s memory configuration doesn't meet the performance
     * requirements of your workload and there is an alternative instance type that provides better memory performance.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>EBSThroughputOverprovisioned</code> </b> — The instance’s EBS throughput configuration can be sized
     * down while still meeting the performance requirements of your workload.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>EBSThroughputUnderprovisioned</code> </b> — The instance’s EBS throughput configuration doesn't meet
     * the performance requirements of your workload and there is an alternative instance type that provides better EBS
     * throughput performance.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>EBSIOPSOverprovisioned</code> </b> — The instance’s EBS IOPS configuration can be sized down while
     * still meeting the performance requirements of your workload.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>EBSIOPSUnderprovisioned</code> </b> — The instance’s EBS IOPS configuration doesn't meet the
     * performance requirements of your workload and there is an alternative instance type that provides better EBS IOPS
     * performance.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>NetworkBandwidthOverprovisioned</code> </b> — The instance’s network bandwidth configuration can be
     * sized down while still meeting the performance requirements of your workload.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>NetworkBandwidthUnderprovisioned</code> </b> — The instance’s network bandwidth configuration doesn't
     * meet the performance requirements of your workload and there is an alternative instance type that provides better
     * network bandwidth performance. This finding reason happens when the <code>NetworkIn</code> or
     * <code>NetworkOut</code> performance of an instance is impacted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>NetworkPPSOverprovisioned</code> </b> — The instance’s network PPS (packets per second) configuration
     * can be sized down while still meeting the performance requirements of your workload.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>NetworkPPSUnderprovisioned</code> </b> — The instance’s network PPS (packets per second) configuration
     * doesn't meet the performance requirements of your workload and there is an alternative instance type that
     * provides better network PPS performance.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>DiskIOPSOverprovisioned</code> </b> — The instance’s disk IOPS configuration can be sized down while
     * still meeting the performance requirements of your workload.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>DiskIOPSUnderprovisioned</code> </b> — The instance’s disk IOPS configuration doesn't meet the
     * performance requirements of your workload and there is an alternative instance type that provides better disk
     * IOPS performance.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>DiskThroughputOverprovisioned</code> </b> — The instance’s disk throughput configuration can be sized
     * down while still meeting the performance requirements of your workload.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>DiskThroughputUnderprovisioned</code> </b> — The instance’s disk throughput configuration doesn't meet
     * the performance requirements of your workload and there is an alternative instance type that provides better disk
     * throughput performance.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasValues} method.
     * </p>
     * 
     * @return The value of the filter.</p>
     *         <p>
     *         The valid values for this parameter are as follows, depending on what you specify for the
     *         <code>name</code> parameter and the resource type that you wish to filter results for:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Specify <code>Optimized</code> or <code>NotOptimized</code> if you specify the <code>name</code>
     *         parameter as <code>Finding</code> and you want to filter results for Auto Scaling groups.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Specify <code>Underprovisioned</code>, <code>Overprovisioned</code>, or <code>Optimized</code> if you
     *         specify the <code>name</code> parameter as <code>Finding</code> and you want to filter results for EC2
     *         instances.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Specify <code>Ec2Instance</code> or <code>AutoScalingGroup</code> if you specify the <code>name</code>
     *         parameter as <code>RecommendationSourceType</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Specify one of the following options if you specify the <code>name</code> parameter as
     *         <code>FindingReasonCodes</code>:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b> <code>CPUOverprovisioned</code> </b> — The instance’s CPU configuration can be sized down while still
     *         meeting the performance requirements of your workload.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>CPUUnderprovisioned</code> </b> — The instance’s CPU configuration doesn't meet the performance
     *         requirements of your workload and there is an alternative instance type that provides better CPU
     *         performance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>MemoryOverprovisioned</code> </b> — The instance’s memory configuration can be sized down while
     *         still meeting the performance requirements of your workload.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>MemoryUnderprovisioned</code> </b> — The instance’s memory configuration doesn't meet the
     *         performance requirements of your workload and there is an alternative instance type that provides better
     *         memory performance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>EBSThroughputOverprovisioned</code> </b> — The instance’s EBS throughput configuration can be
     *         sized down while still meeting the performance requirements of your workload.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>EBSThroughputUnderprovisioned</code> </b> — The instance’s EBS throughput configuration doesn't
     *         meet the performance requirements of your workload and there is an alternative instance type that
     *         provides better EBS throughput performance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>EBSIOPSOverprovisioned</code> </b> — The instance’s EBS IOPS configuration can be sized down
     *         while still meeting the performance requirements of your workload.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>EBSIOPSUnderprovisioned</code> </b> — The instance’s EBS IOPS configuration doesn't meet the
     *         performance requirements of your workload and there is an alternative instance type that provides better
     *         EBS IOPS performance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>NetworkBandwidthOverprovisioned</code> </b> — The instance’s network bandwidth configuration
     *         can be sized down while still meeting the performance requirements of your workload.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>NetworkBandwidthUnderprovisioned</code> </b> — The instance’s network bandwidth configuration
     *         doesn't meet the performance requirements of your workload and there is an alternative instance type that
     *         provides better network bandwidth performance. This finding reason happens when the
     *         <code>NetworkIn</code> or <code>NetworkOut</code> performance of an instance is impacted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>NetworkPPSOverprovisioned</code> </b> — The instance’s network PPS (packets per second)
     *         configuration can be sized down while still meeting the performance requirements of your workload.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>NetworkPPSUnderprovisioned</code> </b> — The instance’s network PPS (packets per second)
     *         configuration doesn't meet the performance requirements of your workload and there is an alternative
     *         instance type that provides better network PPS performance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>DiskIOPSOverprovisioned</code> </b> — The instance’s disk IOPS configuration can be sized down
     *         while still meeting the performance requirements of your workload.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>DiskIOPSUnderprovisioned</code> </b> — The instance’s disk IOPS configuration doesn't meet the
     *         performance requirements of your workload and there is an alternative instance type that provides better
     *         disk IOPS performance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>DiskThroughputOverprovisioned</code> </b> — The instance’s disk throughput configuration can be
     *         sized down while still meeting the performance requirements of your workload.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>DiskThroughputUnderprovisioned</code> </b> — The instance’s disk throughput configuration
     *         doesn't meet the performance requirements of your workload and there is an alternative instance type that
     *         provides better disk throughput performance.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     */
    public final List<String> values() {
        return values;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(nameAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasValues() ? values() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Filter)) {
            return false;
        }
        Filter other = (Filter) obj;
        return Objects.equals(nameAsString(), other.nameAsString()) && hasValues() == other.hasValues()
                && Objects.equals(values(), other.values());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Filter").add("Name", nameAsString()).add("Values", hasValues() ? values() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "name":
            return Optional.ofNullable(clazz.cast(nameAsString()));
        case "values":
            return Optional.ofNullable(clazz.cast(values()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Filter, T> g) {
        return obj -> g.apply((Filter) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Filter> {
        /**
         * <p>
         * The name of the filter.
         * </p>
         * <p>
         * Specify <code>Finding</code> to return recommendations with a specific finding classification (for example,
         * <code>Underprovisioned</code>).
         * </p>
         * <p>
         * Specify <code>RecommendationSourceType</code> to return recommendations of a specific resource type (for
         * example, <code>Ec2Instance</code>).
         * </p>
         * <p>
         * Specify <code>FindingReasonCodes</code> to return recommendations with a specific finding reason code (for
         * example, <code>CPUUnderprovisioned</code>).
         * </p>
         * 
         * @param name
         *        The name of the filter.</p>
         *        <p>
         *        Specify <code>Finding</code> to return recommendations with a specific finding classification (for
         *        example, <code>Underprovisioned</code>).
         *        </p>
         *        <p>
         *        Specify <code>RecommendationSourceType</code> to return recommendations of a specific resource type
         *        (for example, <code>Ec2Instance</code>).
         *        </p>
         *        <p>
         *        Specify <code>FindingReasonCodes</code> to return recommendations with a specific finding reason code
         *        (for example, <code>CPUUnderprovisioned</code>).
         * @see FilterName
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FilterName
         */
        Builder name(String name);

        /**
         * <p>
         * The name of the filter.
         * </p>
         * <p>
         * Specify <code>Finding</code> to return recommendations with a specific finding classification (for example,
         * <code>Underprovisioned</code>).
         * </p>
         * <p>
         * Specify <code>RecommendationSourceType</code> to return recommendations of a specific resource type (for
         * example, <code>Ec2Instance</code>).
         * </p>
         * <p>
         * Specify <code>FindingReasonCodes</code> to return recommendations with a specific finding reason code (for
         * example, <code>CPUUnderprovisioned</code>).
         * </p>
         * 
         * @param name
         *        The name of the filter.</p>
         *        <p>
         *        Specify <code>Finding</code> to return recommendations with a specific finding classification (for
         *        example, <code>Underprovisioned</code>).
         *        </p>
         *        <p>
         *        Specify <code>RecommendationSourceType</code> to return recommendations of a specific resource type
         *        (for example, <code>Ec2Instance</code>).
         *        </p>
         *        <p>
         *        Specify <code>FindingReasonCodes</code> to return recommendations with a specific finding reason code
         *        (for example, <code>CPUUnderprovisioned</code>).
         * @see FilterName
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FilterName
         */
        Builder name(FilterName name);

        /**
         * <p>
         * The value of the filter.
         * </p>
         * <p>
         * The valid values for this parameter are as follows, depending on what you specify for the <code>name</code>
         * parameter and the resource type that you wish to filter results for:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Specify <code>Optimized</code> or <code>NotOptimized</code> if you specify the <code>name</code> parameter as
         * <code>Finding</code> and you want to filter results for Auto Scaling groups.
         * </p>
         * </li>
         * <li>
         * <p>
         * Specify <code>Underprovisioned</code>, <code>Overprovisioned</code>, or <code>Optimized</code> if you specify
         * the <code>name</code> parameter as <code>Finding</code> and you want to filter results for EC2 instances.
         * </p>
         * </li>
         * <li>
         * <p>
         * Specify <code>Ec2Instance</code> or <code>AutoScalingGroup</code> if you specify the <code>name</code>
         * parameter as <code>RecommendationSourceType</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Specify one of the following options if you specify the <code>name</code> parameter as
         * <code>FindingReasonCodes</code>:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b> <code>CPUOverprovisioned</code> </b> — The instance’s CPU configuration can be sized down while still
         * meeting the performance requirements of your workload.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>CPUUnderprovisioned</code> </b> — The instance’s CPU configuration doesn't meet the performance
         * requirements of your workload and there is an alternative instance type that provides better CPU performance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>MemoryOverprovisioned</code> </b> — The instance’s memory configuration can be sized down while
         * still meeting the performance requirements of your workload.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>MemoryUnderprovisioned</code> </b> — The instance’s memory configuration doesn't meet the
         * performance requirements of your workload and there is an alternative instance type that provides better
         * memory performance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>EBSThroughputOverprovisioned</code> </b> — The instance’s EBS throughput configuration can be sized
         * down while still meeting the performance requirements of your workload.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>EBSThroughputUnderprovisioned</code> </b> — The instance’s EBS throughput configuration doesn't
         * meet the performance requirements of your workload and there is an alternative instance type that provides
         * better EBS throughput performance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>EBSIOPSOverprovisioned</code> </b> — The instance’s EBS IOPS configuration can be sized down while
         * still meeting the performance requirements of your workload.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>EBSIOPSUnderprovisioned</code> </b> — The instance’s EBS IOPS configuration doesn't meet the
         * performance requirements of your workload and there is an alternative instance type that provides better EBS
         * IOPS performance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>NetworkBandwidthOverprovisioned</code> </b> — The instance’s network bandwidth configuration can be
         * sized down while still meeting the performance requirements of your workload.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>NetworkBandwidthUnderprovisioned</code> </b> — The instance’s network bandwidth configuration
         * doesn't meet the performance requirements of your workload and there is an alternative instance type that
         * provides better network bandwidth performance. This finding reason happens when the <code>NetworkIn</code> or
         * <code>NetworkOut</code> performance of an instance is impacted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>NetworkPPSOverprovisioned</code> </b> — The instance’s network PPS (packets per second)
         * configuration can be sized down while still meeting the performance requirements of your workload.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>NetworkPPSUnderprovisioned</code> </b> — The instance’s network PPS (packets per second)
         * configuration doesn't meet the performance requirements of your workload and there is an alternative instance
         * type that provides better network PPS performance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>DiskIOPSOverprovisioned</code> </b> — The instance’s disk IOPS configuration can be sized down
         * while still meeting the performance requirements of your workload.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>DiskIOPSUnderprovisioned</code> </b> — The instance’s disk IOPS configuration doesn't meet the
         * performance requirements of your workload and there is an alternative instance type that provides better disk
         * IOPS performance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>DiskThroughputOverprovisioned</code> </b> — The instance’s disk throughput configuration can be
         * sized down while still meeting the performance requirements of your workload.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>DiskThroughputUnderprovisioned</code> </b> — The instance’s disk throughput configuration doesn't
         * meet the performance requirements of your workload and there is an alternative instance type that provides
         * better disk throughput performance.
         * </p>
         * </li>
         * </ul>
         * </li>
         * </ul>
         * 
         * @param values
         *        The value of the filter.</p>
         *        <p>
         *        The valid values for this parameter are as follows, depending on what you specify for the
         *        <code>name</code> parameter and the resource type that you wish to filter results for:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Specify <code>Optimized</code> or <code>NotOptimized</code> if you specify the <code>name</code>
         *        parameter as <code>Finding</code> and you want to filter results for Auto Scaling groups.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Specify <code>Underprovisioned</code>, <code>Overprovisioned</code>, or <code>Optimized</code> if you
         *        specify the <code>name</code> parameter as <code>Finding</code> and you want to filter results for EC2
         *        instances.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Specify <code>Ec2Instance</code> or <code>AutoScalingGroup</code> if you specify the <code>name</code>
         *        parameter as <code>RecommendationSourceType</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Specify one of the following options if you specify the <code>name</code> parameter as
         *        <code>FindingReasonCodes</code>:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b> <code>CPUOverprovisioned</code> </b> — The instance’s CPU configuration can be sized down while
         *        still meeting the performance requirements of your workload.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>CPUUnderprovisioned</code> </b> — The instance’s CPU configuration doesn't meet the
         *        performance requirements of your workload and there is an alternative instance type that provides
         *        better CPU performance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>MemoryOverprovisioned</code> </b> — The instance’s memory configuration can be sized down
         *        while still meeting the performance requirements of your workload.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>MemoryUnderprovisioned</code> </b> — The instance’s memory configuration doesn't meet the
         *        performance requirements of your workload and there is an alternative instance type that provides
         *        better memory performance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>EBSThroughputOverprovisioned</code> </b> — The instance’s EBS throughput configuration can
         *        be sized down while still meeting the performance requirements of your workload.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>EBSThroughputUnderprovisioned</code> </b> — The instance’s EBS throughput configuration
         *        doesn't meet the performance requirements of your workload and there is an alternative instance type
         *        that provides better EBS throughput performance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>EBSIOPSOverprovisioned</code> </b> — The instance’s EBS IOPS configuration can be sized down
         *        while still meeting the performance requirements of your workload.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>EBSIOPSUnderprovisioned</code> </b> — The instance’s EBS IOPS configuration doesn't meet the
         *        performance requirements of your workload and there is an alternative instance type that provides
         *        better EBS IOPS performance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>NetworkBandwidthOverprovisioned</code> </b> — The instance’s network bandwidth configuration
         *        can be sized down while still meeting the performance requirements of your workload.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>NetworkBandwidthUnderprovisioned</code> </b> — The instance’s network bandwidth
         *        configuration doesn't meet the performance requirements of your workload and there is an alternative
         *        instance type that provides better network bandwidth performance. This finding reason happens when the
         *        <code>NetworkIn</code> or <code>NetworkOut</code> performance of an instance is impacted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>NetworkPPSOverprovisioned</code> </b> — The instance’s network PPS (packets per second)
         *        configuration can be sized down while still meeting the performance requirements of your workload.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>NetworkPPSUnderprovisioned</code> </b> — The instance’s network PPS (packets per second)
         *        configuration doesn't meet the performance requirements of your workload and there is an alternative
         *        instance type that provides better network PPS performance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>DiskIOPSOverprovisioned</code> </b> — The instance’s disk IOPS configuration can be sized
         *        down while still meeting the performance requirements of your workload.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>DiskIOPSUnderprovisioned</code> </b> — The instance’s disk IOPS configuration doesn't meet
         *        the performance requirements of your workload and there is an alternative instance type that provides
         *        better disk IOPS performance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>DiskThroughputOverprovisioned</code> </b> — The instance’s disk throughput configuration can
         *        be sized down while still meeting the performance requirements of your workload.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>DiskThroughputUnderprovisioned</code> </b> — The instance’s disk throughput configuration
         *        doesn't meet the performance requirements of your workload and there is an alternative instance type
         *        that provides better disk throughput performance.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder values(Collection<String> values);

        /**
         * <p>
         * The value of the filter.
         * </p>
         * <p>
         * The valid values for this parameter are as follows, depending on what you specify for the <code>name</code>
         * parameter and the resource type that you wish to filter results for:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Specify <code>Optimized</code> or <code>NotOptimized</code> if you specify the <code>name</code> parameter as
         * <code>Finding</code> and you want to filter results for Auto Scaling groups.
         * </p>
         * </li>
         * <li>
         * <p>
         * Specify <code>Underprovisioned</code>, <code>Overprovisioned</code>, or <code>Optimized</code> if you specify
         * the <code>name</code> parameter as <code>Finding</code> and you want to filter results for EC2 instances.
         * </p>
         * </li>
         * <li>
         * <p>
         * Specify <code>Ec2Instance</code> or <code>AutoScalingGroup</code> if you specify the <code>name</code>
         * parameter as <code>RecommendationSourceType</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Specify one of the following options if you specify the <code>name</code> parameter as
         * <code>FindingReasonCodes</code>:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b> <code>CPUOverprovisioned</code> </b> — The instance’s CPU configuration can be sized down while still
         * meeting the performance requirements of your workload.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>CPUUnderprovisioned</code> </b> — The instance’s CPU configuration doesn't meet the performance
         * requirements of your workload and there is an alternative instance type that provides better CPU performance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>MemoryOverprovisioned</code> </b> — The instance’s memory configuration can be sized down while
         * still meeting the performance requirements of your workload.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>MemoryUnderprovisioned</code> </b> — The instance’s memory configuration doesn't meet the
         * performance requirements of your workload and there is an alternative instance type that provides better
         * memory performance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>EBSThroughputOverprovisioned</code> </b> — The instance’s EBS throughput configuration can be sized
         * down while still meeting the performance requirements of your workload.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>EBSThroughputUnderprovisioned</code> </b> — The instance’s EBS throughput configuration doesn't
         * meet the performance requirements of your workload and there is an alternative instance type that provides
         * better EBS throughput performance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>EBSIOPSOverprovisioned</code> </b> — The instance’s EBS IOPS configuration can be sized down while
         * still meeting the performance requirements of your workload.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>EBSIOPSUnderprovisioned</code> </b> — The instance’s EBS IOPS configuration doesn't meet the
         * performance requirements of your workload and there is an alternative instance type that provides better EBS
         * IOPS performance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>NetworkBandwidthOverprovisioned</code> </b> — The instance’s network bandwidth configuration can be
         * sized down while still meeting the performance requirements of your workload.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>NetworkBandwidthUnderprovisioned</code> </b> — The instance’s network bandwidth configuration
         * doesn't meet the performance requirements of your workload and there is an alternative instance type that
         * provides better network bandwidth performance. This finding reason happens when the <code>NetworkIn</code> or
         * <code>NetworkOut</code> performance of an instance is impacted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>NetworkPPSOverprovisioned</code> </b> — The instance’s network PPS (packets per second)
         * configuration can be sized down while still meeting the performance requirements of your workload.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>NetworkPPSUnderprovisioned</code> </b> — The instance’s network PPS (packets per second)
         * configuration doesn't meet the performance requirements of your workload and there is an alternative instance
         * type that provides better network PPS performance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>DiskIOPSOverprovisioned</code> </b> — The instance’s disk IOPS configuration can be sized down
         * while still meeting the performance requirements of your workload.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>DiskIOPSUnderprovisioned</code> </b> — The instance’s disk IOPS configuration doesn't meet the
         * performance requirements of your workload and there is an alternative instance type that provides better disk
         * IOPS performance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>DiskThroughputOverprovisioned</code> </b> — The instance’s disk throughput configuration can be
         * sized down while still meeting the performance requirements of your workload.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>DiskThroughputUnderprovisioned</code> </b> — The instance’s disk throughput configuration doesn't
         * meet the performance requirements of your workload and there is an alternative instance type that provides
         * better disk throughput performance.
         * </p>
         * </li>
         * </ul>
         * </li>
         * </ul>
         * 
         * @param values
         *        The value of the filter.</p>
         *        <p>
         *        The valid values for this parameter are as follows, depending on what you specify for the
         *        <code>name</code> parameter and the resource type that you wish to filter results for:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Specify <code>Optimized</code> or <code>NotOptimized</code> if you specify the <code>name</code>
         *        parameter as <code>Finding</code> and you want to filter results for Auto Scaling groups.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Specify <code>Underprovisioned</code>, <code>Overprovisioned</code>, or <code>Optimized</code> if you
         *        specify the <code>name</code> parameter as <code>Finding</code> and you want to filter results for EC2
         *        instances.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Specify <code>Ec2Instance</code> or <code>AutoScalingGroup</code> if you specify the <code>name</code>
         *        parameter as <code>RecommendationSourceType</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Specify one of the following options if you specify the <code>name</code> parameter as
         *        <code>FindingReasonCodes</code>:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b> <code>CPUOverprovisioned</code> </b> — The instance’s CPU configuration can be sized down while
         *        still meeting the performance requirements of your workload.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>CPUUnderprovisioned</code> </b> — The instance’s CPU configuration doesn't meet the
         *        performance requirements of your workload and there is an alternative instance type that provides
         *        better CPU performance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>MemoryOverprovisioned</code> </b> — The instance’s memory configuration can be sized down
         *        while still meeting the performance requirements of your workload.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>MemoryUnderprovisioned</code> </b> — The instance’s memory configuration doesn't meet the
         *        performance requirements of your workload and there is an alternative instance type that provides
         *        better memory performance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>EBSThroughputOverprovisioned</code> </b> — The instance’s EBS throughput configuration can
         *        be sized down while still meeting the performance requirements of your workload.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>EBSThroughputUnderprovisioned</code> </b> — The instance’s EBS throughput configuration
         *        doesn't meet the performance requirements of your workload and there is an alternative instance type
         *        that provides better EBS throughput performance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>EBSIOPSOverprovisioned</code> </b> — The instance’s EBS IOPS configuration can be sized down
         *        while still meeting the performance requirements of your workload.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>EBSIOPSUnderprovisioned</code> </b> — The instance’s EBS IOPS configuration doesn't meet the
         *        performance requirements of your workload and there is an alternative instance type that provides
         *        better EBS IOPS performance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>NetworkBandwidthOverprovisioned</code> </b> — The instance’s network bandwidth configuration
         *        can be sized down while still meeting the performance requirements of your workload.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>NetworkBandwidthUnderprovisioned</code> </b> — The instance’s network bandwidth
         *        configuration doesn't meet the performance requirements of your workload and there is an alternative
         *        instance type that provides better network bandwidth performance. This finding reason happens when the
         *        <code>NetworkIn</code> or <code>NetworkOut</code> performance of an instance is impacted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>NetworkPPSOverprovisioned</code> </b> — The instance’s network PPS (packets per second)
         *        configuration can be sized down while still meeting the performance requirements of your workload.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>NetworkPPSUnderprovisioned</code> </b> — The instance’s network PPS (packets per second)
         *        configuration doesn't meet the performance requirements of your workload and there is an alternative
         *        instance type that provides better network PPS performance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>DiskIOPSOverprovisioned</code> </b> — The instance’s disk IOPS configuration can be sized
         *        down while still meeting the performance requirements of your workload.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>DiskIOPSUnderprovisioned</code> </b> — The instance’s disk IOPS configuration doesn't meet
         *        the performance requirements of your workload and there is an alternative instance type that provides
         *        better disk IOPS performance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>DiskThroughputOverprovisioned</code> </b> — The instance’s disk throughput configuration can
         *        be sized down while still meeting the performance requirements of your workload.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>DiskThroughputUnderprovisioned</code> </b> — The instance’s disk throughput configuration
         *        doesn't meet the performance requirements of your workload and there is an alternative instance type
         *        that provides better disk throughput performance.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder values(String... values);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private List<String> values = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Filter model) {
            name(model.name);
            values(model.values);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        @Override
        public final Builder name(FilterName name) {
            this.name(name == null ? null : name.toString());
            return this;
        }

        public final Collection<String> getValues() {
            if (values instanceof SdkAutoConstructList) {
                return null;
            }
            return values;
        }

        public final void setValues(Collection<String> values) {
            this.values = FilterValuesCopier.copy(values);
        }

        @Override
        public final Builder values(Collection<String> values) {
            this.values = FilterValuesCopier.copy(values);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder values(String... values) {
            values(Arrays.asList(values));
            return this;
        }

        @Override
        public Filter build() {
            return new Filter(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
