/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.computeoptimizer.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a recommendation option for an Amazon EC2 instance.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class InstanceRecommendationOption implements SdkPojo, Serializable,
        ToCopyableBuilder<InstanceRecommendationOption.Builder, InstanceRecommendationOption> {
    private static final SdkField<String> INSTANCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("instanceType").getter(getter(InstanceRecommendationOption::instanceType))
            .setter(setter(Builder::instanceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("instanceType").build()).build();

    private static final SdkField<List<UtilizationMetric>> PROJECTED_UTILIZATION_METRICS_FIELD = SdkField
            .<List<UtilizationMetric>> builder(MarshallingType.LIST)
            .memberName("projectedUtilizationMetrics")
            .getter(getter(InstanceRecommendationOption::projectedUtilizationMetrics))
            .setter(setter(Builder::projectedUtilizationMetrics))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("projectedUtilizationMetrics")
                    .build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<UtilizationMetric> builder(MarshallingType.SDK_POJO)
                                            .constructor(UtilizationMetric::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> PLATFORM_DIFFERENCES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("platformDifferences")
            .getter(getter(InstanceRecommendationOption::platformDifferencesAsStrings))
            .setter(setter(Builder::platformDifferencesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("platformDifferences").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Double> PERFORMANCE_RISK_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .memberName("performanceRisk").getter(getter(InstanceRecommendationOption::performanceRisk))
            .setter(setter(Builder::performanceRisk))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("performanceRisk").build()).build();

    private static final SdkField<Integer> RANK_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER).memberName("rank")
            .getter(getter(InstanceRecommendationOption::rank)).setter(setter(Builder::rank))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("rank").build()).build();

    private static final SdkField<SavingsOpportunity> SAVINGS_OPPORTUNITY_FIELD = SdkField
            .<SavingsOpportunity> builder(MarshallingType.SDK_POJO).memberName("savingsOpportunity")
            .getter(getter(InstanceRecommendationOption::savingsOpportunity)).setter(setter(Builder::savingsOpportunity))
            .constructor(SavingsOpportunity::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("savingsOpportunity").build())
            .build();

    private static final SdkField<String> MIGRATION_EFFORT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("migrationEffort").getter(getter(InstanceRecommendationOption::migrationEffortAsString))
            .setter(setter(Builder::migrationEffort))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("migrationEffort").build()).build();

    private static final SdkField<GpuInfo> INSTANCE_GPU_INFO_FIELD = SdkField.<GpuInfo> builder(MarshallingType.SDK_POJO)
            .memberName("instanceGpuInfo").getter(getter(InstanceRecommendationOption::instanceGpuInfo))
            .setter(setter(Builder::instanceGpuInfo)).constructor(GpuInfo::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("instanceGpuInfo").build()).build();

    private static final SdkField<InstanceSavingsOpportunityAfterDiscounts> SAVINGS_OPPORTUNITY_AFTER_DISCOUNTS_FIELD = SdkField
            .<InstanceSavingsOpportunityAfterDiscounts> builder(MarshallingType.SDK_POJO)
            .memberName("savingsOpportunityAfterDiscounts")
            .getter(getter(InstanceRecommendationOption::savingsOpportunityAfterDiscounts))
            .setter(setter(Builder::savingsOpportunityAfterDiscounts))
            .constructor(InstanceSavingsOpportunityAfterDiscounts::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("savingsOpportunityAfterDiscounts")
                    .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays
            .asList(INSTANCE_TYPE_FIELD, PROJECTED_UTILIZATION_METRICS_FIELD, PLATFORM_DIFFERENCES_FIELD, PERFORMANCE_RISK_FIELD,
                    RANK_FIELD, SAVINGS_OPPORTUNITY_FIELD, MIGRATION_EFFORT_FIELD, INSTANCE_GPU_INFO_FIELD,
                    SAVINGS_OPPORTUNITY_AFTER_DISCOUNTS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String instanceType;

    private final List<UtilizationMetric> projectedUtilizationMetrics;

    private final List<String> platformDifferences;

    private final Double performanceRisk;

    private final Integer rank;

    private final SavingsOpportunity savingsOpportunity;

    private final String migrationEffort;

    private final GpuInfo instanceGpuInfo;

    private final InstanceSavingsOpportunityAfterDiscounts savingsOpportunityAfterDiscounts;

    private InstanceRecommendationOption(BuilderImpl builder) {
        this.instanceType = builder.instanceType;
        this.projectedUtilizationMetrics = builder.projectedUtilizationMetrics;
        this.platformDifferences = builder.platformDifferences;
        this.performanceRisk = builder.performanceRisk;
        this.rank = builder.rank;
        this.savingsOpportunity = builder.savingsOpportunity;
        this.migrationEffort = builder.migrationEffort;
        this.instanceGpuInfo = builder.instanceGpuInfo;
        this.savingsOpportunityAfterDiscounts = builder.savingsOpportunityAfterDiscounts;
    }

    /**
     * <p>
     * The instance type of the instance recommendation.
     * </p>
     * 
     * @return The instance type of the instance recommendation.
     */
    public final String instanceType() {
        return instanceType;
    }

    /**
     * For responses, this returns true if the service returned a value for the ProjectedUtilizationMetrics property.
     * This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasProjectedUtilizationMetrics() {
        return projectedUtilizationMetrics != null && !(projectedUtilizationMetrics instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of objects that describe the projected utilization metrics of the instance recommendation option.
     * </p>
     * <note>
     * <p>
     * The <code>Cpu</code> and <code>Memory</code> metrics are the only projected utilization metrics returned.
     * Additionally, the <code>Memory</code> metric is returned only for resources that have the unified CloudWatch
     * agent installed on them. For more information, see <a
     * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/metrics.html#cw-agent">Enabling Memory Utilization
     * with the CloudWatch Agent</a>.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasProjectedUtilizationMetrics} method.
     * </p>
     * 
     * @return An array of objects that describe the projected utilization metrics of the instance recommendation
     *         option.</p> <note>
     *         <p>
     *         The <code>Cpu</code> and <code>Memory</code> metrics are the only projected utilization metrics returned.
     *         Additionally, the <code>Memory</code> metric is returned only for resources that have the unified
     *         CloudWatch agent installed on them. For more information, see <a
     *         href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/metrics.html#cw-agent">Enabling Memory
     *         Utilization with the CloudWatch Agent</a>.
     *         </p>
     */
    public final List<UtilizationMetric> projectedUtilizationMetrics() {
        return projectedUtilizationMetrics;
    }

    /**
     * <p>
     * Describes the configuration differences between the current instance and the recommended instance type. You
     * should consider the configuration differences before migrating your workloads from the current instance to the
     * recommended instance type. The <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-instance-resize.html">Change the instance type
     * guide for Linux</a> and <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/ec2-instance-resize.html">Change the instance type
     * guide for Windows</a> provide general guidance for getting started with an instance migration.
     * </p>
     * <p>
     * Platform differences include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b> <code>Hypervisor</code> </b> — The hypervisor of the recommended instance type is different than that of the
     * current instance. For example, the recommended instance type uses a Nitro hypervisor and the current instance
     * uses a Xen hypervisor. The differences that you should consider between these hypervisors are covered in the <a
     * href="http://aws.amazon.com/ec2/faqs/#Nitro_Hypervisor">Nitro Hypervisor</a> section of the Amazon EC2 frequently
     * asked questions. For more information, see <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#ec2-nitro-instances">Instances
     * built on the Nitro System</a> in the <i>Amazon EC2 User Guide for Linux</i>, or <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/instance-types.html#ec2-nitro-instances">Instances
     * built on the Nitro System</a> in the <i>Amazon EC2 User Guide for Windows</i>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>NetworkInterface</code> </b> — The network interface of the recommended instance type is different than
     * that of the current instance. For example, the recommended instance type supports enhanced networking and the
     * current instance might not. To enable enhanced networking for the recommended instance type, you must install the
     * Elastic Network Adapter (ENA) driver or the Intel 82599 Virtual Function driver. For more information, see <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#instance-networking-storage">
     * Networking and storage features</a> and <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/enhanced-networking.html">Enhanced networking on
     * Linux</a> in the <i>Amazon EC2 User Guide for Linux</i>, or <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/instance-types.html#instance-networking-storage"
     * >Networking and storage features</a> and <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/enhanced-networking.html">Enhanced networking on
     * Windows</a> in the <i>Amazon EC2 User Guide for Windows</i>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>StorageInterface</code> </b> — The storage interface of the recommended instance type is different than
     * that of the current instance. For example, the recommended instance type uses an NVMe storage interface and the
     * current instance does not. To access NVMe volumes for the recommended instance type, you will need to install or
     * upgrade the NVMe driver. For more information, see <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#instance-networking-storage"
     * >Networking and storage features</a> and <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/nvme-ebs-volumes.html">Amazon EBS and NVMe on Linux
     * instances</a> in the <i>Amazon EC2 User Guide for Linux</i>, or <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/instance-types.html#instance-networking-storage"
     * >Networking and storage features</a> and <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/nvme-ebs-volumes.html">Amazon EBS and NVMe on
     * Windows instances</a> in the <i>Amazon EC2 User Guide for Windows</i>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>InstanceStoreAvailability</code> </b> — The recommended instance type does not support instance store
     * volumes and the current instance does. Before migrating, you might need to back up the data on your instance
     * store volumes if you want to preserve them. For more information, see <a
     * href="https://aws.amazon.com/premiumsupport/knowledge-center/back-up-instance-store-ebs/">How do I back up an
     * instance store volume on my Amazon EC2 instance to Amazon EBS?</a> in the <i>Amazon Web Services Premium Support
     * Knowledge Base</i>. For more information, see <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#instance-networking-storage"
     * >Networking and storage features</a> and <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/InstanceStorage.html">Amazon EC2 instance store</a> in
     * the <i>Amazon EC2 User Guide for Linux</i>, or see <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/instance-types.html#instance-networking-storage"
     * >Networking and storage features</a> and <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/InstanceStorage.html">Amazon EC2 instance store</a>
     * in the <i>Amazon EC2 User Guide for Windows</i>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>VirtualizationType</code> </b> — The recommended instance type uses the hardware virtual machine (HVM)
     * virtualization type and the current instance uses the paravirtual (PV) virtualization type. For more information
     * about the differences between these virtualization types, see <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/virtualization_types.html">Linux AMI virtualization
     * types</a> in the <i>Amazon EC2 User Guide for Linux</i>, or <a href=
     * "https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/windows-ami-version-history.html#virtualization-types"
     * >Windows AMI virtualization types</a> in the <i>Amazon EC2 User Guide for Windows</i>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>Architecture</code> </b> — The CPU architecture between the recommended instance type and the current
     * instance is different. For example, the recommended instance type might use an Arm CPU architecture and the
     * current instance type might use a different one, such as x86. Before migrating, you should consider recompiling
     * the software on your instance for the new architecture. Alternatively, you might switch to an Amazon Machine
     * Image (AMI) that supports the new architecture. For more information about the CPU architecture for each instance
     * type, see <a href="http://aws.amazon.com/ec2/instance-types/">Amazon EC2 Instance Types</a>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPlatformDifferences} method.
     * </p>
     * 
     * @return Describes the configuration differences between the current instance and the recommended instance type.
     *         You should consider the configuration differences before migrating your workloads from the current
     *         instance to the recommended instance type. The <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-instance-resize.html">Change the instance
     *         type guide for Linux</a> and <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/ec2-instance-resize.html">Change the
     *         instance type guide for Windows</a> provide general guidance for getting started with an instance
     *         migration.</p>
     *         <p>
     *         Platform differences include:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b> <code>Hypervisor</code> </b> — The hypervisor of the recommended instance type is different than that
     *         of the current instance. For example, the recommended instance type uses a Nitro hypervisor and the
     *         current instance uses a Xen hypervisor. The differences that you should consider between these
     *         hypervisors are covered in the <a href="http://aws.amazon.com/ec2/faqs/#Nitro_Hypervisor">Nitro
     *         Hypervisor</a> section of the Amazon EC2 frequently asked questions. For more information, see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#ec2-nitro-instances"
     *         >Instances built on the Nitro System</a> in the <i>Amazon EC2 User Guide for Linux</i>, or <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/instance-types.html#ec2-nitro-instances"
     *         >Instances built on the Nitro System</a> in the <i>Amazon EC2 User Guide for Windows</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>NetworkInterface</code> </b> — The network interface of the recommended instance type is
     *         different than that of the current instance. For example, the recommended instance type supports enhanced
     *         networking and the current instance might not. To enable enhanced networking for the recommended instance
     *         type, you must install the Elastic Network Adapter (ENA) driver or the Intel 82599 Virtual Function
     *         driver. For more information, see <a href=
     *         "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#instance-networking-storage"
     *         >Networking and storage features</a> and <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/enhanced-networking.html">Enhanced networking
     *         on Linux</a> in the <i>Amazon EC2 User Guide for Linux</i>, or <a href=
     *         "https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/instance-types.html#instance-networking-storage"
     *         >Networking and storage features</a> and <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/enhanced-networking.html">Enhanced
     *         networking on Windows</a> in the <i>Amazon EC2 User Guide for Windows</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>StorageInterface</code> </b> — The storage interface of the recommended instance type is
     *         different than that of the current instance. For example, the recommended instance type uses an NVMe
     *         storage interface and the current instance does not. To access NVMe volumes for the recommended instance
     *         type, you will need to install or upgrade the NVMe driver. For more information, see <a href=
     *         "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#instance-networking-storage"
     *         >Networking and storage features</a> and <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/nvme-ebs-volumes.html">Amazon EBS and NVMe on
     *         Linux instances</a> in the <i>Amazon EC2 User Guide for Linux</i>, or <a href=
     *         "https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/instance-types.html#instance-networking-storage"
     *         >Networking and storage features</a> and <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/nvme-ebs-volumes.html">Amazon EBS and NVMe
     *         on Windows instances</a> in the <i>Amazon EC2 User Guide for Windows</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>InstanceStoreAvailability</code> </b> — The recommended instance type does not support instance
     *         store volumes and the current instance does. Before migrating, you might need to back up the data on your
     *         instance store volumes if you want to preserve them. For more information, see <a
     *         href="https://aws.amazon.com/premiumsupport/knowledge-center/back-up-instance-store-ebs/">How do I back
     *         up an instance store volume on my Amazon EC2 instance to Amazon EBS?</a> in the <i>Amazon Web Services
     *         Premium Support Knowledge Base</i>. For more information, see <a href=
     *         "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#instance-networking-storage"
     *         >Networking and storage features</a> and <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/InstanceStorage.html">Amazon EC2 instance
     *         store</a> in the <i>Amazon EC2 User Guide for Linux</i>, or see <a href=
     *         "https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/instance-types.html#instance-networking-storage"
     *         >Networking and storage features</a> and <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/InstanceStorage.html">Amazon EC2 instance
     *         store</a> in the <i>Amazon EC2 User Guide for Windows</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>VirtualizationType</code> </b> — The recommended instance type uses the hardware virtual
     *         machine (HVM) virtualization type and the current instance uses the paravirtual (PV) virtualization type.
     *         For more information about the differences between these virtualization types, see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/virtualization_types.html">Linux AMI
     *         virtualization types</a> in the <i>Amazon EC2 User Guide for Linux</i>, or <a href=
     *         "https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/windows-ami-version-history.html#virtualization-types"
     *         >Windows AMI virtualization types</a> in the <i>Amazon EC2 User Guide for Windows</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>Architecture</code> </b> — The CPU architecture between the recommended instance type and the
     *         current instance is different. For example, the recommended instance type might use an Arm CPU
     *         architecture and the current instance type might use a different one, such as x86. Before migrating, you
     *         should consider recompiling the software on your instance for the new architecture. Alternatively, you
     *         might switch to an Amazon Machine Image (AMI) that supports the new architecture. For more information
     *         about the CPU architecture for each instance type, see <a
     *         href="http://aws.amazon.com/ec2/instance-types/">Amazon EC2 Instance Types</a>.
     *         </p>
     *         </li>
     */
    public final List<PlatformDifference> platformDifferences() {
        return PlatformDifferencesCopier.copyStringToEnum(platformDifferences);
    }

    /**
     * For responses, this returns true if the service returned a value for the PlatformDifferences property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasPlatformDifferences() {
        return platformDifferences != null && !(platformDifferences instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Describes the configuration differences between the current instance and the recommended instance type. You
     * should consider the configuration differences before migrating your workloads from the current instance to the
     * recommended instance type. The <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-instance-resize.html">Change the instance type
     * guide for Linux</a> and <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/ec2-instance-resize.html">Change the instance type
     * guide for Windows</a> provide general guidance for getting started with an instance migration.
     * </p>
     * <p>
     * Platform differences include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b> <code>Hypervisor</code> </b> — The hypervisor of the recommended instance type is different than that of the
     * current instance. For example, the recommended instance type uses a Nitro hypervisor and the current instance
     * uses a Xen hypervisor. The differences that you should consider between these hypervisors are covered in the <a
     * href="http://aws.amazon.com/ec2/faqs/#Nitro_Hypervisor">Nitro Hypervisor</a> section of the Amazon EC2 frequently
     * asked questions. For more information, see <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#ec2-nitro-instances">Instances
     * built on the Nitro System</a> in the <i>Amazon EC2 User Guide for Linux</i>, or <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/instance-types.html#ec2-nitro-instances">Instances
     * built on the Nitro System</a> in the <i>Amazon EC2 User Guide for Windows</i>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>NetworkInterface</code> </b> — The network interface of the recommended instance type is different than
     * that of the current instance. For example, the recommended instance type supports enhanced networking and the
     * current instance might not. To enable enhanced networking for the recommended instance type, you must install the
     * Elastic Network Adapter (ENA) driver or the Intel 82599 Virtual Function driver. For more information, see <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#instance-networking-storage">
     * Networking and storage features</a> and <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/enhanced-networking.html">Enhanced networking on
     * Linux</a> in the <i>Amazon EC2 User Guide for Linux</i>, or <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/instance-types.html#instance-networking-storage"
     * >Networking and storage features</a> and <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/enhanced-networking.html">Enhanced networking on
     * Windows</a> in the <i>Amazon EC2 User Guide for Windows</i>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>StorageInterface</code> </b> — The storage interface of the recommended instance type is different than
     * that of the current instance. For example, the recommended instance type uses an NVMe storage interface and the
     * current instance does not. To access NVMe volumes for the recommended instance type, you will need to install or
     * upgrade the NVMe driver. For more information, see <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#instance-networking-storage"
     * >Networking and storage features</a> and <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/nvme-ebs-volumes.html">Amazon EBS and NVMe on Linux
     * instances</a> in the <i>Amazon EC2 User Guide for Linux</i>, or <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/instance-types.html#instance-networking-storage"
     * >Networking and storage features</a> and <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/nvme-ebs-volumes.html">Amazon EBS and NVMe on
     * Windows instances</a> in the <i>Amazon EC2 User Guide for Windows</i>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>InstanceStoreAvailability</code> </b> — The recommended instance type does not support instance store
     * volumes and the current instance does. Before migrating, you might need to back up the data on your instance
     * store volumes if you want to preserve them. For more information, see <a
     * href="https://aws.amazon.com/premiumsupport/knowledge-center/back-up-instance-store-ebs/">How do I back up an
     * instance store volume on my Amazon EC2 instance to Amazon EBS?</a> in the <i>Amazon Web Services Premium Support
     * Knowledge Base</i>. For more information, see <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#instance-networking-storage"
     * >Networking and storage features</a> and <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/InstanceStorage.html">Amazon EC2 instance store</a> in
     * the <i>Amazon EC2 User Guide for Linux</i>, or see <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/instance-types.html#instance-networking-storage"
     * >Networking and storage features</a> and <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/InstanceStorage.html">Amazon EC2 instance store</a>
     * in the <i>Amazon EC2 User Guide for Windows</i>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>VirtualizationType</code> </b> — The recommended instance type uses the hardware virtual machine (HVM)
     * virtualization type and the current instance uses the paravirtual (PV) virtualization type. For more information
     * about the differences between these virtualization types, see <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/virtualization_types.html">Linux AMI virtualization
     * types</a> in the <i>Amazon EC2 User Guide for Linux</i>, or <a href=
     * "https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/windows-ami-version-history.html#virtualization-types"
     * >Windows AMI virtualization types</a> in the <i>Amazon EC2 User Guide for Windows</i>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>Architecture</code> </b> — The CPU architecture between the recommended instance type and the current
     * instance is different. For example, the recommended instance type might use an Arm CPU architecture and the
     * current instance type might use a different one, such as x86. Before migrating, you should consider recompiling
     * the software on your instance for the new architecture. Alternatively, you might switch to an Amazon Machine
     * Image (AMI) that supports the new architecture. For more information about the CPU architecture for each instance
     * type, see <a href="http://aws.amazon.com/ec2/instance-types/">Amazon EC2 Instance Types</a>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPlatformDifferences} method.
     * </p>
     * 
     * @return Describes the configuration differences between the current instance and the recommended instance type.
     *         You should consider the configuration differences before migrating your workloads from the current
     *         instance to the recommended instance type. The <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-instance-resize.html">Change the instance
     *         type guide for Linux</a> and <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/ec2-instance-resize.html">Change the
     *         instance type guide for Windows</a> provide general guidance for getting started with an instance
     *         migration.</p>
     *         <p>
     *         Platform differences include:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b> <code>Hypervisor</code> </b> — The hypervisor of the recommended instance type is different than that
     *         of the current instance. For example, the recommended instance type uses a Nitro hypervisor and the
     *         current instance uses a Xen hypervisor. The differences that you should consider between these
     *         hypervisors are covered in the <a href="http://aws.amazon.com/ec2/faqs/#Nitro_Hypervisor">Nitro
     *         Hypervisor</a> section of the Amazon EC2 frequently asked questions. For more information, see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#ec2-nitro-instances"
     *         >Instances built on the Nitro System</a> in the <i>Amazon EC2 User Guide for Linux</i>, or <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/instance-types.html#ec2-nitro-instances"
     *         >Instances built on the Nitro System</a> in the <i>Amazon EC2 User Guide for Windows</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>NetworkInterface</code> </b> — The network interface of the recommended instance type is
     *         different than that of the current instance. For example, the recommended instance type supports enhanced
     *         networking and the current instance might not. To enable enhanced networking for the recommended instance
     *         type, you must install the Elastic Network Adapter (ENA) driver or the Intel 82599 Virtual Function
     *         driver. For more information, see <a href=
     *         "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#instance-networking-storage"
     *         >Networking and storage features</a> and <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/enhanced-networking.html">Enhanced networking
     *         on Linux</a> in the <i>Amazon EC2 User Guide for Linux</i>, or <a href=
     *         "https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/instance-types.html#instance-networking-storage"
     *         >Networking and storage features</a> and <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/enhanced-networking.html">Enhanced
     *         networking on Windows</a> in the <i>Amazon EC2 User Guide for Windows</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>StorageInterface</code> </b> — The storage interface of the recommended instance type is
     *         different than that of the current instance. For example, the recommended instance type uses an NVMe
     *         storage interface and the current instance does not. To access NVMe volumes for the recommended instance
     *         type, you will need to install or upgrade the NVMe driver. For more information, see <a href=
     *         "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#instance-networking-storage"
     *         >Networking and storage features</a> and <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/nvme-ebs-volumes.html">Amazon EBS and NVMe on
     *         Linux instances</a> in the <i>Amazon EC2 User Guide for Linux</i>, or <a href=
     *         "https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/instance-types.html#instance-networking-storage"
     *         >Networking and storage features</a> and <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/nvme-ebs-volumes.html">Amazon EBS and NVMe
     *         on Windows instances</a> in the <i>Amazon EC2 User Guide for Windows</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>InstanceStoreAvailability</code> </b> — The recommended instance type does not support instance
     *         store volumes and the current instance does. Before migrating, you might need to back up the data on your
     *         instance store volumes if you want to preserve them. For more information, see <a
     *         href="https://aws.amazon.com/premiumsupport/knowledge-center/back-up-instance-store-ebs/">How do I back
     *         up an instance store volume on my Amazon EC2 instance to Amazon EBS?</a> in the <i>Amazon Web Services
     *         Premium Support Knowledge Base</i>. For more information, see <a href=
     *         "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#instance-networking-storage"
     *         >Networking and storage features</a> and <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/InstanceStorage.html">Amazon EC2 instance
     *         store</a> in the <i>Amazon EC2 User Guide for Linux</i>, or see <a href=
     *         "https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/instance-types.html#instance-networking-storage"
     *         >Networking and storage features</a> and <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/InstanceStorage.html">Amazon EC2 instance
     *         store</a> in the <i>Amazon EC2 User Guide for Windows</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>VirtualizationType</code> </b> — The recommended instance type uses the hardware virtual
     *         machine (HVM) virtualization type and the current instance uses the paravirtual (PV) virtualization type.
     *         For more information about the differences between these virtualization types, see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/virtualization_types.html">Linux AMI
     *         virtualization types</a> in the <i>Amazon EC2 User Guide for Linux</i>, or <a href=
     *         "https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/windows-ami-version-history.html#virtualization-types"
     *         >Windows AMI virtualization types</a> in the <i>Amazon EC2 User Guide for Windows</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>Architecture</code> </b> — The CPU architecture between the recommended instance type and the
     *         current instance is different. For example, the recommended instance type might use an Arm CPU
     *         architecture and the current instance type might use a different one, such as x86. Before migrating, you
     *         should consider recompiling the software on your instance for the new architecture. Alternatively, you
     *         might switch to an Amazon Machine Image (AMI) that supports the new architecture. For more information
     *         about the CPU architecture for each instance type, see <a
     *         href="http://aws.amazon.com/ec2/instance-types/">Amazon EC2 Instance Types</a>.
     *         </p>
     *         </li>
     */
    public final List<String> platformDifferencesAsStrings() {
        return platformDifferences;
    }

    /**
     * <p>
     * The performance risk of the instance recommendation option.
     * </p>
     * <p>
     * Performance risk indicates the likelihood of the recommended instance type not meeting the resource needs of your
     * workload. Compute Optimizer calculates an individual performance risk score for each specification of the
     * recommended instance, including CPU, memory, EBS throughput, EBS IOPS, disk throughput, disk IOPS, network
     * throughput, and network PPS. The performance risk of the recommended instance is calculated as the maximum
     * performance risk score across the analyzed resource specifications.
     * </p>
     * <p>
     * The value ranges from <code>0</code> - <code>4</code>, with <code>0</code> meaning that the recommended resource
     * is predicted to always provide enough hardware capability. The higher the performance risk is, the more likely
     * you should validate whether the recommendation will meet the performance requirements of your workload before
     * migrating your resource.
     * </p>
     * 
     * @return The performance risk of the instance recommendation option.</p>
     *         <p>
     *         Performance risk indicates the likelihood of the recommended instance type not meeting the resource needs
     *         of your workload. Compute Optimizer calculates an individual performance risk score for each
     *         specification of the recommended instance, including CPU, memory, EBS throughput, EBS IOPS, disk
     *         throughput, disk IOPS, network throughput, and network PPS. The performance risk of the recommended
     *         instance is calculated as the maximum performance risk score across the analyzed resource specifications.
     *         </p>
     *         <p>
     *         The value ranges from <code>0</code> - <code>4</code>, with <code>0</code> meaning that the recommended
     *         resource is predicted to always provide enough hardware capability. The higher the performance risk is,
     *         the more likely you should validate whether the recommendation will meet the performance requirements of
     *         your workload before migrating your resource.
     */
    public final Double performanceRisk() {
        return performanceRisk;
    }

    /**
     * <p>
     * The rank of the instance recommendation option.
     * </p>
     * <p>
     * The top recommendation option is ranked as <code>1</code>.
     * </p>
     * 
     * @return The rank of the instance recommendation option.</p>
     *         <p>
     *         The top recommendation option is ranked as <code>1</code>.
     */
    public final Integer rank() {
        return rank;
    }

    /**
     * <p>
     * An object that describes the savings opportunity for the instance recommendation option. Savings opportunity
     * includes the estimated monthly savings amount and percentage.
     * </p>
     * 
     * @return An object that describes the savings opportunity for the instance recommendation option. Savings
     *         opportunity includes the estimated monthly savings amount and percentage.
     */
    public final SavingsOpportunity savingsOpportunity() {
        return savingsOpportunity;
    }

    /**
     * <p>
     * The level of effort required to migrate from the current instance type to the recommended instance type.
     * </p>
     * <p>
     * For example, the migration effort is <code>Low</code> if Amazon EMR is the inferred workload type and an Amazon
     * Web Services Graviton instance type is recommended. The migration effort is <code>Medium</code> if a workload
     * type couldn't be inferred but an Amazon Web Services Graviton instance type is recommended. The migration effort
     * is <code>VeryLow</code> if both the current and recommended instance types are of the same CPU architecture.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #migrationEffort}
     * will return {@link MigrationEffort#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #migrationEffortAsString}.
     * </p>
     * 
     * @return The level of effort required to migrate from the current instance type to the recommended instance
     *         type.</p>
     *         <p>
     *         For example, the migration effort is <code>Low</code> if Amazon EMR is the inferred workload type and an
     *         Amazon Web Services Graviton instance type is recommended. The migration effort is <code>Medium</code> if
     *         a workload type couldn't be inferred but an Amazon Web Services Graviton instance type is recommended.
     *         The migration effort is <code>VeryLow</code> if both the current and recommended instance types are of
     *         the same CPU architecture.
     * @see MigrationEffort
     */
    public final MigrationEffort migrationEffort() {
        return MigrationEffort.fromValue(migrationEffort);
    }

    /**
     * <p>
     * The level of effort required to migrate from the current instance type to the recommended instance type.
     * </p>
     * <p>
     * For example, the migration effort is <code>Low</code> if Amazon EMR is the inferred workload type and an Amazon
     * Web Services Graviton instance type is recommended. The migration effort is <code>Medium</code> if a workload
     * type couldn't be inferred but an Amazon Web Services Graviton instance type is recommended. The migration effort
     * is <code>VeryLow</code> if both the current and recommended instance types are of the same CPU architecture.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #migrationEffort}
     * will return {@link MigrationEffort#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #migrationEffortAsString}.
     * </p>
     * 
     * @return The level of effort required to migrate from the current instance type to the recommended instance
     *         type.</p>
     *         <p>
     *         For example, the migration effort is <code>Low</code> if Amazon EMR is the inferred workload type and an
     *         Amazon Web Services Graviton instance type is recommended. The migration effort is <code>Medium</code> if
     *         a workload type couldn't be inferred but an Amazon Web Services Graviton instance type is recommended.
     *         The migration effort is <code>VeryLow</code> if both the current and recommended instance types are of
     *         the same CPU architecture.
     * @see MigrationEffort
     */
    public final String migrationEffortAsString() {
        return migrationEffort;
    }

    /**
     * <p>
     * Describes the GPU accelerator settings for the recommended instance type.
     * </p>
     * 
     * @return Describes the GPU accelerator settings for the recommended instance type.
     */
    public final GpuInfo instanceGpuInfo() {
        return instanceGpuInfo;
    }

    /**
     * <p>
     * An object that describes the savings opportunity for the instance recommendation option that includes Savings
     * Plans and Reserved Instances discounts. Savings opportunity includes the estimated monthly savings and
     * percentage.
     * </p>
     * 
     * @return An object that describes the savings opportunity for the instance recommendation option that includes
     *         Savings Plans and Reserved Instances discounts. Savings opportunity includes the estimated monthly
     *         savings and percentage.
     */
    public final InstanceSavingsOpportunityAfterDiscounts savingsOpportunityAfterDiscounts() {
        return savingsOpportunityAfterDiscounts;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(instanceType());
        hashCode = 31 * hashCode + Objects.hashCode(hasProjectedUtilizationMetrics() ? projectedUtilizationMetrics() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasPlatformDifferences() ? platformDifferencesAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(performanceRisk());
        hashCode = 31 * hashCode + Objects.hashCode(rank());
        hashCode = 31 * hashCode + Objects.hashCode(savingsOpportunity());
        hashCode = 31 * hashCode + Objects.hashCode(migrationEffortAsString());
        hashCode = 31 * hashCode + Objects.hashCode(instanceGpuInfo());
        hashCode = 31 * hashCode + Objects.hashCode(savingsOpportunityAfterDiscounts());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof InstanceRecommendationOption)) {
            return false;
        }
        InstanceRecommendationOption other = (InstanceRecommendationOption) obj;
        return Objects.equals(instanceType(), other.instanceType())
                && hasProjectedUtilizationMetrics() == other.hasProjectedUtilizationMetrics()
                && Objects.equals(projectedUtilizationMetrics(), other.projectedUtilizationMetrics())
                && hasPlatformDifferences() == other.hasPlatformDifferences()
                && Objects.equals(platformDifferencesAsStrings(), other.platformDifferencesAsStrings())
                && Objects.equals(performanceRisk(), other.performanceRisk()) && Objects.equals(rank(), other.rank())
                && Objects.equals(savingsOpportunity(), other.savingsOpportunity())
                && Objects.equals(migrationEffortAsString(), other.migrationEffortAsString())
                && Objects.equals(instanceGpuInfo(), other.instanceGpuInfo())
                && Objects.equals(savingsOpportunityAfterDiscounts(), other.savingsOpportunityAfterDiscounts());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("InstanceRecommendationOption").add("InstanceType", instanceType())
                .add("ProjectedUtilizationMetrics", hasProjectedUtilizationMetrics() ? projectedUtilizationMetrics() : null)
                .add("PlatformDifferences", hasPlatformDifferences() ? platformDifferencesAsStrings() : null)
                .add("PerformanceRisk", performanceRisk()).add("Rank", rank()).add("SavingsOpportunity", savingsOpportunity())
                .add("MigrationEffort", migrationEffortAsString()).add("InstanceGpuInfo", instanceGpuInfo())
                .add("SavingsOpportunityAfterDiscounts", savingsOpportunityAfterDiscounts()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "instanceType":
            return Optional.ofNullable(clazz.cast(instanceType()));
        case "projectedUtilizationMetrics":
            return Optional.ofNullable(clazz.cast(projectedUtilizationMetrics()));
        case "platformDifferences":
            return Optional.ofNullable(clazz.cast(platformDifferencesAsStrings()));
        case "performanceRisk":
            return Optional.ofNullable(clazz.cast(performanceRisk()));
        case "rank":
            return Optional.ofNullable(clazz.cast(rank()));
        case "savingsOpportunity":
            return Optional.ofNullable(clazz.cast(savingsOpportunity()));
        case "migrationEffort":
            return Optional.ofNullable(clazz.cast(migrationEffortAsString()));
        case "instanceGpuInfo":
            return Optional.ofNullable(clazz.cast(instanceGpuInfo()));
        case "savingsOpportunityAfterDiscounts":
            return Optional.ofNullable(clazz.cast(savingsOpportunityAfterDiscounts()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<InstanceRecommendationOption, T> g) {
        return obj -> g.apply((InstanceRecommendationOption) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, InstanceRecommendationOption> {
        /**
         * <p>
         * The instance type of the instance recommendation.
         * </p>
         * 
         * @param instanceType
         *        The instance type of the instance recommendation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceType(String instanceType);

        /**
         * <p>
         * An array of objects that describe the projected utilization metrics of the instance recommendation option.
         * </p>
         * <note>
         * <p>
         * The <code>Cpu</code> and <code>Memory</code> metrics are the only projected utilization metrics returned.
         * Additionally, the <code>Memory</code> metric is returned only for resources that have the unified CloudWatch
         * agent installed on them. For more information, see <a
         * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/metrics.html#cw-agent">Enabling Memory
         * Utilization with the CloudWatch Agent</a>.
         * </p>
         * </note>
         * 
         * @param projectedUtilizationMetrics
         *        An array of objects that describe the projected utilization metrics of the instance recommendation
         *        option.</p> <note>
         *        <p>
         *        The <code>Cpu</code> and <code>Memory</code> metrics are the only projected utilization metrics
         *        returned. Additionally, the <code>Memory</code> metric is returned only for resources that have the
         *        unified CloudWatch agent installed on them. For more information, see <a
         *        href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/metrics.html#cw-agent">Enabling Memory
         *        Utilization with the CloudWatch Agent</a>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder projectedUtilizationMetrics(Collection<UtilizationMetric> projectedUtilizationMetrics);

        /**
         * <p>
         * An array of objects that describe the projected utilization metrics of the instance recommendation option.
         * </p>
         * <note>
         * <p>
         * The <code>Cpu</code> and <code>Memory</code> metrics are the only projected utilization metrics returned.
         * Additionally, the <code>Memory</code> metric is returned only for resources that have the unified CloudWatch
         * agent installed on them. For more information, see <a
         * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/metrics.html#cw-agent">Enabling Memory
         * Utilization with the CloudWatch Agent</a>.
         * </p>
         * </note>
         * 
         * @param projectedUtilizationMetrics
         *        An array of objects that describe the projected utilization metrics of the instance recommendation
         *        option.</p> <note>
         *        <p>
         *        The <code>Cpu</code> and <code>Memory</code> metrics are the only projected utilization metrics
         *        returned. Additionally, the <code>Memory</code> metric is returned only for resources that have the
         *        unified CloudWatch agent installed on them. For more information, see <a
         *        href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/metrics.html#cw-agent">Enabling Memory
         *        Utilization with the CloudWatch Agent</a>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder projectedUtilizationMetrics(UtilizationMetric... projectedUtilizationMetrics);

        /**
         * <p>
         * An array of objects that describe the projected utilization metrics of the instance recommendation option.
         * </p>
         * <note>
         * <p>
         * The <code>Cpu</code> and <code>Memory</code> metrics are the only projected utilization metrics returned.
         * Additionally, the <code>Memory</code> metric is returned only for resources that have the unified CloudWatch
         * agent installed on them. For more information, see <a
         * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/metrics.html#cw-agent">Enabling Memory
         * Utilization with the CloudWatch Agent</a>.
         * </p>
         * </note> This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.computeoptimizer.model.UtilizationMetric.Builder} avoiding the need to
         * create one manually via
         * {@link software.amazon.awssdk.services.computeoptimizer.model.UtilizationMetric#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.computeoptimizer.model.UtilizationMetric.Builder#build()} is called
         * immediately and its result is passed to {@link #projectedUtilizationMetrics(List<UtilizationMetric>)}.
         * 
         * @param projectedUtilizationMetrics
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.computeoptimizer.model.UtilizationMetric.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #projectedUtilizationMetrics(java.util.Collection<UtilizationMetric>)
         */
        Builder projectedUtilizationMetrics(Consumer<UtilizationMetric.Builder>... projectedUtilizationMetrics);

        /**
         * <p>
         * Describes the configuration differences between the current instance and the recommended instance type. You
         * should consider the configuration differences before migrating your workloads from the current instance to
         * the recommended instance type. The <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-instance-resize.html">Change the instance type
         * guide for Linux</a> and <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/ec2-instance-resize.html">Change the instance
         * type guide for Windows</a> provide general guidance for getting started with an instance migration.
         * </p>
         * <p>
         * Platform differences include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b> <code>Hypervisor</code> </b> — The hypervisor of the recommended instance type is different than that of
         * the current instance. For example, the recommended instance type uses a Nitro hypervisor and the current
         * instance uses a Xen hypervisor. The differences that you should consider between these hypervisors are
         * covered in the <a href="http://aws.amazon.com/ec2/faqs/#Nitro_Hypervisor">Nitro Hypervisor</a> section of the
         * Amazon EC2 frequently asked questions. For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#ec2-nitro-instances">Instances
         * built on the Nitro System</a> in the <i>Amazon EC2 User Guide for Linux</i>, or <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/instance-types.html#ec2-nitro-instances"
         * >Instances built on the Nitro System</a> in the <i>Amazon EC2 User Guide for Windows</i>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>NetworkInterface</code> </b> — The network interface of the recommended instance type is different
         * than that of the current instance. For example, the recommended instance type supports enhanced networking
         * and the current instance might not. To enable enhanced networking for the recommended instance type, you must
         * install the Elastic Network Adapter (ENA) driver or the Intel 82599 Virtual Function driver. For more
         * information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#instance-networking-storage"
         * >Networking and storage features</a> and <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/enhanced-networking.html">Enhanced networking on
         * Linux</a> in the <i>Amazon EC2 User Guide for Linux</i>, or <a href=
         * "https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/instance-types.html#instance-networking-storage"
         * >Networking and storage features</a> and <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/enhanced-networking.html">Enhanced networking on
         * Windows</a> in the <i>Amazon EC2 User Guide for Windows</i>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>StorageInterface</code> </b> — The storage interface of the recommended instance type is different
         * than that of the current instance. For example, the recommended instance type uses an NVMe storage interface
         * and the current instance does not. To access NVMe volumes for the recommended instance type, you will need to
         * install or upgrade the NVMe driver. For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#instance-networking-storage"
         * >Networking and storage features</a> and <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/nvme-ebs-volumes.html">Amazon EBS and NVMe on Linux
         * instances</a> in the <i>Amazon EC2 User Guide for Linux</i>, or <a href=
         * "https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/instance-types.html#instance-networking-storage"
         * >Networking and storage features</a> and <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/nvme-ebs-volumes.html">Amazon EBS and NVMe on
         * Windows instances</a> in the <i>Amazon EC2 User Guide for Windows</i>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>InstanceStoreAvailability</code> </b> — The recommended instance type does not support instance
         * store volumes and the current instance does. Before migrating, you might need to back up the data on your
         * instance store volumes if you want to preserve them. For more information, see <a
         * href="https://aws.amazon.com/premiumsupport/knowledge-center/back-up-instance-store-ebs/">How do I back up an
         * instance store volume on my Amazon EC2 instance to Amazon EBS?</a> in the <i>Amazon Web Services Premium
         * Support Knowledge Base</i>. For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#instance-networking-storage"
         * >Networking and storage features</a> and <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/InstanceStorage.html">Amazon EC2 instance store</a>
         * in the <i>Amazon EC2 User Guide for Linux</i>, or see <a href=
         * "https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/instance-types.html#instance-networking-storage"
         * >Networking and storage features</a> and <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/InstanceStorage.html">Amazon EC2 instance
         * store</a> in the <i>Amazon EC2 User Guide for Windows</i>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>VirtualizationType</code> </b> — The recommended instance type uses the hardware virtual machine
         * (HVM) virtualization type and the current instance uses the paravirtual (PV) virtualization type. For more
         * information about the differences between these virtualization types, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/virtualization_types.html">Linux AMI virtualization
         * types</a> in the <i>Amazon EC2 User Guide for Linux</i>, or <a href=
         * "https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/windows-ami-version-history.html#virtualization-types"
         * >Windows AMI virtualization types</a> in the <i>Amazon EC2 User Guide for Windows</i>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>Architecture</code> </b> — The CPU architecture between the recommended instance type and the
         * current instance is different. For example, the recommended instance type might use an Arm CPU architecture
         * and the current instance type might use a different one, such as x86. Before migrating, you should consider
         * recompiling the software on your instance for the new architecture. Alternatively, you might switch to an
         * Amazon Machine Image (AMI) that supports the new architecture. For more information about the CPU
         * architecture for each instance type, see <a href="http://aws.amazon.com/ec2/instance-types/">Amazon EC2
         * Instance Types</a>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param platformDifferences
         *        Describes the configuration differences between the current instance and the recommended instance
         *        type. You should consider the configuration differences before migrating your workloads from the
         *        current instance to the recommended instance type. The <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-instance-resize.html">Change the
         *        instance type guide for Linux</a> and <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/ec2-instance-resize.html">Change the
         *        instance type guide for Windows</a> provide general guidance for getting started with an instance
         *        migration.</p>
         *        <p>
         *        Platform differences include:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b> <code>Hypervisor</code> </b> — The hypervisor of the recommended instance type is different than
         *        that of the current instance. For example, the recommended instance type uses a Nitro hypervisor and
         *        the current instance uses a Xen hypervisor. The differences that you should consider between these
         *        hypervisors are covered in the <a href="http://aws.amazon.com/ec2/faqs/#Nitro_Hypervisor">Nitro
         *        Hypervisor</a> section of the Amazon EC2 frequently asked questions. For more information, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#ec2-nitro-instances">
         *        Instances built on the Nitro System</a> in the <i>Amazon EC2 User Guide for Linux</i>, or <a href=
         *        "https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/instance-types.html#ec2-nitro-instances"
         *        >Instances built on the Nitro System</a> in the <i>Amazon EC2 User Guide for Windows</i>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>NetworkInterface</code> </b> — The network interface of the recommended instance type is
         *        different than that of the current instance. For example, the recommended instance type supports
         *        enhanced networking and the current instance might not. To enable enhanced networking for the
         *        recommended instance type, you must install the Elastic Network Adapter (ENA) driver or the Intel
         *        82599 Virtual Function driver. For more information, see <a href=
         *        "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#instance-networking-storage"
         *        >Networking and storage features</a> and <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/enhanced-networking.html">Enhanced
         *        networking on Linux</a> in the <i>Amazon EC2 User Guide for Linux</i>, or <a href=
         *        "https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/instance-types.html#instance-networking-storage"
         *        >Networking and storage features</a> and <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/enhanced-networking.html">Enhanced
         *        networking on Windows</a> in the <i>Amazon EC2 User Guide for Windows</i>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>StorageInterface</code> </b> — The storage interface of the recommended instance type is
         *        different than that of the current instance. For example, the recommended instance type uses an NVMe
         *        storage interface and the current instance does not. To access NVMe volumes for the recommended
         *        instance type, you will need to install or upgrade the NVMe driver. For more information, see <a href=
         *        "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#instance-networking-storage"
         *        >Networking and storage features</a> and <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/nvme-ebs-volumes.html">Amazon EBS and NVMe
         *        on Linux instances</a> in the <i>Amazon EC2 User Guide for Linux</i>, or <a href=
         *        "https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/instance-types.html#instance-networking-storage"
         *        >Networking and storage features</a> and <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/nvme-ebs-volumes.html">Amazon EBS and
         *        NVMe on Windows instances</a> in the <i>Amazon EC2 User Guide for Windows</i>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>InstanceStoreAvailability</code> </b> — The recommended instance type does not support
         *        instance store volumes and the current instance does. Before migrating, you might need to back up the
         *        data on your instance store volumes if you want to preserve them. For more information, see <a
         *        href="https://aws.amazon.com/premiumsupport/knowledge-center/back-up-instance-store-ebs/">How do I
         *        back up an instance store volume on my Amazon EC2 instance to Amazon EBS?</a> in the <i>Amazon Web
         *        Services Premium Support Knowledge Base</i>. For more information, see <a href=
         *        "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#instance-networking-storage"
         *        >Networking and storage features</a> and <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/InstanceStorage.html">Amazon EC2 instance
         *        store</a> in the <i>Amazon EC2 User Guide for Linux</i>, or see <a href=
         *        "https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/instance-types.html#instance-networking-storage"
         *        >Networking and storage features</a> and <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/InstanceStorage.html">Amazon EC2 instance
         *        store</a> in the <i>Amazon EC2 User Guide for Windows</i>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>VirtualizationType</code> </b> — The recommended instance type uses the hardware virtual
         *        machine (HVM) virtualization type and the current instance uses the paravirtual (PV) virtualization
         *        type. For more information about the differences between these virtualization types, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/virtualization_types.html">Linux AMI
         *        virtualization types</a> in the <i>Amazon EC2 User Guide for Linux</i>, or <a href=
         *        "https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/windows-ami-version-history.html#virtualization-types"
         *        >Windows AMI virtualization types</a> in the <i>Amazon EC2 User Guide for Windows</i>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>Architecture</code> </b> — The CPU architecture between the recommended instance type and
         *        the current instance is different. For example, the recommended instance type might use an Arm CPU
         *        architecture and the current instance type might use a different one, such as x86. Before migrating,
         *        you should consider recompiling the software on your instance for the new architecture. Alternatively,
         *        you might switch to an Amazon Machine Image (AMI) that supports the new architecture. For more
         *        information about the CPU architecture for each instance type, see <a
         *        href="http://aws.amazon.com/ec2/instance-types/">Amazon EC2 Instance Types</a>.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder platformDifferencesWithStrings(Collection<String> platformDifferences);

        /**
         * <p>
         * Describes the configuration differences between the current instance and the recommended instance type. You
         * should consider the configuration differences before migrating your workloads from the current instance to
         * the recommended instance type. The <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-instance-resize.html">Change the instance type
         * guide for Linux</a> and <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/ec2-instance-resize.html">Change the instance
         * type guide for Windows</a> provide general guidance for getting started with an instance migration.
         * </p>
         * <p>
         * Platform differences include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b> <code>Hypervisor</code> </b> — The hypervisor of the recommended instance type is different than that of
         * the current instance. For example, the recommended instance type uses a Nitro hypervisor and the current
         * instance uses a Xen hypervisor. The differences that you should consider between these hypervisors are
         * covered in the <a href="http://aws.amazon.com/ec2/faqs/#Nitro_Hypervisor">Nitro Hypervisor</a> section of the
         * Amazon EC2 frequently asked questions. For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#ec2-nitro-instances">Instances
         * built on the Nitro System</a> in the <i>Amazon EC2 User Guide for Linux</i>, or <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/instance-types.html#ec2-nitro-instances"
         * >Instances built on the Nitro System</a> in the <i>Amazon EC2 User Guide for Windows</i>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>NetworkInterface</code> </b> — The network interface of the recommended instance type is different
         * than that of the current instance. For example, the recommended instance type supports enhanced networking
         * and the current instance might not. To enable enhanced networking for the recommended instance type, you must
         * install the Elastic Network Adapter (ENA) driver or the Intel 82599 Virtual Function driver. For more
         * information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#instance-networking-storage"
         * >Networking and storage features</a> and <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/enhanced-networking.html">Enhanced networking on
         * Linux</a> in the <i>Amazon EC2 User Guide for Linux</i>, or <a href=
         * "https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/instance-types.html#instance-networking-storage"
         * >Networking and storage features</a> and <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/enhanced-networking.html">Enhanced networking on
         * Windows</a> in the <i>Amazon EC2 User Guide for Windows</i>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>StorageInterface</code> </b> — The storage interface of the recommended instance type is different
         * than that of the current instance. For example, the recommended instance type uses an NVMe storage interface
         * and the current instance does not. To access NVMe volumes for the recommended instance type, you will need to
         * install or upgrade the NVMe driver. For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#instance-networking-storage"
         * >Networking and storage features</a> and <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/nvme-ebs-volumes.html">Amazon EBS and NVMe on Linux
         * instances</a> in the <i>Amazon EC2 User Guide for Linux</i>, or <a href=
         * "https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/instance-types.html#instance-networking-storage"
         * >Networking and storage features</a> and <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/nvme-ebs-volumes.html">Amazon EBS and NVMe on
         * Windows instances</a> in the <i>Amazon EC2 User Guide for Windows</i>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>InstanceStoreAvailability</code> </b> — The recommended instance type does not support instance
         * store volumes and the current instance does. Before migrating, you might need to back up the data on your
         * instance store volumes if you want to preserve them. For more information, see <a
         * href="https://aws.amazon.com/premiumsupport/knowledge-center/back-up-instance-store-ebs/">How do I back up an
         * instance store volume on my Amazon EC2 instance to Amazon EBS?</a> in the <i>Amazon Web Services Premium
         * Support Knowledge Base</i>. For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#instance-networking-storage"
         * >Networking and storage features</a> and <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/InstanceStorage.html">Amazon EC2 instance store</a>
         * in the <i>Amazon EC2 User Guide for Linux</i>, or see <a href=
         * "https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/instance-types.html#instance-networking-storage"
         * >Networking and storage features</a> and <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/InstanceStorage.html">Amazon EC2 instance
         * store</a> in the <i>Amazon EC2 User Guide for Windows</i>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>VirtualizationType</code> </b> — The recommended instance type uses the hardware virtual machine
         * (HVM) virtualization type and the current instance uses the paravirtual (PV) virtualization type. For more
         * information about the differences between these virtualization types, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/virtualization_types.html">Linux AMI virtualization
         * types</a> in the <i>Amazon EC2 User Guide for Linux</i>, or <a href=
         * "https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/windows-ami-version-history.html#virtualization-types"
         * >Windows AMI virtualization types</a> in the <i>Amazon EC2 User Guide for Windows</i>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>Architecture</code> </b> — The CPU architecture between the recommended instance type and the
         * current instance is different. For example, the recommended instance type might use an Arm CPU architecture
         * and the current instance type might use a different one, such as x86. Before migrating, you should consider
         * recompiling the software on your instance for the new architecture. Alternatively, you might switch to an
         * Amazon Machine Image (AMI) that supports the new architecture. For more information about the CPU
         * architecture for each instance type, see <a href="http://aws.amazon.com/ec2/instance-types/">Amazon EC2
         * Instance Types</a>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param platformDifferences
         *        Describes the configuration differences between the current instance and the recommended instance
         *        type. You should consider the configuration differences before migrating your workloads from the
         *        current instance to the recommended instance type. The <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-instance-resize.html">Change the
         *        instance type guide for Linux</a> and <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/ec2-instance-resize.html">Change the
         *        instance type guide for Windows</a> provide general guidance for getting started with an instance
         *        migration.</p>
         *        <p>
         *        Platform differences include:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b> <code>Hypervisor</code> </b> — The hypervisor of the recommended instance type is different than
         *        that of the current instance. For example, the recommended instance type uses a Nitro hypervisor and
         *        the current instance uses a Xen hypervisor. The differences that you should consider between these
         *        hypervisors are covered in the <a href="http://aws.amazon.com/ec2/faqs/#Nitro_Hypervisor">Nitro
         *        Hypervisor</a> section of the Amazon EC2 frequently asked questions. For more information, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#ec2-nitro-instances">
         *        Instances built on the Nitro System</a> in the <i>Amazon EC2 User Guide for Linux</i>, or <a href=
         *        "https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/instance-types.html#ec2-nitro-instances"
         *        >Instances built on the Nitro System</a> in the <i>Amazon EC2 User Guide for Windows</i>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>NetworkInterface</code> </b> — The network interface of the recommended instance type is
         *        different than that of the current instance. For example, the recommended instance type supports
         *        enhanced networking and the current instance might not. To enable enhanced networking for the
         *        recommended instance type, you must install the Elastic Network Adapter (ENA) driver or the Intel
         *        82599 Virtual Function driver. For more information, see <a href=
         *        "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#instance-networking-storage"
         *        >Networking and storage features</a> and <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/enhanced-networking.html">Enhanced
         *        networking on Linux</a> in the <i>Amazon EC2 User Guide for Linux</i>, or <a href=
         *        "https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/instance-types.html#instance-networking-storage"
         *        >Networking and storage features</a> and <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/enhanced-networking.html">Enhanced
         *        networking on Windows</a> in the <i>Amazon EC2 User Guide for Windows</i>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>StorageInterface</code> </b> — The storage interface of the recommended instance type is
         *        different than that of the current instance. For example, the recommended instance type uses an NVMe
         *        storage interface and the current instance does not. To access NVMe volumes for the recommended
         *        instance type, you will need to install or upgrade the NVMe driver. For more information, see <a href=
         *        "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#instance-networking-storage"
         *        >Networking and storage features</a> and <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/nvme-ebs-volumes.html">Amazon EBS and NVMe
         *        on Linux instances</a> in the <i>Amazon EC2 User Guide for Linux</i>, or <a href=
         *        "https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/instance-types.html#instance-networking-storage"
         *        >Networking and storage features</a> and <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/nvme-ebs-volumes.html">Amazon EBS and
         *        NVMe on Windows instances</a> in the <i>Amazon EC2 User Guide for Windows</i>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>InstanceStoreAvailability</code> </b> — The recommended instance type does not support
         *        instance store volumes and the current instance does. Before migrating, you might need to back up the
         *        data on your instance store volumes if you want to preserve them. For more information, see <a
         *        href="https://aws.amazon.com/premiumsupport/knowledge-center/back-up-instance-store-ebs/">How do I
         *        back up an instance store volume on my Amazon EC2 instance to Amazon EBS?</a> in the <i>Amazon Web
         *        Services Premium Support Knowledge Base</i>. For more information, see <a href=
         *        "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#instance-networking-storage"
         *        >Networking and storage features</a> and <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/InstanceStorage.html">Amazon EC2 instance
         *        store</a> in the <i>Amazon EC2 User Guide for Linux</i>, or see <a href=
         *        "https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/instance-types.html#instance-networking-storage"
         *        >Networking and storage features</a> and <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/InstanceStorage.html">Amazon EC2 instance
         *        store</a> in the <i>Amazon EC2 User Guide for Windows</i>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>VirtualizationType</code> </b> — The recommended instance type uses the hardware virtual
         *        machine (HVM) virtualization type and the current instance uses the paravirtual (PV) virtualization
         *        type. For more information about the differences between these virtualization types, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/virtualization_types.html">Linux AMI
         *        virtualization types</a> in the <i>Amazon EC2 User Guide for Linux</i>, or <a href=
         *        "https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/windows-ami-version-history.html#virtualization-types"
         *        >Windows AMI virtualization types</a> in the <i>Amazon EC2 User Guide for Windows</i>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>Architecture</code> </b> — The CPU architecture between the recommended instance type and
         *        the current instance is different. For example, the recommended instance type might use an Arm CPU
         *        architecture and the current instance type might use a different one, such as x86. Before migrating,
         *        you should consider recompiling the software on your instance for the new architecture. Alternatively,
         *        you might switch to an Amazon Machine Image (AMI) that supports the new architecture. For more
         *        information about the CPU architecture for each instance type, see <a
         *        href="http://aws.amazon.com/ec2/instance-types/">Amazon EC2 Instance Types</a>.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder platformDifferencesWithStrings(String... platformDifferences);

        /**
         * <p>
         * Describes the configuration differences between the current instance and the recommended instance type. You
         * should consider the configuration differences before migrating your workloads from the current instance to
         * the recommended instance type. The <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-instance-resize.html">Change the instance type
         * guide for Linux</a> and <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/ec2-instance-resize.html">Change the instance
         * type guide for Windows</a> provide general guidance for getting started with an instance migration.
         * </p>
         * <p>
         * Platform differences include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b> <code>Hypervisor</code> </b> — The hypervisor of the recommended instance type is different than that of
         * the current instance. For example, the recommended instance type uses a Nitro hypervisor and the current
         * instance uses a Xen hypervisor. The differences that you should consider between these hypervisors are
         * covered in the <a href="http://aws.amazon.com/ec2/faqs/#Nitro_Hypervisor">Nitro Hypervisor</a> section of the
         * Amazon EC2 frequently asked questions. For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#ec2-nitro-instances">Instances
         * built on the Nitro System</a> in the <i>Amazon EC2 User Guide for Linux</i>, or <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/instance-types.html#ec2-nitro-instances"
         * >Instances built on the Nitro System</a> in the <i>Amazon EC2 User Guide for Windows</i>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>NetworkInterface</code> </b> — The network interface of the recommended instance type is different
         * than that of the current instance. For example, the recommended instance type supports enhanced networking
         * and the current instance might not. To enable enhanced networking for the recommended instance type, you must
         * install the Elastic Network Adapter (ENA) driver or the Intel 82599 Virtual Function driver. For more
         * information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#instance-networking-storage"
         * >Networking and storage features</a> and <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/enhanced-networking.html">Enhanced networking on
         * Linux</a> in the <i>Amazon EC2 User Guide for Linux</i>, or <a href=
         * "https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/instance-types.html#instance-networking-storage"
         * >Networking and storage features</a> and <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/enhanced-networking.html">Enhanced networking on
         * Windows</a> in the <i>Amazon EC2 User Guide for Windows</i>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>StorageInterface</code> </b> — The storage interface of the recommended instance type is different
         * than that of the current instance. For example, the recommended instance type uses an NVMe storage interface
         * and the current instance does not. To access NVMe volumes for the recommended instance type, you will need to
         * install or upgrade the NVMe driver. For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#instance-networking-storage"
         * >Networking and storage features</a> and <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/nvme-ebs-volumes.html">Amazon EBS and NVMe on Linux
         * instances</a> in the <i>Amazon EC2 User Guide for Linux</i>, or <a href=
         * "https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/instance-types.html#instance-networking-storage"
         * >Networking and storage features</a> and <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/nvme-ebs-volumes.html">Amazon EBS and NVMe on
         * Windows instances</a> in the <i>Amazon EC2 User Guide for Windows</i>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>InstanceStoreAvailability</code> </b> — The recommended instance type does not support instance
         * store volumes and the current instance does. Before migrating, you might need to back up the data on your
         * instance store volumes if you want to preserve them. For more information, see <a
         * href="https://aws.amazon.com/premiumsupport/knowledge-center/back-up-instance-store-ebs/">How do I back up an
         * instance store volume on my Amazon EC2 instance to Amazon EBS?</a> in the <i>Amazon Web Services Premium
         * Support Knowledge Base</i>. For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#instance-networking-storage"
         * >Networking and storage features</a> and <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/InstanceStorage.html">Amazon EC2 instance store</a>
         * in the <i>Amazon EC2 User Guide for Linux</i>, or see <a href=
         * "https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/instance-types.html#instance-networking-storage"
         * >Networking and storage features</a> and <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/InstanceStorage.html">Amazon EC2 instance
         * store</a> in the <i>Amazon EC2 User Guide for Windows</i>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>VirtualizationType</code> </b> — The recommended instance type uses the hardware virtual machine
         * (HVM) virtualization type and the current instance uses the paravirtual (PV) virtualization type. For more
         * information about the differences between these virtualization types, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/virtualization_types.html">Linux AMI virtualization
         * types</a> in the <i>Amazon EC2 User Guide for Linux</i>, or <a href=
         * "https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/windows-ami-version-history.html#virtualization-types"
         * >Windows AMI virtualization types</a> in the <i>Amazon EC2 User Guide for Windows</i>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>Architecture</code> </b> — The CPU architecture between the recommended instance type and the
         * current instance is different. For example, the recommended instance type might use an Arm CPU architecture
         * and the current instance type might use a different one, such as x86. Before migrating, you should consider
         * recompiling the software on your instance for the new architecture. Alternatively, you might switch to an
         * Amazon Machine Image (AMI) that supports the new architecture. For more information about the CPU
         * architecture for each instance type, see <a href="http://aws.amazon.com/ec2/instance-types/">Amazon EC2
         * Instance Types</a>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param platformDifferences
         *        Describes the configuration differences between the current instance and the recommended instance
         *        type. You should consider the configuration differences before migrating your workloads from the
         *        current instance to the recommended instance type. The <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-instance-resize.html">Change the
         *        instance type guide for Linux</a> and <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/ec2-instance-resize.html">Change the
         *        instance type guide for Windows</a> provide general guidance for getting started with an instance
         *        migration.</p>
         *        <p>
         *        Platform differences include:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b> <code>Hypervisor</code> </b> — The hypervisor of the recommended instance type is different than
         *        that of the current instance. For example, the recommended instance type uses a Nitro hypervisor and
         *        the current instance uses a Xen hypervisor. The differences that you should consider between these
         *        hypervisors are covered in the <a href="http://aws.amazon.com/ec2/faqs/#Nitro_Hypervisor">Nitro
         *        Hypervisor</a> section of the Amazon EC2 frequently asked questions. For more information, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#ec2-nitro-instances">
         *        Instances built on the Nitro System</a> in the <i>Amazon EC2 User Guide for Linux</i>, or <a href=
         *        "https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/instance-types.html#ec2-nitro-instances"
         *        >Instances built on the Nitro System</a> in the <i>Amazon EC2 User Guide for Windows</i>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>NetworkInterface</code> </b> — The network interface of the recommended instance type is
         *        different than that of the current instance. For example, the recommended instance type supports
         *        enhanced networking and the current instance might not. To enable enhanced networking for the
         *        recommended instance type, you must install the Elastic Network Adapter (ENA) driver or the Intel
         *        82599 Virtual Function driver. For more information, see <a href=
         *        "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#instance-networking-storage"
         *        >Networking and storage features</a> and <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/enhanced-networking.html">Enhanced
         *        networking on Linux</a> in the <i>Amazon EC2 User Guide for Linux</i>, or <a href=
         *        "https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/instance-types.html#instance-networking-storage"
         *        >Networking and storage features</a> and <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/enhanced-networking.html">Enhanced
         *        networking on Windows</a> in the <i>Amazon EC2 User Guide for Windows</i>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>StorageInterface</code> </b> — The storage interface of the recommended instance type is
         *        different than that of the current instance. For example, the recommended instance type uses an NVMe
         *        storage interface and the current instance does not. To access NVMe volumes for the recommended
         *        instance type, you will need to install or upgrade the NVMe driver. For more information, see <a href=
         *        "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#instance-networking-storage"
         *        >Networking and storage features</a> and <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/nvme-ebs-volumes.html">Amazon EBS and NVMe
         *        on Linux instances</a> in the <i>Amazon EC2 User Guide for Linux</i>, or <a href=
         *        "https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/instance-types.html#instance-networking-storage"
         *        >Networking and storage features</a> and <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/nvme-ebs-volumes.html">Amazon EBS and
         *        NVMe on Windows instances</a> in the <i>Amazon EC2 User Guide for Windows</i>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>InstanceStoreAvailability</code> </b> — The recommended instance type does not support
         *        instance store volumes and the current instance does. Before migrating, you might need to back up the
         *        data on your instance store volumes if you want to preserve them. For more information, see <a
         *        href="https://aws.amazon.com/premiumsupport/knowledge-center/back-up-instance-store-ebs/">How do I
         *        back up an instance store volume on my Amazon EC2 instance to Amazon EBS?</a> in the <i>Amazon Web
         *        Services Premium Support Knowledge Base</i>. For more information, see <a href=
         *        "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#instance-networking-storage"
         *        >Networking and storage features</a> and <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/InstanceStorage.html">Amazon EC2 instance
         *        store</a> in the <i>Amazon EC2 User Guide for Linux</i>, or see <a href=
         *        "https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/instance-types.html#instance-networking-storage"
         *        >Networking and storage features</a> and <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/InstanceStorage.html">Amazon EC2 instance
         *        store</a> in the <i>Amazon EC2 User Guide for Windows</i>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>VirtualizationType</code> </b> — The recommended instance type uses the hardware virtual
         *        machine (HVM) virtualization type and the current instance uses the paravirtual (PV) virtualization
         *        type. For more information about the differences between these virtualization types, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/virtualization_types.html">Linux AMI
         *        virtualization types</a> in the <i>Amazon EC2 User Guide for Linux</i>, or <a href=
         *        "https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/windows-ami-version-history.html#virtualization-types"
         *        >Windows AMI virtualization types</a> in the <i>Amazon EC2 User Guide for Windows</i>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>Architecture</code> </b> — The CPU architecture between the recommended instance type and
         *        the current instance is different. For example, the recommended instance type might use an Arm CPU
         *        architecture and the current instance type might use a different one, such as x86. Before migrating,
         *        you should consider recompiling the software on your instance for the new architecture. Alternatively,
         *        you might switch to an Amazon Machine Image (AMI) that supports the new architecture. For more
         *        information about the CPU architecture for each instance type, see <a
         *        href="http://aws.amazon.com/ec2/instance-types/">Amazon EC2 Instance Types</a>.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder platformDifferences(Collection<PlatformDifference> platformDifferences);

        /**
         * <p>
         * Describes the configuration differences between the current instance and the recommended instance type. You
         * should consider the configuration differences before migrating your workloads from the current instance to
         * the recommended instance type. The <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-instance-resize.html">Change the instance type
         * guide for Linux</a> and <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/ec2-instance-resize.html">Change the instance
         * type guide for Windows</a> provide general guidance for getting started with an instance migration.
         * </p>
         * <p>
         * Platform differences include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b> <code>Hypervisor</code> </b> — The hypervisor of the recommended instance type is different than that of
         * the current instance. For example, the recommended instance type uses a Nitro hypervisor and the current
         * instance uses a Xen hypervisor. The differences that you should consider between these hypervisors are
         * covered in the <a href="http://aws.amazon.com/ec2/faqs/#Nitro_Hypervisor">Nitro Hypervisor</a> section of the
         * Amazon EC2 frequently asked questions. For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#ec2-nitro-instances">Instances
         * built on the Nitro System</a> in the <i>Amazon EC2 User Guide for Linux</i>, or <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/instance-types.html#ec2-nitro-instances"
         * >Instances built on the Nitro System</a> in the <i>Amazon EC2 User Guide for Windows</i>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>NetworkInterface</code> </b> — The network interface of the recommended instance type is different
         * than that of the current instance. For example, the recommended instance type supports enhanced networking
         * and the current instance might not. To enable enhanced networking for the recommended instance type, you must
         * install the Elastic Network Adapter (ENA) driver or the Intel 82599 Virtual Function driver. For more
         * information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#instance-networking-storage"
         * >Networking and storage features</a> and <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/enhanced-networking.html">Enhanced networking on
         * Linux</a> in the <i>Amazon EC2 User Guide for Linux</i>, or <a href=
         * "https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/instance-types.html#instance-networking-storage"
         * >Networking and storage features</a> and <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/enhanced-networking.html">Enhanced networking on
         * Windows</a> in the <i>Amazon EC2 User Guide for Windows</i>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>StorageInterface</code> </b> — The storage interface of the recommended instance type is different
         * than that of the current instance. For example, the recommended instance type uses an NVMe storage interface
         * and the current instance does not. To access NVMe volumes for the recommended instance type, you will need to
         * install or upgrade the NVMe driver. For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#instance-networking-storage"
         * >Networking and storage features</a> and <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/nvme-ebs-volumes.html">Amazon EBS and NVMe on Linux
         * instances</a> in the <i>Amazon EC2 User Guide for Linux</i>, or <a href=
         * "https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/instance-types.html#instance-networking-storage"
         * >Networking and storage features</a> and <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/nvme-ebs-volumes.html">Amazon EBS and NVMe on
         * Windows instances</a> in the <i>Amazon EC2 User Guide for Windows</i>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>InstanceStoreAvailability</code> </b> — The recommended instance type does not support instance
         * store volumes and the current instance does. Before migrating, you might need to back up the data on your
         * instance store volumes if you want to preserve them. For more information, see <a
         * href="https://aws.amazon.com/premiumsupport/knowledge-center/back-up-instance-store-ebs/">How do I back up an
         * instance store volume on my Amazon EC2 instance to Amazon EBS?</a> in the <i>Amazon Web Services Premium
         * Support Knowledge Base</i>. For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#instance-networking-storage"
         * >Networking and storage features</a> and <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/InstanceStorage.html">Amazon EC2 instance store</a>
         * in the <i>Amazon EC2 User Guide for Linux</i>, or see <a href=
         * "https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/instance-types.html#instance-networking-storage"
         * >Networking and storage features</a> and <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/InstanceStorage.html">Amazon EC2 instance
         * store</a> in the <i>Amazon EC2 User Guide for Windows</i>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>VirtualizationType</code> </b> — The recommended instance type uses the hardware virtual machine
         * (HVM) virtualization type and the current instance uses the paravirtual (PV) virtualization type. For more
         * information about the differences between these virtualization types, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/virtualization_types.html">Linux AMI virtualization
         * types</a> in the <i>Amazon EC2 User Guide for Linux</i>, or <a href=
         * "https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/windows-ami-version-history.html#virtualization-types"
         * >Windows AMI virtualization types</a> in the <i>Amazon EC2 User Guide for Windows</i>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>Architecture</code> </b> — The CPU architecture between the recommended instance type and the
         * current instance is different. For example, the recommended instance type might use an Arm CPU architecture
         * and the current instance type might use a different one, such as x86. Before migrating, you should consider
         * recompiling the software on your instance for the new architecture. Alternatively, you might switch to an
         * Amazon Machine Image (AMI) that supports the new architecture. For more information about the CPU
         * architecture for each instance type, see <a href="http://aws.amazon.com/ec2/instance-types/">Amazon EC2
         * Instance Types</a>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param platformDifferences
         *        Describes the configuration differences between the current instance and the recommended instance
         *        type. You should consider the configuration differences before migrating your workloads from the
         *        current instance to the recommended instance type. The <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-instance-resize.html">Change the
         *        instance type guide for Linux</a> and <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/ec2-instance-resize.html">Change the
         *        instance type guide for Windows</a> provide general guidance for getting started with an instance
         *        migration.</p>
         *        <p>
         *        Platform differences include:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b> <code>Hypervisor</code> </b> — The hypervisor of the recommended instance type is different than
         *        that of the current instance. For example, the recommended instance type uses a Nitro hypervisor and
         *        the current instance uses a Xen hypervisor. The differences that you should consider between these
         *        hypervisors are covered in the <a href="http://aws.amazon.com/ec2/faqs/#Nitro_Hypervisor">Nitro
         *        Hypervisor</a> section of the Amazon EC2 frequently asked questions. For more information, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#ec2-nitro-instances">
         *        Instances built on the Nitro System</a> in the <i>Amazon EC2 User Guide for Linux</i>, or <a href=
         *        "https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/instance-types.html#ec2-nitro-instances"
         *        >Instances built on the Nitro System</a> in the <i>Amazon EC2 User Guide for Windows</i>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>NetworkInterface</code> </b> — The network interface of the recommended instance type is
         *        different than that of the current instance. For example, the recommended instance type supports
         *        enhanced networking and the current instance might not. To enable enhanced networking for the
         *        recommended instance type, you must install the Elastic Network Adapter (ENA) driver or the Intel
         *        82599 Virtual Function driver. For more information, see <a href=
         *        "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#instance-networking-storage"
         *        >Networking and storage features</a> and <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/enhanced-networking.html">Enhanced
         *        networking on Linux</a> in the <i>Amazon EC2 User Guide for Linux</i>, or <a href=
         *        "https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/instance-types.html#instance-networking-storage"
         *        >Networking and storage features</a> and <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/enhanced-networking.html">Enhanced
         *        networking on Windows</a> in the <i>Amazon EC2 User Guide for Windows</i>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>StorageInterface</code> </b> — The storage interface of the recommended instance type is
         *        different than that of the current instance. For example, the recommended instance type uses an NVMe
         *        storage interface and the current instance does not. To access NVMe volumes for the recommended
         *        instance type, you will need to install or upgrade the NVMe driver. For more information, see <a href=
         *        "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#instance-networking-storage"
         *        >Networking and storage features</a> and <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/nvme-ebs-volumes.html">Amazon EBS and NVMe
         *        on Linux instances</a> in the <i>Amazon EC2 User Guide for Linux</i>, or <a href=
         *        "https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/instance-types.html#instance-networking-storage"
         *        >Networking and storage features</a> and <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/nvme-ebs-volumes.html">Amazon EBS and
         *        NVMe on Windows instances</a> in the <i>Amazon EC2 User Guide for Windows</i>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>InstanceStoreAvailability</code> </b> — The recommended instance type does not support
         *        instance store volumes and the current instance does. Before migrating, you might need to back up the
         *        data on your instance store volumes if you want to preserve them. For more information, see <a
         *        href="https://aws.amazon.com/premiumsupport/knowledge-center/back-up-instance-store-ebs/">How do I
         *        back up an instance store volume on my Amazon EC2 instance to Amazon EBS?</a> in the <i>Amazon Web
         *        Services Premium Support Knowledge Base</i>. For more information, see <a href=
         *        "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#instance-networking-storage"
         *        >Networking and storage features</a> and <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/InstanceStorage.html">Amazon EC2 instance
         *        store</a> in the <i>Amazon EC2 User Guide for Linux</i>, or see <a href=
         *        "https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/instance-types.html#instance-networking-storage"
         *        >Networking and storage features</a> and <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/InstanceStorage.html">Amazon EC2 instance
         *        store</a> in the <i>Amazon EC2 User Guide for Windows</i>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>VirtualizationType</code> </b> — The recommended instance type uses the hardware virtual
         *        machine (HVM) virtualization type and the current instance uses the paravirtual (PV) virtualization
         *        type. For more information about the differences between these virtualization types, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/virtualization_types.html">Linux AMI
         *        virtualization types</a> in the <i>Amazon EC2 User Guide for Linux</i>, or <a href=
         *        "https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/windows-ami-version-history.html#virtualization-types"
         *        >Windows AMI virtualization types</a> in the <i>Amazon EC2 User Guide for Windows</i>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>Architecture</code> </b> — The CPU architecture between the recommended instance type and
         *        the current instance is different. For example, the recommended instance type might use an Arm CPU
         *        architecture and the current instance type might use a different one, such as x86. Before migrating,
         *        you should consider recompiling the software on your instance for the new architecture. Alternatively,
         *        you might switch to an Amazon Machine Image (AMI) that supports the new architecture. For more
         *        information about the CPU architecture for each instance type, see <a
         *        href="http://aws.amazon.com/ec2/instance-types/">Amazon EC2 Instance Types</a>.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder platformDifferences(PlatformDifference... platformDifferences);

        /**
         * <p>
         * The performance risk of the instance recommendation option.
         * </p>
         * <p>
         * Performance risk indicates the likelihood of the recommended instance type not meeting the resource needs of
         * your workload. Compute Optimizer calculates an individual performance risk score for each specification of
         * the recommended instance, including CPU, memory, EBS throughput, EBS IOPS, disk throughput, disk IOPS,
         * network throughput, and network PPS. The performance risk of the recommended instance is calculated as the
         * maximum performance risk score across the analyzed resource specifications.
         * </p>
         * <p>
         * The value ranges from <code>0</code> - <code>4</code>, with <code>0</code> meaning that the recommended
         * resource is predicted to always provide enough hardware capability. The higher the performance risk is, the
         * more likely you should validate whether the recommendation will meet the performance requirements of your
         * workload before migrating your resource.
         * </p>
         * 
         * @param performanceRisk
         *        The performance risk of the instance recommendation option.</p>
         *        <p>
         *        Performance risk indicates the likelihood of the recommended instance type not meeting the resource
         *        needs of your workload. Compute Optimizer calculates an individual performance risk score for each
         *        specification of the recommended instance, including CPU, memory, EBS throughput, EBS IOPS, disk
         *        throughput, disk IOPS, network throughput, and network PPS. The performance risk of the recommended
         *        instance is calculated as the maximum performance risk score across the analyzed resource
         *        specifications.
         *        </p>
         *        <p>
         *        The value ranges from <code>0</code> - <code>4</code>, with <code>0</code> meaning that the
         *        recommended resource is predicted to always provide enough hardware capability. The higher the
         *        performance risk is, the more likely you should validate whether the recommendation will meet the
         *        performance requirements of your workload before migrating your resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder performanceRisk(Double performanceRisk);

        /**
         * <p>
         * The rank of the instance recommendation option.
         * </p>
         * <p>
         * The top recommendation option is ranked as <code>1</code>.
         * </p>
         * 
         * @param rank
         *        The rank of the instance recommendation option.</p>
         *        <p>
         *        The top recommendation option is ranked as <code>1</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder rank(Integer rank);

        /**
         * <p>
         * An object that describes the savings opportunity for the instance recommendation option. Savings opportunity
         * includes the estimated monthly savings amount and percentage.
         * </p>
         * 
         * @param savingsOpportunity
         *        An object that describes the savings opportunity for the instance recommendation option. Savings
         *        opportunity includes the estimated monthly savings amount and percentage.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder savingsOpportunity(SavingsOpportunity savingsOpportunity);

        /**
         * <p>
         * An object that describes the savings opportunity for the instance recommendation option. Savings opportunity
         * includes the estimated monthly savings amount and percentage.
         * </p>
         * This is a convenience method that creates an instance of the {@link SavingsOpportunity.Builder} avoiding the
         * need to create one manually via {@link SavingsOpportunity#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link SavingsOpportunity.Builder#build()} is called immediately and its
         * result is passed to {@link #savingsOpportunity(SavingsOpportunity)}.
         * 
         * @param savingsOpportunity
         *        a consumer that will call methods on {@link SavingsOpportunity.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #savingsOpportunity(SavingsOpportunity)
         */
        default Builder savingsOpportunity(Consumer<SavingsOpportunity.Builder> savingsOpportunity) {
            return savingsOpportunity(SavingsOpportunity.builder().applyMutation(savingsOpportunity).build());
        }

        /**
         * <p>
         * The level of effort required to migrate from the current instance type to the recommended instance type.
         * </p>
         * <p>
         * For example, the migration effort is <code>Low</code> if Amazon EMR is the inferred workload type and an
         * Amazon Web Services Graviton instance type is recommended. The migration effort is <code>Medium</code> if a
         * workload type couldn't be inferred but an Amazon Web Services Graviton instance type is recommended. The
         * migration effort is <code>VeryLow</code> if both the current and recommended instance types are of the same
         * CPU architecture.
         * </p>
         * 
         * @param migrationEffort
         *        The level of effort required to migrate from the current instance type to the recommended instance
         *        type.</p>
         *        <p>
         *        For example, the migration effort is <code>Low</code> if Amazon EMR is the inferred workload type and
         *        an Amazon Web Services Graviton instance type is recommended. The migration effort is
         *        <code>Medium</code> if a workload type couldn't be inferred but an Amazon Web Services Graviton
         *        instance type is recommended. The migration effort is <code>VeryLow</code> if both the current and
         *        recommended instance types are of the same CPU architecture.
         * @see MigrationEffort
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MigrationEffort
         */
        Builder migrationEffort(String migrationEffort);

        /**
         * <p>
         * The level of effort required to migrate from the current instance type to the recommended instance type.
         * </p>
         * <p>
         * For example, the migration effort is <code>Low</code> if Amazon EMR is the inferred workload type and an
         * Amazon Web Services Graviton instance type is recommended. The migration effort is <code>Medium</code> if a
         * workload type couldn't be inferred but an Amazon Web Services Graviton instance type is recommended. The
         * migration effort is <code>VeryLow</code> if both the current and recommended instance types are of the same
         * CPU architecture.
         * </p>
         * 
         * @param migrationEffort
         *        The level of effort required to migrate from the current instance type to the recommended instance
         *        type.</p>
         *        <p>
         *        For example, the migration effort is <code>Low</code> if Amazon EMR is the inferred workload type and
         *        an Amazon Web Services Graviton instance type is recommended. The migration effort is
         *        <code>Medium</code> if a workload type couldn't be inferred but an Amazon Web Services Graviton
         *        instance type is recommended. The migration effort is <code>VeryLow</code> if both the current and
         *        recommended instance types are of the same CPU architecture.
         * @see MigrationEffort
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MigrationEffort
         */
        Builder migrationEffort(MigrationEffort migrationEffort);

        /**
         * <p>
         * Describes the GPU accelerator settings for the recommended instance type.
         * </p>
         * 
         * @param instanceGpuInfo
         *        Describes the GPU accelerator settings for the recommended instance type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceGpuInfo(GpuInfo instanceGpuInfo);

        /**
         * <p>
         * Describes the GPU accelerator settings for the recommended instance type.
         * </p>
         * This is a convenience method that creates an instance of the {@link GpuInfo.Builder} avoiding the need to
         * create one manually via {@link GpuInfo#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link GpuInfo.Builder#build()} is called immediately and its result is
         * passed to {@link #instanceGpuInfo(GpuInfo)}.
         * 
         * @param instanceGpuInfo
         *        a consumer that will call methods on {@link GpuInfo.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #instanceGpuInfo(GpuInfo)
         */
        default Builder instanceGpuInfo(Consumer<GpuInfo.Builder> instanceGpuInfo) {
            return instanceGpuInfo(GpuInfo.builder().applyMutation(instanceGpuInfo).build());
        }

        /**
         * <p>
         * An object that describes the savings opportunity for the instance recommendation option that includes Savings
         * Plans and Reserved Instances discounts. Savings opportunity includes the estimated monthly savings and
         * percentage.
         * </p>
         * 
         * @param savingsOpportunityAfterDiscounts
         *        An object that describes the savings opportunity for the instance recommendation option that includes
         *        Savings Plans and Reserved Instances discounts. Savings opportunity includes the estimated monthly
         *        savings and percentage.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder savingsOpportunityAfterDiscounts(InstanceSavingsOpportunityAfterDiscounts savingsOpportunityAfterDiscounts);

        /**
         * <p>
         * An object that describes the savings opportunity for the instance recommendation option that includes Savings
         * Plans and Reserved Instances discounts. Savings opportunity includes the estimated monthly savings and
         * percentage.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link InstanceSavingsOpportunityAfterDiscounts.Builder} avoiding the need to create one manually via
         * {@link InstanceSavingsOpportunityAfterDiscounts#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link InstanceSavingsOpportunityAfterDiscounts.Builder#build()} is
         * called immediately and its result is passed to
         * {@link #savingsOpportunityAfterDiscounts(InstanceSavingsOpportunityAfterDiscounts)}.
         * 
         * @param savingsOpportunityAfterDiscounts
         *        a consumer that will call methods on {@link InstanceSavingsOpportunityAfterDiscounts.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #savingsOpportunityAfterDiscounts(InstanceSavingsOpportunityAfterDiscounts)
         */
        default Builder savingsOpportunityAfterDiscounts(
                Consumer<InstanceSavingsOpportunityAfterDiscounts.Builder> savingsOpportunityAfterDiscounts) {
            return savingsOpportunityAfterDiscounts(InstanceSavingsOpportunityAfterDiscounts.builder()
                    .applyMutation(savingsOpportunityAfterDiscounts).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String instanceType;

        private List<UtilizationMetric> projectedUtilizationMetrics = DefaultSdkAutoConstructList.getInstance();

        private List<String> platformDifferences = DefaultSdkAutoConstructList.getInstance();

        private Double performanceRisk;

        private Integer rank;

        private SavingsOpportunity savingsOpportunity;

        private String migrationEffort;

        private GpuInfo instanceGpuInfo;

        private InstanceSavingsOpportunityAfterDiscounts savingsOpportunityAfterDiscounts;

        private BuilderImpl() {
        }

        private BuilderImpl(InstanceRecommendationOption model) {
            instanceType(model.instanceType);
            projectedUtilizationMetrics(model.projectedUtilizationMetrics);
            platformDifferencesWithStrings(model.platformDifferences);
            performanceRisk(model.performanceRisk);
            rank(model.rank);
            savingsOpportunity(model.savingsOpportunity);
            migrationEffort(model.migrationEffort);
            instanceGpuInfo(model.instanceGpuInfo);
            savingsOpportunityAfterDiscounts(model.savingsOpportunityAfterDiscounts);
        }

        public final String getInstanceType() {
            return instanceType;
        }

        public final void setInstanceType(String instanceType) {
            this.instanceType = instanceType;
        }

        @Override
        public final Builder instanceType(String instanceType) {
            this.instanceType = instanceType;
            return this;
        }

        public final List<UtilizationMetric.Builder> getProjectedUtilizationMetrics() {
            List<UtilizationMetric.Builder> result = ProjectedUtilizationMetricsCopier
                    .copyToBuilder(this.projectedUtilizationMetrics);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setProjectedUtilizationMetrics(Collection<UtilizationMetric.BuilderImpl> projectedUtilizationMetrics) {
            this.projectedUtilizationMetrics = ProjectedUtilizationMetricsCopier.copyFromBuilder(projectedUtilizationMetrics);
        }

        @Override
        public final Builder projectedUtilizationMetrics(Collection<UtilizationMetric> projectedUtilizationMetrics) {
            this.projectedUtilizationMetrics = ProjectedUtilizationMetricsCopier.copy(projectedUtilizationMetrics);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder projectedUtilizationMetrics(UtilizationMetric... projectedUtilizationMetrics) {
            projectedUtilizationMetrics(Arrays.asList(projectedUtilizationMetrics));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder projectedUtilizationMetrics(Consumer<UtilizationMetric.Builder>... projectedUtilizationMetrics) {
            projectedUtilizationMetrics(Stream.of(projectedUtilizationMetrics)
                    .map(c -> UtilizationMetric.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final Collection<String> getPlatformDifferences() {
            if (platformDifferences instanceof SdkAutoConstructList) {
                return null;
            }
            return platformDifferences;
        }

        public final void setPlatformDifferences(Collection<String> platformDifferences) {
            this.platformDifferences = PlatformDifferencesCopier.copy(platformDifferences);
        }

        @Override
        public final Builder platformDifferencesWithStrings(Collection<String> platformDifferences) {
            this.platformDifferences = PlatformDifferencesCopier.copy(platformDifferences);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder platformDifferencesWithStrings(String... platformDifferences) {
            platformDifferencesWithStrings(Arrays.asList(platformDifferences));
            return this;
        }

        @Override
        public final Builder platformDifferences(Collection<PlatformDifference> platformDifferences) {
            this.platformDifferences = PlatformDifferencesCopier.copyEnumToString(platformDifferences);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder platformDifferences(PlatformDifference... platformDifferences) {
            platformDifferences(Arrays.asList(platformDifferences));
            return this;
        }

        public final Double getPerformanceRisk() {
            return performanceRisk;
        }

        public final void setPerformanceRisk(Double performanceRisk) {
            this.performanceRisk = performanceRisk;
        }

        @Override
        public final Builder performanceRisk(Double performanceRisk) {
            this.performanceRisk = performanceRisk;
            return this;
        }

        public final Integer getRank() {
            return rank;
        }

        public final void setRank(Integer rank) {
            this.rank = rank;
        }

        @Override
        public final Builder rank(Integer rank) {
            this.rank = rank;
            return this;
        }

        public final SavingsOpportunity.Builder getSavingsOpportunity() {
            return savingsOpportunity != null ? savingsOpportunity.toBuilder() : null;
        }

        public final void setSavingsOpportunity(SavingsOpportunity.BuilderImpl savingsOpportunity) {
            this.savingsOpportunity = savingsOpportunity != null ? savingsOpportunity.build() : null;
        }

        @Override
        public final Builder savingsOpportunity(SavingsOpportunity savingsOpportunity) {
            this.savingsOpportunity = savingsOpportunity;
            return this;
        }

        public final String getMigrationEffort() {
            return migrationEffort;
        }

        public final void setMigrationEffort(String migrationEffort) {
            this.migrationEffort = migrationEffort;
        }

        @Override
        public final Builder migrationEffort(String migrationEffort) {
            this.migrationEffort = migrationEffort;
            return this;
        }

        @Override
        public final Builder migrationEffort(MigrationEffort migrationEffort) {
            this.migrationEffort(migrationEffort == null ? null : migrationEffort.toString());
            return this;
        }

        public final GpuInfo.Builder getInstanceGpuInfo() {
            return instanceGpuInfo != null ? instanceGpuInfo.toBuilder() : null;
        }

        public final void setInstanceGpuInfo(GpuInfo.BuilderImpl instanceGpuInfo) {
            this.instanceGpuInfo = instanceGpuInfo != null ? instanceGpuInfo.build() : null;
        }

        @Override
        public final Builder instanceGpuInfo(GpuInfo instanceGpuInfo) {
            this.instanceGpuInfo = instanceGpuInfo;
            return this;
        }

        public final InstanceSavingsOpportunityAfterDiscounts.Builder getSavingsOpportunityAfterDiscounts() {
            return savingsOpportunityAfterDiscounts != null ? savingsOpportunityAfterDiscounts.toBuilder() : null;
        }

        public final void setSavingsOpportunityAfterDiscounts(
                InstanceSavingsOpportunityAfterDiscounts.BuilderImpl savingsOpportunityAfterDiscounts) {
            this.savingsOpportunityAfterDiscounts = savingsOpportunityAfterDiscounts != null ? savingsOpportunityAfterDiscounts
                    .build() : null;
        }

        @Override
        public final Builder savingsOpportunityAfterDiscounts(
                InstanceSavingsOpportunityAfterDiscounts savingsOpportunityAfterDiscounts) {
            this.savingsOpportunityAfterDiscounts = savingsOpportunityAfterDiscounts;
            return this;
        }

        @Override
        public InstanceRecommendationOption build() {
            return new InstanceRecommendationOption(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
