/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.computeoptimizer.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a recommendation preference.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class RecommendationPreferencesDetail implements SdkPojo, Serializable,
        ToCopyableBuilder<RecommendationPreferencesDetail.Builder, RecommendationPreferencesDetail> {
    private static final SdkField<Scope> SCOPE_FIELD = SdkField.<Scope> builder(MarshallingType.SDK_POJO).memberName("scope")
            .getter(getter(RecommendationPreferencesDetail::scope)).setter(setter(Builder::scope)).constructor(Scope::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("scope").build()).build();

    private static final SdkField<String> RESOURCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("resourceType").getter(getter(RecommendationPreferencesDetail::resourceTypeAsString))
            .setter(setter(Builder::resourceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("resourceType").build()).build();

    private static final SdkField<String> ENHANCED_INFRASTRUCTURE_METRICS_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("enhancedInfrastructureMetrics")
            .getter(getter(RecommendationPreferencesDetail::enhancedInfrastructureMetricsAsString))
            .setter(setter(Builder::enhancedInfrastructureMetrics))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("enhancedInfrastructureMetrics")
                    .build()).build();

    private static final SdkField<String> INFERRED_WORKLOAD_TYPES_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("inferredWorkloadTypes").getter(getter(RecommendationPreferencesDetail::inferredWorkloadTypesAsString))
            .setter(setter(Builder::inferredWorkloadTypes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("inferredWorkloadTypes").build())
            .build();

    private static final SdkField<ExternalMetricsPreference> EXTERNAL_METRICS_PREFERENCE_FIELD = SdkField
            .<ExternalMetricsPreference> builder(MarshallingType.SDK_POJO).memberName("externalMetricsPreference")
            .getter(getter(RecommendationPreferencesDetail::externalMetricsPreference))
            .setter(setter(Builder::externalMetricsPreference)).constructor(ExternalMetricsPreference::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("externalMetricsPreference").build())
            .build();

    private static final SdkField<String> LOOK_BACK_PERIOD_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("lookBackPeriod").getter(getter(RecommendationPreferencesDetail::lookBackPeriodAsString))
            .setter(setter(Builder::lookBackPeriod))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lookBackPeriod").build()).build();

    private static final SdkField<List<UtilizationPreference>> UTILIZATION_PREFERENCES_FIELD = SdkField
            .<List<UtilizationPreference>> builder(MarshallingType.LIST)
            .memberName("utilizationPreferences")
            .getter(getter(RecommendationPreferencesDetail::utilizationPreferences))
            .setter(setter(Builder::utilizationPreferences))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("utilizationPreferences").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<UtilizationPreference> builder(MarshallingType.SDK_POJO)
                                            .constructor(UtilizationPreference::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<EffectivePreferredResource>> PREFERRED_RESOURCES_FIELD = SdkField
            .<List<EffectivePreferredResource>> builder(MarshallingType.LIST)
            .memberName("preferredResources")
            .getter(getter(RecommendationPreferencesDetail::preferredResources))
            .setter(setter(Builder::preferredResources))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("preferredResources").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<EffectivePreferredResource> builder(MarshallingType.SDK_POJO)
                                            .constructor(EffectivePreferredResource::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> SAVINGS_ESTIMATION_MODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("savingsEstimationMode").getter(getter(RecommendationPreferencesDetail::savingsEstimationModeAsString))
            .setter(setter(Builder::savingsEstimationMode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("savingsEstimationMode").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SCOPE_FIELD,
            RESOURCE_TYPE_FIELD, ENHANCED_INFRASTRUCTURE_METRICS_FIELD, INFERRED_WORKLOAD_TYPES_FIELD,
            EXTERNAL_METRICS_PREFERENCE_FIELD, LOOK_BACK_PERIOD_FIELD, UTILIZATION_PREFERENCES_FIELD, PREFERRED_RESOURCES_FIELD,
            SAVINGS_ESTIMATION_MODE_FIELD));

    private static final long serialVersionUID = 1L;

    private final Scope scope;

    private final String resourceType;

    private final String enhancedInfrastructureMetrics;

    private final String inferredWorkloadTypes;

    private final ExternalMetricsPreference externalMetricsPreference;

    private final String lookBackPeriod;

    private final List<UtilizationPreference> utilizationPreferences;

    private final List<EffectivePreferredResource> preferredResources;

    private final String savingsEstimationMode;

    private RecommendationPreferencesDetail(BuilderImpl builder) {
        this.scope = builder.scope;
        this.resourceType = builder.resourceType;
        this.enhancedInfrastructureMetrics = builder.enhancedInfrastructureMetrics;
        this.inferredWorkloadTypes = builder.inferredWorkloadTypes;
        this.externalMetricsPreference = builder.externalMetricsPreference;
        this.lookBackPeriod = builder.lookBackPeriod;
        this.utilizationPreferences = builder.utilizationPreferences;
        this.preferredResources = builder.preferredResources;
        this.savingsEstimationMode = builder.savingsEstimationMode;
    }

    /**
     * <p>
     * An object that describes the scope of the recommendation preference.
     * </p>
     * <p>
     * Recommendation preferences can be created at the organization level (for management accounts of an organization
     * only), account level, and resource level. For more information, see <a
     * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/enhanced-infrastructure-metrics.html">Activating
     * enhanced infrastructure metrics</a> in the <i>Compute Optimizer User Guide</i>.
     * </p>
     * 
     * @return An object that describes the scope of the recommendation preference.</p>
     *         <p>
     *         Recommendation preferences can be created at the organization level (for management accounts of an
     *         organization only), account level, and resource level. For more information, see <a
     *         href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/enhanced-infrastructure-metrics.html"
     *         >Activating enhanced infrastructure metrics</a> in the <i>Compute Optimizer User Guide</i>.
     */
    public final Scope scope() {
        return scope;
    }

    /**
     * <p>
     * The target resource type of the recommendation preference to create.
     * </p>
     * <p>
     * The <code>Ec2Instance</code> option encompasses standalone instances and instances that are part of Auto Scaling
     * groups. The <code>AutoScalingGroup</code> option encompasses only instances that are part of an Auto Scaling
     * group.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #resourceType} will
     * return {@link ResourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #resourceTypeAsString}.
     * </p>
     * 
     * @return The target resource type of the recommendation preference to create.</p>
     *         <p>
     *         The <code>Ec2Instance</code> option encompasses standalone instances and instances that are part of Auto
     *         Scaling groups. The <code>AutoScalingGroup</code> option encompasses only instances that are part of an
     *         Auto Scaling group.
     * @see ResourceType
     */
    public final ResourceType resourceType() {
        return ResourceType.fromValue(resourceType);
    }

    /**
     * <p>
     * The target resource type of the recommendation preference to create.
     * </p>
     * <p>
     * The <code>Ec2Instance</code> option encompasses standalone instances and instances that are part of Auto Scaling
     * groups. The <code>AutoScalingGroup</code> option encompasses only instances that are part of an Auto Scaling
     * group.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #resourceType} will
     * return {@link ResourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #resourceTypeAsString}.
     * </p>
     * 
     * @return The target resource type of the recommendation preference to create.</p>
     *         <p>
     *         The <code>Ec2Instance</code> option encompasses standalone instances and instances that are part of Auto
     *         Scaling groups. The <code>AutoScalingGroup</code> option encompasses only instances that are part of an
     *         Auto Scaling group.
     * @see ResourceType
     */
    public final String resourceTypeAsString() {
        return resourceType;
    }

    /**
     * <p>
     * The status of the enhanced infrastructure metrics recommendation preference.
     * </p>
     * <p>
     * When the recommendations page is refreshed, a status of <code>Active</code> confirms that the preference is
     * applied to the recommendations, and a status of <code>Inactive</code> confirms that the preference isn't yet
     * applied to recommendations.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/enhanced-infrastructure-metrics.html">Enhanced
     * infrastructure metrics</a> in the <i>Compute Optimizer User Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #enhancedInfrastructureMetrics} will return {@link EnhancedInfrastructureMetrics#UNKNOWN_TO_SDK_VERSION}.
     * The raw value returned by the service is available from {@link #enhancedInfrastructureMetricsAsString}.
     * </p>
     * 
     * @return The status of the enhanced infrastructure metrics recommendation preference.</p>
     *         <p>
     *         When the recommendations page is refreshed, a status of <code>Active</code> confirms that the preference
     *         is applied to the recommendations, and a status of <code>Inactive</code> confirms that the preference
     *         isn't yet applied to recommendations.
     *         </p>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/enhanced-infrastructure-metrics.html"
     *         >Enhanced infrastructure metrics</a> in the <i>Compute Optimizer User Guide</i>.
     * @see EnhancedInfrastructureMetrics
     */
    public final EnhancedInfrastructureMetrics enhancedInfrastructureMetrics() {
        return EnhancedInfrastructureMetrics.fromValue(enhancedInfrastructureMetrics);
    }

    /**
     * <p>
     * The status of the enhanced infrastructure metrics recommendation preference.
     * </p>
     * <p>
     * When the recommendations page is refreshed, a status of <code>Active</code> confirms that the preference is
     * applied to the recommendations, and a status of <code>Inactive</code> confirms that the preference isn't yet
     * applied to recommendations.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/enhanced-infrastructure-metrics.html">Enhanced
     * infrastructure metrics</a> in the <i>Compute Optimizer User Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #enhancedInfrastructureMetrics} will return {@link EnhancedInfrastructureMetrics#UNKNOWN_TO_SDK_VERSION}.
     * The raw value returned by the service is available from {@link #enhancedInfrastructureMetricsAsString}.
     * </p>
     * 
     * @return The status of the enhanced infrastructure metrics recommendation preference.</p>
     *         <p>
     *         When the recommendations page is refreshed, a status of <code>Active</code> confirms that the preference
     *         is applied to the recommendations, and a status of <code>Inactive</code> confirms that the preference
     *         isn't yet applied to recommendations.
     *         </p>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/enhanced-infrastructure-metrics.html"
     *         >Enhanced infrastructure metrics</a> in the <i>Compute Optimizer User Guide</i>.
     * @see EnhancedInfrastructureMetrics
     */
    public final String enhancedInfrastructureMetricsAsString() {
        return enhancedInfrastructureMetrics;
    }

    /**
     * <p>
     * The status of the inferred workload types recommendation preference.
     * </p>
     * <p>
     * When the recommendations page is refreshed, a status of <code>Active</code> confirms that the preference is
     * applied to the recommendations, and a status of <code>Inactive</code> confirms that the preference isn't yet
     * applied to recommendations.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #inferredWorkloadTypes} will return {@link InferredWorkloadTypesPreference#UNKNOWN_TO_SDK_VERSION}. The
     * raw value returned by the service is available from {@link #inferredWorkloadTypesAsString}.
     * </p>
     * 
     * @return The status of the inferred workload types recommendation preference.</p>
     *         <p>
     *         When the recommendations page is refreshed, a status of <code>Active</code> confirms that the preference
     *         is applied to the recommendations, and a status of <code>Inactive</code> confirms that the preference
     *         isn't yet applied to recommendations.
     * @see InferredWorkloadTypesPreference
     */
    public final InferredWorkloadTypesPreference inferredWorkloadTypes() {
        return InferredWorkloadTypesPreference.fromValue(inferredWorkloadTypes);
    }

    /**
     * <p>
     * The status of the inferred workload types recommendation preference.
     * </p>
     * <p>
     * When the recommendations page is refreshed, a status of <code>Active</code> confirms that the preference is
     * applied to the recommendations, and a status of <code>Inactive</code> confirms that the preference isn't yet
     * applied to recommendations.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #inferredWorkloadTypes} will return {@link InferredWorkloadTypesPreference#UNKNOWN_TO_SDK_VERSION}. The
     * raw value returned by the service is available from {@link #inferredWorkloadTypesAsString}.
     * </p>
     * 
     * @return The status of the inferred workload types recommendation preference.</p>
     *         <p>
     *         When the recommendations page is refreshed, a status of <code>Active</code> confirms that the preference
     *         is applied to the recommendations, and a status of <code>Inactive</code> confirms that the preference
     *         isn't yet applied to recommendations.
     * @see InferredWorkloadTypesPreference
     */
    public final String inferredWorkloadTypesAsString() {
        return inferredWorkloadTypes;
    }

    /**
     * <p>
     * An object that describes the external metrics recommendation preference.
     * </p>
     * <p>
     * If the preference is applied in the latest recommendation refresh, an object with a valid <code>source</code>
     * value appears in the response. If the preference isn't applied to the recommendations already, then this object
     * doesn't appear in the response.
     * </p>
     * 
     * @return An object that describes the external metrics recommendation preference. </p>
     *         <p>
     *         If the preference is applied in the latest recommendation refresh, an object with a valid
     *         <code>source</code> value appears in the response. If the preference isn't applied to the recommendations
     *         already, then this object doesn't appear in the response.
     */
    public final ExternalMetricsPreference externalMetricsPreference() {
        return externalMetricsPreference;
    }

    /**
     * <p>
     * The preference to control the number of days the utilization metrics of the Amazon Web Services resource are
     * analyzed. If the preference isn’t set, this object is null.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #lookBackPeriod}
     * will return {@link LookBackPeriodPreference#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #lookBackPeriodAsString}.
     * </p>
     * 
     * @return The preference to control the number of days the utilization metrics of the Amazon Web Services resource
     *         are analyzed. If the preference isn’t set, this object is null.
     * @see LookBackPeriodPreference
     */
    public final LookBackPeriodPreference lookBackPeriod() {
        return LookBackPeriodPreference.fromValue(lookBackPeriod);
    }

    /**
     * <p>
     * The preference to control the number of days the utilization metrics of the Amazon Web Services resource are
     * analyzed. If the preference isn’t set, this object is null.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #lookBackPeriod}
     * will return {@link LookBackPeriodPreference#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #lookBackPeriodAsString}.
     * </p>
     * 
     * @return The preference to control the number of days the utilization metrics of the Amazon Web Services resource
     *         are analyzed. If the preference isn’t set, this object is null.
     * @see LookBackPeriodPreference
     */
    public final String lookBackPeriodAsString() {
        return lookBackPeriod;
    }

    /**
     * For responses, this returns true if the service returned a value for the UtilizationPreferences property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasUtilizationPreferences() {
        return utilizationPreferences != null && !(utilizationPreferences instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The preference to control the resource’s CPU utilization thresholds - threshold and headroom. If the preference
     * isn’t set, this object is null.
     * </p>
     * <note>
     * <p>
     * This preference is only available for the Amazon EC2 instance resource type.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasUtilizationPreferences} method.
     * </p>
     * 
     * @return The preference to control the resource’s CPU utilization thresholds - threshold and headroom. If the
     *         preference isn’t set, this object is null. </p> <note>
     *         <p>
     *         This preference is only available for the Amazon EC2 instance resource type.
     *         </p>
     */
    public final List<UtilizationPreference> utilizationPreferences() {
        return utilizationPreferences;
    }

    /**
     * For responses, this returns true if the service returned a value for the PreferredResources property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasPreferredResources() {
        return preferredResources != null && !(preferredResources instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The preference to control which resource type values are considered when generating rightsizing recommendations.
     * This object resolves any wildcard expressions and returns the effective list of candidate resource type values.
     * If the preference isn’t set, this object is null.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPreferredResources} method.
     * </p>
     * 
     * @return The preference to control which resource type values are considered when generating rightsizing
     *         recommendations. This object resolves any wildcard expressions and returns the effective list of
     *         candidate resource type values. If the preference isn’t set, this object is null.
     */
    public final List<EffectivePreferredResource> preferredResources() {
        return preferredResources;
    }

    /**
     * <p>
     * Describes the savings estimation mode used for calculating savings opportunity.
     * </p>
     * <p>
     * Only the account manager or delegated administrator of your organization can activate this preference.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #savingsEstimationMode} will return {@link SavingsEstimationMode#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #savingsEstimationModeAsString}.
     * </p>
     * 
     * @return Describes the savings estimation mode used for calculating savings opportunity. </p>
     *         <p>
     *         Only the account manager or delegated administrator of your organization can activate this preference.
     * @see SavingsEstimationMode
     */
    public final SavingsEstimationMode savingsEstimationMode() {
        return SavingsEstimationMode.fromValue(savingsEstimationMode);
    }

    /**
     * <p>
     * Describes the savings estimation mode used for calculating savings opportunity.
     * </p>
     * <p>
     * Only the account manager or delegated administrator of your organization can activate this preference.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #savingsEstimationMode} will return {@link SavingsEstimationMode#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #savingsEstimationModeAsString}.
     * </p>
     * 
     * @return Describes the savings estimation mode used for calculating savings opportunity. </p>
     *         <p>
     *         Only the account manager or delegated administrator of your organization can activate this preference.
     * @see SavingsEstimationMode
     */
    public final String savingsEstimationModeAsString() {
        return savingsEstimationMode;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(scope());
        hashCode = 31 * hashCode + Objects.hashCode(resourceTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(enhancedInfrastructureMetricsAsString());
        hashCode = 31 * hashCode + Objects.hashCode(inferredWorkloadTypesAsString());
        hashCode = 31 * hashCode + Objects.hashCode(externalMetricsPreference());
        hashCode = 31 * hashCode + Objects.hashCode(lookBackPeriodAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasUtilizationPreferences() ? utilizationPreferences() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasPreferredResources() ? preferredResources() : null);
        hashCode = 31 * hashCode + Objects.hashCode(savingsEstimationModeAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RecommendationPreferencesDetail)) {
            return false;
        }
        RecommendationPreferencesDetail other = (RecommendationPreferencesDetail) obj;
        return Objects.equals(scope(), other.scope()) && Objects.equals(resourceTypeAsString(), other.resourceTypeAsString())
                && Objects.equals(enhancedInfrastructureMetricsAsString(), other.enhancedInfrastructureMetricsAsString())
                && Objects.equals(inferredWorkloadTypesAsString(), other.inferredWorkloadTypesAsString())
                && Objects.equals(externalMetricsPreference(), other.externalMetricsPreference())
                && Objects.equals(lookBackPeriodAsString(), other.lookBackPeriodAsString())
                && hasUtilizationPreferences() == other.hasUtilizationPreferences()
                && Objects.equals(utilizationPreferences(), other.utilizationPreferences())
                && hasPreferredResources() == other.hasPreferredResources()
                && Objects.equals(preferredResources(), other.preferredResources())
                && Objects.equals(savingsEstimationModeAsString(), other.savingsEstimationModeAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("RecommendationPreferencesDetail").add("Scope", scope())
                .add("ResourceType", resourceTypeAsString())
                .add("EnhancedInfrastructureMetrics", enhancedInfrastructureMetricsAsString())
                .add("InferredWorkloadTypes", inferredWorkloadTypesAsString())
                .add("ExternalMetricsPreference", externalMetricsPreference()).add("LookBackPeriod", lookBackPeriodAsString())
                .add("UtilizationPreferences", hasUtilizationPreferences() ? utilizationPreferences() : null)
                .add("PreferredResources", hasPreferredResources() ? preferredResources() : null)
                .add("SavingsEstimationMode", savingsEstimationModeAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "scope":
            return Optional.ofNullable(clazz.cast(scope()));
        case "resourceType":
            return Optional.ofNullable(clazz.cast(resourceTypeAsString()));
        case "enhancedInfrastructureMetrics":
            return Optional.ofNullable(clazz.cast(enhancedInfrastructureMetricsAsString()));
        case "inferredWorkloadTypes":
            return Optional.ofNullable(clazz.cast(inferredWorkloadTypesAsString()));
        case "externalMetricsPreference":
            return Optional.ofNullable(clazz.cast(externalMetricsPreference()));
        case "lookBackPeriod":
            return Optional.ofNullable(clazz.cast(lookBackPeriodAsString()));
        case "utilizationPreferences":
            return Optional.ofNullable(clazz.cast(utilizationPreferences()));
        case "preferredResources":
            return Optional.ofNullable(clazz.cast(preferredResources()));
        case "savingsEstimationMode":
            return Optional.ofNullable(clazz.cast(savingsEstimationModeAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<RecommendationPreferencesDetail, T> g) {
        return obj -> g.apply((RecommendationPreferencesDetail) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, RecommendationPreferencesDetail> {
        /**
         * <p>
         * An object that describes the scope of the recommendation preference.
         * </p>
         * <p>
         * Recommendation preferences can be created at the organization level (for management accounts of an
         * organization only), account level, and resource level. For more information, see <a
         * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/enhanced-infrastructure-metrics.html"
         * >Activating enhanced infrastructure metrics</a> in the <i>Compute Optimizer User Guide</i>.
         * </p>
         * 
         * @param scope
         *        An object that describes the scope of the recommendation preference.</p>
         *        <p>
         *        Recommendation preferences can be created at the organization level (for management accounts of an
         *        organization only), account level, and resource level. For more information, see <a
         *        href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/enhanced-infrastructure-metrics.html"
         *        >Activating enhanced infrastructure metrics</a> in the <i>Compute Optimizer User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scope(Scope scope);

        /**
         * <p>
         * An object that describes the scope of the recommendation preference.
         * </p>
         * <p>
         * Recommendation preferences can be created at the organization level (for management accounts of an
         * organization only), account level, and resource level. For more information, see <a
         * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/enhanced-infrastructure-metrics.html"
         * >Activating enhanced infrastructure metrics</a> in the <i>Compute Optimizer User Guide</i>.
         * </p>
         * This is a convenience method that creates an instance of the {@link Scope.Builder} avoiding the need to
         * create one manually via {@link Scope#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Scope.Builder#build()} is called immediately and its result is
         * passed to {@link #scope(Scope)}.
         * 
         * @param scope
         *        a consumer that will call methods on {@link Scope.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #scope(Scope)
         */
        default Builder scope(Consumer<Scope.Builder> scope) {
            return scope(Scope.builder().applyMutation(scope).build());
        }

        /**
         * <p>
         * The target resource type of the recommendation preference to create.
         * </p>
         * <p>
         * The <code>Ec2Instance</code> option encompasses standalone instances and instances that are part of Auto
         * Scaling groups. The <code>AutoScalingGroup</code> option encompasses only instances that are part of an Auto
         * Scaling group.
         * </p>
         * 
         * @param resourceType
         *        The target resource type of the recommendation preference to create.</p>
         *        <p>
         *        The <code>Ec2Instance</code> option encompasses standalone instances and instances that are part of
         *        Auto Scaling groups. The <code>AutoScalingGroup</code> option encompasses only instances that are part
         *        of an Auto Scaling group.
         * @see ResourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResourceType
         */
        Builder resourceType(String resourceType);

        /**
         * <p>
         * The target resource type of the recommendation preference to create.
         * </p>
         * <p>
         * The <code>Ec2Instance</code> option encompasses standalone instances and instances that are part of Auto
         * Scaling groups. The <code>AutoScalingGroup</code> option encompasses only instances that are part of an Auto
         * Scaling group.
         * </p>
         * 
         * @param resourceType
         *        The target resource type of the recommendation preference to create.</p>
         *        <p>
         *        The <code>Ec2Instance</code> option encompasses standalone instances and instances that are part of
         *        Auto Scaling groups. The <code>AutoScalingGroup</code> option encompasses only instances that are part
         *        of an Auto Scaling group.
         * @see ResourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResourceType
         */
        Builder resourceType(ResourceType resourceType);

        /**
         * <p>
         * The status of the enhanced infrastructure metrics recommendation preference.
         * </p>
         * <p>
         * When the recommendations page is refreshed, a status of <code>Active</code> confirms that the preference is
         * applied to the recommendations, and a status of <code>Inactive</code> confirms that the preference isn't yet
         * applied to recommendations.
         * </p>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/enhanced-infrastructure-metrics.html">Enhanced
         * infrastructure metrics</a> in the <i>Compute Optimizer User Guide</i>.
         * </p>
         * 
         * @param enhancedInfrastructureMetrics
         *        The status of the enhanced infrastructure metrics recommendation preference.</p>
         *        <p>
         *        When the recommendations page is refreshed, a status of <code>Active</code> confirms that the
         *        preference is applied to the recommendations, and a status of <code>Inactive</code> confirms that the
         *        preference isn't yet applied to recommendations.
         *        </p>
         *        <p>
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/enhanced-infrastructure-metrics.html"
         *        >Enhanced infrastructure metrics</a> in the <i>Compute Optimizer User Guide</i>.
         * @see EnhancedInfrastructureMetrics
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EnhancedInfrastructureMetrics
         */
        Builder enhancedInfrastructureMetrics(String enhancedInfrastructureMetrics);

        /**
         * <p>
         * The status of the enhanced infrastructure metrics recommendation preference.
         * </p>
         * <p>
         * When the recommendations page is refreshed, a status of <code>Active</code> confirms that the preference is
         * applied to the recommendations, and a status of <code>Inactive</code> confirms that the preference isn't yet
         * applied to recommendations.
         * </p>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/enhanced-infrastructure-metrics.html">Enhanced
         * infrastructure metrics</a> in the <i>Compute Optimizer User Guide</i>.
         * </p>
         * 
         * @param enhancedInfrastructureMetrics
         *        The status of the enhanced infrastructure metrics recommendation preference.</p>
         *        <p>
         *        When the recommendations page is refreshed, a status of <code>Active</code> confirms that the
         *        preference is applied to the recommendations, and a status of <code>Inactive</code> confirms that the
         *        preference isn't yet applied to recommendations.
         *        </p>
         *        <p>
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/enhanced-infrastructure-metrics.html"
         *        >Enhanced infrastructure metrics</a> in the <i>Compute Optimizer User Guide</i>.
         * @see EnhancedInfrastructureMetrics
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EnhancedInfrastructureMetrics
         */
        Builder enhancedInfrastructureMetrics(EnhancedInfrastructureMetrics enhancedInfrastructureMetrics);

        /**
         * <p>
         * The status of the inferred workload types recommendation preference.
         * </p>
         * <p>
         * When the recommendations page is refreshed, a status of <code>Active</code> confirms that the preference is
         * applied to the recommendations, and a status of <code>Inactive</code> confirms that the preference isn't yet
         * applied to recommendations.
         * </p>
         * 
         * @param inferredWorkloadTypes
         *        The status of the inferred workload types recommendation preference.</p>
         *        <p>
         *        When the recommendations page is refreshed, a status of <code>Active</code> confirms that the
         *        preference is applied to the recommendations, and a status of <code>Inactive</code> confirms that the
         *        preference isn't yet applied to recommendations.
         * @see InferredWorkloadTypesPreference
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InferredWorkloadTypesPreference
         */
        Builder inferredWorkloadTypes(String inferredWorkloadTypes);

        /**
         * <p>
         * The status of the inferred workload types recommendation preference.
         * </p>
         * <p>
         * When the recommendations page is refreshed, a status of <code>Active</code> confirms that the preference is
         * applied to the recommendations, and a status of <code>Inactive</code> confirms that the preference isn't yet
         * applied to recommendations.
         * </p>
         * 
         * @param inferredWorkloadTypes
         *        The status of the inferred workload types recommendation preference.</p>
         *        <p>
         *        When the recommendations page is refreshed, a status of <code>Active</code> confirms that the
         *        preference is applied to the recommendations, and a status of <code>Inactive</code> confirms that the
         *        preference isn't yet applied to recommendations.
         * @see InferredWorkloadTypesPreference
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InferredWorkloadTypesPreference
         */
        Builder inferredWorkloadTypes(InferredWorkloadTypesPreference inferredWorkloadTypes);

        /**
         * <p>
         * An object that describes the external metrics recommendation preference.
         * </p>
         * <p>
         * If the preference is applied in the latest recommendation refresh, an object with a valid <code>source</code>
         * value appears in the response. If the preference isn't applied to the recommendations already, then this
         * object doesn't appear in the response.
         * </p>
         * 
         * @param externalMetricsPreference
         *        An object that describes the external metrics recommendation preference. </p>
         *        <p>
         *        If the preference is applied in the latest recommendation refresh, an object with a valid
         *        <code>source</code> value appears in the response. If the preference isn't applied to the
         *        recommendations already, then this object doesn't appear in the response.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder externalMetricsPreference(ExternalMetricsPreference externalMetricsPreference);

        /**
         * <p>
         * An object that describes the external metrics recommendation preference.
         * </p>
         * <p>
         * If the preference is applied in the latest recommendation refresh, an object with a valid <code>source</code>
         * value appears in the response. If the preference isn't applied to the recommendations already, then this
         * object doesn't appear in the response.
         * </p>
         * This is a convenience method that creates an instance of the {@link ExternalMetricsPreference.Builder}
         * avoiding the need to create one manually via {@link ExternalMetricsPreference#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ExternalMetricsPreference.Builder#build()} is called immediately
         * and its result is passed to {@link #externalMetricsPreference(ExternalMetricsPreference)}.
         * 
         * @param externalMetricsPreference
         *        a consumer that will call methods on {@link ExternalMetricsPreference.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #externalMetricsPreference(ExternalMetricsPreference)
         */
        default Builder externalMetricsPreference(Consumer<ExternalMetricsPreference.Builder> externalMetricsPreference) {
            return externalMetricsPreference(ExternalMetricsPreference.builder().applyMutation(externalMetricsPreference).build());
        }

        /**
         * <p>
         * The preference to control the number of days the utilization metrics of the Amazon Web Services resource are
         * analyzed. If the preference isn’t set, this object is null.
         * </p>
         * 
         * @param lookBackPeriod
         *        The preference to control the number of days the utilization metrics of the Amazon Web Services
         *        resource are analyzed. If the preference isn’t set, this object is null.
         * @see LookBackPeriodPreference
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LookBackPeriodPreference
         */
        Builder lookBackPeriod(String lookBackPeriod);

        /**
         * <p>
         * The preference to control the number of days the utilization metrics of the Amazon Web Services resource are
         * analyzed. If the preference isn’t set, this object is null.
         * </p>
         * 
         * @param lookBackPeriod
         *        The preference to control the number of days the utilization metrics of the Amazon Web Services
         *        resource are analyzed. If the preference isn’t set, this object is null.
         * @see LookBackPeriodPreference
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LookBackPeriodPreference
         */
        Builder lookBackPeriod(LookBackPeriodPreference lookBackPeriod);

        /**
         * <p>
         * The preference to control the resource’s CPU utilization thresholds - threshold and headroom. If the
         * preference isn’t set, this object is null.
         * </p>
         * <note>
         * <p>
         * This preference is only available for the Amazon EC2 instance resource type.
         * </p>
         * </note>
         * 
         * @param utilizationPreferences
         *        The preference to control the resource’s CPU utilization thresholds - threshold and headroom. If the
         *        preference isn’t set, this object is null. </p> <note>
         *        <p>
         *        This preference is only available for the Amazon EC2 instance resource type.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder utilizationPreferences(Collection<UtilizationPreference> utilizationPreferences);

        /**
         * <p>
         * The preference to control the resource’s CPU utilization thresholds - threshold and headroom. If the
         * preference isn’t set, this object is null.
         * </p>
         * <note>
         * <p>
         * This preference is only available for the Amazon EC2 instance resource type.
         * </p>
         * </note>
         * 
         * @param utilizationPreferences
         *        The preference to control the resource’s CPU utilization thresholds - threshold and headroom. If the
         *        preference isn’t set, this object is null. </p> <note>
         *        <p>
         *        This preference is only available for the Amazon EC2 instance resource type.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder utilizationPreferences(UtilizationPreference... utilizationPreferences);

        /**
         * <p>
         * The preference to control the resource’s CPU utilization thresholds - threshold and headroom. If the
         * preference isn’t set, this object is null.
         * </p>
         * <note>
         * <p>
         * This preference is only available for the Amazon EC2 instance resource type.
         * </p>
         * </note> This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.computeoptimizer.model.UtilizationPreference.Builder} avoiding the
         * need to create one manually via
         * {@link software.amazon.awssdk.services.computeoptimizer.model.UtilizationPreference#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.computeoptimizer.model.UtilizationPreference.Builder#build()} is
         * called immediately and its result is passed to {@link #utilizationPreferences(List<UtilizationPreference>)}.
         * 
         * @param utilizationPreferences
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.computeoptimizer.model.UtilizationPreference.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #utilizationPreferences(java.util.Collection<UtilizationPreference>)
         */
        Builder utilizationPreferences(Consumer<UtilizationPreference.Builder>... utilizationPreferences);

        /**
         * <p>
         * The preference to control which resource type values are considered when generating rightsizing
         * recommendations. This object resolves any wildcard expressions and returns the effective list of candidate
         * resource type values. If the preference isn’t set, this object is null.
         * </p>
         * 
         * @param preferredResources
         *        The preference to control which resource type values are considered when generating rightsizing
         *        recommendations. This object resolves any wildcard expressions and returns the effective list of
         *        candidate resource type values. If the preference isn’t set, this object is null.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder preferredResources(Collection<EffectivePreferredResource> preferredResources);

        /**
         * <p>
         * The preference to control which resource type values are considered when generating rightsizing
         * recommendations. This object resolves any wildcard expressions and returns the effective list of candidate
         * resource type values. If the preference isn’t set, this object is null.
         * </p>
         * 
         * @param preferredResources
         *        The preference to control which resource type values are considered when generating rightsizing
         *        recommendations. This object resolves any wildcard expressions and returns the effective list of
         *        candidate resource type values. If the preference isn’t set, this object is null.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder preferredResources(EffectivePreferredResource... preferredResources);

        /**
         * <p>
         * The preference to control which resource type values are considered when generating rightsizing
         * recommendations. This object resolves any wildcard expressions and returns the effective list of candidate
         * resource type values. If the preference isn’t set, this object is null.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.computeoptimizer.model.EffectivePreferredResource.Builder} avoiding
         * the need to create one manually via
         * {@link software.amazon.awssdk.services.computeoptimizer.model.EffectivePreferredResource#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.computeoptimizer.model.EffectivePreferredResource.Builder#build()} is
         * called immediately and its result is passed to {@link #preferredResources(List<EffectivePreferredResource>)}.
         * 
         * @param preferredResources
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.computeoptimizer.model.EffectivePreferredResource.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #preferredResources(java.util.Collection<EffectivePreferredResource>)
         */
        Builder preferredResources(Consumer<EffectivePreferredResource.Builder>... preferredResources);

        /**
         * <p>
         * Describes the savings estimation mode used for calculating savings opportunity.
         * </p>
         * <p>
         * Only the account manager or delegated administrator of your organization can activate this preference.
         * </p>
         * 
         * @param savingsEstimationMode
         *        Describes the savings estimation mode used for calculating savings opportunity. </p>
         *        <p>
         *        Only the account manager or delegated administrator of your organization can activate this preference.
         * @see SavingsEstimationMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SavingsEstimationMode
         */
        Builder savingsEstimationMode(String savingsEstimationMode);

        /**
         * <p>
         * Describes the savings estimation mode used for calculating savings opportunity.
         * </p>
         * <p>
         * Only the account manager or delegated administrator of your organization can activate this preference.
         * </p>
         * 
         * @param savingsEstimationMode
         *        Describes the savings estimation mode used for calculating savings opportunity. </p>
         *        <p>
         *        Only the account manager or delegated administrator of your organization can activate this preference.
         * @see SavingsEstimationMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SavingsEstimationMode
         */
        Builder savingsEstimationMode(SavingsEstimationMode savingsEstimationMode);
    }

    static final class BuilderImpl implements Builder {
        private Scope scope;

        private String resourceType;

        private String enhancedInfrastructureMetrics;

        private String inferredWorkloadTypes;

        private ExternalMetricsPreference externalMetricsPreference;

        private String lookBackPeriod;

        private List<UtilizationPreference> utilizationPreferences = DefaultSdkAutoConstructList.getInstance();

        private List<EffectivePreferredResource> preferredResources = DefaultSdkAutoConstructList.getInstance();

        private String savingsEstimationMode;

        private BuilderImpl() {
        }

        private BuilderImpl(RecommendationPreferencesDetail model) {
            scope(model.scope);
            resourceType(model.resourceType);
            enhancedInfrastructureMetrics(model.enhancedInfrastructureMetrics);
            inferredWorkloadTypes(model.inferredWorkloadTypes);
            externalMetricsPreference(model.externalMetricsPreference);
            lookBackPeriod(model.lookBackPeriod);
            utilizationPreferences(model.utilizationPreferences);
            preferredResources(model.preferredResources);
            savingsEstimationMode(model.savingsEstimationMode);
        }

        public final Scope.Builder getScope() {
            return scope != null ? scope.toBuilder() : null;
        }

        public final void setScope(Scope.BuilderImpl scope) {
            this.scope = scope != null ? scope.build() : null;
        }

        @Override
        public final Builder scope(Scope scope) {
            this.scope = scope;
            return this;
        }

        public final String getResourceType() {
            return resourceType;
        }

        public final void setResourceType(String resourceType) {
            this.resourceType = resourceType;
        }

        @Override
        public final Builder resourceType(String resourceType) {
            this.resourceType = resourceType;
            return this;
        }

        @Override
        public final Builder resourceType(ResourceType resourceType) {
            this.resourceType(resourceType == null ? null : resourceType.toString());
            return this;
        }

        public final String getEnhancedInfrastructureMetrics() {
            return enhancedInfrastructureMetrics;
        }

        public final void setEnhancedInfrastructureMetrics(String enhancedInfrastructureMetrics) {
            this.enhancedInfrastructureMetrics = enhancedInfrastructureMetrics;
        }

        @Override
        public final Builder enhancedInfrastructureMetrics(String enhancedInfrastructureMetrics) {
            this.enhancedInfrastructureMetrics = enhancedInfrastructureMetrics;
            return this;
        }

        @Override
        public final Builder enhancedInfrastructureMetrics(EnhancedInfrastructureMetrics enhancedInfrastructureMetrics) {
            this.enhancedInfrastructureMetrics(enhancedInfrastructureMetrics == null ? null : enhancedInfrastructureMetrics
                    .toString());
            return this;
        }

        public final String getInferredWorkloadTypes() {
            return inferredWorkloadTypes;
        }

        public final void setInferredWorkloadTypes(String inferredWorkloadTypes) {
            this.inferredWorkloadTypes = inferredWorkloadTypes;
        }

        @Override
        public final Builder inferredWorkloadTypes(String inferredWorkloadTypes) {
            this.inferredWorkloadTypes = inferredWorkloadTypes;
            return this;
        }

        @Override
        public final Builder inferredWorkloadTypes(InferredWorkloadTypesPreference inferredWorkloadTypes) {
            this.inferredWorkloadTypes(inferredWorkloadTypes == null ? null : inferredWorkloadTypes.toString());
            return this;
        }

        public final ExternalMetricsPreference.Builder getExternalMetricsPreference() {
            return externalMetricsPreference != null ? externalMetricsPreference.toBuilder() : null;
        }

        public final void setExternalMetricsPreference(ExternalMetricsPreference.BuilderImpl externalMetricsPreference) {
            this.externalMetricsPreference = externalMetricsPreference != null ? externalMetricsPreference.build() : null;
        }

        @Override
        public final Builder externalMetricsPreference(ExternalMetricsPreference externalMetricsPreference) {
            this.externalMetricsPreference = externalMetricsPreference;
            return this;
        }

        public final String getLookBackPeriod() {
            return lookBackPeriod;
        }

        public final void setLookBackPeriod(String lookBackPeriod) {
            this.lookBackPeriod = lookBackPeriod;
        }

        @Override
        public final Builder lookBackPeriod(String lookBackPeriod) {
            this.lookBackPeriod = lookBackPeriod;
            return this;
        }

        @Override
        public final Builder lookBackPeriod(LookBackPeriodPreference lookBackPeriod) {
            this.lookBackPeriod(lookBackPeriod == null ? null : lookBackPeriod.toString());
            return this;
        }

        public final List<UtilizationPreference.Builder> getUtilizationPreferences() {
            List<UtilizationPreference.Builder> result = UtilizationPreferencesCopier.copyToBuilder(this.utilizationPreferences);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setUtilizationPreferences(Collection<UtilizationPreference.BuilderImpl> utilizationPreferences) {
            this.utilizationPreferences = UtilizationPreferencesCopier.copyFromBuilder(utilizationPreferences);
        }

        @Override
        public final Builder utilizationPreferences(Collection<UtilizationPreference> utilizationPreferences) {
            this.utilizationPreferences = UtilizationPreferencesCopier.copy(utilizationPreferences);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder utilizationPreferences(UtilizationPreference... utilizationPreferences) {
            utilizationPreferences(Arrays.asList(utilizationPreferences));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder utilizationPreferences(Consumer<UtilizationPreference.Builder>... utilizationPreferences) {
            utilizationPreferences(Stream.of(utilizationPreferences)
                    .map(c -> UtilizationPreference.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final List<EffectivePreferredResource.Builder> getPreferredResources() {
            List<EffectivePreferredResource.Builder> result = EffectivePreferredResourcesCopier
                    .copyToBuilder(this.preferredResources);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setPreferredResources(Collection<EffectivePreferredResource.BuilderImpl> preferredResources) {
            this.preferredResources = EffectivePreferredResourcesCopier.copyFromBuilder(preferredResources);
        }

        @Override
        public final Builder preferredResources(Collection<EffectivePreferredResource> preferredResources) {
            this.preferredResources = EffectivePreferredResourcesCopier.copy(preferredResources);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder preferredResources(EffectivePreferredResource... preferredResources) {
            preferredResources(Arrays.asList(preferredResources));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder preferredResources(Consumer<EffectivePreferredResource.Builder>... preferredResources) {
            preferredResources(Stream.of(preferredResources)
                    .map(c -> EffectivePreferredResource.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getSavingsEstimationMode() {
            return savingsEstimationMode;
        }

        public final void setSavingsEstimationMode(String savingsEstimationMode) {
            this.savingsEstimationMode = savingsEstimationMode;
        }

        @Override
        public final Builder savingsEstimationMode(String savingsEstimationMode) {
            this.savingsEstimationMode = savingsEstimationMode;
            return this;
        }

        @Override
        public final Builder savingsEstimationMode(SavingsEstimationMode savingsEstimationMode) {
            this.savingsEstimationMode(savingsEstimationMode == null ? null : savingsEstimationMode.toString());
            return this;
        }

        @Override
        public RecommendationPreferencesDetail build() {
            return new RecommendationPreferencesDetail(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
