/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.computeoptimizer.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes an Amazon Elastic Block Store (Amazon EBS) volume recommendation.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class VolumeRecommendation implements SdkPojo, Serializable,
        ToCopyableBuilder<VolumeRecommendation.Builder, VolumeRecommendation> {
    private static final SdkField<String> VOLUME_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("volumeArn").getter(getter(VolumeRecommendation::volumeArn)).setter(setter(Builder::volumeArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("volumeArn").build()).build();

    private static final SdkField<String> ACCOUNT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("accountId").getter(getter(VolumeRecommendation::accountId)).setter(setter(Builder::accountId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("accountId").build()).build();

    private static final SdkField<VolumeConfiguration> CURRENT_CONFIGURATION_FIELD = SdkField
            .<VolumeConfiguration> builder(MarshallingType.SDK_POJO).memberName("currentConfiguration")
            .getter(getter(VolumeRecommendation::currentConfiguration)).setter(setter(Builder::currentConfiguration))
            .constructor(VolumeConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("currentConfiguration").build())
            .build();

    private static final SdkField<String> FINDING_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("finding")
            .getter(getter(VolumeRecommendation::findingAsString)).setter(setter(Builder::finding))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("finding").build()).build();

    private static final SdkField<List<EBSUtilizationMetric>> UTILIZATION_METRICS_FIELD = SdkField
            .<List<EBSUtilizationMetric>> builder(MarshallingType.LIST)
            .memberName("utilizationMetrics")
            .getter(getter(VolumeRecommendation::utilizationMetrics))
            .setter(setter(Builder::utilizationMetrics))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("utilizationMetrics").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<EBSUtilizationMetric> builder(MarshallingType.SDK_POJO)
                                            .constructor(EBSUtilizationMetric::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Double> LOOK_BACK_PERIOD_IN_DAYS_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .memberName("lookBackPeriodInDays").getter(getter(VolumeRecommendation::lookBackPeriodInDays))
            .setter(setter(Builder::lookBackPeriodInDays))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lookBackPeriodInDays").build())
            .build();

    private static final SdkField<List<VolumeRecommendationOption>> VOLUME_RECOMMENDATION_OPTIONS_FIELD = SdkField
            .<List<VolumeRecommendationOption>> builder(MarshallingType.LIST)
            .memberName("volumeRecommendationOptions")
            .getter(getter(VolumeRecommendation::volumeRecommendationOptions))
            .setter(setter(Builder::volumeRecommendationOptions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("volumeRecommendationOptions")
                    .build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<VolumeRecommendationOption> builder(MarshallingType.SDK_POJO)
                                            .constructor(VolumeRecommendationOption::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Instant> LAST_REFRESH_TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("lastRefreshTimestamp").getter(getter(VolumeRecommendation::lastRefreshTimestamp))
            .setter(setter(Builder::lastRefreshTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lastRefreshTimestamp").build())
            .build();

    private static final SdkField<String> CURRENT_PERFORMANCE_RISK_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("currentPerformanceRisk").getter(getter(VolumeRecommendation::currentPerformanceRiskAsString))
            .setter(setter(Builder::currentPerformanceRisk))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("currentPerformanceRisk").build())
            .build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("tags")
            .getter(getter(VolumeRecommendation::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<EBSEffectiveRecommendationPreferences> EFFECTIVE_RECOMMENDATION_PREFERENCES_FIELD = SdkField
            .<EBSEffectiveRecommendationPreferences> builder(MarshallingType.SDK_POJO)
            .memberName("effectiveRecommendationPreferences")
            .getter(getter(VolumeRecommendation::effectiveRecommendationPreferences))
            .setter(setter(Builder::effectiveRecommendationPreferences))
            .constructor(EBSEffectiveRecommendationPreferences::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("effectiveRecommendationPreferences")
                    .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(VOLUME_ARN_FIELD,
            ACCOUNT_ID_FIELD, CURRENT_CONFIGURATION_FIELD, FINDING_FIELD, UTILIZATION_METRICS_FIELD,
            LOOK_BACK_PERIOD_IN_DAYS_FIELD, VOLUME_RECOMMENDATION_OPTIONS_FIELD, LAST_REFRESH_TIMESTAMP_FIELD,
            CURRENT_PERFORMANCE_RISK_FIELD, TAGS_FIELD, EFFECTIVE_RECOMMENDATION_PREFERENCES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String volumeArn;

    private final String accountId;

    private final VolumeConfiguration currentConfiguration;

    private final String finding;

    private final List<EBSUtilizationMetric> utilizationMetrics;

    private final Double lookBackPeriodInDays;

    private final List<VolumeRecommendationOption> volumeRecommendationOptions;

    private final Instant lastRefreshTimestamp;

    private final String currentPerformanceRisk;

    private final List<Tag> tags;

    private final EBSEffectiveRecommendationPreferences effectiveRecommendationPreferences;

    private VolumeRecommendation(BuilderImpl builder) {
        this.volumeArn = builder.volumeArn;
        this.accountId = builder.accountId;
        this.currentConfiguration = builder.currentConfiguration;
        this.finding = builder.finding;
        this.utilizationMetrics = builder.utilizationMetrics;
        this.lookBackPeriodInDays = builder.lookBackPeriodInDays;
        this.volumeRecommendationOptions = builder.volumeRecommendationOptions;
        this.lastRefreshTimestamp = builder.lastRefreshTimestamp;
        this.currentPerformanceRisk = builder.currentPerformanceRisk;
        this.tags = builder.tags;
        this.effectiveRecommendationPreferences = builder.effectiveRecommendationPreferences;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the current volume.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the current volume.
     */
    public final String volumeArn() {
        return volumeArn;
    }

    /**
     * <p>
     * The Amazon Web Services account ID of the volume.
     * </p>
     * 
     * @return The Amazon Web Services account ID of the volume.
     */
    public final String accountId() {
        return accountId;
    }

    /**
     * <p>
     * An array of objects that describe the current configuration of the volume.
     * </p>
     * 
     * @return An array of objects that describe the current configuration of the volume.
     */
    public final VolumeConfiguration currentConfiguration() {
        return currentConfiguration;
    }

    /**
     * <p>
     * The finding classification of the volume.
     * </p>
     * <p>
     * Findings for volumes include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b> <code>NotOptimized</code> </b>—A volume is considered not optimized when Compute Optimizer identifies a
     * recommendation that can provide better performance for your workload.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>Optimized</code> </b>—An volume is considered optimized when Compute Optimizer determines that the
     * volume is correctly provisioned to run your workload based on the chosen volume type. For optimized resources,
     * Compute Optimizer might recommend a new generation volume type.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #finding} will
     * return {@link EBSFinding#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #findingAsString}.
     * </p>
     * 
     * @return The finding classification of the volume.</p>
     *         <p>
     *         Findings for volumes include:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b> <code>NotOptimized</code> </b>—A volume is considered not optimized when Compute Optimizer identifies
     *         a recommendation that can provide better performance for your workload.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>Optimized</code> </b>—An volume is considered optimized when Compute Optimizer determines that
     *         the volume is correctly provisioned to run your workload based on the chosen volume type. For optimized
     *         resources, Compute Optimizer might recommend a new generation volume type.
     *         </p>
     *         </li>
     * @see EBSFinding
     */
    public final EBSFinding finding() {
        return EBSFinding.fromValue(finding);
    }

    /**
     * <p>
     * The finding classification of the volume.
     * </p>
     * <p>
     * Findings for volumes include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b> <code>NotOptimized</code> </b>—A volume is considered not optimized when Compute Optimizer identifies a
     * recommendation that can provide better performance for your workload.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>Optimized</code> </b>—An volume is considered optimized when Compute Optimizer determines that the
     * volume is correctly provisioned to run your workload based on the chosen volume type. For optimized resources,
     * Compute Optimizer might recommend a new generation volume type.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #finding} will
     * return {@link EBSFinding#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #findingAsString}.
     * </p>
     * 
     * @return The finding classification of the volume.</p>
     *         <p>
     *         Findings for volumes include:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b> <code>NotOptimized</code> </b>—A volume is considered not optimized when Compute Optimizer identifies
     *         a recommendation that can provide better performance for your workload.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>Optimized</code> </b>—An volume is considered optimized when Compute Optimizer determines that
     *         the volume is correctly provisioned to run your workload based on the chosen volume type. For optimized
     *         resources, Compute Optimizer might recommend a new generation volume type.
     *         </p>
     *         </li>
     * @see EBSFinding
     */
    public final String findingAsString() {
        return finding;
    }

    /**
     * For responses, this returns true if the service returned a value for the UtilizationMetrics property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasUtilizationMetrics() {
        return utilizationMetrics != null && !(utilizationMetrics instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of objects that describe the utilization metrics of the volume.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasUtilizationMetrics} method.
     * </p>
     * 
     * @return An array of objects that describe the utilization metrics of the volume.
     */
    public final List<EBSUtilizationMetric> utilizationMetrics() {
        return utilizationMetrics;
    }

    /**
     * <p>
     * The number of days for which utilization metrics were analyzed for the volume.
     * </p>
     * 
     * @return The number of days for which utilization metrics were analyzed for the volume.
     */
    public final Double lookBackPeriodInDays() {
        return lookBackPeriodInDays;
    }

    /**
     * For responses, this returns true if the service returned a value for the VolumeRecommendationOptions property.
     * This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasVolumeRecommendationOptions() {
        return volumeRecommendationOptions != null && !(volumeRecommendationOptions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of objects that describe the recommendation options for the volume.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasVolumeRecommendationOptions} method.
     * </p>
     * 
     * @return An array of objects that describe the recommendation options for the volume.
     */
    public final List<VolumeRecommendationOption> volumeRecommendationOptions() {
        return volumeRecommendationOptions;
    }

    /**
     * <p>
     * The timestamp of when the volume recommendation was last generated.
     * </p>
     * 
     * @return The timestamp of when the volume recommendation was last generated.
     */
    public final Instant lastRefreshTimestamp() {
        return lastRefreshTimestamp;
    }

    /**
     * <p>
     * The risk of the current EBS volume not meeting the performance needs of its workloads. The higher the risk, the
     * more likely the current EBS volume doesn't have sufficient capacity.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #currentPerformanceRisk} will return {@link CurrentPerformanceRisk#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #currentPerformanceRiskAsString}.
     * </p>
     * 
     * @return The risk of the current EBS volume not meeting the performance needs of its workloads. The higher the
     *         risk, the more likely the current EBS volume doesn't have sufficient capacity.
     * @see CurrentPerformanceRisk
     */
    public final CurrentPerformanceRisk currentPerformanceRisk() {
        return CurrentPerformanceRisk.fromValue(currentPerformanceRisk);
    }

    /**
     * <p>
     * The risk of the current EBS volume not meeting the performance needs of its workloads. The higher the risk, the
     * more likely the current EBS volume doesn't have sufficient capacity.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #currentPerformanceRisk} will return {@link CurrentPerformanceRisk#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #currentPerformanceRiskAsString}.
     * </p>
     * 
     * @return The risk of the current EBS volume not meeting the performance needs of its workloads. The higher the
     *         risk, the more likely the current EBS volume doesn't have sufficient capacity.
     * @see CurrentPerformanceRisk
     */
    public final String currentPerformanceRiskAsString() {
        return currentPerformanceRisk;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of tags assigned to your Amazon EBS volume recommendations.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return A list of tags assigned to your Amazon EBS volume recommendations.
     */
    public final List<Tag> tags() {
        return tags;
    }

    /**
     * <p>
     * Describes the effective recommendation preferences for Amazon EBS volume.
     * </p>
     * 
     * @return Describes the effective recommendation preferences for Amazon EBS volume.
     */
    public final EBSEffectiveRecommendationPreferences effectiveRecommendationPreferences() {
        return effectiveRecommendationPreferences;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(volumeArn());
        hashCode = 31 * hashCode + Objects.hashCode(accountId());
        hashCode = 31 * hashCode + Objects.hashCode(currentConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(findingAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasUtilizationMetrics() ? utilizationMetrics() : null);
        hashCode = 31 * hashCode + Objects.hashCode(lookBackPeriodInDays());
        hashCode = 31 * hashCode + Objects.hashCode(hasVolumeRecommendationOptions() ? volumeRecommendationOptions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(lastRefreshTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(currentPerformanceRiskAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(effectiveRecommendationPreferences());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof VolumeRecommendation)) {
            return false;
        }
        VolumeRecommendation other = (VolumeRecommendation) obj;
        return Objects.equals(volumeArn(), other.volumeArn()) && Objects.equals(accountId(), other.accountId())
                && Objects.equals(currentConfiguration(), other.currentConfiguration())
                && Objects.equals(findingAsString(), other.findingAsString())
                && hasUtilizationMetrics() == other.hasUtilizationMetrics()
                && Objects.equals(utilizationMetrics(), other.utilizationMetrics())
                && Objects.equals(lookBackPeriodInDays(), other.lookBackPeriodInDays())
                && hasVolumeRecommendationOptions() == other.hasVolumeRecommendationOptions()
                && Objects.equals(volumeRecommendationOptions(), other.volumeRecommendationOptions())
                && Objects.equals(lastRefreshTimestamp(), other.lastRefreshTimestamp())
                && Objects.equals(currentPerformanceRiskAsString(), other.currentPerformanceRiskAsString())
                && hasTags() == other.hasTags() && Objects.equals(tags(), other.tags())
                && Objects.equals(effectiveRecommendationPreferences(), other.effectiveRecommendationPreferences());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("VolumeRecommendation").add("VolumeArn", volumeArn()).add("AccountId", accountId())
                .add("CurrentConfiguration", currentConfiguration()).add("Finding", findingAsString())
                .add("UtilizationMetrics", hasUtilizationMetrics() ? utilizationMetrics() : null)
                .add("LookBackPeriodInDays", lookBackPeriodInDays())
                .add("VolumeRecommendationOptions", hasVolumeRecommendationOptions() ? volumeRecommendationOptions() : null)
                .add("LastRefreshTimestamp", lastRefreshTimestamp())
                .add("CurrentPerformanceRisk", currentPerformanceRiskAsString()).add("Tags", hasTags() ? tags() : null)
                .add("EffectiveRecommendationPreferences", effectiveRecommendationPreferences()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "volumeArn":
            return Optional.ofNullable(clazz.cast(volumeArn()));
        case "accountId":
            return Optional.ofNullable(clazz.cast(accountId()));
        case "currentConfiguration":
            return Optional.ofNullable(clazz.cast(currentConfiguration()));
        case "finding":
            return Optional.ofNullable(clazz.cast(findingAsString()));
        case "utilizationMetrics":
            return Optional.ofNullable(clazz.cast(utilizationMetrics()));
        case "lookBackPeriodInDays":
            return Optional.ofNullable(clazz.cast(lookBackPeriodInDays()));
        case "volumeRecommendationOptions":
            return Optional.ofNullable(clazz.cast(volumeRecommendationOptions()));
        case "lastRefreshTimestamp":
            return Optional.ofNullable(clazz.cast(lastRefreshTimestamp()));
        case "currentPerformanceRisk":
            return Optional.ofNullable(clazz.cast(currentPerformanceRiskAsString()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "effectiveRecommendationPreferences":
            return Optional.ofNullable(clazz.cast(effectiveRecommendationPreferences()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<VolumeRecommendation, T> g) {
        return obj -> g.apply((VolumeRecommendation) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, VolumeRecommendation> {
        /**
         * <p>
         * The Amazon Resource Name (ARN) of the current volume.
         * </p>
         * 
         * @param volumeArn
         *        The Amazon Resource Name (ARN) of the current volume.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder volumeArn(String volumeArn);

        /**
         * <p>
         * The Amazon Web Services account ID of the volume.
         * </p>
         * 
         * @param accountId
         *        The Amazon Web Services account ID of the volume.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accountId(String accountId);

        /**
         * <p>
         * An array of objects that describe the current configuration of the volume.
         * </p>
         * 
         * @param currentConfiguration
         *        An array of objects that describe the current configuration of the volume.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder currentConfiguration(VolumeConfiguration currentConfiguration);

        /**
         * <p>
         * An array of objects that describe the current configuration of the volume.
         * </p>
         * This is a convenience method that creates an instance of the {@link VolumeConfiguration.Builder} avoiding the
         * need to create one manually via {@link VolumeConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link VolumeConfiguration.Builder#build()} is called immediately and
         * its result is passed to {@link #currentConfiguration(VolumeConfiguration)}.
         * 
         * @param currentConfiguration
         *        a consumer that will call methods on {@link VolumeConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #currentConfiguration(VolumeConfiguration)
         */
        default Builder currentConfiguration(Consumer<VolumeConfiguration.Builder> currentConfiguration) {
            return currentConfiguration(VolumeConfiguration.builder().applyMutation(currentConfiguration).build());
        }

        /**
         * <p>
         * The finding classification of the volume.
         * </p>
         * <p>
         * Findings for volumes include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b> <code>NotOptimized</code> </b>—A volume is considered not optimized when Compute Optimizer identifies a
         * recommendation that can provide better performance for your workload.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>Optimized</code> </b>—An volume is considered optimized when Compute Optimizer determines that the
         * volume is correctly provisioned to run your workload based on the chosen volume type. For optimized
         * resources, Compute Optimizer might recommend a new generation volume type.
         * </p>
         * </li>
         * </ul>
         * 
         * @param finding
         *        The finding classification of the volume.</p>
         *        <p>
         *        Findings for volumes include:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b> <code>NotOptimized</code> </b>—A volume is considered not optimized when Compute Optimizer
         *        identifies a recommendation that can provide better performance for your workload.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>Optimized</code> </b>—An volume is considered optimized when Compute Optimizer determines
         *        that the volume is correctly provisioned to run your workload based on the chosen volume type. For
         *        optimized resources, Compute Optimizer might recommend a new generation volume type.
         *        </p>
         *        </li>
         * @see EBSFinding
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EBSFinding
         */
        Builder finding(String finding);

        /**
         * <p>
         * The finding classification of the volume.
         * </p>
         * <p>
         * Findings for volumes include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b> <code>NotOptimized</code> </b>—A volume is considered not optimized when Compute Optimizer identifies a
         * recommendation that can provide better performance for your workload.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>Optimized</code> </b>—An volume is considered optimized when Compute Optimizer determines that the
         * volume is correctly provisioned to run your workload based on the chosen volume type. For optimized
         * resources, Compute Optimizer might recommend a new generation volume type.
         * </p>
         * </li>
         * </ul>
         * 
         * @param finding
         *        The finding classification of the volume.</p>
         *        <p>
         *        Findings for volumes include:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b> <code>NotOptimized</code> </b>—A volume is considered not optimized when Compute Optimizer
         *        identifies a recommendation that can provide better performance for your workload.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>Optimized</code> </b>—An volume is considered optimized when Compute Optimizer determines
         *        that the volume is correctly provisioned to run your workload based on the chosen volume type. For
         *        optimized resources, Compute Optimizer might recommend a new generation volume type.
         *        </p>
         *        </li>
         * @see EBSFinding
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EBSFinding
         */
        Builder finding(EBSFinding finding);

        /**
         * <p>
         * An array of objects that describe the utilization metrics of the volume.
         * </p>
         * 
         * @param utilizationMetrics
         *        An array of objects that describe the utilization metrics of the volume.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder utilizationMetrics(Collection<EBSUtilizationMetric> utilizationMetrics);

        /**
         * <p>
         * An array of objects that describe the utilization metrics of the volume.
         * </p>
         * 
         * @param utilizationMetrics
         *        An array of objects that describe the utilization metrics of the volume.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder utilizationMetrics(EBSUtilizationMetric... utilizationMetrics);

        /**
         * <p>
         * An array of objects that describe the utilization metrics of the volume.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.computeoptimizer.model.EBSUtilizationMetric.Builder} avoiding the need
         * to create one manually via
         * {@link software.amazon.awssdk.services.computeoptimizer.model.EBSUtilizationMetric#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.computeoptimizer.model.EBSUtilizationMetric.Builder#build()} is called
         * immediately and its result is passed to {@link #utilizationMetrics(List<EBSUtilizationMetric>)}.
         * 
         * @param utilizationMetrics
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.computeoptimizer.model.EBSUtilizationMetric.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #utilizationMetrics(java.util.Collection<EBSUtilizationMetric>)
         */
        Builder utilizationMetrics(Consumer<EBSUtilizationMetric.Builder>... utilizationMetrics);

        /**
         * <p>
         * The number of days for which utilization metrics were analyzed for the volume.
         * </p>
         * 
         * @param lookBackPeriodInDays
         *        The number of days for which utilization metrics were analyzed for the volume.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lookBackPeriodInDays(Double lookBackPeriodInDays);

        /**
         * <p>
         * An array of objects that describe the recommendation options for the volume.
         * </p>
         * 
         * @param volumeRecommendationOptions
         *        An array of objects that describe the recommendation options for the volume.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder volumeRecommendationOptions(Collection<VolumeRecommendationOption> volumeRecommendationOptions);

        /**
         * <p>
         * An array of objects that describe the recommendation options for the volume.
         * </p>
         * 
         * @param volumeRecommendationOptions
         *        An array of objects that describe the recommendation options for the volume.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder volumeRecommendationOptions(VolumeRecommendationOption... volumeRecommendationOptions);

        /**
         * <p>
         * An array of objects that describe the recommendation options for the volume.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.computeoptimizer.model.VolumeRecommendationOption.Builder} avoiding
         * the need to create one manually via
         * {@link software.amazon.awssdk.services.computeoptimizer.model.VolumeRecommendationOption#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.computeoptimizer.model.VolumeRecommendationOption.Builder#build()} is
         * called immediately and its result is passed to {@link
         * #volumeRecommendationOptions(List<VolumeRecommendationOption>)}.
         * 
         * @param volumeRecommendationOptions
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.computeoptimizer.model.VolumeRecommendationOption.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #volumeRecommendationOptions(java.util.Collection<VolumeRecommendationOption>)
         */
        Builder volumeRecommendationOptions(Consumer<VolumeRecommendationOption.Builder>... volumeRecommendationOptions);

        /**
         * <p>
         * The timestamp of when the volume recommendation was last generated.
         * </p>
         * 
         * @param lastRefreshTimestamp
         *        The timestamp of when the volume recommendation was last generated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastRefreshTimestamp(Instant lastRefreshTimestamp);

        /**
         * <p>
         * The risk of the current EBS volume not meeting the performance needs of its workloads. The higher the risk,
         * the more likely the current EBS volume doesn't have sufficient capacity.
         * </p>
         * 
         * @param currentPerformanceRisk
         *        The risk of the current EBS volume not meeting the performance needs of its workloads. The higher the
         *        risk, the more likely the current EBS volume doesn't have sufficient capacity.
         * @see CurrentPerformanceRisk
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CurrentPerformanceRisk
         */
        Builder currentPerformanceRisk(String currentPerformanceRisk);

        /**
         * <p>
         * The risk of the current EBS volume not meeting the performance needs of its workloads. The higher the risk,
         * the more likely the current EBS volume doesn't have sufficient capacity.
         * </p>
         * 
         * @param currentPerformanceRisk
         *        The risk of the current EBS volume not meeting the performance needs of its workloads. The higher the
         *        risk, the more likely the current EBS volume doesn't have sufficient capacity.
         * @see CurrentPerformanceRisk
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CurrentPerformanceRisk
         */
        Builder currentPerformanceRisk(CurrentPerformanceRisk currentPerformanceRisk);

        /**
         * <p>
         * A list of tags assigned to your Amazon EBS volume recommendations.
         * </p>
         * 
         * @param tags
         *        A list of tags assigned to your Amazon EBS volume recommendations.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * A list of tags assigned to your Amazon EBS volume recommendations.
         * </p>
         * 
         * @param tags
         *        A list of tags assigned to your Amazon EBS volume recommendations.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * A list of tags assigned to your Amazon EBS volume recommendations.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.computeoptimizer.model.Tag.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.computeoptimizer.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.computeoptimizer.model.Tag.Builder#build()} is called immediately and
         * its result is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.computeoptimizer.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * Describes the effective recommendation preferences for Amazon EBS volume.
         * </p>
         * 
         * @param effectiveRecommendationPreferences
         *        Describes the effective recommendation preferences for Amazon EBS volume.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder effectiveRecommendationPreferences(EBSEffectiveRecommendationPreferences effectiveRecommendationPreferences);

        /**
         * <p>
         * Describes the effective recommendation preferences for Amazon EBS volume.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link EBSEffectiveRecommendationPreferences.Builder} avoiding the need to create one manually via
         * {@link EBSEffectiveRecommendationPreferences#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link EBSEffectiveRecommendationPreferences.Builder#build()} is called
         * immediately and its result is passed to
         * {@link #effectiveRecommendationPreferences(EBSEffectiveRecommendationPreferences)}.
         * 
         * @param effectiveRecommendationPreferences
         *        a consumer that will call methods on {@link EBSEffectiveRecommendationPreferences.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #effectiveRecommendationPreferences(EBSEffectiveRecommendationPreferences)
         */
        default Builder effectiveRecommendationPreferences(
                Consumer<EBSEffectiveRecommendationPreferences.Builder> effectiveRecommendationPreferences) {
            return effectiveRecommendationPreferences(EBSEffectiveRecommendationPreferences.builder()
                    .applyMutation(effectiveRecommendationPreferences).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String volumeArn;

        private String accountId;

        private VolumeConfiguration currentConfiguration;

        private String finding;

        private List<EBSUtilizationMetric> utilizationMetrics = DefaultSdkAutoConstructList.getInstance();

        private Double lookBackPeriodInDays;

        private List<VolumeRecommendationOption> volumeRecommendationOptions = DefaultSdkAutoConstructList.getInstance();

        private Instant lastRefreshTimestamp;

        private String currentPerformanceRisk;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private EBSEffectiveRecommendationPreferences effectiveRecommendationPreferences;

        private BuilderImpl() {
        }

        private BuilderImpl(VolumeRecommendation model) {
            volumeArn(model.volumeArn);
            accountId(model.accountId);
            currentConfiguration(model.currentConfiguration);
            finding(model.finding);
            utilizationMetrics(model.utilizationMetrics);
            lookBackPeriodInDays(model.lookBackPeriodInDays);
            volumeRecommendationOptions(model.volumeRecommendationOptions);
            lastRefreshTimestamp(model.lastRefreshTimestamp);
            currentPerformanceRisk(model.currentPerformanceRisk);
            tags(model.tags);
            effectiveRecommendationPreferences(model.effectiveRecommendationPreferences);
        }

        public final String getVolumeArn() {
            return volumeArn;
        }

        public final void setVolumeArn(String volumeArn) {
            this.volumeArn = volumeArn;
        }

        @Override
        public final Builder volumeArn(String volumeArn) {
            this.volumeArn = volumeArn;
            return this;
        }

        public final String getAccountId() {
            return accountId;
        }

        public final void setAccountId(String accountId) {
            this.accountId = accountId;
        }

        @Override
        public final Builder accountId(String accountId) {
            this.accountId = accountId;
            return this;
        }

        public final VolumeConfiguration.Builder getCurrentConfiguration() {
            return currentConfiguration != null ? currentConfiguration.toBuilder() : null;
        }

        public final void setCurrentConfiguration(VolumeConfiguration.BuilderImpl currentConfiguration) {
            this.currentConfiguration = currentConfiguration != null ? currentConfiguration.build() : null;
        }

        @Override
        public final Builder currentConfiguration(VolumeConfiguration currentConfiguration) {
            this.currentConfiguration = currentConfiguration;
            return this;
        }

        public final String getFinding() {
            return finding;
        }

        public final void setFinding(String finding) {
            this.finding = finding;
        }

        @Override
        public final Builder finding(String finding) {
            this.finding = finding;
            return this;
        }

        @Override
        public final Builder finding(EBSFinding finding) {
            this.finding(finding == null ? null : finding.toString());
            return this;
        }

        public final List<EBSUtilizationMetric.Builder> getUtilizationMetrics() {
            List<EBSUtilizationMetric.Builder> result = EBSUtilizationMetricsCopier.copyToBuilder(this.utilizationMetrics);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setUtilizationMetrics(Collection<EBSUtilizationMetric.BuilderImpl> utilizationMetrics) {
            this.utilizationMetrics = EBSUtilizationMetricsCopier.copyFromBuilder(utilizationMetrics);
        }

        @Override
        public final Builder utilizationMetrics(Collection<EBSUtilizationMetric> utilizationMetrics) {
            this.utilizationMetrics = EBSUtilizationMetricsCopier.copy(utilizationMetrics);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder utilizationMetrics(EBSUtilizationMetric... utilizationMetrics) {
            utilizationMetrics(Arrays.asList(utilizationMetrics));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder utilizationMetrics(Consumer<EBSUtilizationMetric.Builder>... utilizationMetrics) {
            utilizationMetrics(Stream.of(utilizationMetrics).map(c -> EBSUtilizationMetric.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final Double getLookBackPeriodInDays() {
            return lookBackPeriodInDays;
        }

        public final void setLookBackPeriodInDays(Double lookBackPeriodInDays) {
            this.lookBackPeriodInDays = lookBackPeriodInDays;
        }

        @Override
        public final Builder lookBackPeriodInDays(Double lookBackPeriodInDays) {
            this.lookBackPeriodInDays = lookBackPeriodInDays;
            return this;
        }

        public final List<VolumeRecommendationOption.Builder> getVolumeRecommendationOptions() {
            List<VolumeRecommendationOption.Builder> result = VolumeRecommendationOptionsCopier
                    .copyToBuilder(this.volumeRecommendationOptions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setVolumeRecommendationOptions(
                Collection<VolumeRecommendationOption.BuilderImpl> volumeRecommendationOptions) {
            this.volumeRecommendationOptions = VolumeRecommendationOptionsCopier.copyFromBuilder(volumeRecommendationOptions);
        }

        @Override
        public final Builder volumeRecommendationOptions(Collection<VolumeRecommendationOption> volumeRecommendationOptions) {
            this.volumeRecommendationOptions = VolumeRecommendationOptionsCopier.copy(volumeRecommendationOptions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder volumeRecommendationOptions(VolumeRecommendationOption... volumeRecommendationOptions) {
            volumeRecommendationOptions(Arrays.asList(volumeRecommendationOptions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder volumeRecommendationOptions(
                Consumer<VolumeRecommendationOption.Builder>... volumeRecommendationOptions) {
            volumeRecommendationOptions(Stream.of(volumeRecommendationOptions)
                    .map(c -> VolumeRecommendationOption.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final Instant getLastRefreshTimestamp() {
            return lastRefreshTimestamp;
        }

        public final void setLastRefreshTimestamp(Instant lastRefreshTimestamp) {
            this.lastRefreshTimestamp = lastRefreshTimestamp;
        }

        @Override
        public final Builder lastRefreshTimestamp(Instant lastRefreshTimestamp) {
            this.lastRefreshTimestamp = lastRefreshTimestamp;
            return this;
        }

        public final String getCurrentPerformanceRisk() {
            return currentPerformanceRisk;
        }

        public final void setCurrentPerformanceRisk(String currentPerformanceRisk) {
            this.currentPerformanceRisk = currentPerformanceRisk;
        }

        @Override
        public final Builder currentPerformanceRisk(String currentPerformanceRisk) {
            this.currentPerformanceRisk = currentPerformanceRisk;
            return this;
        }

        @Override
        public final Builder currentPerformanceRisk(CurrentPerformanceRisk currentPerformanceRisk) {
            this.currentPerformanceRisk(currentPerformanceRisk == null ? null : currentPerformanceRisk.toString());
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagsCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagsCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagsCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final EBSEffectiveRecommendationPreferences.Builder getEffectiveRecommendationPreferences() {
            return effectiveRecommendationPreferences != null ? effectiveRecommendationPreferences.toBuilder() : null;
        }

        public final void setEffectiveRecommendationPreferences(
                EBSEffectiveRecommendationPreferences.BuilderImpl effectiveRecommendationPreferences) {
            this.effectiveRecommendationPreferences = effectiveRecommendationPreferences != null ? effectiveRecommendationPreferences
                    .build() : null;
        }

        @Override
        public final Builder effectiveRecommendationPreferences(
                EBSEffectiveRecommendationPreferences effectiveRecommendationPreferences) {
            this.effectiveRecommendationPreferences = effectiveRecommendationPreferences;
            return this;
        }

        @Override
        public VolumeRecommendation build() {
            return new VolumeRecommendation(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
