/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.computeoptimizer.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes an Lambda function recommendation.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class LambdaFunctionRecommendation implements SdkPojo, Serializable,
        ToCopyableBuilder<LambdaFunctionRecommendation.Builder, LambdaFunctionRecommendation> {
    private static final SdkField<String> FUNCTION_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("functionArn").getter(getter(LambdaFunctionRecommendation::functionArn))
            .setter(setter(Builder::functionArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("functionArn").build()).build();

    private static final SdkField<String> FUNCTION_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("functionVersion").getter(getter(LambdaFunctionRecommendation::functionVersion))
            .setter(setter(Builder::functionVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("functionVersion").build()).build();

    private static final SdkField<String> ACCOUNT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("accountId").getter(getter(LambdaFunctionRecommendation::accountId)).setter(setter(Builder::accountId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("accountId").build()).build();

    private static final SdkField<Integer> CURRENT_MEMORY_SIZE_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("currentMemorySize").getter(getter(LambdaFunctionRecommendation::currentMemorySize))
            .setter(setter(Builder::currentMemorySize))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("currentMemorySize").build()).build();

    private static final SdkField<Long> NUMBER_OF_INVOCATIONS_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("numberOfInvocations").getter(getter(LambdaFunctionRecommendation::numberOfInvocations))
            .setter(setter(Builder::numberOfInvocations))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("numberOfInvocations").build())
            .build();

    private static final SdkField<List<LambdaFunctionUtilizationMetric>> UTILIZATION_METRICS_FIELD = SdkField
            .<List<LambdaFunctionUtilizationMetric>> builder(MarshallingType.LIST)
            .memberName("utilizationMetrics")
            .getter(getter(LambdaFunctionRecommendation::utilizationMetrics))
            .setter(setter(Builder::utilizationMetrics))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("utilizationMetrics").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<LambdaFunctionUtilizationMetric> builder(MarshallingType.SDK_POJO)
                                            .constructor(LambdaFunctionUtilizationMetric::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Double> LOOKBACK_PERIOD_IN_DAYS_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .memberName("lookbackPeriodInDays").getter(getter(LambdaFunctionRecommendation::lookbackPeriodInDays))
            .setter(setter(Builder::lookbackPeriodInDays))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lookbackPeriodInDays").build())
            .build();

    private static final SdkField<Instant> LAST_REFRESH_TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("lastRefreshTimestamp").getter(getter(LambdaFunctionRecommendation::lastRefreshTimestamp))
            .setter(setter(Builder::lastRefreshTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lastRefreshTimestamp").build())
            .build();

    private static final SdkField<String> FINDING_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("finding")
            .getter(getter(LambdaFunctionRecommendation::findingAsString)).setter(setter(Builder::finding))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("finding").build()).build();

    private static final SdkField<List<String>> FINDING_REASON_CODES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("findingReasonCodes")
            .getter(getter(LambdaFunctionRecommendation::findingReasonCodesAsStrings))
            .setter(setter(Builder::findingReasonCodesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("findingReasonCodes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<LambdaFunctionMemoryRecommendationOption>> MEMORY_SIZE_RECOMMENDATION_OPTIONS_FIELD = SdkField
            .<List<LambdaFunctionMemoryRecommendationOption>> builder(MarshallingType.LIST)
            .memberName("memorySizeRecommendationOptions")
            .getter(getter(LambdaFunctionRecommendation::memorySizeRecommendationOptions))
            .setter(setter(Builder::memorySizeRecommendationOptions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("memorySizeRecommendationOptions")
                    .build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<LambdaFunctionMemoryRecommendationOption> builder(MarshallingType.SDK_POJO)
                                            .constructor(LambdaFunctionMemoryRecommendationOption::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> CURRENT_PERFORMANCE_RISK_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("currentPerformanceRisk").getter(getter(LambdaFunctionRecommendation::currentPerformanceRiskAsString))
            .setter(setter(Builder::currentPerformanceRisk))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("currentPerformanceRisk").build())
            .build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("tags")
            .getter(getter(LambdaFunctionRecommendation::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<LambdaEffectiveRecommendationPreferences> EFFECTIVE_RECOMMENDATION_PREFERENCES_FIELD = SdkField
            .<LambdaEffectiveRecommendationPreferences> builder(MarshallingType.SDK_POJO)
            .memberName("effectiveRecommendationPreferences")
            .getter(getter(LambdaFunctionRecommendation::effectiveRecommendationPreferences))
            .setter(setter(Builder::effectiveRecommendationPreferences))
            .constructor(LambdaEffectiveRecommendationPreferences::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("effectiveRecommendationPreferences")
                    .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(FUNCTION_ARN_FIELD,
            FUNCTION_VERSION_FIELD, ACCOUNT_ID_FIELD, CURRENT_MEMORY_SIZE_FIELD, NUMBER_OF_INVOCATIONS_FIELD,
            UTILIZATION_METRICS_FIELD, LOOKBACK_PERIOD_IN_DAYS_FIELD, LAST_REFRESH_TIMESTAMP_FIELD, FINDING_FIELD,
            FINDING_REASON_CODES_FIELD, MEMORY_SIZE_RECOMMENDATION_OPTIONS_FIELD, CURRENT_PERFORMANCE_RISK_FIELD, TAGS_FIELD,
            EFFECTIVE_RECOMMENDATION_PREFERENCES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String functionArn;

    private final String functionVersion;

    private final String accountId;

    private final Integer currentMemorySize;

    private final Long numberOfInvocations;

    private final List<LambdaFunctionUtilizationMetric> utilizationMetrics;

    private final Double lookbackPeriodInDays;

    private final Instant lastRefreshTimestamp;

    private final String finding;

    private final List<String> findingReasonCodes;

    private final List<LambdaFunctionMemoryRecommendationOption> memorySizeRecommendationOptions;

    private final String currentPerformanceRisk;

    private final List<Tag> tags;

    private final LambdaEffectiveRecommendationPreferences effectiveRecommendationPreferences;

    private LambdaFunctionRecommendation(BuilderImpl builder) {
        this.functionArn = builder.functionArn;
        this.functionVersion = builder.functionVersion;
        this.accountId = builder.accountId;
        this.currentMemorySize = builder.currentMemorySize;
        this.numberOfInvocations = builder.numberOfInvocations;
        this.utilizationMetrics = builder.utilizationMetrics;
        this.lookbackPeriodInDays = builder.lookbackPeriodInDays;
        this.lastRefreshTimestamp = builder.lastRefreshTimestamp;
        this.finding = builder.finding;
        this.findingReasonCodes = builder.findingReasonCodes;
        this.memorySizeRecommendationOptions = builder.memorySizeRecommendationOptions;
        this.currentPerformanceRisk = builder.currentPerformanceRisk;
        this.tags = builder.tags;
        this.effectiveRecommendationPreferences = builder.effectiveRecommendationPreferences;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the current function.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the current function.
     */
    public final String functionArn() {
        return functionArn;
    }

    /**
     * <p>
     * The version number of the current function.
     * </p>
     * 
     * @return The version number of the current function.
     */
    public final String functionVersion() {
        return functionVersion;
    }

    /**
     * <p>
     * The Amazon Web Services account ID of the function.
     * </p>
     * 
     * @return The Amazon Web Services account ID of the function.
     */
    public final String accountId() {
        return accountId;
    }

    /**
     * <p>
     * The amount of memory, in MB, that's allocated to the current function.
     * </p>
     * 
     * @return The amount of memory, in MB, that's allocated to the current function.
     */
    public final Integer currentMemorySize() {
        return currentMemorySize;
    }

    /**
     * <p>
     * The number of times your function code was applied during the look-back period.
     * </p>
     * 
     * @return The number of times your function code was applied during the look-back period.
     */
    public final Long numberOfInvocations() {
        return numberOfInvocations;
    }

    /**
     * For responses, this returns true if the service returned a value for the UtilizationMetrics property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasUtilizationMetrics() {
        return utilizationMetrics != null && !(utilizationMetrics instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of objects that describe the utilization metrics of the function.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasUtilizationMetrics} method.
     * </p>
     * 
     * @return An array of objects that describe the utilization metrics of the function.
     */
    public final List<LambdaFunctionUtilizationMetric> utilizationMetrics() {
        return utilizationMetrics;
    }

    /**
     * <p>
     * The number of days for which utilization metrics were analyzed for the function.
     * </p>
     * 
     * @return The number of days for which utilization metrics were analyzed for the function.
     */
    public final Double lookbackPeriodInDays() {
        return lookbackPeriodInDays;
    }

    /**
     * <p>
     * The timestamp of when the function recommendation was last generated.
     * </p>
     * 
     * @return The timestamp of when the function recommendation was last generated.
     */
    public final Instant lastRefreshTimestamp() {
        return lastRefreshTimestamp;
    }

    /**
     * <p>
     * The finding classification of the function.
     * </p>
     * <p>
     * Findings for functions include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b> <code>Optimized</code> </b> — The function is correctly provisioned to run your workload based on its current
     * configuration and its utilization history. This finding classification does not include finding reason codes.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>NotOptimized</code> </b> — The function is performing at a higher level (over-provisioned) or at a
     * lower level (under-provisioned) than required for your workload because its current configuration is not optimal.
     * Over-provisioned resources might lead to unnecessary infrastructure cost, and under-provisioned resources might
     * lead to poor application performance. This finding classification can include the
     * <code>MemoryUnderprovisioned</code> and <code>MemoryUnderprovisioned</code> finding reason codes.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>Unavailable</code> </b> — Compute Optimizer was unable to generate a recommendation for the function.
     * This could be because the function has not accumulated sufficient metric data, or the function does not qualify
     * for a recommendation. This finding classification can include the <code>InsufficientData</code> and
     * <code>Inconclusive</code> finding reason codes.
     * </p>
     * <note>
     * <p>
     * Functions with a finding of unavailable are not returned unless you specify the <code>filter</code> parameter
     * with a value of <code>Unavailable</code> in your <code>GetLambdaFunctionRecommendations</code> request.
     * </p>
     * </note></li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #finding} will
     * return {@link LambdaFunctionRecommendationFinding#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service
     * is available from {@link #findingAsString}.
     * </p>
     * 
     * @return The finding classification of the function.</p>
     *         <p>
     *         Findings for functions include:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b> <code>Optimized</code> </b> — The function is correctly provisioned to run your workload based on its
     *         current configuration and its utilization history. This finding classification does not include finding
     *         reason codes.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>NotOptimized</code> </b> — The function is performing at a higher level (over-provisioned) or
     *         at a lower level (under-provisioned) than required for your workload because its current configuration is
     *         not optimal. Over-provisioned resources might lead to unnecessary infrastructure cost, and
     *         under-provisioned resources might lead to poor application performance. This finding classification can
     *         include the <code>MemoryUnderprovisioned</code> and <code>MemoryUnderprovisioned</code> finding reason
     *         codes.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>Unavailable</code> </b> — Compute Optimizer was unable to generate a recommendation for the
     *         function. This could be because the function has not accumulated sufficient metric data, or the function
     *         does not qualify for a recommendation. This finding classification can include the
     *         <code>InsufficientData</code> and <code>Inconclusive</code> finding reason codes.
     *         </p>
     *         <note>
     *         <p>
     *         Functions with a finding of unavailable are not returned unless you specify the <code>filter</code>
     *         parameter with a value of <code>Unavailable</code> in your <code>GetLambdaFunctionRecommendations</code>
     *         request.
     *         </p>
     *         </note></li>
     * @see LambdaFunctionRecommendationFinding
     */
    public final LambdaFunctionRecommendationFinding finding() {
        return LambdaFunctionRecommendationFinding.fromValue(finding);
    }

    /**
     * <p>
     * The finding classification of the function.
     * </p>
     * <p>
     * Findings for functions include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b> <code>Optimized</code> </b> — The function is correctly provisioned to run your workload based on its current
     * configuration and its utilization history. This finding classification does not include finding reason codes.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>NotOptimized</code> </b> — The function is performing at a higher level (over-provisioned) or at a
     * lower level (under-provisioned) than required for your workload because its current configuration is not optimal.
     * Over-provisioned resources might lead to unnecessary infrastructure cost, and under-provisioned resources might
     * lead to poor application performance. This finding classification can include the
     * <code>MemoryUnderprovisioned</code> and <code>MemoryUnderprovisioned</code> finding reason codes.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>Unavailable</code> </b> — Compute Optimizer was unable to generate a recommendation for the function.
     * This could be because the function has not accumulated sufficient metric data, or the function does not qualify
     * for a recommendation. This finding classification can include the <code>InsufficientData</code> and
     * <code>Inconclusive</code> finding reason codes.
     * </p>
     * <note>
     * <p>
     * Functions with a finding of unavailable are not returned unless you specify the <code>filter</code> parameter
     * with a value of <code>Unavailable</code> in your <code>GetLambdaFunctionRecommendations</code> request.
     * </p>
     * </note></li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #finding} will
     * return {@link LambdaFunctionRecommendationFinding#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service
     * is available from {@link #findingAsString}.
     * </p>
     * 
     * @return The finding classification of the function.</p>
     *         <p>
     *         Findings for functions include:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b> <code>Optimized</code> </b> — The function is correctly provisioned to run your workload based on its
     *         current configuration and its utilization history. This finding classification does not include finding
     *         reason codes.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>NotOptimized</code> </b> — The function is performing at a higher level (over-provisioned) or
     *         at a lower level (under-provisioned) than required for your workload because its current configuration is
     *         not optimal. Over-provisioned resources might lead to unnecessary infrastructure cost, and
     *         under-provisioned resources might lead to poor application performance. This finding classification can
     *         include the <code>MemoryUnderprovisioned</code> and <code>MemoryUnderprovisioned</code> finding reason
     *         codes.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>Unavailable</code> </b> — Compute Optimizer was unable to generate a recommendation for the
     *         function. This could be because the function has not accumulated sufficient metric data, or the function
     *         does not qualify for a recommendation. This finding classification can include the
     *         <code>InsufficientData</code> and <code>Inconclusive</code> finding reason codes.
     *         </p>
     *         <note>
     *         <p>
     *         Functions with a finding of unavailable are not returned unless you specify the <code>filter</code>
     *         parameter with a value of <code>Unavailable</code> in your <code>GetLambdaFunctionRecommendations</code>
     *         request.
     *         </p>
     *         </note></li>
     * @see LambdaFunctionRecommendationFinding
     */
    public final String findingAsString() {
        return finding;
    }

    /**
     * <p>
     * The reason for the finding classification of the function.
     * </p>
     * <note>
     * <p>
     * Functions that have a finding classification of <code>Optimized</code> don't have a finding reason code.
     * </p>
     * </note>
     * <p>
     * Finding reason codes for functions include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b> <code>MemoryOverprovisioned</code> </b> — The function is over-provisioned when its memory configuration can
     * be sized down while still meeting the performance requirements of your workload. An over-provisioned function
     * might lead to unnecessary infrastructure cost. This finding reason code is part of the <code>NotOptimized</code>
     * finding classification.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>MemoryUnderprovisioned</code> </b> — The function is under-provisioned when its memory configuration
     * doesn't meet the performance requirements of the workload. An under-provisioned function might lead to poor
     * application performance. This finding reason code is part of the <code>NotOptimized</code> finding
     * classification.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>InsufficientData</code> </b> — The function does not have sufficient metric data for Compute Optimizer
     * to generate a recommendation. For more information, see the <a
     * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/requirements.html">Supported resources and
     * requirements</a> in the <i>Compute Optimizer User Guide</i>. This finding reason code is part of the
     * <code>Unavailable</code> finding classification.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>Inconclusive</code> </b> — The function does not qualify for a recommendation because Compute Optimizer
     * cannot generate a recommendation with a high degree of confidence. This finding reason code is part of the
     * <code>Unavailable</code> finding classification.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasFindingReasonCodes} method.
     * </p>
     * 
     * @return The reason for the finding classification of the function.</p> <note>
     *         <p>
     *         Functions that have a finding classification of <code>Optimized</code> don't have a finding reason code.
     *         </p>
     *         </note>
     *         <p>
     *         Finding reason codes for functions include:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b> <code>MemoryOverprovisioned</code> </b> — The function is over-provisioned when its memory
     *         configuration can be sized down while still meeting the performance requirements of your workload. An
     *         over-provisioned function might lead to unnecessary infrastructure cost. This finding reason code is part
     *         of the <code>NotOptimized</code> finding classification.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>MemoryUnderprovisioned</code> </b> — The function is under-provisioned when its memory
     *         configuration doesn't meet the performance requirements of the workload. An under-provisioned function
     *         might lead to poor application performance. This finding reason code is part of the
     *         <code>NotOptimized</code> finding classification.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>InsufficientData</code> </b> — The function does not have sufficient metric data for Compute
     *         Optimizer to generate a recommendation. For more information, see the <a
     *         href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/requirements.html">Supported resources and
     *         requirements</a> in the <i>Compute Optimizer User Guide</i>. This finding reason code is part of the
     *         <code>Unavailable</code> finding classification.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>Inconclusive</code> </b> — The function does not qualify for a recommendation because Compute
     *         Optimizer cannot generate a recommendation with a high degree of confidence. This finding reason code is
     *         part of the <code>Unavailable</code> finding classification.
     *         </p>
     *         </li>
     */
    public final List<LambdaFunctionRecommendationFindingReasonCode> findingReasonCodes() {
        return LambdaFunctionRecommendationFindingReasonCodesCopier.copyStringToEnum(findingReasonCodes);
    }

    /**
     * For responses, this returns true if the service returned a value for the FindingReasonCodes property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasFindingReasonCodes() {
        return findingReasonCodes != null && !(findingReasonCodes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The reason for the finding classification of the function.
     * </p>
     * <note>
     * <p>
     * Functions that have a finding classification of <code>Optimized</code> don't have a finding reason code.
     * </p>
     * </note>
     * <p>
     * Finding reason codes for functions include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b> <code>MemoryOverprovisioned</code> </b> — The function is over-provisioned when its memory configuration can
     * be sized down while still meeting the performance requirements of your workload. An over-provisioned function
     * might lead to unnecessary infrastructure cost. This finding reason code is part of the <code>NotOptimized</code>
     * finding classification.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>MemoryUnderprovisioned</code> </b> — The function is under-provisioned when its memory configuration
     * doesn't meet the performance requirements of the workload. An under-provisioned function might lead to poor
     * application performance. This finding reason code is part of the <code>NotOptimized</code> finding
     * classification.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>InsufficientData</code> </b> — The function does not have sufficient metric data for Compute Optimizer
     * to generate a recommendation. For more information, see the <a
     * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/requirements.html">Supported resources and
     * requirements</a> in the <i>Compute Optimizer User Guide</i>. This finding reason code is part of the
     * <code>Unavailable</code> finding classification.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>Inconclusive</code> </b> — The function does not qualify for a recommendation because Compute Optimizer
     * cannot generate a recommendation with a high degree of confidence. This finding reason code is part of the
     * <code>Unavailable</code> finding classification.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasFindingReasonCodes} method.
     * </p>
     * 
     * @return The reason for the finding classification of the function.</p> <note>
     *         <p>
     *         Functions that have a finding classification of <code>Optimized</code> don't have a finding reason code.
     *         </p>
     *         </note>
     *         <p>
     *         Finding reason codes for functions include:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b> <code>MemoryOverprovisioned</code> </b> — The function is over-provisioned when its memory
     *         configuration can be sized down while still meeting the performance requirements of your workload. An
     *         over-provisioned function might lead to unnecessary infrastructure cost. This finding reason code is part
     *         of the <code>NotOptimized</code> finding classification.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>MemoryUnderprovisioned</code> </b> — The function is under-provisioned when its memory
     *         configuration doesn't meet the performance requirements of the workload. An under-provisioned function
     *         might lead to poor application performance. This finding reason code is part of the
     *         <code>NotOptimized</code> finding classification.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>InsufficientData</code> </b> — The function does not have sufficient metric data for Compute
     *         Optimizer to generate a recommendation. For more information, see the <a
     *         href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/requirements.html">Supported resources and
     *         requirements</a> in the <i>Compute Optimizer User Guide</i>. This finding reason code is part of the
     *         <code>Unavailable</code> finding classification.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>Inconclusive</code> </b> — The function does not qualify for a recommendation because Compute
     *         Optimizer cannot generate a recommendation with a high degree of confidence. This finding reason code is
     *         part of the <code>Unavailable</code> finding classification.
     *         </p>
     *         </li>
     */
    public final List<String> findingReasonCodesAsStrings() {
        return findingReasonCodes;
    }

    /**
     * For responses, this returns true if the service returned a value for the MemorySizeRecommendationOptions
     * property. This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()}
     * method on the property). This is useful because the SDK will never return a null collection or map, but you may
     * need to differentiate between the service returning nothing (or null) and the service returning an empty
     * collection or map. For requests, this returns true if a value for the property was specified in the request
     * builder, and false if a value was not specified.
     */
    public final boolean hasMemorySizeRecommendationOptions() {
        return memorySizeRecommendationOptions != null && !(memorySizeRecommendationOptions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of objects that describe the memory configuration recommendation options for the function.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasMemorySizeRecommendationOptions}
     * method.
     * </p>
     * 
     * @return An array of objects that describe the memory configuration recommendation options for the function.
     */
    public final List<LambdaFunctionMemoryRecommendationOption> memorySizeRecommendationOptions() {
        return memorySizeRecommendationOptions;
    }

    /**
     * <p>
     * The risk of the current Lambda function not meeting the performance needs of its workloads. The higher the risk,
     * the more likely the current Lambda function requires more memory.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #currentPerformanceRisk} will return {@link CurrentPerformanceRisk#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #currentPerformanceRiskAsString}.
     * </p>
     * 
     * @return The risk of the current Lambda function not meeting the performance needs of its workloads. The higher
     *         the risk, the more likely the current Lambda function requires more memory.
     * @see CurrentPerformanceRisk
     */
    public final CurrentPerformanceRisk currentPerformanceRisk() {
        return CurrentPerformanceRisk.fromValue(currentPerformanceRisk);
    }

    /**
     * <p>
     * The risk of the current Lambda function not meeting the performance needs of its workloads. The higher the risk,
     * the more likely the current Lambda function requires more memory.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #currentPerformanceRisk} will return {@link CurrentPerformanceRisk#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #currentPerformanceRiskAsString}.
     * </p>
     * 
     * @return The risk of the current Lambda function not meeting the performance needs of its workloads. The higher
     *         the risk, the more likely the current Lambda function requires more memory.
     * @see CurrentPerformanceRisk
     */
    public final String currentPerformanceRiskAsString() {
        return currentPerformanceRisk;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of tags assigned to your Lambda function recommendations.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return A list of tags assigned to your Lambda function recommendations.
     */
    public final List<Tag> tags() {
        return tags;
    }

    /**
     * <p>
     * Describes the effective recommendation preferences for Lambda functions.
     * </p>
     * 
     * @return Describes the effective recommendation preferences for Lambda functions.
     */
    public final LambdaEffectiveRecommendationPreferences effectiveRecommendationPreferences() {
        return effectiveRecommendationPreferences;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(functionArn());
        hashCode = 31 * hashCode + Objects.hashCode(functionVersion());
        hashCode = 31 * hashCode + Objects.hashCode(accountId());
        hashCode = 31 * hashCode + Objects.hashCode(currentMemorySize());
        hashCode = 31 * hashCode + Objects.hashCode(numberOfInvocations());
        hashCode = 31 * hashCode + Objects.hashCode(hasUtilizationMetrics() ? utilizationMetrics() : null);
        hashCode = 31 * hashCode + Objects.hashCode(lookbackPeriodInDays());
        hashCode = 31 * hashCode + Objects.hashCode(lastRefreshTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(findingAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasFindingReasonCodes() ? findingReasonCodesAsStrings() : null);
        hashCode = 31 * hashCode
                + Objects.hashCode(hasMemorySizeRecommendationOptions() ? memorySizeRecommendationOptions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(currentPerformanceRiskAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(effectiveRecommendationPreferences());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof LambdaFunctionRecommendation)) {
            return false;
        }
        LambdaFunctionRecommendation other = (LambdaFunctionRecommendation) obj;
        return Objects.equals(functionArn(), other.functionArn()) && Objects.equals(functionVersion(), other.functionVersion())
                && Objects.equals(accountId(), other.accountId())
                && Objects.equals(currentMemorySize(), other.currentMemorySize())
                && Objects.equals(numberOfInvocations(), other.numberOfInvocations())
                && hasUtilizationMetrics() == other.hasUtilizationMetrics()
                && Objects.equals(utilizationMetrics(), other.utilizationMetrics())
                && Objects.equals(lookbackPeriodInDays(), other.lookbackPeriodInDays())
                && Objects.equals(lastRefreshTimestamp(), other.lastRefreshTimestamp())
                && Objects.equals(findingAsString(), other.findingAsString())
                && hasFindingReasonCodes() == other.hasFindingReasonCodes()
                && Objects.equals(findingReasonCodesAsStrings(), other.findingReasonCodesAsStrings())
                && hasMemorySizeRecommendationOptions() == other.hasMemorySizeRecommendationOptions()
                && Objects.equals(memorySizeRecommendationOptions(), other.memorySizeRecommendationOptions())
                && Objects.equals(currentPerformanceRiskAsString(), other.currentPerformanceRiskAsString())
                && hasTags() == other.hasTags() && Objects.equals(tags(), other.tags())
                && Objects.equals(effectiveRecommendationPreferences(), other.effectiveRecommendationPreferences());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString
                .builder("LambdaFunctionRecommendation")
                .add("FunctionArn", functionArn())
                .add("FunctionVersion", functionVersion())
                .add("AccountId", accountId())
                .add("CurrentMemorySize", currentMemorySize())
                .add("NumberOfInvocations", numberOfInvocations())
                .add("UtilizationMetrics", hasUtilizationMetrics() ? utilizationMetrics() : null)
                .add("LookbackPeriodInDays", lookbackPeriodInDays())
                .add("LastRefreshTimestamp", lastRefreshTimestamp())
                .add("Finding", findingAsString())
                .add("FindingReasonCodes", hasFindingReasonCodes() ? findingReasonCodesAsStrings() : null)
                .add("MemorySizeRecommendationOptions",
                        hasMemorySizeRecommendationOptions() ? memorySizeRecommendationOptions() : null)
                .add("CurrentPerformanceRisk", currentPerformanceRiskAsString()).add("Tags", hasTags() ? tags() : null)
                .add("EffectiveRecommendationPreferences", effectiveRecommendationPreferences()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "functionArn":
            return Optional.ofNullable(clazz.cast(functionArn()));
        case "functionVersion":
            return Optional.ofNullable(clazz.cast(functionVersion()));
        case "accountId":
            return Optional.ofNullable(clazz.cast(accountId()));
        case "currentMemorySize":
            return Optional.ofNullable(clazz.cast(currentMemorySize()));
        case "numberOfInvocations":
            return Optional.ofNullable(clazz.cast(numberOfInvocations()));
        case "utilizationMetrics":
            return Optional.ofNullable(clazz.cast(utilizationMetrics()));
        case "lookbackPeriodInDays":
            return Optional.ofNullable(clazz.cast(lookbackPeriodInDays()));
        case "lastRefreshTimestamp":
            return Optional.ofNullable(clazz.cast(lastRefreshTimestamp()));
        case "finding":
            return Optional.ofNullable(clazz.cast(findingAsString()));
        case "findingReasonCodes":
            return Optional.ofNullable(clazz.cast(findingReasonCodesAsStrings()));
        case "memorySizeRecommendationOptions":
            return Optional.ofNullable(clazz.cast(memorySizeRecommendationOptions()));
        case "currentPerformanceRisk":
            return Optional.ofNullable(clazz.cast(currentPerformanceRiskAsString()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "effectiveRecommendationPreferences":
            return Optional.ofNullable(clazz.cast(effectiveRecommendationPreferences()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<LambdaFunctionRecommendation, T> g) {
        return obj -> g.apply((LambdaFunctionRecommendation) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, LambdaFunctionRecommendation> {
        /**
         * <p>
         * The Amazon Resource Name (ARN) of the current function.
         * </p>
         * 
         * @param functionArn
         *        The Amazon Resource Name (ARN) of the current function.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder functionArn(String functionArn);

        /**
         * <p>
         * The version number of the current function.
         * </p>
         * 
         * @param functionVersion
         *        The version number of the current function.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder functionVersion(String functionVersion);

        /**
         * <p>
         * The Amazon Web Services account ID of the function.
         * </p>
         * 
         * @param accountId
         *        The Amazon Web Services account ID of the function.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accountId(String accountId);

        /**
         * <p>
         * The amount of memory, in MB, that's allocated to the current function.
         * </p>
         * 
         * @param currentMemorySize
         *        The amount of memory, in MB, that's allocated to the current function.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder currentMemorySize(Integer currentMemorySize);

        /**
         * <p>
         * The number of times your function code was applied during the look-back period.
         * </p>
         * 
         * @param numberOfInvocations
         *        The number of times your function code was applied during the look-back period.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder numberOfInvocations(Long numberOfInvocations);

        /**
         * <p>
         * An array of objects that describe the utilization metrics of the function.
         * </p>
         * 
         * @param utilizationMetrics
         *        An array of objects that describe the utilization metrics of the function.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder utilizationMetrics(Collection<LambdaFunctionUtilizationMetric> utilizationMetrics);

        /**
         * <p>
         * An array of objects that describe the utilization metrics of the function.
         * </p>
         * 
         * @param utilizationMetrics
         *        An array of objects that describe the utilization metrics of the function.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder utilizationMetrics(LambdaFunctionUtilizationMetric... utilizationMetrics);

        /**
         * <p>
         * An array of objects that describe the utilization metrics of the function.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.computeoptimizer.model.LambdaFunctionUtilizationMetric.Builder}
         * avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.computeoptimizer.model.LambdaFunctionUtilizationMetric#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.computeoptimizer.model.LambdaFunctionUtilizationMetric.Builder#build()}
         * is called immediately and its result is passed to {@link
         * #utilizationMetrics(List<LambdaFunctionUtilizationMetric>)}.
         * 
         * @param utilizationMetrics
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.computeoptimizer.model.LambdaFunctionUtilizationMetric.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #utilizationMetrics(java.util.Collection<LambdaFunctionUtilizationMetric>)
         */
        Builder utilizationMetrics(Consumer<LambdaFunctionUtilizationMetric.Builder>... utilizationMetrics);

        /**
         * <p>
         * The number of days for which utilization metrics were analyzed for the function.
         * </p>
         * 
         * @param lookbackPeriodInDays
         *        The number of days for which utilization metrics were analyzed for the function.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lookbackPeriodInDays(Double lookbackPeriodInDays);

        /**
         * <p>
         * The timestamp of when the function recommendation was last generated.
         * </p>
         * 
         * @param lastRefreshTimestamp
         *        The timestamp of when the function recommendation was last generated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastRefreshTimestamp(Instant lastRefreshTimestamp);

        /**
         * <p>
         * The finding classification of the function.
         * </p>
         * <p>
         * Findings for functions include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b> <code>Optimized</code> </b> — The function is correctly provisioned to run your workload based on its
         * current configuration and its utilization history. This finding classification does not include finding
         * reason codes.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>NotOptimized</code> </b> — The function is performing at a higher level (over-provisioned) or at a
         * lower level (under-provisioned) than required for your workload because its current configuration is not
         * optimal. Over-provisioned resources might lead to unnecessary infrastructure cost, and under-provisioned
         * resources might lead to poor application performance. This finding classification can include the
         * <code>MemoryUnderprovisioned</code> and <code>MemoryUnderprovisioned</code> finding reason codes.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>Unavailable</code> </b> — Compute Optimizer was unable to generate a recommendation for the
         * function. This could be because the function has not accumulated sufficient metric data, or the function does
         * not qualify for a recommendation. This finding classification can include the <code>InsufficientData</code>
         * and <code>Inconclusive</code> finding reason codes.
         * </p>
         * <note>
         * <p>
         * Functions with a finding of unavailable are not returned unless you specify the <code>filter</code> parameter
         * with a value of <code>Unavailable</code> in your <code>GetLambdaFunctionRecommendations</code> request.
         * </p>
         * </note></li>
         * </ul>
         * 
         * @param finding
         *        The finding classification of the function.</p>
         *        <p>
         *        Findings for functions include:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b> <code>Optimized</code> </b> — The function is correctly provisioned to run your workload based on
         *        its current configuration and its utilization history. This finding classification does not include
         *        finding reason codes.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>NotOptimized</code> </b> — The function is performing at a higher level (over-provisioned)
         *        or at a lower level (under-provisioned) than required for your workload because its current
         *        configuration is not optimal. Over-provisioned resources might lead to unnecessary infrastructure
         *        cost, and under-provisioned resources might lead to poor application performance. This finding
         *        classification can include the <code>MemoryUnderprovisioned</code> and
         *        <code>MemoryUnderprovisioned</code> finding reason codes.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>Unavailable</code> </b> — Compute Optimizer was unable to generate a recommendation for the
         *        function. This could be because the function has not accumulated sufficient metric data, or the
         *        function does not qualify for a recommendation. This finding classification can include the
         *        <code>InsufficientData</code> and <code>Inconclusive</code> finding reason codes.
         *        </p>
         *        <note>
         *        <p>
         *        Functions with a finding of unavailable are not returned unless you specify the <code>filter</code>
         *        parameter with a value of <code>Unavailable</code> in your
         *        <code>GetLambdaFunctionRecommendations</code> request.
         *        </p>
         *        </note></li>
         * @see LambdaFunctionRecommendationFinding
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LambdaFunctionRecommendationFinding
         */
        Builder finding(String finding);

        /**
         * <p>
         * The finding classification of the function.
         * </p>
         * <p>
         * Findings for functions include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b> <code>Optimized</code> </b> — The function is correctly provisioned to run your workload based on its
         * current configuration and its utilization history. This finding classification does not include finding
         * reason codes.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>NotOptimized</code> </b> — The function is performing at a higher level (over-provisioned) or at a
         * lower level (under-provisioned) than required for your workload because its current configuration is not
         * optimal. Over-provisioned resources might lead to unnecessary infrastructure cost, and under-provisioned
         * resources might lead to poor application performance. This finding classification can include the
         * <code>MemoryUnderprovisioned</code> and <code>MemoryUnderprovisioned</code> finding reason codes.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>Unavailable</code> </b> — Compute Optimizer was unable to generate a recommendation for the
         * function. This could be because the function has not accumulated sufficient metric data, or the function does
         * not qualify for a recommendation. This finding classification can include the <code>InsufficientData</code>
         * and <code>Inconclusive</code> finding reason codes.
         * </p>
         * <note>
         * <p>
         * Functions with a finding of unavailable are not returned unless you specify the <code>filter</code> parameter
         * with a value of <code>Unavailable</code> in your <code>GetLambdaFunctionRecommendations</code> request.
         * </p>
         * </note></li>
         * </ul>
         * 
         * @param finding
         *        The finding classification of the function.</p>
         *        <p>
         *        Findings for functions include:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b> <code>Optimized</code> </b> — The function is correctly provisioned to run your workload based on
         *        its current configuration and its utilization history. This finding classification does not include
         *        finding reason codes.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>NotOptimized</code> </b> — The function is performing at a higher level (over-provisioned)
         *        or at a lower level (under-provisioned) than required for your workload because its current
         *        configuration is not optimal. Over-provisioned resources might lead to unnecessary infrastructure
         *        cost, and under-provisioned resources might lead to poor application performance. This finding
         *        classification can include the <code>MemoryUnderprovisioned</code> and
         *        <code>MemoryUnderprovisioned</code> finding reason codes.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>Unavailable</code> </b> — Compute Optimizer was unable to generate a recommendation for the
         *        function. This could be because the function has not accumulated sufficient metric data, or the
         *        function does not qualify for a recommendation. This finding classification can include the
         *        <code>InsufficientData</code> and <code>Inconclusive</code> finding reason codes.
         *        </p>
         *        <note>
         *        <p>
         *        Functions with a finding of unavailable are not returned unless you specify the <code>filter</code>
         *        parameter with a value of <code>Unavailable</code> in your
         *        <code>GetLambdaFunctionRecommendations</code> request.
         *        </p>
         *        </note></li>
         * @see LambdaFunctionRecommendationFinding
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LambdaFunctionRecommendationFinding
         */
        Builder finding(LambdaFunctionRecommendationFinding finding);

        /**
         * <p>
         * The reason for the finding classification of the function.
         * </p>
         * <note>
         * <p>
         * Functions that have a finding classification of <code>Optimized</code> don't have a finding reason code.
         * </p>
         * </note>
         * <p>
         * Finding reason codes for functions include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b> <code>MemoryOverprovisioned</code> </b> — The function is over-provisioned when its memory configuration
         * can be sized down while still meeting the performance requirements of your workload. An over-provisioned
         * function might lead to unnecessary infrastructure cost. This finding reason code is part of the
         * <code>NotOptimized</code> finding classification.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>MemoryUnderprovisioned</code> </b> — The function is under-provisioned when its memory
         * configuration doesn't meet the performance requirements of the workload. An under-provisioned function might
         * lead to poor application performance. This finding reason code is part of the <code>NotOptimized</code>
         * finding classification.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>InsufficientData</code> </b> — The function does not have sufficient metric data for Compute
         * Optimizer to generate a recommendation. For more information, see the <a
         * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/requirements.html">Supported resources and
         * requirements</a> in the <i>Compute Optimizer User Guide</i>. This finding reason code is part of the
         * <code>Unavailable</code> finding classification.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>Inconclusive</code> </b> — The function does not qualify for a recommendation because Compute
         * Optimizer cannot generate a recommendation with a high degree of confidence. This finding reason code is part
         * of the <code>Unavailable</code> finding classification.
         * </p>
         * </li>
         * </ul>
         * 
         * @param findingReasonCodes
         *        The reason for the finding classification of the function.</p> <note>
         *        <p>
         *        Functions that have a finding classification of <code>Optimized</code> don't have a finding reason
         *        code.
         *        </p>
         *        </note>
         *        <p>
         *        Finding reason codes for functions include:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b> <code>MemoryOverprovisioned</code> </b> — The function is over-provisioned when its memory
         *        configuration can be sized down while still meeting the performance requirements of your workload. An
         *        over-provisioned function might lead to unnecessary infrastructure cost. This finding reason code is
         *        part of the <code>NotOptimized</code> finding classification.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>MemoryUnderprovisioned</code> </b> — The function is under-provisioned when its memory
         *        configuration doesn't meet the performance requirements of the workload. An under-provisioned function
         *        might lead to poor application performance. This finding reason code is part of the
         *        <code>NotOptimized</code> finding classification.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>InsufficientData</code> </b> — The function does not have sufficient metric data for Compute
         *        Optimizer to generate a recommendation. For more information, see the <a
         *        href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/requirements.html">Supported resources
         *        and requirements</a> in the <i>Compute Optimizer User Guide</i>. This finding reason code is part of
         *        the <code>Unavailable</code> finding classification.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>Inconclusive</code> </b> — The function does not qualify for a recommendation because
         *        Compute Optimizer cannot generate a recommendation with a high degree of confidence. This finding
         *        reason code is part of the <code>Unavailable</code> finding classification.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder findingReasonCodesWithStrings(Collection<String> findingReasonCodes);

        /**
         * <p>
         * The reason for the finding classification of the function.
         * </p>
         * <note>
         * <p>
         * Functions that have a finding classification of <code>Optimized</code> don't have a finding reason code.
         * </p>
         * </note>
         * <p>
         * Finding reason codes for functions include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b> <code>MemoryOverprovisioned</code> </b> — The function is over-provisioned when its memory configuration
         * can be sized down while still meeting the performance requirements of your workload. An over-provisioned
         * function might lead to unnecessary infrastructure cost. This finding reason code is part of the
         * <code>NotOptimized</code> finding classification.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>MemoryUnderprovisioned</code> </b> — The function is under-provisioned when its memory
         * configuration doesn't meet the performance requirements of the workload. An under-provisioned function might
         * lead to poor application performance. This finding reason code is part of the <code>NotOptimized</code>
         * finding classification.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>InsufficientData</code> </b> — The function does not have sufficient metric data for Compute
         * Optimizer to generate a recommendation. For more information, see the <a
         * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/requirements.html">Supported resources and
         * requirements</a> in the <i>Compute Optimizer User Guide</i>. This finding reason code is part of the
         * <code>Unavailable</code> finding classification.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>Inconclusive</code> </b> — The function does not qualify for a recommendation because Compute
         * Optimizer cannot generate a recommendation with a high degree of confidence. This finding reason code is part
         * of the <code>Unavailable</code> finding classification.
         * </p>
         * </li>
         * </ul>
         * 
         * @param findingReasonCodes
         *        The reason for the finding classification of the function.</p> <note>
         *        <p>
         *        Functions that have a finding classification of <code>Optimized</code> don't have a finding reason
         *        code.
         *        </p>
         *        </note>
         *        <p>
         *        Finding reason codes for functions include:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b> <code>MemoryOverprovisioned</code> </b> — The function is over-provisioned when its memory
         *        configuration can be sized down while still meeting the performance requirements of your workload. An
         *        over-provisioned function might lead to unnecessary infrastructure cost. This finding reason code is
         *        part of the <code>NotOptimized</code> finding classification.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>MemoryUnderprovisioned</code> </b> — The function is under-provisioned when its memory
         *        configuration doesn't meet the performance requirements of the workload. An under-provisioned function
         *        might lead to poor application performance. This finding reason code is part of the
         *        <code>NotOptimized</code> finding classification.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>InsufficientData</code> </b> — The function does not have sufficient metric data for Compute
         *        Optimizer to generate a recommendation. For more information, see the <a
         *        href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/requirements.html">Supported resources
         *        and requirements</a> in the <i>Compute Optimizer User Guide</i>. This finding reason code is part of
         *        the <code>Unavailable</code> finding classification.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>Inconclusive</code> </b> — The function does not qualify for a recommendation because
         *        Compute Optimizer cannot generate a recommendation with a high degree of confidence. This finding
         *        reason code is part of the <code>Unavailable</code> finding classification.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder findingReasonCodesWithStrings(String... findingReasonCodes);

        /**
         * <p>
         * The reason for the finding classification of the function.
         * </p>
         * <note>
         * <p>
         * Functions that have a finding classification of <code>Optimized</code> don't have a finding reason code.
         * </p>
         * </note>
         * <p>
         * Finding reason codes for functions include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b> <code>MemoryOverprovisioned</code> </b> — The function is over-provisioned when its memory configuration
         * can be sized down while still meeting the performance requirements of your workload. An over-provisioned
         * function might lead to unnecessary infrastructure cost. This finding reason code is part of the
         * <code>NotOptimized</code> finding classification.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>MemoryUnderprovisioned</code> </b> — The function is under-provisioned when its memory
         * configuration doesn't meet the performance requirements of the workload. An under-provisioned function might
         * lead to poor application performance. This finding reason code is part of the <code>NotOptimized</code>
         * finding classification.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>InsufficientData</code> </b> — The function does not have sufficient metric data for Compute
         * Optimizer to generate a recommendation. For more information, see the <a
         * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/requirements.html">Supported resources and
         * requirements</a> in the <i>Compute Optimizer User Guide</i>. This finding reason code is part of the
         * <code>Unavailable</code> finding classification.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>Inconclusive</code> </b> — The function does not qualify for a recommendation because Compute
         * Optimizer cannot generate a recommendation with a high degree of confidence. This finding reason code is part
         * of the <code>Unavailable</code> finding classification.
         * </p>
         * </li>
         * </ul>
         * 
         * @param findingReasonCodes
         *        The reason for the finding classification of the function.</p> <note>
         *        <p>
         *        Functions that have a finding classification of <code>Optimized</code> don't have a finding reason
         *        code.
         *        </p>
         *        </note>
         *        <p>
         *        Finding reason codes for functions include:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b> <code>MemoryOverprovisioned</code> </b> — The function is over-provisioned when its memory
         *        configuration can be sized down while still meeting the performance requirements of your workload. An
         *        over-provisioned function might lead to unnecessary infrastructure cost. This finding reason code is
         *        part of the <code>NotOptimized</code> finding classification.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>MemoryUnderprovisioned</code> </b> — The function is under-provisioned when its memory
         *        configuration doesn't meet the performance requirements of the workload. An under-provisioned function
         *        might lead to poor application performance. This finding reason code is part of the
         *        <code>NotOptimized</code> finding classification.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>InsufficientData</code> </b> — The function does not have sufficient metric data for Compute
         *        Optimizer to generate a recommendation. For more information, see the <a
         *        href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/requirements.html">Supported resources
         *        and requirements</a> in the <i>Compute Optimizer User Guide</i>. This finding reason code is part of
         *        the <code>Unavailable</code> finding classification.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>Inconclusive</code> </b> — The function does not qualify for a recommendation because
         *        Compute Optimizer cannot generate a recommendation with a high degree of confidence. This finding
         *        reason code is part of the <code>Unavailable</code> finding classification.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder findingReasonCodes(Collection<LambdaFunctionRecommendationFindingReasonCode> findingReasonCodes);

        /**
         * <p>
         * The reason for the finding classification of the function.
         * </p>
         * <note>
         * <p>
         * Functions that have a finding classification of <code>Optimized</code> don't have a finding reason code.
         * </p>
         * </note>
         * <p>
         * Finding reason codes for functions include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b> <code>MemoryOverprovisioned</code> </b> — The function is over-provisioned when its memory configuration
         * can be sized down while still meeting the performance requirements of your workload. An over-provisioned
         * function might lead to unnecessary infrastructure cost. This finding reason code is part of the
         * <code>NotOptimized</code> finding classification.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>MemoryUnderprovisioned</code> </b> — The function is under-provisioned when its memory
         * configuration doesn't meet the performance requirements of the workload. An under-provisioned function might
         * lead to poor application performance. This finding reason code is part of the <code>NotOptimized</code>
         * finding classification.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>InsufficientData</code> </b> — The function does not have sufficient metric data for Compute
         * Optimizer to generate a recommendation. For more information, see the <a
         * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/requirements.html">Supported resources and
         * requirements</a> in the <i>Compute Optimizer User Guide</i>. This finding reason code is part of the
         * <code>Unavailable</code> finding classification.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>Inconclusive</code> </b> — The function does not qualify for a recommendation because Compute
         * Optimizer cannot generate a recommendation with a high degree of confidence. This finding reason code is part
         * of the <code>Unavailable</code> finding classification.
         * </p>
         * </li>
         * </ul>
         * 
         * @param findingReasonCodes
         *        The reason for the finding classification of the function.</p> <note>
         *        <p>
         *        Functions that have a finding classification of <code>Optimized</code> don't have a finding reason
         *        code.
         *        </p>
         *        </note>
         *        <p>
         *        Finding reason codes for functions include:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b> <code>MemoryOverprovisioned</code> </b> — The function is over-provisioned when its memory
         *        configuration can be sized down while still meeting the performance requirements of your workload. An
         *        over-provisioned function might lead to unnecessary infrastructure cost. This finding reason code is
         *        part of the <code>NotOptimized</code> finding classification.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>MemoryUnderprovisioned</code> </b> — The function is under-provisioned when its memory
         *        configuration doesn't meet the performance requirements of the workload. An under-provisioned function
         *        might lead to poor application performance. This finding reason code is part of the
         *        <code>NotOptimized</code> finding classification.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>InsufficientData</code> </b> — The function does not have sufficient metric data for Compute
         *        Optimizer to generate a recommendation. For more information, see the <a
         *        href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/requirements.html">Supported resources
         *        and requirements</a> in the <i>Compute Optimizer User Guide</i>. This finding reason code is part of
         *        the <code>Unavailable</code> finding classification.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>Inconclusive</code> </b> — The function does not qualify for a recommendation because
         *        Compute Optimizer cannot generate a recommendation with a high degree of confidence. This finding
         *        reason code is part of the <code>Unavailable</code> finding classification.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder findingReasonCodes(LambdaFunctionRecommendationFindingReasonCode... findingReasonCodes);

        /**
         * <p>
         * An array of objects that describe the memory configuration recommendation options for the function.
         * </p>
         * 
         * @param memorySizeRecommendationOptions
         *        An array of objects that describe the memory configuration recommendation options for the function.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder memorySizeRecommendationOptions(
                Collection<LambdaFunctionMemoryRecommendationOption> memorySizeRecommendationOptions);

        /**
         * <p>
         * An array of objects that describe the memory configuration recommendation options for the function.
         * </p>
         * 
         * @param memorySizeRecommendationOptions
         *        An array of objects that describe the memory configuration recommendation options for the function.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder memorySizeRecommendationOptions(LambdaFunctionMemoryRecommendationOption... memorySizeRecommendationOptions);

        /**
         * <p>
         * An array of objects that describe the memory configuration recommendation options for the function.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.computeoptimizer.model.LambdaFunctionMemoryRecommendationOption.Builder}
         * avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.computeoptimizer.model.LambdaFunctionMemoryRecommendationOption#builder()}
         * .
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.computeoptimizer.model.LambdaFunctionMemoryRecommendationOption.Builder#build()}
         * is called immediately and its result is passed to {@link
         * #memorySizeRecommendationOptions(List<LambdaFunctionMemoryRecommendationOption>)}.
         * 
         * @param memorySizeRecommendationOptions
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.computeoptimizer.model.LambdaFunctionMemoryRecommendationOption.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #memorySizeRecommendationOptions(java.util.Collection<LambdaFunctionMemoryRecommendationOption>)
         */
        Builder memorySizeRecommendationOptions(
                Consumer<LambdaFunctionMemoryRecommendationOption.Builder>... memorySizeRecommendationOptions);

        /**
         * <p>
         * The risk of the current Lambda function not meeting the performance needs of its workloads. The higher the
         * risk, the more likely the current Lambda function requires more memory.
         * </p>
         * 
         * @param currentPerformanceRisk
         *        The risk of the current Lambda function not meeting the performance needs of its workloads. The higher
         *        the risk, the more likely the current Lambda function requires more memory.
         * @see CurrentPerformanceRisk
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CurrentPerformanceRisk
         */
        Builder currentPerformanceRisk(String currentPerformanceRisk);

        /**
         * <p>
         * The risk of the current Lambda function not meeting the performance needs of its workloads. The higher the
         * risk, the more likely the current Lambda function requires more memory.
         * </p>
         * 
         * @param currentPerformanceRisk
         *        The risk of the current Lambda function not meeting the performance needs of its workloads. The higher
         *        the risk, the more likely the current Lambda function requires more memory.
         * @see CurrentPerformanceRisk
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CurrentPerformanceRisk
         */
        Builder currentPerformanceRisk(CurrentPerformanceRisk currentPerformanceRisk);

        /**
         * <p>
         * A list of tags assigned to your Lambda function recommendations.
         * </p>
         * 
         * @param tags
         *        A list of tags assigned to your Lambda function recommendations.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * A list of tags assigned to your Lambda function recommendations.
         * </p>
         * 
         * @param tags
         *        A list of tags assigned to your Lambda function recommendations.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * A list of tags assigned to your Lambda function recommendations.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.computeoptimizer.model.Tag.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.computeoptimizer.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.computeoptimizer.model.Tag.Builder#build()} is called immediately and
         * its result is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.computeoptimizer.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * Describes the effective recommendation preferences for Lambda functions.
         * </p>
         * 
         * @param effectiveRecommendationPreferences
         *        Describes the effective recommendation preferences for Lambda functions.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder effectiveRecommendationPreferences(LambdaEffectiveRecommendationPreferences effectiveRecommendationPreferences);

        /**
         * <p>
         * Describes the effective recommendation preferences for Lambda functions.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link LambdaEffectiveRecommendationPreferences.Builder} avoiding the need to create one manually via
         * {@link LambdaEffectiveRecommendationPreferences#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link LambdaEffectiveRecommendationPreferences.Builder#build()} is
         * called immediately and its result is passed to
         * {@link #effectiveRecommendationPreferences(LambdaEffectiveRecommendationPreferences)}.
         * 
         * @param effectiveRecommendationPreferences
         *        a consumer that will call methods on {@link LambdaEffectiveRecommendationPreferences.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #effectiveRecommendationPreferences(LambdaEffectiveRecommendationPreferences)
         */
        default Builder effectiveRecommendationPreferences(
                Consumer<LambdaEffectiveRecommendationPreferences.Builder> effectiveRecommendationPreferences) {
            return effectiveRecommendationPreferences(LambdaEffectiveRecommendationPreferences.builder()
                    .applyMutation(effectiveRecommendationPreferences).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String functionArn;

        private String functionVersion;

        private String accountId;

        private Integer currentMemorySize;

        private Long numberOfInvocations;

        private List<LambdaFunctionUtilizationMetric> utilizationMetrics = DefaultSdkAutoConstructList.getInstance();

        private Double lookbackPeriodInDays;

        private Instant lastRefreshTimestamp;

        private String finding;

        private List<String> findingReasonCodes = DefaultSdkAutoConstructList.getInstance();

        private List<LambdaFunctionMemoryRecommendationOption> memorySizeRecommendationOptions = DefaultSdkAutoConstructList
                .getInstance();

        private String currentPerformanceRisk;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private LambdaEffectiveRecommendationPreferences effectiveRecommendationPreferences;

        private BuilderImpl() {
        }

        private BuilderImpl(LambdaFunctionRecommendation model) {
            functionArn(model.functionArn);
            functionVersion(model.functionVersion);
            accountId(model.accountId);
            currentMemorySize(model.currentMemorySize);
            numberOfInvocations(model.numberOfInvocations);
            utilizationMetrics(model.utilizationMetrics);
            lookbackPeriodInDays(model.lookbackPeriodInDays);
            lastRefreshTimestamp(model.lastRefreshTimestamp);
            finding(model.finding);
            findingReasonCodesWithStrings(model.findingReasonCodes);
            memorySizeRecommendationOptions(model.memorySizeRecommendationOptions);
            currentPerformanceRisk(model.currentPerformanceRisk);
            tags(model.tags);
            effectiveRecommendationPreferences(model.effectiveRecommendationPreferences);
        }

        public final String getFunctionArn() {
            return functionArn;
        }

        public final void setFunctionArn(String functionArn) {
            this.functionArn = functionArn;
        }

        @Override
        public final Builder functionArn(String functionArn) {
            this.functionArn = functionArn;
            return this;
        }

        public final String getFunctionVersion() {
            return functionVersion;
        }

        public final void setFunctionVersion(String functionVersion) {
            this.functionVersion = functionVersion;
        }

        @Override
        public final Builder functionVersion(String functionVersion) {
            this.functionVersion = functionVersion;
            return this;
        }

        public final String getAccountId() {
            return accountId;
        }

        public final void setAccountId(String accountId) {
            this.accountId = accountId;
        }

        @Override
        public final Builder accountId(String accountId) {
            this.accountId = accountId;
            return this;
        }

        public final Integer getCurrentMemorySize() {
            return currentMemorySize;
        }

        public final void setCurrentMemorySize(Integer currentMemorySize) {
            this.currentMemorySize = currentMemorySize;
        }

        @Override
        public final Builder currentMemorySize(Integer currentMemorySize) {
            this.currentMemorySize = currentMemorySize;
            return this;
        }

        public final Long getNumberOfInvocations() {
            return numberOfInvocations;
        }

        public final void setNumberOfInvocations(Long numberOfInvocations) {
            this.numberOfInvocations = numberOfInvocations;
        }

        @Override
        public final Builder numberOfInvocations(Long numberOfInvocations) {
            this.numberOfInvocations = numberOfInvocations;
            return this;
        }

        public final List<LambdaFunctionUtilizationMetric.Builder> getUtilizationMetrics() {
            List<LambdaFunctionUtilizationMetric.Builder> result = LambdaFunctionUtilizationMetricsCopier
                    .copyToBuilder(this.utilizationMetrics);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setUtilizationMetrics(Collection<LambdaFunctionUtilizationMetric.BuilderImpl> utilizationMetrics) {
            this.utilizationMetrics = LambdaFunctionUtilizationMetricsCopier.copyFromBuilder(utilizationMetrics);
        }

        @Override
        public final Builder utilizationMetrics(Collection<LambdaFunctionUtilizationMetric> utilizationMetrics) {
            this.utilizationMetrics = LambdaFunctionUtilizationMetricsCopier.copy(utilizationMetrics);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder utilizationMetrics(LambdaFunctionUtilizationMetric... utilizationMetrics) {
            utilizationMetrics(Arrays.asList(utilizationMetrics));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder utilizationMetrics(Consumer<LambdaFunctionUtilizationMetric.Builder>... utilizationMetrics) {
            utilizationMetrics(Stream.of(utilizationMetrics)
                    .map(c -> LambdaFunctionUtilizationMetric.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final Double getLookbackPeriodInDays() {
            return lookbackPeriodInDays;
        }

        public final void setLookbackPeriodInDays(Double lookbackPeriodInDays) {
            this.lookbackPeriodInDays = lookbackPeriodInDays;
        }

        @Override
        public final Builder lookbackPeriodInDays(Double lookbackPeriodInDays) {
            this.lookbackPeriodInDays = lookbackPeriodInDays;
            return this;
        }

        public final Instant getLastRefreshTimestamp() {
            return lastRefreshTimestamp;
        }

        public final void setLastRefreshTimestamp(Instant lastRefreshTimestamp) {
            this.lastRefreshTimestamp = lastRefreshTimestamp;
        }

        @Override
        public final Builder lastRefreshTimestamp(Instant lastRefreshTimestamp) {
            this.lastRefreshTimestamp = lastRefreshTimestamp;
            return this;
        }

        public final String getFinding() {
            return finding;
        }

        public final void setFinding(String finding) {
            this.finding = finding;
        }

        @Override
        public final Builder finding(String finding) {
            this.finding = finding;
            return this;
        }

        @Override
        public final Builder finding(LambdaFunctionRecommendationFinding finding) {
            this.finding(finding == null ? null : finding.toString());
            return this;
        }

        public final Collection<String> getFindingReasonCodes() {
            if (findingReasonCodes instanceof SdkAutoConstructList) {
                return null;
            }
            return findingReasonCodes;
        }

        public final void setFindingReasonCodes(Collection<String> findingReasonCodes) {
            this.findingReasonCodes = LambdaFunctionRecommendationFindingReasonCodesCopier.copy(findingReasonCodes);
        }

        @Override
        public final Builder findingReasonCodesWithStrings(Collection<String> findingReasonCodes) {
            this.findingReasonCodes = LambdaFunctionRecommendationFindingReasonCodesCopier.copy(findingReasonCodes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder findingReasonCodesWithStrings(String... findingReasonCodes) {
            findingReasonCodesWithStrings(Arrays.asList(findingReasonCodes));
            return this;
        }

        @Override
        public final Builder findingReasonCodes(Collection<LambdaFunctionRecommendationFindingReasonCode> findingReasonCodes) {
            this.findingReasonCodes = LambdaFunctionRecommendationFindingReasonCodesCopier.copyEnumToString(findingReasonCodes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder findingReasonCodes(LambdaFunctionRecommendationFindingReasonCode... findingReasonCodes) {
            findingReasonCodes(Arrays.asList(findingReasonCodes));
            return this;
        }

        public final List<LambdaFunctionMemoryRecommendationOption.Builder> getMemorySizeRecommendationOptions() {
            List<LambdaFunctionMemoryRecommendationOption.Builder> result = LambdaFunctionMemoryRecommendationOptionsCopier
                    .copyToBuilder(this.memorySizeRecommendationOptions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setMemorySizeRecommendationOptions(
                Collection<LambdaFunctionMemoryRecommendationOption.BuilderImpl> memorySizeRecommendationOptions) {
            this.memorySizeRecommendationOptions = LambdaFunctionMemoryRecommendationOptionsCopier
                    .copyFromBuilder(memorySizeRecommendationOptions);
        }

        @Override
        public final Builder memorySizeRecommendationOptions(
                Collection<LambdaFunctionMemoryRecommendationOption> memorySizeRecommendationOptions) {
            this.memorySizeRecommendationOptions = LambdaFunctionMemoryRecommendationOptionsCopier
                    .copy(memorySizeRecommendationOptions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder memorySizeRecommendationOptions(
                LambdaFunctionMemoryRecommendationOption... memorySizeRecommendationOptions) {
            memorySizeRecommendationOptions(Arrays.asList(memorySizeRecommendationOptions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder memorySizeRecommendationOptions(
                Consumer<LambdaFunctionMemoryRecommendationOption.Builder>... memorySizeRecommendationOptions) {
            memorySizeRecommendationOptions(Stream.of(memorySizeRecommendationOptions)
                    .map(c -> LambdaFunctionMemoryRecommendationOption.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getCurrentPerformanceRisk() {
            return currentPerformanceRisk;
        }

        public final void setCurrentPerformanceRisk(String currentPerformanceRisk) {
            this.currentPerformanceRisk = currentPerformanceRisk;
        }

        @Override
        public final Builder currentPerformanceRisk(String currentPerformanceRisk) {
            this.currentPerformanceRisk = currentPerformanceRisk;
            return this;
        }

        @Override
        public final Builder currentPerformanceRisk(CurrentPerformanceRisk currentPerformanceRisk) {
            this.currentPerformanceRisk(currentPerformanceRisk == null ? null : currentPerformanceRisk.toString());
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagsCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagsCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagsCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final LambdaEffectiveRecommendationPreferences.Builder getEffectiveRecommendationPreferences() {
            return effectiveRecommendationPreferences != null ? effectiveRecommendationPreferences.toBuilder() : null;
        }

        public final void setEffectiveRecommendationPreferences(
                LambdaEffectiveRecommendationPreferences.BuilderImpl effectiveRecommendationPreferences) {
            this.effectiveRecommendationPreferences = effectiveRecommendationPreferences != null ? effectiveRecommendationPreferences
                    .build() : null;
        }

        @Override
        public final Builder effectiveRecommendationPreferences(
                LambdaEffectiveRecommendationPreferences effectiveRecommendationPreferences) {
            this.effectiveRecommendationPreferences = effectiveRecommendationPreferences;
            return this;
        }

        @Override
        public LambdaFunctionRecommendation build() {
            return new LambdaFunctionRecommendation(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
