/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.connectcases;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.connectcases.model.AccessDeniedException;
import software.amazon.awssdk.services.connectcases.model.BatchGetFieldRequest;
import software.amazon.awssdk.services.connectcases.model.BatchGetFieldResponse;
import software.amazon.awssdk.services.connectcases.model.BatchPutFieldOptionsRequest;
import software.amazon.awssdk.services.connectcases.model.BatchPutFieldOptionsResponse;
import software.amazon.awssdk.services.connectcases.model.ConflictException;
import software.amazon.awssdk.services.connectcases.model.ConnectCasesException;
import software.amazon.awssdk.services.connectcases.model.ConnectCasesRequest;
import software.amazon.awssdk.services.connectcases.model.CreateCaseRequest;
import software.amazon.awssdk.services.connectcases.model.CreateCaseResponse;
import software.amazon.awssdk.services.connectcases.model.CreateDomainRequest;
import software.amazon.awssdk.services.connectcases.model.CreateDomainResponse;
import software.amazon.awssdk.services.connectcases.model.CreateFieldRequest;
import software.amazon.awssdk.services.connectcases.model.CreateFieldResponse;
import software.amazon.awssdk.services.connectcases.model.CreateLayoutRequest;
import software.amazon.awssdk.services.connectcases.model.CreateLayoutResponse;
import software.amazon.awssdk.services.connectcases.model.CreateRelatedItemRequest;
import software.amazon.awssdk.services.connectcases.model.CreateRelatedItemResponse;
import software.amazon.awssdk.services.connectcases.model.CreateTemplateRequest;
import software.amazon.awssdk.services.connectcases.model.CreateTemplateResponse;
import software.amazon.awssdk.services.connectcases.model.GetCaseEventConfigurationRequest;
import software.amazon.awssdk.services.connectcases.model.GetCaseEventConfigurationResponse;
import software.amazon.awssdk.services.connectcases.model.GetCaseRequest;
import software.amazon.awssdk.services.connectcases.model.GetCaseResponse;
import software.amazon.awssdk.services.connectcases.model.GetDomainRequest;
import software.amazon.awssdk.services.connectcases.model.GetDomainResponse;
import software.amazon.awssdk.services.connectcases.model.GetLayoutRequest;
import software.amazon.awssdk.services.connectcases.model.GetLayoutResponse;
import software.amazon.awssdk.services.connectcases.model.GetTemplateRequest;
import software.amazon.awssdk.services.connectcases.model.GetTemplateResponse;
import software.amazon.awssdk.services.connectcases.model.InternalServerException;
import software.amazon.awssdk.services.connectcases.model.ListCasesForContactRequest;
import software.amazon.awssdk.services.connectcases.model.ListCasesForContactResponse;
import software.amazon.awssdk.services.connectcases.model.ListDomainsRequest;
import software.amazon.awssdk.services.connectcases.model.ListDomainsResponse;
import software.amazon.awssdk.services.connectcases.model.ListFieldOptionsRequest;
import software.amazon.awssdk.services.connectcases.model.ListFieldOptionsResponse;
import software.amazon.awssdk.services.connectcases.model.ListFieldsRequest;
import software.amazon.awssdk.services.connectcases.model.ListFieldsResponse;
import software.amazon.awssdk.services.connectcases.model.ListLayoutsRequest;
import software.amazon.awssdk.services.connectcases.model.ListLayoutsResponse;
import software.amazon.awssdk.services.connectcases.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.connectcases.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.connectcases.model.ListTemplatesRequest;
import software.amazon.awssdk.services.connectcases.model.ListTemplatesResponse;
import software.amazon.awssdk.services.connectcases.model.PutCaseEventConfigurationRequest;
import software.amazon.awssdk.services.connectcases.model.PutCaseEventConfigurationResponse;
import software.amazon.awssdk.services.connectcases.model.ResourceNotFoundException;
import software.amazon.awssdk.services.connectcases.model.SearchCasesRequest;
import software.amazon.awssdk.services.connectcases.model.SearchCasesResponse;
import software.amazon.awssdk.services.connectcases.model.SearchRelatedItemsRequest;
import software.amazon.awssdk.services.connectcases.model.SearchRelatedItemsResponse;
import software.amazon.awssdk.services.connectcases.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.connectcases.model.TagResourceRequest;
import software.amazon.awssdk.services.connectcases.model.TagResourceResponse;
import software.amazon.awssdk.services.connectcases.model.ThrottlingException;
import software.amazon.awssdk.services.connectcases.model.UntagResourceRequest;
import software.amazon.awssdk.services.connectcases.model.UntagResourceResponse;
import software.amazon.awssdk.services.connectcases.model.UpdateCaseRequest;
import software.amazon.awssdk.services.connectcases.model.UpdateCaseResponse;
import software.amazon.awssdk.services.connectcases.model.UpdateFieldRequest;
import software.amazon.awssdk.services.connectcases.model.UpdateFieldResponse;
import software.amazon.awssdk.services.connectcases.model.UpdateLayoutRequest;
import software.amazon.awssdk.services.connectcases.model.UpdateLayoutResponse;
import software.amazon.awssdk.services.connectcases.model.UpdateTemplateRequest;
import software.amazon.awssdk.services.connectcases.model.UpdateTemplateResponse;
import software.amazon.awssdk.services.connectcases.model.ValidationException;
import software.amazon.awssdk.services.connectcases.paginators.GetCaseIterable;
import software.amazon.awssdk.services.connectcases.paginators.ListCasesForContactIterable;
import software.amazon.awssdk.services.connectcases.paginators.ListDomainsIterable;
import software.amazon.awssdk.services.connectcases.paginators.ListFieldOptionsIterable;
import software.amazon.awssdk.services.connectcases.paginators.ListFieldsIterable;
import software.amazon.awssdk.services.connectcases.paginators.ListLayoutsIterable;
import software.amazon.awssdk.services.connectcases.paginators.ListTemplatesIterable;
import software.amazon.awssdk.services.connectcases.paginators.SearchCasesIterable;
import software.amazon.awssdk.services.connectcases.paginators.SearchRelatedItemsIterable;
import software.amazon.awssdk.services.connectcases.transform.BatchGetFieldRequestMarshaller;
import software.amazon.awssdk.services.connectcases.transform.BatchPutFieldOptionsRequestMarshaller;
import software.amazon.awssdk.services.connectcases.transform.CreateCaseRequestMarshaller;
import software.amazon.awssdk.services.connectcases.transform.CreateDomainRequestMarshaller;
import software.amazon.awssdk.services.connectcases.transform.CreateFieldRequestMarshaller;
import software.amazon.awssdk.services.connectcases.transform.CreateLayoutRequestMarshaller;
import software.amazon.awssdk.services.connectcases.transform.CreateRelatedItemRequestMarshaller;
import software.amazon.awssdk.services.connectcases.transform.CreateTemplateRequestMarshaller;
import software.amazon.awssdk.services.connectcases.transform.GetCaseEventConfigurationRequestMarshaller;
import software.amazon.awssdk.services.connectcases.transform.GetCaseRequestMarshaller;
import software.amazon.awssdk.services.connectcases.transform.GetDomainRequestMarshaller;
import software.amazon.awssdk.services.connectcases.transform.GetLayoutRequestMarshaller;
import software.amazon.awssdk.services.connectcases.transform.GetTemplateRequestMarshaller;
import software.amazon.awssdk.services.connectcases.transform.ListCasesForContactRequestMarshaller;
import software.amazon.awssdk.services.connectcases.transform.ListDomainsRequestMarshaller;
import software.amazon.awssdk.services.connectcases.transform.ListFieldOptionsRequestMarshaller;
import software.amazon.awssdk.services.connectcases.transform.ListFieldsRequestMarshaller;
import software.amazon.awssdk.services.connectcases.transform.ListLayoutsRequestMarshaller;
import software.amazon.awssdk.services.connectcases.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.connectcases.transform.ListTemplatesRequestMarshaller;
import software.amazon.awssdk.services.connectcases.transform.PutCaseEventConfigurationRequestMarshaller;
import software.amazon.awssdk.services.connectcases.transform.SearchCasesRequestMarshaller;
import software.amazon.awssdk.services.connectcases.transform.SearchRelatedItemsRequestMarshaller;
import software.amazon.awssdk.services.connectcases.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.connectcases.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.connectcases.transform.UpdateCaseRequestMarshaller;
import software.amazon.awssdk.services.connectcases.transform.UpdateFieldRequestMarshaller;
import software.amazon.awssdk.services.connectcases.transform.UpdateLayoutRequestMarshaller;
import software.amazon.awssdk.services.connectcases.transform.UpdateTemplateRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link ConnectCasesClient}.
 *
 * @see ConnectCasesClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultConnectCasesClient implements ConnectCasesClient {
    private static final Logger log = Logger.loggerFor(DefaultConnectCasesClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultConnectCasesClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Returns the description for the list of fields in the request parameters.
     * </p>
     *
     * @param batchGetFieldRequest
     * @return Result of the BatchGetField operation returned by the service.
     * @throws InternalServerException
     *         We couldn't process your request because of an issue with the server. Try again later.
     * @throws ResourceNotFoundException
     *         We couldn't find the requested resource. Check that your resources exists and were created in the same
     *         Amazon Web Services Region as your request, and try your request again.
     * @throws ValidationException
     *         The request isn't valid. Check the syntax and try again.
     * @throws ThrottlingException
     *         The rate has been exceeded for this API. Please try again after a few minutes.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectCasesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectCasesClient.BatchGetField
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcases-2022-10-03/BatchGetField" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public BatchGetFieldResponse batchGetField(BatchGetFieldRequest batchGetFieldRequest) throws InternalServerException,
            ResourceNotFoundException, ValidationException, ThrottlingException, AccessDeniedException, AwsServiceException,
            SdkClientException, ConnectCasesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<BatchGetFieldResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                BatchGetFieldResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchGetFieldRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCases");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchGetField");

            return clientHandler.execute(new ClientExecutionParams<BatchGetFieldRequest, BatchGetFieldResponse>()
                    .withOperationName("BatchGetField").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(batchGetFieldRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new BatchGetFieldRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates and updates a set of field options for a single select field in a Cases domain.
     * </p>
     *
     * @param batchPutFieldOptionsRequest
     * @return Result of the BatchPutFieldOptions operation returned by the service.
     * @throws InternalServerException
     *         We couldn't process your request because of an issue with the server. Try again later.
     * @throws ResourceNotFoundException
     *         We couldn't find the requested resource. Check that your resources exists and were created in the same
     *         Amazon Web Services Region as your request, and try your request again.
     * @throws ValidationException
     *         The request isn't valid. Check the syntax and try again.
     * @throws ThrottlingException
     *         The rate has been exceeded for this API. Please try again after a few minutes.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ServiceQuotaExceededException
     *         The service quota has been exceeded. For a list of service quotas, see <a
     *         href="https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html">Amazon
     *         Connect Service Quotas</a> in the <i>Amazon Connect Administrator Guide</i>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectCasesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectCasesClient.BatchPutFieldOptions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcases-2022-10-03/BatchPutFieldOptions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public BatchPutFieldOptionsResponse batchPutFieldOptions(BatchPutFieldOptionsRequest batchPutFieldOptionsRequest)
            throws InternalServerException, ResourceNotFoundException, ValidationException, ThrottlingException,
            AccessDeniedException, ServiceQuotaExceededException, AwsServiceException, SdkClientException, ConnectCasesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<BatchPutFieldOptionsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, BatchPutFieldOptionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchPutFieldOptionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCases");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchPutFieldOptions");

            return clientHandler.execute(new ClientExecutionParams<BatchPutFieldOptionsRequest, BatchPutFieldOptionsResponse>()
                    .withOperationName("BatchPutFieldOptions").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(batchPutFieldOptionsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new BatchPutFieldOptionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a case in the specified Cases domain. Case system and custom fields are taken as an array id/value pairs
     * with a declared data types.
     * </p>
     * <note>
     * <p>
     * <code>customer_id</code> is a required field when creating a case.
     * </p>
     * </note>
     *
     * @param createCaseRequest
     * @return Result of the CreateCase operation returned by the service.
     * @throws InternalServerException
     *         We couldn't process your request because of an issue with the server. Try again later.
     * @throws ResourceNotFoundException
     *         We couldn't find the requested resource. Check that your resources exists and were created in the same
     *         Amazon Web Services Region as your request, and try your request again.
     * @throws ValidationException
     *         The request isn't valid. Check the syntax and try again.
     * @throws ThrottlingException
     *         The rate has been exceeded for this API. Please try again after a few minutes.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ConflictException
     *         The requested operation would cause a conflict with the current state of a service resource associated
     *         with the request. Resolve the conflict before retrying this request. See the accompanying error message
     *         for details.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectCasesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectCasesClient.CreateCase
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcases-2022-10-03/CreateCase" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateCaseResponse createCase(CreateCaseRequest createCaseRequest) throws InternalServerException,
            ResourceNotFoundException, ValidationException, ThrottlingException, AccessDeniedException, ConflictException,
            AwsServiceException, SdkClientException, ConnectCasesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateCaseResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateCaseResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createCaseRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCases");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateCase");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateCaseRequest, CreateCaseResponse>().withOperationName("CreateCase")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createCaseRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateCaseRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a domain, which is a container for all case data, such as cases, fields, templates and layouts. Each
     * Amazon Connect instance can be associated with only one Cases domain.
     * </p>
     * <important>
     * <p>
     * This will not associate your connect instance to Cases domain. Instead, use the Amazon Connect <a
     * href="https://docs.aws.amazon.com/connect/latest/APIReference/API_CreateIntegrationAssociation.html"
     * >CreateIntegrationAssociation</a> API.
     * </p>
     * </important>
     *
     * @param createDomainRequest
     * @return Result of the CreateDomain operation returned by the service.
     * @throws InternalServerException
     *         We couldn't process your request because of an issue with the server. Try again later.
     * @throws ValidationException
     *         The request isn't valid. Check the syntax and try again.
     * @throws ThrottlingException
     *         The rate has been exceeded for this API. Please try again after a few minutes.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ConflictException
     *         The requested operation would cause a conflict with the current state of a service resource associated
     *         with the request. Resolve the conflict before retrying this request. See the accompanying error message
     *         for details.
     * @throws ServiceQuotaExceededException
     *         The service quota has been exceeded. For a list of service quotas, see <a
     *         href="https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html">Amazon
     *         Connect Service Quotas</a> in the <i>Amazon Connect Administrator Guide</i>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectCasesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectCasesClient.CreateDomain
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcases-2022-10-03/CreateDomain" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateDomainResponse createDomain(CreateDomainRequest createDomainRequest) throws InternalServerException,
            ValidationException, ThrottlingException, AccessDeniedException, ConflictException, ServiceQuotaExceededException,
            AwsServiceException, SdkClientException, ConnectCasesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateDomainResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateDomainResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDomainRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCases");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDomain");

            return clientHandler.execute(new ClientExecutionParams<CreateDomainRequest, CreateDomainResponse>()
                    .withOperationName("CreateDomain").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createDomainRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateDomainRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a field in the Cases domain. This field is used to define the case object model (that is, defines what
     * data can be captured on cases) in a Cases domain.
     * </p>
     *
     * @param createFieldRequest
     * @return Result of the CreateField operation returned by the service.
     * @throws InternalServerException
     *         We couldn't process your request because of an issue with the server. Try again later.
     * @throws ResourceNotFoundException
     *         We couldn't find the requested resource. Check that your resources exists and were created in the same
     *         Amazon Web Services Region as your request, and try your request again.
     * @throws ValidationException
     *         The request isn't valid. Check the syntax and try again.
     * @throws ThrottlingException
     *         The rate has been exceeded for this API. Please try again after a few minutes.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ConflictException
     *         The requested operation would cause a conflict with the current state of a service resource associated
     *         with the request. Resolve the conflict before retrying this request. See the accompanying error message
     *         for details.
     * @throws ServiceQuotaExceededException
     *         The service quota has been exceeded. For a list of service quotas, see <a
     *         href="https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html">Amazon
     *         Connect Service Quotas</a> in the <i>Amazon Connect Administrator Guide</i>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectCasesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectCasesClient.CreateField
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcases-2022-10-03/CreateField" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateFieldResponse createField(CreateFieldRequest createFieldRequest) throws InternalServerException,
            ResourceNotFoundException, ValidationException, ThrottlingException, AccessDeniedException, ConflictException,
            ServiceQuotaExceededException, AwsServiceException, SdkClientException, ConnectCasesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateFieldResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateFieldResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createFieldRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCases");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateField");

            return clientHandler.execute(new ClientExecutionParams<CreateFieldRequest, CreateFieldResponse>()
                    .withOperationName("CreateField").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createFieldRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateFieldRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a layout in the Cases domain. Layouts define the following configuration in the top section and More Info
     * tab of the Cases user interface:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Fields to display to the users
     * </p>
     * </li>
     * <li>
     * <p>
     * Field ordering
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * Title and Status fields cannot be part of layouts since they are not configurable.
     * </p>
     * </note>
     *
     * @param createLayoutRequest
     * @return Result of the CreateLayout operation returned by the service.
     * @throws InternalServerException
     *         We couldn't process your request because of an issue with the server. Try again later.
     * @throws ResourceNotFoundException
     *         We couldn't find the requested resource. Check that your resources exists and were created in the same
     *         Amazon Web Services Region as your request, and try your request again.
     * @throws ValidationException
     *         The request isn't valid. Check the syntax and try again.
     * @throws ThrottlingException
     *         The rate has been exceeded for this API. Please try again after a few minutes.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ConflictException
     *         The requested operation would cause a conflict with the current state of a service resource associated
     *         with the request. Resolve the conflict before retrying this request. See the accompanying error message
     *         for details.
     * @throws ServiceQuotaExceededException
     *         The service quota has been exceeded. For a list of service quotas, see <a
     *         href="https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html">Amazon
     *         Connect Service Quotas</a> in the <i>Amazon Connect Administrator Guide</i>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectCasesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectCasesClient.CreateLayout
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcases-2022-10-03/CreateLayout" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateLayoutResponse createLayout(CreateLayoutRequest createLayoutRequest) throws InternalServerException,
            ResourceNotFoundException, ValidationException, ThrottlingException, AccessDeniedException, ConflictException,
            ServiceQuotaExceededException, AwsServiceException, SdkClientException, ConnectCasesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateLayoutResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateLayoutResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createLayoutRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCases");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateLayout");

            return clientHandler.execute(new ClientExecutionParams<CreateLayoutRequest, CreateLayoutResponse>()
                    .withOperationName("CreateLayout").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createLayoutRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateLayoutRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a related item (comments, tasks, and contacts) and associates it with a case.
     * </p>
     * <note>
     * <p>
     * A Related Item is a resource that is associated with a case. It may or may not have an external identifier
     * linking it to an external resource (for example, a <code>contactArn</code>). All Related Items have their own
     * internal identifier, the <code>relatedItemArn</code>. Examples of related items include <code>comments</code> and
     * <code>contacts</code>.
     * </p>
     * </note>
     *
     * @param createRelatedItemRequest
     * @return Result of the CreateRelatedItem operation returned by the service.
     * @throws InternalServerException
     *         We couldn't process your request because of an issue with the server. Try again later.
     * @throws ResourceNotFoundException
     *         We couldn't find the requested resource. Check that your resources exists and were created in the same
     *         Amazon Web Services Region as your request, and try your request again.
     * @throws ValidationException
     *         The request isn't valid. Check the syntax and try again.
     * @throws ThrottlingException
     *         The rate has been exceeded for this API. Please try again after a few minutes.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ServiceQuotaExceededException
     *         The service quota has been exceeded. For a list of service quotas, see <a
     *         href="https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html">Amazon
     *         Connect Service Quotas</a> in the <i>Amazon Connect Administrator Guide</i>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectCasesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectCasesClient.CreateRelatedItem
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcases-2022-10-03/CreateRelatedItem"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateRelatedItemResponse createRelatedItem(CreateRelatedItemRequest createRelatedItemRequest)
            throws InternalServerException, ResourceNotFoundException, ValidationException, ThrottlingException,
            AccessDeniedException, ServiceQuotaExceededException, AwsServiceException, SdkClientException, ConnectCasesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateRelatedItemResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateRelatedItemResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createRelatedItemRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCases");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateRelatedItem");

            return clientHandler.execute(new ClientExecutionParams<CreateRelatedItemRequest, CreateRelatedItemResponse>()
                    .withOperationName("CreateRelatedItem").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createRelatedItemRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateRelatedItemRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a template in the Cases domain. This template is used to define the case object model (that is, to define
     * what data can be captured on cases) in a Cases domain. A template must have a unique name within a domain, and it
     * must reference existing field IDs and layout IDs. Additionally, multiple fields with same IDs are not allowed
     * within the same Template. A template can be either Active or Inactive, as indicated by its status. Inactive
     * templates cannot be used to create cases.
     * </p>
     *
     * @param createTemplateRequest
     * @return Result of the CreateTemplate operation returned by the service.
     * @throws InternalServerException
     *         We couldn't process your request because of an issue with the server. Try again later.
     * @throws ResourceNotFoundException
     *         We couldn't find the requested resource. Check that your resources exists and were created in the same
     *         Amazon Web Services Region as your request, and try your request again.
     * @throws ValidationException
     *         The request isn't valid. Check the syntax and try again.
     * @throws ThrottlingException
     *         The rate has been exceeded for this API. Please try again after a few minutes.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ConflictException
     *         The requested operation would cause a conflict with the current state of a service resource associated
     *         with the request. Resolve the conflict before retrying this request. See the accompanying error message
     *         for details.
     * @throws ServiceQuotaExceededException
     *         The service quota has been exceeded. For a list of service quotas, see <a
     *         href="https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html">Amazon
     *         Connect Service Quotas</a> in the <i>Amazon Connect Administrator Guide</i>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectCasesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectCasesClient.CreateTemplate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcases-2022-10-03/CreateTemplate" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateTemplateResponse createTemplate(CreateTemplateRequest createTemplateRequest) throws InternalServerException,
            ResourceNotFoundException, ValidationException, ThrottlingException, AccessDeniedException, ConflictException,
            ServiceQuotaExceededException, AwsServiceException, SdkClientException, ConnectCasesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateTemplateResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateTemplateResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createTemplateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCases");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateTemplate");

            return clientHandler.execute(new ClientExecutionParams<CreateTemplateRequest, CreateTemplateResponse>()
                    .withOperationName("CreateTemplate").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createTemplateRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateTemplateRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information about a specific case if it exists.
     * </p>
     *
     * @param getCaseRequest
     * @return Result of the GetCase operation returned by the service.
     * @throws InternalServerException
     *         We couldn't process your request because of an issue with the server. Try again later.
     * @throws ResourceNotFoundException
     *         We couldn't find the requested resource. Check that your resources exists and were created in the same
     *         Amazon Web Services Region as your request, and try your request again.
     * @throws ValidationException
     *         The request isn't valid. Check the syntax and try again.
     * @throws ThrottlingException
     *         The rate has been exceeded for this API. Please try again after a few minutes.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectCasesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectCasesClient.GetCase
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcases-2022-10-03/GetCase" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetCaseResponse getCase(GetCaseRequest getCaseRequest) throws InternalServerException, ResourceNotFoundException,
            ValidationException, ThrottlingException, AccessDeniedException, AwsServiceException, SdkClientException,
            ConnectCasesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetCaseResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetCaseResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getCaseRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCases");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetCase");

            return clientHandler.execute(new ClientExecutionParams<GetCaseRequest, GetCaseResponse>()
                    .withOperationName("GetCase").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getCaseRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new GetCaseRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information about a specific case if it exists.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #getCase(software.amazon.awssdk.services.connectcases.model.GetCaseRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.connectcases.paginators.GetCaseIterable responses = client.getCasePaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.connectcases.paginators.GetCaseIterable responses = client.getCasePaginator(request);
     *     for (software.amazon.awssdk.services.connectcases.model.GetCaseResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.connectcases.paginators.GetCaseIterable responses = client.getCasePaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of null won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getCase(software.amazon.awssdk.services.connectcases.model.GetCaseRequest)} operation.</b>
     * </p>
     *
     * @param getCaseRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerException
     *         We couldn't process your request because of an issue with the server. Try again later.
     * @throws ResourceNotFoundException
     *         We couldn't find the requested resource. Check that your resources exists and were created in the same
     *         Amazon Web Services Region as your request, and try your request again.
     * @throws ValidationException
     *         The request isn't valid. Check the syntax and try again.
     * @throws ThrottlingException
     *         The rate has been exceeded for this API. Please try again after a few minutes.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectCasesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectCasesClient.GetCase
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcases-2022-10-03/GetCase" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetCaseIterable getCasePaginator(GetCaseRequest getCaseRequest) throws InternalServerException,
            ResourceNotFoundException, ValidationException, ThrottlingException, AccessDeniedException, AwsServiceException,
            SdkClientException, ConnectCasesException {
        return new GetCaseIterable(this, applyPaginatorUserAgent(getCaseRequest));
    }

    /**
     * <p>
     * Returns the case event publishing configuration.
     * </p>
     *
     * @param getCaseEventConfigurationRequest
     * @return Result of the GetCaseEventConfiguration operation returned by the service.
     * @throws InternalServerException
     *         We couldn't process your request because of an issue with the server. Try again later.
     * @throws ResourceNotFoundException
     *         We couldn't find the requested resource. Check that your resources exists and were created in the same
     *         Amazon Web Services Region as your request, and try your request again.
     * @throws ValidationException
     *         The request isn't valid. Check the syntax and try again.
     * @throws ThrottlingException
     *         The rate has been exceeded for this API. Please try again after a few minutes.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectCasesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectCasesClient.GetCaseEventConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcases-2022-10-03/GetCaseEventConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetCaseEventConfigurationResponse getCaseEventConfiguration(
            GetCaseEventConfigurationRequest getCaseEventConfigurationRequest) throws InternalServerException,
            ResourceNotFoundException, ValidationException, ThrottlingException, AccessDeniedException, AwsServiceException,
            SdkClientException, ConnectCasesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetCaseEventConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetCaseEventConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getCaseEventConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCases");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetCaseEventConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<GetCaseEventConfigurationRequest, GetCaseEventConfigurationResponse>()
                            .withOperationName("GetCaseEventConfiguration").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(getCaseEventConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetCaseEventConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information about a specific domain if it exists.
     * </p>
     *
     * @param getDomainRequest
     * @return Result of the GetDomain operation returned by the service.
     * @throws InternalServerException
     *         We couldn't process your request because of an issue with the server. Try again later.
     * @throws ResourceNotFoundException
     *         We couldn't find the requested resource. Check that your resources exists and were created in the same
     *         Amazon Web Services Region as your request, and try your request again.
     * @throws ValidationException
     *         The request isn't valid. Check the syntax and try again.
     * @throws ThrottlingException
     *         The rate has been exceeded for this API. Please try again after a few minutes.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectCasesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectCasesClient.GetDomain
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcases-2022-10-03/GetDomain" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetDomainResponse getDomain(GetDomainRequest getDomainRequest) throws InternalServerException,
            ResourceNotFoundException, ValidationException, ThrottlingException, AccessDeniedException, AwsServiceException,
            SdkClientException, ConnectCasesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetDomainResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetDomainResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDomainRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCases");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDomain");

            return clientHandler.execute(new ClientExecutionParams<GetDomainRequest, GetDomainResponse>()
                    .withOperationName("GetDomain").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getDomainRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new GetDomainRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the details for the requested layout.
     * </p>
     *
     * @param getLayoutRequest
     * @return Result of the GetLayout operation returned by the service.
     * @throws InternalServerException
     *         We couldn't process your request because of an issue with the server. Try again later.
     * @throws ResourceNotFoundException
     *         We couldn't find the requested resource. Check that your resources exists and were created in the same
     *         Amazon Web Services Region as your request, and try your request again.
     * @throws ValidationException
     *         The request isn't valid. Check the syntax and try again.
     * @throws ThrottlingException
     *         The rate has been exceeded for this API. Please try again after a few minutes.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectCasesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectCasesClient.GetLayout
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcases-2022-10-03/GetLayout" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetLayoutResponse getLayout(GetLayoutRequest getLayoutRequest) throws InternalServerException,
            ResourceNotFoundException, ValidationException, ThrottlingException, AccessDeniedException, AwsServiceException,
            SdkClientException, ConnectCasesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetLayoutResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetLayoutResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getLayoutRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCases");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetLayout");

            return clientHandler.execute(new ClientExecutionParams<GetLayoutRequest, GetLayoutResponse>()
                    .withOperationName("GetLayout").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getLayoutRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new GetLayoutRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the details for the requested template.
     * </p>
     *
     * @param getTemplateRequest
     * @return Result of the GetTemplate operation returned by the service.
     * @throws InternalServerException
     *         We couldn't process your request because of an issue with the server. Try again later.
     * @throws ResourceNotFoundException
     *         We couldn't find the requested resource. Check that your resources exists and were created in the same
     *         Amazon Web Services Region as your request, and try your request again.
     * @throws ValidationException
     *         The request isn't valid. Check the syntax and try again.
     * @throws ThrottlingException
     *         The rate has been exceeded for this API. Please try again after a few minutes.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectCasesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectCasesClient.GetTemplate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcases-2022-10-03/GetTemplate" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetTemplateResponse getTemplate(GetTemplateRequest getTemplateRequest) throws InternalServerException,
            ResourceNotFoundException, ValidationException, ThrottlingException, AccessDeniedException, AwsServiceException,
            SdkClientException, ConnectCasesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetTemplateResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetTemplateResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getTemplateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCases");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetTemplate");

            return clientHandler.execute(new ClientExecutionParams<GetTemplateRequest, GetTemplateResponse>()
                    .withOperationName("GetTemplate").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getTemplateRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetTemplateRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists cases for a given contact.
     * </p>
     *
     * @param listCasesForContactRequest
     * @return Result of the ListCasesForContact operation returned by the service.
     * @throws InternalServerException
     *         We couldn't process your request because of an issue with the server. Try again later.
     * @throws ResourceNotFoundException
     *         We couldn't find the requested resource. Check that your resources exists and were created in the same
     *         Amazon Web Services Region as your request, and try your request again.
     * @throws ValidationException
     *         The request isn't valid. Check the syntax and try again.
     * @throws ThrottlingException
     *         The rate has been exceeded for this API. Please try again after a few minutes.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectCasesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectCasesClient.ListCasesForContact
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcases-2022-10-03/ListCasesForContact"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListCasesForContactResponse listCasesForContact(ListCasesForContactRequest listCasesForContactRequest)
            throws InternalServerException, ResourceNotFoundException, ValidationException, ThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, ConnectCasesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListCasesForContactResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListCasesForContactResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listCasesForContactRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCases");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListCasesForContact");

            return clientHandler.execute(new ClientExecutionParams<ListCasesForContactRequest, ListCasesForContactResponse>()
                    .withOperationName("ListCasesForContact").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listCasesForContactRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListCasesForContactRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists cases for a given contact.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listCasesForContact(software.amazon.awssdk.services.connectcases.model.ListCasesForContactRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.connectcases.paginators.ListCasesForContactIterable responses = client.listCasesForContactPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.connectcases.paginators.ListCasesForContactIterable responses = client
     *             .listCasesForContactPaginator(request);
     *     for (software.amazon.awssdk.services.connectcases.model.ListCasesForContactResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.connectcases.paginators.ListCasesForContactIterable responses = client.listCasesForContactPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listCasesForContact(software.amazon.awssdk.services.connectcases.model.ListCasesForContactRequest)}
     * operation.</b>
     * </p>
     *
     * @param listCasesForContactRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerException
     *         We couldn't process your request because of an issue with the server. Try again later.
     * @throws ResourceNotFoundException
     *         We couldn't find the requested resource. Check that your resources exists and were created in the same
     *         Amazon Web Services Region as your request, and try your request again.
     * @throws ValidationException
     *         The request isn't valid. Check the syntax and try again.
     * @throws ThrottlingException
     *         The rate has been exceeded for this API. Please try again after a few minutes.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectCasesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectCasesClient.ListCasesForContact
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcases-2022-10-03/ListCasesForContact"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListCasesForContactIterable listCasesForContactPaginator(ListCasesForContactRequest listCasesForContactRequest)
            throws InternalServerException, ResourceNotFoundException, ValidationException, ThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, ConnectCasesException {
        return new ListCasesForContactIterable(this, applyPaginatorUserAgent(listCasesForContactRequest));
    }

    /**
     * <p>
     * Lists all cases domains in the Amazon Web Services account. Each list item is a condensed summary object of the
     * domain.
     * </p>
     *
     * @param listDomainsRequest
     * @return Result of the ListDomains operation returned by the service.
     * @throws InternalServerException
     *         We couldn't process your request because of an issue with the server. Try again later.
     * @throws ValidationException
     *         The request isn't valid. Check the syntax and try again.
     * @throws ThrottlingException
     *         The rate has been exceeded for this API. Please try again after a few minutes.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectCasesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectCasesClient.ListDomains
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcases-2022-10-03/ListDomains" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListDomainsResponse listDomains(ListDomainsRequest listDomainsRequest) throws InternalServerException,
            ValidationException, ThrottlingException, AccessDeniedException, AwsServiceException, SdkClientException,
            ConnectCasesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListDomainsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListDomainsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDomainsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCases");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDomains");

            return clientHandler.execute(new ClientExecutionParams<ListDomainsRequest, ListDomainsResponse>()
                    .withOperationName("ListDomains").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listDomainsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListDomainsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all cases domains in the Amazon Web Services account. Each list item is a condensed summary object of the
     * domain.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listDomains(software.amazon.awssdk.services.connectcases.model.ListDomainsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.connectcases.paginators.ListDomainsIterable responses = client.listDomainsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.connectcases.paginators.ListDomainsIterable responses = client.listDomainsPaginator(request);
     *     for (software.amazon.awssdk.services.connectcases.model.ListDomainsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.connectcases.paginators.ListDomainsIterable responses = client.listDomainsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listDomains(software.amazon.awssdk.services.connectcases.model.ListDomainsRequest)} operation.</b>
     * </p>
     *
     * @param listDomainsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerException
     *         We couldn't process your request because of an issue with the server. Try again later.
     * @throws ValidationException
     *         The request isn't valid. Check the syntax and try again.
     * @throws ThrottlingException
     *         The rate has been exceeded for this API. Please try again after a few minutes.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectCasesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectCasesClient.ListDomains
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcases-2022-10-03/ListDomains" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListDomainsIterable listDomainsPaginator(ListDomainsRequest listDomainsRequest) throws InternalServerException,
            ValidationException, ThrottlingException, AccessDeniedException, AwsServiceException, SdkClientException,
            ConnectCasesException {
        return new ListDomainsIterable(this, applyPaginatorUserAgent(listDomainsRequest));
    }

    /**
     * <p>
     * Lists all of the field options for a field identifier in the domain.
     * </p>
     *
     * @param listFieldOptionsRequest
     * @return Result of the ListFieldOptions operation returned by the service.
     * @throws InternalServerException
     *         We couldn't process your request because of an issue with the server. Try again later.
     * @throws ResourceNotFoundException
     *         We couldn't find the requested resource. Check that your resources exists and were created in the same
     *         Amazon Web Services Region as your request, and try your request again.
     * @throws ValidationException
     *         The request isn't valid. Check the syntax and try again.
     * @throws ThrottlingException
     *         The rate has been exceeded for this API. Please try again after a few minutes.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectCasesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectCasesClient.ListFieldOptions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcases-2022-10-03/ListFieldOptions" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListFieldOptionsResponse listFieldOptions(ListFieldOptionsRequest listFieldOptionsRequest)
            throws InternalServerException, ResourceNotFoundException, ValidationException, ThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, ConnectCasesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListFieldOptionsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListFieldOptionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listFieldOptionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCases");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListFieldOptions");

            return clientHandler.execute(new ClientExecutionParams<ListFieldOptionsRequest, ListFieldOptionsResponse>()
                    .withOperationName("ListFieldOptions").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listFieldOptionsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListFieldOptionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all of the field options for a field identifier in the domain.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listFieldOptions(software.amazon.awssdk.services.connectcases.model.ListFieldOptionsRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.connectcases.paginators.ListFieldOptionsIterable responses = client.listFieldOptionsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.connectcases.paginators.ListFieldOptionsIterable responses = client
     *             .listFieldOptionsPaginator(request);
     *     for (software.amazon.awssdk.services.connectcases.model.ListFieldOptionsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.connectcases.paginators.ListFieldOptionsIterable responses = client.listFieldOptionsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listFieldOptions(software.amazon.awssdk.services.connectcases.model.ListFieldOptionsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listFieldOptionsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerException
     *         We couldn't process your request because of an issue with the server. Try again later.
     * @throws ResourceNotFoundException
     *         We couldn't find the requested resource. Check that your resources exists and were created in the same
     *         Amazon Web Services Region as your request, and try your request again.
     * @throws ValidationException
     *         The request isn't valid. Check the syntax and try again.
     * @throws ThrottlingException
     *         The rate has been exceeded for this API. Please try again after a few minutes.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectCasesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectCasesClient.ListFieldOptions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcases-2022-10-03/ListFieldOptions" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListFieldOptionsIterable listFieldOptionsPaginator(ListFieldOptionsRequest listFieldOptionsRequest)
            throws InternalServerException, ResourceNotFoundException, ValidationException, ThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, ConnectCasesException {
        return new ListFieldOptionsIterable(this, applyPaginatorUserAgent(listFieldOptionsRequest));
    }

    /**
     * <p>
     * Lists all fields in a Cases domain.
     * </p>
     *
     * @param listFieldsRequest
     * @return Result of the ListFields operation returned by the service.
     * @throws InternalServerException
     *         We couldn't process your request because of an issue with the server. Try again later.
     * @throws ResourceNotFoundException
     *         We couldn't find the requested resource. Check that your resources exists and were created in the same
     *         Amazon Web Services Region as your request, and try your request again.
     * @throws ValidationException
     *         The request isn't valid. Check the syntax and try again.
     * @throws ThrottlingException
     *         The rate has been exceeded for this API. Please try again after a few minutes.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectCasesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectCasesClient.ListFields
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcases-2022-10-03/ListFields" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListFieldsResponse listFields(ListFieldsRequest listFieldsRequest) throws InternalServerException,
            ResourceNotFoundException, ValidationException, ThrottlingException, AccessDeniedException, AwsServiceException,
            SdkClientException, ConnectCasesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListFieldsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListFieldsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listFieldsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCases");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListFields");

            return clientHandler
                    .execute(new ClientExecutionParams<ListFieldsRequest, ListFieldsResponse>().withOperationName("ListFields")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listFieldsRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListFieldsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all fields in a Cases domain.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listFields(software.amazon.awssdk.services.connectcases.model.ListFieldsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.connectcases.paginators.ListFieldsIterable responses = client.listFieldsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.connectcases.paginators.ListFieldsIterable responses = client.listFieldsPaginator(request);
     *     for (software.amazon.awssdk.services.connectcases.model.ListFieldsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.connectcases.paginators.ListFieldsIterable responses = client.listFieldsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listFields(software.amazon.awssdk.services.connectcases.model.ListFieldsRequest)} operation.</b>
     * </p>
     *
     * @param listFieldsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerException
     *         We couldn't process your request because of an issue with the server. Try again later.
     * @throws ResourceNotFoundException
     *         We couldn't find the requested resource. Check that your resources exists and were created in the same
     *         Amazon Web Services Region as your request, and try your request again.
     * @throws ValidationException
     *         The request isn't valid. Check the syntax and try again.
     * @throws ThrottlingException
     *         The rate has been exceeded for this API. Please try again after a few minutes.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectCasesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectCasesClient.ListFields
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcases-2022-10-03/ListFields" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListFieldsIterable listFieldsPaginator(ListFieldsRequest listFieldsRequest) throws InternalServerException,
            ResourceNotFoundException, ValidationException, ThrottlingException, AccessDeniedException, AwsServiceException,
            SdkClientException, ConnectCasesException {
        return new ListFieldsIterable(this, applyPaginatorUserAgent(listFieldsRequest));
    }

    /**
     * <p>
     * Lists all layouts in the given cases domain. Each list item is a condensed summary object of the layout.
     * </p>
     *
     * @param listLayoutsRequest
     * @return Result of the ListLayouts operation returned by the service.
     * @throws InternalServerException
     *         We couldn't process your request because of an issue with the server. Try again later.
     * @throws ResourceNotFoundException
     *         We couldn't find the requested resource. Check that your resources exists and were created in the same
     *         Amazon Web Services Region as your request, and try your request again.
     * @throws ValidationException
     *         The request isn't valid. Check the syntax and try again.
     * @throws ThrottlingException
     *         The rate has been exceeded for this API. Please try again after a few minutes.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectCasesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectCasesClient.ListLayouts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcases-2022-10-03/ListLayouts" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListLayoutsResponse listLayouts(ListLayoutsRequest listLayoutsRequest) throws InternalServerException,
            ResourceNotFoundException, ValidationException, ThrottlingException, AccessDeniedException, AwsServiceException,
            SdkClientException, ConnectCasesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListLayoutsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListLayoutsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listLayoutsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCases");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListLayouts");

            return clientHandler.execute(new ClientExecutionParams<ListLayoutsRequest, ListLayoutsResponse>()
                    .withOperationName("ListLayouts").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listLayoutsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListLayoutsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all layouts in the given cases domain. Each list item is a condensed summary object of the layout.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listLayouts(software.amazon.awssdk.services.connectcases.model.ListLayoutsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.connectcases.paginators.ListLayoutsIterable responses = client.listLayoutsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.connectcases.paginators.ListLayoutsIterable responses = client.listLayoutsPaginator(request);
     *     for (software.amazon.awssdk.services.connectcases.model.ListLayoutsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.connectcases.paginators.ListLayoutsIterable responses = client.listLayoutsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listLayouts(software.amazon.awssdk.services.connectcases.model.ListLayoutsRequest)} operation.</b>
     * </p>
     *
     * @param listLayoutsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerException
     *         We couldn't process your request because of an issue with the server. Try again later.
     * @throws ResourceNotFoundException
     *         We couldn't find the requested resource. Check that your resources exists and were created in the same
     *         Amazon Web Services Region as your request, and try your request again.
     * @throws ValidationException
     *         The request isn't valid. Check the syntax and try again.
     * @throws ThrottlingException
     *         The rate has been exceeded for this API. Please try again after a few minutes.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectCasesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectCasesClient.ListLayouts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcases-2022-10-03/ListLayouts" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListLayoutsIterable listLayoutsPaginator(ListLayoutsRequest listLayoutsRequest) throws InternalServerException,
            ResourceNotFoundException, ValidationException, ThrottlingException, AccessDeniedException, AwsServiceException,
            SdkClientException, ConnectCasesException {
        return new ListLayoutsIterable(this, applyPaginatorUserAgent(listLayoutsRequest));
    }

    /**
     * <p>
     * Lists tags for a resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws InternalServerException
     *         We couldn't process your request because of an issue with the server. Try again later.
     * @throws ResourceNotFoundException
     *         We couldn't find the requested resource. Check that your resources exists and were created in the same
     *         Amazon Web Services Region as your request, and try your request again.
     * @throws ValidationException
     *         The request isn't valid. Check the syntax and try again.
     * @throws ThrottlingException
     *         The rate has been exceeded for this API. Please try again after a few minutes.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectCasesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectCasesClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcases-2022-10-03/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws InternalServerException, ResourceNotFoundException, ValidationException, ThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, ConnectCasesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCases");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all of the templates in a Cases domain. Each list item is a condensed summary object of the template.
     * </p>
     *
     * @param listTemplatesRequest
     * @return Result of the ListTemplates operation returned by the service.
     * @throws InternalServerException
     *         We couldn't process your request because of an issue with the server. Try again later.
     * @throws ResourceNotFoundException
     *         We couldn't find the requested resource. Check that your resources exists and were created in the same
     *         Amazon Web Services Region as your request, and try your request again.
     * @throws ValidationException
     *         The request isn't valid. Check the syntax and try again.
     * @throws ThrottlingException
     *         The rate has been exceeded for this API. Please try again after a few minutes.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectCasesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectCasesClient.ListTemplates
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcases-2022-10-03/ListTemplates" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListTemplatesResponse listTemplates(ListTemplatesRequest listTemplatesRequest) throws InternalServerException,
            ResourceNotFoundException, ValidationException, ThrottlingException, AccessDeniedException, AwsServiceException,
            SdkClientException, ConnectCasesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTemplatesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListTemplatesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTemplatesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCases");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTemplates");

            return clientHandler.execute(new ClientExecutionParams<ListTemplatesRequest, ListTemplatesResponse>()
                    .withOperationName("ListTemplates").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listTemplatesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTemplatesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all of the templates in a Cases domain. Each list item is a condensed summary object of the template.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listTemplates(software.amazon.awssdk.services.connectcases.model.ListTemplatesRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.connectcases.paginators.ListTemplatesIterable responses = client.listTemplatesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.connectcases.paginators.ListTemplatesIterable responses = client
     *             .listTemplatesPaginator(request);
     *     for (software.amazon.awssdk.services.connectcases.model.ListTemplatesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.connectcases.paginators.ListTemplatesIterable responses = client.listTemplatesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listTemplates(software.amazon.awssdk.services.connectcases.model.ListTemplatesRequest)} operation.</b>
     * </p>
     *
     * @param listTemplatesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerException
     *         We couldn't process your request because of an issue with the server. Try again later.
     * @throws ResourceNotFoundException
     *         We couldn't find the requested resource. Check that your resources exists and were created in the same
     *         Amazon Web Services Region as your request, and try your request again.
     * @throws ValidationException
     *         The request isn't valid. Check the syntax and try again.
     * @throws ThrottlingException
     *         The rate has been exceeded for this API. Please try again after a few minutes.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectCasesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectCasesClient.ListTemplates
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcases-2022-10-03/ListTemplates" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListTemplatesIterable listTemplatesPaginator(ListTemplatesRequest listTemplatesRequest)
            throws InternalServerException, ResourceNotFoundException, ValidationException, ThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, ConnectCasesException {
        return new ListTemplatesIterable(this, applyPaginatorUserAgent(listTemplatesRequest));
    }

    /**
     * <p>
     * API for adding case event publishing configuration
     * </p>
     *
     * @param putCaseEventConfigurationRequest
     * @return Result of the PutCaseEventConfiguration operation returned by the service.
     * @throws InternalServerException
     *         We couldn't process your request because of an issue with the server. Try again later.
     * @throws ResourceNotFoundException
     *         We couldn't find the requested resource. Check that your resources exists and were created in the same
     *         Amazon Web Services Region as your request, and try your request again.
     * @throws ValidationException
     *         The request isn't valid. Check the syntax and try again.
     * @throws ThrottlingException
     *         The rate has been exceeded for this API. Please try again after a few minutes.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectCasesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectCasesClient.PutCaseEventConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcases-2022-10-03/PutCaseEventConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public PutCaseEventConfigurationResponse putCaseEventConfiguration(
            PutCaseEventConfigurationRequest putCaseEventConfigurationRequest) throws InternalServerException,
            ResourceNotFoundException, ValidationException, ThrottlingException, AccessDeniedException, AwsServiceException,
            SdkClientException, ConnectCasesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutCaseEventConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, PutCaseEventConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putCaseEventConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCases");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutCaseEventConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<PutCaseEventConfigurationRequest, PutCaseEventConfigurationResponse>()
                            .withOperationName("PutCaseEventConfiguration").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(putCaseEventConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new PutCaseEventConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Searches for cases within their associated Cases domain. Search results are returned as a paginated list of
     * abridged case documents.
     * </p>
     *
     * @param searchCasesRequest
     * @return Result of the SearchCases operation returned by the service.
     * @throws InternalServerException
     *         We couldn't process your request because of an issue with the server. Try again later.
     * @throws ResourceNotFoundException
     *         We couldn't find the requested resource. Check that your resources exists and were created in the same
     *         Amazon Web Services Region as your request, and try your request again.
     * @throws ValidationException
     *         The request isn't valid. Check the syntax and try again.
     * @throws ThrottlingException
     *         The rate has been exceeded for this API. Please try again after a few minutes.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectCasesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectCasesClient.SearchCases
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcases-2022-10-03/SearchCases" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public SearchCasesResponse searchCases(SearchCasesRequest searchCasesRequest) throws InternalServerException,
            ResourceNotFoundException, ValidationException, ThrottlingException, AccessDeniedException, AwsServiceException,
            SdkClientException, ConnectCasesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<SearchCasesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                SearchCasesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, searchCasesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCases");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SearchCases");

            return clientHandler.execute(new ClientExecutionParams<SearchCasesRequest, SearchCasesResponse>()
                    .withOperationName("SearchCases").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(searchCasesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new SearchCasesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Searches for cases within their associated Cases domain. Search results are returned as a paginated list of
     * abridged case documents.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #searchCases(software.amazon.awssdk.services.connectcases.model.SearchCasesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.connectcases.paginators.SearchCasesIterable responses = client.searchCasesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.connectcases.paginators.SearchCasesIterable responses = client.searchCasesPaginator(request);
     *     for (software.amazon.awssdk.services.connectcases.model.SearchCasesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.connectcases.paginators.SearchCasesIterable responses = client.searchCasesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #searchCases(software.amazon.awssdk.services.connectcases.model.SearchCasesRequest)} operation.</b>
     * </p>
     *
     * @param searchCasesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerException
     *         We couldn't process your request because of an issue with the server. Try again later.
     * @throws ResourceNotFoundException
     *         We couldn't find the requested resource. Check that your resources exists and were created in the same
     *         Amazon Web Services Region as your request, and try your request again.
     * @throws ValidationException
     *         The request isn't valid. Check the syntax and try again.
     * @throws ThrottlingException
     *         The rate has been exceeded for this API. Please try again after a few minutes.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectCasesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectCasesClient.SearchCases
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcases-2022-10-03/SearchCases" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public SearchCasesIterable searchCasesPaginator(SearchCasesRequest searchCasesRequest) throws InternalServerException,
            ResourceNotFoundException, ValidationException, ThrottlingException, AccessDeniedException, AwsServiceException,
            SdkClientException, ConnectCasesException {
        return new SearchCasesIterable(this, applyPaginatorUserAgent(searchCasesRequest));
    }

    /**
     * <p>
     * Searches for related items that are associated with a case.
     * </p>
     * <note>
     * <p>
     * If no filters are provided, this returns all related items associated with a case.
     * </p>
     * </note>
     *
     * @param searchRelatedItemsRequest
     * @return Result of the SearchRelatedItems operation returned by the service.
     * @throws InternalServerException
     *         We couldn't process your request because of an issue with the server. Try again later.
     * @throws ResourceNotFoundException
     *         We couldn't find the requested resource. Check that your resources exists and were created in the same
     *         Amazon Web Services Region as your request, and try your request again.
     * @throws ValidationException
     *         The request isn't valid. Check the syntax and try again.
     * @throws ThrottlingException
     *         The rate has been exceeded for this API. Please try again after a few minutes.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectCasesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectCasesClient.SearchRelatedItems
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcases-2022-10-03/SearchRelatedItems"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public SearchRelatedItemsResponse searchRelatedItems(SearchRelatedItemsRequest searchRelatedItemsRequest)
            throws InternalServerException, ResourceNotFoundException, ValidationException, ThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, ConnectCasesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<SearchRelatedItemsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, SearchRelatedItemsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, searchRelatedItemsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCases");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SearchRelatedItems");

            return clientHandler.execute(new ClientExecutionParams<SearchRelatedItemsRequest, SearchRelatedItemsResponse>()
                    .withOperationName("SearchRelatedItems").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(searchRelatedItemsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new SearchRelatedItemsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Searches for related items that are associated with a case.
     * </p>
     * <note>
     * <p>
     * If no filters are provided, this returns all related items associated with a case.
     * </p>
     * </note><br/>
     * <p>
     * This is a variant of
     * {@link #searchRelatedItems(software.amazon.awssdk.services.connectcases.model.SearchRelatedItemsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.connectcases.paginators.SearchRelatedItemsIterable responses = client.searchRelatedItemsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.connectcases.paginators.SearchRelatedItemsIterable responses = client
     *             .searchRelatedItemsPaginator(request);
     *     for (software.amazon.awssdk.services.connectcases.model.SearchRelatedItemsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.connectcases.paginators.SearchRelatedItemsIterable responses = client.searchRelatedItemsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #searchRelatedItems(software.amazon.awssdk.services.connectcases.model.SearchRelatedItemsRequest)}
     * operation.</b>
     * </p>
     *
     * @param searchRelatedItemsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerException
     *         We couldn't process your request because of an issue with the server. Try again later.
     * @throws ResourceNotFoundException
     *         We couldn't find the requested resource. Check that your resources exists and were created in the same
     *         Amazon Web Services Region as your request, and try your request again.
     * @throws ValidationException
     *         The request isn't valid. Check the syntax and try again.
     * @throws ThrottlingException
     *         The rate has been exceeded for this API. Please try again after a few minutes.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectCasesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectCasesClient.SearchRelatedItems
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcases-2022-10-03/SearchRelatedItems"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public SearchRelatedItemsIterable searchRelatedItemsPaginator(SearchRelatedItemsRequest searchRelatedItemsRequest)
            throws InternalServerException, ResourceNotFoundException, ValidationException, ThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, ConnectCasesException {
        return new SearchRelatedItemsIterable(this, applyPaginatorUserAgent(searchRelatedItemsRequest));
    }

    /**
     * <p>
     * Adds tags to a resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws InternalServerException
     *         We couldn't process your request because of an issue with the server. Try again later.
     * @throws ResourceNotFoundException
     *         We couldn't find the requested resource. Check that your resources exists and were created in the same
     *         Amazon Web Services Region as your request, and try your request again.
     * @throws ValidationException
     *         The request isn't valid. Check the syntax and try again.
     * @throws ThrottlingException
     *         The rate has been exceeded for this API. Please try again after a few minutes.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectCasesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectCasesClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcases-2022-10-03/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws InternalServerException,
            ResourceNotFoundException, ValidationException, ThrottlingException, AccessDeniedException, AwsServiceException,
            SdkClientException, ConnectCasesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCases");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(tagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Untags a resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws InternalServerException
     *         We couldn't process your request because of an issue with the server. Try again later.
     * @throws ResourceNotFoundException
     *         We couldn't find the requested resource. Check that your resources exists and were created in the same
     *         Amazon Web Services Region as your request, and try your request again.
     * @throws ValidationException
     *         The request isn't valid. Check the syntax and try again.
     * @throws ThrottlingException
     *         The rate has been exceeded for this API. Please try again after a few minutes.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectCasesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectCasesClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcases-2022-10-03/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws InternalServerException,
            ResourceNotFoundException, ValidationException, ThrottlingException, AccessDeniedException, AwsServiceException,
            SdkClientException, ConnectCasesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCases");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the values of fields on a case. Fields to be updated are received as an array of id/value pairs identical
     * to the <code>CreateCase</code> input .
     * </p>
     * <p>
     * If the action is successful, the service sends back an HTTP 200 response with an empty HTTP body.
     * </p>
     *
     * @param updateCaseRequest
     * @return Result of the UpdateCase operation returned by the service.
     * @throws InternalServerException
     *         We couldn't process your request because of an issue with the server. Try again later.
     * @throws ResourceNotFoundException
     *         We couldn't find the requested resource. Check that your resources exists and were created in the same
     *         Amazon Web Services Region as your request, and try your request again.
     * @throws ValidationException
     *         The request isn't valid. Check the syntax and try again.
     * @throws ThrottlingException
     *         The rate has been exceeded for this API. Please try again after a few minutes.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectCasesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectCasesClient.UpdateCase
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcases-2022-10-03/UpdateCase" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateCaseResponse updateCase(UpdateCaseRequest updateCaseRequest) throws InternalServerException,
            ResourceNotFoundException, ValidationException, ThrottlingException, AccessDeniedException, AwsServiceException,
            SdkClientException, ConnectCasesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateCaseResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateCaseResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateCaseRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCases");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateCase");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateCaseRequest, UpdateCaseResponse>().withOperationName("UpdateCase")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateCaseRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateCaseRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the properties of an existing field.
     * </p>
     *
     * @param updateFieldRequest
     * @return Result of the UpdateField operation returned by the service.
     * @throws InternalServerException
     *         We couldn't process your request because of an issue with the server. Try again later.
     * @throws ResourceNotFoundException
     *         We couldn't find the requested resource. Check that your resources exists and were created in the same
     *         Amazon Web Services Region as your request, and try your request again.
     * @throws ValidationException
     *         The request isn't valid. Check the syntax and try again.
     * @throws ThrottlingException
     *         The rate has been exceeded for this API. Please try again after a few minutes.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ConflictException
     *         The requested operation would cause a conflict with the current state of a service resource associated
     *         with the request. Resolve the conflict before retrying this request. See the accompanying error message
     *         for details.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectCasesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectCasesClient.UpdateField
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcases-2022-10-03/UpdateField" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateFieldResponse updateField(UpdateFieldRequest updateFieldRequest) throws InternalServerException,
            ResourceNotFoundException, ValidationException, ThrottlingException, AccessDeniedException, ConflictException,
            AwsServiceException, SdkClientException, ConnectCasesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateFieldResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateFieldResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateFieldRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCases");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateField");

            return clientHandler.execute(new ClientExecutionParams<UpdateFieldRequest, UpdateFieldResponse>()
                    .withOperationName("UpdateField").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateFieldRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateFieldRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the attributes of an existing layout.
     * </p>
     * <p>
     * If the action is successful, the service sends back an HTTP 200 response with an empty HTTP body.
     * </p>
     * <p>
     * A <code>ValidationException</code> is returned when you add non-existent <code>fieldIds</code> to a layout.
     * </p>
     * <note>
     * <p>
     * Title and Status fields cannot be part of layouts because they are not configurable.
     * </p>
     * </note>
     *
     * @param updateLayoutRequest
     * @return Result of the UpdateLayout operation returned by the service.
     * @throws InternalServerException
     *         We couldn't process your request because of an issue with the server. Try again later.
     * @throws ResourceNotFoundException
     *         We couldn't find the requested resource. Check that your resources exists and were created in the same
     *         Amazon Web Services Region as your request, and try your request again.
     * @throws ValidationException
     *         The request isn't valid. Check the syntax and try again.
     * @throws ThrottlingException
     *         The rate has been exceeded for this API. Please try again after a few minutes.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ConflictException
     *         The requested operation would cause a conflict with the current state of a service resource associated
     *         with the request. Resolve the conflict before retrying this request. See the accompanying error message
     *         for details.
     * @throws ServiceQuotaExceededException
     *         The service quota has been exceeded. For a list of service quotas, see <a
     *         href="https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html">Amazon
     *         Connect Service Quotas</a> in the <i>Amazon Connect Administrator Guide</i>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectCasesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectCasesClient.UpdateLayout
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcases-2022-10-03/UpdateLayout" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateLayoutResponse updateLayout(UpdateLayoutRequest updateLayoutRequest) throws InternalServerException,
            ResourceNotFoundException, ValidationException, ThrottlingException, AccessDeniedException, ConflictException,
            ServiceQuotaExceededException, AwsServiceException, SdkClientException, ConnectCasesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateLayoutResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateLayoutResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateLayoutRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCases");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateLayout");

            return clientHandler.execute(new ClientExecutionParams<UpdateLayoutRequest, UpdateLayoutResponse>()
                    .withOperationName("UpdateLayout").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateLayoutRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateLayoutRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the attributes of an existing template. The template attributes that can be modified include
     * <code>name</code>, <code>description</code>, <code>layoutConfiguration</code>, <code>requiredFields</code>, and
     * <code>status</code>. At least one of these attributes must not be null. If a null value is provided for a given
     * attribute, that attribute is ignored and its current value is preserved.
     * </p>
     *
     * @param updateTemplateRequest
     * @return Result of the UpdateTemplate operation returned by the service.
     * @throws InternalServerException
     *         We couldn't process your request because of an issue with the server. Try again later.
     * @throws ResourceNotFoundException
     *         We couldn't find the requested resource. Check that your resources exists and were created in the same
     *         Amazon Web Services Region as your request, and try your request again.
     * @throws ValidationException
     *         The request isn't valid. Check the syntax and try again.
     * @throws ThrottlingException
     *         The rate has been exceeded for this API. Please try again after a few minutes.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ConflictException
     *         The requested operation would cause a conflict with the current state of a service resource associated
     *         with the request. Resolve the conflict before retrying this request. See the accompanying error message
     *         for details.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectCasesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectCasesClient.UpdateTemplate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcases-2022-10-03/UpdateTemplate" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateTemplateResponse updateTemplate(UpdateTemplateRequest updateTemplateRequest) throws InternalServerException,
            ResourceNotFoundException, ValidationException, ThrottlingException, AccessDeniedException, ConflictException,
            AwsServiceException, SdkClientException, ConnectCasesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateTemplateResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateTemplateResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateTemplateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCases");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateTemplate");

            return clientHandler.execute(new ClientExecutionParams<UpdateTemplateRequest, UpdateTemplateResponse>()
                    .withOperationName("UpdateTemplate").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateTemplateRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateTemplateRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(ConnectCasesException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends ConnectCasesRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
