/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.dataexchange.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An asset in AWS Data Exchange is a piece of data (S3 object) or a means of fulfilling data (Amazon Redshift datashare
 * or Amazon API Gateway API). The asset can be a structured data file, an image file, or some other data file that can
 * be stored as an S3 object, an Amazon API Gateway API, or an Amazon Redshift datashare. When you create an import job
 * for your files, API Gateway APIs, or Amazon Redshift datashares, you create an asset in AWS Data Exchange.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AssetEntry implements SdkPojo, Serializable, ToCopyableBuilder<AssetEntry.Builder, AssetEntry> {
    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Arn")
            .getter(getter(AssetEntry::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Arn").build()).build();

    private static final SdkField<AssetDetails> ASSET_DETAILS_FIELD = SdkField.<AssetDetails> builder(MarshallingType.SDK_POJO)
            .memberName("AssetDetails").getter(getter(AssetEntry::assetDetails)).setter(setter(Builder::assetDetails))
            .constructor(AssetDetails::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AssetDetails").build()).build();

    private static final SdkField<String> ASSET_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AssetType").getter(getter(AssetEntry::assetTypeAsString)).setter(setter(Builder::assetType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AssetType").build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreatedAt")
            .getter(getter(AssetEntry::createdAt))
            .setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatedAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<String> DATA_SET_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DataSetId").getter(getter(AssetEntry::dataSetId)).setter(setter(Builder::dataSetId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DataSetId").build()).build();

    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Id")
            .getter(getter(AssetEntry::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Id").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(AssetEntry::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> REVISION_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RevisionId").getter(getter(AssetEntry::revisionId)).setter(setter(Builder::revisionId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RevisionId").build()).build();

    private static final SdkField<String> SOURCE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SourceId").getter(getter(AssetEntry::sourceId)).setter(setter(Builder::sourceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceId").build()).build();

    private static final SdkField<Instant> UPDATED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("UpdatedAt")
            .getter(getter(AssetEntry::updatedAt))
            .setter(setter(Builder::updatedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UpdatedAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ARN_FIELD,
            ASSET_DETAILS_FIELD, ASSET_TYPE_FIELD, CREATED_AT_FIELD, DATA_SET_ID_FIELD, ID_FIELD, NAME_FIELD, REVISION_ID_FIELD,
            SOURCE_ID_FIELD, UPDATED_AT_FIELD));

    private static final long serialVersionUID = 1L;

    private final String arn;

    private final AssetDetails assetDetails;

    private final String assetType;

    private final Instant createdAt;

    private final String dataSetId;

    private final String id;

    private final String name;

    private final String revisionId;

    private final String sourceId;

    private final Instant updatedAt;

    private AssetEntry(BuilderImpl builder) {
        this.arn = builder.arn;
        this.assetDetails = builder.assetDetails;
        this.assetType = builder.assetType;
        this.createdAt = builder.createdAt;
        this.dataSetId = builder.dataSetId;
        this.id = builder.id;
        this.name = builder.name;
        this.revisionId = builder.revisionId;
        this.sourceId = builder.sourceId;
        this.updatedAt = builder.updatedAt;
    }

    /**
     * <p>
     * The ARN for the asset.
     * </p>
     * 
     * @return The ARN for the asset.
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * Information about the asset.
     * </p>
     * 
     * @return Information about the asset.
     */
    public final AssetDetails assetDetails() {
        return assetDetails;
    }

    /**
     * <p>
     * The type of asset that is added to a data set.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #assetType} will
     * return {@link AssetType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #assetTypeAsString}.
     * </p>
     * 
     * @return The type of asset that is added to a data set.
     * @see AssetType
     */
    public final AssetType assetType() {
        return AssetType.fromValue(assetType);
    }

    /**
     * <p>
     * The type of asset that is added to a data set.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #assetType} will
     * return {@link AssetType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #assetTypeAsString}.
     * </p>
     * 
     * @return The type of asset that is added to a data set.
     * @see AssetType
     */
    public final String assetTypeAsString() {
        return assetType;
    }

    /**
     * <p>
     * The date and time that the asset was created, in ISO 8601 format.
     * </p>
     * 
     * @return The date and time that the asset was created, in ISO 8601 format.
     */
    public final Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * The unique identifier for the data set associated with this asset.
     * </p>
     * 
     * @return The unique identifier for the data set associated with this asset.
     */
    public final String dataSetId() {
        return dataSetId;
    }

    /**
     * <p>
     * The unique identifier for the asset.
     * </p>
     * 
     * @return The unique identifier for the asset.
     */
    public final String id() {
        return id;
    }

    /**
     * <p>
     * The name of the asset. When importing from Amazon S3, the S3 object key is used as the asset name. When exporting
     * to Amazon S3, the asset name is used as default target S3 object key. When importing from Amazon API Gateway API,
     * the API name is used as the asset name. When importing from Amazon Redshift, the datashare name is used as the
     * asset name.
     * </p>
     * 
     * @return The name of the asset. When importing from Amazon S3, the S3 object key is used as the asset name. When
     *         exporting to Amazon S3, the asset name is used as default target S3 object key. When importing from
     *         Amazon API Gateway API, the API name is used as the asset name. When importing from Amazon Redshift, the
     *         datashare name is used as the asset name.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The unique identifier for the revision associated with this asset.
     * </p>
     * 
     * @return The unique identifier for the revision associated with this asset.
     */
    public final String revisionId() {
        return revisionId;
    }

    /**
     * <p>
     * The asset ID of the owned asset corresponding to the entitled asset being viewed. This parameter is returned when
     * an asset owner is viewing the entitled copy of its owned asset.
     * </p>
     * 
     * @return The asset ID of the owned asset corresponding to the entitled asset being viewed. This parameter is
     *         returned when an asset owner is viewing the entitled copy of its owned asset.
     */
    public final String sourceId() {
        return sourceId;
    }

    /**
     * <p>
     * The date and time that the asset was last updated, in ISO 8601 format.
     * </p>
     * 
     * @return The date and time that the asset was last updated, in ISO 8601 format.
     */
    public final Instant updatedAt() {
        return updatedAt;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(assetDetails());
        hashCode = 31 * hashCode + Objects.hashCode(assetTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(dataSetId());
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(revisionId());
        hashCode = 31 * hashCode + Objects.hashCode(sourceId());
        hashCode = 31 * hashCode + Objects.hashCode(updatedAt());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AssetEntry)) {
            return false;
        }
        AssetEntry other = (AssetEntry) obj;
        return Objects.equals(arn(), other.arn()) && Objects.equals(assetDetails(), other.assetDetails())
                && Objects.equals(assetTypeAsString(), other.assetTypeAsString())
                && Objects.equals(createdAt(), other.createdAt()) && Objects.equals(dataSetId(), other.dataSetId())
                && Objects.equals(id(), other.id()) && Objects.equals(name(), other.name())
                && Objects.equals(revisionId(), other.revisionId()) && Objects.equals(sourceId(), other.sourceId())
                && Objects.equals(updatedAt(), other.updatedAt());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AssetEntry").add("Arn", arn()).add("AssetDetails", assetDetails())
                .add("AssetType", assetTypeAsString()).add("CreatedAt", createdAt()).add("DataSetId", dataSetId())
                .add("Id", id()).add("Name", name()).add("RevisionId", revisionId()).add("SourceId", sourceId())
                .add("UpdatedAt", updatedAt()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "AssetDetails":
            return Optional.ofNullable(clazz.cast(assetDetails()));
        case "AssetType":
            return Optional.ofNullable(clazz.cast(assetTypeAsString()));
        case "CreatedAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "DataSetId":
            return Optional.ofNullable(clazz.cast(dataSetId()));
        case "Id":
            return Optional.ofNullable(clazz.cast(id()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "RevisionId":
            return Optional.ofNullable(clazz.cast(revisionId()));
        case "SourceId":
            return Optional.ofNullable(clazz.cast(sourceId()));
        case "UpdatedAt":
            return Optional.ofNullable(clazz.cast(updatedAt()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<AssetEntry, T> g) {
        return obj -> g.apply((AssetEntry) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AssetEntry> {
        /**
         * <p>
         * The ARN for the asset.
         * </p>
         * 
         * @param arn
         *        The ARN for the asset.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * Information about the asset.
         * </p>
         * 
         * @param assetDetails
         *        Information about the asset.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder assetDetails(AssetDetails assetDetails);

        /**
         * <p>
         * Information about the asset.
         * </p>
         * This is a convenience method that creates an instance of the {@link AssetDetails.Builder} avoiding the need
         * to create one manually via {@link AssetDetails#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AssetDetails.Builder#build()} is called immediately and its
         * result is passed to {@link #assetDetails(AssetDetails)}.
         * 
         * @param assetDetails
         *        a consumer that will call methods on {@link AssetDetails.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #assetDetails(AssetDetails)
         */
        default Builder assetDetails(Consumer<AssetDetails.Builder> assetDetails) {
            return assetDetails(AssetDetails.builder().applyMutation(assetDetails).build());
        }

        /**
         * <p>
         * The type of asset that is added to a data set.
         * </p>
         * 
         * @param assetType
         *        The type of asset that is added to a data set.
         * @see AssetType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AssetType
         */
        Builder assetType(String assetType);

        /**
         * <p>
         * The type of asset that is added to a data set.
         * </p>
         * 
         * @param assetType
         *        The type of asset that is added to a data set.
         * @see AssetType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AssetType
         */
        Builder assetType(AssetType assetType);

        /**
         * <p>
         * The date and time that the asset was created, in ISO 8601 format.
         * </p>
         * 
         * @param createdAt
         *        The date and time that the asset was created, in ISO 8601 format.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * The unique identifier for the data set associated with this asset.
         * </p>
         * 
         * @param dataSetId
         *        The unique identifier for the data set associated with this asset.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataSetId(String dataSetId);

        /**
         * <p>
         * The unique identifier for the asset.
         * </p>
         * 
         * @param id
         *        The unique identifier for the asset.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The name of the asset. When importing from Amazon S3, the S3 object key is used as the asset name. When
         * exporting to Amazon S3, the asset name is used as default target S3 object key. When importing from Amazon
         * API Gateway API, the API name is used as the asset name. When importing from Amazon Redshift, the datashare
         * name is used as the asset name.
         * </p>
         * 
         * @param name
         *        The name of the asset. When importing from Amazon S3, the S3 object key is used as the asset name.
         *        When exporting to Amazon S3, the asset name is used as default target S3 object key. When importing
         *        from Amazon API Gateway API, the API name is used as the asset name. When importing from Amazon
         *        Redshift, the datashare name is used as the asset name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The unique identifier for the revision associated with this asset.
         * </p>
         * 
         * @param revisionId
         *        The unique identifier for the revision associated with this asset.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder revisionId(String revisionId);

        /**
         * <p>
         * The asset ID of the owned asset corresponding to the entitled asset being viewed. This parameter is returned
         * when an asset owner is viewing the entitled copy of its owned asset.
         * </p>
         * 
         * @param sourceId
         *        The asset ID of the owned asset corresponding to the entitled asset being viewed. This parameter is
         *        returned when an asset owner is viewing the entitled copy of its owned asset.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceId(String sourceId);

        /**
         * <p>
         * The date and time that the asset was last updated, in ISO 8601 format.
         * </p>
         * 
         * @param updatedAt
         *        The date and time that the asset was last updated, in ISO 8601 format.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder updatedAt(Instant updatedAt);
    }

    static final class BuilderImpl implements Builder {
        private String arn;

        private AssetDetails assetDetails;

        private String assetType;

        private Instant createdAt;

        private String dataSetId;

        private String id;

        private String name;

        private String revisionId;

        private String sourceId;

        private Instant updatedAt;

        private BuilderImpl() {
        }

        private BuilderImpl(AssetEntry model) {
            arn(model.arn);
            assetDetails(model.assetDetails);
            assetType(model.assetType);
            createdAt(model.createdAt);
            dataSetId(model.dataSetId);
            id(model.id);
            name(model.name);
            revisionId(model.revisionId);
            sourceId(model.sourceId);
            updatedAt(model.updatedAt);
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final AssetDetails.Builder getAssetDetails() {
            return assetDetails != null ? assetDetails.toBuilder() : null;
        }

        public final void setAssetDetails(AssetDetails.BuilderImpl assetDetails) {
            this.assetDetails = assetDetails != null ? assetDetails.build() : null;
        }

        @Override
        public final Builder assetDetails(AssetDetails assetDetails) {
            this.assetDetails = assetDetails;
            return this;
        }

        public final String getAssetType() {
            return assetType;
        }

        public final void setAssetType(String assetType) {
            this.assetType = assetType;
        }

        @Override
        public final Builder assetType(String assetType) {
            this.assetType = assetType;
            return this;
        }

        @Override
        public final Builder assetType(AssetType assetType) {
            this.assetType(assetType == null ? null : assetType.toString());
            return this;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final String getDataSetId() {
            return dataSetId;
        }

        public final void setDataSetId(String dataSetId) {
            this.dataSetId = dataSetId;
        }

        @Override
        public final Builder dataSetId(String dataSetId) {
            this.dataSetId = dataSetId;
            return this;
        }

        public final String getId() {
            return id;
        }

        public final void setId(String id) {
            this.id = id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getRevisionId() {
            return revisionId;
        }

        public final void setRevisionId(String revisionId) {
            this.revisionId = revisionId;
        }

        @Override
        public final Builder revisionId(String revisionId) {
            this.revisionId = revisionId;
            return this;
        }

        public final String getSourceId() {
            return sourceId;
        }

        public final void setSourceId(String sourceId) {
            this.sourceId = sourceId;
        }

        @Override
        public final Builder sourceId(String sourceId) {
            this.sourceId = sourceId;
            return this;
        }

        public final Instant getUpdatedAt() {
            return updatedAt;
        }

        public final void setUpdatedAt(Instant updatedAt) {
            this.updatedAt = updatedAt;
        }

        @Override
        public final Builder updatedAt(Instant updatedAt) {
            this.updatedAt = updatedAt;
            return this;
        }

        @Override
        public AssetEntry build() {
            return new AssetEntry(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
