/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.devopsguru.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A single query to be processed. Use these parameters to query the Performance Insights
 * <code>GetResourceMetrics</code> API to retrieve the metrics for an anomaly. For more information, see
 * <code> <a href="https://docs.aws.amazon.com/performance-insights/latest/APIReference/API_GetResourceMetrics.html">GetResourceMetrics</a> </code>
 * in the <i>Amazon RDS Performance Insights API Reference</i>.
 * </p>
 * <p>
 * Amazon RDS Performance Insights enables you to monitor and explore different dimensions of database load based on
 * data captured from a running DB instance. DB load is measured as average active sessions. Performance Insights
 * provides the data to API consumers as a two-dimensional time-series dataset. The time dimension provides DB load data
 * for each time point in the queried time range. Each time point decomposes overall load in relation to the requested
 * dimensions, measured at that time point. Examples include SQL, Wait event, User, and Host.
 * </p>
 * <ul>
 * <li>
 * <p>
 * To learn more about Performance Insights and Amazon Aurora DB instances, go to the <a
 * href="https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_PerfInsights.html"> Amazon Aurora User
 * Guide</a>.
 * </p>
 * </li>
 * <li>
 * <p>
 * To learn more about Performance Insights and Amazon RDS DB instances, go to the <a
 * href="https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_PerfInsights.html"> Amazon RDS User Guide</a>.
 * </p>
 * </li>
 * </ul>
 */
@Generated("software.amazon.awssdk:codegen")
public final class PerformanceInsightsMetricQuery implements SdkPojo, Serializable,
        ToCopyableBuilder<PerformanceInsightsMetricQuery.Builder, PerformanceInsightsMetricQuery> {
    private static final SdkField<String> METRIC_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Metric")
            .getter(getter(PerformanceInsightsMetricQuery::metric)).setter(setter(Builder::metric))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Metric").build()).build();

    private static final SdkField<PerformanceInsightsMetricDimensionGroup> GROUP_BY_FIELD = SdkField
            .<PerformanceInsightsMetricDimensionGroup> builder(MarshallingType.SDK_POJO).memberName("GroupBy")
            .getter(getter(PerformanceInsightsMetricQuery::groupBy)).setter(setter(Builder::groupBy))
            .constructor(PerformanceInsightsMetricDimensionGroup::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GroupBy").build()).build();

    private static final SdkField<Map<String, String>> FILTER_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("Filter")
            .getter(getter(PerformanceInsightsMetricQuery::filter))
            .setter(setter(Builder::filter))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Filter").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(METRIC_FIELD, GROUP_BY_FIELD,
            FILTER_FIELD));

    private static final long serialVersionUID = 1L;

    private final String metric;

    private final PerformanceInsightsMetricDimensionGroup groupBy;

    private final Map<String, String> filter;

    private PerformanceInsightsMetricQuery(BuilderImpl builder) {
        this.metric = builder.metric;
        this.groupBy = builder.groupBy;
        this.filter = builder.filter;
    }

    /**
     * <p>
     * The name of the meteric used used when querying an Performance Insights <code>GetResourceMetrics</code> API for
     * anomaly metrics.
     * </p>
     * <p>
     * Valid values for <code>Metric</code> are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>db.load.avg</code> - a scaled representation of the number of active sessions for the database engine.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>db.sampledload.avg</code> - the raw number of active sessions for the database engine.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the number of active sessions is less than an internal Performance Insights threshold,
     * <code>db.load.avg</code> and <code>db.sampledload.avg</code> are the same value. If the number of active sessions
     * is greater than the internal threshold, Performance Insights samples the active sessions, with
     * <code>db.load.avg</code> showing the scaled values, <code>db.sampledload.avg</code> showing the raw values, and
     * <code>db.sampledload.avg</code> less than <code>db.load.avg</code>. For most use cases, you can query
     * <code>db.load.avg</code> only.
     * </p>
     * 
     * @return The name of the meteric used used when querying an Performance Insights <code>GetResourceMetrics</code>
     *         API for anomaly metrics.</p>
     *         <p>
     *         Valid values for <code>Metric</code> are:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>db.load.avg</code> - a scaled representation of the number of active sessions for the database
     *         engine.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>db.sampledload.avg</code> - the raw number of active sessions for the database engine.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If the number of active sessions is less than an internal Performance Insights threshold,
     *         <code>db.load.avg</code> and <code>db.sampledload.avg</code> are the same value. If the number of active
     *         sessions is greater than the internal threshold, Performance Insights samples the active sessions, with
     *         <code>db.load.avg</code> showing the scaled values, <code>db.sampledload.avg</code> showing the raw
     *         values, and <code>db.sampledload.avg</code> less than <code>db.load.avg</code>. For most use cases, you
     *         can query <code>db.load.avg</code> only.
     */
    public final String metric() {
        return metric;
    }

    /**
     * <p>
     * The specification for how to aggregate the data points from a Performance Insights
     * <code>GetResourceMetrics</code> API query. The Performance Insights query returns all of the dimensions within
     * that group, unless you provide the names of specific dimensions within that group. You can also request that
     * Performance Insights return a limited number of values for a dimension.
     * </p>
     * 
     * @return The specification for how to aggregate the data points from a Performance Insights
     *         <code>GetResourceMetrics</code> API query. The Performance Insights query returns all of the dimensions
     *         within that group, unless you provide the names of specific dimensions within that group. You can also
     *         request that Performance Insights return a limited number of values for a dimension.
     */
    public final PerformanceInsightsMetricDimensionGroup groupBy() {
        return groupBy;
    }

    /**
     * For responses, this returns true if the service returned a value for the Filter property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasFilter() {
        return filter != null && !(filter instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * One or more filters to apply to a Performance Insights <code>GetResourceMetrics</code> API query. Restrictions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Any number of filters by the same dimension, as specified in the <code>GroupBy</code> parameter.
     * </p>
     * </li>
     * <li>
     * <p>
     * A single filter for any other dimension in this dimension group.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasFilter} method.
     * </p>
     * 
     * @return One or more filters to apply to a Performance Insights <code>GetResourceMetrics</code> API query.
     *         Restrictions:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Any number of filters by the same dimension, as specified in the <code>GroupBy</code> parameter.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A single filter for any other dimension in this dimension group.
     *         </p>
     *         </li>
     */
    public final Map<String, String> filter() {
        return filter;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(metric());
        hashCode = 31 * hashCode + Objects.hashCode(groupBy());
        hashCode = 31 * hashCode + Objects.hashCode(hasFilter() ? filter() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PerformanceInsightsMetricQuery)) {
            return false;
        }
        PerformanceInsightsMetricQuery other = (PerformanceInsightsMetricQuery) obj;
        return Objects.equals(metric(), other.metric()) && Objects.equals(groupBy(), other.groupBy())
                && hasFilter() == other.hasFilter() && Objects.equals(filter(), other.filter());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("PerformanceInsightsMetricQuery").add("Metric", metric()).add("GroupBy", groupBy())
                .add("Filter", hasFilter() ? filter() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Metric":
            return Optional.ofNullable(clazz.cast(metric()));
        case "GroupBy":
            return Optional.ofNullable(clazz.cast(groupBy()));
        case "Filter":
            return Optional.ofNullable(clazz.cast(filter()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<PerformanceInsightsMetricQuery, T> g) {
        return obj -> g.apply((PerformanceInsightsMetricQuery) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, PerformanceInsightsMetricQuery> {
        /**
         * <p>
         * The name of the meteric used used when querying an Performance Insights <code>GetResourceMetrics</code> API
         * for anomaly metrics.
         * </p>
         * <p>
         * Valid values for <code>Metric</code> are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>db.load.avg</code> - a scaled representation of the number of active sessions for the database engine.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>db.sampledload.avg</code> - the raw number of active sessions for the database engine.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If the number of active sessions is less than an internal Performance Insights threshold,
         * <code>db.load.avg</code> and <code>db.sampledload.avg</code> are the same value. If the number of active
         * sessions is greater than the internal threshold, Performance Insights samples the active sessions, with
         * <code>db.load.avg</code> showing the scaled values, <code>db.sampledload.avg</code> showing the raw values,
         * and <code>db.sampledload.avg</code> less than <code>db.load.avg</code>. For most use cases, you can query
         * <code>db.load.avg</code> only.
         * </p>
         * 
         * @param metric
         *        The name of the meteric used used when querying an Performance Insights
         *        <code>GetResourceMetrics</code> API for anomaly metrics.</p>
         *        <p>
         *        Valid values for <code>Metric</code> are:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>db.load.avg</code> - a scaled representation of the number of active sessions for the database
         *        engine.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>db.sampledload.avg</code> - the raw number of active sessions for the database engine.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If the number of active sessions is less than an internal Performance Insights threshold,
         *        <code>db.load.avg</code> and <code>db.sampledload.avg</code> are the same value. If the number of
         *        active sessions is greater than the internal threshold, Performance Insights samples the active
         *        sessions, with <code>db.load.avg</code> showing the scaled values, <code>db.sampledload.avg</code>
         *        showing the raw values, and <code>db.sampledload.avg</code> less than <code>db.load.avg</code>. For
         *        most use cases, you can query <code>db.load.avg</code> only.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metric(String metric);

        /**
         * <p>
         * The specification for how to aggregate the data points from a Performance Insights
         * <code>GetResourceMetrics</code> API query. The Performance Insights query returns all of the dimensions
         * within that group, unless you provide the names of specific dimensions within that group. You can also
         * request that Performance Insights return a limited number of values for a dimension.
         * </p>
         * 
         * @param groupBy
         *        The specification for how to aggregate the data points from a Performance Insights
         *        <code>GetResourceMetrics</code> API query. The Performance Insights query returns all of the
         *        dimensions within that group, unless you provide the names of specific dimensions within that group.
         *        You can also request that Performance Insights return a limited number of values for a dimension.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder groupBy(PerformanceInsightsMetricDimensionGroup groupBy);

        /**
         * <p>
         * The specification for how to aggregate the data points from a Performance Insights
         * <code>GetResourceMetrics</code> API query. The Performance Insights query returns all of the dimensions
         * within that group, unless you provide the names of specific dimensions within that group. You can also
         * request that Performance Insights return a limited number of values for a dimension.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link PerformanceInsightsMetricDimensionGroup.Builder} avoiding the need to create one manually via
         * {@link PerformanceInsightsMetricDimensionGroup#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link PerformanceInsightsMetricDimensionGroup.Builder#build()} is
         * called immediately and its result is passed to {@link #groupBy(PerformanceInsightsMetricDimensionGroup)}.
         * 
         * @param groupBy
         *        a consumer that will call methods on {@link PerformanceInsightsMetricDimensionGroup.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #groupBy(PerformanceInsightsMetricDimensionGroup)
         */
        default Builder groupBy(Consumer<PerformanceInsightsMetricDimensionGroup.Builder> groupBy) {
            return groupBy(PerformanceInsightsMetricDimensionGroup.builder().applyMutation(groupBy).build());
        }

        /**
         * <p>
         * One or more filters to apply to a Performance Insights <code>GetResourceMetrics</code> API query.
         * Restrictions:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Any number of filters by the same dimension, as specified in the <code>GroupBy</code> parameter.
         * </p>
         * </li>
         * <li>
         * <p>
         * A single filter for any other dimension in this dimension group.
         * </p>
         * </li>
         * </ul>
         * 
         * @param filter
         *        One or more filters to apply to a Performance Insights <code>GetResourceMetrics</code> API query.
         *        Restrictions:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        Any number of filters by the same dimension, as specified in the <code>GroupBy</code> parameter.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        A single filter for any other dimension in this dimension group.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder filter(Map<String, String> filter);
    }

    static final class BuilderImpl implements Builder {
        private String metric;

        private PerformanceInsightsMetricDimensionGroup groupBy;

        private Map<String, String> filter = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(PerformanceInsightsMetricQuery model) {
            metric(model.metric);
            groupBy(model.groupBy);
            filter(model.filter);
        }

        public final String getMetric() {
            return metric;
        }

        public final void setMetric(String metric) {
            this.metric = metric;
        }

        @Override
        public final Builder metric(String metric) {
            this.metric = metric;
            return this;
        }

        public final PerformanceInsightsMetricDimensionGroup.Builder getGroupBy() {
            return groupBy != null ? groupBy.toBuilder() : null;
        }

        public final void setGroupBy(PerformanceInsightsMetricDimensionGroup.BuilderImpl groupBy) {
            this.groupBy = groupBy != null ? groupBy.build() : null;
        }

        @Override
        public final Builder groupBy(PerformanceInsightsMetricDimensionGroup groupBy) {
            this.groupBy = groupBy;
            return this;
        }

        public final Map<String, String> getFilter() {
            if (filter instanceof SdkAutoConstructMap) {
                return null;
            }
            return filter;
        }

        public final void setFilter(Map<String, String> filter) {
            this.filter = PerformanceInsightsMetricFilterMapCopier.copy(filter);
        }

        @Override
        public final Builder filter(Map<String, String> filter) {
            this.filter = PerformanceInsightsMetricFilterMapCopier.copy(filter);
            return this;
        }

        @Override
        public PerformanceInsightsMetricQuery build() {
            return new PerformanceInsightsMetricQuery(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
