/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.directconnect.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about an interconnect.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Interconnect implements SdkPojo, Serializable, ToCopyableBuilder<Interconnect.Builder, Interconnect> {
    private static final SdkField<String> INTERCONNECT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("interconnectId").getter(getter(Interconnect::interconnectId)).setter(setter(Builder::interconnectId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("interconnectId").build()).build();

    private static final SdkField<String> INTERCONNECT_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("interconnectName").getter(getter(Interconnect::interconnectName))
            .setter(setter(Builder::interconnectName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("interconnectName").build()).build();

    private static final SdkField<String> INTERCONNECT_STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("interconnectState").getter(getter(Interconnect::interconnectStateAsString))
            .setter(setter(Builder::interconnectState))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("interconnectState").build()).build();

    private static final SdkField<String> REGION_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("region")
            .getter(getter(Interconnect::region)).setter(setter(Builder::region))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("region").build()).build();

    private static final SdkField<String> LOCATION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("location").getter(getter(Interconnect::location)).setter(setter(Builder::location))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("location").build()).build();

    private static final SdkField<String> BANDWIDTH_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("bandwidth").getter(getter(Interconnect::bandwidth)).setter(setter(Builder::bandwidth))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("bandwidth").build()).build();

    private static final SdkField<Instant> LOA_ISSUE_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("loaIssueTime").getter(getter(Interconnect::loaIssueTime)).setter(setter(Builder::loaIssueTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("loaIssueTime").build()).build();

    private static final SdkField<String> LAG_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("lagId")
            .getter(getter(Interconnect::lagId)).setter(setter(Builder::lagId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lagId").build()).build();

    private static final SdkField<String> AWS_DEVICE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("awsDevice").getter(getter(Interconnect::awsDevice)).setter(setter(Builder::awsDevice))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("awsDevice").build()).build();

    private static final SdkField<Boolean> JUMBO_FRAME_CAPABLE_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("jumboFrameCapable").getter(getter(Interconnect::jumboFrameCapable))
            .setter(setter(Builder::jumboFrameCapable))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("jumboFrameCapable").build()).build();

    private static final SdkField<String> AWS_DEVICE_V2_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("awsDeviceV2").getter(getter(Interconnect::awsDeviceV2)).setter(setter(Builder::awsDeviceV2))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("awsDeviceV2").build()).build();

    private static final SdkField<String> AWS_LOGICAL_DEVICE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("awsLogicalDeviceId").getter(getter(Interconnect::awsLogicalDeviceId))
            .setter(setter(Builder::awsLogicalDeviceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("awsLogicalDeviceId").build())
            .build();

    private static final SdkField<String> HAS_LOGICAL_REDUNDANCY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("hasLogicalRedundancy").getter(getter(Interconnect::hasLogicalRedundancyAsString))
            .setter(setter(Builder::hasLogicalRedundancy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("hasLogicalRedundancy").build())
            .build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("tags")
            .getter(getter(Interconnect::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> PROVIDER_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("providerName").getter(getter(Interconnect::providerName)).setter(setter(Builder::providerName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("providerName").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(INTERCONNECT_ID_FIELD,
            INTERCONNECT_NAME_FIELD, INTERCONNECT_STATE_FIELD, REGION_FIELD, LOCATION_FIELD, BANDWIDTH_FIELD,
            LOA_ISSUE_TIME_FIELD, LAG_ID_FIELD, AWS_DEVICE_FIELD, JUMBO_FRAME_CAPABLE_FIELD, AWS_DEVICE_V2_FIELD,
            AWS_LOGICAL_DEVICE_ID_FIELD, HAS_LOGICAL_REDUNDANCY_FIELD, TAGS_FIELD, PROVIDER_NAME_FIELD));

    private static final long serialVersionUID = 1L;

    private final String interconnectId;

    private final String interconnectName;

    private final String interconnectState;

    private final String region;

    private final String location;

    private final String bandwidth;

    private final Instant loaIssueTime;

    private final String lagId;

    private final String awsDevice;

    private final Boolean jumboFrameCapable;

    private final String awsDeviceV2;

    private final String awsLogicalDeviceId;

    private final String hasLogicalRedundancy;

    private final List<Tag> tags;

    private final String providerName;

    private Interconnect(BuilderImpl builder) {
        this.interconnectId = builder.interconnectId;
        this.interconnectName = builder.interconnectName;
        this.interconnectState = builder.interconnectState;
        this.region = builder.region;
        this.location = builder.location;
        this.bandwidth = builder.bandwidth;
        this.loaIssueTime = builder.loaIssueTime;
        this.lagId = builder.lagId;
        this.awsDevice = builder.awsDevice;
        this.jumboFrameCapable = builder.jumboFrameCapable;
        this.awsDeviceV2 = builder.awsDeviceV2;
        this.awsLogicalDeviceId = builder.awsLogicalDeviceId;
        this.hasLogicalRedundancy = builder.hasLogicalRedundancy;
        this.tags = builder.tags;
        this.providerName = builder.providerName;
    }

    /**
     * <p>
     * The ID of the interconnect.
     * </p>
     * 
     * @return The ID of the interconnect.
     */
    public final String interconnectId() {
        return interconnectId;
    }

    /**
     * <p>
     * The name of the interconnect.
     * </p>
     * 
     * @return The name of the interconnect.
     */
    public final String interconnectName() {
        return interconnectName;
    }

    /**
     * <p>
     * The state of the interconnect. The following are the possible values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>requested</code>: The initial state of an interconnect. The interconnect stays in the requested state until
     * the Letter of Authorization (LOA) is sent to the customer.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>pending</code>: The interconnect is approved, and is being initialized.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>available</code>: The network link is up, and the interconnect is ready for use.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>down</code>: The network link is down.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>deleting</code>: The interconnect is being deleted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>deleted</code>: The interconnect is deleted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>unknown</code>: The state of the interconnect is not available.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #interconnectState}
     * will return {@link InterconnectState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #interconnectStateAsString}.
     * </p>
     * 
     * @return The state of the interconnect. The following are the possible values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>requested</code>: The initial state of an interconnect. The interconnect stays in the requested
     *         state until the Letter of Authorization (LOA) is sent to the customer.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>pending</code>: The interconnect is approved, and is being initialized.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>available</code>: The network link is up, and the interconnect is ready for use.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>down</code>: The network link is down.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>deleting</code>: The interconnect is being deleted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>deleted</code>: The interconnect is deleted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>unknown</code>: The state of the interconnect is not available.
     *         </p>
     *         </li>
     * @see InterconnectState
     */
    public final InterconnectState interconnectState() {
        return InterconnectState.fromValue(interconnectState);
    }

    /**
     * <p>
     * The state of the interconnect. The following are the possible values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>requested</code>: The initial state of an interconnect. The interconnect stays in the requested state until
     * the Letter of Authorization (LOA) is sent to the customer.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>pending</code>: The interconnect is approved, and is being initialized.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>available</code>: The network link is up, and the interconnect is ready for use.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>down</code>: The network link is down.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>deleting</code>: The interconnect is being deleted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>deleted</code>: The interconnect is deleted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>unknown</code>: The state of the interconnect is not available.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #interconnectState}
     * will return {@link InterconnectState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #interconnectStateAsString}.
     * </p>
     * 
     * @return The state of the interconnect. The following are the possible values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>requested</code>: The initial state of an interconnect. The interconnect stays in the requested
     *         state until the Letter of Authorization (LOA) is sent to the customer.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>pending</code>: The interconnect is approved, and is being initialized.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>available</code>: The network link is up, and the interconnect is ready for use.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>down</code>: The network link is down.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>deleting</code>: The interconnect is being deleted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>deleted</code>: The interconnect is deleted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>unknown</code>: The state of the interconnect is not available.
     *         </p>
     *         </li>
     * @see InterconnectState
     */
    public final String interconnectStateAsString() {
        return interconnectState;
    }

    /**
     * <p>
     * The Amazon Web Services Region where the connection is located.
     * </p>
     * 
     * @return The Amazon Web Services Region where the connection is located.
     */
    public final String region() {
        return region;
    }

    /**
     * <p>
     * The location of the connection.
     * </p>
     * 
     * @return The location of the connection.
     */
    public final String location() {
        return location;
    }

    /**
     * <p>
     * The bandwidth of the connection.
     * </p>
     * 
     * @return The bandwidth of the connection.
     */
    public final String bandwidth() {
        return bandwidth;
    }

    /**
     * <p>
     * The time of the most recent call to <a>DescribeLoa</a> for this connection.
     * </p>
     * 
     * @return The time of the most recent call to <a>DescribeLoa</a> for this connection.
     */
    public final Instant loaIssueTime() {
        return loaIssueTime;
    }

    /**
     * <p>
     * The ID of the LAG.
     * </p>
     * 
     * @return The ID of the LAG.
     */
    public final String lagId() {
        return lagId;
    }

    /**
     * <p>
     * The Direct Connect endpoint on which the physical connection terminates.
     * </p>
     * 
     * @return The Direct Connect endpoint on which the physical connection terminates.
     */
    public final String awsDevice() {
        return awsDevice;
    }

    /**
     * <p>
     * Indicates whether jumbo frames (9001 MTU) are supported.
     * </p>
     * 
     * @return Indicates whether jumbo frames (9001 MTU) are supported.
     */
    public final Boolean jumboFrameCapable() {
        return jumboFrameCapable;
    }

    /**
     * <p>
     * The Direct Connect endpoint that terminates the physical connection.
     * </p>
     * 
     * @return The Direct Connect endpoint that terminates the physical connection.
     */
    public final String awsDeviceV2() {
        return awsDeviceV2;
    }

    /**
     * <p>
     * The Direct Connect endpoint that terminates the logical connection. This device might be different than the
     * device that terminates the physical connection.
     * </p>
     * 
     * @return The Direct Connect endpoint that terminates the logical connection. This device might be different than
     *         the device that terminates the physical connection.
     */
    public final String awsLogicalDeviceId() {
        return awsLogicalDeviceId;
    }

    /**
     * <p>
     * Indicates whether the interconnect supports a secondary BGP in the same address family (IPv4/IPv6).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #hasLogicalRedundancy} will return {@link HasLogicalRedundancy#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #hasLogicalRedundancyAsString}.
     * </p>
     * 
     * @return Indicates whether the interconnect supports a secondary BGP in the same address family (IPv4/IPv6).
     * @see HasLogicalRedundancy
     */
    public final HasLogicalRedundancy hasLogicalRedundancy() {
        return HasLogicalRedundancy.fromValue(hasLogicalRedundancy);
    }

    /**
     * <p>
     * Indicates whether the interconnect supports a secondary BGP in the same address family (IPv4/IPv6).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #hasLogicalRedundancy} will return {@link HasLogicalRedundancy#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #hasLogicalRedundancyAsString}.
     * </p>
     * 
     * @return Indicates whether the interconnect supports a secondary BGP in the same address family (IPv4/IPv6).
     * @see HasLogicalRedundancy
     */
    public final String hasLogicalRedundancyAsString() {
        return hasLogicalRedundancy;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The tags associated with the interconnect.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The tags associated with the interconnect.
     */
    public final List<Tag> tags() {
        return tags;
    }

    /**
     * <p>
     * The name of the service provider associated with the interconnect.
     * </p>
     * 
     * @return The name of the service provider associated with the interconnect.
     */
    public final String providerName() {
        return providerName;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(interconnectId());
        hashCode = 31 * hashCode + Objects.hashCode(interconnectName());
        hashCode = 31 * hashCode + Objects.hashCode(interconnectStateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(region());
        hashCode = 31 * hashCode + Objects.hashCode(location());
        hashCode = 31 * hashCode + Objects.hashCode(bandwidth());
        hashCode = 31 * hashCode + Objects.hashCode(loaIssueTime());
        hashCode = 31 * hashCode + Objects.hashCode(lagId());
        hashCode = 31 * hashCode + Objects.hashCode(awsDevice());
        hashCode = 31 * hashCode + Objects.hashCode(jumboFrameCapable());
        hashCode = 31 * hashCode + Objects.hashCode(awsDeviceV2());
        hashCode = 31 * hashCode + Objects.hashCode(awsLogicalDeviceId());
        hashCode = 31 * hashCode + Objects.hashCode(hasLogicalRedundancyAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(providerName());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Interconnect)) {
            return false;
        }
        Interconnect other = (Interconnect) obj;
        return Objects.equals(interconnectId(), other.interconnectId())
                && Objects.equals(interconnectName(), other.interconnectName())
                && Objects.equals(interconnectStateAsString(), other.interconnectStateAsString())
                && Objects.equals(region(), other.region()) && Objects.equals(location(), other.location())
                && Objects.equals(bandwidth(), other.bandwidth()) && Objects.equals(loaIssueTime(), other.loaIssueTime())
                && Objects.equals(lagId(), other.lagId()) && Objects.equals(awsDevice(), other.awsDevice())
                && Objects.equals(jumboFrameCapable(), other.jumboFrameCapable())
                && Objects.equals(awsDeviceV2(), other.awsDeviceV2())
                && Objects.equals(awsLogicalDeviceId(), other.awsLogicalDeviceId())
                && Objects.equals(hasLogicalRedundancyAsString(), other.hasLogicalRedundancyAsString())
                && hasTags() == other.hasTags() && Objects.equals(tags(), other.tags())
                && Objects.equals(providerName(), other.providerName());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Interconnect").add("InterconnectId", interconnectId())
                .add("InterconnectName", interconnectName()).add("InterconnectState", interconnectStateAsString())
                .add("Region", region()).add("Location", location()).add("Bandwidth", bandwidth())
                .add("LoaIssueTime", loaIssueTime()).add("LagId", lagId()).add("AwsDevice", awsDevice())
                .add("JumboFrameCapable", jumboFrameCapable()).add("AwsDeviceV2", awsDeviceV2())
                .add("AwsLogicalDeviceId", awsLogicalDeviceId()).add("HasLogicalRedundancy", hasLogicalRedundancyAsString())
                .add("Tags", hasTags() ? tags() : null).add("ProviderName", providerName()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "interconnectId":
            return Optional.ofNullable(clazz.cast(interconnectId()));
        case "interconnectName":
            return Optional.ofNullable(clazz.cast(interconnectName()));
        case "interconnectState":
            return Optional.ofNullable(clazz.cast(interconnectStateAsString()));
        case "region":
            return Optional.ofNullable(clazz.cast(region()));
        case "location":
            return Optional.ofNullable(clazz.cast(location()));
        case "bandwidth":
            return Optional.ofNullable(clazz.cast(bandwidth()));
        case "loaIssueTime":
            return Optional.ofNullable(clazz.cast(loaIssueTime()));
        case "lagId":
            return Optional.ofNullable(clazz.cast(lagId()));
        case "awsDevice":
            return Optional.ofNullable(clazz.cast(awsDevice()));
        case "jumboFrameCapable":
            return Optional.ofNullable(clazz.cast(jumboFrameCapable()));
        case "awsDeviceV2":
            return Optional.ofNullable(clazz.cast(awsDeviceV2()));
        case "awsLogicalDeviceId":
            return Optional.ofNullable(clazz.cast(awsLogicalDeviceId()));
        case "hasLogicalRedundancy":
            return Optional.ofNullable(clazz.cast(hasLogicalRedundancyAsString()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "providerName":
            return Optional.ofNullable(clazz.cast(providerName()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Interconnect, T> g) {
        return obj -> g.apply((Interconnect) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Interconnect> {
        /**
         * <p>
         * The ID of the interconnect.
         * </p>
         * 
         * @param interconnectId
         *        The ID of the interconnect.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder interconnectId(String interconnectId);

        /**
         * <p>
         * The name of the interconnect.
         * </p>
         * 
         * @param interconnectName
         *        The name of the interconnect.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder interconnectName(String interconnectName);

        /**
         * <p>
         * The state of the interconnect. The following are the possible values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>requested</code>: The initial state of an interconnect. The interconnect stays in the requested state
         * until the Letter of Authorization (LOA) is sent to the customer.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>pending</code>: The interconnect is approved, and is being initialized.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>available</code>: The network link is up, and the interconnect is ready for use.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>down</code>: The network link is down.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>deleting</code>: The interconnect is being deleted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>deleted</code>: The interconnect is deleted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>unknown</code>: The state of the interconnect is not available.
         * </p>
         * </li>
         * </ul>
         * 
         * @param interconnectState
         *        The state of the interconnect. The following are the possible values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>requested</code>: The initial state of an interconnect. The interconnect stays in the requested
         *        state until the Letter of Authorization (LOA) is sent to the customer.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>pending</code>: The interconnect is approved, and is being initialized.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>available</code>: The network link is up, and the interconnect is ready for use.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>down</code>: The network link is down.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>deleting</code>: The interconnect is being deleted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>deleted</code>: The interconnect is deleted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>unknown</code>: The state of the interconnect is not available.
         *        </p>
         *        </li>
         * @see InterconnectState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InterconnectState
         */
        Builder interconnectState(String interconnectState);

        /**
         * <p>
         * The state of the interconnect. The following are the possible values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>requested</code>: The initial state of an interconnect. The interconnect stays in the requested state
         * until the Letter of Authorization (LOA) is sent to the customer.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>pending</code>: The interconnect is approved, and is being initialized.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>available</code>: The network link is up, and the interconnect is ready for use.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>down</code>: The network link is down.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>deleting</code>: The interconnect is being deleted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>deleted</code>: The interconnect is deleted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>unknown</code>: The state of the interconnect is not available.
         * </p>
         * </li>
         * </ul>
         * 
         * @param interconnectState
         *        The state of the interconnect. The following are the possible values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>requested</code>: The initial state of an interconnect. The interconnect stays in the requested
         *        state until the Letter of Authorization (LOA) is sent to the customer.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>pending</code>: The interconnect is approved, and is being initialized.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>available</code>: The network link is up, and the interconnect is ready for use.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>down</code>: The network link is down.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>deleting</code>: The interconnect is being deleted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>deleted</code>: The interconnect is deleted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>unknown</code>: The state of the interconnect is not available.
         *        </p>
         *        </li>
         * @see InterconnectState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InterconnectState
         */
        Builder interconnectState(InterconnectState interconnectState);

        /**
         * <p>
         * The Amazon Web Services Region where the connection is located.
         * </p>
         * 
         * @param region
         *        The Amazon Web Services Region where the connection is located.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder region(String region);

        /**
         * <p>
         * The location of the connection.
         * </p>
         * 
         * @param location
         *        The location of the connection.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder location(String location);

        /**
         * <p>
         * The bandwidth of the connection.
         * </p>
         * 
         * @param bandwidth
         *        The bandwidth of the connection.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bandwidth(String bandwidth);

        /**
         * <p>
         * The time of the most recent call to <a>DescribeLoa</a> for this connection.
         * </p>
         * 
         * @param loaIssueTime
         *        The time of the most recent call to <a>DescribeLoa</a> for this connection.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder loaIssueTime(Instant loaIssueTime);

        /**
         * <p>
         * The ID of the LAG.
         * </p>
         * 
         * @param lagId
         *        The ID of the LAG.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lagId(String lagId);

        /**
         * <p>
         * The Direct Connect endpoint on which the physical connection terminates.
         * </p>
         * 
         * @param awsDevice
         *        The Direct Connect endpoint on which the physical connection terminates.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder awsDevice(String awsDevice);

        /**
         * <p>
         * Indicates whether jumbo frames (9001 MTU) are supported.
         * </p>
         * 
         * @param jumboFrameCapable
         *        Indicates whether jumbo frames (9001 MTU) are supported.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jumboFrameCapable(Boolean jumboFrameCapable);

        /**
         * <p>
         * The Direct Connect endpoint that terminates the physical connection.
         * </p>
         * 
         * @param awsDeviceV2
         *        The Direct Connect endpoint that terminates the physical connection.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder awsDeviceV2(String awsDeviceV2);

        /**
         * <p>
         * The Direct Connect endpoint that terminates the logical connection. This device might be different than the
         * device that terminates the physical connection.
         * </p>
         * 
         * @param awsLogicalDeviceId
         *        The Direct Connect endpoint that terminates the logical connection. This device might be different
         *        than the device that terminates the physical connection.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder awsLogicalDeviceId(String awsLogicalDeviceId);

        /**
         * <p>
         * Indicates whether the interconnect supports a secondary BGP in the same address family (IPv4/IPv6).
         * </p>
         * 
         * @param hasLogicalRedundancy
         *        Indicates whether the interconnect supports a secondary BGP in the same address family (IPv4/IPv6).
         * @see HasLogicalRedundancy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see HasLogicalRedundancy
         */
        Builder hasLogicalRedundancy(String hasLogicalRedundancy);

        /**
         * <p>
         * Indicates whether the interconnect supports a secondary BGP in the same address family (IPv4/IPv6).
         * </p>
         * 
         * @param hasLogicalRedundancy
         *        Indicates whether the interconnect supports a secondary BGP in the same address family (IPv4/IPv6).
         * @see HasLogicalRedundancy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see HasLogicalRedundancy
         */
        Builder hasLogicalRedundancy(HasLogicalRedundancy hasLogicalRedundancy);

        /**
         * <p>
         * The tags associated with the interconnect.
         * </p>
         * 
         * @param tags
         *        The tags associated with the interconnect.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * The tags associated with the interconnect.
         * </p>
         * 
         * @param tags
         *        The tags associated with the interconnect.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * The tags associated with the interconnect.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.directconnect.model.Tag.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.directconnect.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.directconnect.model.Tag.Builder#build()} is called immediately and its
         * result is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.directconnect.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * The name of the service provider associated with the interconnect.
         * </p>
         * 
         * @param providerName
         *        The name of the service provider associated with the interconnect.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder providerName(String providerName);
    }

    static final class BuilderImpl implements Builder {
        private String interconnectId;

        private String interconnectName;

        private String interconnectState;

        private String region;

        private String location;

        private String bandwidth;

        private Instant loaIssueTime;

        private String lagId;

        private String awsDevice;

        private Boolean jumboFrameCapable;

        private String awsDeviceV2;

        private String awsLogicalDeviceId;

        private String hasLogicalRedundancy;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private String providerName;

        private BuilderImpl() {
        }

        private BuilderImpl(Interconnect model) {
            interconnectId(model.interconnectId);
            interconnectName(model.interconnectName);
            interconnectState(model.interconnectState);
            region(model.region);
            location(model.location);
            bandwidth(model.bandwidth);
            loaIssueTime(model.loaIssueTime);
            lagId(model.lagId);
            awsDevice(model.awsDevice);
            jumboFrameCapable(model.jumboFrameCapable);
            awsDeviceV2(model.awsDeviceV2);
            awsLogicalDeviceId(model.awsLogicalDeviceId);
            hasLogicalRedundancy(model.hasLogicalRedundancy);
            tags(model.tags);
            providerName(model.providerName);
        }

        public final String getInterconnectId() {
            return interconnectId;
        }

        public final void setInterconnectId(String interconnectId) {
            this.interconnectId = interconnectId;
        }

        @Override
        public final Builder interconnectId(String interconnectId) {
            this.interconnectId = interconnectId;
            return this;
        }

        public final String getInterconnectName() {
            return interconnectName;
        }

        public final void setInterconnectName(String interconnectName) {
            this.interconnectName = interconnectName;
        }

        @Override
        public final Builder interconnectName(String interconnectName) {
            this.interconnectName = interconnectName;
            return this;
        }

        public final String getInterconnectState() {
            return interconnectState;
        }

        public final void setInterconnectState(String interconnectState) {
            this.interconnectState = interconnectState;
        }

        @Override
        public final Builder interconnectState(String interconnectState) {
            this.interconnectState = interconnectState;
            return this;
        }

        @Override
        public final Builder interconnectState(InterconnectState interconnectState) {
            this.interconnectState(interconnectState == null ? null : interconnectState.toString());
            return this;
        }

        public final String getRegion() {
            return region;
        }

        public final void setRegion(String region) {
            this.region = region;
        }

        @Override
        public final Builder region(String region) {
            this.region = region;
            return this;
        }

        public final String getLocation() {
            return location;
        }

        public final void setLocation(String location) {
            this.location = location;
        }

        @Override
        public final Builder location(String location) {
            this.location = location;
            return this;
        }

        public final String getBandwidth() {
            return bandwidth;
        }

        public final void setBandwidth(String bandwidth) {
            this.bandwidth = bandwidth;
        }

        @Override
        public final Builder bandwidth(String bandwidth) {
            this.bandwidth = bandwidth;
            return this;
        }

        public final Instant getLoaIssueTime() {
            return loaIssueTime;
        }

        public final void setLoaIssueTime(Instant loaIssueTime) {
            this.loaIssueTime = loaIssueTime;
        }

        @Override
        public final Builder loaIssueTime(Instant loaIssueTime) {
            this.loaIssueTime = loaIssueTime;
            return this;
        }

        public final String getLagId() {
            return lagId;
        }

        public final void setLagId(String lagId) {
            this.lagId = lagId;
        }

        @Override
        public final Builder lagId(String lagId) {
            this.lagId = lagId;
            return this;
        }

        public final String getAwsDevice() {
            return awsDevice;
        }

        public final void setAwsDevice(String awsDevice) {
            this.awsDevice = awsDevice;
        }

        @Override
        public final Builder awsDevice(String awsDevice) {
            this.awsDevice = awsDevice;
            return this;
        }

        public final Boolean getJumboFrameCapable() {
            return jumboFrameCapable;
        }

        public final void setJumboFrameCapable(Boolean jumboFrameCapable) {
            this.jumboFrameCapable = jumboFrameCapable;
        }

        @Override
        public final Builder jumboFrameCapable(Boolean jumboFrameCapable) {
            this.jumboFrameCapable = jumboFrameCapable;
            return this;
        }

        public final String getAwsDeviceV2() {
            return awsDeviceV2;
        }

        public final void setAwsDeviceV2(String awsDeviceV2) {
            this.awsDeviceV2 = awsDeviceV2;
        }

        @Override
        public final Builder awsDeviceV2(String awsDeviceV2) {
            this.awsDeviceV2 = awsDeviceV2;
            return this;
        }

        public final String getAwsLogicalDeviceId() {
            return awsLogicalDeviceId;
        }

        public final void setAwsLogicalDeviceId(String awsLogicalDeviceId) {
            this.awsLogicalDeviceId = awsLogicalDeviceId;
        }

        @Override
        public final Builder awsLogicalDeviceId(String awsLogicalDeviceId) {
            this.awsLogicalDeviceId = awsLogicalDeviceId;
            return this;
        }

        public final String getHasLogicalRedundancy() {
            return hasLogicalRedundancy;
        }

        public final void setHasLogicalRedundancy(String hasLogicalRedundancy) {
            this.hasLogicalRedundancy = hasLogicalRedundancy;
        }

        @Override
        public final Builder hasLogicalRedundancy(String hasLogicalRedundancy) {
            this.hasLogicalRedundancy = hasLogicalRedundancy;
            return this;
        }

        @Override
        public final Builder hasLogicalRedundancy(HasLogicalRedundancy hasLogicalRedundancy) {
            this.hasLogicalRedundancy(hasLogicalRedundancy == null ? null : hasLogicalRedundancy.toString());
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getProviderName() {
            return providerName;
        }

        public final void setProviderName(String providerName) {
            this.providerName = providerName;
        }

        @Override
        public final Builder providerName(String providerName) {
            this.providerName = providerName;
            return this;
        }

        @Override
        public Interconnect build() {
            return new Interconnect(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
