/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.dlm.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * <b>[Snapshot policies only]</b> Describes the retention rule for archived snapshots. Once the archive retention
 * threshold is met, the snapshots are permanently deleted from the archive tier.
 * </p>
 * <note>
 * <p>
 * The archive retention rule must retain snapshots in the archive tier for a minimum of 90 days.
 * </p>
 * </note>
 * <p>
 * For <b>count-based schedules</b>, you must specify <b>Count</b>. For <b>age-based schedules</b>, you must specify
 * <b>Interval</b> and <b> IntervalUnit</b>.
 * </p>
 * <p>
 * For more information about using snapshot archiving, see <a
 * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/snapshot-ami-policy.html#dlm-archive">Considerations for
 * snapshot lifecycle policies</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class RetentionArchiveTier implements SdkPojo, Serializable,
        ToCopyableBuilder<RetentionArchiveTier.Builder, RetentionArchiveTier> {
    private static final SdkField<Integer> COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER).memberName("Count")
            .getter(getter(RetentionArchiveTier::count)).setter(setter(Builder::count))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Count").build()).build();

    private static final SdkField<Integer> INTERVAL_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("Interval").getter(getter(RetentionArchiveTier::interval)).setter(setter(Builder::interval))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Interval").build()).build();

    private static final SdkField<String> INTERVAL_UNIT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("IntervalUnit").getter(getter(RetentionArchiveTier::intervalUnitAsString))
            .setter(setter(Builder::intervalUnit))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IntervalUnit").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(COUNT_FIELD, INTERVAL_FIELD,
            INTERVAL_UNIT_FIELD));

    private static final long serialVersionUID = 1L;

    private final Integer count;

    private final Integer interval;

    private final String intervalUnit;

    private RetentionArchiveTier(BuilderImpl builder) {
        this.count = builder.count;
        this.interval = builder.interval;
        this.intervalUnit = builder.intervalUnit;
    }

    /**
     * <p>
     * The maximum number of snapshots to retain in the archive storage tier for each volume. The count must ensure that
     * each snapshot remains in the archive tier for at least 90 days. For example, if the schedule creates snapshots
     * every 30 days, you must specify a count of 3 or more to ensure that each snapshot is archived for at least 90
     * days.
     * </p>
     * 
     * @return The maximum number of snapshots to retain in the archive storage tier for each volume. The count must
     *         ensure that each snapshot remains in the archive tier for at least 90 days. For example, if the schedule
     *         creates snapshots every 30 days, you must specify a count of 3 or more to ensure that each snapshot is
     *         archived for at least 90 days.
     */
    public final Integer count() {
        return count;
    }

    /**
     * <p>
     * Specifies the period of time to retain snapshots in the archive tier. After this period expires, the snapshot is
     * permanently deleted.
     * </p>
     * 
     * @return Specifies the period of time to retain snapshots in the archive tier. After this period expires, the
     *         snapshot is permanently deleted.
     */
    public final Integer interval() {
        return interval;
    }

    /**
     * <p>
     * The unit of time in which to measure the <b>Interval</b>. For example, to retain a snapshots in the archive tier
     * for 6 months, specify <code>Interval=6</code> and <code>IntervalUnit=MONTHS</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #intervalUnit} will
     * return {@link RetentionIntervalUnitValues#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #intervalUnitAsString}.
     * </p>
     * 
     * @return The unit of time in which to measure the <b>Interval</b>. For example, to retain a snapshots in the
     *         archive tier for 6 months, specify <code>Interval=6</code> and <code>IntervalUnit=MONTHS</code>.
     * @see RetentionIntervalUnitValues
     */
    public final RetentionIntervalUnitValues intervalUnit() {
        return RetentionIntervalUnitValues.fromValue(intervalUnit);
    }

    /**
     * <p>
     * The unit of time in which to measure the <b>Interval</b>. For example, to retain a snapshots in the archive tier
     * for 6 months, specify <code>Interval=6</code> and <code>IntervalUnit=MONTHS</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #intervalUnit} will
     * return {@link RetentionIntervalUnitValues#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #intervalUnitAsString}.
     * </p>
     * 
     * @return The unit of time in which to measure the <b>Interval</b>. For example, to retain a snapshots in the
     *         archive tier for 6 months, specify <code>Interval=6</code> and <code>IntervalUnit=MONTHS</code>.
     * @see RetentionIntervalUnitValues
     */
    public final String intervalUnitAsString() {
        return intervalUnit;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(count());
        hashCode = 31 * hashCode + Objects.hashCode(interval());
        hashCode = 31 * hashCode + Objects.hashCode(intervalUnitAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RetentionArchiveTier)) {
            return false;
        }
        RetentionArchiveTier other = (RetentionArchiveTier) obj;
        return Objects.equals(count(), other.count()) && Objects.equals(interval(), other.interval())
                && Objects.equals(intervalUnitAsString(), other.intervalUnitAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("RetentionArchiveTier").add("Count", count()).add("Interval", interval())
                .add("IntervalUnit", intervalUnitAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Count":
            return Optional.ofNullable(clazz.cast(count()));
        case "Interval":
            return Optional.ofNullable(clazz.cast(interval()));
        case "IntervalUnit":
            return Optional.ofNullable(clazz.cast(intervalUnitAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<RetentionArchiveTier, T> g) {
        return obj -> g.apply((RetentionArchiveTier) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, RetentionArchiveTier> {
        /**
         * <p>
         * The maximum number of snapshots to retain in the archive storage tier for each volume. The count must ensure
         * that each snapshot remains in the archive tier for at least 90 days. For example, if the schedule creates
         * snapshots every 30 days, you must specify a count of 3 or more to ensure that each snapshot is archived for
         * at least 90 days.
         * </p>
         * 
         * @param count
         *        The maximum number of snapshots to retain in the archive storage tier for each volume. The count must
         *        ensure that each snapshot remains in the archive tier for at least 90 days. For example, if the
         *        schedule creates snapshots every 30 days, you must specify a count of 3 or more to ensure that each
         *        snapshot is archived for at least 90 days.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder count(Integer count);

        /**
         * <p>
         * Specifies the period of time to retain snapshots in the archive tier. After this period expires, the snapshot
         * is permanently deleted.
         * </p>
         * 
         * @param interval
         *        Specifies the period of time to retain snapshots in the archive tier. After this period expires, the
         *        snapshot is permanently deleted.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder interval(Integer interval);

        /**
         * <p>
         * The unit of time in which to measure the <b>Interval</b>. For example, to retain a snapshots in the archive
         * tier for 6 months, specify <code>Interval=6</code> and <code>IntervalUnit=MONTHS</code>.
         * </p>
         * 
         * @param intervalUnit
         *        The unit of time in which to measure the <b>Interval</b>. For example, to retain a snapshots in the
         *        archive tier for 6 months, specify <code>Interval=6</code> and <code>IntervalUnit=MONTHS</code>.
         * @see RetentionIntervalUnitValues
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RetentionIntervalUnitValues
         */
        Builder intervalUnit(String intervalUnit);

        /**
         * <p>
         * The unit of time in which to measure the <b>Interval</b>. For example, to retain a snapshots in the archive
         * tier for 6 months, specify <code>Interval=6</code> and <code>IntervalUnit=MONTHS</code>.
         * </p>
         * 
         * @param intervalUnit
         *        The unit of time in which to measure the <b>Interval</b>. For example, to retain a snapshots in the
         *        archive tier for 6 months, specify <code>Interval=6</code> and <code>IntervalUnit=MONTHS</code>.
         * @see RetentionIntervalUnitValues
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RetentionIntervalUnitValues
         */
        Builder intervalUnit(RetentionIntervalUnitValues intervalUnit);
    }

    static final class BuilderImpl implements Builder {
        private Integer count;

        private Integer interval;

        private String intervalUnit;

        private BuilderImpl() {
        }

        private BuilderImpl(RetentionArchiveTier model) {
            count(model.count);
            interval(model.interval);
            intervalUnit(model.intervalUnit);
        }

        public final Integer getCount() {
            return count;
        }

        public final void setCount(Integer count) {
            this.count = count;
        }

        @Override
        public final Builder count(Integer count) {
            this.count = count;
            return this;
        }

        public final Integer getInterval() {
            return interval;
        }

        public final void setInterval(Integer interval) {
            this.interval = interval;
        }

        @Override
        public final Builder interval(Integer interval) {
            this.interval = interval;
            return this;
        }

        public final String getIntervalUnit() {
            return intervalUnit;
        }

        public final void setIntervalUnit(String intervalUnit) {
            this.intervalUnit = intervalUnit;
        }

        @Override
        public final Builder intervalUnit(String intervalUnit) {
            this.intervalUnit = intervalUnit;
            return this;
        }

        @Override
        public final Builder intervalUnit(RetentionIntervalUnitValues intervalUnit) {
            this.intervalUnit(intervalUnit == null ? null : intervalUnit.toString());
            return this;
        }

        @Override
        public RetentionArchiveTier build() {
            return new RetentionArchiveTier(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
