/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ebs;

import static software.amazon.awssdk.utils.FunctionalUtils.runAndLogError;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.auth.signer.Aws4UnsignedPayloadSigner;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.async.AsyncRequestBody;
import software.amazon.awssdk.core.async.AsyncResponseTransformer;
import software.amazon.awssdk.core.async.AsyncResponseTransformerUtils;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.runtime.transform.AsyncStreamingRequestMarshaller;
import software.amazon.awssdk.core.signer.Signer;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.ebs.internal.EbsServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.ebs.model.AccessDeniedException;
import software.amazon.awssdk.services.ebs.model.CompleteSnapshotRequest;
import software.amazon.awssdk.services.ebs.model.CompleteSnapshotResponse;
import software.amazon.awssdk.services.ebs.model.ConcurrentLimitExceededException;
import software.amazon.awssdk.services.ebs.model.ConflictException;
import software.amazon.awssdk.services.ebs.model.EbsException;
import software.amazon.awssdk.services.ebs.model.EbsRequest;
import software.amazon.awssdk.services.ebs.model.GetSnapshotBlockRequest;
import software.amazon.awssdk.services.ebs.model.GetSnapshotBlockResponse;
import software.amazon.awssdk.services.ebs.model.InternalServerException;
import software.amazon.awssdk.services.ebs.model.ListChangedBlocksRequest;
import software.amazon.awssdk.services.ebs.model.ListChangedBlocksResponse;
import software.amazon.awssdk.services.ebs.model.ListSnapshotBlocksRequest;
import software.amazon.awssdk.services.ebs.model.ListSnapshotBlocksResponse;
import software.amazon.awssdk.services.ebs.model.PutSnapshotBlockRequest;
import software.amazon.awssdk.services.ebs.model.PutSnapshotBlockResponse;
import software.amazon.awssdk.services.ebs.model.RequestThrottledException;
import software.amazon.awssdk.services.ebs.model.ResourceNotFoundException;
import software.amazon.awssdk.services.ebs.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.ebs.model.StartSnapshotRequest;
import software.amazon.awssdk.services.ebs.model.StartSnapshotResponse;
import software.amazon.awssdk.services.ebs.model.ValidationException;
import software.amazon.awssdk.services.ebs.transform.CompleteSnapshotRequestMarshaller;
import software.amazon.awssdk.services.ebs.transform.GetSnapshotBlockRequestMarshaller;
import software.amazon.awssdk.services.ebs.transform.ListChangedBlocksRequestMarshaller;
import software.amazon.awssdk.services.ebs.transform.ListSnapshotBlocksRequestMarshaller;
import software.amazon.awssdk.services.ebs.transform.PutSnapshotBlockRequestMarshaller;
import software.amazon.awssdk.services.ebs.transform.StartSnapshotRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;
import software.amazon.awssdk.utils.Pair;

/**
 * Internal implementation of {@link EbsAsyncClient}.
 *
 * @see EbsAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultEbsAsyncClient implements EbsAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultEbsAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultEbsAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Seals and completes the snapshot after all of the required blocks of data have been written to it. Completing the
     * snapshot changes the status to <code>completed</code>. You cannot write new blocks to a snapshot after it has
     * been completed.
     * </p>
     * <note>
     * <p>
     * You should always retry requests that receive server (<code>5xx</code>) error responses, and
     * <code>ThrottlingException</code> and <code>RequestThrottledException</code> client error responses. For more
     * information see <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error
     * retries</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
     * </p>
     * </note>
     *
     * @param completeSnapshotRequest
     * @return A Java Future containing the result of the CompleteSnapshot operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints of the EBS direct APIs.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>RequestThrottledException The number of API requests has exceeded the maximum allowed API request
     *         throttling limit for the snapshot. For more information see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error retries</a>.</li>
     *         <li>ServiceQuotaExceededException Your current service quotas do not allow you to perform this action.</li>
     *         <li>InternalServerException An internal error has occurred. For more information see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error retries</a>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EbsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EbsAsyncClient.CompleteSnapshot
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ebs-2019-11-02/CompleteSnapshot" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CompleteSnapshotResponse> completeSnapshot(CompleteSnapshotRequest completeSnapshotRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(completeSnapshotRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, completeSnapshotRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EBS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CompleteSnapshot");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CompleteSnapshotResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CompleteSnapshotResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CompleteSnapshotResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CompleteSnapshotRequest, CompleteSnapshotResponse>()
                            .withOperationName("CompleteSnapshot").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CompleteSnapshotRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(completeSnapshotRequest));
            CompletableFuture<CompleteSnapshotResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the data in a block in an Amazon Elastic Block Store snapshot.
     * </p>
     * <note>
     * <p>
     * You should always retry requests that receive server (<code>5xx</code>) error responses, and
     * <code>ThrottlingException</code> and <code>RequestThrottledException</code> client error responses. For more
     * information see <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error
     * retries</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
     * </p>
     * </note>
     *
     * @param getSnapshotBlockRequest
     * @param asyncResponseTransformer
     *        The response transformer for processing the streaming response in a non-blocking manner. See
     *        {@link AsyncResponseTransformer} for details on how this callback should be implemented and for links to
     *        precanned implementations for common scenarios like downloading to a file. The service documentation for
     *        the response content is as follows '
     *        <p>
     *        The data content of the block.
     *        </p>
     *        '.
     * @return A future to the transformed result of the AsyncResponseTransformer.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints of the EBS direct APIs.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>RequestThrottledException The number of API requests has exceeded the maximum allowed API request
     *         throttling limit for the snapshot. For more information see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error retries</a>.</li>
     *         <li>ServiceQuotaExceededException Your current service quotas do not allow you to perform this action.</li>
     *         <li>InternalServerException An internal error has occurred. For more information see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error retries</a>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EbsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EbsAsyncClient.GetSnapshotBlock
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ebs-2019-11-02/GetSnapshotBlock" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public <ReturnT> CompletableFuture<ReturnT> getSnapshotBlock(GetSnapshotBlockRequest getSnapshotBlockRequest,
            AsyncResponseTransformer<GetSnapshotBlockResponse, ReturnT> asyncResponseTransformer) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getSnapshotBlockRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSnapshotBlockRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EBS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSnapshotBlock");
            Pair<AsyncResponseTransformer<GetSnapshotBlockResponse, ReturnT>, CompletableFuture<Void>> pair = AsyncResponseTransformerUtils
                    .wrapWithEndOfStreamFuture(asyncResponseTransformer);
            asyncResponseTransformer = pair.left();
            CompletableFuture<Void> endOfStreamFuture = pair.right();
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(true)
                    .isPayloadJson(false).build();

            HttpResponseHandler<GetSnapshotBlockResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetSnapshotBlockResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ReturnT> executeFuture = clientHandler.execute(
                    new ClientExecutionParams<GetSnapshotBlockRequest, GetSnapshotBlockResponse>()
                            .withOperationName("GetSnapshotBlock").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetSnapshotBlockRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getSnapshotBlockRequest), asyncResponseTransformer);
            AsyncResponseTransformer<GetSnapshotBlockResponse, ReturnT> finalAsyncResponseTransformer = asyncResponseTransformer;
            CompletableFuture<ReturnT> whenCompleted = executeFuture.whenComplete((r, e) -> {
                if (e != null) {
                    runAndLogError(log, "Exception thrown in exceptionOccurred callback, ignoring",
                            () -> finalAsyncResponseTransformer.exceptionOccurred(e));
                }
                endOfStreamFuture.whenComplete((r2, e2) -> {
                    metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                });
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            AsyncResponseTransformer<GetSnapshotBlockResponse, ReturnT> finalAsyncResponseTransformer = asyncResponseTransformer;
            runAndLogError(log, "Exception thrown in exceptionOccurred callback, ignoring",
                    () -> finalAsyncResponseTransformer.exceptionOccurred(t));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about the blocks that are different between two Amazon Elastic Block Store snapshots of the
     * same volume/snapshot lineage.
     * </p>
     * <note>
     * <p>
     * You should always retry requests that receive server (<code>5xx</code>) error responses, and
     * <code>ThrottlingException</code> and <code>RequestThrottledException</code> client error responses. For more
     * information see <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error
     * retries</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
     * </p>
     * </note>
     *
     * @param listChangedBlocksRequest
     * @return A Java Future containing the result of the ListChangedBlocks operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints of the EBS direct APIs.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>RequestThrottledException The number of API requests has exceeded the maximum allowed API request
     *         throttling limit for the snapshot. For more information see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error retries</a>.</li>
     *         <li>ServiceQuotaExceededException Your current service quotas do not allow you to perform this action.</li>
     *         <li>InternalServerException An internal error has occurred. For more information see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error retries</a>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EbsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EbsAsyncClient.ListChangedBlocks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ebs-2019-11-02/ListChangedBlocks" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListChangedBlocksResponse> listChangedBlocks(ListChangedBlocksRequest listChangedBlocksRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listChangedBlocksRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listChangedBlocksRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EBS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListChangedBlocks");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListChangedBlocksResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListChangedBlocksResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListChangedBlocksResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListChangedBlocksRequest, ListChangedBlocksResponse>()
                            .withOperationName("ListChangedBlocks").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListChangedBlocksRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listChangedBlocksRequest));
            CompletableFuture<ListChangedBlocksResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about the blocks in an Amazon Elastic Block Store snapshot.
     * </p>
     * <note>
     * <p>
     * You should always retry requests that receive server (<code>5xx</code>) error responses, and
     * <code>ThrottlingException</code> and <code>RequestThrottledException</code> client error responses. For more
     * information see <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error
     * retries</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
     * </p>
     * </note>
     *
     * @param listSnapshotBlocksRequest
     * @return A Java Future containing the result of the ListSnapshotBlocks operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints of the EBS direct APIs.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>RequestThrottledException The number of API requests has exceeded the maximum allowed API request
     *         throttling limit for the snapshot. For more information see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error retries</a>.</li>
     *         <li>ServiceQuotaExceededException Your current service quotas do not allow you to perform this action.</li>
     *         <li>InternalServerException An internal error has occurred. For more information see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error retries</a>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EbsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EbsAsyncClient.ListSnapshotBlocks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ebs-2019-11-02/ListSnapshotBlocks" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListSnapshotBlocksResponse> listSnapshotBlocks(ListSnapshotBlocksRequest listSnapshotBlocksRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listSnapshotBlocksRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSnapshotBlocksRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EBS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSnapshotBlocks");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListSnapshotBlocksResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListSnapshotBlocksResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListSnapshotBlocksResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListSnapshotBlocksRequest, ListSnapshotBlocksResponse>()
                            .withOperationName("ListSnapshotBlocks").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListSnapshotBlocksRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listSnapshotBlocksRequest));
            CompletableFuture<ListSnapshotBlocksResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Writes a block of data to a snapshot. If the specified block contains data, the existing data is overwritten. The
     * target snapshot must be in the <code>pending</code> state.
     * </p>
     * <p>
     * Data written to a snapshot must be aligned with 512-KiB sectors.
     * </p>
     * <note>
     * <p>
     * You should always retry requests that receive server (<code>5xx</code>) error responses, and
     * <code>ThrottlingException</code> and <code>RequestThrottledException</code> client error responses. For more
     * information see <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error
     * retries</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
     * </p>
     * </note>
     *
     * @param putSnapshotBlockRequest
     * @param requestBody
     *        Functional interface that can be implemented to produce the request content in a non-blocking manner. The
     *        size of the content is expected to be known up front. See {@link AsyncRequestBody} for specific details on
     *        implementing this interface as well as links to precanned implementations for common scenarios like
     *        uploading from a file. The service documentation for the request content is as follows '
     *        <p>
     *        The data to write to the block.
     *        </p>
     *        <p>
     *        The block data is not signed as part of the Signature Version 4 signing process. As a result, you must
     *        generate and provide a Base64-encoded SHA256 checksum for the block data using the <b>x-amz-Checksum</b>
     *        header. Also, you must specify the checksum algorithm using the <b>x-amz-Checksum-Algorithm</b> header.
     *        The checksum that you provide is part of the Signature Version 4 signing process. It is validated against
     *        a checksum generated by Amazon EBS to ensure the validity and authenticity of the data. If the checksums
     *        do not correspond, the request fails. For more information, see <a href=
     *        "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ebs-accessing-snapshot.html#ebsapis-using-checksums">
     *        Using checksums with the EBS direct APIs</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
     *        </p>
     *        '
     * @return A Java Future containing the result of the PutSnapshotBlock operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints of the EBS direct APIs.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>RequestThrottledException The number of API requests has exceeded the maximum allowed API request
     *         throttling limit for the snapshot. For more information see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error retries</a>.</li>
     *         <li>ServiceQuotaExceededException Your current service quotas do not allow you to perform this action.</li>
     *         <li>InternalServerException An internal error has occurred. For more information see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error retries</a>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EbsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EbsAsyncClient.PutSnapshotBlock
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ebs-2019-11-02/PutSnapshotBlock" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PutSnapshotBlockResponse> putSnapshotBlock(PutSnapshotBlockRequest putSnapshotBlockRequest,
            AsyncRequestBody requestBody) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putSnapshotBlockRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putSnapshotBlockRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EBS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutSnapshotBlock");
            putSnapshotBlockRequest = applySignerOverride(putSnapshotBlockRequest, Aws4UnsignedPayloadSigner.create());
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutSnapshotBlockResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutSnapshotBlockResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutSnapshotBlockResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutSnapshotBlockRequest, PutSnapshotBlockResponse>()
                            .withOperationName("PutSnapshotBlock")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(
                                    AsyncStreamingRequestMarshaller.builder()
                                            .delegateMarshaller(new PutSnapshotBlockRequestMarshaller(protocolFactory))
                                            .asyncRequestBody(requestBody).transferEncoding(true).build())
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withAsyncRequestBody(requestBody).withInput(putSnapshotBlockRequest));
            CompletableFuture<PutSnapshotBlockResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new Amazon EBS snapshot. The new snapshot enters the <code>pending</code> state after the request
     * completes.
     * </p>
     * <p>
     * After creating the snapshot, use <a
     * href="https://docs.aws.amazon.com/ebs/latest/APIReference/API_PutSnapshotBlock.html"> PutSnapshotBlock</a> to
     * write blocks of data to the snapshot.
     * </p>
     * <note>
     * <p>
     * You should always retry requests that receive server (<code>5xx</code>) error responses, and
     * <code>ThrottlingException</code> and <code>RequestThrottledException</code> client error responses. For more
     * information see <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error
     * retries</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
     * </p>
     * </note>
     *
     * @param startSnapshotRequest
     * @return A Java Future containing the result of the StartSnapshot operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints of the EBS direct APIs.</li>
     *         <li>RequestThrottledException The number of API requests has exceeded the maximum allowed API request
     *         throttling limit for the snapshot. For more information see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error retries</a>.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>ServiceQuotaExceededException Your current service quotas do not allow you to perform this action.</li>
     *         <li>InternalServerException An internal error has occurred. For more information see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/error-retries.html">Error retries</a>.</li>
     *         <li>ConcurrentLimitExceededException You have reached the limit for concurrent API requests. For more
     *         information, see <a href=
     *         "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ebs-accessing-snapshot.html#ebsapi-performance"
     *         >Optimizing performance of the EBS direct APIs</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
     *         </li>
     *         <li>ConflictException The request uses the same client token as a previous, but non-identical request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EbsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EbsAsyncClient.StartSnapshot
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ebs-2019-11-02/StartSnapshot" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StartSnapshotResponse> startSnapshot(StartSnapshotRequest startSnapshotRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startSnapshotRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startSnapshotRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EBS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartSnapshot");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartSnapshotResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    StartSnapshotResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartSnapshotResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartSnapshotRequest, StartSnapshotResponse>()
                            .withOperationName("StartSnapshot").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StartSnapshotRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(startSnapshotRequest));
            CompletableFuture<StartSnapshotResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final EbsServiceClientConfiguration serviceClientConfiguration() {
        return new EbsServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(EbsException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("RequestThrottledException")
                                .exceptionBuilderSupplier(RequestThrottledException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConcurrentLimitExceededException")
                                .exceptionBuilderSupplier(ConcurrentLimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends EbsRequest> T applySignerOverride(T request, Signer signer) {
        if (request.overrideConfiguration().flatMap(c -> c.signer()).isPresent()) {
            return request;
        }
        Consumer<AwsRequestOverrideConfiguration.Builder> signerOverride = b -> b.signer(signer).build();
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(signerOverride).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(signerOverride).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private static boolean isSignerOverridden(SdkClientConfiguration clientConfiguration) {
        return Boolean.TRUE.equals(clientConfiguration.option(SdkClientOption.SIGNER_OVERRIDDEN));
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        EbsServiceClientConfigurationBuilder serviceConfigBuilder = new EbsServiceClientConfigurationBuilder(configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
