/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a Spot Instance request.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SpotInstanceRequest implements SdkPojo, Serializable,
        ToCopyableBuilder<SpotInstanceRequest.Builder, SpotInstanceRequest> {
    private static final SdkField<String> ACTUAL_BLOCK_HOURLY_PRICE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ActualBlockHourlyPrice")
            .getter(getter(SpotInstanceRequest::actualBlockHourlyPrice))
            .setter(setter(Builder::actualBlockHourlyPrice))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ActualBlockHourlyPrice")
                    .unmarshallLocationName("actualBlockHourlyPrice").build()).build();

    private static final SdkField<String> AVAILABILITY_ZONE_GROUP_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("AvailabilityZoneGroup")
            .getter(getter(SpotInstanceRequest::availabilityZoneGroup))
            .setter(setter(Builder::availabilityZoneGroup))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AvailabilityZoneGroup")
                    .unmarshallLocationName("availabilityZoneGroup").build()).build();

    private static final SdkField<Integer> BLOCK_DURATION_MINUTES_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("BlockDurationMinutes")
            .getter(getter(SpotInstanceRequest::blockDurationMinutes))
            .setter(setter(Builder::blockDurationMinutes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BlockDurationMinutes")
                    .unmarshallLocationName("blockDurationMinutes").build()).build();

    private static final SdkField<Instant> CREATE_TIME_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreateTime")
            .getter(getter(SpotInstanceRequest::createTime))
            .setter(setter(Builder::createTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreateTime")
                    .unmarshallLocationName("createTime").build()).build();

    private static final SdkField<SpotInstanceStateFault> FAULT_FIELD = SdkField
            .<SpotInstanceStateFault> builder(MarshallingType.SDK_POJO)
            .memberName("Fault")
            .getter(getter(SpotInstanceRequest::fault))
            .setter(setter(Builder::fault))
            .constructor(SpotInstanceStateFault::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Fault")
                    .unmarshallLocationName("fault").build()).build();

    private static final SdkField<String> INSTANCE_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("InstanceId")
            .getter(getter(SpotInstanceRequest::instanceId))
            .setter(setter(Builder::instanceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceId")
                    .unmarshallLocationName("instanceId").build()).build();

    private static final SdkField<String> LAUNCH_GROUP_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("LaunchGroup")
            .getter(getter(SpotInstanceRequest::launchGroup))
            .setter(setter(Builder::launchGroup))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LaunchGroup")
                    .unmarshallLocationName("launchGroup").build()).build();

    private static final SdkField<LaunchSpecification> LAUNCH_SPECIFICATION_FIELD = SdkField
            .<LaunchSpecification> builder(MarshallingType.SDK_POJO)
            .memberName("LaunchSpecification")
            .getter(getter(SpotInstanceRequest::launchSpecification))
            .setter(setter(Builder::launchSpecification))
            .constructor(LaunchSpecification::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LaunchSpecification")
                    .unmarshallLocationName("launchSpecification").build()).build();

    private static final SdkField<String> LAUNCHED_AVAILABILITY_ZONE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("LaunchedAvailabilityZone")
            .getter(getter(SpotInstanceRequest::launchedAvailabilityZone))
            .setter(setter(Builder::launchedAvailabilityZone))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LaunchedAvailabilityZone")
                    .unmarshallLocationName("launchedAvailabilityZone").build()).build();

    private static final SdkField<String> PRODUCT_DESCRIPTION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ProductDescription")
            .getter(getter(SpotInstanceRequest::productDescriptionAsString))
            .setter(setter(Builder::productDescription))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ProductDescription")
                    .unmarshallLocationName("productDescription").build()).build();

    private static final SdkField<String> SPOT_INSTANCE_REQUEST_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("SpotInstanceRequestId")
            .getter(getter(SpotInstanceRequest::spotInstanceRequestId))
            .setter(setter(Builder::spotInstanceRequestId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SpotInstanceRequestId")
                    .unmarshallLocationName("spotInstanceRequestId").build()).build();

    private static final SdkField<String> SPOT_PRICE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("SpotPrice")
            .getter(getter(SpotInstanceRequest::spotPrice))
            .setter(setter(Builder::spotPrice))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SpotPrice")
                    .unmarshallLocationName("spotPrice").build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("State")
            .getter(getter(SpotInstanceRequest::stateAsString))
            .setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("State")
                    .unmarshallLocationName("state").build()).build();

    private static final SdkField<SpotInstanceStatus> STATUS_FIELD = SdkField
            .<SpotInstanceStatus> builder(MarshallingType.SDK_POJO)
            .memberName("Status")
            .getter(getter(SpotInstanceRequest::status))
            .setter(setter(Builder::status))
            .constructor(SpotInstanceStatus::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status")
                    .unmarshallLocationName("status").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(SpotInstanceRequest::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TagSet")
                    .unmarshallLocationName("tagSet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<String> TYPE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Type")
            .getter(getter(SpotInstanceRequest::typeAsString))
            .setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Type")
                    .unmarshallLocationName("type").build()).build();

    private static final SdkField<Instant> VALID_FROM_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("ValidFrom")
            .getter(getter(SpotInstanceRequest::validFrom))
            .setter(setter(Builder::validFrom))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ValidFrom")
                    .unmarshallLocationName("validFrom").build()).build();

    private static final SdkField<Instant> VALID_UNTIL_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("ValidUntil")
            .getter(getter(SpotInstanceRequest::validUntil))
            .setter(setter(Builder::validUntil))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ValidUntil")
                    .unmarshallLocationName("validUntil").build()).build();

    private static final SdkField<String> INSTANCE_INTERRUPTION_BEHAVIOR_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("InstanceInterruptionBehavior")
            .getter(getter(SpotInstanceRequest::instanceInterruptionBehaviorAsString))
            .setter(setter(Builder::instanceInterruptionBehavior))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceInterruptionBehavior")
                    .unmarshallLocationName("instanceInterruptionBehavior").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            ACTUAL_BLOCK_HOURLY_PRICE_FIELD, AVAILABILITY_ZONE_GROUP_FIELD, BLOCK_DURATION_MINUTES_FIELD, CREATE_TIME_FIELD,
            FAULT_FIELD, INSTANCE_ID_FIELD, LAUNCH_GROUP_FIELD, LAUNCH_SPECIFICATION_FIELD, LAUNCHED_AVAILABILITY_ZONE_FIELD,
            PRODUCT_DESCRIPTION_FIELD, SPOT_INSTANCE_REQUEST_ID_FIELD, SPOT_PRICE_FIELD, STATE_FIELD, STATUS_FIELD, TAGS_FIELD,
            TYPE_FIELD, VALID_FROM_FIELD, VALID_UNTIL_FIELD, INSTANCE_INTERRUPTION_BEHAVIOR_FIELD));

    private static final long serialVersionUID = 1L;

    private final String actualBlockHourlyPrice;

    private final String availabilityZoneGroup;

    private final Integer blockDurationMinutes;

    private final Instant createTime;

    private final SpotInstanceStateFault fault;

    private final String instanceId;

    private final String launchGroup;

    private final LaunchSpecification launchSpecification;

    private final String launchedAvailabilityZone;

    private final String productDescription;

    private final String spotInstanceRequestId;

    private final String spotPrice;

    private final String state;

    private final SpotInstanceStatus status;

    private final List<Tag> tags;

    private final String type;

    private final Instant validFrom;

    private final Instant validUntil;

    private final String instanceInterruptionBehavior;

    private SpotInstanceRequest(BuilderImpl builder) {
        this.actualBlockHourlyPrice = builder.actualBlockHourlyPrice;
        this.availabilityZoneGroup = builder.availabilityZoneGroup;
        this.blockDurationMinutes = builder.blockDurationMinutes;
        this.createTime = builder.createTime;
        this.fault = builder.fault;
        this.instanceId = builder.instanceId;
        this.launchGroup = builder.launchGroup;
        this.launchSpecification = builder.launchSpecification;
        this.launchedAvailabilityZone = builder.launchedAvailabilityZone;
        this.productDescription = builder.productDescription;
        this.spotInstanceRequestId = builder.spotInstanceRequestId;
        this.spotPrice = builder.spotPrice;
        this.state = builder.state;
        this.status = builder.status;
        this.tags = builder.tags;
        this.type = builder.type;
        this.validFrom = builder.validFrom;
        this.validUntil = builder.validUntil;
        this.instanceInterruptionBehavior = builder.instanceInterruptionBehavior;
    }

    /**
     * <p>
     * If you specified a duration and your Spot Instance request was fulfilled, this is the fixed hourly price in
     * effect for the Spot Instance while it runs.
     * </p>
     * 
     * @return If you specified a duration and your Spot Instance request was fulfilled, this is the fixed hourly price
     *         in effect for the Spot Instance while it runs.
     */
    public String actualBlockHourlyPrice() {
        return actualBlockHourlyPrice;
    }

    /**
     * <p>
     * The Availability Zone group. If you specify the same Availability Zone group for all Spot Instance requests, all
     * Spot Instances are launched in the same Availability Zone.
     * </p>
     * 
     * @return The Availability Zone group. If you specify the same Availability Zone group for all Spot Instance
     *         requests, all Spot Instances are launched in the same Availability Zone.
     */
    public String availabilityZoneGroup() {
        return availabilityZoneGroup;
    }

    /**
     * <p>
     * The duration for the Spot Instance, in minutes.
     * </p>
     * 
     * @return The duration for the Spot Instance, in minutes.
     */
    public Integer blockDurationMinutes() {
        return blockDurationMinutes;
    }

    /**
     * <p>
     * The date and time when the Spot Instance request was created, in UTC format (for example,
     * <i>YYYY</i>-<i>MM</i>-<i>DD</i>T<i>HH</i>:<i>MM</i>:<i>SS</i>Z).
     * </p>
     * 
     * @return The date and time when the Spot Instance request was created, in UTC format (for example,
     *         <i>YYYY</i>-<i>MM</i>-<i>DD</i>T<i>HH</i>:<i>MM</i>:<i>SS</i>Z).
     */
    public Instant createTime() {
        return createTime;
    }

    /**
     * <p>
     * The fault codes for the Spot Instance request, if any.
     * </p>
     * 
     * @return The fault codes for the Spot Instance request, if any.
     */
    public SpotInstanceStateFault fault() {
        return fault;
    }

    /**
     * <p>
     * The instance ID, if an instance has been launched to fulfill the Spot Instance request.
     * </p>
     * 
     * @return The instance ID, if an instance has been launched to fulfill the Spot Instance request.
     */
    public String instanceId() {
        return instanceId;
    }

    /**
     * <p>
     * The instance launch group. Launch groups are Spot Instances that launch together and terminate together.
     * </p>
     * 
     * @return The instance launch group. Launch groups are Spot Instances that launch together and terminate together.
     */
    public String launchGroup() {
        return launchGroup;
    }

    /**
     * <p>
     * Additional information for launching instances.
     * </p>
     * 
     * @return Additional information for launching instances.
     */
    public LaunchSpecification launchSpecification() {
        return launchSpecification;
    }

    /**
     * <p>
     * The Availability Zone in which the request is launched.
     * </p>
     * 
     * @return The Availability Zone in which the request is launched.
     */
    public String launchedAvailabilityZone() {
        return launchedAvailabilityZone;
    }

    /**
     * <p>
     * The product description associated with the Spot Instance.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #productDescription} will return {@link RIProductDescription#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #productDescriptionAsString}.
     * </p>
     * 
     * @return The product description associated with the Spot Instance.
     * @see RIProductDescription
     */
    public RIProductDescription productDescription() {
        return RIProductDescription.fromValue(productDescription);
    }

    /**
     * <p>
     * The product description associated with the Spot Instance.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #productDescription} will return {@link RIProductDescription#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #productDescriptionAsString}.
     * </p>
     * 
     * @return The product description associated with the Spot Instance.
     * @see RIProductDescription
     */
    public String productDescriptionAsString() {
        return productDescription;
    }

    /**
     * <p>
     * The ID of the Spot Instance request.
     * </p>
     * 
     * @return The ID of the Spot Instance request.
     */
    public String spotInstanceRequestId() {
        return spotInstanceRequestId;
    }

    /**
     * <p>
     * The maximum price per hour that you are willing to pay for a Spot Instance.
     * </p>
     * 
     * @return The maximum price per hour that you are willing to pay for a Spot Instance.
     */
    public String spotPrice() {
        return spotPrice;
    }

    /**
     * <p>
     * The state of the Spot Instance request. Spot status information helps track your Spot Instance requests. For more
     * information, see <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/spot-bid-status.html">Spot
     * status</a> in the <i>Amazon EC2 User Guide for Linux Instances</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link SpotInstanceState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The state of the Spot Instance request. Spot status information helps track your Spot Instance requests.
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/spot-bid-status.html">Spot status</a> in the
     *         <i>Amazon EC2 User Guide for Linux Instances</i>.
     * @see SpotInstanceState
     */
    public SpotInstanceState state() {
        return SpotInstanceState.fromValue(state);
    }

    /**
     * <p>
     * The state of the Spot Instance request. Spot status information helps track your Spot Instance requests. For more
     * information, see <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/spot-bid-status.html">Spot
     * status</a> in the <i>Amazon EC2 User Guide for Linux Instances</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link SpotInstanceState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The state of the Spot Instance request. Spot status information helps track your Spot Instance requests.
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/spot-bid-status.html">Spot status</a> in the
     *         <i>Amazon EC2 User Guide for Linux Instances</i>.
     * @see SpotInstanceState
     */
    public String stateAsString() {
        return state;
    }

    /**
     * <p>
     * The status code and status message describing the Spot Instance request.
     * </p>
     * 
     * @return The status code and status message describing the Spot Instance request.
     */
    public SpotInstanceStatus status() {
        return status;
    }

    /**
     * Returns true if the Tags property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Any tags assigned to the resource.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTags()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Any tags assigned to the resource.
     */
    public List<Tag> tags() {
        return tags;
    }

    /**
     * <p>
     * The Spot Instance request type.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link SpotInstanceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The Spot Instance request type.
     * @see SpotInstanceType
     */
    public SpotInstanceType type() {
        return SpotInstanceType.fromValue(type);
    }

    /**
     * <p>
     * The Spot Instance request type.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link SpotInstanceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The Spot Instance request type.
     * @see SpotInstanceType
     */
    public String typeAsString() {
        return type;
    }

    /**
     * <p>
     * The start date of the request, in UTC format (for example,
     * <i>YYYY</i>-<i>MM</i>-<i>DD</i>T<i>HH</i>:<i>MM</i>:<i>SS</i>Z). The request becomes active at this date and
     * time.
     * </p>
     * 
     * @return The start date of the request, in UTC format (for example,
     *         <i>YYYY</i>-<i>MM</i>-<i>DD</i>T<i>HH</i>:<i>MM</i>:<i>SS</i>Z). The request becomes active at this date
     *         and time.
     */
    public Instant validFrom() {
        return validFrom;
    }

    /**
     * <p>
     * The end date of the request, in UTC format (for example,
     * <i>YYYY</i>-<i>MM</i>-<i>DD</i>T<i>HH</i>:<i>MM</i>:<i>SS</i>Z). If this is a one-time request, it remains active
     * until all instances launch, the request is canceled, or this date is reached. If the request is persistent, it
     * remains active until it is canceled or this date is reached. The default end date is 7 days from the current
     * date.
     * </p>
     * 
     * @return The end date of the request, in UTC format (for example,
     *         <i>YYYY</i>-<i>MM</i>-<i>DD</i>T<i>HH</i>:<i>MM</i>:<i>SS</i>Z). If this is a one-time request, it
     *         remains active until all instances launch, the request is canceled, or this date is reached. If the
     *         request is persistent, it remains active until it is canceled or this date is reached. The default end
     *         date is 7 days from the current date.
     */
    public Instant validUntil() {
        return validUntil;
    }

    /**
     * <p>
     * The behavior when a Spot Instance is interrupted.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #instanceInterruptionBehavior} will return {@link InstanceInterruptionBehavior#UNKNOWN_TO_SDK_VERSION}.
     * The raw value returned by the service is available from {@link #instanceInterruptionBehaviorAsString}.
     * </p>
     * 
     * @return The behavior when a Spot Instance is interrupted.
     * @see InstanceInterruptionBehavior
     */
    public InstanceInterruptionBehavior instanceInterruptionBehavior() {
        return InstanceInterruptionBehavior.fromValue(instanceInterruptionBehavior);
    }

    /**
     * <p>
     * The behavior when a Spot Instance is interrupted.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #instanceInterruptionBehavior} will return {@link InstanceInterruptionBehavior#UNKNOWN_TO_SDK_VERSION}.
     * The raw value returned by the service is available from {@link #instanceInterruptionBehaviorAsString}.
     * </p>
     * 
     * @return The behavior when a Spot Instance is interrupted.
     * @see InstanceInterruptionBehavior
     */
    public String instanceInterruptionBehaviorAsString() {
        return instanceInterruptionBehavior;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(actualBlockHourlyPrice());
        hashCode = 31 * hashCode + Objects.hashCode(availabilityZoneGroup());
        hashCode = 31 * hashCode + Objects.hashCode(blockDurationMinutes());
        hashCode = 31 * hashCode + Objects.hashCode(createTime());
        hashCode = 31 * hashCode + Objects.hashCode(fault());
        hashCode = 31 * hashCode + Objects.hashCode(instanceId());
        hashCode = 31 * hashCode + Objects.hashCode(launchGroup());
        hashCode = 31 * hashCode + Objects.hashCode(launchSpecification());
        hashCode = 31 * hashCode + Objects.hashCode(launchedAvailabilityZone());
        hashCode = 31 * hashCode + Objects.hashCode(productDescriptionAsString());
        hashCode = 31 * hashCode + Objects.hashCode(spotInstanceRequestId());
        hashCode = 31 * hashCode + Objects.hashCode(spotPrice());
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(status());
        hashCode = 31 * hashCode + Objects.hashCode(tags());
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(validFrom());
        hashCode = 31 * hashCode + Objects.hashCode(validUntil());
        hashCode = 31 * hashCode + Objects.hashCode(instanceInterruptionBehaviorAsString());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SpotInstanceRequest)) {
            return false;
        }
        SpotInstanceRequest other = (SpotInstanceRequest) obj;
        return Objects.equals(actualBlockHourlyPrice(), other.actualBlockHourlyPrice())
                && Objects.equals(availabilityZoneGroup(), other.availabilityZoneGroup())
                && Objects.equals(blockDurationMinutes(), other.blockDurationMinutes())
                && Objects.equals(createTime(), other.createTime()) && Objects.equals(fault(), other.fault())
                && Objects.equals(instanceId(), other.instanceId()) && Objects.equals(launchGroup(), other.launchGroup())
                && Objects.equals(launchSpecification(), other.launchSpecification())
                && Objects.equals(launchedAvailabilityZone(), other.launchedAvailabilityZone())
                && Objects.equals(productDescriptionAsString(), other.productDescriptionAsString())
                && Objects.equals(spotInstanceRequestId(), other.spotInstanceRequestId())
                && Objects.equals(spotPrice(), other.spotPrice()) && Objects.equals(stateAsString(), other.stateAsString())
                && Objects.equals(status(), other.status()) && Objects.equals(tags(), other.tags())
                && Objects.equals(typeAsString(), other.typeAsString()) && Objects.equals(validFrom(), other.validFrom())
                && Objects.equals(validUntil(), other.validUntil())
                && Objects.equals(instanceInterruptionBehaviorAsString(), other.instanceInterruptionBehaviorAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("SpotInstanceRequest").add("ActualBlockHourlyPrice", actualBlockHourlyPrice())
                .add("AvailabilityZoneGroup", availabilityZoneGroup()).add("BlockDurationMinutes", blockDurationMinutes())
                .add("CreateTime", createTime()).add("Fault", fault()).add("InstanceId", instanceId())
                .add("LaunchGroup", launchGroup()).add("LaunchSpecification", launchSpecification())
                .add("LaunchedAvailabilityZone", launchedAvailabilityZone())
                .add("ProductDescription", productDescriptionAsString()).add("SpotInstanceRequestId", spotInstanceRequestId())
                .add("SpotPrice", spotPrice()).add("State", stateAsString()).add("Status", status()).add("Tags", tags())
                .add("Type", typeAsString()).add("ValidFrom", validFrom()).add("ValidUntil", validUntil())
                .add("InstanceInterruptionBehavior", instanceInterruptionBehaviorAsString()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ActualBlockHourlyPrice":
            return Optional.ofNullable(clazz.cast(actualBlockHourlyPrice()));
        case "AvailabilityZoneGroup":
            return Optional.ofNullable(clazz.cast(availabilityZoneGroup()));
        case "BlockDurationMinutes":
            return Optional.ofNullable(clazz.cast(blockDurationMinutes()));
        case "CreateTime":
            return Optional.ofNullable(clazz.cast(createTime()));
        case "Fault":
            return Optional.ofNullable(clazz.cast(fault()));
        case "InstanceId":
            return Optional.ofNullable(clazz.cast(instanceId()));
        case "LaunchGroup":
            return Optional.ofNullable(clazz.cast(launchGroup()));
        case "LaunchSpecification":
            return Optional.ofNullable(clazz.cast(launchSpecification()));
        case "LaunchedAvailabilityZone":
            return Optional.ofNullable(clazz.cast(launchedAvailabilityZone()));
        case "ProductDescription":
            return Optional.ofNullable(clazz.cast(productDescriptionAsString()));
        case "SpotInstanceRequestId":
            return Optional.ofNullable(clazz.cast(spotInstanceRequestId()));
        case "SpotPrice":
            return Optional.ofNullable(clazz.cast(spotPrice()));
        case "State":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "Status":
            return Optional.ofNullable(clazz.cast(status()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "Type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "ValidFrom":
            return Optional.ofNullable(clazz.cast(validFrom()));
        case "ValidUntil":
            return Optional.ofNullable(clazz.cast(validUntil()));
        case "InstanceInterruptionBehavior":
            return Optional.ofNullable(clazz.cast(instanceInterruptionBehaviorAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<SpotInstanceRequest, T> g) {
        return obj -> g.apply((SpotInstanceRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, SpotInstanceRequest> {
        /**
         * <p>
         * If you specified a duration and your Spot Instance request was fulfilled, this is the fixed hourly price in
         * effect for the Spot Instance while it runs.
         * </p>
         * 
         * @param actualBlockHourlyPrice
         *        If you specified a duration and your Spot Instance request was fulfilled, this is the fixed hourly
         *        price in effect for the Spot Instance while it runs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder actualBlockHourlyPrice(String actualBlockHourlyPrice);

        /**
         * <p>
         * The Availability Zone group. If you specify the same Availability Zone group for all Spot Instance requests,
         * all Spot Instances are launched in the same Availability Zone.
         * </p>
         * 
         * @param availabilityZoneGroup
         *        The Availability Zone group. If you specify the same Availability Zone group for all Spot Instance
         *        requests, all Spot Instances are launched in the same Availability Zone.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder availabilityZoneGroup(String availabilityZoneGroup);

        /**
         * <p>
         * The duration for the Spot Instance, in minutes.
         * </p>
         * 
         * @param blockDurationMinutes
         *        The duration for the Spot Instance, in minutes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder blockDurationMinutes(Integer blockDurationMinutes);

        /**
         * <p>
         * The date and time when the Spot Instance request was created, in UTC format (for example,
         * <i>YYYY</i>-<i>MM</i>-<i>DD</i>T<i>HH</i>:<i>MM</i>:<i>SS</i>Z).
         * </p>
         * 
         * @param createTime
         *        The date and time when the Spot Instance request was created, in UTC format (for example,
         *        <i>YYYY</i>-<i>MM</i>-<i>DD</i>T<i>HH</i>:<i>MM</i>:<i>SS</i>Z).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createTime(Instant createTime);

        /**
         * <p>
         * The fault codes for the Spot Instance request, if any.
         * </p>
         * 
         * @param fault
         *        The fault codes for the Spot Instance request, if any.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fault(SpotInstanceStateFault fault);

        /**
         * <p>
         * The fault codes for the Spot Instance request, if any.
         * </p>
         * This is a convenience that creates an instance of the {@link SpotInstanceStateFault.Builder} avoiding the
         * need to create one manually via {@link SpotInstanceStateFault#builder()}.
         *
         * When the {@link Consumer} completes, {@link SpotInstanceStateFault.Builder#build()} is called immediately and
         * its result is passed to {@link #fault(SpotInstanceStateFault)}.
         * 
         * @param fault
         *        a consumer that will call methods on {@link SpotInstanceStateFault.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #fault(SpotInstanceStateFault)
         */
        default Builder fault(Consumer<SpotInstanceStateFault.Builder> fault) {
            return fault(SpotInstanceStateFault.builder().applyMutation(fault).build());
        }

        /**
         * <p>
         * The instance ID, if an instance has been launched to fulfill the Spot Instance request.
         * </p>
         * 
         * @param instanceId
         *        The instance ID, if an instance has been launched to fulfill the Spot Instance request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceId(String instanceId);

        /**
         * <p>
         * The instance launch group. Launch groups are Spot Instances that launch together and terminate together.
         * </p>
         * 
         * @param launchGroup
         *        The instance launch group. Launch groups are Spot Instances that launch together and terminate
         *        together.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder launchGroup(String launchGroup);

        /**
         * <p>
         * Additional information for launching instances.
         * </p>
         * 
         * @param launchSpecification
         *        Additional information for launching instances.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder launchSpecification(LaunchSpecification launchSpecification);

        /**
         * <p>
         * Additional information for launching instances.
         * </p>
         * This is a convenience that creates an instance of the {@link LaunchSpecification.Builder} avoiding the need
         * to create one manually via {@link LaunchSpecification#builder()}.
         *
         * When the {@link Consumer} completes, {@link LaunchSpecification.Builder#build()} is called immediately and
         * its result is passed to {@link #launchSpecification(LaunchSpecification)}.
         * 
         * @param launchSpecification
         *        a consumer that will call methods on {@link LaunchSpecification.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #launchSpecification(LaunchSpecification)
         */
        default Builder launchSpecification(Consumer<LaunchSpecification.Builder> launchSpecification) {
            return launchSpecification(LaunchSpecification.builder().applyMutation(launchSpecification).build());
        }

        /**
         * <p>
         * The Availability Zone in which the request is launched.
         * </p>
         * 
         * @param launchedAvailabilityZone
         *        The Availability Zone in which the request is launched.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder launchedAvailabilityZone(String launchedAvailabilityZone);

        /**
         * <p>
         * The product description associated with the Spot Instance.
         * </p>
         * 
         * @param productDescription
         *        The product description associated with the Spot Instance.
         * @see RIProductDescription
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RIProductDescription
         */
        Builder productDescription(String productDescription);

        /**
         * <p>
         * The product description associated with the Spot Instance.
         * </p>
         * 
         * @param productDescription
         *        The product description associated with the Spot Instance.
         * @see RIProductDescription
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RIProductDescription
         */
        Builder productDescription(RIProductDescription productDescription);

        /**
         * <p>
         * The ID of the Spot Instance request.
         * </p>
         * 
         * @param spotInstanceRequestId
         *        The ID of the Spot Instance request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder spotInstanceRequestId(String spotInstanceRequestId);

        /**
         * <p>
         * The maximum price per hour that you are willing to pay for a Spot Instance.
         * </p>
         * 
         * @param spotPrice
         *        The maximum price per hour that you are willing to pay for a Spot Instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder spotPrice(String spotPrice);

        /**
         * <p>
         * The state of the Spot Instance request. Spot status information helps track your Spot Instance requests. For
         * more information, see <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/spot-bid-status.html">Spot
         * status</a> in the <i>Amazon EC2 User Guide for Linux Instances</i>.
         * </p>
         * 
         * @param state
         *        The state of the Spot Instance request. Spot status information helps track your Spot Instance
         *        requests. For more information, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/spot-bid-status.html">Spot status</a> in the
         *        <i>Amazon EC2 User Guide for Linux Instances</i>.
         * @see SpotInstanceState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SpotInstanceState
         */
        Builder state(String state);

        /**
         * <p>
         * The state of the Spot Instance request. Spot status information helps track your Spot Instance requests. For
         * more information, see <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/spot-bid-status.html">Spot
         * status</a> in the <i>Amazon EC2 User Guide for Linux Instances</i>.
         * </p>
         * 
         * @param state
         *        The state of the Spot Instance request. Spot status information helps track your Spot Instance
         *        requests. For more information, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/spot-bid-status.html">Spot status</a> in the
         *        <i>Amazon EC2 User Guide for Linux Instances</i>.
         * @see SpotInstanceState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SpotInstanceState
         */
        Builder state(SpotInstanceState state);

        /**
         * <p>
         * The status code and status message describing the Spot Instance request.
         * </p>
         * 
         * @param status
         *        The status code and status message describing the Spot Instance request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder status(SpotInstanceStatus status);

        /**
         * <p>
         * The status code and status message describing the Spot Instance request.
         * </p>
         * This is a convenience that creates an instance of the {@link SpotInstanceStatus.Builder} avoiding the need to
         * create one manually via {@link SpotInstanceStatus#builder()}.
         *
         * When the {@link Consumer} completes, {@link SpotInstanceStatus.Builder#build()} is called immediately and its
         * result is passed to {@link #status(SpotInstanceStatus)}.
         * 
         * @param status
         *        a consumer that will call methods on {@link SpotInstanceStatus.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #status(SpotInstanceStatus)
         */
        default Builder status(Consumer<SpotInstanceStatus.Builder> status) {
            return status(SpotInstanceStatus.builder().applyMutation(status).build());
        }

        /**
         * <p>
         * Any tags assigned to the resource.
         * </p>
         * 
         * @param tags
         *        Any tags assigned to the resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * Any tags assigned to the resource.
         * </p>
         * 
         * @param tags
         *        Any tags assigned to the resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * Any tags assigned to the resource.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Tag>.Builder} avoiding the need to create
         * one manually via {@link List<Tag>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Tag>.Builder#build()} is called immediately and its result
         * is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link List<Tag>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(List<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * The Spot Instance request type.
         * </p>
         * 
         * @param type
         *        The Spot Instance request type.
         * @see SpotInstanceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SpotInstanceType
         */
        Builder type(String type);

        /**
         * <p>
         * The Spot Instance request type.
         * </p>
         * 
         * @param type
         *        The Spot Instance request type.
         * @see SpotInstanceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SpotInstanceType
         */
        Builder type(SpotInstanceType type);

        /**
         * <p>
         * The start date of the request, in UTC format (for example,
         * <i>YYYY</i>-<i>MM</i>-<i>DD</i>T<i>HH</i>:<i>MM</i>:<i>SS</i>Z). The request becomes active at this date and
         * time.
         * </p>
         * 
         * @param validFrom
         *        The start date of the request, in UTC format (for example,
         *        <i>YYYY</i>-<i>MM</i>-<i>DD</i>T<i>HH</i>:<i>MM</i>:<i>SS</i>Z). The request becomes active at this
         *        date and time.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder validFrom(Instant validFrom);

        /**
         * <p>
         * The end date of the request, in UTC format (for example,
         * <i>YYYY</i>-<i>MM</i>-<i>DD</i>T<i>HH</i>:<i>MM</i>:<i>SS</i>Z). If this is a one-time request, it remains
         * active until all instances launch, the request is canceled, or this date is reached. If the request is
         * persistent, it remains active until it is canceled or this date is reached. The default end date is 7 days
         * from the current date.
         * </p>
         * 
         * @param validUntil
         *        The end date of the request, in UTC format (for example,
         *        <i>YYYY</i>-<i>MM</i>-<i>DD</i>T<i>HH</i>:<i>MM</i>:<i>SS</i>Z). If this is a one-time request, it
         *        remains active until all instances launch, the request is canceled, or this date is reached. If the
         *        request is persistent, it remains active until it is canceled or this date is reached. The default end
         *        date is 7 days from the current date.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder validUntil(Instant validUntil);

        /**
         * <p>
         * The behavior when a Spot Instance is interrupted.
         * </p>
         * 
         * @param instanceInterruptionBehavior
         *        The behavior when a Spot Instance is interrupted.
         * @see InstanceInterruptionBehavior
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InstanceInterruptionBehavior
         */
        Builder instanceInterruptionBehavior(String instanceInterruptionBehavior);

        /**
         * <p>
         * The behavior when a Spot Instance is interrupted.
         * </p>
         * 
         * @param instanceInterruptionBehavior
         *        The behavior when a Spot Instance is interrupted.
         * @see InstanceInterruptionBehavior
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InstanceInterruptionBehavior
         */
        Builder instanceInterruptionBehavior(InstanceInterruptionBehavior instanceInterruptionBehavior);
    }

    static final class BuilderImpl implements Builder {
        private String actualBlockHourlyPrice;

        private String availabilityZoneGroup;

        private Integer blockDurationMinutes;

        private Instant createTime;

        private SpotInstanceStateFault fault;

        private String instanceId;

        private String launchGroup;

        private LaunchSpecification launchSpecification;

        private String launchedAvailabilityZone;

        private String productDescription;

        private String spotInstanceRequestId;

        private String spotPrice;

        private String state;

        private SpotInstanceStatus status;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private String type;

        private Instant validFrom;

        private Instant validUntil;

        private String instanceInterruptionBehavior;

        private BuilderImpl() {
        }

        private BuilderImpl(SpotInstanceRequest model) {
            actualBlockHourlyPrice(model.actualBlockHourlyPrice);
            availabilityZoneGroup(model.availabilityZoneGroup);
            blockDurationMinutes(model.blockDurationMinutes);
            createTime(model.createTime);
            fault(model.fault);
            instanceId(model.instanceId);
            launchGroup(model.launchGroup);
            launchSpecification(model.launchSpecification);
            launchedAvailabilityZone(model.launchedAvailabilityZone);
            productDescription(model.productDescription);
            spotInstanceRequestId(model.spotInstanceRequestId);
            spotPrice(model.spotPrice);
            state(model.state);
            status(model.status);
            tags(model.tags);
            type(model.type);
            validFrom(model.validFrom);
            validUntil(model.validUntil);
            instanceInterruptionBehavior(model.instanceInterruptionBehavior);
        }

        public final String getActualBlockHourlyPrice() {
            return actualBlockHourlyPrice;
        }

        @Override
        public final Builder actualBlockHourlyPrice(String actualBlockHourlyPrice) {
            this.actualBlockHourlyPrice = actualBlockHourlyPrice;
            return this;
        }

        public final void setActualBlockHourlyPrice(String actualBlockHourlyPrice) {
            this.actualBlockHourlyPrice = actualBlockHourlyPrice;
        }

        public final String getAvailabilityZoneGroup() {
            return availabilityZoneGroup;
        }

        @Override
        public final Builder availabilityZoneGroup(String availabilityZoneGroup) {
            this.availabilityZoneGroup = availabilityZoneGroup;
            return this;
        }

        public final void setAvailabilityZoneGroup(String availabilityZoneGroup) {
            this.availabilityZoneGroup = availabilityZoneGroup;
        }

        public final Integer getBlockDurationMinutes() {
            return blockDurationMinutes;
        }

        @Override
        public final Builder blockDurationMinutes(Integer blockDurationMinutes) {
            this.blockDurationMinutes = blockDurationMinutes;
            return this;
        }

        public final void setBlockDurationMinutes(Integer blockDurationMinutes) {
            this.blockDurationMinutes = blockDurationMinutes;
        }

        public final Instant getCreateTime() {
            return createTime;
        }

        @Override
        public final Builder createTime(Instant createTime) {
            this.createTime = createTime;
            return this;
        }

        public final void setCreateTime(Instant createTime) {
            this.createTime = createTime;
        }

        public final SpotInstanceStateFault.Builder getFault() {
            return fault != null ? fault.toBuilder() : null;
        }

        @Override
        public final Builder fault(SpotInstanceStateFault fault) {
            this.fault = fault;
            return this;
        }

        public final void setFault(SpotInstanceStateFault.BuilderImpl fault) {
            this.fault = fault != null ? fault.build() : null;
        }

        public final String getInstanceId() {
            return instanceId;
        }

        @Override
        public final Builder instanceId(String instanceId) {
            this.instanceId = instanceId;
            return this;
        }

        public final void setInstanceId(String instanceId) {
            this.instanceId = instanceId;
        }

        public final String getLaunchGroup() {
            return launchGroup;
        }

        @Override
        public final Builder launchGroup(String launchGroup) {
            this.launchGroup = launchGroup;
            return this;
        }

        public final void setLaunchGroup(String launchGroup) {
            this.launchGroup = launchGroup;
        }

        public final LaunchSpecification.Builder getLaunchSpecification() {
            return launchSpecification != null ? launchSpecification.toBuilder() : null;
        }

        @Override
        public final Builder launchSpecification(LaunchSpecification launchSpecification) {
            this.launchSpecification = launchSpecification;
            return this;
        }

        public final void setLaunchSpecification(LaunchSpecification.BuilderImpl launchSpecification) {
            this.launchSpecification = launchSpecification != null ? launchSpecification.build() : null;
        }

        public final String getLaunchedAvailabilityZone() {
            return launchedAvailabilityZone;
        }

        @Override
        public final Builder launchedAvailabilityZone(String launchedAvailabilityZone) {
            this.launchedAvailabilityZone = launchedAvailabilityZone;
            return this;
        }

        public final void setLaunchedAvailabilityZone(String launchedAvailabilityZone) {
            this.launchedAvailabilityZone = launchedAvailabilityZone;
        }

        public final String getProductDescription() {
            return productDescription;
        }

        @Override
        public final Builder productDescription(String productDescription) {
            this.productDescription = productDescription;
            return this;
        }

        @Override
        public final Builder productDescription(RIProductDescription productDescription) {
            this.productDescription(productDescription == null ? null : productDescription.toString());
            return this;
        }

        public final void setProductDescription(String productDescription) {
            this.productDescription = productDescription;
        }

        public final String getSpotInstanceRequestId() {
            return spotInstanceRequestId;
        }

        @Override
        public final Builder spotInstanceRequestId(String spotInstanceRequestId) {
            this.spotInstanceRequestId = spotInstanceRequestId;
            return this;
        }

        public final void setSpotInstanceRequestId(String spotInstanceRequestId) {
            this.spotInstanceRequestId = spotInstanceRequestId;
        }

        public final String getSpotPrice() {
            return spotPrice;
        }

        @Override
        public final Builder spotPrice(String spotPrice) {
            this.spotPrice = spotPrice;
            return this;
        }

        public final void setSpotPrice(String spotPrice) {
            this.spotPrice = spotPrice;
        }

        public final String getState() {
            return state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        public final Builder state(SpotInstanceState state) {
            this.state(state == null ? null : state.toString());
            return this;
        }

        public final void setState(String state) {
            this.state = state;
        }

        public final SpotInstanceStatus.Builder getStatus() {
            return status != null ? status.toBuilder() : null;
        }

        @Override
        public final Builder status(SpotInstanceStatus status) {
            this.status = status;
            return this;
        }

        public final void setStatus(SpotInstanceStatus.BuilderImpl status) {
            this.status = status != null ? status.build() : null;
        }

        public final Collection<Tag.Builder> getTags() {
            return tags != null ? tags.stream().map(Tag::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        public final String getType() {
            return type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(SpotInstanceType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final void setType(String type) {
            this.type = type;
        }

        public final Instant getValidFrom() {
            return validFrom;
        }

        @Override
        public final Builder validFrom(Instant validFrom) {
            this.validFrom = validFrom;
            return this;
        }

        public final void setValidFrom(Instant validFrom) {
            this.validFrom = validFrom;
        }

        public final Instant getValidUntil() {
            return validUntil;
        }

        @Override
        public final Builder validUntil(Instant validUntil) {
            this.validUntil = validUntil;
            return this;
        }

        public final void setValidUntil(Instant validUntil) {
            this.validUntil = validUntil;
        }

        public final String getInstanceInterruptionBehavior() {
            return instanceInterruptionBehavior;
        }

        @Override
        public final Builder instanceInterruptionBehavior(String instanceInterruptionBehavior) {
            this.instanceInterruptionBehavior = instanceInterruptionBehavior;
            return this;
        }

        @Override
        public final Builder instanceInterruptionBehavior(InstanceInterruptionBehavior instanceInterruptionBehavior) {
            this.instanceInterruptionBehavior(instanceInterruptionBehavior == null ? null : instanceInterruptionBehavior
                    .toString());
            return this;
        }

        public final void setInstanceInterruptionBehavior(String instanceInterruptionBehavior) {
            this.instanceInterruptionBehavior = instanceInterruptionBehavior;
        }

        @Override
        public SpotInstanceRequest build() {
            return new SpotInstanceRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
