/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2instanceconnect;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.ec2instanceconnect.internal.Ec2InstanceConnectServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.ec2instanceconnect.model.AuthException;
import software.amazon.awssdk.services.ec2instanceconnect.model.Ec2InstanceConnectException;
import software.amazon.awssdk.services.ec2instanceconnect.model.Ec2InstanceNotFoundException;
import software.amazon.awssdk.services.ec2instanceconnect.model.Ec2InstanceStateInvalidException;
import software.amazon.awssdk.services.ec2instanceconnect.model.Ec2InstanceTypeInvalidException;
import software.amazon.awssdk.services.ec2instanceconnect.model.Ec2InstanceUnavailableException;
import software.amazon.awssdk.services.ec2instanceconnect.model.InvalidArgsException;
import software.amazon.awssdk.services.ec2instanceconnect.model.SendSerialConsoleSshPublicKeyRequest;
import software.amazon.awssdk.services.ec2instanceconnect.model.SendSerialConsoleSshPublicKeyResponse;
import software.amazon.awssdk.services.ec2instanceconnect.model.SendSshPublicKeyRequest;
import software.amazon.awssdk.services.ec2instanceconnect.model.SendSshPublicKeyResponse;
import software.amazon.awssdk.services.ec2instanceconnect.model.SerialConsoleAccessDisabledException;
import software.amazon.awssdk.services.ec2instanceconnect.model.SerialConsoleSessionLimitExceededException;
import software.amazon.awssdk.services.ec2instanceconnect.model.SerialConsoleSessionUnavailableException;
import software.amazon.awssdk.services.ec2instanceconnect.model.ServiceException;
import software.amazon.awssdk.services.ec2instanceconnect.model.ThrottlingException;
import software.amazon.awssdk.services.ec2instanceconnect.transform.SendSerialConsoleSshPublicKeyRequestMarshaller;
import software.amazon.awssdk.services.ec2instanceconnect.transform.SendSshPublicKeyRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link Ec2InstanceConnectAsyncClient}.
 *
 * @see Ec2InstanceConnectAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultEc2InstanceConnectAsyncClient implements Ec2InstanceConnectAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultEc2InstanceConnectAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.AWS_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultEc2InstanceConnectAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Pushes an SSH public key to the specified EC2 instance for use by the specified user. The key remains for 60
     * seconds. For more information, see <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/Connect-using-EC2-Instance-Connect.html">Connect to
     * your Linux instance using EC2 Instance Connect</a> in the <i>Amazon EC2 User Guide</i>.
     * </p>
     *
     * @param sendSshPublicKeyRequest
     * @return A Java Future containing the result of the SendSSHPublicKey operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AuthException Either your AWS credentials are not valid or you do not have access to the EC2
     *         instance.</li>
     *         <li>InvalidArgsException One of the parameters is not valid.</li>
     *         <li>ServiceException The service encountered an error. Follow the instructions in the error message and
     *         try again.</li>
     *         <li>ThrottlingException The requests were made too frequently and have been throttled. Wait a while and
     *         try again. To increase the limit on your request frequency, contact AWS Support.</li>
     *         <li>Ec2InstanceNotFoundException The specified instance was not found.</li>
     *         <li>Ec2InstanceStateInvalidException Unable to connect because the instance is not in a valid state.
     *         Connecting to a stopped or terminated instance is not supported. If the instance is stopped, start your
     *         instance, and try to connect again.</li>
     *         <li>Ec2InstanceUnavailableException The instance is currently unavailable. Wait a few minutes and try
     *         again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Ec2InstanceConnectException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample Ec2InstanceConnectAsyncClient.SendSSHPublicKey
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ec2-instance-connect-2018-04-02/SendSSHPublicKey"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<SendSshPublicKeyResponse> sendSSHPublicKey(SendSshPublicKeyRequest sendSshPublicKeyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(sendSshPublicKeyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, sendSshPublicKeyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EC2 Instance Connect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SendSSHPublicKey");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SendSshPublicKeyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, SendSshPublicKeyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<SendSshPublicKeyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SendSshPublicKeyRequest, SendSshPublicKeyResponse>()
                            .withOperationName("SendSSHPublicKey").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new SendSshPublicKeyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(sendSshPublicKeyRequest));
            CompletableFuture<SendSshPublicKeyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Pushes an SSH public key to the specified EC2 instance. The key remains for 60 seconds, which gives you 60
     * seconds to establish a serial console connection to the instance using SSH. For more information, see <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-serial-console.html">EC2 Serial Console</a> in the
     * <i>Amazon EC2 User Guide</i>.
     * </p>
     *
     * @param sendSerialConsoleSshPublicKeyRequest
     * @return A Java Future containing the result of the SendSerialConsoleSSHPublicKey operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AuthException Either your AWS credentials are not valid or you do not have access to the EC2
     *         instance.</li>
     *         <li>SerialConsoleAccessDisabledException Your account is not authorized to use the EC2 Serial Console. To
     *         authorize your account, run the EnableSerialConsoleAccess API. For more information, see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_EnableSerialConsoleAccess.html"
     *         >EnableSerialConsoleAccess</a> in the <i>Amazon EC2 API Reference</i>.</li>
     *         <li>InvalidArgsException One of the parameters is not valid.</li>
     *         <li>ServiceException The service encountered an error. Follow the instructions in the error message and
     *         try again.</li>
     *         <li>ThrottlingException The requests were made too frequently and have been throttled. Wait a while and
     *         try again. To increase the limit on your request frequency, contact AWS Support.</li>
     *         <li>Ec2InstanceNotFoundException The specified instance was not found.</li>
     *         <li>Ec2InstanceTypeInvalidException The instance type is not supported for connecting via the serial
     *         console. Only Nitro instance types are currently supported.</li>
     *         <li>SerialConsoleSessionLimitExceededException The instance currently has 1 active serial console
     *         session. Only 1 session is supported at a time.</li>
     *         <li>SerialConsoleSessionUnavailableException Unable to start a serial console session. Please try again.</li>
     *         <li>Ec2InstanceStateInvalidException Unable to connect because the instance is not in a valid state.
     *         Connecting to a stopped or terminated instance is not supported. If the instance is stopped, start your
     *         instance, and try to connect again.</li>
     *         <li>Ec2InstanceUnavailableException The instance is currently unavailable. Wait a few minutes and try
     *         again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Ec2InstanceConnectException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample Ec2InstanceConnectAsyncClient.SendSerialConsoleSSHPublicKey
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/ec2-instance-connect-2018-04-02/SendSerialConsoleSSHPublicKey"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<SendSerialConsoleSshPublicKeyResponse> sendSerialConsoleSSHPublicKey(
            SendSerialConsoleSshPublicKeyRequest sendSerialConsoleSshPublicKeyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(sendSerialConsoleSshPublicKeyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                sendSerialConsoleSshPublicKeyRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EC2 Instance Connect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SendSerialConsoleSSHPublicKey");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SendSerialConsoleSshPublicKeyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, SendSerialConsoleSshPublicKeyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<SendSerialConsoleSshPublicKeyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SendSerialConsoleSshPublicKeyRequest, SendSerialConsoleSshPublicKeyResponse>()
                            .withOperationName("SendSerialConsoleSSHPublicKey").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new SendSerialConsoleSshPublicKeyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(sendSerialConsoleSshPublicKeyRequest));
            CompletableFuture<SendSerialConsoleSshPublicKeyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final Ec2InstanceConnectServiceClientConfiguration serviceClientConfiguration() {
        return new Ec2InstanceConnectServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(Ec2InstanceConnectException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AuthException").exceptionBuilderSupplier(AuthException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("EC2InstanceUnavailableException")
                                .exceptionBuilderSupplier(Ec2InstanceUnavailableException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("SerialConsoleSessionLimitExceededException")
                                .exceptionBuilderSupplier(SerialConsoleSessionLimitExceededException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("SerialConsoleAccessDisabledException")
                                .exceptionBuilderSupplier(SerialConsoleAccessDisabledException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("SerialConsoleSessionUnavailableException")
                                .exceptionBuilderSupplier(SerialConsoleSessionUnavailableException::builder).httpStatusCode(500)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("EC2InstanceTypeInvalidException")
                                .exceptionBuilderSupplier(Ec2InstanceTypeInvalidException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("EC2InstanceStateInvalidException")
                                .exceptionBuilderSupplier(Ec2InstanceStateInvalidException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceException")
                                .exceptionBuilderSupplier(ServiceException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("EC2InstanceNotFoundException")
                                .exceptionBuilderSupplier(Ec2InstanceNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidArgsException")
                                .exceptionBuilderSupplier(InvalidArgsException::builder).httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        Ec2InstanceConnectServiceClientConfigurationBuilder serviceConfigBuilder = new Ec2InstanceConnectServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
