/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.elastictranscoder;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.elastictranscoder.internal.ElasticTranscoderServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.elastictranscoder.model.AccessDeniedException;
import software.amazon.awssdk.services.elastictranscoder.model.CancelJobRequest;
import software.amazon.awssdk.services.elastictranscoder.model.CancelJobResponse;
import software.amazon.awssdk.services.elastictranscoder.model.CreateJobRequest;
import software.amazon.awssdk.services.elastictranscoder.model.CreateJobResponse;
import software.amazon.awssdk.services.elastictranscoder.model.CreatePipelineRequest;
import software.amazon.awssdk.services.elastictranscoder.model.CreatePipelineResponse;
import software.amazon.awssdk.services.elastictranscoder.model.CreatePresetRequest;
import software.amazon.awssdk.services.elastictranscoder.model.CreatePresetResponse;
import software.amazon.awssdk.services.elastictranscoder.model.DeletePipelineRequest;
import software.amazon.awssdk.services.elastictranscoder.model.DeletePipelineResponse;
import software.amazon.awssdk.services.elastictranscoder.model.DeletePresetRequest;
import software.amazon.awssdk.services.elastictranscoder.model.DeletePresetResponse;
import software.amazon.awssdk.services.elastictranscoder.model.ElasticTranscoderException;
import software.amazon.awssdk.services.elastictranscoder.model.IncompatibleVersionException;
import software.amazon.awssdk.services.elastictranscoder.model.InternalServiceException;
import software.amazon.awssdk.services.elastictranscoder.model.LimitExceededException;
import software.amazon.awssdk.services.elastictranscoder.model.ListJobsByPipelineRequest;
import software.amazon.awssdk.services.elastictranscoder.model.ListJobsByPipelineResponse;
import software.amazon.awssdk.services.elastictranscoder.model.ListJobsByStatusRequest;
import software.amazon.awssdk.services.elastictranscoder.model.ListJobsByStatusResponse;
import software.amazon.awssdk.services.elastictranscoder.model.ListPipelinesRequest;
import software.amazon.awssdk.services.elastictranscoder.model.ListPipelinesResponse;
import software.amazon.awssdk.services.elastictranscoder.model.ListPresetsRequest;
import software.amazon.awssdk.services.elastictranscoder.model.ListPresetsResponse;
import software.amazon.awssdk.services.elastictranscoder.model.ReadJobRequest;
import software.amazon.awssdk.services.elastictranscoder.model.ReadJobResponse;
import software.amazon.awssdk.services.elastictranscoder.model.ReadPipelineRequest;
import software.amazon.awssdk.services.elastictranscoder.model.ReadPipelineResponse;
import software.amazon.awssdk.services.elastictranscoder.model.ReadPresetRequest;
import software.amazon.awssdk.services.elastictranscoder.model.ReadPresetResponse;
import software.amazon.awssdk.services.elastictranscoder.model.ResourceInUseException;
import software.amazon.awssdk.services.elastictranscoder.model.ResourceNotFoundException;
import software.amazon.awssdk.services.elastictranscoder.model.UpdatePipelineNotificationsRequest;
import software.amazon.awssdk.services.elastictranscoder.model.UpdatePipelineNotificationsResponse;
import software.amazon.awssdk.services.elastictranscoder.model.UpdatePipelineRequest;
import software.amazon.awssdk.services.elastictranscoder.model.UpdatePipelineResponse;
import software.amazon.awssdk.services.elastictranscoder.model.UpdatePipelineStatusRequest;
import software.amazon.awssdk.services.elastictranscoder.model.UpdatePipelineStatusResponse;
import software.amazon.awssdk.services.elastictranscoder.model.ValidationException;
import software.amazon.awssdk.services.elastictranscoder.transform.CancelJobRequestMarshaller;
import software.amazon.awssdk.services.elastictranscoder.transform.CreateJobRequestMarshaller;
import software.amazon.awssdk.services.elastictranscoder.transform.CreatePipelineRequestMarshaller;
import software.amazon.awssdk.services.elastictranscoder.transform.CreatePresetRequestMarshaller;
import software.amazon.awssdk.services.elastictranscoder.transform.DeletePipelineRequestMarshaller;
import software.amazon.awssdk.services.elastictranscoder.transform.DeletePresetRequestMarshaller;
import software.amazon.awssdk.services.elastictranscoder.transform.ListJobsByPipelineRequestMarshaller;
import software.amazon.awssdk.services.elastictranscoder.transform.ListJobsByStatusRequestMarshaller;
import software.amazon.awssdk.services.elastictranscoder.transform.ListPipelinesRequestMarshaller;
import software.amazon.awssdk.services.elastictranscoder.transform.ListPresetsRequestMarshaller;
import software.amazon.awssdk.services.elastictranscoder.transform.ReadJobRequestMarshaller;
import software.amazon.awssdk.services.elastictranscoder.transform.ReadPipelineRequestMarshaller;
import software.amazon.awssdk.services.elastictranscoder.transform.ReadPresetRequestMarshaller;
import software.amazon.awssdk.services.elastictranscoder.transform.UpdatePipelineNotificationsRequestMarshaller;
import software.amazon.awssdk.services.elastictranscoder.transform.UpdatePipelineRequestMarshaller;
import software.amazon.awssdk.services.elastictranscoder.transform.UpdatePipelineStatusRequestMarshaller;
import software.amazon.awssdk.services.elastictranscoder.waiters.ElasticTranscoderWaiter;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link ElasticTranscoderClient}.
 *
 * @see ElasticTranscoderClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultElasticTranscoderClient implements ElasticTranscoderClient {
    private static final Logger log = Logger.loggerFor(DefaultElasticTranscoderClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultElasticTranscoderClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * The CancelJob operation cancels an unfinished job.
     * </p>
     * <note>
     * <p>
     * You can only cancel a job that has a status of <code>Submitted</code>. To prevent a pipeline from starting to
     * process a job while you're getting the job identifier, use <a>UpdatePipelineStatus</a> to temporarily pause the
     * pipeline.
     * </p>
     * </note>
     *
     * @param cancelJobRequest
     *        The <code>CancelJobRequest</code> structure.
     * @return Result of the CancelJob operation returned by the service.
     * @throws ValidationException
     *         One or more required parameter values were not provided in the request.
     * @throws IncompatibleVersionException
     * @throws ResourceNotFoundException
     *         The requested resource does not exist or is not available. For example, the pipeline to which you're
     *         trying to add a job doesn't exist or is still being created.
     * @throws ResourceInUseException
     *         The resource you are attempting to change is in use. For example, you are attempting to delete a pipeline
     *         that is currently in use.
     * @throws AccessDeniedException
     *         General authentication failure. The request was not signed correctly.
     * @throws InternalServiceException
     *         Elastic Transcoder encountered an unexpected exception while trying to fulfill the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ElasticTranscoderException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ElasticTranscoderClient.CancelJob
     */
    @Override
    public CancelJobResponse cancelJob(CancelJobRequest cancelJobRequest) throws ValidationException,
            IncompatibleVersionException, ResourceNotFoundException, ResourceInUseException, AccessDeniedException,
            InternalServiceException, AwsServiceException, SdkClientException, ElasticTranscoderException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CancelJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CancelJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(cancelJobRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, cancelJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Elastic Transcoder");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CancelJob");

            return clientHandler.execute(new ClientExecutionParams<CancelJobRequest, CancelJobResponse>()
                    .withOperationName("CancelJob").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(cancelJobRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CancelJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * When you create a job, Elastic Transcoder returns JSON data that includes the values that you specified plus
     * information about the job that is created.
     * </p>
     * <p>
     * If you have specified more than one output for your jobs (for example, one output for the Kindle Fire and another
     * output for the Apple iPhone 4s), you currently must use the Elastic Transcoder API to list the jobs (as opposed
     * to the AWS Console).
     * </p>
     *
     * @param createJobRequest
     *        The <code>CreateJobRequest</code> structure.
     * @return Result of the CreateJob operation returned by the service.
     * @throws ValidationException
     *         One or more required parameter values were not provided in the request.
     * @throws IncompatibleVersionException
     * @throws ResourceNotFoundException
     *         The requested resource does not exist or is not available. For example, the pipeline to which you're
     *         trying to add a job doesn't exist or is still being created.
     * @throws AccessDeniedException
     *         General authentication failure. The request was not signed correctly.
     * @throws LimitExceededException
     *         Too many operations for a given AWS account. For example, the number of pipelines exceeds the maximum
     *         allowed.
     * @throws InternalServiceException
     *         Elastic Transcoder encountered an unexpected exception while trying to fulfill the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ElasticTranscoderException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ElasticTranscoderClient.CreateJob
     */
    @Override
    public CreateJobResponse createJob(CreateJobRequest createJobRequest) throws ValidationException,
            IncompatibleVersionException, ResourceNotFoundException, AccessDeniedException, LimitExceededException,
            InternalServiceException, AwsServiceException, SdkClientException, ElasticTranscoderException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createJobRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Elastic Transcoder");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateJob");

            return clientHandler.execute(new ClientExecutionParams<CreateJobRequest, CreateJobResponse>()
                    .withOperationName("CreateJob").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(createJobRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The CreatePipeline operation creates a pipeline with settings that you specify.
     * </p>
     *
     * @param createPipelineRequest
     *        The <code>CreatePipelineRequest</code> structure.
     * @return Result of the CreatePipeline operation returned by the service.
     * @throws ValidationException
     *         One or more required parameter values were not provided in the request.
     * @throws IncompatibleVersionException
     * @throws AccessDeniedException
     *         General authentication failure. The request was not signed correctly.
     * @throws ResourceNotFoundException
     *         The requested resource does not exist or is not available. For example, the pipeline to which you're
     *         trying to add a job doesn't exist or is still being created.
     * @throws LimitExceededException
     *         Too many operations for a given AWS account. For example, the number of pipelines exceeds the maximum
     *         allowed.
     * @throws InternalServiceException
     *         Elastic Transcoder encountered an unexpected exception while trying to fulfill the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ElasticTranscoderException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ElasticTranscoderClient.CreatePipeline
     */
    @Override
    public CreatePipelineResponse createPipeline(CreatePipelineRequest createPipelineRequest) throws ValidationException,
            IncompatibleVersionException, AccessDeniedException, ResourceNotFoundException, LimitExceededException,
            InternalServiceException, AwsServiceException, SdkClientException, ElasticTranscoderException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreatePipelineResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreatePipelineResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createPipelineRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createPipelineRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Elastic Transcoder");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreatePipeline");

            return clientHandler.execute(new ClientExecutionParams<CreatePipelineRequest, CreatePipelineResponse>()
                    .withOperationName("CreatePipeline").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createPipelineRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreatePipelineRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The CreatePreset operation creates a preset with settings that you specify.
     * </p>
     * <important>
     * <p>
     * Elastic Transcoder checks the CreatePreset settings to ensure that they meet Elastic Transcoder requirements and
     * to determine whether they comply with H.264 standards. If your settings are not valid for Elastic Transcoder,
     * Elastic Transcoder returns an HTTP 400 response (<code>ValidationException</code>) and does not create the
     * preset. If the settings are valid for Elastic Transcoder but aren't strictly compliant with the H.264 standard,
     * Elastic Transcoder creates the preset and returns a warning message in the response. This helps you determine
     * whether your settings comply with the H.264 standard while giving you greater flexibility with respect to the
     * video that Elastic Transcoder produces.
     * </p>
     * </important>
     * <p>
     * Elastic Transcoder uses the H.264 video-compression format. For more information, see the International
     * Telecommunication Union publication <i>Recommendation ITU-T H.264: Advanced video coding for generic audiovisual
     * services</i>.
     * </p>
     *
     * @param createPresetRequest
     *        The <code>CreatePresetRequest</code> structure.
     * @return Result of the CreatePreset operation returned by the service.
     * @throws ValidationException
     *         One or more required parameter values were not provided in the request.
     * @throws IncompatibleVersionException
     * @throws AccessDeniedException
     *         General authentication failure. The request was not signed correctly.
     * @throws LimitExceededException
     *         Too many operations for a given AWS account. For example, the number of pipelines exceeds the maximum
     *         allowed.
     * @throws InternalServiceException
     *         Elastic Transcoder encountered an unexpected exception while trying to fulfill the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ElasticTranscoderException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ElasticTranscoderClient.CreatePreset
     */
    @Override
    public CreatePresetResponse createPreset(CreatePresetRequest createPresetRequest) throws ValidationException,
            IncompatibleVersionException, AccessDeniedException, LimitExceededException, InternalServiceException,
            AwsServiceException, SdkClientException, ElasticTranscoderException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreatePresetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreatePresetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createPresetRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createPresetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Elastic Transcoder");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreatePreset");

            return clientHandler.execute(new ClientExecutionParams<CreatePresetRequest, CreatePresetResponse>()
                    .withOperationName("CreatePreset").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createPresetRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreatePresetRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The DeletePipeline operation removes a pipeline.
     * </p>
     * <p>
     * You can only delete a pipeline that has never been used or that is not currently in use (doesn't contain any
     * active jobs). If the pipeline is currently in use, <code>DeletePipeline</code> returns an error.
     * </p>
     *
     * @param deletePipelineRequest
     *        The <code>DeletePipelineRequest</code> structure.
     * @return Result of the DeletePipeline operation returned by the service.
     * @throws ValidationException
     *         One or more required parameter values were not provided in the request.
     * @throws IncompatibleVersionException
     * @throws ResourceNotFoundException
     *         The requested resource does not exist or is not available. For example, the pipeline to which you're
     *         trying to add a job doesn't exist or is still being created.
     * @throws ResourceInUseException
     *         The resource you are attempting to change is in use. For example, you are attempting to delete a pipeline
     *         that is currently in use.
     * @throws AccessDeniedException
     *         General authentication failure. The request was not signed correctly.
     * @throws InternalServiceException
     *         Elastic Transcoder encountered an unexpected exception while trying to fulfill the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ElasticTranscoderException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ElasticTranscoderClient.DeletePipeline
     */
    @Override
    public DeletePipelineResponse deletePipeline(DeletePipelineRequest deletePipelineRequest) throws ValidationException,
            IncompatibleVersionException, ResourceNotFoundException, ResourceInUseException, AccessDeniedException,
            InternalServiceException, AwsServiceException, SdkClientException, ElasticTranscoderException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeletePipelineResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeletePipelineResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deletePipelineRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deletePipelineRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Elastic Transcoder");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeletePipeline");

            return clientHandler.execute(new ClientExecutionParams<DeletePipelineRequest, DeletePipelineResponse>()
                    .withOperationName("DeletePipeline").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deletePipelineRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeletePipelineRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The DeletePreset operation removes a preset that you've added in an AWS region.
     * </p>
     * <note>
     * <p>
     * You can't delete the default presets that are included with Elastic Transcoder.
     * </p>
     * </note>
     *
     * @param deletePresetRequest
     *        The <code>DeletePresetRequest</code> structure.
     * @return Result of the DeletePreset operation returned by the service.
     * @throws ValidationException
     *         One or more required parameter values were not provided in the request.
     * @throws IncompatibleVersionException
     * @throws ResourceNotFoundException
     *         The requested resource does not exist or is not available. For example, the pipeline to which you're
     *         trying to add a job doesn't exist or is still being created.
     * @throws AccessDeniedException
     *         General authentication failure. The request was not signed correctly.
     * @throws InternalServiceException
     *         Elastic Transcoder encountered an unexpected exception while trying to fulfill the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ElasticTranscoderException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ElasticTranscoderClient.DeletePreset
     */
    @Override
    public DeletePresetResponse deletePreset(DeletePresetRequest deletePresetRequest) throws ValidationException,
            IncompatibleVersionException, ResourceNotFoundException, AccessDeniedException, InternalServiceException,
            AwsServiceException, SdkClientException, ElasticTranscoderException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeletePresetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeletePresetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deletePresetRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deletePresetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Elastic Transcoder");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeletePreset");

            return clientHandler.execute(new ClientExecutionParams<DeletePresetRequest, DeletePresetResponse>()
                    .withOperationName("DeletePreset").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deletePresetRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeletePresetRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The ListJobsByPipeline operation gets a list of the jobs currently in a pipeline.
     * </p>
     * <p>
     * Elastic Transcoder returns all of the jobs currently in the specified pipeline. The response body contains one
     * element for each job that satisfies the search criteria.
     * </p>
     *
     * @param listJobsByPipelineRequest
     *        The <code>ListJobsByPipelineRequest</code> structure.
     * @return Result of the ListJobsByPipeline operation returned by the service.
     * @throws ValidationException
     *         One or more required parameter values were not provided in the request.
     * @throws IncompatibleVersionException
     * @throws ResourceNotFoundException
     *         The requested resource does not exist or is not available. For example, the pipeline to which you're
     *         trying to add a job doesn't exist or is still being created.
     * @throws AccessDeniedException
     *         General authentication failure. The request was not signed correctly.
     * @throws InternalServiceException
     *         Elastic Transcoder encountered an unexpected exception while trying to fulfill the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ElasticTranscoderException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ElasticTranscoderClient.ListJobsByPipeline
     */
    @Override
    public ListJobsByPipelineResponse listJobsByPipeline(ListJobsByPipelineRequest listJobsByPipelineRequest)
            throws ValidationException, IncompatibleVersionException, ResourceNotFoundException, AccessDeniedException,
            InternalServiceException, AwsServiceException, SdkClientException, ElasticTranscoderException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListJobsByPipelineResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListJobsByPipelineResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listJobsByPipelineRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listJobsByPipelineRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Elastic Transcoder");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListJobsByPipeline");

            return clientHandler.execute(new ClientExecutionParams<ListJobsByPipelineRequest, ListJobsByPipelineResponse>()
                    .withOperationName("ListJobsByPipeline").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listJobsByPipelineRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListJobsByPipelineRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The ListJobsByStatus operation gets a list of jobs that have a specified status. The response body contains one
     * element for each job that satisfies the search criteria.
     * </p>
     *
     * @param listJobsByStatusRequest
     *        The <code>ListJobsByStatusRequest</code> structure.
     * @return Result of the ListJobsByStatus operation returned by the service.
     * @throws ValidationException
     *         One or more required parameter values were not provided in the request.
     * @throws IncompatibleVersionException
     * @throws ResourceNotFoundException
     *         The requested resource does not exist or is not available. For example, the pipeline to which you're
     *         trying to add a job doesn't exist or is still being created.
     * @throws AccessDeniedException
     *         General authentication failure. The request was not signed correctly.
     * @throws InternalServiceException
     *         Elastic Transcoder encountered an unexpected exception while trying to fulfill the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ElasticTranscoderException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ElasticTranscoderClient.ListJobsByStatus
     */
    @Override
    public ListJobsByStatusResponse listJobsByStatus(ListJobsByStatusRequest listJobsByStatusRequest) throws ValidationException,
            IncompatibleVersionException, ResourceNotFoundException, AccessDeniedException, InternalServiceException,
            AwsServiceException, SdkClientException, ElasticTranscoderException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListJobsByStatusResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListJobsByStatusResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listJobsByStatusRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listJobsByStatusRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Elastic Transcoder");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListJobsByStatus");

            return clientHandler.execute(new ClientExecutionParams<ListJobsByStatusRequest, ListJobsByStatusResponse>()
                    .withOperationName("ListJobsByStatus").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listJobsByStatusRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListJobsByStatusRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The ListPipelines operation gets a list of the pipelines associated with the current AWS account.
     * </p>
     *
     * @param listPipelinesRequest
     *        The <code>ListPipelineRequest</code> structure.
     * @return Result of the ListPipelines operation returned by the service.
     * @throws ValidationException
     *         One or more required parameter values were not provided in the request.
     * @throws IncompatibleVersionException
     * @throws AccessDeniedException
     *         General authentication failure. The request was not signed correctly.
     * @throws InternalServiceException
     *         Elastic Transcoder encountered an unexpected exception while trying to fulfill the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ElasticTranscoderException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ElasticTranscoderClient.ListPipelines
     */
    @Override
    public ListPipelinesResponse listPipelines(ListPipelinesRequest listPipelinesRequest) throws ValidationException,
            IncompatibleVersionException, AccessDeniedException, InternalServiceException, AwsServiceException,
            SdkClientException, ElasticTranscoderException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListPipelinesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListPipelinesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listPipelinesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPipelinesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Elastic Transcoder");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPipelines");

            return clientHandler.execute(new ClientExecutionParams<ListPipelinesRequest, ListPipelinesResponse>()
                    .withOperationName("ListPipelines").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listPipelinesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListPipelinesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The ListPresets operation gets a list of the default presets included with Elastic Transcoder and the presets
     * that you've added in an AWS region.
     * </p>
     *
     * @param listPresetsRequest
     *        The <code>ListPresetsRequest</code> structure.
     * @return Result of the ListPresets operation returned by the service.
     * @throws ValidationException
     *         One or more required parameter values were not provided in the request.
     * @throws IncompatibleVersionException
     * @throws AccessDeniedException
     *         General authentication failure. The request was not signed correctly.
     * @throws InternalServiceException
     *         Elastic Transcoder encountered an unexpected exception while trying to fulfill the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ElasticTranscoderException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ElasticTranscoderClient.ListPresets
     */
    @Override
    public ListPresetsResponse listPresets(ListPresetsRequest listPresetsRequest) throws ValidationException,
            IncompatibleVersionException, AccessDeniedException, InternalServiceException, AwsServiceException,
            SdkClientException, ElasticTranscoderException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListPresetsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListPresetsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listPresetsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPresetsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Elastic Transcoder");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPresets");

            return clientHandler.execute(new ClientExecutionParams<ListPresetsRequest, ListPresetsResponse>()
                    .withOperationName("ListPresets").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(listPresetsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListPresetsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The ReadJob operation returns detailed information about a job.
     * </p>
     *
     * @param readJobRequest
     *        The <code>ReadJobRequest</code> structure.
     * @return Result of the ReadJob operation returned by the service.
     * @throws ValidationException
     *         One or more required parameter values were not provided in the request.
     * @throws IncompatibleVersionException
     * @throws ResourceNotFoundException
     *         The requested resource does not exist or is not available. For example, the pipeline to which you're
     *         trying to add a job doesn't exist or is still being created.
     * @throws AccessDeniedException
     *         General authentication failure. The request was not signed correctly.
     * @throws InternalServiceException
     *         Elastic Transcoder encountered an unexpected exception while trying to fulfill the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ElasticTranscoderException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ElasticTranscoderClient.ReadJob
     */
    @Override
    public ReadJobResponse readJob(ReadJobRequest readJobRequest) throws ValidationException, IncompatibleVersionException,
            ResourceNotFoundException, AccessDeniedException, InternalServiceException, AwsServiceException, SdkClientException,
            ElasticTranscoderException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ReadJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ReadJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(readJobRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, readJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Elastic Transcoder");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ReadJob");

            return clientHandler.execute(new ClientExecutionParams<ReadJobRequest, ReadJobResponse>()
                    .withOperationName("ReadJob").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(readJobRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ReadJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The ReadPipeline operation gets detailed information about a pipeline.
     * </p>
     *
     * @param readPipelineRequest
     *        The <code>ReadPipelineRequest</code> structure.
     * @return Result of the ReadPipeline operation returned by the service.
     * @throws ValidationException
     *         One or more required parameter values were not provided in the request.
     * @throws IncompatibleVersionException
     * @throws ResourceNotFoundException
     *         The requested resource does not exist or is not available. For example, the pipeline to which you're
     *         trying to add a job doesn't exist or is still being created.
     * @throws AccessDeniedException
     *         General authentication failure. The request was not signed correctly.
     * @throws InternalServiceException
     *         Elastic Transcoder encountered an unexpected exception while trying to fulfill the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ElasticTranscoderException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ElasticTranscoderClient.ReadPipeline
     */
    @Override
    public ReadPipelineResponse readPipeline(ReadPipelineRequest readPipelineRequest) throws ValidationException,
            IncompatibleVersionException, ResourceNotFoundException, AccessDeniedException, InternalServiceException,
            AwsServiceException, SdkClientException, ElasticTranscoderException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ReadPipelineResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ReadPipelineResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(readPipelineRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, readPipelineRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Elastic Transcoder");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ReadPipeline");

            return clientHandler.execute(new ClientExecutionParams<ReadPipelineRequest, ReadPipelineResponse>()
                    .withOperationName("ReadPipeline").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(readPipelineRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ReadPipelineRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The ReadPreset operation gets detailed information about a preset.
     * </p>
     *
     * @param readPresetRequest
     *        The <code>ReadPresetRequest</code> structure.
     * @return Result of the ReadPreset operation returned by the service.
     * @throws ValidationException
     *         One or more required parameter values were not provided in the request.
     * @throws IncompatibleVersionException
     * @throws ResourceNotFoundException
     *         The requested resource does not exist or is not available. For example, the pipeline to which you're
     *         trying to add a job doesn't exist or is still being created.
     * @throws AccessDeniedException
     *         General authentication failure. The request was not signed correctly.
     * @throws InternalServiceException
     *         Elastic Transcoder encountered an unexpected exception while trying to fulfill the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ElasticTranscoderException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ElasticTranscoderClient.ReadPreset
     */
    @Override
    public ReadPresetResponse readPreset(ReadPresetRequest readPresetRequest) throws ValidationException,
            IncompatibleVersionException, ResourceNotFoundException, AccessDeniedException, InternalServiceException,
            AwsServiceException, SdkClientException, ElasticTranscoderException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ReadPresetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ReadPresetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(readPresetRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, readPresetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Elastic Transcoder");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ReadPreset");

            return clientHandler.execute(new ClientExecutionParams<ReadPresetRequest, ReadPresetResponse>()
                    .withOperationName("ReadPreset").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(readPresetRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ReadPresetRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Use the <code>UpdatePipeline</code> operation to update settings for a pipeline.
     * </p>
     * <important>
     * <p>
     * When you change pipeline settings, your changes take effect immediately. Jobs that you have already submitted and
     * that Elastic Transcoder has not started to process are affected in addition to jobs that you submit after you
     * change settings.
     * </p>
     * </important>
     *
     * @param updatePipelineRequest
     *        The <code>UpdatePipelineRequest</code> structure.
     * @return Result of the UpdatePipeline operation returned by the service.
     * @throws ValidationException
     *         One or more required parameter values were not provided in the request.
     * @throws IncompatibleVersionException
     * @throws AccessDeniedException
     *         General authentication failure. The request was not signed correctly.
     * @throws ResourceInUseException
     *         The resource you are attempting to change is in use. For example, you are attempting to delete a pipeline
     *         that is currently in use.
     * @throws ResourceNotFoundException
     *         The requested resource does not exist or is not available. For example, the pipeline to which you're
     *         trying to add a job doesn't exist or is still being created.
     * @throws InternalServiceException
     *         Elastic Transcoder encountered an unexpected exception while trying to fulfill the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ElasticTranscoderException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ElasticTranscoderClient.UpdatePipeline
     */
    @Override
    public UpdatePipelineResponse updatePipeline(UpdatePipelineRequest updatePipelineRequest) throws ValidationException,
            IncompatibleVersionException, AccessDeniedException, ResourceInUseException, ResourceNotFoundException,
            InternalServiceException, AwsServiceException, SdkClientException, ElasticTranscoderException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdatePipelineResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdatePipelineResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updatePipelineRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updatePipelineRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Elastic Transcoder");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdatePipeline");

            return clientHandler.execute(new ClientExecutionParams<UpdatePipelineRequest, UpdatePipelineResponse>()
                    .withOperationName("UpdatePipeline").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updatePipelineRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdatePipelineRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * With the UpdatePipelineNotifications operation, you can update Amazon Simple Notification Service (Amazon SNS)
     * notifications for a pipeline.
     * </p>
     * <p>
     * When you update notifications for a pipeline, Elastic Transcoder returns the values that you specified in the
     * request.
     * </p>
     *
     * @param updatePipelineNotificationsRequest
     *        The <code>UpdatePipelineNotificationsRequest</code> structure.
     * @return Result of the UpdatePipelineNotifications operation returned by the service.
     * @throws ValidationException
     *         One or more required parameter values were not provided in the request.
     * @throws IncompatibleVersionException
     * @throws ResourceNotFoundException
     *         The requested resource does not exist or is not available. For example, the pipeline to which you're
     *         trying to add a job doesn't exist or is still being created.
     * @throws ResourceInUseException
     *         The resource you are attempting to change is in use. For example, you are attempting to delete a pipeline
     *         that is currently in use.
     * @throws AccessDeniedException
     *         General authentication failure. The request was not signed correctly.
     * @throws InternalServiceException
     *         Elastic Transcoder encountered an unexpected exception while trying to fulfill the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ElasticTranscoderException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ElasticTranscoderClient.UpdatePipelineNotifications
     */
    @Override
    public UpdatePipelineNotificationsResponse updatePipelineNotifications(
            UpdatePipelineNotificationsRequest updatePipelineNotificationsRequest) throws ValidationException,
            IncompatibleVersionException, ResourceNotFoundException, ResourceInUseException, AccessDeniedException,
            InternalServiceException, AwsServiceException, SdkClientException, ElasticTranscoderException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdatePipelineNotificationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdatePipelineNotificationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updatePipelineNotificationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updatePipelineNotificationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Elastic Transcoder");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdatePipelineNotifications");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdatePipelineNotificationsRequest, UpdatePipelineNotificationsResponse>()
                            .withOperationName("UpdatePipelineNotifications").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(updatePipelineNotificationsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdatePipelineNotificationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The UpdatePipelineStatus operation pauses or reactivates a pipeline, so that the pipeline stops or restarts the
     * processing of jobs.
     * </p>
     * <p>
     * Changing the pipeline status is useful if you want to cancel one or more jobs. You can't cancel jobs after
     * Elastic Transcoder has started processing them; if you pause the pipeline to which you submitted the jobs, you
     * have more time to get the job IDs for the jobs that you want to cancel, and to send a <a>CancelJob</a> request.
     * </p>
     *
     * @param updatePipelineStatusRequest
     *        The <code>UpdatePipelineStatusRequest</code> structure.
     * @return Result of the UpdatePipelineStatus operation returned by the service.
     * @throws ValidationException
     *         One or more required parameter values were not provided in the request.
     * @throws IncompatibleVersionException
     * @throws ResourceNotFoundException
     *         The requested resource does not exist or is not available. For example, the pipeline to which you're
     *         trying to add a job doesn't exist or is still being created.
     * @throws ResourceInUseException
     *         The resource you are attempting to change is in use. For example, you are attempting to delete a pipeline
     *         that is currently in use.
     * @throws AccessDeniedException
     *         General authentication failure. The request was not signed correctly.
     * @throws InternalServiceException
     *         Elastic Transcoder encountered an unexpected exception while trying to fulfill the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ElasticTranscoderException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ElasticTranscoderClient.UpdatePipelineStatus
     */
    @Override
    public UpdatePipelineStatusResponse updatePipelineStatus(UpdatePipelineStatusRequest updatePipelineStatusRequest)
            throws ValidationException, IncompatibleVersionException, ResourceNotFoundException, ResourceInUseException,
            AccessDeniedException, InternalServiceException, AwsServiceException, SdkClientException, ElasticTranscoderException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdatePipelineStatusResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdatePipelineStatusResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updatePipelineStatusRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updatePipelineStatusRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Elastic Transcoder");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdatePipelineStatus");

            return clientHandler.execute(new ClientExecutionParams<UpdatePipelineStatusRequest, UpdatePipelineStatusResponse>()
                    .withOperationName("UpdatePipelineStatus").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updatePipelineStatusRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdatePipelineStatusRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Create an instance of {@link ElasticTranscoderWaiter} using this client.
     * <p>
     * Waiters created via this method are managed by the SDK and resources will be released when the service client is
     * closed.
     *
     * @return an instance of {@link ElasticTranscoderWaiter}
     */
    @Override
    public ElasticTranscoderWaiter waiter() {
        return ElasticTranscoderWaiter.builder().client(this).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        ElasticTranscoderServiceClientConfigurationBuilder serviceConfigBuilder = new ElasticTranscoderServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(ElasticTranscoderException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceInUseException")
                                .exceptionBuilderSupplier(ResourceInUseException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("IncompatibleVersionException")
                                .exceptionBuilderSupplier(IncompatibleVersionException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServiceException")
                                .exceptionBuilderSupplier(InternalServiceException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(429).build());
    }

    @Override
    public final ElasticTranscoderServiceClientConfiguration serviceClientConfiguration() {
        return new ElasticTranscoderServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
