/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.emrcontainers.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * This entity describes a job run. A job run is a unit of work, such as a Spark jar, PySpark script, or SparkSQL query,
 * that you submit to Amazon EMR on EKS.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class JobRun implements SdkPojo, Serializable, ToCopyableBuilder<JobRun.Builder, JobRun> {
    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("id")
            .getter(getter(JobRun::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("id").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(JobRun::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> VIRTUAL_CLUSTER_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("virtualClusterId").getter(getter(JobRun::virtualClusterId)).setter(setter(Builder::virtualClusterId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("virtualClusterId").build()).build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("arn")
            .getter(getter(JobRun::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("arn").build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("state")
            .getter(getter(JobRun::stateAsString)).setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("state").build()).build();

    private static final SdkField<String> CLIENT_TOKEN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("clientToken").getter(getter(JobRun::clientToken)).setter(setter(Builder::clientToken))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("clientToken").build()).build();

    private static final SdkField<String> EXECUTION_ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("executionRoleArn").getter(getter(JobRun::executionRoleArn)).setter(setter(Builder::executionRoleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("executionRoleArn").build()).build();

    private static final SdkField<String> RELEASE_LABEL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("releaseLabel").getter(getter(JobRun::releaseLabel)).setter(setter(Builder::releaseLabel))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("releaseLabel").build()).build();

    private static final SdkField<ConfigurationOverrides> CONFIGURATION_OVERRIDES_FIELD = SdkField
            .<ConfigurationOverrides> builder(MarshallingType.SDK_POJO).memberName("configurationOverrides")
            .getter(getter(JobRun::configurationOverrides)).setter(setter(Builder::configurationOverrides))
            .constructor(ConfigurationOverrides::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("configurationOverrides").build())
            .build();

    private static final SdkField<JobDriver> JOB_DRIVER_FIELD = SdkField.<JobDriver> builder(MarshallingType.SDK_POJO)
            .memberName("jobDriver").getter(getter(JobRun::jobDriver)).setter(setter(Builder::jobDriver))
            .constructor(JobDriver::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("jobDriver").build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("createdAt")
            .getter(getter(JobRun::createdAt))
            .setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<String> CREATED_BY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("createdBy").getter(getter(JobRun::createdBy)).setter(setter(Builder::createdBy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdBy").build()).build();

    private static final SdkField<Instant> FINISHED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("finishedAt")
            .getter(getter(JobRun::finishedAt))
            .setter(setter(Builder::finishedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("finishedAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<String> STATE_DETAILS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("stateDetails").getter(getter(JobRun::stateDetails)).setter(setter(Builder::stateDetails))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("stateDetails").build()).build();

    private static final SdkField<String> FAILURE_REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("failureReason").getter(getter(JobRun::failureReasonAsString)).setter(setter(Builder::failureReason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("failureReason").build()).build();

    private static final SdkField<Map<String, String>> TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("tags")
            .getter(getter(JobRun::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<RetryPolicyConfiguration> RETRY_POLICY_CONFIGURATION_FIELD = SdkField
            .<RetryPolicyConfiguration> builder(MarshallingType.SDK_POJO).memberName("retryPolicyConfiguration")
            .getter(getter(JobRun::retryPolicyConfiguration)).setter(setter(Builder::retryPolicyConfiguration))
            .constructor(RetryPolicyConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("retryPolicyConfiguration").build())
            .build();

    private static final SdkField<RetryPolicyExecution> RETRY_POLICY_EXECUTION_FIELD = SdkField
            .<RetryPolicyExecution> builder(MarshallingType.SDK_POJO).memberName("retryPolicyExecution")
            .getter(getter(JobRun::retryPolicyExecution)).setter(setter(Builder::retryPolicyExecution))
            .constructor(RetryPolicyExecution::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("retryPolicyExecution").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays
            .asList(ID_FIELD, NAME_FIELD, VIRTUAL_CLUSTER_ID_FIELD, ARN_FIELD, STATE_FIELD, CLIENT_TOKEN_FIELD,
                    EXECUTION_ROLE_ARN_FIELD, RELEASE_LABEL_FIELD, CONFIGURATION_OVERRIDES_FIELD, JOB_DRIVER_FIELD,
                    CREATED_AT_FIELD, CREATED_BY_FIELD, FINISHED_AT_FIELD, STATE_DETAILS_FIELD, FAILURE_REASON_FIELD, TAGS_FIELD,
                    RETRY_POLICY_CONFIGURATION_FIELD, RETRY_POLICY_EXECUTION_FIELD));

    private static final long serialVersionUID = 1L;

    private final String id;

    private final String name;

    private final String virtualClusterId;

    private final String arn;

    private final String state;

    private final String clientToken;

    private final String executionRoleArn;

    private final String releaseLabel;

    private final ConfigurationOverrides configurationOverrides;

    private final JobDriver jobDriver;

    private final Instant createdAt;

    private final String createdBy;

    private final Instant finishedAt;

    private final String stateDetails;

    private final String failureReason;

    private final Map<String, String> tags;

    private final RetryPolicyConfiguration retryPolicyConfiguration;

    private final RetryPolicyExecution retryPolicyExecution;

    private JobRun(BuilderImpl builder) {
        this.id = builder.id;
        this.name = builder.name;
        this.virtualClusterId = builder.virtualClusterId;
        this.arn = builder.arn;
        this.state = builder.state;
        this.clientToken = builder.clientToken;
        this.executionRoleArn = builder.executionRoleArn;
        this.releaseLabel = builder.releaseLabel;
        this.configurationOverrides = builder.configurationOverrides;
        this.jobDriver = builder.jobDriver;
        this.createdAt = builder.createdAt;
        this.createdBy = builder.createdBy;
        this.finishedAt = builder.finishedAt;
        this.stateDetails = builder.stateDetails;
        this.failureReason = builder.failureReason;
        this.tags = builder.tags;
        this.retryPolicyConfiguration = builder.retryPolicyConfiguration;
        this.retryPolicyExecution = builder.retryPolicyExecution;
    }

    /**
     * <p>
     * The ID of the job run.
     * </p>
     * 
     * @return The ID of the job run.
     */
    public final String id() {
        return id;
    }

    /**
     * <p>
     * The name of the job run.
     * </p>
     * 
     * @return The name of the job run.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The ID of the job run's virtual cluster.
     * </p>
     * 
     * @return The ID of the job run's virtual cluster.
     */
    public final String virtualClusterId() {
        return virtualClusterId;
    }

    /**
     * <p>
     * The ARN of job run.
     * </p>
     * 
     * @return The ARN of job run.
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * The state of the job run.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link JobRunState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The state of the job run.
     * @see JobRunState
     */
    public final JobRunState state() {
        return JobRunState.fromValue(state);
    }

    /**
     * <p>
     * The state of the job run.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link JobRunState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The state of the job run.
     * @see JobRunState
     */
    public final String stateAsString() {
        return state;
    }

    /**
     * <p>
     * The client token used to start a job run.
     * </p>
     * 
     * @return The client token used to start a job run.
     */
    public final String clientToken() {
        return clientToken;
    }

    /**
     * <p>
     * The execution role ARN of the job run.
     * </p>
     * 
     * @return The execution role ARN of the job run.
     */
    public final String executionRoleArn() {
        return executionRoleArn;
    }

    /**
     * <p>
     * The release version of Amazon EMR.
     * </p>
     * 
     * @return The release version of Amazon EMR.
     */
    public final String releaseLabel() {
        return releaseLabel;
    }

    /**
     * <p>
     * The configuration settings that are used to override default configuration.
     * </p>
     * 
     * @return The configuration settings that are used to override default configuration.
     */
    public final ConfigurationOverrides configurationOverrides() {
        return configurationOverrides;
    }

    /**
     * <p>
     * Parameters of job driver for the job run.
     * </p>
     * 
     * @return Parameters of job driver for the job run.
     */
    public final JobDriver jobDriver() {
        return jobDriver;
    }

    /**
     * <p>
     * The date and time when the job run was created.
     * </p>
     * 
     * @return The date and time when the job run was created.
     */
    public final Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * The user who created the job run.
     * </p>
     * 
     * @return The user who created the job run.
     */
    public final String createdBy() {
        return createdBy;
    }

    /**
     * <p>
     * The date and time when the job run has finished.
     * </p>
     * 
     * @return The date and time when the job run has finished.
     */
    public final Instant finishedAt() {
        return finishedAt;
    }

    /**
     * <p>
     * Additional details of the job run state.
     * </p>
     * 
     * @return Additional details of the job run state.
     */
    public final String stateDetails() {
        return stateDetails;
    }

    /**
     * <p>
     * The reasons why the job run has failed.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #failureReason}
     * will return {@link FailureReason#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #failureReasonAsString}.
     * </p>
     * 
     * @return The reasons why the job run has failed.
     * @see FailureReason
     */
    public final FailureReason failureReason() {
        return FailureReason.fromValue(failureReason);
    }

    /**
     * <p>
     * The reasons why the job run has failed.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #failureReason}
     * will return {@link FailureReason#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #failureReasonAsString}.
     * </p>
     * 
     * @return The reasons why the job run has failed.
     * @see FailureReason
     */
    public final String failureReasonAsString() {
        return failureReason;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The assigned tags of the job run.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The assigned tags of the job run.
     */
    public final Map<String, String> tags() {
        return tags;
    }

    /**
     * <p>
     * The configuration of the retry policy that the job runs on.
     * </p>
     * 
     * @return The configuration of the retry policy that the job runs on.
     */
    public final RetryPolicyConfiguration retryPolicyConfiguration() {
        return retryPolicyConfiguration;
    }

    /**
     * <p>
     * The current status of the retry policy executed on the job.
     * </p>
     * 
     * @return The current status of the retry policy executed on the job.
     */
    public final RetryPolicyExecution retryPolicyExecution() {
        return retryPolicyExecution;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(virtualClusterId());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(clientToken());
        hashCode = 31 * hashCode + Objects.hashCode(executionRoleArn());
        hashCode = 31 * hashCode + Objects.hashCode(releaseLabel());
        hashCode = 31 * hashCode + Objects.hashCode(configurationOverrides());
        hashCode = 31 * hashCode + Objects.hashCode(jobDriver());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(createdBy());
        hashCode = 31 * hashCode + Objects.hashCode(finishedAt());
        hashCode = 31 * hashCode + Objects.hashCode(stateDetails());
        hashCode = 31 * hashCode + Objects.hashCode(failureReasonAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(retryPolicyConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(retryPolicyExecution());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof JobRun)) {
            return false;
        }
        JobRun other = (JobRun) obj;
        return Objects.equals(id(), other.id()) && Objects.equals(name(), other.name())
                && Objects.equals(virtualClusterId(), other.virtualClusterId()) && Objects.equals(arn(), other.arn())
                && Objects.equals(stateAsString(), other.stateAsString()) && Objects.equals(clientToken(), other.clientToken())
                && Objects.equals(executionRoleArn(), other.executionRoleArn())
                && Objects.equals(releaseLabel(), other.releaseLabel())
                && Objects.equals(configurationOverrides(), other.configurationOverrides())
                && Objects.equals(jobDriver(), other.jobDriver()) && Objects.equals(createdAt(), other.createdAt())
                && Objects.equals(createdBy(), other.createdBy()) && Objects.equals(finishedAt(), other.finishedAt())
                && Objects.equals(stateDetails(), other.stateDetails())
                && Objects.equals(failureReasonAsString(), other.failureReasonAsString()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags())
                && Objects.equals(retryPolicyConfiguration(), other.retryPolicyConfiguration())
                && Objects.equals(retryPolicyExecution(), other.retryPolicyExecution());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("JobRun").add("Id", id()).add("Name", name()).add("VirtualClusterId", virtualClusterId())
                .add("Arn", arn()).add("State", stateAsString()).add("ClientToken", clientToken())
                .add("ExecutionRoleArn", executionRoleArn()).add("ReleaseLabel", releaseLabel())
                .add("ConfigurationOverrides", configurationOverrides()).add("JobDriver", jobDriver())
                .add("CreatedAt", createdAt()).add("CreatedBy", createdBy()).add("FinishedAt", finishedAt())
                .add("StateDetails", stateDetails()).add("FailureReason", failureReasonAsString())
                .add("Tags", hasTags() ? tags() : null).add("RetryPolicyConfiguration", retryPolicyConfiguration())
                .add("RetryPolicyExecution", retryPolicyExecution()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "id":
            return Optional.ofNullable(clazz.cast(id()));
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "virtualClusterId":
            return Optional.ofNullable(clazz.cast(virtualClusterId()));
        case "arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "state":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "clientToken":
            return Optional.ofNullable(clazz.cast(clientToken()));
        case "executionRoleArn":
            return Optional.ofNullable(clazz.cast(executionRoleArn()));
        case "releaseLabel":
            return Optional.ofNullable(clazz.cast(releaseLabel()));
        case "configurationOverrides":
            return Optional.ofNullable(clazz.cast(configurationOverrides()));
        case "jobDriver":
            return Optional.ofNullable(clazz.cast(jobDriver()));
        case "createdAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "createdBy":
            return Optional.ofNullable(clazz.cast(createdBy()));
        case "finishedAt":
            return Optional.ofNullable(clazz.cast(finishedAt()));
        case "stateDetails":
            return Optional.ofNullable(clazz.cast(stateDetails()));
        case "failureReason":
            return Optional.ofNullable(clazz.cast(failureReasonAsString()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "retryPolicyConfiguration":
            return Optional.ofNullable(clazz.cast(retryPolicyConfiguration()));
        case "retryPolicyExecution":
            return Optional.ofNullable(clazz.cast(retryPolicyExecution()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<JobRun, T> g) {
        return obj -> g.apply((JobRun) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, JobRun> {
        /**
         * <p>
         * The ID of the job run.
         * </p>
         * 
         * @param id
         *        The ID of the job run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The name of the job run.
         * </p>
         * 
         * @param name
         *        The name of the job run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The ID of the job run's virtual cluster.
         * </p>
         * 
         * @param virtualClusterId
         *        The ID of the job run's virtual cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder virtualClusterId(String virtualClusterId);

        /**
         * <p>
         * The ARN of job run.
         * </p>
         * 
         * @param arn
         *        The ARN of job run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The state of the job run.
         * </p>
         * 
         * @param state
         *        The state of the job run.
         * @see JobRunState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see JobRunState
         */
        Builder state(String state);

        /**
         * <p>
         * The state of the job run.
         * </p>
         * 
         * @param state
         *        The state of the job run.
         * @see JobRunState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see JobRunState
         */
        Builder state(JobRunState state);

        /**
         * <p>
         * The client token used to start a job run.
         * </p>
         * 
         * @param clientToken
         *        The client token used to start a job run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientToken(String clientToken);

        /**
         * <p>
         * The execution role ARN of the job run.
         * </p>
         * 
         * @param executionRoleArn
         *        The execution role ARN of the job run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder executionRoleArn(String executionRoleArn);

        /**
         * <p>
         * The release version of Amazon EMR.
         * </p>
         * 
         * @param releaseLabel
         *        The release version of Amazon EMR.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder releaseLabel(String releaseLabel);

        /**
         * <p>
         * The configuration settings that are used to override default configuration.
         * </p>
         * 
         * @param configurationOverrides
         *        The configuration settings that are used to override default configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configurationOverrides(ConfigurationOverrides configurationOverrides);

        /**
         * <p>
         * The configuration settings that are used to override default configuration.
         * </p>
         * This is a convenience method that creates an instance of the {@link ConfigurationOverrides.Builder} avoiding
         * the need to create one manually via {@link ConfigurationOverrides#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ConfigurationOverrides.Builder#build()} is called immediately and
         * its result is passed to {@link #configurationOverrides(ConfigurationOverrides)}.
         * 
         * @param configurationOverrides
         *        a consumer that will call methods on {@link ConfigurationOverrides.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #configurationOverrides(ConfigurationOverrides)
         */
        default Builder configurationOverrides(Consumer<ConfigurationOverrides.Builder> configurationOverrides) {
            return configurationOverrides(ConfigurationOverrides.builder().applyMutation(configurationOverrides).build());
        }

        /**
         * <p>
         * Parameters of job driver for the job run.
         * </p>
         * 
         * @param jobDriver
         *        Parameters of job driver for the job run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jobDriver(JobDriver jobDriver);

        /**
         * <p>
         * Parameters of job driver for the job run.
         * </p>
         * This is a convenience method that creates an instance of the {@link JobDriver.Builder} avoiding the need to
         * create one manually via {@link JobDriver#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link JobDriver.Builder#build()} is called immediately and its result
         * is passed to {@link #jobDriver(JobDriver)}.
         * 
         * @param jobDriver
         *        a consumer that will call methods on {@link JobDriver.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #jobDriver(JobDriver)
         */
        default Builder jobDriver(Consumer<JobDriver.Builder> jobDriver) {
            return jobDriver(JobDriver.builder().applyMutation(jobDriver).build());
        }

        /**
         * <p>
         * The date and time when the job run was created.
         * </p>
         * 
         * @param createdAt
         *        The date and time when the job run was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * The user who created the job run.
         * </p>
         * 
         * @param createdBy
         *        The user who created the job run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdBy(String createdBy);

        /**
         * <p>
         * The date and time when the job run has finished.
         * </p>
         * 
         * @param finishedAt
         *        The date and time when the job run has finished.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder finishedAt(Instant finishedAt);

        /**
         * <p>
         * Additional details of the job run state.
         * </p>
         * 
         * @param stateDetails
         *        Additional details of the job run state.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stateDetails(String stateDetails);

        /**
         * <p>
         * The reasons why the job run has failed.
         * </p>
         * 
         * @param failureReason
         *        The reasons why the job run has failed.
         * @see FailureReason
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FailureReason
         */
        Builder failureReason(String failureReason);

        /**
         * <p>
         * The reasons why the job run has failed.
         * </p>
         * 
         * @param failureReason
         *        The reasons why the job run has failed.
         * @see FailureReason
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FailureReason
         */
        Builder failureReason(FailureReason failureReason);

        /**
         * <p>
         * The assigned tags of the job run.
         * </p>
         * 
         * @param tags
         *        The assigned tags of the job run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Map<String, String> tags);

        /**
         * <p>
         * The configuration of the retry policy that the job runs on.
         * </p>
         * 
         * @param retryPolicyConfiguration
         *        The configuration of the retry policy that the job runs on.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder retryPolicyConfiguration(RetryPolicyConfiguration retryPolicyConfiguration);

        /**
         * <p>
         * The configuration of the retry policy that the job runs on.
         * </p>
         * This is a convenience method that creates an instance of the {@link RetryPolicyConfiguration.Builder}
         * avoiding the need to create one manually via {@link RetryPolicyConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link RetryPolicyConfiguration.Builder#build()} is called immediately
         * and its result is passed to {@link #retryPolicyConfiguration(RetryPolicyConfiguration)}.
         * 
         * @param retryPolicyConfiguration
         *        a consumer that will call methods on {@link RetryPolicyConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #retryPolicyConfiguration(RetryPolicyConfiguration)
         */
        default Builder retryPolicyConfiguration(Consumer<RetryPolicyConfiguration.Builder> retryPolicyConfiguration) {
            return retryPolicyConfiguration(RetryPolicyConfiguration.builder().applyMutation(retryPolicyConfiguration).build());
        }

        /**
         * <p>
         * The current status of the retry policy executed on the job.
         * </p>
         * 
         * @param retryPolicyExecution
         *        The current status of the retry policy executed on the job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder retryPolicyExecution(RetryPolicyExecution retryPolicyExecution);

        /**
         * <p>
         * The current status of the retry policy executed on the job.
         * </p>
         * This is a convenience method that creates an instance of the {@link RetryPolicyExecution.Builder} avoiding
         * the need to create one manually via {@link RetryPolicyExecution#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link RetryPolicyExecution.Builder#build()} is called immediately and
         * its result is passed to {@link #retryPolicyExecution(RetryPolicyExecution)}.
         * 
         * @param retryPolicyExecution
         *        a consumer that will call methods on {@link RetryPolicyExecution.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #retryPolicyExecution(RetryPolicyExecution)
         */
        default Builder retryPolicyExecution(Consumer<RetryPolicyExecution.Builder> retryPolicyExecution) {
            return retryPolicyExecution(RetryPolicyExecution.builder().applyMutation(retryPolicyExecution).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String id;

        private String name;

        private String virtualClusterId;

        private String arn;

        private String state;

        private String clientToken;

        private String executionRoleArn;

        private String releaseLabel;

        private ConfigurationOverrides configurationOverrides;

        private JobDriver jobDriver;

        private Instant createdAt;

        private String createdBy;

        private Instant finishedAt;

        private String stateDetails;

        private String failureReason;

        private Map<String, String> tags = DefaultSdkAutoConstructMap.getInstance();

        private RetryPolicyConfiguration retryPolicyConfiguration;

        private RetryPolicyExecution retryPolicyExecution;

        private BuilderImpl() {
        }

        private BuilderImpl(JobRun model) {
            id(model.id);
            name(model.name);
            virtualClusterId(model.virtualClusterId);
            arn(model.arn);
            state(model.state);
            clientToken(model.clientToken);
            executionRoleArn(model.executionRoleArn);
            releaseLabel(model.releaseLabel);
            configurationOverrides(model.configurationOverrides);
            jobDriver(model.jobDriver);
            createdAt(model.createdAt);
            createdBy(model.createdBy);
            finishedAt(model.finishedAt);
            stateDetails(model.stateDetails);
            failureReason(model.failureReason);
            tags(model.tags);
            retryPolicyConfiguration(model.retryPolicyConfiguration);
            retryPolicyExecution(model.retryPolicyExecution);
        }

        public final String getId() {
            return id;
        }

        public final void setId(String id) {
            this.id = id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getVirtualClusterId() {
            return virtualClusterId;
        }

        public final void setVirtualClusterId(String virtualClusterId) {
            this.virtualClusterId = virtualClusterId;
        }

        @Override
        public final Builder virtualClusterId(String virtualClusterId) {
            this.virtualClusterId = virtualClusterId;
            return this;
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final String getState() {
            return state;
        }

        public final void setState(String state) {
            this.state = state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        public final Builder state(JobRunState state) {
            this.state(state == null ? null : state.toString());
            return this;
        }

        public final String getClientToken() {
            return clientToken;
        }

        public final void setClientToken(String clientToken) {
            this.clientToken = clientToken;
        }

        @Override
        public final Builder clientToken(String clientToken) {
            this.clientToken = clientToken;
            return this;
        }

        public final String getExecutionRoleArn() {
            return executionRoleArn;
        }

        public final void setExecutionRoleArn(String executionRoleArn) {
            this.executionRoleArn = executionRoleArn;
        }

        @Override
        public final Builder executionRoleArn(String executionRoleArn) {
            this.executionRoleArn = executionRoleArn;
            return this;
        }

        public final String getReleaseLabel() {
            return releaseLabel;
        }

        public final void setReleaseLabel(String releaseLabel) {
            this.releaseLabel = releaseLabel;
        }

        @Override
        public final Builder releaseLabel(String releaseLabel) {
            this.releaseLabel = releaseLabel;
            return this;
        }

        public final ConfigurationOverrides.Builder getConfigurationOverrides() {
            return configurationOverrides != null ? configurationOverrides.toBuilder() : null;
        }

        public final void setConfigurationOverrides(ConfigurationOverrides.BuilderImpl configurationOverrides) {
            this.configurationOverrides = configurationOverrides != null ? configurationOverrides.build() : null;
        }

        @Override
        public final Builder configurationOverrides(ConfigurationOverrides configurationOverrides) {
            this.configurationOverrides = configurationOverrides;
            return this;
        }

        public final JobDriver.Builder getJobDriver() {
            return jobDriver != null ? jobDriver.toBuilder() : null;
        }

        public final void setJobDriver(JobDriver.BuilderImpl jobDriver) {
            this.jobDriver = jobDriver != null ? jobDriver.build() : null;
        }

        @Override
        public final Builder jobDriver(JobDriver jobDriver) {
            this.jobDriver = jobDriver;
            return this;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final String getCreatedBy() {
            return createdBy;
        }

        public final void setCreatedBy(String createdBy) {
            this.createdBy = createdBy;
        }

        @Override
        public final Builder createdBy(String createdBy) {
            this.createdBy = createdBy;
            return this;
        }

        public final Instant getFinishedAt() {
            return finishedAt;
        }

        public final void setFinishedAt(Instant finishedAt) {
            this.finishedAt = finishedAt;
        }

        @Override
        public final Builder finishedAt(Instant finishedAt) {
            this.finishedAt = finishedAt;
            return this;
        }

        public final String getStateDetails() {
            return stateDetails;
        }

        public final void setStateDetails(String stateDetails) {
            this.stateDetails = stateDetails;
        }

        @Override
        public final Builder stateDetails(String stateDetails) {
            this.stateDetails = stateDetails;
            return this;
        }

        public final String getFailureReason() {
            return failureReason;
        }

        public final void setFailureReason(String failureReason) {
            this.failureReason = failureReason;
        }

        @Override
        public final Builder failureReason(String failureReason) {
            this.failureReason = failureReason;
            return this;
        }

        @Override
        public final Builder failureReason(FailureReason failureReason) {
            this.failureReason(failureReason == null ? null : failureReason.toString());
            return this;
        }

        public final Map<String, String> getTags() {
            if (tags instanceof SdkAutoConstructMap) {
                return null;
            }
            return tags;
        }

        public final void setTags(Map<String, String> tags) {
            this.tags = TagMapCopier.copy(tags);
        }

        @Override
        public final Builder tags(Map<String, String> tags) {
            this.tags = TagMapCopier.copy(tags);
            return this;
        }

        public final RetryPolicyConfiguration.Builder getRetryPolicyConfiguration() {
            return retryPolicyConfiguration != null ? retryPolicyConfiguration.toBuilder() : null;
        }

        public final void setRetryPolicyConfiguration(RetryPolicyConfiguration.BuilderImpl retryPolicyConfiguration) {
            this.retryPolicyConfiguration = retryPolicyConfiguration != null ? retryPolicyConfiguration.build() : null;
        }

        @Override
        public final Builder retryPolicyConfiguration(RetryPolicyConfiguration retryPolicyConfiguration) {
            this.retryPolicyConfiguration = retryPolicyConfiguration;
            return this;
        }

        public final RetryPolicyExecution.Builder getRetryPolicyExecution() {
            return retryPolicyExecution != null ? retryPolicyExecution.toBuilder() : null;
        }

        public final void setRetryPolicyExecution(RetryPolicyExecution.BuilderImpl retryPolicyExecution) {
            this.retryPolicyExecution = retryPolicyExecution != null ? retryPolicyExecution.build() : null;
        }

        @Override
        public final Builder retryPolicyExecution(RetryPolicyExecution retryPolicyExecution) {
            this.retryPolicyExecution = retryPolicyExecution;
            return this;
        }

        @Override
        public JobRun build() {
            return new JobRun(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
