/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.fsx.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Provides a report detailing the data repository task results of the files processed that match the criteria specified
 * in the report <code>Scope</code> parameter. FSx delivers the report to the file system's linked data repository in
 * Amazon S3, using the path specified in the report <code>Path</code> parameter. You can specify whether or not a
 * report gets generated for a task using the <code>Enabled</code> parameter.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CompletionReport implements SdkPojo, Serializable,
        ToCopyableBuilder<CompletionReport.Builder, CompletionReport> {
    private static final SdkField<Boolean> ENABLED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("Enabled").getter(getter(CompletionReport::enabled)).setter(setter(Builder::enabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Enabled").build()).build();

    private static final SdkField<String> PATH_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Path")
            .getter(getter(CompletionReport::path)).setter(setter(Builder::path))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Path").build()).build();

    private static final SdkField<String> FORMAT_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Format")
            .getter(getter(CompletionReport::formatAsString)).setter(setter(Builder::format))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Format").build()).build();

    private static final SdkField<String> SCOPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Scope")
            .getter(getter(CompletionReport::scopeAsString)).setter(setter(Builder::scope))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Scope").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ENABLED_FIELD, PATH_FIELD,
            FORMAT_FIELD, SCOPE_FIELD));

    private static final long serialVersionUID = 1L;

    private final Boolean enabled;

    private final String path;

    private final String format;

    private final String scope;

    private CompletionReport(BuilderImpl builder) {
        this.enabled = builder.enabled;
        this.path = builder.path;
        this.format = builder.format;
        this.scope = builder.scope;
    }

    /**
     * <p>
     * Set <code>Enabled</code> to <code>True</code> to generate a <code>CompletionReport</code> when the task
     * completes. If set to <code>true</code>, then you need to provide a report <code>Scope</code>, <code>Path</code>,
     * and <code>Format</code>. Set <code>Enabled</code> to <code>False</code> if you do not want a
     * <code>CompletionReport</code> generated when the task completes.
     * </p>
     * 
     * @return Set <code>Enabled</code> to <code>True</code> to generate a <code>CompletionReport</code> when the task
     *         completes. If set to <code>true</code>, then you need to provide a report <code>Scope</code>,
     *         <code>Path</code>, and <code>Format</code>. Set <code>Enabled</code> to <code>False</code> if you do not
     *         want a <code>CompletionReport</code> generated when the task completes.
     */
    public final Boolean enabled() {
        return enabled;
    }

    /**
     * <p>
     * Required if <code>Enabled</code> is set to <code>true</code>. Specifies the location of the report on the file
     * system's linked S3 data repository. An absolute path that defines where the completion report will be stored in
     * the destination location. The <code>Path</code> you provide must be located within the file system’s ExportPath.
     * An example <code>Path</code> value is "s3://myBucket/myExportPath/optionalPrefix". The report provides the
     * following information for each file in the report: FilePath, FileStatus, and ErrorCode. To learn more about a
     * file system's <code>ExportPath</code>, see .
     * </p>
     * 
     * @return Required if <code>Enabled</code> is set to <code>true</code>. Specifies the location of the report on the
     *         file system's linked S3 data repository. An absolute path that defines where the completion report will
     *         be stored in the destination location. The <code>Path</code> you provide must be located within the file
     *         system’s ExportPath. An example <code>Path</code> value is "s3://myBucket/myExportPath/optionalPrefix".
     *         The report provides the following information for each file in the report: FilePath, FileStatus, and
     *         ErrorCode. To learn more about a file system's <code>ExportPath</code>, see .
     */
    public final String path() {
        return path;
    }

    /**
     * <p>
     * Required if <code>Enabled</code> is set to <code>true</code>. Specifies the format of the
     * <code>CompletionReport</code>. <code>REPORT_CSV_20191124</code> is the only format currently supported. When
     * <code>Format</code> is set to <code>REPORT_CSV_20191124</code>, the <code>CompletionReport</code> is provided in
     * CSV format, and is delivered to <code>{path}/task-{id}/failures.csv</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #format} will
     * return {@link ReportFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #formatAsString}.
     * </p>
     * 
     * @return Required if <code>Enabled</code> is set to <code>true</code>. Specifies the format of the
     *         <code>CompletionReport</code>. <code>REPORT_CSV_20191124</code> is the only format currently supported.
     *         When <code>Format</code> is set to <code>REPORT_CSV_20191124</code>, the <code>CompletionReport</code> is
     *         provided in CSV format, and is delivered to <code>{path}/task-{id}/failures.csv</code>.
     * @see ReportFormat
     */
    public final ReportFormat format() {
        return ReportFormat.fromValue(format);
    }

    /**
     * <p>
     * Required if <code>Enabled</code> is set to <code>true</code>. Specifies the format of the
     * <code>CompletionReport</code>. <code>REPORT_CSV_20191124</code> is the only format currently supported. When
     * <code>Format</code> is set to <code>REPORT_CSV_20191124</code>, the <code>CompletionReport</code> is provided in
     * CSV format, and is delivered to <code>{path}/task-{id}/failures.csv</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #format} will
     * return {@link ReportFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #formatAsString}.
     * </p>
     * 
     * @return Required if <code>Enabled</code> is set to <code>true</code>. Specifies the format of the
     *         <code>CompletionReport</code>. <code>REPORT_CSV_20191124</code> is the only format currently supported.
     *         When <code>Format</code> is set to <code>REPORT_CSV_20191124</code>, the <code>CompletionReport</code> is
     *         provided in CSV format, and is delivered to <code>{path}/task-{id}/failures.csv</code>.
     * @see ReportFormat
     */
    public final String formatAsString() {
        return format;
    }

    /**
     * <p>
     * Required if <code>Enabled</code> is set to <code>true</code>. Specifies the scope of the
     * <code>CompletionReport</code>; <code>FAILED_FILES_ONLY</code> is the only scope currently supported. When
     * <code>Scope</code> is set to <code>FAILED_FILES_ONLY</code>, the <code>CompletionReport</code> only contains
     * information about files that the data repository task failed to process.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #scope} will return
     * {@link ReportScope#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #scopeAsString}.
     * </p>
     * 
     * @return Required if <code>Enabled</code> is set to <code>true</code>. Specifies the scope of the
     *         <code>CompletionReport</code>; <code>FAILED_FILES_ONLY</code> is the only scope currently supported. When
     *         <code>Scope</code> is set to <code>FAILED_FILES_ONLY</code>, the <code>CompletionReport</code> only
     *         contains information about files that the data repository task failed to process.
     * @see ReportScope
     */
    public final ReportScope scope() {
        return ReportScope.fromValue(scope);
    }

    /**
     * <p>
     * Required if <code>Enabled</code> is set to <code>true</code>. Specifies the scope of the
     * <code>CompletionReport</code>; <code>FAILED_FILES_ONLY</code> is the only scope currently supported. When
     * <code>Scope</code> is set to <code>FAILED_FILES_ONLY</code>, the <code>CompletionReport</code> only contains
     * information about files that the data repository task failed to process.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #scope} will return
     * {@link ReportScope#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #scopeAsString}.
     * </p>
     * 
     * @return Required if <code>Enabled</code> is set to <code>true</code>. Specifies the scope of the
     *         <code>CompletionReport</code>; <code>FAILED_FILES_ONLY</code> is the only scope currently supported. When
     *         <code>Scope</code> is set to <code>FAILED_FILES_ONLY</code>, the <code>CompletionReport</code> only
     *         contains information about files that the data repository task failed to process.
     * @see ReportScope
     */
    public final String scopeAsString() {
        return scope;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(enabled());
        hashCode = 31 * hashCode + Objects.hashCode(path());
        hashCode = 31 * hashCode + Objects.hashCode(formatAsString());
        hashCode = 31 * hashCode + Objects.hashCode(scopeAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CompletionReport)) {
            return false;
        }
        CompletionReport other = (CompletionReport) obj;
        return Objects.equals(enabled(), other.enabled()) && Objects.equals(path(), other.path())
                && Objects.equals(formatAsString(), other.formatAsString())
                && Objects.equals(scopeAsString(), other.scopeAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CompletionReport").add("Enabled", enabled()).add("Path", path()).add("Format", formatAsString())
                .add("Scope", scopeAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Enabled":
            return Optional.ofNullable(clazz.cast(enabled()));
        case "Path":
            return Optional.ofNullable(clazz.cast(path()));
        case "Format":
            return Optional.ofNullable(clazz.cast(formatAsString()));
        case "Scope":
            return Optional.ofNullable(clazz.cast(scopeAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CompletionReport, T> g) {
        return obj -> g.apply((CompletionReport) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CompletionReport> {
        /**
         * <p>
         * Set <code>Enabled</code> to <code>True</code> to generate a <code>CompletionReport</code> when the task
         * completes. If set to <code>true</code>, then you need to provide a report <code>Scope</code>,
         * <code>Path</code>, and <code>Format</code>. Set <code>Enabled</code> to <code>False</code> if you do not want
         * a <code>CompletionReport</code> generated when the task completes.
         * </p>
         * 
         * @param enabled
         *        Set <code>Enabled</code> to <code>True</code> to generate a <code>CompletionReport</code> when the
         *        task completes. If set to <code>true</code>, then you need to provide a report <code>Scope</code>,
         *        <code>Path</code>, and <code>Format</code>. Set <code>Enabled</code> to <code>False</code> if you do
         *        not want a <code>CompletionReport</code> generated when the task completes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enabled(Boolean enabled);

        /**
         * <p>
         * Required if <code>Enabled</code> is set to <code>true</code>. Specifies the location of the report on the
         * file system's linked S3 data repository. An absolute path that defines where the completion report will be
         * stored in the destination location. The <code>Path</code> you provide must be located within the file
         * system’s ExportPath. An example <code>Path</code> value is "s3://myBucket/myExportPath/optionalPrefix". The
         * report provides the following information for each file in the report: FilePath, FileStatus, and ErrorCode.
         * To learn more about a file system's <code>ExportPath</code>, see .
         * </p>
         * 
         * @param path
         *        Required if <code>Enabled</code> is set to <code>true</code>. Specifies the location of the report on
         *        the file system's linked S3 data repository. An absolute path that defines where the completion report
         *        will be stored in the destination location. The <code>Path</code> you provide must be located within
         *        the file system’s ExportPath. An example <code>Path</code> value is
         *        "s3://myBucket/myExportPath/optionalPrefix". The report provides the following information for each
         *        file in the report: FilePath, FileStatus, and ErrorCode. To learn more about a file system's
         *        <code>ExportPath</code>, see .
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder path(String path);

        /**
         * <p>
         * Required if <code>Enabled</code> is set to <code>true</code>. Specifies the format of the
         * <code>CompletionReport</code>. <code>REPORT_CSV_20191124</code> is the only format currently supported. When
         * <code>Format</code> is set to <code>REPORT_CSV_20191124</code>, the <code>CompletionReport</code> is provided
         * in CSV format, and is delivered to <code>{path}/task-{id}/failures.csv</code>.
         * </p>
         * 
         * @param format
         *        Required if <code>Enabled</code> is set to <code>true</code>. Specifies the format of the
         *        <code>CompletionReport</code>. <code>REPORT_CSV_20191124</code> is the only format currently
         *        supported. When <code>Format</code> is set to <code>REPORT_CSV_20191124</code>, the
         *        <code>CompletionReport</code> is provided in CSV format, and is delivered to
         *        <code>{path}/task-{id}/failures.csv</code>.
         * @see ReportFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ReportFormat
         */
        Builder format(String format);

        /**
         * <p>
         * Required if <code>Enabled</code> is set to <code>true</code>. Specifies the format of the
         * <code>CompletionReport</code>. <code>REPORT_CSV_20191124</code> is the only format currently supported. When
         * <code>Format</code> is set to <code>REPORT_CSV_20191124</code>, the <code>CompletionReport</code> is provided
         * in CSV format, and is delivered to <code>{path}/task-{id}/failures.csv</code>.
         * </p>
         * 
         * @param format
         *        Required if <code>Enabled</code> is set to <code>true</code>. Specifies the format of the
         *        <code>CompletionReport</code>. <code>REPORT_CSV_20191124</code> is the only format currently
         *        supported. When <code>Format</code> is set to <code>REPORT_CSV_20191124</code>, the
         *        <code>CompletionReport</code> is provided in CSV format, and is delivered to
         *        <code>{path}/task-{id}/failures.csv</code>.
         * @see ReportFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ReportFormat
         */
        Builder format(ReportFormat format);

        /**
         * <p>
         * Required if <code>Enabled</code> is set to <code>true</code>. Specifies the scope of the
         * <code>CompletionReport</code>; <code>FAILED_FILES_ONLY</code> is the only scope currently supported. When
         * <code>Scope</code> is set to <code>FAILED_FILES_ONLY</code>, the <code>CompletionReport</code> only contains
         * information about files that the data repository task failed to process.
         * </p>
         * 
         * @param scope
         *        Required if <code>Enabled</code> is set to <code>true</code>. Specifies the scope of the
         *        <code>CompletionReport</code>; <code>FAILED_FILES_ONLY</code> is the only scope currently supported.
         *        When <code>Scope</code> is set to <code>FAILED_FILES_ONLY</code>, the <code>CompletionReport</code>
         *        only contains information about files that the data repository task failed to process.
         * @see ReportScope
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ReportScope
         */
        Builder scope(String scope);

        /**
         * <p>
         * Required if <code>Enabled</code> is set to <code>true</code>. Specifies the scope of the
         * <code>CompletionReport</code>; <code>FAILED_FILES_ONLY</code> is the only scope currently supported. When
         * <code>Scope</code> is set to <code>FAILED_FILES_ONLY</code>, the <code>CompletionReport</code> only contains
         * information about files that the data repository task failed to process.
         * </p>
         * 
         * @param scope
         *        Required if <code>Enabled</code> is set to <code>true</code>. Specifies the scope of the
         *        <code>CompletionReport</code>; <code>FAILED_FILES_ONLY</code> is the only scope currently supported.
         *        When <code>Scope</code> is set to <code>FAILED_FILES_ONLY</code>, the <code>CompletionReport</code>
         *        only contains information about files that the data repository task failed to process.
         * @see ReportScope
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ReportScope
         */
        Builder scope(ReportScope scope);
    }

    static final class BuilderImpl implements Builder {
        private Boolean enabled;

        private String path;

        private String format;

        private String scope;

        private BuilderImpl() {
        }

        private BuilderImpl(CompletionReport model) {
            enabled(model.enabled);
            path(model.path);
            format(model.format);
            scope(model.scope);
        }

        public final Boolean getEnabled() {
            return enabled;
        }

        public final void setEnabled(Boolean enabled) {
            this.enabled = enabled;
        }

        @Override
        public final Builder enabled(Boolean enabled) {
            this.enabled = enabled;
            return this;
        }

        public final String getPath() {
            return path;
        }

        public final void setPath(String path) {
            this.path = path;
        }

        @Override
        public final Builder path(String path) {
            this.path = path;
            return this;
        }

        public final String getFormat() {
            return format;
        }

        public final void setFormat(String format) {
            this.format = format;
        }

        @Override
        public final Builder format(String format) {
            this.format = format;
            return this;
        }

        @Override
        public final Builder format(ReportFormat format) {
            this.format(format == null ? null : format.toString());
            return this;
        }

        public final String getScope() {
            return scope;
        }

        public final void setScope(String scope) {
            this.scope = scope;
        }

        @Override
        public final Builder scope(String scope) {
            this.scope = scope;
            return this;
        }

        @Override
        public final Builder scope(ReportScope scope) {
            this.scope(scope == null ? null : scope.toString());
            return this;
        }

        @Override
        public CompletionReport build() {
            return new CompletionReport(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
