/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.iam.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information about a condition context key. It includes the name of the key and specifies the value (or
 * values, if the context key supports multiple values) to use in the simulation. This information is used when
 * evaluating the <code>Condition</code> elements of the input policies.
 * </p>
 * <p>
 * This data type is used as an input parameter to <a>SimulateCustomPolicy</a> and <a>SimulatePrincipalPolicy</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ContextEntry implements SdkPojo, Serializable, ToCopyableBuilder<ContextEntry.Builder, ContextEntry> {
    private static final SdkField<String> CONTEXT_KEY_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ContextKeyName").getter(getter(ContextEntry::contextKeyName)).setter(setter(Builder::contextKeyName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ContextKeyName").build()).build();

    private static final SdkField<List<String>> CONTEXT_KEY_VALUES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ContextKeyValues")
            .getter(getter(ContextEntry::contextKeyValues))
            .setter(setter(Builder::contextKeyValues))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ContextKeyValues").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> CONTEXT_KEY_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ContextKeyType").getter(getter(ContextEntry::contextKeyTypeAsString))
            .setter(setter(Builder::contextKeyType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ContextKeyType").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CONTEXT_KEY_NAME_FIELD,
            CONTEXT_KEY_VALUES_FIELD, CONTEXT_KEY_TYPE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String contextKeyName;

    private final List<String> contextKeyValues;

    private final String contextKeyType;

    private ContextEntry(BuilderImpl builder) {
        this.contextKeyName = builder.contextKeyName;
        this.contextKeyValues = builder.contextKeyValues;
        this.contextKeyType = builder.contextKeyType;
    }

    /**
     * <p>
     * The full name of a condition context key, including the service prefix. For example, <code>aws:SourceIp</code> or
     * <code>s3:VersionId</code>.
     * </p>
     * 
     * @return The full name of a condition context key, including the service prefix. For example,
     *         <code>aws:SourceIp</code> or <code>s3:VersionId</code>.
     */
    public final String contextKeyName() {
        return contextKeyName;
    }

    /**
     * For responses, this returns true if the service returned a value for the ContextKeyValues property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasContextKeyValues() {
        return contextKeyValues != null && !(contextKeyValues instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The value (or values, if the condition context key supports multiple values) to provide to the simulation when
     * the key is referenced by a <code>Condition</code> element in an input policy.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasContextKeyValues} method.
     * </p>
     * 
     * @return The value (or values, if the condition context key supports multiple values) to provide to the simulation
     *         when the key is referenced by a <code>Condition</code> element in an input policy.
     */
    public final List<String> contextKeyValues() {
        return contextKeyValues;
    }

    /**
     * <p>
     * The data type of the value (or values) specified in the <code>ContextKeyValues</code> parameter.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #contextKeyType}
     * will return {@link ContextKeyTypeEnum#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #contextKeyTypeAsString}.
     * </p>
     * 
     * @return The data type of the value (or values) specified in the <code>ContextKeyValues</code> parameter.
     * @see ContextKeyTypeEnum
     */
    public final ContextKeyTypeEnum contextKeyType() {
        return ContextKeyTypeEnum.fromValue(contextKeyType);
    }

    /**
     * <p>
     * The data type of the value (or values) specified in the <code>ContextKeyValues</code> parameter.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #contextKeyType}
     * will return {@link ContextKeyTypeEnum#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #contextKeyTypeAsString}.
     * </p>
     * 
     * @return The data type of the value (or values) specified in the <code>ContextKeyValues</code> parameter.
     * @see ContextKeyTypeEnum
     */
    public final String contextKeyTypeAsString() {
        return contextKeyType;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(contextKeyName());
        hashCode = 31 * hashCode + Objects.hashCode(hasContextKeyValues() ? contextKeyValues() : null);
        hashCode = 31 * hashCode + Objects.hashCode(contextKeyTypeAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ContextEntry)) {
            return false;
        }
        ContextEntry other = (ContextEntry) obj;
        return Objects.equals(contextKeyName(), other.contextKeyName()) && hasContextKeyValues() == other.hasContextKeyValues()
                && Objects.equals(contextKeyValues(), other.contextKeyValues())
                && Objects.equals(contextKeyTypeAsString(), other.contextKeyTypeAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ContextEntry").add("ContextKeyName", contextKeyName())
                .add("ContextKeyValues", hasContextKeyValues() ? contextKeyValues() : null)
                .add("ContextKeyType", contextKeyTypeAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ContextKeyName":
            return Optional.ofNullable(clazz.cast(contextKeyName()));
        case "ContextKeyValues":
            return Optional.ofNullable(clazz.cast(contextKeyValues()));
        case "ContextKeyType":
            return Optional.ofNullable(clazz.cast(contextKeyTypeAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ContextEntry, T> g) {
        return obj -> g.apply((ContextEntry) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ContextEntry> {
        /**
         * <p>
         * The full name of a condition context key, including the service prefix. For example,
         * <code>aws:SourceIp</code> or <code>s3:VersionId</code>.
         * </p>
         * 
         * @param contextKeyName
         *        The full name of a condition context key, including the service prefix. For example,
         *        <code>aws:SourceIp</code> or <code>s3:VersionId</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder contextKeyName(String contextKeyName);

        /**
         * <p>
         * The value (or values, if the condition context key supports multiple values) to provide to the simulation
         * when the key is referenced by a <code>Condition</code> element in an input policy.
         * </p>
         * 
         * @param contextKeyValues
         *        The value (or values, if the condition context key supports multiple values) to provide to the
         *        simulation when the key is referenced by a <code>Condition</code> element in an input policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder contextKeyValues(Collection<String> contextKeyValues);

        /**
         * <p>
         * The value (or values, if the condition context key supports multiple values) to provide to the simulation
         * when the key is referenced by a <code>Condition</code> element in an input policy.
         * </p>
         * 
         * @param contextKeyValues
         *        The value (or values, if the condition context key supports multiple values) to provide to the
         *        simulation when the key is referenced by a <code>Condition</code> element in an input policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder contextKeyValues(String... contextKeyValues);

        /**
         * <p>
         * The data type of the value (or values) specified in the <code>ContextKeyValues</code> parameter.
         * </p>
         * 
         * @param contextKeyType
         *        The data type of the value (or values) specified in the <code>ContextKeyValues</code> parameter.
         * @see ContextKeyTypeEnum
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ContextKeyTypeEnum
         */
        Builder contextKeyType(String contextKeyType);

        /**
         * <p>
         * The data type of the value (or values) specified in the <code>ContextKeyValues</code> parameter.
         * </p>
         * 
         * @param contextKeyType
         *        The data type of the value (or values) specified in the <code>ContextKeyValues</code> parameter.
         * @see ContextKeyTypeEnum
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ContextKeyTypeEnum
         */
        Builder contextKeyType(ContextKeyTypeEnum contextKeyType);
    }

    static final class BuilderImpl implements Builder {
        private String contextKeyName;

        private List<String> contextKeyValues = DefaultSdkAutoConstructList.getInstance();

        private String contextKeyType;

        private BuilderImpl() {
        }

        private BuilderImpl(ContextEntry model) {
            contextKeyName(model.contextKeyName);
            contextKeyValues(model.contextKeyValues);
            contextKeyType(model.contextKeyType);
        }

        public final String getContextKeyName() {
            return contextKeyName;
        }

        public final void setContextKeyName(String contextKeyName) {
            this.contextKeyName = contextKeyName;
        }

        @Override
        public final Builder contextKeyName(String contextKeyName) {
            this.contextKeyName = contextKeyName;
            return this;
        }

        public final Collection<String> getContextKeyValues() {
            if (contextKeyValues instanceof SdkAutoConstructList) {
                return null;
            }
            return contextKeyValues;
        }

        public final void setContextKeyValues(Collection<String> contextKeyValues) {
            this.contextKeyValues = ContextKeyValueListTypeCopier.copy(contextKeyValues);
        }

        @Override
        public final Builder contextKeyValues(Collection<String> contextKeyValues) {
            this.contextKeyValues = ContextKeyValueListTypeCopier.copy(contextKeyValues);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder contextKeyValues(String... contextKeyValues) {
            contextKeyValues(Arrays.asList(contextKeyValues));
            return this;
        }

        public final String getContextKeyType() {
            return contextKeyType;
        }

        public final void setContextKeyType(String contextKeyType) {
            this.contextKeyType = contextKeyType;
        }

        @Override
        public final Builder contextKeyType(String contextKeyType) {
            this.contextKeyType = contextKeyType;
            return this;
        }

        @Override
        public final Builder contextKeyType(ContextKeyTypeEnum contextKeyType) {
            this.contextKeyType(contextKeyType == null ? null : contextKeyType.toString());
            return this;
        }

        @Override
        public ContextEntry build() {
            return new ContextEntry(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
