/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.iotevents;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.iotevents.internal.IotEventsServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.iotevents.model.CreateAlarmModelRequest;
import software.amazon.awssdk.services.iotevents.model.CreateAlarmModelResponse;
import software.amazon.awssdk.services.iotevents.model.CreateDetectorModelRequest;
import software.amazon.awssdk.services.iotevents.model.CreateDetectorModelResponse;
import software.amazon.awssdk.services.iotevents.model.CreateInputRequest;
import software.amazon.awssdk.services.iotevents.model.CreateInputResponse;
import software.amazon.awssdk.services.iotevents.model.DeleteAlarmModelRequest;
import software.amazon.awssdk.services.iotevents.model.DeleteAlarmModelResponse;
import software.amazon.awssdk.services.iotevents.model.DeleteDetectorModelRequest;
import software.amazon.awssdk.services.iotevents.model.DeleteDetectorModelResponse;
import software.amazon.awssdk.services.iotevents.model.DeleteInputRequest;
import software.amazon.awssdk.services.iotevents.model.DeleteInputResponse;
import software.amazon.awssdk.services.iotevents.model.DescribeAlarmModelRequest;
import software.amazon.awssdk.services.iotevents.model.DescribeAlarmModelResponse;
import software.amazon.awssdk.services.iotevents.model.DescribeDetectorModelAnalysisRequest;
import software.amazon.awssdk.services.iotevents.model.DescribeDetectorModelAnalysisResponse;
import software.amazon.awssdk.services.iotevents.model.DescribeDetectorModelRequest;
import software.amazon.awssdk.services.iotevents.model.DescribeDetectorModelResponse;
import software.amazon.awssdk.services.iotevents.model.DescribeInputRequest;
import software.amazon.awssdk.services.iotevents.model.DescribeInputResponse;
import software.amazon.awssdk.services.iotevents.model.DescribeLoggingOptionsRequest;
import software.amazon.awssdk.services.iotevents.model.DescribeLoggingOptionsResponse;
import software.amazon.awssdk.services.iotevents.model.GetDetectorModelAnalysisResultsRequest;
import software.amazon.awssdk.services.iotevents.model.GetDetectorModelAnalysisResultsResponse;
import software.amazon.awssdk.services.iotevents.model.InternalFailureException;
import software.amazon.awssdk.services.iotevents.model.InvalidRequestException;
import software.amazon.awssdk.services.iotevents.model.IotEventsException;
import software.amazon.awssdk.services.iotevents.model.LimitExceededException;
import software.amazon.awssdk.services.iotevents.model.ListAlarmModelVersionsRequest;
import software.amazon.awssdk.services.iotevents.model.ListAlarmModelVersionsResponse;
import software.amazon.awssdk.services.iotevents.model.ListAlarmModelsRequest;
import software.amazon.awssdk.services.iotevents.model.ListAlarmModelsResponse;
import software.amazon.awssdk.services.iotevents.model.ListDetectorModelVersionsRequest;
import software.amazon.awssdk.services.iotevents.model.ListDetectorModelVersionsResponse;
import software.amazon.awssdk.services.iotevents.model.ListDetectorModelsRequest;
import software.amazon.awssdk.services.iotevents.model.ListDetectorModelsResponse;
import software.amazon.awssdk.services.iotevents.model.ListInputRoutingsRequest;
import software.amazon.awssdk.services.iotevents.model.ListInputRoutingsResponse;
import software.amazon.awssdk.services.iotevents.model.ListInputsRequest;
import software.amazon.awssdk.services.iotevents.model.ListInputsResponse;
import software.amazon.awssdk.services.iotevents.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.iotevents.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.iotevents.model.PutLoggingOptionsRequest;
import software.amazon.awssdk.services.iotevents.model.PutLoggingOptionsResponse;
import software.amazon.awssdk.services.iotevents.model.ResourceAlreadyExistsException;
import software.amazon.awssdk.services.iotevents.model.ResourceInUseException;
import software.amazon.awssdk.services.iotevents.model.ResourceNotFoundException;
import software.amazon.awssdk.services.iotevents.model.ServiceUnavailableException;
import software.amazon.awssdk.services.iotevents.model.StartDetectorModelAnalysisRequest;
import software.amazon.awssdk.services.iotevents.model.StartDetectorModelAnalysisResponse;
import software.amazon.awssdk.services.iotevents.model.TagResourceRequest;
import software.amazon.awssdk.services.iotevents.model.TagResourceResponse;
import software.amazon.awssdk.services.iotevents.model.ThrottlingException;
import software.amazon.awssdk.services.iotevents.model.UnsupportedOperationException;
import software.amazon.awssdk.services.iotevents.model.UntagResourceRequest;
import software.amazon.awssdk.services.iotevents.model.UntagResourceResponse;
import software.amazon.awssdk.services.iotevents.model.UpdateAlarmModelRequest;
import software.amazon.awssdk.services.iotevents.model.UpdateAlarmModelResponse;
import software.amazon.awssdk.services.iotevents.model.UpdateDetectorModelRequest;
import software.amazon.awssdk.services.iotevents.model.UpdateDetectorModelResponse;
import software.amazon.awssdk.services.iotevents.model.UpdateInputRequest;
import software.amazon.awssdk.services.iotevents.model.UpdateInputResponse;
import software.amazon.awssdk.services.iotevents.transform.CreateAlarmModelRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.CreateDetectorModelRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.CreateInputRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.DeleteAlarmModelRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.DeleteDetectorModelRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.DeleteInputRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.DescribeAlarmModelRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.DescribeDetectorModelAnalysisRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.DescribeDetectorModelRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.DescribeInputRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.DescribeLoggingOptionsRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.GetDetectorModelAnalysisResultsRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.ListAlarmModelVersionsRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.ListAlarmModelsRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.ListDetectorModelVersionsRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.ListDetectorModelsRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.ListInputRoutingsRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.ListInputsRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.PutLoggingOptionsRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.StartDetectorModelAnalysisRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.UpdateAlarmModelRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.UpdateDetectorModelRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.UpdateInputRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link IotEventsAsyncClient}.
 *
 * @see IotEventsAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultIotEventsAsyncClient implements IotEventsAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultIotEventsAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultIotEventsAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates an alarm model to monitor an AWS IoT Events input attribute. You can use the alarm to get notified when
     * the value is outside a specified range. For more information, see <a
     * href="https://docs.aws.amazon.com/iotevents/latest/developerguide/create-alarms.html">Create an alarm model</a>
     * in the <i>AWS IoT Events Developer Guide</i>.
     * </p>
     *
     * @param createAlarmModelRequest
     * @return A Java Future containing the result of the CreateAlarmModel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request was invalid.</li>
     *         <li>ResourceInUseException The resource is in use.</li>
     *         <li>ResourceAlreadyExistsException The resource already exists.</li>
     *         <li>LimitExceededException A limit was exceeded.</li>
     *         <li>ThrottlingException The request could not be completed due to throttling.</li>
     *         <li>InternalFailureException An internal failure occurred.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotEventsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotEventsAsyncClient.CreateAlarmModel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/CreateAlarmModel" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateAlarmModelResponse> createAlarmModel(CreateAlarmModelRequest createAlarmModelRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createAlarmModelRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createAlarmModelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateAlarmModel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateAlarmModelResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateAlarmModelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateAlarmModelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateAlarmModelRequest, CreateAlarmModelResponse>()
                            .withOperationName("CreateAlarmModel").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateAlarmModelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createAlarmModelRequest));
            CompletableFuture<CreateAlarmModelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a detector model.
     * </p>
     *
     * @param createDetectorModelRequest
     * @return A Java Future containing the result of the CreateDetectorModel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request was invalid.</li>
     *         <li>ResourceInUseException The resource is in use.</li>
     *         <li>ResourceAlreadyExistsException The resource already exists.</li>
     *         <li>LimitExceededException A limit was exceeded.</li>
     *         <li>ThrottlingException The request could not be completed due to throttling.</li>
     *         <li>InternalFailureException An internal failure occurred.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotEventsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotEventsAsyncClient.CreateDetectorModel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/CreateDetectorModel" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateDetectorModelResponse> createDetectorModel(
            CreateDetectorModelRequest createDetectorModelRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createDetectorModelRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDetectorModelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDetectorModel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateDetectorModelResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateDetectorModelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateDetectorModelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateDetectorModelRequest, CreateDetectorModelResponse>()
                            .withOperationName("CreateDetectorModel").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateDetectorModelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createDetectorModelRequest));
            CompletableFuture<CreateDetectorModelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an input.
     * </p>
     *
     * @param createInputRequest
     * @return A Java Future containing the result of the CreateInput operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request was invalid.</li>
     *         <li>ThrottlingException The request could not be completed due to throttling.</li>
     *         <li>InternalFailureException An internal failure occurred.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ResourceAlreadyExistsException The resource already exists.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotEventsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotEventsAsyncClient.CreateInput
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/CreateInput" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateInputResponse> createInput(CreateInputRequest createInputRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createInputRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createInputRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateInput");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateInputResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateInputResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateInputResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateInputRequest, CreateInputResponse>()
                            .withOperationName("CreateInput").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateInputRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createInputRequest));
            CompletableFuture<CreateInputResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an alarm model. Any alarm instances that were created based on this alarm model are also deleted. This
     * action can't be undone.
     * </p>
     *
     * @param deleteAlarmModelRequest
     * @return A Java Future containing the result of the DeleteAlarmModel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request was invalid.</li>
     *         <li>ResourceInUseException The resource is in use.</li>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>ThrottlingException The request could not be completed due to throttling.</li>
     *         <li>InternalFailureException An internal failure occurred.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotEventsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotEventsAsyncClient.DeleteAlarmModel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/DeleteAlarmModel" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteAlarmModelResponse> deleteAlarmModel(DeleteAlarmModelRequest deleteAlarmModelRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteAlarmModelRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAlarmModelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAlarmModel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteAlarmModelResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteAlarmModelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteAlarmModelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteAlarmModelRequest, DeleteAlarmModelResponse>()
                            .withOperationName("DeleteAlarmModel").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteAlarmModelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteAlarmModelRequest));
            CompletableFuture<DeleteAlarmModelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a detector model. Any active instances of the detector model are also deleted.
     * </p>
     *
     * @param deleteDetectorModelRequest
     * @return A Java Future containing the result of the DeleteDetectorModel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request was invalid.</li>
     *         <li>ResourceInUseException The resource is in use.</li>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>ThrottlingException The request could not be completed due to throttling.</li>
     *         <li>InternalFailureException An internal failure occurred.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotEventsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotEventsAsyncClient.DeleteDetectorModel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/DeleteDetectorModel" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteDetectorModelResponse> deleteDetectorModel(
            DeleteDetectorModelRequest deleteDetectorModelRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteDetectorModelRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDetectorModelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDetectorModel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteDetectorModelResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteDetectorModelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteDetectorModelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteDetectorModelRequest, DeleteDetectorModelResponse>()
                            .withOperationName("DeleteDetectorModel").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteDetectorModelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteDetectorModelRequest));
            CompletableFuture<DeleteDetectorModelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an input.
     * </p>
     *
     * @param deleteInputRequest
     * @return A Java Future containing the result of the DeleteInput operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request was invalid.</li>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>ThrottlingException The request could not be completed due to throttling.</li>
     *         <li>InternalFailureException An internal failure occurred.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ResourceInUseException The resource is in use.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotEventsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotEventsAsyncClient.DeleteInput
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/DeleteInput" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteInputResponse> deleteInput(DeleteInputRequest deleteInputRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteInputRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteInputRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteInput");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteInputResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteInputResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteInputResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteInputRequest, DeleteInputResponse>()
                            .withOperationName("DeleteInput").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteInputRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteInputRequest));
            CompletableFuture<DeleteInputResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves information about an alarm model. If you don't specify a value for the <code>alarmModelVersion</code>
     * parameter, the latest version is returned.
     * </p>
     *
     * @param describeAlarmModelRequest
     * @return A Java Future containing the result of the DescribeAlarmModel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request was invalid.</li>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>ThrottlingException The request could not be completed due to throttling.</li>
     *         <li>InternalFailureException An internal failure occurred.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotEventsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotEventsAsyncClient.DescribeAlarmModel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/DescribeAlarmModel" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeAlarmModelResponse> describeAlarmModel(DescribeAlarmModelRequest describeAlarmModelRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeAlarmModelRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAlarmModelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAlarmModel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeAlarmModelResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeAlarmModelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeAlarmModelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeAlarmModelRequest, DescribeAlarmModelResponse>()
                            .withOperationName("DescribeAlarmModel").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeAlarmModelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeAlarmModelRequest));
            CompletableFuture<DescribeAlarmModelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes a detector model. If the <code>version</code> parameter is not specified, information about the latest
     * version is returned.
     * </p>
     *
     * @param describeDetectorModelRequest
     * @return A Java Future containing the result of the DescribeDetectorModel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request was invalid.</li>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>ThrottlingException The request could not be completed due to throttling.</li>
     *         <li>InternalFailureException An internal failure occurred.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotEventsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotEventsAsyncClient.DescribeDetectorModel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/DescribeDetectorModel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeDetectorModelResponse> describeDetectorModel(
            DescribeDetectorModelRequest describeDetectorModelRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeDetectorModelRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeDetectorModelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeDetectorModel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeDetectorModelResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeDetectorModelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeDetectorModelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeDetectorModelRequest, DescribeDetectorModelResponse>()
                            .withOperationName("DescribeDetectorModel").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeDetectorModelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeDetectorModelRequest));
            CompletableFuture<DescribeDetectorModelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves runtime information about a detector model analysis.
     * </p>
     * <note>
     * <p>
     * After AWS IoT Events starts analyzing your detector model, you have up to 24 hours to retrieve the analysis
     * results.
     * </p>
     * </note>
     *
     * @param describeDetectorModelAnalysisRequest
     * @return A Java Future containing the result of the DescribeDetectorModelAnalysis operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request was invalid.</li>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>ThrottlingException The request could not be completed due to throttling.</li>
     *         <li>InternalFailureException An internal failure occurred.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotEventsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotEventsAsyncClient.DescribeDetectorModelAnalysis
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/DescribeDetectorModelAnalysis"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeDetectorModelAnalysisResponse> describeDetectorModelAnalysis(
            DescribeDetectorModelAnalysisRequest describeDetectorModelAnalysisRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeDetectorModelAnalysisRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeDetectorModelAnalysisRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeDetectorModelAnalysis");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeDetectorModelAnalysisResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeDetectorModelAnalysisResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeDetectorModelAnalysisResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeDetectorModelAnalysisRequest, DescribeDetectorModelAnalysisResponse>()
                            .withOperationName("DescribeDetectorModelAnalysis").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeDetectorModelAnalysisRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeDetectorModelAnalysisRequest));
            CompletableFuture<DescribeDetectorModelAnalysisResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes an input.
     * </p>
     *
     * @param describeInputRequest
     * @return A Java Future containing the result of the DescribeInput operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request was invalid.</li>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>ThrottlingException The request could not be completed due to throttling.</li>
     *         <li>InternalFailureException An internal failure occurred.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotEventsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotEventsAsyncClient.DescribeInput
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/DescribeInput" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeInputResponse> describeInput(DescribeInputRequest describeInputRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeInputRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeInputRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeInput");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeInputResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DescribeInputResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeInputResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeInputRequest, DescribeInputResponse>()
                            .withOperationName("DescribeInput").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeInputRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeInputRequest));
            CompletableFuture<DescribeInputResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the current settings of the AWS IoT Events logging options.
     * </p>
     *
     * @param describeLoggingOptionsRequest
     * @return A Java Future containing the result of the DescribeLoggingOptions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request was invalid.</li>
     *         <li>ThrottlingException The request could not be completed due to throttling.</li>
     *         <li>InternalFailureException An internal failure occurred.</li>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>UnsupportedOperationException The requested operation is not supported.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotEventsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotEventsAsyncClient.DescribeLoggingOptions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/DescribeLoggingOptions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeLoggingOptionsResponse> describeLoggingOptions(
            DescribeLoggingOptionsRequest describeLoggingOptionsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeLoggingOptionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeLoggingOptionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeLoggingOptions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeLoggingOptionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeLoggingOptionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeLoggingOptionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeLoggingOptionsRequest, DescribeLoggingOptionsResponse>()
                            .withOperationName("DescribeLoggingOptions").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeLoggingOptionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeLoggingOptionsRequest));
            CompletableFuture<DescribeLoggingOptionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves one or more analysis results of the detector model.
     * </p>
     * <note>
     * <p>
     * After AWS IoT Events starts analyzing your detector model, you have up to 24 hours to retrieve the analysis
     * results.
     * </p>
     * </note>
     *
     * @param getDetectorModelAnalysisResultsRequest
     * @return A Java Future containing the result of the GetDetectorModelAnalysisResults operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request was invalid.</li>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>ThrottlingException The request could not be completed due to throttling.</li>
     *         <li>InternalFailureException An internal failure occurred.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotEventsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotEventsAsyncClient.GetDetectorModelAnalysisResults
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/GetDetectorModelAnalysisResults"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetDetectorModelAnalysisResultsResponse> getDetectorModelAnalysisResults(
            GetDetectorModelAnalysisResultsRequest getDetectorModelAnalysisResultsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getDetectorModelAnalysisResultsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getDetectorModelAnalysisResultsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDetectorModelAnalysisResults");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDetectorModelAnalysisResultsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetDetectorModelAnalysisResultsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetDetectorModelAnalysisResultsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDetectorModelAnalysisResultsRequest, GetDetectorModelAnalysisResultsResponse>()
                            .withOperationName("GetDetectorModelAnalysisResults").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetDetectorModelAnalysisResultsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getDetectorModelAnalysisResultsRequest));
            CompletableFuture<GetDetectorModelAnalysisResultsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all the versions of an alarm model. The operation returns only the metadata associated with each alarm
     * model version.
     * </p>
     *
     * @param listAlarmModelVersionsRequest
     * @return A Java Future containing the result of the ListAlarmModelVersions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request was invalid.</li>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>ThrottlingException The request could not be completed due to throttling.</li>
     *         <li>InternalFailureException An internal failure occurred.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotEventsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotEventsAsyncClient.ListAlarmModelVersions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/ListAlarmModelVersions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListAlarmModelVersionsResponse> listAlarmModelVersions(
            ListAlarmModelVersionsRequest listAlarmModelVersionsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listAlarmModelVersionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAlarmModelVersionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAlarmModelVersions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAlarmModelVersionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListAlarmModelVersionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListAlarmModelVersionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAlarmModelVersionsRequest, ListAlarmModelVersionsResponse>()
                            .withOperationName("ListAlarmModelVersions").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListAlarmModelVersionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listAlarmModelVersionsRequest));
            CompletableFuture<ListAlarmModelVersionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the alarm models that you created. The operation returns only the metadata associated with each alarm
     * model.
     * </p>
     *
     * @param listAlarmModelsRequest
     * @return A Java Future containing the result of the ListAlarmModels operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request was invalid.</li>
     *         <li>ThrottlingException The request could not be completed due to throttling.</li>
     *         <li>InternalFailureException An internal failure occurred.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotEventsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotEventsAsyncClient.ListAlarmModels
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/ListAlarmModels" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListAlarmModelsResponse> listAlarmModels(ListAlarmModelsRequest listAlarmModelsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listAlarmModelsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAlarmModelsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAlarmModels");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAlarmModelsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListAlarmModelsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListAlarmModelsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAlarmModelsRequest, ListAlarmModelsResponse>()
                            .withOperationName("ListAlarmModels").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListAlarmModelsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listAlarmModelsRequest));
            CompletableFuture<ListAlarmModelsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all the versions of a detector model. Only the metadata associated with each detector model version is
     * returned.
     * </p>
     *
     * @param listDetectorModelVersionsRequest
     * @return A Java Future containing the result of the ListDetectorModelVersions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request was invalid.</li>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>ThrottlingException The request could not be completed due to throttling.</li>
     *         <li>InternalFailureException An internal failure occurred.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotEventsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotEventsAsyncClient.ListDetectorModelVersions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/ListDetectorModelVersions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListDetectorModelVersionsResponse> listDetectorModelVersions(
            ListDetectorModelVersionsRequest listDetectorModelVersionsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listDetectorModelVersionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDetectorModelVersionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDetectorModelVersions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDetectorModelVersionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListDetectorModelVersionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListDetectorModelVersionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDetectorModelVersionsRequest, ListDetectorModelVersionsResponse>()
                            .withOperationName("ListDetectorModelVersions").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListDetectorModelVersionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listDetectorModelVersionsRequest));
            CompletableFuture<ListDetectorModelVersionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the detector models you have created. Only the metadata associated with each detector model is returned.
     * </p>
     *
     * @param listDetectorModelsRequest
     * @return A Java Future containing the result of the ListDetectorModels operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request was invalid.</li>
     *         <li>ThrottlingException The request could not be completed due to throttling.</li>
     *         <li>InternalFailureException An internal failure occurred.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotEventsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotEventsAsyncClient.ListDetectorModels
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/ListDetectorModels" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListDetectorModelsResponse> listDetectorModels(ListDetectorModelsRequest listDetectorModelsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listDetectorModelsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDetectorModelsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDetectorModels");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDetectorModelsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListDetectorModelsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListDetectorModelsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDetectorModelsRequest, ListDetectorModelsResponse>()
                            .withOperationName("ListDetectorModels").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListDetectorModelsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listDetectorModelsRequest));
            CompletableFuture<ListDetectorModelsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists one or more input routings.
     * </p>
     *
     * @param listInputRoutingsRequest
     * @return A Java Future containing the result of the ListInputRoutings operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request was invalid.</li>
     *         <li>ThrottlingException The request could not be completed due to throttling.</li>
     *         <li>InternalFailureException An internal failure occurred.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotEventsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotEventsAsyncClient.ListInputRoutings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/ListInputRoutings" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListInputRoutingsResponse> listInputRoutings(ListInputRoutingsRequest listInputRoutingsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listInputRoutingsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listInputRoutingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListInputRoutings");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListInputRoutingsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListInputRoutingsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListInputRoutingsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListInputRoutingsRequest, ListInputRoutingsResponse>()
                            .withOperationName("ListInputRoutings").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListInputRoutingsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listInputRoutingsRequest));
            CompletableFuture<ListInputRoutingsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the inputs you have created.
     * </p>
     *
     * @param listInputsRequest
     * @return A Java Future containing the result of the ListInputs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request was invalid.</li>
     *         <li>ThrottlingException The request could not be completed due to throttling.</li>
     *         <li>InternalFailureException An internal failure occurred.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotEventsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotEventsAsyncClient.ListInputs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/ListInputs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListInputsResponse> listInputs(ListInputsRequest listInputsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listInputsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listInputsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListInputs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListInputsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListInputsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListInputsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListInputsRequest, ListInputsResponse>().withOperationName("ListInputs")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListInputsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listInputsRequest));
            CompletableFuture<ListInputsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the tags (metadata) you have assigned to the resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request was invalid.</li>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>ResourceInUseException The resource is in use.</li>
     *         <li>ThrottlingException The request could not be completed due to throttling.</li>
     *         <li>InternalFailureException An internal failure occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotEventsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotEventsAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Sets or updates the AWS IoT Events logging options.
     * </p>
     * <p>
     * If you update the value of any <code>loggingOptions</code> field, it takes up to one minute for the change to
     * take effect. If you change the policy attached to the role you specified in the <code>roleArn</code> field (for
     * example, to correct an invalid policy), it takes up to five minutes for that change to take effect.
     * </p>
     *
     * @param putLoggingOptionsRequest
     * @return A Java Future containing the result of the PutLoggingOptions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request was invalid.</li>
     *         <li>ThrottlingException The request could not be completed due to throttling.</li>
     *         <li>InternalFailureException An internal failure occurred.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>UnsupportedOperationException The requested operation is not supported.</li>
     *         <li>ResourceInUseException The resource is in use.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotEventsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotEventsAsyncClient.PutLoggingOptions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/PutLoggingOptions" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<PutLoggingOptionsResponse> putLoggingOptions(PutLoggingOptionsRequest putLoggingOptionsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putLoggingOptionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putLoggingOptionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutLoggingOptions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutLoggingOptionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutLoggingOptionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutLoggingOptionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutLoggingOptionsRequest, PutLoggingOptionsResponse>()
                            .withOperationName("PutLoggingOptions").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PutLoggingOptionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putLoggingOptionsRequest));
            CompletableFuture<PutLoggingOptionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Performs an analysis of your detector model. For more information, see <a
     * href="https://docs.aws.amazon.com/iotevents/latest/developerguide/iotevents-analyze-api.html">Troubleshooting a
     * detector model</a> in the <i>AWS IoT Events Developer Guide</i>.
     * </p>
     *
     * @param startDetectorModelAnalysisRequest
     * @return A Java Future containing the result of the StartDetectorModelAnalysis operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request was invalid.</li>
     *         <li>LimitExceededException A limit was exceeded.</li>
     *         <li>ThrottlingException The request could not be completed due to throttling.</li>
     *         <li>InternalFailureException An internal failure occurred.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotEventsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotEventsAsyncClient.StartDetectorModelAnalysis
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/StartDetectorModelAnalysis"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartDetectorModelAnalysisResponse> startDetectorModelAnalysis(
            StartDetectorModelAnalysisRequest startDetectorModelAnalysisRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startDetectorModelAnalysisRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startDetectorModelAnalysisRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartDetectorModelAnalysis");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartDetectorModelAnalysisResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartDetectorModelAnalysisResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartDetectorModelAnalysisResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartDetectorModelAnalysisRequest, StartDetectorModelAnalysisResponse>()
                            .withOperationName("StartDetectorModelAnalysis").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StartDetectorModelAnalysisRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(startDetectorModelAnalysisRequest));
            CompletableFuture<StartDetectorModelAnalysisResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds to or modifies the tags of the given resource. Tags are metadata that can be used to manage a resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request was invalid.</li>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>ResourceInUseException The resource is in use.</li>
     *         <li>ThrottlingException The request could not be completed due to throttling.</li>
     *         <li>LimitExceededException A limit was exceeded.</li>
     *         <li>InternalFailureException An internal failure occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotEventsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotEventsAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes the given tags (metadata) from the resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request was invalid.</li>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>ResourceInUseException The resource is in use.</li>
     *         <li>ThrottlingException The request could not be completed due to throttling.</li>
     *         <li>InternalFailureException An internal failure occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotEventsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotEventsAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an alarm model. Any alarms that were created based on the previous version are deleted and then created
     * again as new data arrives.
     * </p>
     *
     * @param updateAlarmModelRequest
     * @return A Java Future containing the result of the UpdateAlarmModel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request was invalid.</li>
     *         <li>ResourceInUseException The resource is in use.</li>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>ThrottlingException The request could not be completed due to throttling.</li>
     *         <li>InternalFailureException An internal failure occurred.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotEventsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotEventsAsyncClient.UpdateAlarmModel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/UpdateAlarmModel" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateAlarmModelResponse> updateAlarmModel(UpdateAlarmModelRequest updateAlarmModelRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateAlarmModelRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateAlarmModelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateAlarmModel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateAlarmModelResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateAlarmModelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateAlarmModelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateAlarmModelRequest, UpdateAlarmModelResponse>()
                            .withOperationName("UpdateAlarmModel").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateAlarmModelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateAlarmModelRequest));
            CompletableFuture<UpdateAlarmModelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a detector model. Detectors (instances) spawned by the previous version are deleted and then re-created
     * as new inputs arrive.
     * </p>
     *
     * @param updateDetectorModelRequest
     * @return A Java Future containing the result of the UpdateDetectorModel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request was invalid.</li>
     *         <li>ResourceInUseException The resource is in use.</li>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>ThrottlingException The request could not be completed due to throttling.</li>
     *         <li>InternalFailureException An internal failure occurred.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotEventsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotEventsAsyncClient.UpdateDetectorModel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/UpdateDetectorModel" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateDetectorModelResponse> updateDetectorModel(
            UpdateDetectorModelRequest updateDetectorModelRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateDetectorModelRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateDetectorModelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateDetectorModel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateDetectorModelResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateDetectorModelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateDetectorModelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateDetectorModelRequest, UpdateDetectorModelResponse>()
                            .withOperationName("UpdateDetectorModel").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateDetectorModelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateDetectorModelRequest));
            CompletableFuture<UpdateDetectorModelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an input.
     * </p>
     *
     * @param updateInputRequest
     * @return A Java Future containing the result of the UpdateInput operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request was invalid.</li>
     *         <li>ThrottlingException The request could not be completed due to throttling.</li>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>InternalFailureException An internal failure occurred.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ResourceInUseException The resource is in use.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotEventsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotEventsAsyncClient.UpdateInput
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/UpdateInput" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateInputResponse> updateInput(UpdateInputRequest updateInputRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateInputRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateInputRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateInput");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateInputResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateInputResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateInputResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateInputRequest, UpdateInputResponse>()
                            .withOperationName("UpdateInput").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateInputRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateInputRequest));
            CompletableFuture<UpdateInputResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final IotEventsServiceClientConfiguration serviceClientConfiguration() {
        return new IotEventsServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(IotEventsException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalFailureException")
                                .exceptionBuilderSupplier(InternalFailureException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceInUseException")
                                .exceptionBuilderSupplier(ResourceInUseException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRequestException")
                                .exceptionBuilderSupplier(InvalidRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(410).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnsupportedOperationException")
                                .exceptionBuilderSupplier(UnsupportedOperationException::builder).httpStatusCode(501).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).httpStatusCode(503).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException")
                                .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).httpStatusCode(409).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        IotEventsServiceClientConfigurationBuilder serviceConfigBuilder = new IotEventsServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
