/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ioteventsdata;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.ioteventsdata.internal.IotEventsDataServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.ioteventsdata.model.BatchAcknowledgeAlarmRequest;
import software.amazon.awssdk.services.ioteventsdata.model.BatchAcknowledgeAlarmResponse;
import software.amazon.awssdk.services.ioteventsdata.model.BatchDeleteDetectorRequest;
import software.amazon.awssdk.services.ioteventsdata.model.BatchDeleteDetectorResponse;
import software.amazon.awssdk.services.ioteventsdata.model.BatchDisableAlarmRequest;
import software.amazon.awssdk.services.ioteventsdata.model.BatchDisableAlarmResponse;
import software.amazon.awssdk.services.ioteventsdata.model.BatchEnableAlarmRequest;
import software.amazon.awssdk.services.ioteventsdata.model.BatchEnableAlarmResponse;
import software.amazon.awssdk.services.ioteventsdata.model.BatchPutMessageRequest;
import software.amazon.awssdk.services.ioteventsdata.model.BatchPutMessageResponse;
import software.amazon.awssdk.services.ioteventsdata.model.BatchResetAlarmRequest;
import software.amazon.awssdk.services.ioteventsdata.model.BatchResetAlarmResponse;
import software.amazon.awssdk.services.ioteventsdata.model.BatchSnoozeAlarmRequest;
import software.amazon.awssdk.services.ioteventsdata.model.BatchSnoozeAlarmResponse;
import software.amazon.awssdk.services.ioteventsdata.model.BatchUpdateDetectorRequest;
import software.amazon.awssdk.services.ioteventsdata.model.BatchUpdateDetectorResponse;
import software.amazon.awssdk.services.ioteventsdata.model.DescribeAlarmRequest;
import software.amazon.awssdk.services.ioteventsdata.model.DescribeAlarmResponse;
import software.amazon.awssdk.services.ioteventsdata.model.DescribeDetectorRequest;
import software.amazon.awssdk.services.ioteventsdata.model.DescribeDetectorResponse;
import software.amazon.awssdk.services.ioteventsdata.model.InternalFailureException;
import software.amazon.awssdk.services.ioteventsdata.model.InvalidRequestException;
import software.amazon.awssdk.services.ioteventsdata.model.IotEventsDataException;
import software.amazon.awssdk.services.ioteventsdata.model.ListAlarmsRequest;
import software.amazon.awssdk.services.ioteventsdata.model.ListAlarmsResponse;
import software.amazon.awssdk.services.ioteventsdata.model.ListDetectorsRequest;
import software.amazon.awssdk.services.ioteventsdata.model.ListDetectorsResponse;
import software.amazon.awssdk.services.ioteventsdata.model.ResourceNotFoundException;
import software.amazon.awssdk.services.ioteventsdata.model.ServiceUnavailableException;
import software.amazon.awssdk.services.ioteventsdata.model.ThrottlingException;
import software.amazon.awssdk.services.ioteventsdata.transform.BatchAcknowledgeAlarmRequestMarshaller;
import software.amazon.awssdk.services.ioteventsdata.transform.BatchDeleteDetectorRequestMarshaller;
import software.amazon.awssdk.services.ioteventsdata.transform.BatchDisableAlarmRequestMarshaller;
import software.amazon.awssdk.services.ioteventsdata.transform.BatchEnableAlarmRequestMarshaller;
import software.amazon.awssdk.services.ioteventsdata.transform.BatchPutMessageRequestMarshaller;
import software.amazon.awssdk.services.ioteventsdata.transform.BatchResetAlarmRequestMarshaller;
import software.amazon.awssdk.services.ioteventsdata.transform.BatchSnoozeAlarmRequestMarshaller;
import software.amazon.awssdk.services.ioteventsdata.transform.BatchUpdateDetectorRequestMarshaller;
import software.amazon.awssdk.services.ioteventsdata.transform.DescribeAlarmRequestMarshaller;
import software.amazon.awssdk.services.ioteventsdata.transform.DescribeDetectorRequestMarshaller;
import software.amazon.awssdk.services.ioteventsdata.transform.ListAlarmsRequestMarshaller;
import software.amazon.awssdk.services.ioteventsdata.transform.ListDetectorsRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link IotEventsDataAsyncClient}.
 *
 * @see IotEventsDataAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultIotEventsDataAsyncClient implements IotEventsDataAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultIotEventsDataAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultIotEventsDataAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Acknowledges one or more alarms. The alarms change to the <code>ACKNOWLEDGED</code> state after you acknowledge
     * them.
     * </p>
     *
     * @param batchAcknowledgeAlarmRequest
     * @return A Java Future containing the result of the BatchAcknowledgeAlarm operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request was invalid.</li>
     *         <li>InternalFailureException An internal failure occurred.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ThrottlingException The request could not be completed due to throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotEventsDataException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotEventsDataAsyncClient.BatchAcknowledgeAlarm
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotevents-data-2018-10-23/BatchAcknowledgeAlarm"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchAcknowledgeAlarmResponse> batchAcknowledgeAlarm(
            BatchAcknowledgeAlarmRequest batchAcknowledgeAlarmRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(batchAcknowledgeAlarmRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchAcknowledgeAlarmRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Events Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchAcknowledgeAlarm");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchAcknowledgeAlarmResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchAcknowledgeAlarmResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BatchAcknowledgeAlarmResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchAcknowledgeAlarmRequest, BatchAcknowledgeAlarmResponse>()
                            .withOperationName("BatchAcknowledgeAlarm").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new BatchAcknowledgeAlarmRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(batchAcknowledgeAlarmRequest));
            CompletableFuture<BatchAcknowledgeAlarmResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes one or more detectors that were created. When a detector is deleted, its state will be cleared and the
     * detector will be removed from the list of detectors. The deleted detector will no longer appear if referenced in
     * the <a href="https://docs.aws.amazon.com/iotevents/latest/apireference/API_iotevents-data_ListDetectors.html">
     * ListDetectors</a> API call.
     * </p>
     *
     * @param batchDeleteDetectorRequest
     * @return A Java Future containing the result of the BatchDeleteDetector operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request was invalid.</li>
     *         <li>InternalFailureException An internal failure occurred.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ThrottlingException The request could not be completed due to throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotEventsDataException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotEventsDataAsyncClient.BatchDeleteDetector
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotevents-data-2018-10-23/BatchDeleteDetector"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchDeleteDetectorResponse> batchDeleteDetector(
            BatchDeleteDetectorRequest batchDeleteDetectorRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(batchDeleteDetectorRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchDeleteDetectorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Events Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchDeleteDetector");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchDeleteDetectorResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchDeleteDetectorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BatchDeleteDetectorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchDeleteDetectorRequest, BatchDeleteDetectorResponse>()
                            .withOperationName("BatchDeleteDetector").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new BatchDeleteDetectorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(batchDeleteDetectorRequest));
            CompletableFuture<BatchDeleteDetectorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Disables one or more alarms. The alarms change to the <code>DISABLED</code> state after you disable them.
     * </p>
     *
     * @param batchDisableAlarmRequest
     * @return A Java Future containing the result of the BatchDisableAlarm operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request was invalid.</li>
     *         <li>InternalFailureException An internal failure occurred.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ThrottlingException The request could not be completed due to throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotEventsDataException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotEventsDataAsyncClient.BatchDisableAlarm
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotevents-data-2018-10-23/BatchDisableAlarm"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchDisableAlarmResponse> batchDisableAlarm(BatchDisableAlarmRequest batchDisableAlarmRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(batchDisableAlarmRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchDisableAlarmRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Events Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchDisableAlarm");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchDisableAlarmResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchDisableAlarmResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BatchDisableAlarmResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchDisableAlarmRequest, BatchDisableAlarmResponse>()
                            .withOperationName("BatchDisableAlarm").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new BatchDisableAlarmRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(batchDisableAlarmRequest));
            CompletableFuture<BatchDisableAlarmResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Enables one or more alarms. The alarms change to the <code>NORMAL</code> state after you enable them.
     * </p>
     *
     * @param batchEnableAlarmRequest
     * @return A Java Future containing the result of the BatchEnableAlarm operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request was invalid.</li>
     *         <li>InternalFailureException An internal failure occurred.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ThrottlingException The request could not be completed due to throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotEventsDataException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotEventsDataAsyncClient.BatchEnableAlarm
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotevents-data-2018-10-23/BatchEnableAlarm"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchEnableAlarmResponse> batchEnableAlarm(BatchEnableAlarmRequest batchEnableAlarmRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(batchEnableAlarmRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchEnableAlarmRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Events Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchEnableAlarm");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchEnableAlarmResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchEnableAlarmResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BatchEnableAlarmResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchEnableAlarmRequest, BatchEnableAlarmResponse>()
                            .withOperationName("BatchEnableAlarm").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new BatchEnableAlarmRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(batchEnableAlarmRequest));
            CompletableFuture<BatchEnableAlarmResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Sends a set of messages to the IoT Events system. Each message payload is transformed into the input you specify
     * (<code>"inputName"</code>) and ingested into any detectors that monitor that input. If multiple messages are
     * sent, the order in which the messages are processed isn't guaranteed. To guarantee ordering, you must send
     * messages one at a time and wait for a successful response.
     * </p>
     *
     * @param batchPutMessageRequest
     * @return A Java Future containing the result of the BatchPutMessage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request was invalid.</li>
     *         <li>InternalFailureException An internal failure occurred.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ThrottlingException The request could not be completed due to throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotEventsDataException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotEventsDataAsyncClient.BatchPutMessage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotevents-data-2018-10-23/BatchPutMessage"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchPutMessageResponse> batchPutMessage(BatchPutMessageRequest batchPutMessageRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(batchPutMessageRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchPutMessageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Events Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchPutMessage");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchPutMessageResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchPutMessageResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BatchPutMessageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchPutMessageRequest, BatchPutMessageResponse>()
                            .withOperationName("BatchPutMessage").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new BatchPutMessageRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(batchPutMessageRequest));
            CompletableFuture<BatchPutMessageResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Resets one or more alarms. The alarms return to the <code>NORMAL</code> state after you reset them.
     * </p>
     *
     * @param batchResetAlarmRequest
     * @return A Java Future containing the result of the BatchResetAlarm operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request was invalid.</li>
     *         <li>InternalFailureException An internal failure occurred.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ThrottlingException The request could not be completed due to throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotEventsDataException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotEventsDataAsyncClient.BatchResetAlarm
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotevents-data-2018-10-23/BatchResetAlarm"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchResetAlarmResponse> batchResetAlarm(BatchResetAlarmRequest batchResetAlarmRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(batchResetAlarmRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchResetAlarmRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Events Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchResetAlarm");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchResetAlarmResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchResetAlarmResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BatchResetAlarmResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchResetAlarmRequest, BatchResetAlarmResponse>()
                            .withOperationName("BatchResetAlarm").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new BatchResetAlarmRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(batchResetAlarmRequest));
            CompletableFuture<BatchResetAlarmResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Changes one or more alarms to the snooze mode. The alarms change to the <code>SNOOZE_DISABLED</code> state after
     * you set them to the snooze mode.
     * </p>
     *
     * @param batchSnoozeAlarmRequest
     * @return A Java Future containing the result of the BatchSnoozeAlarm operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request was invalid.</li>
     *         <li>InternalFailureException An internal failure occurred.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ThrottlingException The request could not be completed due to throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotEventsDataException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotEventsDataAsyncClient.BatchSnoozeAlarm
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotevents-data-2018-10-23/BatchSnoozeAlarm"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchSnoozeAlarmResponse> batchSnoozeAlarm(BatchSnoozeAlarmRequest batchSnoozeAlarmRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(batchSnoozeAlarmRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchSnoozeAlarmRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Events Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchSnoozeAlarm");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchSnoozeAlarmResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchSnoozeAlarmResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BatchSnoozeAlarmResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchSnoozeAlarmRequest, BatchSnoozeAlarmResponse>()
                            .withOperationName("BatchSnoozeAlarm").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new BatchSnoozeAlarmRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(batchSnoozeAlarmRequest));
            CompletableFuture<BatchSnoozeAlarmResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the state, variable values, and timer settings of one or more detectors (instances) of a specified
     * detector model.
     * </p>
     *
     * @param batchUpdateDetectorRequest
     * @return A Java Future containing the result of the BatchUpdateDetector operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request was invalid.</li>
     *         <li>InternalFailureException An internal failure occurred.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ThrottlingException The request could not be completed due to throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotEventsDataException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotEventsDataAsyncClient.BatchUpdateDetector
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotevents-data-2018-10-23/BatchUpdateDetector"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchUpdateDetectorResponse> batchUpdateDetector(
            BatchUpdateDetectorRequest batchUpdateDetectorRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(batchUpdateDetectorRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchUpdateDetectorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Events Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchUpdateDetector");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchUpdateDetectorResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchUpdateDetectorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BatchUpdateDetectorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchUpdateDetectorRequest, BatchUpdateDetectorResponse>()
                            .withOperationName("BatchUpdateDetector").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new BatchUpdateDetectorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(batchUpdateDetectorRequest));
            CompletableFuture<BatchUpdateDetectorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves information about an alarm.
     * </p>
     *
     * @param describeAlarmRequest
     * @return A Java Future containing the result of the DescribeAlarm operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request was invalid.</li>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>ThrottlingException The request could not be completed due to throttling.</li>
     *         <li>InternalFailureException An internal failure occurred.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotEventsDataException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotEventsDataAsyncClient.DescribeAlarm
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotevents-data-2018-10-23/DescribeAlarm" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeAlarmResponse> describeAlarm(DescribeAlarmRequest describeAlarmRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeAlarmRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAlarmRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Events Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAlarm");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeAlarmResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DescribeAlarmResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeAlarmResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeAlarmRequest, DescribeAlarmResponse>()
                            .withOperationName("DescribeAlarm").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeAlarmRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeAlarmRequest));
            CompletableFuture<DescribeAlarmResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about the specified detector (instance).
     * </p>
     *
     * @param describeDetectorRequest
     * @return A Java Future containing the result of the DescribeDetector operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request was invalid.</li>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>ThrottlingException The request could not be completed due to throttling.</li>
     *         <li>InternalFailureException An internal failure occurred.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotEventsDataException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotEventsDataAsyncClient.DescribeDetector
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotevents-data-2018-10-23/DescribeDetector"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeDetectorResponse> describeDetector(DescribeDetectorRequest describeDetectorRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeDetectorRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeDetectorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Events Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeDetector");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeDetectorResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeDetectorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeDetectorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeDetectorRequest, DescribeDetectorResponse>()
                            .withOperationName("DescribeDetector").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeDetectorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeDetectorRequest));
            CompletableFuture<DescribeDetectorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists one or more alarms. The operation returns only the metadata associated with each alarm.
     * </p>
     *
     * @param listAlarmsRequest
     * @return A Java Future containing the result of the ListAlarms operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request was invalid.</li>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>ThrottlingException The request could not be completed due to throttling.</li>
     *         <li>InternalFailureException An internal failure occurred.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotEventsDataException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotEventsDataAsyncClient.ListAlarms
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotevents-data-2018-10-23/ListAlarms" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListAlarmsResponse> listAlarms(ListAlarmsRequest listAlarmsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listAlarmsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAlarmsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Events Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAlarms");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAlarmsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListAlarmsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListAlarmsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAlarmsRequest, ListAlarmsResponse>().withOperationName("ListAlarms")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListAlarmsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listAlarmsRequest));
            CompletableFuture<ListAlarmsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists detectors (the instances of a detector model).
     * </p>
     *
     * @param listDetectorsRequest
     * @return A Java Future containing the result of the ListDetectors operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request was invalid.</li>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>ThrottlingException The request could not be completed due to throttling.</li>
     *         <li>InternalFailureException An internal failure occurred.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotEventsDataException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotEventsDataAsyncClient.ListDetectors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotevents-data-2018-10-23/ListDetectors" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListDetectorsResponse> listDetectors(ListDetectorsRequest listDetectorsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listDetectorsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDetectorsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Events Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDetectors");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDetectorsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListDetectorsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListDetectorsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDetectorsRequest, ListDetectorsResponse>()
                            .withOperationName("ListDetectors").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListDetectorsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listDetectorsRequest));
            CompletableFuture<ListDetectorsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final IotEventsDataServiceClientConfiguration serviceClientConfiguration() {
        return new IotEventsDataServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(IotEventsDataException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalFailureException")
                                .exceptionBuilderSupplier(InternalFailureException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRequestException")
                                .exceptionBuilderSupplier(InvalidRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).httpStatusCode(503).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        IotEventsDataServiceClientConfigurationBuilder serviceConfigBuilder = new IotEventsDataServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
