/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lexmodelsv2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains fields describing a condition by which to filter the intent stages. The expression may be understood as
 * <code>name</code> <code>operator</code> <code>values</code>. For example:
 * </p>
 * <ul>
 * <li>
 * <p>
 * <code>IntentName CO Book</code> – The intent name contains the string "Book."
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>BotVersion EQ 2</code> – The bot version is equal to two.
 * </p>
 * </li>
 * </ul>
 * <p>
 * The operators that each filter supports are listed below:
 * </p>
 * <ul>
 * <li>
 * <p>
 * <code>BotAlias</code> – <code>EQ</code>.
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>BotVersion</code> – <code>EQ</code>.
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>LocaleId</code> – <code>EQ</code>.
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>Modality</code> – <code>EQ</code>.
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>Channel</code> – <code>EQ</code>.
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>SessionId</code> – <code>EQ</code>.
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>OriginatingRequestId</code> – <code>EQ</code>.
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>IntentName</code> – <code>EQ</code>, <code>CO</code>.
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>IntentStageName</code> – <code>EQ</code>, <code>CO</code>.
 * </p>
 * </li>
 * </ul>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AnalyticsIntentStageFilter implements SdkPojo, Serializable,
        ToCopyableBuilder<AnalyticsIntentStageFilter.Builder, AnalyticsIntentStageFilter> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(AnalyticsIntentStageFilter::nameAsString)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> OPERATOR_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("operator").getter(getter(AnalyticsIntentStageFilter::operatorAsString))
            .setter(setter(Builder::operator))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("operator").build()).build();

    private static final SdkField<List<String>> VALUES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("values")
            .getter(getter(AnalyticsIntentStageFilter::values))
            .setter(setter(Builder::values))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("values").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, OPERATOR_FIELD,
            VALUES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String operator;

    private final List<String> values;

    private AnalyticsIntentStageFilter(BuilderImpl builder) {
        this.name = builder.name;
        this.operator = builder.operator;
        this.values = builder.values;
    }

    /**
     * <p>
     * The category by which to filter the intent stages. The descriptions for each option are as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>BotAlias</code> – The name of the bot alias.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>BotVersion</code> – The version of the bot.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>LocaleId</code> – The locale of the bot.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Modality</code> – The modality of the session with the bot (audio, DTMF, or text).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Channel</code> – The channel that the bot is integrated with.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SessionId</code> – The identifier of the session with the bot.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>OriginatingRequestId</code> – The identifier of the first request in a session.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>IntentName</code> – The name of the intent.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>IntentStageName</code> – The stage in the intent.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #name} will return
     * {@link AnalyticsIntentStageFilterName#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #nameAsString}.
     * </p>
     * 
     * @return The category by which to filter the intent stages. The descriptions for each option are as follows:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>BotAlias</code> – The name of the bot alias.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>BotVersion</code> – The version of the bot.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>LocaleId</code> – The locale of the bot.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Modality</code> – The modality of the session with the bot (audio, DTMF, or text).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Channel</code> – The channel that the bot is integrated with.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SessionId</code> – The identifier of the session with the bot.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>OriginatingRequestId</code> – The identifier of the first request in a session.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>IntentName</code> – The name of the intent.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>IntentStageName</code> – The stage in the intent.
     *         </p>
     *         </li>
     * @see AnalyticsIntentStageFilterName
     */
    public final AnalyticsIntentStageFilterName name() {
        return AnalyticsIntentStageFilterName.fromValue(name);
    }

    /**
     * <p>
     * The category by which to filter the intent stages. The descriptions for each option are as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>BotAlias</code> – The name of the bot alias.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>BotVersion</code> – The version of the bot.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>LocaleId</code> – The locale of the bot.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Modality</code> – The modality of the session with the bot (audio, DTMF, or text).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Channel</code> – The channel that the bot is integrated with.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SessionId</code> – The identifier of the session with the bot.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>OriginatingRequestId</code> – The identifier of the first request in a session.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>IntentName</code> – The name of the intent.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>IntentStageName</code> – The stage in the intent.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #name} will return
     * {@link AnalyticsIntentStageFilterName#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #nameAsString}.
     * </p>
     * 
     * @return The category by which to filter the intent stages. The descriptions for each option are as follows:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>BotAlias</code> – The name of the bot alias.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>BotVersion</code> – The version of the bot.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>LocaleId</code> – The locale of the bot.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Modality</code> – The modality of the session with the bot (audio, DTMF, or text).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Channel</code> – The channel that the bot is integrated with.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SessionId</code> – The identifier of the session with the bot.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>OriginatingRequestId</code> – The identifier of the first request in a session.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>IntentName</code> – The name of the intent.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>IntentStageName</code> – The stage in the intent.
     *         </p>
     *         </li>
     * @see AnalyticsIntentStageFilterName
     */
    public final String nameAsString() {
        return name;
    }

    /**
     * <p>
     * The operation by which to filter the category. The following operations are possible:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CO</code> – Contains
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>EQ</code> – Equals
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>GT</code> – Greater than
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>LT</code> – Less than
     * </p>
     * </li>
     * </ul>
     * <p>
     * The operators that each filter supports are listed below:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>BotAlias</code> – <code>EQ</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>BotVersion</code> – <code>EQ</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>LocaleId</code> – <code>EQ</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Modality</code> – <code>EQ</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Channel</code> – <code>EQ</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SessionId</code> – <code>EQ</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>OriginatingRequestId</code> – <code>EQ</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>IntentName</code> – <code>EQ</code>, <code>CO</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>IntentStageName</code> – <code>EQ</code>, <code>CO</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #operator} will
     * return {@link AnalyticsFilterOperator#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #operatorAsString}.
     * </p>
     * 
     * @return The operation by which to filter the category. The following operations are possible:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>CO</code> – Contains
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>EQ</code> – Equals
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>GT</code> – Greater than
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>LT</code> – Less than
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         The operators that each filter supports are listed below:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>BotAlias</code> – <code>EQ</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>BotVersion</code> – <code>EQ</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>LocaleId</code> – <code>EQ</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Modality</code> – <code>EQ</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Channel</code> – <code>EQ</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SessionId</code> – <code>EQ</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>OriginatingRequestId</code> – <code>EQ</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>IntentName</code> – <code>EQ</code>, <code>CO</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>IntentStageName</code> – <code>EQ</code>, <code>CO</code>.
     *         </p>
     *         </li>
     * @see AnalyticsFilterOperator
     */
    public final AnalyticsFilterOperator operator() {
        return AnalyticsFilterOperator.fromValue(operator);
    }

    /**
     * <p>
     * The operation by which to filter the category. The following operations are possible:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CO</code> – Contains
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>EQ</code> – Equals
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>GT</code> – Greater than
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>LT</code> – Less than
     * </p>
     * </li>
     * </ul>
     * <p>
     * The operators that each filter supports are listed below:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>BotAlias</code> – <code>EQ</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>BotVersion</code> – <code>EQ</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>LocaleId</code> – <code>EQ</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Modality</code> – <code>EQ</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Channel</code> – <code>EQ</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SessionId</code> – <code>EQ</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>OriginatingRequestId</code> – <code>EQ</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>IntentName</code> – <code>EQ</code>, <code>CO</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>IntentStageName</code> – <code>EQ</code>, <code>CO</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #operator} will
     * return {@link AnalyticsFilterOperator#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #operatorAsString}.
     * </p>
     * 
     * @return The operation by which to filter the category. The following operations are possible:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>CO</code> – Contains
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>EQ</code> – Equals
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>GT</code> – Greater than
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>LT</code> – Less than
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         The operators that each filter supports are listed below:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>BotAlias</code> – <code>EQ</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>BotVersion</code> – <code>EQ</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>LocaleId</code> – <code>EQ</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Modality</code> – <code>EQ</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Channel</code> – <code>EQ</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SessionId</code> – <code>EQ</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>OriginatingRequestId</code> – <code>EQ</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>IntentName</code> – <code>EQ</code>, <code>CO</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>IntentStageName</code> – <code>EQ</code>, <code>CO</code>.
     *         </p>
     *         </li>
     * @see AnalyticsFilterOperator
     */
    public final String operatorAsString() {
        return operator;
    }

    /**
     * For responses, this returns true if the service returned a value for the Values property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasValues() {
        return values != null && !(values instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array containing the values of the category by which to apply the operator to filter the results. You can
     * provide multiple values if the operator is <code>EQ</code> or <code>CO</code>. If you provide multiple values,
     * you filter for results that equal/contain any of the values. For example, if the <code>name</code>,
     * <code>operator</code>, and <code>values</code> fields are <code>Modality</code>, <code>EQ</code>, and
     * <code>[Speech, Text]</code>, the operation filters for results where the modality was either <code>Speech</code>
     * or <code>Text</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasValues} method.
     * </p>
     * 
     * @return An array containing the values of the category by which to apply the operator to filter the results. You
     *         can provide multiple values if the operator is <code>EQ</code> or <code>CO</code>. If you provide
     *         multiple values, you filter for results that equal/contain any of the values. For example, if the
     *         <code>name</code>, <code>operator</code>, and <code>values</code> fields are <code>Modality</code>,
     *         <code>EQ</code>, and <code>[Speech, Text]</code>, the operation filters for results where the modality
     *         was either <code>Speech</code> or <code>Text</code>.
     */
    public final List<String> values() {
        return values;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(nameAsString());
        hashCode = 31 * hashCode + Objects.hashCode(operatorAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasValues() ? values() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AnalyticsIntentStageFilter)) {
            return false;
        }
        AnalyticsIntentStageFilter other = (AnalyticsIntentStageFilter) obj;
        return Objects.equals(nameAsString(), other.nameAsString())
                && Objects.equals(operatorAsString(), other.operatorAsString()) && hasValues() == other.hasValues()
                && Objects.equals(values(), other.values());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AnalyticsIntentStageFilter").add("Name", nameAsString()).add("Operator", operatorAsString())
                .add("Values", hasValues() ? values() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "name":
            return Optional.ofNullable(clazz.cast(nameAsString()));
        case "operator":
            return Optional.ofNullable(clazz.cast(operatorAsString()));
        case "values":
            return Optional.ofNullable(clazz.cast(values()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<AnalyticsIntentStageFilter, T> g) {
        return obj -> g.apply((AnalyticsIntentStageFilter) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AnalyticsIntentStageFilter> {
        /**
         * <p>
         * The category by which to filter the intent stages. The descriptions for each option are as follows:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>BotAlias</code> – The name of the bot alias.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>BotVersion</code> – The version of the bot.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>LocaleId</code> – The locale of the bot.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Modality</code> – The modality of the session with the bot (audio, DTMF, or text).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Channel</code> – The channel that the bot is integrated with.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SessionId</code> – The identifier of the session with the bot.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>OriginatingRequestId</code> – The identifier of the first request in a session.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IntentName</code> – The name of the intent.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IntentStageName</code> – The stage in the intent.
         * </p>
         * </li>
         * </ul>
         * 
         * @param name
         *        The category by which to filter the intent stages. The descriptions for each option are as
         *        follows:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>BotAlias</code> – The name of the bot alias.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>BotVersion</code> – The version of the bot.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>LocaleId</code> – The locale of the bot.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Modality</code> – The modality of the session with the bot (audio, DTMF, or text).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Channel</code> – The channel that the bot is integrated with.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SessionId</code> – The identifier of the session with the bot.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>OriginatingRequestId</code> – The identifier of the first request in a session.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>IntentName</code> – The name of the intent.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>IntentStageName</code> – The stage in the intent.
         *        </p>
         *        </li>
         * @see AnalyticsIntentStageFilterName
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AnalyticsIntentStageFilterName
         */
        Builder name(String name);

        /**
         * <p>
         * The category by which to filter the intent stages. The descriptions for each option are as follows:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>BotAlias</code> – The name of the bot alias.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>BotVersion</code> – The version of the bot.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>LocaleId</code> – The locale of the bot.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Modality</code> – The modality of the session with the bot (audio, DTMF, or text).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Channel</code> – The channel that the bot is integrated with.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SessionId</code> – The identifier of the session with the bot.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>OriginatingRequestId</code> – The identifier of the first request in a session.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IntentName</code> – The name of the intent.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IntentStageName</code> – The stage in the intent.
         * </p>
         * </li>
         * </ul>
         * 
         * @param name
         *        The category by which to filter the intent stages. The descriptions for each option are as
         *        follows:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>BotAlias</code> – The name of the bot alias.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>BotVersion</code> – The version of the bot.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>LocaleId</code> – The locale of the bot.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Modality</code> – The modality of the session with the bot (audio, DTMF, or text).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Channel</code> – The channel that the bot is integrated with.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SessionId</code> – The identifier of the session with the bot.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>OriginatingRequestId</code> – The identifier of the first request in a session.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>IntentName</code> – The name of the intent.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>IntentStageName</code> – The stage in the intent.
         *        </p>
         *        </li>
         * @see AnalyticsIntentStageFilterName
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AnalyticsIntentStageFilterName
         */
        Builder name(AnalyticsIntentStageFilterName name);

        /**
         * <p>
         * The operation by which to filter the category. The following operations are possible:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>CO</code> – Contains
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>EQ</code> – Equals
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>GT</code> – Greater than
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>LT</code> – Less than
         * </p>
         * </li>
         * </ul>
         * <p>
         * The operators that each filter supports are listed below:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>BotAlias</code> – <code>EQ</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>BotVersion</code> – <code>EQ</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>LocaleId</code> – <code>EQ</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Modality</code> – <code>EQ</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Channel</code> – <code>EQ</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SessionId</code> – <code>EQ</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>OriginatingRequestId</code> – <code>EQ</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IntentName</code> – <code>EQ</code>, <code>CO</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IntentStageName</code> – <code>EQ</code>, <code>CO</code>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param operator
         *        The operation by which to filter the category. The following operations are possible:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>CO</code> – Contains
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>EQ</code> – Equals
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>GT</code> – Greater than
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>LT</code> – Less than
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        The operators that each filter supports are listed below:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>BotAlias</code> – <code>EQ</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>BotVersion</code> – <code>EQ</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>LocaleId</code> – <code>EQ</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Modality</code> – <code>EQ</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Channel</code> – <code>EQ</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SessionId</code> – <code>EQ</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>OriginatingRequestId</code> – <code>EQ</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>IntentName</code> – <code>EQ</code>, <code>CO</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>IntentStageName</code> – <code>EQ</code>, <code>CO</code>.
         *        </p>
         *        </li>
         * @see AnalyticsFilterOperator
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AnalyticsFilterOperator
         */
        Builder operator(String operator);

        /**
         * <p>
         * The operation by which to filter the category. The following operations are possible:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>CO</code> – Contains
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>EQ</code> – Equals
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>GT</code> – Greater than
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>LT</code> – Less than
         * </p>
         * </li>
         * </ul>
         * <p>
         * The operators that each filter supports are listed below:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>BotAlias</code> – <code>EQ</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>BotVersion</code> – <code>EQ</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>LocaleId</code> – <code>EQ</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Modality</code> – <code>EQ</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Channel</code> – <code>EQ</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SessionId</code> – <code>EQ</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>OriginatingRequestId</code> – <code>EQ</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IntentName</code> – <code>EQ</code>, <code>CO</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IntentStageName</code> – <code>EQ</code>, <code>CO</code>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param operator
         *        The operation by which to filter the category. The following operations are possible:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>CO</code> – Contains
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>EQ</code> – Equals
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>GT</code> – Greater than
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>LT</code> – Less than
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        The operators that each filter supports are listed below:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>BotAlias</code> – <code>EQ</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>BotVersion</code> – <code>EQ</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>LocaleId</code> – <code>EQ</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Modality</code> – <code>EQ</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Channel</code> – <code>EQ</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SessionId</code> – <code>EQ</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>OriginatingRequestId</code> – <code>EQ</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>IntentName</code> – <code>EQ</code>, <code>CO</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>IntentStageName</code> – <code>EQ</code>, <code>CO</code>.
         *        </p>
         *        </li>
         * @see AnalyticsFilterOperator
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AnalyticsFilterOperator
         */
        Builder operator(AnalyticsFilterOperator operator);

        /**
         * <p>
         * An array containing the values of the category by which to apply the operator to filter the results. You can
         * provide multiple values if the operator is <code>EQ</code> or <code>CO</code>. If you provide multiple
         * values, you filter for results that equal/contain any of the values. For example, if the <code>name</code>,
         * <code>operator</code>, and <code>values</code> fields are <code>Modality</code>, <code>EQ</code>, and
         * <code>[Speech, Text]</code>, the operation filters for results where the modality was either
         * <code>Speech</code> or <code>Text</code>.
         * </p>
         * 
         * @param values
         *        An array containing the values of the category by which to apply the operator to filter the results.
         *        You can provide multiple values if the operator is <code>EQ</code> or <code>CO</code>. If you provide
         *        multiple values, you filter for results that equal/contain any of the values. For example, if the
         *        <code>name</code>, <code>operator</code>, and <code>values</code> fields are <code>Modality</code>,
         *        <code>EQ</code>, and <code>[Speech, Text]</code>, the operation filters for results where the modality
         *        was either <code>Speech</code> or <code>Text</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder values(Collection<String> values);

        /**
         * <p>
         * An array containing the values of the category by which to apply the operator to filter the results. You can
         * provide multiple values if the operator is <code>EQ</code> or <code>CO</code>. If you provide multiple
         * values, you filter for results that equal/contain any of the values. For example, if the <code>name</code>,
         * <code>operator</code>, and <code>values</code> fields are <code>Modality</code>, <code>EQ</code>, and
         * <code>[Speech, Text]</code>, the operation filters for results where the modality was either
         * <code>Speech</code> or <code>Text</code>.
         * </p>
         * 
         * @param values
         *        An array containing the values of the category by which to apply the operator to filter the results.
         *        You can provide multiple values if the operator is <code>EQ</code> or <code>CO</code>. If you provide
         *        multiple values, you filter for results that equal/contain any of the values. For example, if the
         *        <code>name</code>, <code>operator</code>, and <code>values</code> fields are <code>Modality</code>,
         *        <code>EQ</code>, and <code>[Speech, Text]</code>, the operation filters for results where the modality
         *        was either <code>Speech</code> or <code>Text</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder values(String... values);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String operator;

        private List<String> values = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(AnalyticsIntentStageFilter model) {
            name(model.name);
            operator(model.operator);
            values(model.values);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        @Override
        public final Builder name(AnalyticsIntentStageFilterName name) {
            this.name(name == null ? null : name.toString());
            return this;
        }

        public final String getOperator() {
            return operator;
        }

        public final void setOperator(String operator) {
            this.operator = operator;
        }

        @Override
        public final Builder operator(String operator) {
            this.operator = operator;
            return this;
        }

        @Override
        public final Builder operator(AnalyticsFilterOperator operator) {
            this.operator(operator == null ? null : operator.toString());
            return this;
        }

        public final Collection<String> getValues() {
            if (values instanceof SdkAutoConstructList) {
                return null;
            }
            return values;
        }

        public final void setValues(Collection<String> values) {
            this.values = AnalyticsFilterValuesCopier.copy(values);
        }

        @Override
        public final Builder values(Collection<String> values) {
            this.values = AnalyticsFilterValuesCopier.copy(values);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder values(String... values) {
            values(Arrays.asList(values));
            return this;
        }

        @Override
        public AnalyticsIntentStageFilter build() {
            return new AnalyticsIntentStageFilter(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
