/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.marketplaceagreement;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.marketplaceagreement.internal.MarketplaceAgreementServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.marketplaceagreement.model.AccessDeniedException;
import software.amazon.awssdk.services.marketplaceagreement.model.DescribeAgreementRequest;
import software.amazon.awssdk.services.marketplaceagreement.model.DescribeAgreementResponse;
import software.amazon.awssdk.services.marketplaceagreement.model.GetAgreementTermsRequest;
import software.amazon.awssdk.services.marketplaceagreement.model.GetAgreementTermsResponse;
import software.amazon.awssdk.services.marketplaceagreement.model.InternalServerException;
import software.amazon.awssdk.services.marketplaceagreement.model.MarketplaceAgreementException;
import software.amazon.awssdk.services.marketplaceagreement.model.ResourceNotFoundException;
import software.amazon.awssdk.services.marketplaceagreement.model.SearchAgreementsRequest;
import software.amazon.awssdk.services.marketplaceagreement.model.SearchAgreementsResponse;
import software.amazon.awssdk.services.marketplaceagreement.model.ThrottlingException;
import software.amazon.awssdk.services.marketplaceagreement.model.ValidationException;
import software.amazon.awssdk.services.marketplaceagreement.transform.DescribeAgreementRequestMarshaller;
import software.amazon.awssdk.services.marketplaceagreement.transform.GetAgreementTermsRequestMarshaller;
import software.amazon.awssdk.services.marketplaceagreement.transform.SearchAgreementsRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link MarketplaceAgreementClient}.
 *
 * @see MarketplaceAgreementClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultMarketplaceAgreementClient implements MarketplaceAgreementClient {
    private static final Logger log = Logger.loggerFor(DefaultMarketplaceAgreementClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.AWS_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultMarketplaceAgreementClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Provides details about an agreement, such as the proposer, acceptor, start date, and end date.
     * </p>
     *
     * @param describeAgreementRequest
     * @return Result of the DescribeAgreement operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by the service.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         Request references a resource which does not exist.
     * @throws InternalServerException
     *         Unexpected error during processing of request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MarketplaceAgreementException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MarketplaceAgreementClient.DescribeAgreement
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/marketplace-agreement-2020-03-01/DescribeAgreement"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeAgreementResponse describeAgreement(DescribeAgreementRequest describeAgreementRequest)
            throws ValidationException, AccessDeniedException, ThrottlingException, ResourceNotFoundException,
            InternalServerException, AwsServiceException, SdkClientException, MarketplaceAgreementException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeAgreementResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeAgreementResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeAgreementRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAgreementRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Marketplace Agreement");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAgreement");

            return clientHandler.execute(new ClientExecutionParams<DescribeAgreementRequest, DescribeAgreementResponse>()
                    .withOperationName("DescribeAgreement").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describeAgreementRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeAgreementRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Obtains details about the terms in an agreement that you participated in as proposer or acceptor.
     * </p>
     * <p>
     * The details include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>TermType</code> – The type of term, such as <code>LegalTerm</code>, <code>RenewalTerm</code>, or
     * <code>ConfigurableUpfrontPricingTerm</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>TermID</code> – The ID of the particular term, which is common between offer and agreement.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>TermPayload</code> – The key information contained in the term, such as the EULA for <code>LegalTerm</code>
     * or pricing and dimensions for various pricing terms, such as <code>ConfigurableUpfrontPricingTerm</code> or
     * <code>UsageBasedPricingTerm</code>.
     * </p>
     * </li>
     * </ul>
     * <ul>
     * <li>
     * <p>
     * <code>Configuration</code> – The buyer/acceptor's selection at the time of agreement creation, such as the number
     * of units purchased for a dimension or setting the <code>EnableAutoRenew</code> flag.
     * </p>
     * </li>
     * </ul>
     *
     * @param getAgreementTermsRequest
     * @return Result of the GetAgreementTerms operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by the service.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         Request references a resource which does not exist.
     * @throws InternalServerException
     *         Unexpected error during processing of request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MarketplaceAgreementException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MarketplaceAgreementClient.GetAgreementTerms
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/marketplace-agreement-2020-03-01/GetAgreementTerms"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetAgreementTermsResponse getAgreementTerms(GetAgreementTermsRequest getAgreementTermsRequest)
            throws ValidationException, AccessDeniedException, ThrottlingException, ResourceNotFoundException,
            InternalServerException, AwsServiceException, SdkClientException, MarketplaceAgreementException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetAgreementTermsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetAgreementTermsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getAgreementTermsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAgreementTermsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Marketplace Agreement");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAgreementTerms");

            return clientHandler.execute(new ClientExecutionParams<GetAgreementTermsRequest, GetAgreementTermsResponse>()
                    .withOperationName("GetAgreementTerms").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getAgreementTermsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetAgreementTermsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Searches across all agreements that a proposer or an acceptor has in AWS Marketplace. The search returns a list
     * of agreements with basic agreement information.
     * </p>
     * <p>
     * The following filter combinations are supported:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>PartyType</code> as <code>Proposer</code> + <code>AgreementType</code> + <code>ResourceIdentifier</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PartyType</code> as <code>Proposer</code> + <code>AgreementType</code> + <code>OfferId</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PartyType</code> as <code>Proposer</code> + <code>AgreementType</code> + <code>AcceptorAccountId</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PartyType</code> as <code>Proposer</code> + <code>AgreementType</code> + <code>Status</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PartyType</code> as <code>Proposer</code> + <code>AgreementType</code> + <code>ResourceIdentifier</code> +
     * <code>Status</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PartyType</code> as <code>Proposer</code> + <code>AgreementType</code> + <code>OfferId</code> +
     * <code>Status</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PartyType</code> as <code>Proposer</code> + <code>AgreementType</code> + <code>AcceptorAccountId</code> +
     * <code>Status</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PartyType</code> as <code>Proposer</code> + <code>AgreementType</code> + <code>ResourceType</code> +
     * <code>Status</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PartyType</code> as <code>Proposer</code> + <code>AgreementType</code> + <code>AcceptorAccountId</code> +
     * <code>ResourceType</code> + <code>Status</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PartyType</code> as <code>Proposer</code> + <code>AgreementType</code> + <code>AcceptorAccountId</code> +
     * <code>OfferId</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PartyType</code> as <code>Proposer</code> + <code>AgreementType</code> + <code>AcceptorAccountId</code> +
     * <code>OfferId</code> + <code>Status</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PartyType</code> as <code>Proposer</code> + <code>AgreementType</code> + <code>AcceptorAccountId</code> +
     * <code>ResourceIdentifier</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PartyType</code> as <code>Proposer</code> + <code>AgreementType</code> + <code>AcceptorAccountId</code> +
     * <code>ResourceIdentifier</code> + <code>Status</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PartyType</code> as <code>Proposer</code> + <code>AgreementType</code> + <code>AcceptorAccountId</code> +
     * <code>ResourceType</code>
     * </p>
     * </li>
     * </ul>
     *
     * @param searchAgreementsRequest
     * @return Result of the SearchAgreements operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by the service.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws InternalServerException
     *         Unexpected error during processing of request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MarketplaceAgreementException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MarketplaceAgreementClient.SearchAgreements
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/marketplace-agreement-2020-03-01/SearchAgreements"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public SearchAgreementsResponse searchAgreements(SearchAgreementsRequest searchAgreementsRequest) throws ValidationException,
            AccessDeniedException, ThrottlingException, InternalServerException, AwsServiceException, SdkClientException,
            MarketplaceAgreementException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<SearchAgreementsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                SearchAgreementsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(searchAgreementsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, searchAgreementsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Marketplace Agreement");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SearchAgreements");

            return clientHandler.execute(new ClientExecutionParams<SearchAgreementsRequest, SearchAgreementsResponse>()
                    .withOperationName("SearchAgreements").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(searchAgreementsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new SearchAgreementsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        MarketplaceAgreementServiceClientConfigurationBuilder serviceConfigBuilder = new MarketplaceAgreementServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(MarketplaceAgreementException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.0")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(400).build());
    }

    @Override
    public final MarketplaceAgreementServiceClientConfiguration serviceClientConfiguration() {
        return new MarketplaceAgreementServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
