/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.marketplacecatalog.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.document.Document;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An object that contains the <code>ChangeType</code>, <code>Details</code>, and <code>Entity</code>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Change implements SdkPojo, Serializable, ToCopyableBuilder<Change.Builder, Change> {
    private static final SdkField<String> CHANGE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ChangeType").getter(getter(Change::changeType)).setter(setter(Builder::changeType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ChangeType").build()).build();

    private static final SdkField<Entity> ENTITY_FIELD = SdkField.<Entity> builder(MarshallingType.SDK_POJO).memberName("Entity")
            .getter(getter(Change::entity)).setter(setter(Builder::entity)).constructor(Entity::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Entity").build()).build();

    private static final SdkField<List<Tag>> ENTITY_TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("EntityTags")
            .getter(getter(Change::entityTags))
            .setter(setter(Builder::entityTags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EntityTags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> DETAILS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Details")
            .getter(getter(Change::details)).setter(setter(Builder::details))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Details").build()).build();

    private static final SdkField<Document> DETAILS_DOCUMENT_FIELD = SdkField.<Document> builder(MarshallingType.DOCUMENT)
            .memberName("DetailsDocument").getter(getter(Change::detailsDocument)).setter(setter(Builder::detailsDocument))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DetailsDocument").build()).build();

    private static final SdkField<String> CHANGE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ChangeName").getter(getter(Change::changeName)).setter(setter(Builder::changeName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ChangeName").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CHANGE_TYPE_FIELD,
            ENTITY_FIELD, ENTITY_TAGS_FIELD, DETAILS_FIELD, DETAILS_DOCUMENT_FIELD, CHANGE_NAME_FIELD));

    private static final long serialVersionUID = 1L;

    private final String changeType;

    private final Entity entity;

    private final List<Tag> entityTags;

    private final String details;

    private final Document detailsDocument;

    private final String changeName;

    private Change(BuilderImpl builder) {
        this.changeType = builder.changeType;
        this.entity = builder.entity;
        this.entityTags = builder.entityTags;
        this.details = builder.details;
        this.detailsDocument = builder.detailsDocument;
        this.changeName = builder.changeName;
    }

    /**
     * <p>
     * Change types are single string values that describe your intention for the change. Each change type is unique for
     * each <code>EntityType</code> provided in the change's scope. For more information on change types available for
     * single-AMI products, see <a href=
     * "https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/ami-products.html#working-with-single-AMI-products"
     * >Working with single-AMI products</a>. Also, for more information about change types available for
     * container-based products, see <a href=
     * "https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/container-products.html#working-with-container-products"
     * >Working with container products</a>.
     * </p>
     * 
     * @return Change types are single string values that describe your intention for the change. Each change type is
     *         unique for each <code>EntityType</code> provided in the change's scope. For more information on change
     *         types available for single-AMI products, see <a href=
     *         "https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/ami-products.html#working-with-single-AMI-products"
     *         >Working with single-AMI products</a>. Also, for more information about change types available for
     *         container-based products, see <a href=
     *         "https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/container-products.html#working-with-container-products"
     *         >Working with container products</a>.
     */
    public final String changeType() {
        return changeType;
    }

    /**
     * <p>
     * The entity to be changed.
     * </p>
     * 
     * @return The entity to be changed.
     */
    public final Entity entity() {
        return entity;
    }

    /**
     * For responses, this returns true if the service returned a value for the EntityTags property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasEntityTags() {
        return entityTags != null && !(entityTags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The tags associated with the change.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEntityTags} method.
     * </p>
     * 
     * @return The tags associated with the change.
     */
    public final List<Tag> entityTags() {
        return entityTags;
    }

    /**
     * <p>
     * This object contains details specific to the change type of the requested change. For more information about
     * change types available for single-AMI products, see <a href=
     * "https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/ami-products.html#working-with-single-AMI-products"
     * >Working with single-AMI products</a>. Also, for more information about change types available for
     * container-based products, see <a href=
     * "https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/container-products.html#working-with-container-products"
     * >Working with container products</a>.
     * </p>
     * 
     * @return This object contains details specific to the change type of the requested change. For more information
     *         about change types available for single-AMI products, see <a href=
     *         "https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/ami-products.html#working-with-single-AMI-products"
     *         >Working with single-AMI products</a>. Also, for more information about change types available for
     *         container-based products, see <a href=
     *         "https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/container-products.html#working-with-container-products"
     *         >Working with container products</a>.
     */
    public final String details() {
        return details;
    }

    /**
     * <p>
     * Alternative field that accepts a JSON value instead of a string for <code>ChangeType</code> details. You can use
     * either <code>Details</code> or <code>DetailsDocument</code>, but not both.
     * </p>
     * 
     * @return Alternative field that accepts a JSON value instead of a string for <code>ChangeType</code> details. You
     *         can use either <code>Details</code> or <code>DetailsDocument</code>, but not both.
     */
    public final Document detailsDocument() {
        return detailsDocument;
    }

    /**
     * <p>
     * Optional name for the change.
     * </p>
     * 
     * @return Optional name for the change.
     */
    public final String changeName() {
        return changeName;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(changeType());
        hashCode = 31 * hashCode + Objects.hashCode(entity());
        hashCode = 31 * hashCode + Objects.hashCode(hasEntityTags() ? entityTags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(details());
        hashCode = 31 * hashCode + Objects.hashCode(detailsDocument());
        hashCode = 31 * hashCode + Objects.hashCode(changeName());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Change)) {
            return false;
        }
        Change other = (Change) obj;
        return Objects.equals(changeType(), other.changeType()) && Objects.equals(entity(), other.entity())
                && hasEntityTags() == other.hasEntityTags() && Objects.equals(entityTags(), other.entityTags())
                && Objects.equals(details(), other.details()) && Objects.equals(detailsDocument(), other.detailsDocument())
                && Objects.equals(changeName(), other.changeName());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Change").add("ChangeType", changeType()).add("Entity", entity())
                .add("EntityTags", hasEntityTags() ? entityTags() : null).add("Details", details())
                .add("DetailsDocument", detailsDocument()).add("ChangeName", changeName()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ChangeType":
            return Optional.ofNullable(clazz.cast(changeType()));
        case "Entity":
            return Optional.ofNullable(clazz.cast(entity()));
        case "EntityTags":
            return Optional.ofNullable(clazz.cast(entityTags()));
        case "Details":
            return Optional.ofNullable(clazz.cast(details()));
        case "DetailsDocument":
            return Optional.ofNullable(clazz.cast(detailsDocument()));
        case "ChangeName":
            return Optional.ofNullable(clazz.cast(changeName()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Change, T> g) {
        return obj -> g.apply((Change) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Change> {
        /**
         * <p>
         * Change types are single string values that describe your intention for the change. Each change type is unique
         * for each <code>EntityType</code> provided in the change's scope. For more information on change types
         * available for single-AMI products, see <a href=
         * "https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/ami-products.html#working-with-single-AMI-products"
         * >Working with single-AMI products</a>. Also, for more information about change types available for
         * container-based products, see <a href=
         * "https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/container-products.html#working-with-container-products"
         * >Working with container products</a>.
         * </p>
         * 
         * @param changeType
         *        Change types are single string values that describe your intention for the change. Each change type is
         *        unique for each <code>EntityType</code> provided in the change's scope. For more information on change
         *        types available for single-AMI products, see <a href=
         *        "https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/ami-products.html#working-with-single-AMI-products"
         *        >Working with single-AMI products</a>. Also, for more information about change types available for
         *        container-based products, see <a href=
         *        "https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/container-products.html#working-with-container-products"
         *        >Working with container products</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder changeType(String changeType);

        /**
         * <p>
         * The entity to be changed.
         * </p>
         * 
         * @param entity
         *        The entity to be changed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder entity(Entity entity);

        /**
         * <p>
         * The entity to be changed.
         * </p>
         * This is a convenience method that creates an instance of the {@link Entity.Builder} avoiding the need to
         * create one manually via {@link Entity#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Entity.Builder#build()} is called immediately and its result is
         * passed to {@link #entity(Entity)}.
         * 
         * @param entity
         *        a consumer that will call methods on {@link Entity.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #entity(Entity)
         */
        default Builder entity(Consumer<Entity.Builder> entity) {
            return entity(Entity.builder().applyMutation(entity).build());
        }

        /**
         * <p>
         * The tags associated with the change.
         * </p>
         * 
         * @param entityTags
         *        The tags associated with the change.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder entityTags(Collection<Tag> entityTags);

        /**
         * <p>
         * The tags associated with the change.
         * </p>
         * 
         * @param entityTags
         *        The tags associated with the change.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder entityTags(Tag... entityTags);

        /**
         * <p>
         * The tags associated with the change.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.marketplacecatalog.model.Tag.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.marketplacecatalog.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.marketplacecatalog.model.Tag.Builder#build()} is called immediately
         * and its result is passed to {@link #entityTags(List<Tag>)}.
         * 
         * @param entityTags
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.marketplacecatalog.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #entityTags(java.util.Collection<Tag>)
         */
        Builder entityTags(Consumer<Tag.Builder>... entityTags);

        /**
         * <p>
         * This object contains details specific to the change type of the requested change. For more information about
         * change types available for single-AMI products, see <a href=
         * "https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/ami-products.html#working-with-single-AMI-products"
         * >Working with single-AMI products</a>. Also, for more information about change types available for
         * container-based products, see <a href=
         * "https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/container-products.html#working-with-container-products"
         * >Working with container products</a>.
         * </p>
         * 
         * @param details
         *        This object contains details specific to the change type of the requested change. For more information
         *        about change types available for single-AMI products, see <a href=
         *        "https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/ami-products.html#working-with-single-AMI-products"
         *        >Working with single-AMI products</a>. Also, for more information about change types available for
         *        container-based products, see <a href=
         *        "https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/container-products.html#working-with-container-products"
         *        >Working with container products</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder details(String details);

        /**
         * <p>
         * Alternative field that accepts a JSON value instead of a string for <code>ChangeType</code> details. You can
         * use either <code>Details</code> or <code>DetailsDocument</code>, but not both.
         * </p>
         * 
         * @param detailsDocument
         *        Alternative field that accepts a JSON value instead of a string for <code>ChangeType</code> details.
         *        You can use either <code>Details</code> or <code>DetailsDocument</code>, but not both.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder detailsDocument(Document detailsDocument);

        /**
         * <p>
         * Optional name for the change.
         * </p>
         * 
         * @param changeName
         *        Optional name for the change.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder changeName(String changeName);
    }

    static final class BuilderImpl implements Builder {
        private String changeType;

        private Entity entity;

        private List<Tag> entityTags = DefaultSdkAutoConstructList.getInstance();

        private String details;

        private Document detailsDocument;

        private String changeName;

        private BuilderImpl() {
        }

        private BuilderImpl(Change model) {
            changeType(model.changeType);
            entity(model.entity);
            entityTags(model.entityTags);
            details(model.details);
            detailsDocument(model.detailsDocument);
            changeName(model.changeName);
        }

        public final String getChangeType() {
            return changeType;
        }

        public final void setChangeType(String changeType) {
            this.changeType = changeType;
        }

        @Override
        public final Builder changeType(String changeType) {
            this.changeType = changeType;
            return this;
        }

        public final Entity.Builder getEntity() {
            return entity != null ? entity.toBuilder() : null;
        }

        public final void setEntity(Entity.BuilderImpl entity) {
            this.entity = entity != null ? entity.build() : null;
        }

        @Override
        public final Builder entity(Entity entity) {
            this.entity = entity;
            return this;
        }

        public final List<Tag.Builder> getEntityTags() {
            List<Tag.Builder> result = TagListCopier.copyToBuilder(this.entityTags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setEntityTags(Collection<Tag.BuilderImpl> entityTags) {
            this.entityTags = TagListCopier.copyFromBuilder(entityTags);
        }

        @Override
        public final Builder entityTags(Collection<Tag> entityTags) {
            this.entityTags = TagListCopier.copy(entityTags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder entityTags(Tag... entityTags) {
            entityTags(Arrays.asList(entityTags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder entityTags(Consumer<Tag.Builder>... entityTags) {
            entityTags(Stream.of(entityTags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getDetails() {
            return details;
        }

        public final void setDetails(String details) {
            this.details = details;
        }

        @Override
        public final Builder details(String details) {
            this.details = details;
            return this;
        }

        public final Document getDetailsDocument() {
            return detailsDocument;
        }

        public final void setDetailsDocument(Document detailsDocument) {
            this.detailsDocument = detailsDocument;
        }

        @Override
        public final Builder detailsDocument(Document detailsDocument) {
            this.detailsDocument = detailsDocument;
            return this;
        }

        public final String getChangeName() {
            return changeName;
        }

        public final void setChangeName(String changeName) {
            this.changeName = changeName;
        }

        @Override
        public final Builder changeName(String changeName) {
            this.changeName = changeName;
            return this;
        }

        @Override
        public Change build() {
            return new Change(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
