/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.marketplacecatalog;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.marketplacecatalog.internal.MarketplaceCatalogServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.marketplacecatalog.model.AccessDeniedException;
import software.amazon.awssdk.services.marketplacecatalog.model.BatchDescribeEntitiesRequest;
import software.amazon.awssdk.services.marketplacecatalog.model.BatchDescribeEntitiesResponse;
import software.amazon.awssdk.services.marketplacecatalog.model.CancelChangeSetRequest;
import software.amazon.awssdk.services.marketplacecatalog.model.CancelChangeSetResponse;
import software.amazon.awssdk.services.marketplacecatalog.model.DeleteResourcePolicyRequest;
import software.amazon.awssdk.services.marketplacecatalog.model.DeleteResourcePolicyResponse;
import software.amazon.awssdk.services.marketplacecatalog.model.DescribeChangeSetRequest;
import software.amazon.awssdk.services.marketplacecatalog.model.DescribeChangeSetResponse;
import software.amazon.awssdk.services.marketplacecatalog.model.DescribeEntityRequest;
import software.amazon.awssdk.services.marketplacecatalog.model.DescribeEntityResponse;
import software.amazon.awssdk.services.marketplacecatalog.model.GetResourcePolicyRequest;
import software.amazon.awssdk.services.marketplacecatalog.model.GetResourcePolicyResponse;
import software.amazon.awssdk.services.marketplacecatalog.model.InternalServiceException;
import software.amazon.awssdk.services.marketplacecatalog.model.ListChangeSetsRequest;
import software.amazon.awssdk.services.marketplacecatalog.model.ListChangeSetsResponse;
import software.amazon.awssdk.services.marketplacecatalog.model.ListEntitiesRequest;
import software.amazon.awssdk.services.marketplacecatalog.model.ListEntitiesResponse;
import software.amazon.awssdk.services.marketplacecatalog.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.marketplacecatalog.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.marketplacecatalog.model.MarketplaceCatalogException;
import software.amazon.awssdk.services.marketplacecatalog.model.PutResourcePolicyRequest;
import software.amazon.awssdk.services.marketplacecatalog.model.PutResourcePolicyResponse;
import software.amazon.awssdk.services.marketplacecatalog.model.ResourceInUseException;
import software.amazon.awssdk.services.marketplacecatalog.model.ResourceNotFoundException;
import software.amazon.awssdk.services.marketplacecatalog.model.ResourceNotSupportedException;
import software.amazon.awssdk.services.marketplacecatalog.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.marketplacecatalog.model.StartChangeSetRequest;
import software.amazon.awssdk.services.marketplacecatalog.model.StartChangeSetResponse;
import software.amazon.awssdk.services.marketplacecatalog.model.TagResourceRequest;
import software.amazon.awssdk.services.marketplacecatalog.model.TagResourceResponse;
import software.amazon.awssdk.services.marketplacecatalog.model.ThrottlingException;
import software.amazon.awssdk.services.marketplacecatalog.model.UntagResourceRequest;
import software.amazon.awssdk.services.marketplacecatalog.model.UntagResourceResponse;
import software.amazon.awssdk.services.marketplacecatalog.model.ValidationException;
import software.amazon.awssdk.services.marketplacecatalog.transform.BatchDescribeEntitiesRequestMarshaller;
import software.amazon.awssdk.services.marketplacecatalog.transform.CancelChangeSetRequestMarshaller;
import software.amazon.awssdk.services.marketplacecatalog.transform.DeleteResourcePolicyRequestMarshaller;
import software.amazon.awssdk.services.marketplacecatalog.transform.DescribeChangeSetRequestMarshaller;
import software.amazon.awssdk.services.marketplacecatalog.transform.DescribeEntityRequestMarshaller;
import software.amazon.awssdk.services.marketplacecatalog.transform.GetResourcePolicyRequestMarshaller;
import software.amazon.awssdk.services.marketplacecatalog.transform.ListChangeSetsRequestMarshaller;
import software.amazon.awssdk.services.marketplacecatalog.transform.ListEntitiesRequestMarshaller;
import software.amazon.awssdk.services.marketplacecatalog.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.marketplacecatalog.transform.PutResourcePolicyRequestMarshaller;
import software.amazon.awssdk.services.marketplacecatalog.transform.StartChangeSetRequestMarshaller;
import software.amazon.awssdk.services.marketplacecatalog.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.marketplacecatalog.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link MarketplaceCatalogAsyncClient}.
 *
 * @see MarketplaceCatalogAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultMarketplaceCatalogAsyncClient implements MarketplaceCatalogAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultMarketplaceCatalogAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultMarketplaceCatalogAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Returns metadata and content for multiple entities.
     * </p>
     *
     * @param batchDescribeEntitiesRequest
     * @return A Java Future containing the result of the BatchDescribeEntities operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException Access is denied.</p>
     *         <p>
     *         HTTP status code: 403</li>
     *         <li>InternalServiceException There was an internal service exception.</p>
     *         <p>
     *         HTTP status code: 500</li>
     *         <li>ThrottlingException Too many requests.</p>
     *         <p>
     *         HTTP status code: 429</li>
     *         <li>ValidationException An error occurred during validation.</p>
     *         <p>
     *         HTTP status code: 422</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MarketplaceCatalogException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample MarketplaceCatalogAsyncClient.BatchDescribeEntities
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/marketplace-catalog-2018-09-17/BatchDescribeEntities"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchDescribeEntitiesResponse> batchDescribeEntities(
            BatchDescribeEntitiesRequest batchDescribeEntitiesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(batchDescribeEntitiesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchDescribeEntitiesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Marketplace Catalog");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchDescribeEntities");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchDescribeEntitiesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchDescribeEntitiesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BatchDescribeEntitiesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchDescribeEntitiesRequest, BatchDescribeEntitiesResponse>()
                            .withOperationName("BatchDescribeEntities").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new BatchDescribeEntitiesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(batchDescribeEntitiesRequest));
            CompletableFuture<BatchDescribeEntitiesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Used to cancel an open change request. Must be sent before the status of the request changes to
     * <code>APPLYING</code>, the final stage of completing your change request. You can describe a change during the
     * 60-day request history retention period for API calls.
     * </p>
     *
     * @param cancelChangeSetRequest
     * @return A Java Future containing the result of the CancelChangeSet operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceException There was an internal service exception.</p>
     *         <p>
     *         HTTP status code: 500</li>
     *         <li>AccessDeniedException Access is denied.</p>
     *         <p>
     *         HTTP status code: 403</li>
     *         <li>ValidationException An error occurred during validation.</p>
     *         <p>
     *         HTTP status code: 422</li>
     *         <li>ResourceNotFoundException The specified resource wasn't found.</p>
     *         <p>
     *         HTTP status code: 404</li>
     *         <li>ResourceInUseException The resource is currently in use.</li>
     *         <li>ThrottlingException Too many requests.</p>
     *         <p>
     *         HTTP status code: 429</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MarketplaceCatalogException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample MarketplaceCatalogAsyncClient.CancelChangeSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/marketplace-catalog-2018-09-17/CancelChangeSet"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CancelChangeSetResponse> cancelChangeSet(CancelChangeSetRequest cancelChangeSetRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(cancelChangeSetRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, cancelChangeSetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Marketplace Catalog");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CancelChangeSet");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CancelChangeSetResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CancelChangeSetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CancelChangeSetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CancelChangeSetRequest, CancelChangeSetResponse>()
                            .withOperationName("CancelChangeSet").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CancelChangeSetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(cancelChangeSetRequest));
            CompletableFuture<CancelChangeSetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a resource-based policy on an entity that is identified by its resource ARN.
     * </p>
     *
     * @param deleteResourcePolicyRequest
     * @return A Java Future containing the result of the DeleteResourcePolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceException There was an internal service exception.</p>
     *         <p>
     *         HTTP status code: 500</li>
     *         <li>AccessDeniedException Access is denied.</p>
     *         <p>
     *         HTTP status code: 403</li>
     *         <li>ValidationException An error occurred during validation.</p>
     *         <p>
     *         HTTP status code: 422</li>
     *         <li>ResourceNotFoundException The specified resource wasn't found.</p>
     *         <p>
     *         HTTP status code: 404</li>
     *         <li>ThrottlingException Too many requests.</p>
     *         <p>
     *         HTTP status code: 429</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MarketplaceCatalogException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample MarketplaceCatalogAsyncClient.DeleteResourcePolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/marketplace-catalog-2018-09-17/DeleteResourcePolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteResourcePolicyResponse> deleteResourcePolicy(
            DeleteResourcePolicyRequest deleteResourcePolicyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteResourcePolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteResourcePolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Marketplace Catalog");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteResourcePolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteResourcePolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteResourcePolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteResourcePolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteResourcePolicyRequest, DeleteResourcePolicyResponse>()
                            .withOperationName("DeleteResourcePolicy").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteResourcePolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteResourcePolicyRequest));
            CompletableFuture<DeleteResourcePolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides information about a given change set.
     * </p>
     *
     * @param describeChangeSetRequest
     * @return A Java Future containing the result of the DescribeChangeSet operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceException There was an internal service exception.</p>
     *         <p>
     *         HTTP status code: 500</li>
     *         <li>AccessDeniedException Access is denied.</p>
     *         <p>
     *         HTTP status code: 403</li>
     *         <li>ValidationException An error occurred during validation.</p>
     *         <p>
     *         HTTP status code: 422</li>
     *         <li>ResourceNotFoundException The specified resource wasn't found.</p>
     *         <p>
     *         HTTP status code: 404</li>
     *         <li>ThrottlingException Too many requests.</p>
     *         <p>
     *         HTTP status code: 429</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MarketplaceCatalogException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample MarketplaceCatalogAsyncClient.DescribeChangeSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/marketplace-catalog-2018-09-17/DescribeChangeSet"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeChangeSetResponse> describeChangeSet(DescribeChangeSetRequest describeChangeSetRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeChangeSetRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeChangeSetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Marketplace Catalog");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeChangeSet");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeChangeSetResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeChangeSetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeChangeSetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeChangeSetRequest, DescribeChangeSetResponse>()
                            .withOperationName("DescribeChangeSet").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeChangeSetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeChangeSetRequest));
            CompletableFuture<DescribeChangeSetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the metadata and content of the entity.
     * </p>
     *
     * @param describeEntityRequest
     * @return A Java Future containing the result of the DescribeEntity operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceException There was an internal service exception.</p>
     *         <p>
     *         HTTP status code: 500</li>
     *         <li>AccessDeniedException Access is denied.</p>
     *         <p>
     *         HTTP status code: 403</li>
     *         <li>ValidationException An error occurred during validation.</p>
     *         <p>
     *         HTTP status code: 422</li>
     *         <li>ResourceNotSupportedException Currently, the specified resource is not supported.</li>
     *         <li>ResourceNotFoundException The specified resource wasn't found.</p>
     *         <p>
     *         HTTP status code: 404</li>
     *         <li>ThrottlingException Too many requests.</p>
     *         <p>
     *         HTTP status code: 429</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MarketplaceCatalogException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample MarketplaceCatalogAsyncClient.DescribeEntity
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/marketplace-catalog-2018-09-17/DescribeEntity"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeEntityResponse> describeEntity(DescribeEntityRequest describeEntityRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeEntityRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeEntityRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Marketplace Catalog");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeEntity");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeEntityResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeEntityResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeEntityResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeEntityRequest, DescribeEntityResponse>()
                            .withOperationName("DescribeEntity").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeEntityRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeEntityRequest));
            CompletableFuture<DescribeEntityResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets a resource-based policy of an entity that is identified by its resource ARN.
     * </p>
     *
     * @param getResourcePolicyRequest
     * @return A Java Future containing the result of the GetResourcePolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceException There was an internal service exception.</p>
     *         <p>
     *         HTTP status code: 500</li>
     *         <li>AccessDeniedException Access is denied.</p>
     *         <p>
     *         HTTP status code: 403</li>
     *         <li>ValidationException An error occurred during validation.</p>
     *         <p>
     *         HTTP status code: 422</li>
     *         <li>ResourceNotFoundException The specified resource wasn't found.</p>
     *         <p>
     *         HTTP status code: 404</li>
     *         <li>ThrottlingException Too many requests.</p>
     *         <p>
     *         HTTP status code: 429</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MarketplaceCatalogException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample MarketplaceCatalogAsyncClient.GetResourcePolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/marketplace-catalog-2018-09-17/GetResourcePolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetResourcePolicyResponse> getResourcePolicy(GetResourcePolicyRequest getResourcePolicyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getResourcePolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getResourcePolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Marketplace Catalog");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResourcePolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetResourcePolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetResourcePolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetResourcePolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetResourcePolicyRequest, GetResourcePolicyResponse>()
                            .withOperationName("GetResourcePolicy").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetResourcePolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getResourcePolicyRequest));
            CompletableFuture<GetResourcePolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the list of change sets owned by the account being used to make the call. You can filter this list by
     * providing any combination of <code>entityId</code>, <code>ChangeSetName</code>, and status. If you provide more
     * than one filter, the API operation applies a logical AND between the filters.
     * </p>
     * <p>
     * You can describe a change during the 60-day request history retention period for API calls.
     * </p>
     *
     * @param listChangeSetsRequest
     * @return A Java Future containing the result of the ListChangeSets operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceException There was an internal service exception.</p>
     *         <p>
     *         HTTP status code: 500</li>
     *         <li>AccessDeniedException Access is denied.</p>
     *         <p>
     *         HTTP status code: 403</li>
     *         <li>ValidationException An error occurred during validation.</p>
     *         <p>
     *         HTTP status code: 422</li>
     *         <li>ThrottlingException Too many requests.</p>
     *         <p>
     *         HTTP status code: 429</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MarketplaceCatalogException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample MarketplaceCatalogAsyncClient.ListChangeSets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/marketplace-catalog-2018-09-17/ListChangeSets"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListChangeSetsResponse> listChangeSets(ListChangeSetsRequest listChangeSetsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listChangeSetsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listChangeSetsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Marketplace Catalog");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListChangeSets");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListChangeSetsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListChangeSetsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListChangeSetsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListChangeSetsRequest, ListChangeSetsResponse>()
                            .withOperationName("ListChangeSets").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListChangeSetsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listChangeSetsRequest));
            CompletableFuture<ListChangeSetsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides the list of entities of a given type.
     * </p>
     *
     * @param listEntitiesRequest
     * @return A Java Future containing the result of the ListEntities operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceException There was an internal service exception.</p>
     *         <p>
     *         HTTP status code: 500</li>
     *         <li>AccessDeniedException Access is denied.</p>
     *         <p>
     *         HTTP status code: 403</li>
     *         <li>ValidationException An error occurred during validation.</p>
     *         <p>
     *         HTTP status code: 422</li>
     *         <li>ResourceNotFoundException The specified resource wasn't found.</p>
     *         <p>
     *         HTTP status code: 404</li>
     *         <li>ThrottlingException Too many requests.</p>
     *         <p>
     *         HTTP status code: 429</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MarketplaceCatalogException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample MarketplaceCatalogAsyncClient.ListEntities
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/marketplace-catalog-2018-09-17/ListEntities"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListEntitiesResponse> listEntities(ListEntitiesRequest listEntitiesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listEntitiesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listEntitiesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Marketplace Catalog");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListEntities");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListEntitiesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListEntitiesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListEntitiesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListEntitiesRequest, ListEntitiesResponse>()
                            .withOperationName("ListEntities").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListEntitiesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listEntitiesRequest));
            CompletableFuture<ListEntitiesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all tags that have been added to a resource (either an <a
     * href="https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/welcome.html#catalog-api-entities"
     * >entity</a> or <a href=
     * "https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/welcome.html#working-with-change-sets"
     * >change set</a>).
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource wasn't found.</p>
     *         <p>
     *         HTTP status code: 404</li>
     *         <li>InternalServiceException There was an internal service exception.</p>
     *         <p>
     *         HTTP status code: 500</li>
     *         <li>AccessDeniedException Access is denied.</p>
     *         <p>
     *         HTTP status code: 403</li>
     *         <li>ValidationException An error occurred during validation.</p>
     *         <p>
     *         HTTP status code: 422</li>
     *         <li>ThrottlingException Too many requests.</p>
     *         <p>
     *         HTTP status code: 429</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MarketplaceCatalogException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample MarketplaceCatalogAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/marketplace-catalog-2018-09-17/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Marketplace Catalog");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Attaches a resource-based policy to an entity. Examples of an entity include: <code>AmiProduct</code> and
     * <code>ContainerProduct</code>.
     * </p>
     *
     * @param putResourcePolicyRequest
     * @return A Java Future containing the result of the PutResourcePolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceException There was an internal service exception.</p>
     *         <p>
     *         HTTP status code: 500</li>
     *         <li>AccessDeniedException Access is denied.</p>
     *         <p>
     *         HTTP status code: 403</li>
     *         <li>ValidationException An error occurred during validation.</p>
     *         <p>
     *         HTTP status code: 422</li>
     *         <li>ResourceNotFoundException The specified resource wasn't found.</p>
     *         <p>
     *         HTTP status code: 404</li>
     *         <li>ThrottlingException Too many requests.</p>
     *         <p>
     *         HTTP status code: 429</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MarketplaceCatalogException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample MarketplaceCatalogAsyncClient.PutResourcePolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/marketplace-catalog-2018-09-17/PutResourcePolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutResourcePolicyResponse> putResourcePolicy(PutResourcePolicyRequest putResourcePolicyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putResourcePolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putResourcePolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Marketplace Catalog");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutResourcePolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutResourcePolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutResourcePolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutResourcePolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutResourcePolicyRequest, PutResourcePolicyResponse>()
                            .withOperationName("PutResourcePolicy").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PutResourcePolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putResourcePolicyRequest));
            CompletableFuture<PutResourcePolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Allows you to request changes for your entities. Within a single <code>ChangeSet</code>, you can't start the same
     * change type against the same entity multiple times. Additionally, when a <code>ChangeSet</code> is running, all
     * the entities targeted by the different changes are locked until the change set has completed (either succeeded,
     * cancelled, or failed). If you try to start a change set containing a change against an entity that is already
     * locked, you will receive a <code>ResourceInUseException</code> error.
     * </p>
     * <p>
     * For example, you can't start the <code>ChangeSet</code> described in the <a href=
     * "https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/API_StartChangeSet.html#API_StartChangeSet_Examples"
     * >example</a> later in this topic because it contains two changes to run the same change type (
     * <code>AddRevisions</code>) against the same entity (<code>entity-id@1</code>).
     * </p>
     * <p>
     * For more information about working with change sets, see <a href=
     * "https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/welcome.html#working-with-change-sets">
     * Working with change sets</a>. For information about change types for single-AMI products, see <a href=
     * "https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/ami-products.html#working-with-single-AMI-products"
     * >Working with single-AMI products</a>. Also, for more information about change types available for
     * container-based products, see <a href=
     * "https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/container-products.html#working-with-container-products"
     * >Working with container products</a>.
     * </p>
     *
     * @param startChangeSetRequest
     * @return A Java Future containing the result of the StartChangeSet operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceException There was an internal service exception.</p>
     *         <p>
     *         HTTP status code: 500</li>
     *         <li>AccessDeniedException Access is denied.</p>
     *         <p>
     *         HTTP status code: 403</li>
     *         <li>ValidationException An error occurred during validation.</p>
     *         <p>
     *         HTTP status code: 422</li>
     *         <li>ResourceNotFoundException The specified resource wasn't found.</p>
     *         <p>
     *         HTTP status code: 404</li>
     *         <li>ResourceInUseException The resource is currently in use.</li>
     *         <li>ThrottlingException Too many requests.</p>
     *         <p>
     *         HTTP status code: 429</li>
     *         <li>ServiceQuotaExceededException The maximum number of open requests per account has been exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MarketplaceCatalogException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample MarketplaceCatalogAsyncClient.StartChangeSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/marketplace-catalog-2018-09-17/StartChangeSet"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartChangeSetResponse> startChangeSet(StartChangeSetRequest startChangeSetRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startChangeSetRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startChangeSetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Marketplace Catalog");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartChangeSet");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartChangeSetResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartChangeSetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartChangeSetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartChangeSetRequest, StartChangeSetResponse>()
                            .withOperationName("StartChangeSet").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StartChangeSetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(startChangeSetRequest));
            CompletableFuture<StartChangeSetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Tags a resource (either an <a
     * href="https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/welcome.html#catalog-api-entities"
     * >entity</a> or <a href=
     * "https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/welcome.html#working-with-change-sets"
     * >change set</a>).
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource wasn't found.</p>
     *         <p>
     *         HTTP status code: 404</li>
     *         <li>InternalServiceException There was an internal service exception.</p>
     *         <p>
     *         HTTP status code: 500</li>
     *         <li>AccessDeniedException Access is denied.</p>
     *         <p>
     *         HTTP status code: 403</li>
     *         <li>ValidationException An error occurred during validation.</p>
     *         <p>
     *         HTTP status code: 422</li>
     *         <li>ThrottlingException Too many requests.</p>
     *         <p>
     *         HTTP status code: 429</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MarketplaceCatalogException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample MarketplaceCatalogAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/marketplace-catalog-2018-09-17/TagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Marketplace Catalog");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes a tag or list of tags from a resource (either an <a
     * href="https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/welcome.html#catalog-api-entities"
     * >entity</a> or <a href=
     * "https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/welcome.html#working-with-change-sets"
     * >change set</a>).
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource wasn't found.</p>
     *         <p>
     *         HTTP status code: 404</li>
     *         <li>InternalServiceException There was an internal service exception.</p>
     *         <p>
     *         HTTP status code: 500</li>
     *         <li>AccessDeniedException Access is denied.</p>
     *         <p>
     *         HTTP status code: 403</li>
     *         <li>ValidationException An error occurred during validation.</p>
     *         <p>
     *         HTTP status code: 422</li>
     *         <li>ThrottlingException Too many requests.</p>
     *         <p>
     *         HTTP status code: 429</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MarketplaceCatalogException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample MarketplaceCatalogAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/marketplace-catalog-2018-09-17/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Marketplace Catalog");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final MarketplaceCatalogServiceClientConfiguration serviceClientConfiguration() {
        return new MarketplaceCatalogServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(MarketplaceCatalogException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceInUseException")
                                .exceptionBuilderSupplier(ResourceInUseException::builder).httpStatusCode(423).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotSupportedException")
                                .exceptionBuilderSupplier(ResourceNotSupportedException::builder).httpStatusCode(415).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(422).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServiceException")
                                .exceptionBuilderSupplier(InternalServiceException::builder).httpStatusCode(500).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        MarketplaceCatalogServiceClientConfigurationBuilder serviceConfigBuilder = new MarketplaceCatalogServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
