/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mediaconvert.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * Settings related to one audio tab on the MediaConvert console. In your job JSON, an instance of AudioDescription is
 * equivalent to one audio tab in the console. Usually, one audio tab corresponds to one output audio track. Depending
 * on how you set up your input audio selectors and whether you use audio selector groups, one audio tab can correspond
 * to a group of output audio tracks.
 */
@Generated("software.amazon.awssdk:codegen")
public final class AudioDescription implements SdkPojo, Serializable,
        ToCopyableBuilder<AudioDescription.Builder, AudioDescription> {
    private static final SdkField<AudioChannelTaggingSettings> AUDIO_CHANNEL_TAGGING_SETTINGS_FIELD = SdkField
            .<AudioChannelTaggingSettings> builder(MarshallingType.SDK_POJO)
            .memberName("AudioChannelTaggingSettings")
            .getter(getter(AudioDescription::audioChannelTaggingSettings))
            .setter(setter(Builder::audioChannelTaggingSettings))
            .constructor(AudioChannelTaggingSettings::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("audioChannelTaggingSettings")
                    .build()).build();

    private static final SdkField<AudioNormalizationSettings> AUDIO_NORMALIZATION_SETTINGS_FIELD = SdkField
            .<AudioNormalizationSettings> builder(MarshallingType.SDK_POJO)
            .memberName("AudioNormalizationSettings")
            .getter(getter(AudioDescription::audioNormalizationSettings))
            .setter(setter(Builder::audioNormalizationSettings))
            .constructor(AudioNormalizationSettings::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("audioNormalizationSettings").build())
            .build();

    private static final SdkField<String> AUDIO_SOURCE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AudioSourceName").getter(getter(AudioDescription::audioSourceName))
            .setter(setter(Builder::audioSourceName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("audioSourceName").build()).build();

    private static final SdkField<Integer> AUDIO_TYPE_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("AudioType").getter(getter(AudioDescription::audioType)).setter(setter(Builder::audioType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("audioType").build()).build();

    private static final SdkField<String> AUDIO_TYPE_CONTROL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AudioTypeControl").getter(getter(AudioDescription::audioTypeControlAsString))
            .setter(setter(Builder::audioTypeControl))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("audioTypeControl").build()).build();

    private static final SdkField<AudioCodecSettings> CODEC_SETTINGS_FIELD = SdkField
            .<AudioCodecSettings> builder(MarshallingType.SDK_POJO).memberName("CodecSettings")
            .getter(getter(AudioDescription::codecSettings)).setter(setter(Builder::codecSettings))
            .constructor(AudioCodecSettings::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("codecSettings").build()).build();

    private static final SdkField<String> CUSTOM_LANGUAGE_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CustomLanguageCode").getter(getter(AudioDescription::customLanguageCode))
            .setter(setter(Builder::customLanguageCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("customLanguageCode").build())
            .build();

    private static final SdkField<String> LANGUAGE_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LanguageCode").getter(getter(AudioDescription::languageCodeAsString))
            .setter(setter(Builder::languageCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("languageCode").build()).build();

    private static final SdkField<String> LANGUAGE_CODE_CONTROL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LanguageCodeControl").getter(getter(AudioDescription::languageCodeControlAsString))
            .setter(setter(Builder::languageCodeControl))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("languageCodeControl").build())
            .build();

    private static final SdkField<RemixSettings> REMIX_SETTINGS_FIELD = SdkField
            .<RemixSettings> builder(MarshallingType.SDK_POJO).memberName("RemixSettings")
            .getter(getter(AudioDescription::remixSettings)).setter(setter(Builder::remixSettings))
            .constructor(RemixSettings::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("remixSettings").build()).build();

    private static final SdkField<String> STREAM_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StreamName").getter(getter(AudioDescription::streamName)).setter(setter(Builder::streamName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("streamName").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            AUDIO_CHANNEL_TAGGING_SETTINGS_FIELD, AUDIO_NORMALIZATION_SETTINGS_FIELD, AUDIO_SOURCE_NAME_FIELD, AUDIO_TYPE_FIELD,
            AUDIO_TYPE_CONTROL_FIELD, CODEC_SETTINGS_FIELD, CUSTOM_LANGUAGE_CODE_FIELD, LANGUAGE_CODE_FIELD,
            LANGUAGE_CODE_CONTROL_FIELD, REMIX_SETTINGS_FIELD, STREAM_NAME_FIELD));

    private static final long serialVersionUID = 1L;

    private final AudioChannelTaggingSettings audioChannelTaggingSettings;

    private final AudioNormalizationSettings audioNormalizationSettings;

    private final String audioSourceName;

    private final Integer audioType;

    private final String audioTypeControl;

    private final AudioCodecSettings codecSettings;

    private final String customLanguageCode;

    private final String languageCode;

    private final String languageCodeControl;

    private final RemixSettings remixSettings;

    private final String streamName;

    private AudioDescription(BuilderImpl builder) {
        this.audioChannelTaggingSettings = builder.audioChannelTaggingSettings;
        this.audioNormalizationSettings = builder.audioNormalizationSettings;
        this.audioSourceName = builder.audioSourceName;
        this.audioType = builder.audioType;
        this.audioTypeControl = builder.audioTypeControl;
        this.codecSettings = builder.codecSettings;
        this.customLanguageCode = builder.customLanguageCode;
        this.languageCode = builder.languageCode;
        this.languageCodeControl = builder.languageCodeControl;
        this.remixSettings = builder.remixSettings;
        this.streamName = builder.streamName;
    }

    /**
     * When you mimic a multi-channel audio layout with multiple mono-channel tracks, you can tag each channel layout
     * manually. For example, you would tag the tracks that contain your left, right, and center audio with Left (L),
     * Right (R), and Center (C), respectively. When you don't specify a value, MediaConvert labels your track as Center
     * (C) by default. To use audio layout tagging, your output must be in a QuickTime (.mov) container; your audio
     * codec must be AAC, WAV, or AIFF; and you must set up your audio track to have only one channel.
     * 
     * @return When you mimic a multi-channel audio layout with multiple mono-channel tracks, you can tag each channel
     *         layout manually. For example, you would tag the tracks that contain your left, right, and center audio
     *         with Left (L), Right (R), and Center (C), respectively. When you don't specify a value, MediaConvert
     *         labels your track as Center (C) by default. To use audio layout tagging, your output must be in a
     *         QuickTime (.mov) container; your audio codec must be AAC, WAV, or AIFF; and you must set up your audio
     *         track to have only one channel.
     */
    public final AudioChannelTaggingSettings audioChannelTaggingSettings() {
        return audioChannelTaggingSettings;
    }

    /**
     * Advanced audio normalization settings. Ignore these settings unless you need to comply with a loudness standard.
     * 
     * @return Advanced audio normalization settings. Ignore these settings unless you need to comply with a loudness
     *         standard.
     */
    public final AudioNormalizationSettings audioNormalizationSettings() {
        return audioNormalizationSettings;
    }

    /**
     * Specifies which audio data to use from each input. In the simplest case, specify an
     * "Audio Selector":#inputs-audio_selector by name based on its order within each input. For example if you specify
     * "Audio Selector 3", then the third audio selector will be used from each input. If an input does not have an
     * "Audio Selector 3", then the audio selector marked as "default" in that input will be used. If there is no audio
     * selector marked as "default", silence will be inserted for the duration of that input. Alternatively, an
     * "Audio Selector Group":#inputs-audio_selector_group name may be specified, with similar default/silence behavior.
     * If no audio_source_name is specified, then "Audio Selector 1" will be chosen automatically.
     * 
     * @return Specifies which audio data to use from each input. In the simplest case, specify an
     *         "Audio Selector":#inputs-audio_selector by name based on its order within each input. For example if you
     *         specify "Audio Selector 3", then the third audio selector will be used from each input. If an input does
     *         not have an "Audio Selector 3", then the audio selector marked as "default" in that input will be used.
     *         If there is no audio selector marked as "default", silence will be inserted for the duration of that
     *         input. Alternatively, an "Audio Selector Group":#inputs-audio_selector_group name may be specified, with
     *         similar default/silence behavior. If no audio_source_name is specified, then "Audio Selector 1" will be
     *         chosen automatically.
     */
    public final String audioSourceName() {
        return audioSourceName;
    }

    /**
     * Applies only if Follow Input Audio Type is unchecked (false). A number between 0 and 255. The following are
     * defined in ISO-IEC 13818-1: 0 = Undefined, 1 = Clean Effects, 2 = Hearing Impaired, 3 = Visually Impaired
     * Commentary, 4-255 = Reserved.
     * 
     * @return Applies only if Follow Input Audio Type is unchecked (false). A number between 0 and 255. The following
     *         are defined in ISO-IEC 13818-1: 0 = Undefined, 1 = Clean Effects, 2 = Hearing Impaired, 3 = Visually
     *         Impaired Commentary, 4-255 = Reserved.
     */
    public final Integer audioType() {
        return audioType;
    }

    /**
     * When set to FOLLOW_INPUT, if the input contains an ISO 639 audio_type, then that value is passed through to the
     * output. If the input contains no ISO 639 audio_type, the value in Audio Type is included in the output. Otherwise
     * the value in Audio Type is included in the output. Note that this field and audioType are both ignored if
     * audioDescriptionBroadcasterMix is set to BROADCASTER_MIXED_AD.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #audioTypeControl}
     * will return {@link AudioTypeControl#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #audioTypeControlAsString}.
     * </p>
     * 
     * @return When set to FOLLOW_INPUT, if the input contains an ISO 639 audio_type, then that value is passed through
     *         to the output. If the input contains no ISO 639 audio_type, the value in Audio Type is included in the
     *         output. Otherwise the value in Audio Type is included in the output. Note that this field and audioType
     *         are both ignored if audioDescriptionBroadcasterMix is set to BROADCASTER_MIXED_AD.
     * @see AudioTypeControl
     */
    public final AudioTypeControl audioTypeControl() {
        return AudioTypeControl.fromValue(audioTypeControl);
    }

    /**
     * When set to FOLLOW_INPUT, if the input contains an ISO 639 audio_type, then that value is passed through to the
     * output. If the input contains no ISO 639 audio_type, the value in Audio Type is included in the output. Otherwise
     * the value in Audio Type is included in the output. Note that this field and audioType are both ignored if
     * audioDescriptionBroadcasterMix is set to BROADCASTER_MIXED_AD.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #audioTypeControl}
     * will return {@link AudioTypeControl#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #audioTypeControlAsString}.
     * </p>
     * 
     * @return When set to FOLLOW_INPUT, if the input contains an ISO 639 audio_type, then that value is passed through
     *         to the output. If the input contains no ISO 639 audio_type, the value in Audio Type is included in the
     *         output. Otherwise the value in Audio Type is included in the output. Note that this field and audioType
     *         are both ignored if audioDescriptionBroadcasterMix is set to BROADCASTER_MIXED_AD.
     * @see AudioTypeControl
     */
    public final String audioTypeControlAsString() {
        return audioTypeControl;
    }

    /**
     * Settings related to audio encoding. The settings in this group vary depending on the value that you choose for
     * your audio codec.
     * 
     * @return Settings related to audio encoding. The settings in this group vary depending on the value that you
     *         choose for your audio codec.
     */
    public final AudioCodecSettings codecSettings() {
        return codecSettings;
    }

    /**
     * Specify the language for this audio output track. The service puts this language code into your output audio
     * track when you set Language code control (AudioLanguageCodeControl) to Use configured (USE_CONFIGURED). The
     * service also uses your specified custom language code when you set Language code control
     * (AudioLanguageCodeControl) to Follow input (FOLLOW_INPUT), but your input file doesn't specify a language code.
     * For all outputs, you can use an ISO 639-2 or ISO 639-3 code. For streaming outputs, you can also use any other
     * code in the full RFC-5646 specification. Streaming outputs are those that are in one of the following output
     * groups: CMAF, DASH ISO, Apple HLS, or Microsoft Smooth Streaming.
     * 
     * @return Specify the language for this audio output track. The service puts this language code into your output
     *         audio track when you set Language code control (AudioLanguageCodeControl) to Use configured
     *         (USE_CONFIGURED). The service also uses your specified custom language code when you set Language code
     *         control (AudioLanguageCodeControl) to Follow input (FOLLOW_INPUT), but your input file doesn't specify a
     *         language code. For all outputs, you can use an ISO 639-2 or ISO 639-3 code. For streaming outputs, you
     *         can also use any other code in the full RFC-5646 specification. Streaming outputs are those that are in
     *         one of the following output groups: CMAF, DASH ISO, Apple HLS, or Microsoft Smooth Streaming.
     */
    public final String customLanguageCode() {
        return customLanguageCode;
    }

    /**
     * Indicates the language of the audio output track. The ISO 639 language specified in the 'Language Code' drop down
     * will be used when 'Follow Input Language Code' is not selected or when 'Follow Input Language Code' is selected
     * but there is no ISO 639 language code specified by the input.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #languageCode} will
     * return {@link LanguageCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #languageCodeAsString}.
     * </p>
     * 
     * @return Indicates the language of the audio output track. The ISO 639 language specified in the 'Language Code'
     *         drop down will be used when 'Follow Input Language Code' is not selected or when 'Follow Input Language
     *         Code' is selected but there is no ISO 639 language code specified by the input.
     * @see LanguageCode
     */
    public final LanguageCode languageCode() {
        return LanguageCode.fromValue(languageCode);
    }

    /**
     * Indicates the language of the audio output track. The ISO 639 language specified in the 'Language Code' drop down
     * will be used when 'Follow Input Language Code' is not selected or when 'Follow Input Language Code' is selected
     * but there is no ISO 639 language code specified by the input.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #languageCode} will
     * return {@link LanguageCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #languageCodeAsString}.
     * </p>
     * 
     * @return Indicates the language of the audio output track. The ISO 639 language specified in the 'Language Code'
     *         drop down will be used when 'Follow Input Language Code' is not selected or when 'Follow Input Language
     *         Code' is selected but there is no ISO 639 language code specified by the input.
     * @see LanguageCode
     */
    public final String languageCodeAsString() {
        return languageCode;
    }

    /**
     * Specify which source for language code takes precedence for this audio track. When you choose Follow input
     * (FOLLOW_INPUT), the service uses the language code from the input track if it's present. If there's no languge
     * code on the input track, the service uses the code that you specify in the setting Language code (languageCode or
     * customLanguageCode). When you choose Use configured (USE_CONFIGURED), the service uses the language code that you
     * specify.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #languageCodeControl} will return {@link AudioLanguageCodeControl#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #languageCodeControlAsString}.
     * </p>
     * 
     * @return Specify which source for language code takes precedence for this audio track. When you choose Follow
     *         input (FOLLOW_INPUT), the service uses the language code from the input track if it's present. If there's
     *         no languge code on the input track, the service uses the code that you specify in the setting Language
     *         code (languageCode or customLanguageCode). When you choose Use configured (USE_CONFIGURED), the service
     *         uses the language code that you specify.
     * @see AudioLanguageCodeControl
     */
    public final AudioLanguageCodeControl languageCodeControl() {
        return AudioLanguageCodeControl.fromValue(languageCodeControl);
    }

    /**
     * Specify which source for language code takes precedence for this audio track. When you choose Follow input
     * (FOLLOW_INPUT), the service uses the language code from the input track if it's present. If there's no languge
     * code on the input track, the service uses the code that you specify in the setting Language code (languageCode or
     * customLanguageCode). When you choose Use configured (USE_CONFIGURED), the service uses the language code that you
     * specify.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #languageCodeControl} will return {@link AudioLanguageCodeControl#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #languageCodeControlAsString}.
     * </p>
     * 
     * @return Specify which source for language code takes precedence for this audio track. When you choose Follow
     *         input (FOLLOW_INPUT), the service uses the language code from the input track if it's present. If there's
     *         no languge code on the input track, the service uses the code that you specify in the setting Language
     *         code (languageCode or customLanguageCode). When you choose Use configured (USE_CONFIGURED), the service
     *         uses the language code that you specify.
     * @see AudioLanguageCodeControl
     */
    public final String languageCodeControlAsString() {
        return languageCodeControl;
    }

    /**
     * Advanced audio remixing settings.
     * 
     * @return Advanced audio remixing settings.
     */
    public final RemixSettings remixSettings() {
        return remixSettings;
    }

    /**
     * Specify a label for this output audio stream. For example, "English", "Director commentary", or "track_2". For
     * streaming outputs, MediaConvert passes this information into destination manifests for display on the
     * end-viewer's player device. For outputs in other output groups, the service ignores this setting.
     * 
     * @return Specify a label for this output audio stream. For example, "English", "Director commentary", or
     *         "track_2". For streaming outputs, MediaConvert passes this information into destination manifests for
     *         display on the end-viewer's player device. For outputs in other output groups, the service ignores this
     *         setting.
     */
    public final String streamName() {
        return streamName;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(audioChannelTaggingSettings());
        hashCode = 31 * hashCode + Objects.hashCode(audioNormalizationSettings());
        hashCode = 31 * hashCode + Objects.hashCode(audioSourceName());
        hashCode = 31 * hashCode + Objects.hashCode(audioType());
        hashCode = 31 * hashCode + Objects.hashCode(audioTypeControlAsString());
        hashCode = 31 * hashCode + Objects.hashCode(codecSettings());
        hashCode = 31 * hashCode + Objects.hashCode(customLanguageCode());
        hashCode = 31 * hashCode + Objects.hashCode(languageCodeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(languageCodeControlAsString());
        hashCode = 31 * hashCode + Objects.hashCode(remixSettings());
        hashCode = 31 * hashCode + Objects.hashCode(streamName());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AudioDescription)) {
            return false;
        }
        AudioDescription other = (AudioDescription) obj;
        return Objects.equals(audioChannelTaggingSettings(), other.audioChannelTaggingSettings())
                && Objects.equals(audioNormalizationSettings(), other.audioNormalizationSettings())
                && Objects.equals(audioSourceName(), other.audioSourceName()) && Objects.equals(audioType(), other.audioType())
                && Objects.equals(audioTypeControlAsString(), other.audioTypeControlAsString())
                && Objects.equals(codecSettings(), other.codecSettings())
                && Objects.equals(customLanguageCode(), other.customLanguageCode())
                && Objects.equals(languageCodeAsString(), other.languageCodeAsString())
                && Objects.equals(languageCodeControlAsString(), other.languageCodeControlAsString())
                && Objects.equals(remixSettings(), other.remixSettings()) && Objects.equals(streamName(), other.streamName());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AudioDescription").add("AudioChannelTaggingSettings", audioChannelTaggingSettings())
                .add("AudioNormalizationSettings", audioNormalizationSettings()).add("AudioSourceName", audioSourceName())
                .add("AudioType", audioType()).add("AudioTypeControl", audioTypeControlAsString())
                .add("CodecSettings", codecSettings()).add("CustomLanguageCode", customLanguageCode())
                .add("LanguageCode", languageCodeAsString()).add("LanguageCodeControl", languageCodeControlAsString())
                .add("RemixSettings", remixSettings()).add("StreamName", streamName()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AudioChannelTaggingSettings":
            return Optional.ofNullable(clazz.cast(audioChannelTaggingSettings()));
        case "AudioNormalizationSettings":
            return Optional.ofNullable(clazz.cast(audioNormalizationSettings()));
        case "AudioSourceName":
            return Optional.ofNullable(clazz.cast(audioSourceName()));
        case "AudioType":
            return Optional.ofNullable(clazz.cast(audioType()));
        case "AudioTypeControl":
            return Optional.ofNullable(clazz.cast(audioTypeControlAsString()));
        case "CodecSettings":
            return Optional.ofNullable(clazz.cast(codecSettings()));
        case "CustomLanguageCode":
            return Optional.ofNullable(clazz.cast(customLanguageCode()));
        case "LanguageCode":
            return Optional.ofNullable(clazz.cast(languageCodeAsString()));
        case "LanguageCodeControl":
            return Optional.ofNullable(clazz.cast(languageCodeControlAsString()));
        case "RemixSettings":
            return Optional.ofNullable(clazz.cast(remixSettings()));
        case "StreamName":
            return Optional.ofNullable(clazz.cast(streamName()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<AudioDescription, T> g) {
        return obj -> g.apply((AudioDescription) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AudioDescription> {
        /**
         * When you mimic a multi-channel audio layout with multiple mono-channel tracks, you can tag each channel
         * layout manually. For example, you would tag the tracks that contain your left, right, and center audio with
         * Left (L), Right (R), and Center (C), respectively. When you don't specify a value, MediaConvert labels your
         * track as Center (C) by default. To use audio layout tagging, your output must be in a QuickTime (.mov)
         * container; your audio codec must be AAC, WAV, or AIFF; and you must set up your audio track to have only one
         * channel.
         * 
         * @param audioChannelTaggingSettings
         *        When you mimic a multi-channel audio layout with multiple mono-channel tracks, you can tag each
         *        channel layout manually. For example, you would tag the tracks that contain your left, right, and
         *        center audio with Left (L), Right (R), and Center (C), respectively. When you don't specify a value,
         *        MediaConvert labels your track as Center (C) by default. To use audio layout tagging, your output must
         *        be in a QuickTime (.mov) container; your audio codec must be AAC, WAV, or AIFF; and you must set up
         *        your audio track to have only one channel.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder audioChannelTaggingSettings(AudioChannelTaggingSettings audioChannelTaggingSettings);

        /**
         * When you mimic a multi-channel audio layout with multiple mono-channel tracks, you can tag each channel
         * layout manually. For example, you would tag the tracks that contain your left, right, and center audio with
         * Left (L), Right (R), and Center (C), respectively. When you don't specify a value, MediaConvert labels your
         * track as Center (C) by default. To use audio layout tagging, your output must be in a QuickTime (.mov)
         * container; your audio codec must be AAC, WAV, or AIFF; and you must set up your audio track to have only one
         * channel. This is a convenience that creates an instance of the {@link AudioChannelTaggingSettings.Builder}
         * avoiding the need to create one manually via {@link AudioChannelTaggingSettings#builder()}.
         *
         * When the {@link Consumer} completes, {@link AudioChannelTaggingSettings.Builder#build()} is called
         * immediately and its result is passed to {@link #audioChannelTaggingSettings(AudioChannelTaggingSettings)}.
         * 
         * @param audioChannelTaggingSettings
         *        a consumer that will call methods on {@link AudioChannelTaggingSettings.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #audioChannelTaggingSettings(AudioChannelTaggingSettings)
         */
        default Builder audioChannelTaggingSettings(Consumer<AudioChannelTaggingSettings.Builder> audioChannelTaggingSettings) {
            return audioChannelTaggingSettings(AudioChannelTaggingSettings.builder().applyMutation(audioChannelTaggingSettings)
                    .build());
        }

        /**
         * Advanced audio normalization settings. Ignore these settings unless you need to comply with a loudness
         * standard.
         * 
         * @param audioNormalizationSettings
         *        Advanced audio normalization settings. Ignore these settings unless you need to comply with a loudness
         *        standard.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder audioNormalizationSettings(AudioNormalizationSettings audioNormalizationSettings);

        /**
         * Advanced audio normalization settings. Ignore these settings unless you need to comply with a loudness
         * standard. This is a convenience that creates an instance of the {@link AudioNormalizationSettings.Builder}
         * avoiding the need to create one manually via {@link AudioNormalizationSettings#builder()}.
         *
         * When the {@link Consumer} completes, {@link AudioNormalizationSettings.Builder#build()} is called immediately
         * and its result is passed to {@link #audioNormalizationSettings(AudioNormalizationSettings)}.
         * 
         * @param audioNormalizationSettings
         *        a consumer that will call methods on {@link AudioNormalizationSettings.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #audioNormalizationSettings(AudioNormalizationSettings)
         */
        default Builder audioNormalizationSettings(Consumer<AudioNormalizationSettings.Builder> audioNormalizationSettings) {
            return audioNormalizationSettings(AudioNormalizationSettings.builder().applyMutation(audioNormalizationSettings)
                    .build());
        }

        /**
         * Specifies which audio data to use from each input. In the simplest case, specify an
         * "Audio Selector":#inputs-audio_selector by name based on its order within each input. For example if you
         * specify "Audio Selector 3", then the third audio selector will be used from each input. If an input does not
         * have an "Audio Selector 3", then the audio selector marked as "default" in that input will be used. If there
         * is no audio selector marked as "default", silence will be inserted for the duration of that input.
         * Alternatively, an "Audio Selector Group":#inputs-audio_selector_group name may be specified, with similar
         * default/silence behavior. If no audio_source_name is specified, then "Audio Selector 1" will be chosen
         * automatically.
         * 
         * @param audioSourceName
         *        Specifies which audio data to use from each input. In the simplest case, specify an
         *        "Audio Selector":#inputs-audio_selector by name based on its order within each input. For example if
         *        you specify "Audio Selector 3", then the third audio selector will be used from each input. If an
         *        input does not have an "Audio Selector 3", then the audio selector marked as "default" in that input
         *        will be used. If there is no audio selector marked as "default", silence will be inserted for the
         *        duration of that input. Alternatively, an "Audio Selector Group":#inputs-audio_selector_group name may
         *        be specified, with similar default/silence behavior. If no audio_source_name is specified, then
         *        "Audio Selector 1" will be chosen automatically.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder audioSourceName(String audioSourceName);

        /**
         * Applies only if Follow Input Audio Type is unchecked (false). A number between 0 and 255. The following are
         * defined in ISO-IEC 13818-1: 0 = Undefined, 1 = Clean Effects, 2 = Hearing Impaired, 3 = Visually Impaired
         * Commentary, 4-255 = Reserved.
         * 
         * @param audioType
         *        Applies only if Follow Input Audio Type is unchecked (false). A number between 0 and 255. The
         *        following are defined in ISO-IEC 13818-1: 0 = Undefined, 1 = Clean Effects, 2 = Hearing Impaired, 3 =
         *        Visually Impaired Commentary, 4-255 = Reserved.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder audioType(Integer audioType);

        /**
         * When set to FOLLOW_INPUT, if the input contains an ISO 639 audio_type, then that value is passed through to
         * the output. If the input contains no ISO 639 audio_type, the value in Audio Type is included in the output.
         * Otherwise the value in Audio Type is included in the output. Note that this field and audioType are both
         * ignored if audioDescriptionBroadcasterMix is set to BROADCASTER_MIXED_AD.
         * 
         * @param audioTypeControl
         *        When set to FOLLOW_INPUT, if the input contains an ISO 639 audio_type, then that value is passed
         *        through to the output. If the input contains no ISO 639 audio_type, the value in Audio Type is
         *        included in the output. Otherwise the value in Audio Type is included in the output. Note that this
         *        field and audioType are both ignored if audioDescriptionBroadcasterMix is set to BROADCASTER_MIXED_AD.
         * @see AudioTypeControl
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AudioTypeControl
         */
        Builder audioTypeControl(String audioTypeControl);

        /**
         * When set to FOLLOW_INPUT, if the input contains an ISO 639 audio_type, then that value is passed through to
         * the output. If the input contains no ISO 639 audio_type, the value in Audio Type is included in the output.
         * Otherwise the value in Audio Type is included in the output. Note that this field and audioType are both
         * ignored if audioDescriptionBroadcasterMix is set to BROADCASTER_MIXED_AD.
         * 
         * @param audioTypeControl
         *        When set to FOLLOW_INPUT, if the input contains an ISO 639 audio_type, then that value is passed
         *        through to the output. If the input contains no ISO 639 audio_type, the value in Audio Type is
         *        included in the output. Otherwise the value in Audio Type is included in the output. Note that this
         *        field and audioType are both ignored if audioDescriptionBroadcasterMix is set to BROADCASTER_MIXED_AD.
         * @see AudioTypeControl
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AudioTypeControl
         */
        Builder audioTypeControl(AudioTypeControl audioTypeControl);

        /**
         * Settings related to audio encoding. The settings in this group vary depending on the value that you choose
         * for your audio codec.
         * 
         * @param codecSettings
         *        Settings related to audio encoding. The settings in this group vary depending on the value that you
         *        choose for your audio codec.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder codecSettings(AudioCodecSettings codecSettings);

        /**
         * Settings related to audio encoding. The settings in this group vary depending on the value that you choose
         * for your audio codec. This is a convenience that creates an instance of the
         * {@link AudioCodecSettings.Builder} avoiding the need to create one manually via
         * {@link AudioCodecSettings#builder()}.
         *
         * When the {@link Consumer} completes, {@link AudioCodecSettings.Builder#build()} is called immediately and its
         * result is passed to {@link #codecSettings(AudioCodecSettings)}.
         * 
         * @param codecSettings
         *        a consumer that will call methods on {@link AudioCodecSettings.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #codecSettings(AudioCodecSettings)
         */
        default Builder codecSettings(Consumer<AudioCodecSettings.Builder> codecSettings) {
            return codecSettings(AudioCodecSettings.builder().applyMutation(codecSettings).build());
        }

        /**
         * Specify the language for this audio output track. The service puts this language code into your output audio
         * track when you set Language code control (AudioLanguageCodeControl) to Use configured (USE_CONFIGURED). The
         * service also uses your specified custom language code when you set Language code control
         * (AudioLanguageCodeControl) to Follow input (FOLLOW_INPUT), but your input file doesn't specify a language
         * code. For all outputs, you can use an ISO 639-2 or ISO 639-3 code. For streaming outputs, you can also use
         * any other code in the full RFC-5646 specification. Streaming outputs are those that are in one of the
         * following output groups: CMAF, DASH ISO, Apple HLS, or Microsoft Smooth Streaming.
         * 
         * @param customLanguageCode
         *        Specify the language for this audio output track. The service puts this language code into your output
         *        audio track when you set Language code control (AudioLanguageCodeControl) to Use configured
         *        (USE_CONFIGURED). The service also uses your specified custom language code when you set Language code
         *        control (AudioLanguageCodeControl) to Follow input (FOLLOW_INPUT), but your input file doesn't specify
         *        a language code. For all outputs, you can use an ISO 639-2 or ISO 639-3 code. For streaming outputs,
         *        you can also use any other code in the full RFC-5646 specification. Streaming outputs are those that
         *        are in one of the following output groups: CMAF, DASH ISO, Apple HLS, or Microsoft Smooth Streaming.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder customLanguageCode(String customLanguageCode);

        /**
         * Indicates the language of the audio output track. The ISO 639 language specified in the 'Language Code' drop
         * down will be used when 'Follow Input Language Code' is not selected or when 'Follow Input Language Code' is
         * selected but there is no ISO 639 language code specified by the input.
         * 
         * @param languageCode
         *        Indicates the language of the audio output track. The ISO 639 language specified in the 'Language
         *        Code' drop down will be used when 'Follow Input Language Code' is not selected or when 'Follow Input
         *        Language Code' is selected but there is no ISO 639 language code specified by the input.
         * @see LanguageCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LanguageCode
         */
        Builder languageCode(String languageCode);

        /**
         * Indicates the language of the audio output track. The ISO 639 language specified in the 'Language Code' drop
         * down will be used when 'Follow Input Language Code' is not selected or when 'Follow Input Language Code' is
         * selected but there is no ISO 639 language code specified by the input.
         * 
         * @param languageCode
         *        Indicates the language of the audio output track. The ISO 639 language specified in the 'Language
         *        Code' drop down will be used when 'Follow Input Language Code' is not selected or when 'Follow Input
         *        Language Code' is selected but there is no ISO 639 language code specified by the input.
         * @see LanguageCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LanguageCode
         */
        Builder languageCode(LanguageCode languageCode);

        /**
         * Specify which source for language code takes precedence for this audio track. When you choose Follow input
         * (FOLLOW_INPUT), the service uses the language code from the input track if it's present. If there's no
         * languge code on the input track, the service uses the code that you specify in the setting Language code
         * (languageCode or customLanguageCode). When you choose Use configured (USE_CONFIGURED), the service uses the
         * language code that you specify.
         * 
         * @param languageCodeControl
         *        Specify which source for language code takes precedence for this audio track. When you choose Follow
         *        input (FOLLOW_INPUT), the service uses the language code from the input track if it's present. If
         *        there's no languge code on the input track, the service uses the code that you specify in the setting
         *        Language code (languageCode or customLanguageCode). When you choose Use configured (USE_CONFIGURED),
         *        the service uses the language code that you specify.
         * @see AudioLanguageCodeControl
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AudioLanguageCodeControl
         */
        Builder languageCodeControl(String languageCodeControl);

        /**
         * Specify which source for language code takes precedence for this audio track. When you choose Follow input
         * (FOLLOW_INPUT), the service uses the language code from the input track if it's present. If there's no
         * languge code on the input track, the service uses the code that you specify in the setting Language code
         * (languageCode or customLanguageCode). When you choose Use configured (USE_CONFIGURED), the service uses the
         * language code that you specify.
         * 
         * @param languageCodeControl
         *        Specify which source for language code takes precedence for this audio track. When you choose Follow
         *        input (FOLLOW_INPUT), the service uses the language code from the input track if it's present. If
         *        there's no languge code on the input track, the service uses the code that you specify in the setting
         *        Language code (languageCode or customLanguageCode). When you choose Use configured (USE_CONFIGURED),
         *        the service uses the language code that you specify.
         * @see AudioLanguageCodeControl
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AudioLanguageCodeControl
         */
        Builder languageCodeControl(AudioLanguageCodeControl languageCodeControl);

        /**
         * Advanced audio remixing settings.
         * 
         * @param remixSettings
         *        Advanced audio remixing settings.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder remixSettings(RemixSettings remixSettings);

        /**
         * Advanced audio remixing settings. This is a convenience that creates an instance of the
         * {@link RemixSettings.Builder} avoiding the need to create one manually via {@link RemixSettings#builder()}.
         *
         * When the {@link Consumer} completes, {@link RemixSettings.Builder#build()} is called immediately and its
         * result is passed to {@link #remixSettings(RemixSettings)}.
         * 
         * @param remixSettings
         *        a consumer that will call methods on {@link RemixSettings.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #remixSettings(RemixSettings)
         */
        default Builder remixSettings(Consumer<RemixSettings.Builder> remixSettings) {
            return remixSettings(RemixSettings.builder().applyMutation(remixSettings).build());
        }

        /**
         * Specify a label for this output audio stream. For example, "English", "Director commentary", or "track_2".
         * For streaming outputs, MediaConvert passes this information into destination manifests for display on the
         * end-viewer's player device. For outputs in other output groups, the service ignores this setting.
         * 
         * @param streamName
         *        Specify a label for this output audio stream. For example, "English", "Director commentary", or
         *        "track_2". For streaming outputs, MediaConvert passes this information into destination manifests for
         *        display on the end-viewer's player device. For outputs in other output groups, the service ignores
         *        this setting.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder streamName(String streamName);
    }

    static final class BuilderImpl implements Builder {
        private AudioChannelTaggingSettings audioChannelTaggingSettings;

        private AudioNormalizationSettings audioNormalizationSettings;

        private String audioSourceName;

        private Integer audioType;

        private String audioTypeControl;

        private AudioCodecSettings codecSettings;

        private String customLanguageCode;

        private String languageCode;

        private String languageCodeControl;

        private RemixSettings remixSettings;

        private String streamName;

        private BuilderImpl() {
        }

        private BuilderImpl(AudioDescription model) {
            audioChannelTaggingSettings(model.audioChannelTaggingSettings);
            audioNormalizationSettings(model.audioNormalizationSettings);
            audioSourceName(model.audioSourceName);
            audioType(model.audioType);
            audioTypeControl(model.audioTypeControl);
            codecSettings(model.codecSettings);
            customLanguageCode(model.customLanguageCode);
            languageCode(model.languageCode);
            languageCodeControl(model.languageCodeControl);
            remixSettings(model.remixSettings);
            streamName(model.streamName);
        }

        public final AudioChannelTaggingSettings.Builder getAudioChannelTaggingSettings() {
            return audioChannelTaggingSettings != null ? audioChannelTaggingSettings.toBuilder() : null;
        }

        @Override
        public final Builder audioChannelTaggingSettings(AudioChannelTaggingSettings audioChannelTaggingSettings) {
            this.audioChannelTaggingSettings = audioChannelTaggingSettings;
            return this;
        }

        public final void setAudioChannelTaggingSettings(AudioChannelTaggingSettings.BuilderImpl audioChannelTaggingSettings) {
            this.audioChannelTaggingSettings = audioChannelTaggingSettings != null ? audioChannelTaggingSettings.build() : null;
        }

        public final AudioNormalizationSettings.Builder getAudioNormalizationSettings() {
            return audioNormalizationSettings != null ? audioNormalizationSettings.toBuilder() : null;
        }

        @Override
        public final Builder audioNormalizationSettings(AudioNormalizationSettings audioNormalizationSettings) {
            this.audioNormalizationSettings = audioNormalizationSettings;
            return this;
        }

        public final void setAudioNormalizationSettings(AudioNormalizationSettings.BuilderImpl audioNormalizationSettings) {
            this.audioNormalizationSettings = audioNormalizationSettings != null ? audioNormalizationSettings.build() : null;
        }

        public final String getAudioSourceName() {
            return audioSourceName;
        }

        @Override
        public final Builder audioSourceName(String audioSourceName) {
            this.audioSourceName = audioSourceName;
            return this;
        }

        public final void setAudioSourceName(String audioSourceName) {
            this.audioSourceName = audioSourceName;
        }

        public final Integer getAudioType() {
            return audioType;
        }

        @Override
        public final Builder audioType(Integer audioType) {
            this.audioType = audioType;
            return this;
        }

        public final void setAudioType(Integer audioType) {
            this.audioType = audioType;
        }

        public final String getAudioTypeControl() {
            return audioTypeControl;
        }

        @Override
        public final Builder audioTypeControl(String audioTypeControl) {
            this.audioTypeControl = audioTypeControl;
            return this;
        }

        @Override
        public final Builder audioTypeControl(AudioTypeControl audioTypeControl) {
            this.audioTypeControl(audioTypeControl == null ? null : audioTypeControl.toString());
            return this;
        }

        public final void setAudioTypeControl(String audioTypeControl) {
            this.audioTypeControl = audioTypeControl;
        }

        public final AudioCodecSettings.Builder getCodecSettings() {
            return codecSettings != null ? codecSettings.toBuilder() : null;
        }

        @Override
        public final Builder codecSettings(AudioCodecSettings codecSettings) {
            this.codecSettings = codecSettings;
            return this;
        }

        public final void setCodecSettings(AudioCodecSettings.BuilderImpl codecSettings) {
            this.codecSettings = codecSettings != null ? codecSettings.build() : null;
        }

        public final String getCustomLanguageCode() {
            return customLanguageCode;
        }

        @Override
        public final Builder customLanguageCode(String customLanguageCode) {
            this.customLanguageCode = customLanguageCode;
            return this;
        }

        public final void setCustomLanguageCode(String customLanguageCode) {
            this.customLanguageCode = customLanguageCode;
        }

        public final String getLanguageCode() {
            return languageCode;
        }

        @Override
        public final Builder languageCode(String languageCode) {
            this.languageCode = languageCode;
            return this;
        }

        @Override
        public final Builder languageCode(LanguageCode languageCode) {
            this.languageCode(languageCode == null ? null : languageCode.toString());
            return this;
        }

        public final void setLanguageCode(String languageCode) {
            this.languageCode = languageCode;
        }

        public final String getLanguageCodeControl() {
            return languageCodeControl;
        }

        @Override
        public final Builder languageCodeControl(String languageCodeControl) {
            this.languageCodeControl = languageCodeControl;
            return this;
        }

        @Override
        public final Builder languageCodeControl(AudioLanguageCodeControl languageCodeControl) {
            this.languageCodeControl(languageCodeControl == null ? null : languageCodeControl.toString());
            return this;
        }

        public final void setLanguageCodeControl(String languageCodeControl) {
            this.languageCodeControl = languageCodeControl;
        }

        public final RemixSettings.Builder getRemixSettings() {
            return remixSettings != null ? remixSettings.toBuilder() : null;
        }

        @Override
        public final Builder remixSettings(RemixSettings remixSettings) {
            this.remixSettings = remixSettings;
            return this;
        }

        public final void setRemixSettings(RemixSettings.BuilderImpl remixSettings) {
            this.remixSettings = remixSettings != null ? remixSettings.build() : null;
        }

        public final String getStreamName() {
            return streamName;
        }

        @Override
        public final Builder streamName(String streamName) {
            this.streamName = streamName;
            return this;
        }

        public final void setStreamName(String streamName) {
            this.streamName = streamName;
        }

        @Override
        public AudioDescription build() {
            return new AudioDescription(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
