/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mediaconvert.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * Settings for CMAF encryption
 */
@Generated("software.amazon.awssdk:codegen")
public final class CmafEncryptionSettings implements SdkPojo, Serializable,
        ToCopyableBuilder<CmafEncryptionSettings.Builder, CmafEncryptionSettings> {
    private static final SdkField<String> CONSTANT_INITIALIZATION_VECTOR_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ConstantInitializationVector")
            .getter(getter(CmafEncryptionSettings::constantInitializationVector))
            .setter(setter(Builder::constantInitializationVector))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("constantInitializationVector")
                    .build()).build();

    private static final SdkField<String> ENCRYPTION_METHOD_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EncryptionMethod").getter(getter(CmafEncryptionSettings::encryptionMethodAsString))
            .setter(setter(Builder::encryptionMethod))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("encryptionMethod").build()).build();

    private static final SdkField<String> INITIALIZATION_VECTOR_IN_MANIFEST_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("InitializationVectorInManifest")
            .getter(getter(CmafEncryptionSettings::initializationVectorInManifestAsString))
            .setter(setter(Builder::initializationVectorInManifest))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("initializationVectorInManifest")
                    .build()).build();

    private static final SdkField<SpekeKeyProviderCmaf> SPEKE_KEY_PROVIDER_FIELD = SdkField
            .<SpekeKeyProviderCmaf> builder(MarshallingType.SDK_POJO).memberName("SpekeKeyProvider")
            .getter(getter(CmafEncryptionSettings::spekeKeyProvider)).setter(setter(Builder::spekeKeyProvider))
            .constructor(SpekeKeyProviderCmaf::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("spekeKeyProvider").build()).build();

    private static final SdkField<StaticKeyProvider> STATIC_KEY_PROVIDER_FIELD = SdkField
            .<StaticKeyProvider> builder(MarshallingType.SDK_POJO).memberName("StaticKeyProvider")
            .getter(getter(CmafEncryptionSettings::staticKeyProvider)).setter(setter(Builder::staticKeyProvider))
            .constructor(StaticKeyProvider::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("staticKeyProvider").build()).build();

    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Type")
            .getter(getter(CmafEncryptionSettings::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("type").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            CONSTANT_INITIALIZATION_VECTOR_FIELD, ENCRYPTION_METHOD_FIELD, INITIALIZATION_VECTOR_IN_MANIFEST_FIELD,
            SPEKE_KEY_PROVIDER_FIELD, STATIC_KEY_PROVIDER_FIELD, TYPE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String constantInitializationVector;

    private final String encryptionMethod;

    private final String initializationVectorInManifest;

    private final SpekeKeyProviderCmaf spekeKeyProvider;

    private final StaticKeyProvider staticKeyProvider;

    private final String type;

    private CmafEncryptionSettings(BuilderImpl builder) {
        this.constantInitializationVector = builder.constantInitializationVector;
        this.encryptionMethod = builder.encryptionMethod;
        this.initializationVectorInManifest = builder.initializationVectorInManifest;
        this.spekeKeyProvider = builder.spekeKeyProvider;
        this.staticKeyProvider = builder.staticKeyProvider;
        this.type = builder.type;
    }

    /**
     * This is a 128-bit, 16-byte hex value represented by a 32-character text string. If this parameter is not set then
     * the Initialization Vector will follow the segment number by default.
     * 
     * @return This is a 128-bit, 16-byte hex value represented by a 32-character text string. If this parameter is not
     *         set then the Initialization Vector will follow the segment number by default.
     */
    public final String constantInitializationVector() {
        return constantInitializationVector;
    }

    /**
     * Specify the encryption scheme that you want the service to use when encrypting your CMAF segments. Choose AES-CBC
     * subsample or AES_CTR.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #encryptionMethod}
     * will return {@link CmafEncryptionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #encryptionMethodAsString}.
     * </p>
     * 
     * @return Specify the encryption scheme that you want the service to use when encrypting your CMAF segments. Choose
     *         AES-CBC subsample or AES_CTR.
     * @see CmafEncryptionType
     */
    public final CmafEncryptionType encryptionMethod() {
        return CmafEncryptionType.fromValue(encryptionMethod);
    }

    /**
     * Specify the encryption scheme that you want the service to use when encrypting your CMAF segments. Choose AES-CBC
     * subsample or AES_CTR.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #encryptionMethod}
     * will return {@link CmafEncryptionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #encryptionMethodAsString}.
     * </p>
     * 
     * @return Specify the encryption scheme that you want the service to use when encrypting your CMAF segments. Choose
     *         AES-CBC subsample or AES_CTR.
     * @see CmafEncryptionType
     */
    public final String encryptionMethodAsString() {
        return encryptionMethod;
    }

    /**
     * When you use DRM with CMAF outputs, choose whether the service writes the 128-bit encryption initialization
     * vector in the HLS and DASH manifests.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #initializationVectorInManifest} will return
     * {@link CmafInitializationVectorInManifest#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #initializationVectorInManifestAsString}.
     * </p>
     * 
     * @return When you use DRM with CMAF outputs, choose whether the service writes the 128-bit encryption
     *         initialization vector in the HLS and DASH manifests.
     * @see CmafInitializationVectorInManifest
     */
    public final CmafInitializationVectorInManifest initializationVectorInManifest() {
        return CmafInitializationVectorInManifest.fromValue(initializationVectorInManifest);
    }

    /**
     * When you use DRM with CMAF outputs, choose whether the service writes the 128-bit encryption initialization
     * vector in the HLS and DASH manifests.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #initializationVectorInManifest} will return
     * {@link CmafInitializationVectorInManifest#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #initializationVectorInManifestAsString}.
     * </p>
     * 
     * @return When you use DRM with CMAF outputs, choose whether the service writes the 128-bit encryption
     *         initialization vector in the HLS and DASH manifests.
     * @see CmafInitializationVectorInManifest
     */
    public final String initializationVectorInManifestAsString() {
        return initializationVectorInManifest;
    }

    /**
     * If your output group type is CMAF, use these settings when doing DRM encryption with a SPEKE-compliant key
     * provider. If your output group type is HLS, DASH, or Microsoft Smooth, use the SpekeKeyProvider settings instead.
     * 
     * @return If your output group type is CMAF, use these settings when doing DRM encryption with a SPEKE-compliant
     *         key provider. If your output group type is HLS, DASH, or Microsoft Smooth, use the SpekeKeyProvider
     *         settings instead.
     */
    public final SpekeKeyProviderCmaf spekeKeyProvider() {
        return spekeKeyProvider;
    }

    /**
     * Use these settings to set up encryption with a static key provider.
     * 
     * @return Use these settings to set up encryption with a static key provider.
     */
    public final StaticKeyProvider staticKeyProvider() {
        return staticKeyProvider;
    }

    /**
     * Specify whether your DRM encryption key is static or from a key provider that follows the SPEKE standard. For
     * more information about SPEKE, see https://docs.aws.amazon.com/speke/latest/documentation/what-is-speke.html.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link CmafKeyProviderType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return Specify whether your DRM encryption key is static or from a key provider that follows the SPEKE standard.
     *         For more information about SPEKE, see
     *         https://docs.aws.amazon.com/speke/latest/documentation/what-is-speke.html.
     * @see CmafKeyProviderType
     */
    public final CmafKeyProviderType type() {
        return CmafKeyProviderType.fromValue(type);
    }

    /**
     * Specify whether your DRM encryption key is static or from a key provider that follows the SPEKE standard. For
     * more information about SPEKE, see https://docs.aws.amazon.com/speke/latest/documentation/what-is-speke.html.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link CmafKeyProviderType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return Specify whether your DRM encryption key is static or from a key provider that follows the SPEKE standard.
     *         For more information about SPEKE, see
     *         https://docs.aws.amazon.com/speke/latest/documentation/what-is-speke.html.
     * @see CmafKeyProviderType
     */
    public final String typeAsString() {
        return type;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(constantInitializationVector());
        hashCode = 31 * hashCode + Objects.hashCode(encryptionMethodAsString());
        hashCode = 31 * hashCode + Objects.hashCode(initializationVectorInManifestAsString());
        hashCode = 31 * hashCode + Objects.hashCode(spekeKeyProvider());
        hashCode = 31 * hashCode + Objects.hashCode(staticKeyProvider());
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CmafEncryptionSettings)) {
            return false;
        }
        CmafEncryptionSettings other = (CmafEncryptionSettings) obj;
        return Objects.equals(constantInitializationVector(), other.constantInitializationVector())
                && Objects.equals(encryptionMethodAsString(), other.encryptionMethodAsString())
                && Objects.equals(initializationVectorInManifestAsString(), other.initializationVectorInManifestAsString())
                && Objects.equals(spekeKeyProvider(), other.spekeKeyProvider())
                && Objects.equals(staticKeyProvider(), other.staticKeyProvider())
                && Objects.equals(typeAsString(), other.typeAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CmafEncryptionSettings").add("ConstantInitializationVector", constantInitializationVector())
                .add("EncryptionMethod", encryptionMethodAsString())
                .add("InitializationVectorInManifest", initializationVectorInManifestAsString())
                .add("SpekeKeyProvider", spekeKeyProvider()).add("StaticKeyProvider", staticKeyProvider())
                .add("Type", typeAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ConstantInitializationVector":
            return Optional.ofNullable(clazz.cast(constantInitializationVector()));
        case "EncryptionMethod":
            return Optional.ofNullable(clazz.cast(encryptionMethodAsString()));
        case "InitializationVectorInManifest":
            return Optional.ofNullable(clazz.cast(initializationVectorInManifestAsString()));
        case "SpekeKeyProvider":
            return Optional.ofNullable(clazz.cast(spekeKeyProvider()));
        case "StaticKeyProvider":
            return Optional.ofNullable(clazz.cast(staticKeyProvider()));
        case "Type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CmafEncryptionSettings, T> g) {
        return obj -> g.apply((CmafEncryptionSettings) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CmafEncryptionSettings> {
        /**
         * This is a 128-bit, 16-byte hex value represented by a 32-character text string. If this parameter is not set
         * then the Initialization Vector will follow the segment number by default.
         * 
         * @param constantInitializationVector
         *        This is a 128-bit, 16-byte hex value represented by a 32-character text string. If this parameter is
         *        not set then the Initialization Vector will follow the segment number by default.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder constantInitializationVector(String constantInitializationVector);

        /**
         * Specify the encryption scheme that you want the service to use when encrypting your CMAF segments. Choose
         * AES-CBC subsample or AES_CTR.
         * 
         * @param encryptionMethod
         *        Specify the encryption scheme that you want the service to use when encrypting your CMAF segments.
         *        Choose AES-CBC subsample or AES_CTR.
         * @see CmafEncryptionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CmafEncryptionType
         */
        Builder encryptionMethod(String encryptionMethod);

        /**
         * Specify the encryption scheme that you want the service to use when encrypting your CMAF segments. Choose
         * AES-CBC subsample or AES_CTR.
         * 
         * @param encryptionMethod
         *        Specify the encryption scheme that you want the service to use when encrypting your CMAF segments.
         *        Choose AES-CBC subsample or AES_CTR.
         * @see CmafEncryptionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CmafEncryptionType
         */
        Builder encryptionMethod(CmafEncryptionType encryptionMethod);

        /**
         * When you use DRM with CMAF outputs, choose whether the service writes the 128-bit encryption initialization
         * vector in the HLS and DASH manifests.
         * 
         * @param initializationVectorInManifest
         *        When you use DRM with CMAF outputs, choose whether the service writes the 128-bit encryption
         *        initialization vector in the HLS and DASH manifests.
         * @see CmafInitializationVectorInManifest
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CmafInitializationVectorInManifest
         */
        Builder initializationVectorInManifest(String initializationVectorInManifest);

        /**
         * When you use DRM with CMAF outputs, choose whether the service writes the 128-bit encryption initialization
         * vector in the HLS and DASH manifests.
         * 
         * @param initializationVectorInManifest
         *        When you use DRM with CMAF outputs, choose whether the service writes the 128-bit encryption
         *        initialization vector in the HLS and DASH manifests.
         * @see CmafInitializationVectorInManifest
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CmafInitializationVectorInManifest
         */
        Builder initializationVectorInManifest(CmafInitializationVectorInManifest initializationVectorInManifest);

        /**
         * If your output group type is CMAF, use these settings when doing DRM encryption with a SPEKE-compliant key
         * provider. If your output group type is HLS, DASH, or Microsoft Smooth, use the SpekeKeyProvider settings
         * instead.
         * 
         * @param spekeKeyProvider
         *        If your output group type is CMAF, use these settings when doing DRM encryption with a SPEKE-compliant
         *        key provider. If your output group type is HLS, DASH, or Microsoft Smooth, use the SpekeKeyProvider
         *        settings instead.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder spekeKeyProvider(SpekeKeyProviderCmaf spekeKeyProvider);

        /**
         * If your output group type is CMAF, use these settings when doing DRM encryption with a SPEKE-compliant key
         * provider. If your output group type is HLS, DASH, or Microsoft Smooth, use the SpekeKeyProvider settings
         * instead. This is a convenience method that creates an instance of the {@link SpekeKeyProviderCmaf.Builder}
         * avoiding the need to create one manually via {@link SpekeKeyProviderCmaf#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link SpekeKeyProviderCmaf.Builder#build()} is called immediately and
         * its result is passed to {@link #spekeKeyProvider(SpekeKeyProviderCmaf)}.
         * 
         * @param spekeKeyProvider
         *        a consumer that will call methods on {@link SpekeKeyProviderCmaf.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #spekeKeyProvider(SpekeKeyProviderCmaf)
         */
        default Builder spekeKeyProvider(Consumer<SpekeKeyProviderCmaf.Builder> spekeKeyProvider) {
            return spekeKeyProvider(SpekeKeyProviderCmaf.builder().applyMutation(spekeKeyProvider).build());
        }

        /**
         * Use these settings to set up encryption with a static key provider.
         * 
         * @param staticKeyProvider
         *        Use these settings to set up encryption with a static key provider.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder staticKeyProvider(StaticKeyProvider staticKeyProvider);

        /**
         * Use these settings to set up encryption with a static key provider. This is a convenience method that creates
         * an instance of the {@link StaticKeyProvider.Builder} avoiding the need to create one manually via
         * {@link StaticKeyProvider#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link StaticKeyProvider.Builder#build()} is called immediately and its
         * result is passed to {@link #staticKeyProvider(StaticKeyProvider)}.
         * 
         * @param staticKeyProvider
         *        a consumer that will call methods on {@link StaticKeyProvider.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #staticKeyProvider(StaticKeyProvider)
         */
        default Builder staticKeyProvider(Consumer<StaticKeyProvider.Builder> staticKeyProvider) {
            return staticKeyProvider(StaticKeyProvider.builder().applyMutation(staticKeyProvider).build());
        }

        /**
         * Specify whether your DRM encryption key is static or from a key provider that follows the SPEKE standard. For
         * more information about SPEKE, see https://docs.aws.amazon.com/speke/latest/documentation/what-is-speke.html.
         * 
         * @param type
         *        Specify whether your DRM encryption key is static or from a key provider that follows the SPEKE
         *        standard. For more information about SPEKE, see
         *        https://docs.aws.amazon.com/speke/latest/documentation/what-is-speke.html.
         * @see CmafKeyProviderType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CmafKeyProviderType
         */
        Builder type(String type);

        /**
         * Specify whether your DRM encryption key is static or from a key provider that follows the SPEKE standard. For
         * more information about SPEKE, see https://docs.aws.amazon.com/speke/latest/documentation/what-is-speke.html.
         * 
         * @param type
         *        Specify whether your DRM encryption key is static or from a key provider that follows the SPEKE
         *        standard. For more information about SPEKE, see
         *        https://docs.aws.amazon.com/speke/latest/documentation/what-is-speke.html.
         * @see CmafKeyProviderType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CmafKeyProviderType
         */
        Builder type(CmafKeyProviderType type);
    }

    static final class BuilderImpl implements Builder {
        private String constantInitializationVector;

        private String encryptionMethod;

        private String initializationVectorInManifest;

        private SpekeKeyProviderCmaf spekeKeyProvider;

        private StaticKeyProvider staticKeyProvider;

        private String type;

        private BuilderImpl() {
        }

        private BuilderImpl(CmafEncryptionSettings model) {
            constantInitializationVector(model.constantInitializationVector);
            encryptionMethod(model.encryptionMethod);
            initializationVectorInManifest(model.initializationVectorInManifest);
            spekeKeyProvider(model.spekeKeyProvider);
            staticKeyProvider(model.staticKeyProvider);
            type(model.type);
        }

        public final String getConstantInitializationVector() {
            return constantInitializationVector;
        }

        public final void setConstantInitializationVector(String constantInitializationVector) {
            this.constantInitializationVector = constantInitializationVector;
        }

        @Override
        public final Builder constantInitializationVector(String constantInitializationVector) {
            this.constantInitializationVector = constantInitializationVector;
            return this;
        }

        public final String getEncryptionMethod() {
            return encryptionMethod;
        }

        public final void setEncryptionMethod(String encryptionMethod) {
            this.encryptionMethod = encryptionMethod;
        }

        @Override
        public final Builder encryptionMethod(String encryptionMethod) {
            this.encryptionMethod = encryptionMethod;
            return this;
        }

        @Override
        public final Builder encryptionMethod(CmafEncryptionType encryptionMethod) {
            this.encryptionMethod(encryptionMethod == null ? null : encryptionMethod.toString());
            return this;
        }

        public final String getInitializationVectorInManifest() {
            return initializationVectorInManifest;
        }

        public final void setInitializationVectorInManifest(String initializationVectorInManifest) {
            this.initializationVectorInManifest = initializationVectorInManifest;
        }

        @Override
        public final Builder initializationVectorInManifest(String initializationVectorInManifest) {
            this.initializationVectorInManifest = initializationVectorInManifest;
            return this;
        }

        @Override
        public final Builder initializationVectorInManifest(CmafInitializationVectorInManifest initializationVectorInManifest) {
            this.initializationVectorInManifest(initializationVectorInManifest == null ? null : initializationVectorInManifest
                    .toString());
            return this;
        }

        public final SpekeKeyProviderCmaf.Builder getSpekeKeyProvider() {
            return spekeKeyProvider != null ? spekeKeyProvider.toBuilder() : null;
        }

        public final void setSpekeKeyProvider(SpekeKeyProviderCmaf.BuilderImpl spekeKeyProvider) {
            this.spekeKeyProvider = spekeKeyProvider != null ? spekeKeyProvider.build() : null;
        }

        @Override
        public final Builder spekeKeyProvider(SpekeKeyProviderCmaf spekeKeyProvider) {
            this.spekeKeyProvider = spekeKeyProvider;
            return this;
        }

        public final StaticKeyProvider.Builder getStaticKeyProvider() {
            return staticKeyProvider != null ? staticKeyProvider.toBuilder() : null;
        }

        public final void setStaticKeyProvider(StaticKeyProvider.BuilderImpl staticKeyProvider) {
            this.staticKeyProvider = staticKeyProvider != null ? staticKeyProvider.build() : null;
        }

        @Override
        public final Builder staticKeyProvider(StaticKeyProvider staticKeyProvider) {
            this.staticKeyProvider = staticKeyProvider;
            return this;
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(CmafKeyProviderType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        @Override
        public CmafEncryptionSettings build() {
            return new CmafEncryptionSettings(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
