/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.medialive;

import static software.amazon.awssdk.utils.FunctionalUtils.runAndLogError;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ScheduledExecutorService;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.async.AsyncResponseTransformer;
import software.amazon.awssdk.core.async.AsyncResponseTransformerUtils;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.medialive.model.AcceptInputDeviceTransferRequest;
import software.amazon.awssdk.services.medialive.model.AcceptInputDeviceTransferResponse;
import software.amazon.awssdk.services.medialive.model.BadGatewayException;
import software.amazon.awssdk.services.medialive.model.BadRequestException;
import software.amazon.awssdk.services.medialive.model.BatchDeleteRequest;
import software.amazon.awssdk.services.medialive.model.BatchDeleteResponse;
import software.amazon.awssdk.services.medialive.model.BatchStartRequest;
import software.amazon.awssdk.services.medialive.model.BatchStartResponse;
import software.amazon.awssdk.services.medialive.model.BatchStopRequest;
import software.amazon.awssdk.services.medialive.model.BatchStopResponse;
import software.amazon.awssdk.services.medialive.model.BatchUpdateScheduleRequest;
import software.amazon.awssdk.services.medialive.model.BatchUpdateScheduleResponse;
import software.amazon.awssdk.services.medialive.model.CancelInputDeviceTransferRequest;
import software.amazon.awssdk.services.medialive.model.CancelInputDeviceTransferResponse;
import software.amazon.awssdk.services.medialive.model.ClaimDeviceRequest;
import software.amazon.awssdk.services.medialive.model.ClaimDeviceResponse;
import software.amazon.awssdk.services.medialive.model.ConflictException;
import software.amazon.awssdk.services.medialive.model.CreateChannelRequest;
import software.amazon.awssdk.services.medialive.model.CreateChannelResponse;
import software.amazon.awssdk.services.medialive.model.CreateInputRequest;
import software.amazon.awssdk.services.medialive.model.CreateInputResponse;
import software.amazon.awssdk.services.medialive.model.CreateInputSecurityGroupRequest;
import software.amazon.awssdk.services.medialive.model.CreateInputSecurityGroupResponse;
import software.amazon.awssdk.services.medialive.model.CreateMultiplexProgramRequest;
import software.amazon.awssdk.services.medialive.model.CreateMultiplexProgramResponse;
import software.amazon.awssdk.services.medialive.model.CreateMultiplexRequest;
import software.amazon.awssdk.services.medialive.model.CreateMultiplexResponse;
import software.amazon.awssdk.services.medialive.model.CreatePartnerInputRequest;
import software.amazon.awssdk.services.medialive.model.CreatePartnerInputResponse;
import software.amazon.awssdk.services.medialive.model.CreateTagsRequest;
import software.amazon.awssdk.services.medialive.model.CreateTagsResponse;
import software.amazon.awssdk.services.medialive.model.DeleteChannelRequest;
import software.amazon.awssdk.services.medialive.model.DeleteChannelResponse;
import software.amazon.awssdk.services.medialive.model.DeleteInputRequest;
import software.amazon.awssdk.services.medialive.model.DeleteInputResponse;
import software.amazon.awssdk.services.medialive.model.DeleteInputSecurityGroupRequest;
import software.amazon.awssdk.services.medialive.model.DeleteInputSecurityGroupResponse;
import software.amazon.awssdk.services.medialive.model.DeleteMultiplexProgramRequest;
import software.amazon.awssdk.services.medialive.model.DeleteMultiplexProgramResponse;
import software.amazon.awssdk.services.medialive.model.DeleteMultiplexRequest;
import software.amazon.awssdk.services.medialive.model.DeleteMultiplexResponse;
import software.amazon.awssdk.services.medialive.model.DeleteReservationRequest;
import software.amazon.awssdk.services.medialive.model.DeleteReservationResponse;
import software.amazon.awssdk.services.medialive.model.DeleteScheduleRequest;
import software.amazon.awssdk.services.medialive.model.DeleteScheduleResponse;
import software.amazon.awssdk.services.medialive.model.DeleteTagsRequest;
import software.amazon.awssdk.services.medialive.model.DeleteTagsResponse;
import software.amazon.awssdk.services.medialive.model.DescribeChannelRequest;
import software.amazon.awssdk.services.medialive.model.DescribeChannelResponse;
import software.amazon.awssdk.services.medialive.model.DescribeInputDeviceRequest;
import software.amazon.awssdk.services.medialive.model.DescribeInputDeviceResponse;
import software.amazon.awssdk.services.medialive.model.DescribeInputDeviceThumbnailRequest;
import software.amazon.awssdk.services.medialive.model.DescribeInputDeviceThumbnailResponse;
import software.amazon.awssdk.services.medialive.model.DescribeInputRequest;
import software.amazon.awssdk.services.medialive.model.DescribeInputResponse;
import software.amazon.awssdk.services.medialive.model.DescribeInputSecurityGroupRequest;
import software.amazon.awssdk.services.medialive.model.DescribeInputSecurityGroupResponse;
import software.amazon.awssdk.services.medialive.model.DescribeMultiplexProgramRequest;
import software.amazon.awssdk.services.medialive.model.DescribeMultiplexProgramResponse;
import software.amazon.awssdk.services.medialive.model.DescribeMultiplexRequest;
import software.amazon.awssdk.services.medialive.model.DescribeMultiplexResponse;
import software.amazon.awssdk.services.medialive.model.DescribeOfferingRequest;
import software.amazon.awssdk.services.medialive.model.DescribeOfferingResponse;
import software.amazon.awssdk.services.medialive.model.DescribeReservationRequest;
import software.amazon.awssdk.services.medialive.model.DescribeReservationResponse;
import software.amazon.awssdk.services.medialive.model.DescribeScheduleRequest;
import software.amazon.awssdk.services.medialive.model.DescribeScheduleResponse;
import software.amazon.awssdk.services.medialive.model.ForbiddenException;
import software.amazon.awssdk.services.medialive.model.GatewayTimeoutException;
import software.amazon.awssdk.services.medialive.model.InternalServerErrorException;
import software.amazon.awssdk.services.medialive.model.ListChannelsRequest;
import software.amazon.awssdk.services.medialive.model.ListChannelsResponse;
import software.amazon.awssdk.services.medialive.model.ListInputDeviceTransfersRequest;
import software.amazon.awssdk.services.medialive.model.ListInputDeviceTransfersResponse;
import software.amazon.awssdk.services.medialive.model.ListInputDevicesRequest;
import software.amazon.awssdk.services.medialive.model.ListInputDevicesResponse;
import software.amazon.awssdk.services.medialive.model.ListInputSecurityGroupsRequest;
import software.amazon.awssdk.services.medialive.model.ListInputSecurityGroupsResponse;
import software.amazon.awssdk.services.medialive.model.ListInputsRequest;
import software.amazon.awssdk.services.medialive.model.ListInputsResponse;
import software.amazon.awssdk.services.medialive.model.ListMultiplexProgramsRequest;
import software.amazon.awssdk.services.medialive.model.ListMultiplexProgramsResponse;
import software.amazon.awssdk.services.medialive.model.ListMultiplexesRequest;
import software.amazon.awssdk.services.medialive.model.ListMultiplexesResponse;
import software.amazon.awssdk.services.medialive.model.ListOfferingsRequest;
import software.amazon.awssdk.services.medialive.model.ListOfferingsResponse;
import software.amazon.awssdk.services.medialive.model.ListReservationsRequest;
import software.amazon.awssdk.services.medialive.model.ListReservationsResponse;
import software.amazon.awssdk.services.medialive.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.medialive.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.medialive.model.MediaLiveException;
import software.amazon.awssdk.services.medialive.model.MediaLiveRequest;
import software.amazon.awssdk.services.medialive.model.NotFoundException;
import software.amazon.awssdk.services.medialive.model.PurchaseOfferingRequest;
import software.amazon.awssdk.services.medialive.model.PurchaseOfferingResponse;
import software.amazon.awssdk.services.medialive.model.RebootInputDeviceRequest;
import software.amazon.awssdk.services.medialive.model.RebootInputDeviceResponse;
import software.amazon.awssdk.services.medialive.model.RejectInputDeviceTransferRequest;
import software.amazon.awssdk.services.medialive.model.RejectInputDeviceTransferResponse;
import software.amazon.awssdk.services.medialive.model.StartChannelRequest;
import software.amazon.awssdk.services.medialive.model.StartChannelResponse;
import software.amazon.awssdk.services.medialive.model.StartInputDeviceMaintenanceWindowRequest;
import software.amazon.awssdk.services.medialive.model.StartInputDeviceMaintenanceWindowResponse;
import software.amazon.awssdk.services.medialive.model.StartMultiplexRequest;
import software.amazon.awssdk.services.medialive.model.StartMultiplexResponse;
import software.amazon.awssdk.services.medialive.model.StopChannelRequest;
import software.amazon.awssdk.services.medialive.model.StopChannelResponse;
import software.amazon.awssdk.services.medialive.model.StopMultiplexRequest;
import software.amazon.awssdk.services.medialive.model.StopMultiplexResponse;
import software.amazon.awssdk.services.medialive.model.TooManyRequestsException;
import software.amazon.awssdk.services.medialive.model.TransferInputDeviceRequest;
import software.amazon.awssdk.services.medialive.model.TransferInputDeviceResponse;
import software.amazon.awssdk.services.medialive.model.UnprocessableEntityException;
import software.amazon.awssdk.services.medialive.model.UpdateChannelClassRequest;
import software.amazon.awssdk.services.medialive.model.UpdateChannelClassResponse;
import software.amazon.awssdk.services.medialive.model.UpdateChannelRequest;
import software.amazon.awssdk.services.medialive.model.UpdateChannelResponse;
import software.amazon.awssdk.services.medialive.model.UpdateInputDeviceRequest;
import software.amazon.awssdk.services.medialive.model.UpdateInputDeviceResponse;
import software.amazon.awssdk.services.medialive.model.UpdateInputRequest;
import software.amazon.awssdk.services.medialive.model.UpdateInputResponse;
import software.amazon.awssdk.services.medialive.model.UpdateInputSecurityGroupRequest;
import software.amazon.awssdk.services.medialive.model.UpdateInputSecurityGroupResponse;
import software.amazon.awssdk.services.medialive.model.UpdateMultiplexProgramRequest;
import software.amazon.awssdk.services.medialive.model.UpdateMultiplexProgramResponse;
import software.amazon.awssdk.services.medialive.model.UpdateMultiplexRequest;
import software.amazon.awssdk.services.medialive.model.UpdateMultiplexResponse;
import software.amazon.awssdk.services.medialive.model.UpdateReservationRequest;
import software.amazon.awssdk.services.medialive.model.UpdateReservationResponse;
import software.amazon.awssdk.services.medialive.paginators.DescribeSchedulePublisher;
import software.amazon.awssdk.services.medialive.paginators.ListChannelsPublisher;
import software.amazon.awssdk.services.medialive.paginators.ListInputDeviceTransfersPublisher;
import software.amazon.awssdk.services.medialive.paginators.ListInputDevicesPublisher;
import software.amazon.awssdk.services.medialive.paginators.ListInputSecurityGroupsPublisher;
import software.amazon.awssdk.services.medialive.paginators.ListInputsPublisher;
import software.amazon.awssdk.services.medialive.paginators.ListMultiplexProgramsPublisher;
import software.amazon.awssdk.services.medialive.paginators.ListMultiplexesPublisher;
import software.amazon.awssdk.services.medialive.paginators.ListOfferingsPublisher;
import software.amazon.awssdk.services.medialive.paginators.ListReservationsPublisher;
import software.amazon.awssdk.services.medialive.transform.AcceptInputDeviceTransferRequestMarshaller;
import software.amazon.awssdk.services.medialive.transform.BatchDeleteRequestMarshaller;
import software.amazon.awssdk.services.medialive.transform.BatchStartRequestMarshaller;
import software.amazon.awssdk.services.medialive.transform.BatchStopRequestMarshaller;
import software.amazon.awssdk.services.medialive.transform.BatchUpdateScheduleRequestMarshaller;
import software.amazon.awssdk.services.medialive.transform.CancelInputDeviceTransferRequestMarshaller;
import software.amazon.awssdk.services.medialive.transform.ClaimDeviceRequestMarshaller;
import software.amazon.awssdk.services.medialive.transform.CreateChannelRequestMarshaller;
import software.amazon.awssdk.services.medialive.transform.CreateInputRequestMarshaller;
import software.amazon.awssdk.services.medialive.transform.CreateInputSecurityGroupRequestMarshaller;
import software.amazon.awssdk.services.medialive.transform.CreateMultiplexProgramRequestMarshaller;
import software.amazon.awssdk.services.medialive.transform.CreateMultiplexRequestMarshaller;
import software.amazon.awssdk.services.medialive.transform.CreatePartnerInputRequestMarshaller;
import software.amazon.awssdk.services.medialive.transform.CreateTagsRequestMarshaller;
import software.amazon.awssdk.services.medialive.transform.DeleteChannelRequestMarshaller;
import software.amazon.awssdk.services.medialive.transform.DeleteInputRequestMarshaller;
import software.amazon.awssdk.services.medialive.transform.DeleteInputSecurityGroupRequestMarshaller;
import software.amazon.awssdk.services.medialive.transform.DeleteMultiplexProgramRequestMarshaller;
import software.amazon.awssdk.services.medialive.transform.DeleteMultiplexRequestMarshaller;
import software.amazon.awssdk.services.medialive.transform.DeleteReservationRequestMarshaller;
import software.amazon.awssdk.services.medialive.transform.DeleteScheduleRequestMarshaller;
import software.amazon.awssdk.services.medialive.transform.DeleteTagsRequestMarshaller;
import software.amazon.awssdk.services.medialive.transform.DescribeChannelRequestMarshaller;
import software.amazon.awssdk.services.medialive.transform.DescribeInputDeviceRequestMarshaller;
import software.amazon.awssdk.services.medialive.transform.DescribeInputDeviceThumbnailRequestMarshaller;
import software.amazon.awssdk.services.medialive.transform.DescribeInputRequestMarshaller;
import software.amazon.awssdk.services.medialive.transform.DescribeInputSecurityGroupRequestMarshaller;
import software.amazon.awssdk.services.medialive.transform.DescribeMultiplexProgramRequestMarshaller;
import software.amazon.awssdk.services.medialive.transform.DescribeMultiplexRequestMarshaller;
import software.amazon.awssdk.services.medialive.transform.DescribeOfferingRequestMarshaller;
import software.amazon.awssdk.services.medialive.transform.DescribeReservationRequestMarshaller;
import software.amazon.awssdk.services.medialive.transform.DescribeScheduleRequestMarshaller;
import software.amazon.awssdk.services.medialive.transform.ListChannelsRequestMarshaller;
import software.amazon.awssdk.services.medialive.transform.ListInputDeviceTransfersRequestMarshaller;
import software.amazon.awssdk.services.medialive.transform.ListInputDevicesRequestMarshaller;
import software.amazon.awssdk.services.medialive.transform.ListInputSecurityGroupsRequestMarshaller;
import software.amazon.awssdk.services.medialive.transform.ListInputsRequestMarshaller;
import software.amazon.awssdk.services.medialive.transform.ListMultiplexProgramsRequestMarshaller;
import software.amazon.awssdk.services.medialive.transform.ListMultiplexesRequestMarshaller;
import software.amazon.awssdk.services.medialive.transform.ListOfferingsRequestMarshaller;
import software.amazon.awssdk.services.medialive.transform.ListReservationsRequestMarshaller;
import software.amazon.awssdk.services.medialive.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.medialive.transform.PurchaseOfferingRequestMarshaller;
import software.amazon.awssdk.services.medialive.transform.RebootInputDeviceRequestMarshaller;
import software.amazon.awssdk.services.medialive.transform.RejectInputDeviceTransferRequestMarshaller;
import software.amazon.awssdk.services.medialive.transform.StartChannelRequestMarshaller;
import software.amazon.awssdk.services.medialive.transform.StartInputDeviceMaintenanceWindowRequestMarshaller;
import software.amazon.awssdk.services.medialive.transform.StartMultiplexRequestMarshaller;
import software.amazon.awssdk.services.medialive.transform.StopChannelRequestMarshaller;
import software.amazon.awssdk.services.medialive.transform.StopMultiplexRequestMarshaller;
import software.amazon.awssdk.services.medialive.transform.TransferInputDeviceRequestMarshaller;
import software.amazon.awssdk.services.medialive.transform.UpdateChannelClassRequestMarshaller;
import software.amazon.awssdk.services.medialive.transform.UpdateChannelRequestMarshaller;
import software.amazon.awssdk.services.medialive.transform.UpdateInputDeviceRequestMarshaller;
import software.amazon.awssdk.services.medialive.transform.UpdateInputRequestMarshaller;
import software.amazon.awssdk.services.medialive.transform.UpdateInputSecurityGroupRequestMarshaller;
import software.amazon.awssdk.services.medialive.transform.UpdateMultiplexProgramRequestMarshaller;
import software.amazon.awssdk.services.medialive.transform.UpdateMultiplexRequestMarshaller;
import software.amazon.awssdk.services.medialive.transform.UpdateReservationRequestMarshaller;
import software.amazon.awssdk.services.medialive.waiters.MediaLiveAsyncWaiter;
import software.amazon.awssdk.utils.CompletableFutureUtils;
import software.amazon.awssdk.utils.Pair;

/**
 * Internal implementation of {@link MediaLiveAsyncClient}.
 *
 * @see MediaLiveAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultMediaLiveAsyncClient implements MediaLiveAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultMediaLiveAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final ScheduledExecutorService executorService;

    protected DefaultMediaLiveAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
        this.executorService = clientConfiguration.option(SdkClientOption.SCHEDULED_EXECUTOR_SERVICE);
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * Accept an incoming input device transfer. The ownership of the device will transfer to your AWS account.
     *
     * @param acceptInputDeviceTransferRequest
     *        Placeholder documentation for AcceptInputDeviceTransferRequest
     * @return A Java Future containing the result of the AcceptInputDeviceTransfer operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException This request was invalid.</li>
     *         <li>UnprocessableEntityException Transfer operation failed, input could not be validated.</li>
     *         <li>InternalServerErrorException Unexpected internal service error.</li>
     *         <li>ForbiddenException You do not have permission to accept input device transfers.</li>
     *         <li>BadGatewayException Bad gateway error.</li>
     *         <li>NotFoundException Input device not found.</li>
     *         <li>GatewayTimeoutException Gateway Timeout Error</li>
     *         <li>TooManyRequestsException Request limit exceeded on transfer device calls to the input device service.
     *         </li>
     *         <li>ConflictException Input device transfer could not be accepted.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaLiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaLiveAsyncClient.AcceptInputDeviceTransfer
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medialive-2017-10-14/AcceptInputDeviceTransfer"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<AcceptInputDeviceTransferResponse> acceptInputDeviceTransfer(
            AcceptInputDeviceTransferRequest acceptInputDeviceTransferRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, acceptInputDeviceTransferRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaLive");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AcceptInputDeviceTransfer");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AcceptInputDeviceTransferResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AcceptInputDeviceTransferResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AcceptInputDeviceTransferResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AcceptInputDeviceTransferRequest, AcceptInputDeviceTransferResponse>()
                            .withOperationName("AcceptInputDeviceTransfer")
                            .withMarshaller(new AcceptInputDeviceTransferRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(acceptInputDeviceTransferRequest));
            CompletableFuture<AcceptInputDeviceTransferResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Starts delete of resources.
     *
     * @param batchDeleteRequest
     *        A request to delete resources
     * @return A Java Future containing the result of the BatchDelete operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException This request was invalid</li>
     *         <li>InternalServerErrorException Unexpected internal service error.</li>
     *         <li>ForbiddenException You do not have permission to delete the resources.</li>
     *         <li>BadGatewayException Bad Gateway Error</li>
     *         <li>NotFoundException The resources you're requesting to delete do not exist.</li>
     *         <li>GatewayTimeoutException Gateway Timeout Error</li>
     *         <li>TooManyRequestsException Request limit exceeded on delete resources calls.</li>
     *         <li>ConflictException The resources are unable to delete.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaLiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaLiveAsyncClient.BatchDelete
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medialive-2017-10-14/BatchDelete" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<BatchDeleteResponse> batchDelete(BatchDeleteRequest batchDeleteRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchDeleteRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaLive");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchDelete");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchDeleteResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    BatchDeleteResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BatchDeleteResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchDeleteRequest, BatchDeleteResponse>()
                            .withOperationName("BatchDelete").withMarshaller(new BatchDeleteRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(batchDeleteRequest));
            CompletableFuture<BatchDeleteResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Starts existing resources
     *
     * @param batchStartRequest
     *        A request to start resources
     * @return A Java Future containing the result of the BatchStart operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException This request was invalid.</li>
     *         <li>InternalServerErrorException Unexpected internal service error.</li>
     *         <li>ForbiddenException You do not have permission to start the resources.</li>
     *         <li>BadGatewayException Bad Gateway Error</li>
     *         <li>NotFoundException The resources you're requesting to start do not exist.</li>
     *         <li>GatewayTimeoutException Gateway Timeout Error</li>
     *         <li>TooManyRequestsException Request limit exceeded on start resources calls to resource service.</li>
     *         <li>ConflictException The resources are unable to start.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaLiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaLiveAsyncClient.BatchStart
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medialive-2017-10-14/BatchStart" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<BatchStartResponse> batchStart(BatchStartRequest batchStartRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchStartRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaLive");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchStart");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchStartResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    BatchStartResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BatchStartResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchStartRequest, BatchStartResponse>().withOperationName("BatchStart")
                            .withMarshaller(new BatchStartRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(batchStartRequest));
            CompletableFuture<BatchStartResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Stops running resources
     *
     * @param batchStopRequest
     *        A request to stop resources
     * @return A Java Future containing the result of the BatchStop operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException This request was invalid.</li>
     *         <li>InternalServerErrorException Unexpected internal service error.</li>
     *         <li>ForbiddenException You do not have permission to stop the resources.</li>
     *         <li>BadGatewayException Bad Gateway Error</li>
     *         <li>NotFoundException The resources you're requesting to stop do not exist.</li>
     *         <li>GatewayTimeoutException Gateway Timeout Error</li>
     *         <li>TooManyRequestsException Request limit exceeded on stop resources.</li>
     *         <li>ConflictException The resources are unable to stop.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaLiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaLiveAsyncClient.BatchStop
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medialive-2017-10-14/BatchStop" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<BatchStopResponse> batchStop(BatchStopRequest batchStopRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchStopRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaLive");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchStop");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchStopResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    BatchStopResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BatchStopResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchStopRequest, BatchStopResponse>().withOperationName("BatchStop")
                            .withMarshaller(new BatchStopRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(batchStopRequest));
            CompletableFuture<BatchStopResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Update a channel schedule
     *
     * @param batchUpdateScheduleRequest
     *        List of actions to create and list of actions to delete.
     * @return A Java Future containing the result of the BatchUpdateSchedule operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException This request was invalid.</li>
     *         <li>UnprocessableEntityException The update schedule request failed validation.</li>
     *         <li>InternalServerErrorException Unexpected internal service error.</li>
     *         <li>ForbiddenException You do not have permission to update the channel schedule.</li>
     *         <li>BadGatewayException Bad Gateway Error</li>
     *         <li>NotFoundException The specified channel id does not exist.</li>
     *         <li>GatewayTimeoutException Gateway Timeout Error</li>
     *         <li>TooManyRequestsException Request limit exceeded on update schedule calls.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaLiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaLiveAsyncClient.BatchUpdateSchedule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medialive-2017-10-14/BatchUpdateSchedule" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchUpdateScheduleResponse> batchUpdateSchedule(
            BatchUpdateScheduleRequest batchUpdateScheduleRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchUpdateScheduleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaLive");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchUpdateSchedule");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchUpdateScheduleResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchUpdateScheduleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BatchUpdateScheduleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchUpdateScheduleRequest, BatchUpdateScheduleResponse>()
                            .withOperationName("BatchUpdateSchedule")
                            .withMarshaller(new BatchUpdateScheduleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(batchUpdateScheduleRequest));
            CompletableFuture<BatchUpdateScheduleResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Cancel an input device transfer that you have requested.
     *
     * @param cancelInputDeviceTransferRequest
     *        Placeholder documentation for CancelInputDeviceTransferRequest
     * @return A Java Future containing the result of the CancelInputDeviceTransfer operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException This request was invalid.</li>
     *         <li>UnprocessableEntityException Transfer operation failed, input could not be validated.</li>
     *         <li>InternalServerErrorException Unexpected internal service error.</li>
     *         <li>ForbiddenException You do not have permission to cancel input device transfers.</li>
     *         <li>BadGatewayException Bad gateway error.</li>
     *         <li>NotFoundException Input device not found.</li>
     *         <li>GatewayTimeoutException Gateway Timeout Error</li>
     *         <li>TooManyRequestsException Request limit exceeded on transfer device calls to the input device service.
     *         </li>
     *         <li>ConflictException Input device transfer could not be canceled.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaLiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaLiveAsyncClient.CancelInputDeviceTransfer
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medialive-2017-10-14/CancelInputDeviceTransfer"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CancelInputDeviceTransferResponse> cancelInputDeviceTransfer(
            CancelInputDeviceTransferRequest cancelInputDeviceTransferRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, cancelInputDeviceTransferRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaLive");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CancelInputDeviceTransfer");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CancelInputDeviceTransferResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CancelInputDeviceTransferResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CancelInputDeviceTransferResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CancelInputDeviceTransferRequest, CancelInputDeviceTransferResponse>()
                            .withOperationName("CancelInputDeviceTransfer")
                            .withMarshaller(new CancelInputDeviceTransferRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(cancelInputDeviceTransferRequest));
            CompletableFuture<CancelInputDeviceTransferResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Send a request to claim an AWS Elemental device that you have purchased from a third-party vendor. After the
     * request succeeds, you will own the device.
     *
     * @param claimDeviceRequest
     *        A request to claim an AWS Elemental device that you have purchased from a third-party vendor.
     * @return A Java Future containing the result of the ClaimDevice operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException This request was invalid.</li>
     *         <li>UnprocessableEntityException Claim operation failed, device id could not be validated.</li>
     *         <li>InternalServerErrorException Unexpected internal service error.</li>
     *         <li>ForbiddenException You don't have the IAM access permissions to claim this device. You must have
     *         permission for the WriteClaim operation.</li>
     *         <li>BadGatewayException Bad gateway error.</li>
     *         <li>NotFoundException Device not found or you called an unsupported region.</li>
     *         <li>GatewayTimeoutException Bad gateway error. Try again.</li>
     *         <li>TooManyRequestsException Request limit exceeded. Your have sent this request too many times. There
     *         might be a bug in your API client.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaLiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaLiveAsyncClient.ClaimDevice
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medialive-2017-10-14/ClaimDevice" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ClaimDeviceResponse> claimDevice(ClaimDeviceRequest claimDeviceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, claimDeviceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaLive");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ClaimDevice");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ClaimDeviceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ClaimDeviceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ClaimDeviceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ClaimDeviceRequest, ClaimDeviceResponse>()
                            .withOperationName("ClaimDevice").withMarshaller(new ClaimDeviceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(claimDeviceRequest));
            CompletableFuture<ClaimDeviceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Creates a new channel
     *
     * @param createChannelRequest
     *        A request to create a channel
     * @return A Java Future containing the result of the CreateChannel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException This request was invalid.</li>
     *         <li>UnprocessableEntityException The Channel failed validation and could not be created.</li>
     *         <li>InternalServerErrorException Unexpected internal service error.</li>
     *         <li>ForbiddenException You do not have permission to create the channel.</li>
     *         <li>BadGatewayException Bad Gateway Error</li>
     *         <li>GatewayTimeoutException Gateway Timeout Error</li>
     *         <li>TooManyRequestsException Request limit exceeded on create channel calls to channel service.</li>
     *         <li>ConflictException The channel is unable to create due to an issue with channel resources.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaLiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaLiveAsyncClient.CreateChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medialive-2017-10-14/CreateChannel" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateChannelResponse> createChannel(CreateChannelRequest createChannelRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaLive");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateChannel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateChannelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateChannelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateChannelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateChannelRequest, CreateChannelResponse>()
                            .withOperationName("CreateChannel")
                            .withMarshaller(new CreateChannelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createChannelRequest));
            CompletableFuture<CreateChannelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Create an input
     *
     * @param createInputRequest
     *        The name of the input
     * @return A Java Future containing the result of the CreateInput operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException This request was invalid</li>
     *         <li>InternalServerErrorException Internal Service Error</li>
     *         <li>ForbiddenException Access was denied</li>
     *         <li>BadGatewayException Bad Gateway Error</li>
     *         <li>GatewayTimeoutException Gateway Timeout Error</li>
     *         <li>TooManyRequestsException Limit exceeded</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaLiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaLiveAsyncClient.CreateInput
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medialive-2017-10-14/CreateInput" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateInputResponse> createInput(CreateInputRequest createInputRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createInputRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaLive");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateInput");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateInputResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateInputResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateInputResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateInputRequest, CreateInputResponse>()
                            .withOperationName("CreateInput").withMarshaller(new CreateInputRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createInputRequest));
            CompletableFuture<CreateInputResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Creates a Input Security Group
     *
     * @param createInputSecurityGroupRequest
     *        The IPv4 CIDRs to whitelist for this Input Security Group
     * @return A Java Future containing the result of the CreateInputSecurityGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request to create an Input Security Group was Invalid</li>
     *         <li>InternalServerErrorException Internal Server Error</li>
     *         <li>ForbiddenException The requester does not have permission to create an Input Security Group</li>
     *         <li>BadGatewayException Bad Gateway Error</li>
     *         <li>GatewayTimeoutException Gateway Timeout Error</li>
     *         <li>TooManyRequestsException Limit Exceeded Error</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaLiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaLiveAsyncClient.CreateInputSecurityGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medialive-2017-10-14/CreateInputSecurityGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateInputSecurityGroupResponse> createInputSecurityGroup(
            CreateInputSecurityGroupRequest createInputSecurityGroupRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createInputSecurityGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaLive");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateInputSecurityGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateInputSecurityGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateInputSecurityGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateInputSecurityGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateInputSecurityGroupRequest, CreateInputSecurityGroupResponse>()
                            .withOperationName("CreateInputSecurityGroup")
                            .withMarshaller(new CreateInputSecurityGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createInputSecurityGroupRequest));
            CompletableFuture<CreateInputSecurityGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Create a new multiplex.
     *
     * @param createMultiplexRequest
     *        A request to create a multiplex.
     * @return A Java Future containing the result of the CreateMultiplex operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException MediaLive can't process your request because of a problem in the request. Please
     *         check your request form and syntax.</li>
     *         <li>UnprocessableEntityException The Multiplex failed validation and could not be created.</li>
     *         <li>InternalServerErrorException Unexpected internal service error.</li>
     *         <li>ForbiddenException You don't have permission to create the multiplex.</li>
     *         <li>BadGatewayException Bad Gateway Error</li>
     *         <li>GatewayTimeoutException Gateway Timeout Error</li>
     *         <li>TooManyRequestsException Request limit exceeded on create multiplex calls to multiplex service.</li>
     *         <li>ConflictException The multiplex is unable to create due to an issue with multiplex resources.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaLiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaLiveAsyncClient.CreateMultiplex
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medialive-2017-10-14/CreateMultiplex" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateMultiplexResponse> createMultiplex(CreateMultiplexRequest createMultiplexRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createMultiplexRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaLive");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateMultiplex");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateMultiplexResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateMultiplexResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateMultiplexResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateMultiplexRequest, CreateMultiplexResponse>()
                            .withOperationName("CreateMultiplex")
                            .withMarshaller(new CreateMultiplexRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createMultiplexRequest));
            CompletableFuture<CreateMultiplexResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Create a new program in the multiplex.
     *
     * @param createMultiplexProgramRequest
     *        A request to create a program in a multiplex.
     * @return A Java Future containing the result of the CreateMultiplexProgram operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException MediaLive can't process your request because of a problem in the request. Please
     *         check your request form and syntax.</li>
     *         <li>UnprocessableEntityException The Multiplex program failed validation and could not be created.</li>
     *         <li>InternalServerErrorException Unexpected internal service error.</li>
     *         <li>ForbiddenException You do not have permission to create a program.</li>
     *         <li>BadGatewayException Bad Gateway Error</li>
     *         <li>GatewayTimeoutException Gateway Timeout Error</li>
     *         <li>TooManyRequestsException Request limit exceeded on create multiplex program calls to multiplex
     *         service.</li>
     *         <li>ConflictException The multiplex program is unable to create due to an issue with multiplex resources.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaLiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaLiveAsyncClient.CreateMultiplexProgram
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medialive-2017-10-14/CreateMultiplexProgram"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateMultiplexProgramResponse> createMultiplexProgram(
            CreateMultiplexProgramRequest createMultiplexProgramRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createMultiplexProgramRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaLive");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateMultiplexProgram");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateMultiplexProgramResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateMultiplexProgramResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateMultiplexProgramResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateMultiplexProgramRequest, CreateMultiplexProgramResponse>()
                            .withOperationName("CreateMultiplexProgram")
                            .withMarshaller(new CreateMultiplexProgramRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createMultiplexProgramRequest));
            CompletableFuture<CreateMultiplexProgramResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Create a partner input
     *
     * @param createPartnerInputRequest
     *        A request to create a partner input
     * @return A Java Future containing the result of the CreatePartnerInput operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException This request was invalid</li>
     *         <li>InternalServerErrorException Internal Service Error</li>
     *         <li>ForbiddenException Access was denied</li>
     *         <li>BadGatewayException Bad Gateway Error</li>
     *         <li>GatewayTimeoutException Gateway Timeout Error</li>
     *         <li>TooManyRequestsException Limit exceeded</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaLiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaLiveAsyncClient.CreatePartnerInput
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medialive-2017-10-14/CreatePartnerInput" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreatePartnerInputResponse> createPartnerInput(CreatePartnerInputRequest createPartnerInputRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createPartnerInputRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaLive");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreatePartnerInput");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreatePartnerInputResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreatePartnerInputResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreatePartnerInputResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreatePartnerInputRequest, CreatePartnerInputResponse>()
                            .withOperationName("CreatePartnerInput")
                            .withMarshaller(new CreatePartnerInputRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createPartnerInputRequest));
            CompletableFuture<CreatePartnerInputResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Create tags for a resource
     *
     * @param createTagsRequest
     *        Placeholder documentation for CreateTagsRequest
     * @return A Java Future containing the result of the CreateTags operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException The arn was not found.</li>
     *         <li>BadRequestException This request was invalid</li>
     *         <li>InternalServerErrorException Internal Service Error</li>
     *         <li>ForbiddenException Access was denied</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaLiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaLiveAsyncClient.CreateTags
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medialive-2017-10-14/CreateTags" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateTagsResponse> createTags(CreateTagsRequest createTagsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createTagsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaLive");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateTags");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateTagsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateTagsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateTagsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateTagsRequest, CreateTagsResponse>().withOperationName("CreateTags")
                            .withMarshaller(new CreateTagsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createTagsRequest));
            CompletableFuture<CreateTagsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Starts deletion of channel. The associated outputs are also deleted.
     *
     * @param deleteChannelRequest
     *        Placeholder documentation for DeleteChannelRequest
     * @return A Java Future containing the result of the DeleteChannel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException This request was invalid</li>
     *         <li>InternalServerErrorException Unexpected internal service error.</li>
     *         <li>ForbiddenException You do not have permission to delete the channel.</li>
     *         <li>BadGatewayException Bad Gateway Error</li>
     *         <li>NotFoundException The channel you're requesting to delete does not exist.</li>
     *         <li>GatewayTimeoutException Gateway Timeout Error</li>
     *         <li>TooManyRequestsException Request limit exceeded on delete channel calls to channel service.</li>
     *         <li>ConflictException The channel is unable to delete due to an issue with channel resources.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaLiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaLiveAsyncClient.DeleteChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medialive-2017-10-14/DeleteChannel" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteChannelResponse> deleteChannel(DeleteChannelRequest deleteChannelRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaLive");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteChannel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteChannelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteChannelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteChannelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteChannelRequest, DeleteChannelResponse>()
                            .withOperationName("DeleteChannel")
                            .withMarshaller(new DeleteChannelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteChannelRequest));
            CompletableFuture<DeleteChannelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Deletes the input end point
     *
     * @param deleteInputRequest
     *        Placeholder documentation for DeleteInputRequest
     * @return A Java Future containing the result of the DeleteInput operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException This request was invalid</li>
     *         <li>InternalServerErrorException Internal Service Error</li>
     *         <li>ForbiddenException Access was denied</li>
     *         <li>BadGatewayException Bad Gateway Error</li>
     *         <li>NotFoundException Input not found</li>
     *         <li>GatewayTimeoutException Gateway Timeout Error</li>
     *         <li>TooManyRequestsException Limit exceeded</li>
     *         <li>ConflictException Resource conflict</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaLiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaLiveAsyncClient.DeleteInput
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medialive-2017-10-14/DeleteInput" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteInputResponse> deleteInput(DeleteInputRequest deleteInputRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteInputRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaLive");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteInput");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteInputResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteInputResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteInputResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteInputRequest, DeleteInputResponse>()
                            .withOperationName("DeleteInput").withMarshaller(new DeleteInputRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteInputRequest));
            CompletableFuture<DeleteInputResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Deletes an Input Security Group
     *
     * @param deleteInputSecurityGroupRequest
     *        Placeholder documentation for DeleteInputSecurityGroupRequest
     * @return A Java Future containing the result of the DeleteInputSecurityGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request to delete the Input Security Group was Invalid</li>
     *         <li>InternalServerErrorException Internal Server Error</li>
     *         <li>ForbiddenException The requester does not have permission to delete this Input Security Group</li>
     *         <li>BadGatewayException Bad Gateway Error</li>
     *         <li>NotFoundException Input Security Group not found</li>
     *         <li>GatewayTimeoutException Gateway Timeout Error</li>
     *         <li>TooManyRequestsException Limit Exceeded Error</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaLiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaLiveAsyncClient.DeleteInputSecurityGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medialive-2017-10-14/DeleteInputSecurityGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteInputSecurityGroupResponse> deleteInputSecurityGroup(
            DeleteInputSecurityGroupRequest deleteInputSecurityGroupRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteInputSecurityGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaLive");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteInputSecurityGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteInputSecurityGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteInputSecurityGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteInputSecurityGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteInputSecurityGroupRequest, DeleteInputSecurityGroupResponse>()
                            .withOperationName("DeleteInputSecurityGroup")
                            .withMarshaller(new DeleteInputSecurityGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteInputSecurityGroupRequest));
            CompletableFuture<DeleteInputSecurityGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Delete a multiplex. The multiplex must be idle.
     *
     * @param deleteMultiplexRequest
     *        Placeholder documentation for DeleteMultiplexRequest
     * @return A Java Future containing the result of the DeleteMultiplex operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException This request was invalid.</li>
     *         <li>InternalServerErrorException Unexpected internal service error.</li>
     *         <li>ForbiddenException You don't have permission to delete the multiplex.</li>
     *         <li>BadGatewayException Bad Gateway Error</li>
     *         <li>NotFoundException The multiplex that you are trying to delete doesn’t exist. Check the ID and try
     *         again.</li>
     *         <li>GatewayTimeoutException Gateway Timeout Error</li>
     *         <li>TooManyRequestsException Request limit exceeded on delete multiplex calls to multiplex service.</li>
     *         <li>ConflictException The multiplex is unable to delete due to an issue with multiplex resources.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaLiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaLiveAsyncClient.DeleteMultiplex
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medialive-2017-10-14/DeleteMultiplex" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteMultiplexResponse> deleteMultiplex(DeleteMultiplexRequest deleteMultiplexRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteMultiplexRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaLive");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteMultiplex");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteMultiplexResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteMultiplexResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteMultiplexResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteMultiplexRequest, DeleteMultiplexResponse>()
                            .withOperationName("DeleteMultiplex")
                            .withMarshaller(new DeleteMultiplexRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteMultiplexRequest));
            CompletableFuture<DeleteMultiplexResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Delete a program from a multiplex.
     *
     * @param deleteMultiplexProgramRequest
     *        Placeholder documentation for DeleteMultiplexProgramRequest
     * @return A Java Future containing the result of the DeleteMultiplexProgram operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException This request was invalid.</li>
     *         <li>InternalServerErrorException Unexpected internal service error.</li>
     *         <li>ForbiddenException You don't have permission to delete the multiplex program.</li>
     *         <li>BadGatewayException Bad Gateway Error</li>
     *         <li>NotFoundException The program that you are trying to delete doesn’t exist. Check the ID and try
     *         again.</li>
     *         <li>GatewayTimeoutException Gateway Timeout Error</li>
     *         <li>TooManyRequestsException Request limit exceeded on delete multiplex program calls to multiplex
     *         service.</li>
     *         <li>ConflictException The multiplex program is unable to delete due to an issue with multiplex resources.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaLiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaLiveAsyncClient.DeleteMultiplexProgram
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medialive-2017-10-14/DeleteMultiplexProgram"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteMultiplexProgramResponse> deleteMultiplexProgram(
            DeleteMultiplexProgramRequest deleteMultiplexProgramRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteMultiplexProgramRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaLive");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteMultiplexProgram");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteMultiplexProgramResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteMultiplexProgramResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteMultiplexProgramResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteMultiplexProgramRequest, DeleteMultiplexProgramResponse>()
                            .withOperationName("DeleteMultiplexProgram")
                            .withMarshaller(new DeleteMultiplexProgramRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteMultiplexProgramRequest));
            CompletableFuture<DeleteMultiplexProgramResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Delete an expired reservation.
     *
     * @param deleteReservationRequest
     *        Placeholder documentation for DeleteReservationRequest
     * @return A Java Future containing the result of the DeleteReservation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException This request was invalid</li>
     *         <li>InternalServerErrorException Internal service error</li>
     *         <li>ForbiddenException You do not have permission to delete reservation</li>
     *         <li>BadGatewayException Bad gateway error</li>
     *         <li>NotFoundException Reservation you're attempting to delete does not exist</li>
     *         <li>GatewayTimeoutException Gateway Timeout Error</li>
     *         <li>TooManyRequestsException Request limit exceeded on delete reservation request</li>
     *         <li>ConflictException The reservation could not be deleted because it is currently active.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaLiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaLiveAsyncClient.DeleteReservation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medialive-2017-10-14/DeleteReservation" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteReservationResponse> deleteReservation(DeleteReservationRequest deleteReservationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteReservationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaLive");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteReservation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteReservationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteReservationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteReservationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteReservationRequest, DeleteReservationResponse>()
                            .withOperationName("DeleteReservation")
                            .withMarshaller(new DeleteReservationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteReservationRequest));
            CompletableFuture<DeleteReservationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Delete all schedule actions on a channel.
     *
     * @param deleteScheduleRequest
     *        Placeholder documentation for DeleteScheduleRequest
     * @return A Java Future containing the result of the DeleteSchedule operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException This request to delete the schedule on this channel was invalid.</li>
     *         <li>InternalServerErrorException Unexpected internal service error.</li>
     *         <li>ForbiddenException You do not have permission to delete the channel schedule.</li>
     *         <li>BadGatewayException Bad Gateway Error</li>
     *         <li>NotFoundException The specified channel does not exist to have its schedule deleted.</li>
     *         <li>GatewayTimeoutException Gateway Timeout Error</li>
     *         <li>TooManyRequestsException Request limit exceeded on delete schedule calls.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaLiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaLiveAsyncClient.DeleteSchedule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medialive-2017-10-14/DeleteSchedule" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteScheduleResponse> deleteSchedule(DeleteScheduleRequest deleteScheduleRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteScheduleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaLive");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSchedule");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteScheduleResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteScheduleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteScheduleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteScheduleRequest, DeleteScheduleResponse>()
                            .withOperationName("DeleteSchedule")
                            .withMarshaller(new DeleteScheduleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteScheduleRequest));
            CompletableFuture<DeleteScheduleResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Removes tags for a resource
     *
     * @param deleteTagsRequest
     *        Placeholder documentation for DeleteTagsRequest
     * @return A Java Future containing the result of the DeleteTags operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException The arn was not found.</li>
     *         <li>BadRequestException This request was invalid</li>
     *         <li>InternalServerErrorException Internal Service Error</li>
     *         <li>ForbiddenException Access was denied</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaLiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaLiveAsyncClient.DeleteTags
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medialive-2017-10-14/DeleteTags" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteTagsResponse> deleteTags(DeleteTagsRequest deleteTagsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteTagsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaLive");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteTags");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteTagsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteTagsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteTagsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteTagsRequest, DeleteTagsResponse>().withOperationName("DeleteTags")
                            .withMarshaller(new DeleteTagsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteTagsRequest));
            CompletableFuture<DeleteTagsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Gets details about a channel
     *
     * @param describeChannelRequest
     *        Placeholder documentation for DescribeChannelRequest
     * @return A Java Future containing the result of the DescribeChannel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException This request was invalid</li>
     *         <li>InternalServerErrorException Unexpected internal service error.</li>
     *         <li>ForbiddenException You do not have permission to describe the channel.</li>
     *         <li>BadGatewayException Bad Gateway Error</li>
     *         <li>NotFoundException The channel you're requesting to describe does not exist.</li>
     *         <li>GatewayTimeoutException Gateway Timeout Error</li>
     *         <li>TooManyRequestsException Request limit exceeded on describe channel calls to channel service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaLiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaLiveAsyncClient.DescribeChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medialive-2017-10-14/DescribeChannel" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeChannelResponse> describeChannel(DescribeChannelRequest describeChannelRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaLive");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeChannel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeChannelResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeChannelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeChannelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeChannelRequest, DescribeChannelResponse>()
                            .withOperationName("DescribeChannel")
                            .withMarshaller(new DescribeChannelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeChannelRequest));
            CompletableFuture<DescribeChannelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Produces details about an input
     *
     * @param describeInputRequest
     *        Placeholder documentation for DescribeInputRequest
     * @return A Java Future containing the result of the DescribeInput operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException This request was invalid</li>
     *         <li>InternalServerErrorException Internal Service Error</li>
     *         <li>ForbiddenException Access was denied</li>
     *         <li>BadGatewayException Bad Gateway Error</li>
     *         <li>NotFoundException Input not found</li>
     *         <li>GatewayTimeoutException Gateway Timeout</li>
     *         <li>TooManyRequestsException Limit exceeded</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaLiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaLiveAsyncClient.DescribeInput
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medialive-2017-10-14/DescribeInput" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeInputResponse> describeInput(DescribeInputRequest describeInputRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeInputRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaLive");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeInput");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeInputResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DescribeInputResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeInputResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeInputRequest, DescribeInputResponse>()
                            .withOperationName("DescribeInput")
                            .withMarshaller(new DescribeInputRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeInputRequest));
            CompletableFuture<DescribeInputResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Gets the details for the input device
     *
     * @param describeInputDeviceRequest
     *        Placeholder documentation for DescribeInputDeviceRequest
     * @return A Java Future containing the result of the DescribeInputDevice operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException This request was invalid.</li>
     *         <li>InternalServerErrorException Unexpected internal service error.</li>
     *         <li>ForbiddenException You do not have permission to describe the input device.</li>
     *         <li>BadGatewayException Bad gateway error.</li>
     *         <li>NotFoundException The input device you're requesting to describe does not exist. Check the ID.</li>
     *         <li>GatewayTimeoutException Gateway Timeout Error</li>
     *         <li>TooManyRequestsException Request limit exceeded on describe calls to the input device service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaLiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaLiveAsyncClient.DescribeInputDevice
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medialive-2017-10-14/DescribeInputDevice" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeInputDeviceResponse> describeInputDevice(
            DescribeInputDeviceRequest describeInputDeviceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeInputDeviceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaLive");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeInputDevice");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeInputDeviceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeInputDeviceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeInputDeviceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeInputDeviceRequest, DescribeInputDeviceResponse>()
                            .withOperationName("DescribeInputDevice")
                            .withMarshaller(new DescribeInputDeviceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeInputDeviceRequest));
            CompletableFuture<DescribeInputDeviceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Get the latest thumbnail data for the input device.
     *
     * @param describeInputDeviceThumbnailRequest
     *        Placeholder documentation for DescribeInputDeviceThumbnailRequest
     * @param asyncResponseTransformer
     *        The response transformer for processing the streaming response in a non-blocking manner. See
     *        {@link AsyncResponseTransformer} for details on how this callback should be implemented and for links to
     *        precanned implementations for common scenarios like downloading to a file. The service documentation for
     *        the response content is as follows 'The binary data for the thumbnail that the Link device has most
     *        recently sent to MediaLive.'.
     * @return A future to the transformed result of the AsyncResponseTransformer.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException This request was invalid.</li>
     *         <li>InternalServerErrorException Unexpected internal service error.</li>
     *         <li>ForbiddenException You do not have permission to describe input device thumbnail.</li>
     *         <li>BadGatewayException Bad gateway error.</li>
     *         <li>NotFoundException Input device not found.</li>
     *         <li>GatewayTimeoutException Gateway Timeout Error</li>
     *         <li>TooManyRequestsException Request limit exceeded on get thumbnail calls to the input device service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaLiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaLiveAsyncClient.DescribeInputDeviceThumbnail
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medialive-2017-10-14/DescribeInputDeviceThumbnail"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public <ReturnT> CompletableFuture<ReturnT> describeInputDeviceThumbnail(
            DescribeInputDeviceThumbnailRequest describeInputDeviceThumbnailRequest,
            AsyncResponseTransformer<DescribeInputDeviceThumbnailResponse, ReturnT> asyncResponseTransformer) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeInputDeviceThumbnailRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaLive");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeInputDeviceThumbnail");
            Pair<AsyncResponseTransformer<DescribeInputDeviceThumbnailResponse, ReturnT>, CompletableFuture<Void>> pair = AsyncResponseTransformerUtils
                    .wrapWithEndOfStreamFuture(asyncResponseTransformer);
            asyncResponseTransformer = pair.left();
            CompletableFuture<Void> endOfStreamFuture = pair.right();
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(true)
                    .isPayloadJson(false).build();

            HttpResponseHandler<DescribeInputDeviceThumbnailResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeInputDeviceThumbnailResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ReturnT> executeFuture = clientHandler.execute(
                    new ClientExecutionParams<DescribeInputDeviceThumbnailRequest, DescribeInputDeviceThumbnailResponse>()
                            .withOperationName("DescribeInputDeviceThumbnail")
                            .withMarshaller(new DescribeInputDeviceThumbnailRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeInputDeviceThumbnailRequest),
                    asyncResponseTransformer);
            AsyncResponseTransformer<DescribeInputDeviceThumbnailResponse, ReturnT> finalAsyncResponseTransformer = asyncResponseTransformer;
            CompletableFuture<ReturnT> whenCompleted = executeFuture.whenComplete((r, e) -> {
                if (e != null) {
                    runAndLogError(log, "Exception thrown in exceptionOccurred callback, ignoring",
                            () -> finalAsyncResponseTransformer.exceptionOccurred(e));
                }
                endOfStreamFuture.whenComplete((r2, e2) -> {
                    metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                });
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            AsyncResponseTransformer<DescribeInputDeviceThumbnailResponse, ReturnT> finalAsyncResponseTransformer = asyncResponseTransformer;
            runAndLogError(log, "Exception thrown in exceptionOccurred callback, ignoring",
                    () -> finalAsyncResponseTransformer.exceptionOccurred(t));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Produces a summary of an Input Security Group
     *
     * @param describeInputSecurityGroupRequest
     *        Placeholder documentation for DescribeInputSecurityGroupRequest
     * @return A Java Future containing the result of the DescribeInputSecurityGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request to describe an Input Security Group was Invalid</li>
     *         <li>InternalServerErrorException Internal Server Error</li>
     *         <li>ForbiddenException The requester does not have permission to describe this Input Security Group</li>
     *         <li>BadGatewayException Bad Gateway Error</li>
     *         <li>NotFoundException Input Security Group not found</li>
     *         <li>GatewayTimeoutException Gateway Timeout Error</li>
     *         <li>TooManyRequestsException Limit Exceeded Error</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaLiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaLiveAsyncClient.DescribeInputSecurityGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medialive-2017-10-14/DescribeInputSecurityGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeInputSecurityGroupResponse> describeInputSecurityGroup(
            DescribeInputSecurityGroupRequest describeInputSecurityGroupRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeInputSecurityGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaLive");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeInputSecurityGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeInputSecurityGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeInputSecurityGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeInputSecurityGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeInputSecurityGroupRequest, DescribeInputSecurityGroupResponse>()
                            .withOperationName("DescribeInputSecurityGroup")
                            .withMarshaller(new DescribeInputSecurityGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeInputSecurityGroupRequest));
            CompletableFuture<DescribeInputSecurityGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Gets details about a multiplex.
     *
     * @param describeMultiplexRequest
     *        Placeholder documentation for DescribeMultiplexRequest
     * @return A Java Future containing the result of the DescribeMultiplex operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException This request was invalid.</li>
     *         <li>InternalServerErrorException Unexpected internal service error.</li>
     *         <li>ForbiddenException You don't have permission to describe the multiplex.</li>
     *         <li>BadGatewayException Bad Gateway Error</li>
     *         <li>NotFoundException The multiplex that you are trying to describe doesn’t exist. Check the ID and try
     *         again.</li>
     *         <li>GatewayTimeoutException Gateway Timeout Error</li>
     *         <li>TooManyRequestsException Request limit exceeded on describe multiplex calls to multiplex service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaLiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaLiveAsyncClient.DescribeMultiplex
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medialive-2017-10-14/DescribeMultiplex" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeMultiplexResponse> describeMultiplex(DescribeMultiplexRequest describeMultiplexRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeMultiplexRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaLive");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeMultiplex");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeMultiplexResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeMultiplexResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeMultiplexResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeMultiplexRequest, DescribeMultiplexResponse>()
                            .withOperationName("DescribeMultiplex")
                            .withMarshaller(new DescribeMultiplexRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeMultiplexRequest));
            CompletableFuture<DescribeMultiplexResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Get the details for a program in a multiplex.
     *
     * @param describeMultiplexProgramRequest
     *        Placeholder documentation for DescribeMultiplexProgramRequest
     * @return A Java Future containing the result of the DescribeMultiplexProgram operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException This request was invalid.</li>
     *         <li>InternalServerErrorException Unexpected internal service error.</li>
     *         <li>ForbiddenException You don't have permission to describe the multiplex program.</li>
     *         <li>BadGatewayException Bad Gateway Error</li>
     *         <li>NotFoundException MediaLive can't describe the program. The multiplex or the program that you
     *         specified doesn’t exist. Check the IDs and try again.</li>
     *         <li>GatewayTimeoutException Gateway Timeout Error</li>
     *         <li>TooManyRequestsException Request limit exceeded on describe multiplex program calls to multiplex
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaLiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaLiveAsyncClient.DescribeMultiplexProgram
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medialive-2017-10-14/DescribeMultiplexProgram"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeMultiplexProgramResponse> describeMultiplexProgram(
            DescribeMultiplexProgramRequest describeMultiplexProgramRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeMultiplexProgramRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaLive");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeMultiplexProgram");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeMultiplexProgramResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeMultiplexProgramResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeMultiplexProgramResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeMultiplexProgramRequest, DescribeMultiplexProgramResponse>()
                            .withOperationName("DescribeMultiplexProgram")
                            .withMarshaller(new DescribeMultiplexProgramRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeMultiplexProgramRequest));
            CompletableFuture<DescribeMultiplexProgramResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Get details for an offering.
     *
     * @param describeOfferingRequest
     *        Placeholder documentation for DescribeOfferingRequest
     * @return A Java Future containing the result of the DescribeOffering operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException This request was invalid</li>
     *         <li>InternalServerErrorException Internal service error</li>
     *         <li>ForbiddenException You do not have permission to describe offering</li>
     *         <li>BadGatewayException Bad gateway error</li>
     *         <li>NotFoundException Offering you're attempting to describe does not exist</li>
     *         <li>GatewayTimeoutException Gateway Timeout Error</li>
     *         <li>TooManyRequestsException Request limit exceeded on describe offering request</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaLiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaLiveAsyncClient.DescribeOffering
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medialive-2017-10-14/DescribeOffering" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeOfferingResponse> describeOffering(DescribeOfferingRequest describeOfferingRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeOfferingRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaLive");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeOffering");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeOfferingResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeOfferingResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeOfferingResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeOfferingRequest, DescribeOfferingResponse>()
                            .withOperationName("DescribeOffering")
                            .withMarshaller(new DescribeOfferingRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeOfferingRequest));
            CompletableFuture<DescribeOfferingResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Get details for a reservation.
     *
     * @param describeReservationRequest
     *        Placeholder documentation for DescribeReservationRequest
     * @return A Java Future containing the result of the DescribeReservation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException This request was invalid</li>
     *         <li>InternalServerErrorException Internal service error</li>
     *         <li>ForbiddenException You do not have permission to describe reservation</li>
     *         <li>BadGatewayException Bad gateway error</li>
     *         <li>NotFoundException Reservation you're attempting to describe does not exist</li>
     *         <li>GatewayTimeoutException Gateway Timeout Error</li>
     *         <li>TooManyRequestsException Request limit exceeded on describe reservation request</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaLiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaLiveAsyncClient.DescribeReservation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medialive-2017-10-14/DescribeReservation" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeReservationResponse> describeReservation(
            DescribeReservationRequest describeReservationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeReservationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaLive");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeReservation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeReservationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeReservationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeReservationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeReservationRequest, DescribeReservationResponse>()
                            .withOperationName("DescribeReservation")
                            .withMarshaller(new DescribeReservationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeReservationRequest));
            CompletableFuture<DescribeReservationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Get a channel schedule
     *
     * @param describeScheduleRequest
     *        Placeholder documentation for DescribeScheduleRequest
     * @return A Java Future containing the result of the DescribeSchedule operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException This request was invalid.</li>
     *         <li>InternalServerErrorException Unexpected internal service error.</li>
     *         <li>ForbiddenException You do not have permission to describe the channel schedule.</li>
     *         <li>BadGatewayException Bad Gateway Error</li>
     *         <li>NotFoundException The channel you're requesting a schedule describe for does not exist.</li>
     *         <li>GatewayTimeoutException Gateway Timeout Error</li>
     *         <li>TooManyRequestsException Request limit exceeded on describe schedule calls.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaLiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaLiveAsyncClient.DescribeSchedule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medialive-2017-10-14/DescribeSchedule" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeScheduleResponse> describeSchedule(DescribeScheduleRequest describeScheduleRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeScheduleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaLive");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeSchedule");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeScheduleResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeScheduleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeScheduleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeScheduleRequest, DescribeScheduleResponse>()
                            .withOperationName("DescribeSchedule")
                            .withMarshaller(new DescribeScheduleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeScheduleRequest));
            CompletableFuture<DescribeScheduleResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Get a channel schedule<br/>
     * <p>
     * This is a variant of
     * {@link #describeSchedule(software.amazon.awssdk.services.medialive.model.DescribeScheduleRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.medialive.paginators.DescribeSchedulePublisher publisher = client.describeSchedulePaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.medialive.paginators.DescribeSchedulePublisher publisher = client.describeSchedulePaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.medialive.model.DescribeScheduleResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.medialive.model.DescribeScheduleResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeSchedule(software.amazon.awssdk.services.medialive.model.DescribeScheduleRequest)} operation.</b>
     * </p>
     *
     * @param describeScheduleRequest
     *        Placeholder documentation for DescribeScheduleRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException This request was invalid.</li>
     *         <li>InternalServerErrorException Unexpected internal service error.</li>
     *         <li>ForbiddenException You do not have permission to describe the channel schedule.</li>
     *         <li>BadGatewayException Bad Gateway Error</li>
     *         <li>NotFoundException The channel you're requesting a schedule describe for does not exist.</li>
     *         <li>GatewayTimeoutException Gateway Timeout Error</li>
     *         <li>TooManyRequestsException Request limit exceeded on describe schedule calls.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaLiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaLiveAsyncClient.DescribeSchedule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medialive-2017-10-14/DescribeSchedule" target="_top">AWS
     *      API Documentation</a>
     */
    public DescribeSchedulePublisher describeSchedulePaginator(DescribeScheduleRequest describeScheduleRequest) {
        return new DescribeSchedulePublisher(this, applyPaginatorUserAgent(describeScheduleRequest));
    }

    /**
     * Produces list of channels that have been created
     *
     * @param listChannelsRequest
     *        Placeholder documentation for ListChannelsRequest
     * @return A Java Future containing the result of the ListChannels operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException This request was invalid.</li>
     *         <li>InternalServerErrorException Unexpected internal service error.</li>
     *         <li>ForbiddenException You do not have permission to list channels.</li>
     *         <li>BadGatewayException Bad Gateway Error</li>
     *         <li>GatewayTimeoutException Gateway Timeout Error</li>
     *         <li>TooManyRequestsException Request limit exceeded on list channel calls to channel service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaLiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaLiveAsyncClient.ListChannels
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medialive-2017-10-14/ListChannels" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListChannelsResponse> listChannels(ListChannelsRequest listChannelsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listChannelsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaLive");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListChannels");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListChannelsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListChannelsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListChannelsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListChannelsRequest, ListChannelsResponse>()
                            .withOperationName("ListChannels").withMarshaller(new ListChannelsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listChannelsRequest));
            CompletableFuture<ListChannelsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Produces list of channels that have been created<br/>
     * <p>
     * This is a variant of {@link #listChannels(software.amazon.awssdk.services.medialive.model.ListChannelsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.medialive.paginators.ListChannelsPublisher publisher = client.listChannelsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.medialive.paginators.ListChannelsPublisher publisher = client.listChannelsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.medialive.model.ListChannelsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.medialive.model.ListChannelsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listChannels(software.amazon.awssdk.services.medialive.model.ListChannelsRequest)} operation.</b>
     * </p>
     *
     * @param listChannelsRequest
     *        Placeholder documentation for ListChannelsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException This request was invalid.</li>
     *         <li>InternalServerErrorException Unexpected internal service error.</li>
     *         <li>ForbiddenException You do not have permission to list channels.</li>
     *         <li>BadGatewayException Bad Gateway Error</li>
     *         <li>GatewayTimeoutException Gateway Timeout Error</li>
     *         <li>TooManyRequestsException Request limit exceeded on list channel calls to channel service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaLiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaLiveAsyncClient.ListChannels
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medialive-2017-10-14/ListChannels" target="_top">AWS API
     *      Documentation</a>
     */
    public ListChannelsPublisher listChannelsPaginator(ListChannelsRequest listChannelsRequest) {
        return new ListChannelsPublisher(this, applyPaginatorUserAgent(listChannelsRequest));
    }

    /**
     * List input devices that are currently being transferred. List input devices that you are transferring from your
     * AWS account or input devices that another AWS account is transferring to you.
     *
     * @param listInputDeviceTransfersRequest
     *        Placeholder documentation for ListInputDeviceTransfersRequest
     * @return A Java Future containing the result of the ListInputDeviceTransfers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException This request was invalid.</li>
     *         <li>UnprocessableEntityException Transfer operation failed, input could not be validated.</li>
     *         <li>InternalServerErrorException Unexpected internal service error.</li>
     *         <li>ForbiddenException You do not have permission to list transferring devices.</li>
     *         <li>BadGatewayException Bad gateway error.</li>
     *         <li>GatewayTimeoutException Gateway Timeout Error</li>
     *         <li>TooManyRequestsException Request limit exceeded on list devices calls to the input device service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaLiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaLiveAsyncClient.ListInputDeviceTransfers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medialive-2017-10-14/ListInputDeviceTransfers"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListInputDeviceTransfersResponse> listInputDeviceTransfers(
            ListInputDeviceTransfersRequest listInputDeviceTransfersRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listInputDeviceTransfersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaLive");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListInputDeviceTransfers");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListInputDeviceTransfersResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListInputDeviceTransfersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListInputDeviceTransfersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListInputDeviceTransfersRequest, ListInputDeviceTransfersResponse>()
                            .withOperationName("ListInputDeviceTransfers")
                            .withMarshaller(new ListInputDeviceTransfersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listInputDeviceTransfersRequest));
            CompletableFuture<ListInputDeviceTransfersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * List input devices that are currently being transferred. List input devices that you are transferring from your
     * AWS account or input devices that another AWS account is transferring to you.<br/>
     * <p>
     * This is a variant of
     * {@link #listInputDeviceTransfers(software.amazon.awssdk.services.medialive.model.ListInputDeviceTransfersRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.medialive.paginators.ListInputDeviceTransfersPublisher publisher = client.listInputDeviceTransfersPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.medialive.paginators.ListInputDeviceTransfersPublisher publisher = client.listInputDeviceTransfersPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.medialive.model.ListInputDeviceTransfersResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.medialive.model.ListInputDeviceTransfersResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listInputDeviceTransfers(software.amazon.awssdk.services.medialive.model.ListInputDeviceTransfersRequest)}
     * operation.</b>
     * </p>
     *
     * @param listInputDeviceTransfersRequest
     *        Placeholder documentation for ListInputDeviceTransfersRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException This request was invalid.</li>
     *         <li>UnprocessableEntityException Transfer operation failed, input could not be validated.</li>
     *         <li>InternalServerErrorException Unexpected internal service error.</li>
     *         <li>ForbiddenException You do not have permission to list transferring devices.</li>
     *         <li>BadGatewayException Bad gateway error.</li>
     *         <li>GatewayTimeoutException Gateway Timeout Error</li>
     *         <li>TooManyRequestsException Request limit exceeded on list devices calls to the input device service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaLiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaLiveAsyncClient.ListInputDeviceTransfers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medialive-2017-10-14/ListInputDeviceTransfers"
     *      target="_top">AWS API Documentation</a>
     */
    public ListInputDeviceTransfersPublisher listInputDeviceTransfersPaginator(
            ListInputDeviceTransfersRequest listInputDeviceTransfersRequest) {
        return new ListInputDeviceTransfersPublisher(this, applyPaginatorUserAgent(listInputDeviceTransfersRequest));
    }

    /**
     * List input devices
     *
     * @param listInputDevicesRequest
     *        Placeholder documentation for ListInputDevicesRequest
     * @return A Java Future containing the result of the ListInputDevices operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException This request was invalid.</li>
     *         <li>InternalServerErrorException Unexpected internal service error.</li>
     *         <li>ForbiddenException You do not have permission to list input devices.</li>
     *         <li>BadGatewayException Bad gateway error.</li>
     *         <li>GatewayTimeoutException Gateway Timeout Error</li>
     *         <li>TooManyRequestsException Request limit exceeded on list devices calls to the input device service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaLiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaLiveAsyncClient.ListInputDevices
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medialive-2017-10-14/ListInputDevices" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListInputDevicesResponse> listInputDevices(ListInputDevicesRequest listInputDevicesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listInputDevicesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaLive");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListInputDevices");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListInputDevicesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListInputDevicesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListInputDevicesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListInputDevicesRequest, ListInputDevicesResponse>()
                            .withOperationName("ListInputDevices")
                            .withMarshaller(new ListInputDevicesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listInputDevicesRequest));
            CompletableFuture<ListInputDevicesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * List input devices<br/>
     * <p>
     * This is a variant of
     * {@link #listInputDevices(software.amazon.awssdk.services.medialive.model.ListInputDevicesRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.medialive.paginators.ListInputDevicesPublisher publisher = client.listInputDevicesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.medialive.paginators.ListInputDevicesPublisher publisher = client.listInputDevicesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.medialive.model.ListInputDevicesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.medialive.model.ListInputDevicesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listInputDevices(software.amazon.awssdk.services.medialive.model.ListInputDevicesRequest)} operation.</b>
     * </p>
     *
     * @param listInputDevicesRequest
     *        Placeholder documentation for ListInputDevicesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException This request was invalid.</li>
     *         <li>InternalServerErrorException Unexpected internal service error.</li>
     *         <li>ForbiddenException You do not have permission to list input devices.</li>
     *         <li>BadGatewayException Bad gateway error.</li>
     *         <li>GatewayTimeoutException Gateway Timeout Error</li>
     *         <li>TooManyRequestsException Request limit exceeded on list devices calls to the input device service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaLiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaLiveAsyncClient.ListInputDevices
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medialive-2017-10-14/ListInputDevices" target="_top">AWS
     *      API Documentation</a>
     */
    public ListInputDevicesPublisher listInputDevicesPaginator(ListInputDevicesRequest listInputDevicesRequest) {
        return new ListInputDevicesPublisher(this, applyPaginatorUserAgent(listInputDevicesRequest));
    }

    /**
     * Produces a list of Input Security Groups for an account
     *
     * @param listInputSecurityGroupsRequest
     *        Placeholder documentation for ListInputSecurityGroupsRequest
     * @return A Java Future containing the result of the ListInputSecurityGroups operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request to list Input Security Groups was invalid</li>
     *         <li>InternalServerErrorException Internal Server Error</li>
     *         <li>ForbiddenException The requester does not have permission to call ListInputSecurityGroups</li>
     *         <li>BadGatewayException Bad Gateway Error</li>
     *         <li>GatewayTimeoutException Gateway Timeout Error</li>
     *         <li>TooManyRequestsException Limit Exceeded Error</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaLiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaLiveAsyncClient.ListInputSecurityGroups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medialive-2017-10-14/ListInputSecurityGroups"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListInputSecurityGroupsResponse> listInputSecurityGroups(
            ListInputSecurityGroupsRequest listInputSecurityGroupsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listInputSecurityGroupsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaLive");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListInputSecurityGroups");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListInputSecurityGroupsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListInputSecurityGroupsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListInputSecurityGroupsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListInputSecurityGroupsRequest, ListInputSecurityGroupsResponse>()
                            .withOperationName("ListInputSecurityGroups")
                            .withMarshaller(new ListInputSecurityGroupsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listInputSecurityGroupsRequest));
            CompletableFuture<ListInputSecurityGroupsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Produces a list of Input Security Groups for an account<br/>
     * <p>
     * This is a variant of
     * {@link #listInputSecurityGroups(software.amazon.awssdk.services.medialive.model.ListInputSecurityGroupsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.medialive.paginators.ListInputSecurityGroupsPublisher publisher = client.listInputSecurityGroupsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.medialive.paginators.ListInputSecurityGroupsPublisher publisher = client.listInputSecurityGroupsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.medialive.model.ListInputSecurityGroupsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.medialive.model.ListInputSecurityGroupsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listInputSecurityGroups(software.amazon.awssdk.services.medialive.model.ListInputSecurityGroupsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listInputSecurityGroupsRequest
     *        Placeholder documentation for ListInputSecurityGroupsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request to list Input Security Groups was invalid</li>
     *         <li>InternalServerErrorException Internal Server Error</li>
     *         <li>ForbiddenException The requester does not have permission to call ListInputSecurityGroups</li>
     *         <li>BadGatewayException Bad Gateway Error</li>
     *         <li>GatewayTimeoutException Gateway Timeout Error</li>
     *         <li>TooManyRequestsException Limit Exceeded Error</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaLiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaLiveAsyncClient.ListInputSecurityGroups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medialive-2017-10-14/ListInputSecurityGroups"
     *      target="_top">AWS API Documentation</a>
     */
    public ListInputSecurityGroupsPublisher listInputSecurityGroupsPaginator(
            ListInputSecurityGroupsRequest listInputSecurityGroupsRequest) {
        return new ListInputSecurityGroupsPublisher(this, applyPaginatorUserAgent(listInputSecurityGroupsRequest));
    }

    /**
     * Produces list of inputs that have been created
     *
     * @param listInputsRequest
     *        Placeholder documentation for ListInputsRequest
     * @return A Java Future containing the result of the ListInputs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException This request was invalid</li>
     *         <li>InternalServerErrorException Internal Service Error</li>
     *         <li>ForbiddenException Access was denied</li>
     *         <li>BadGatewayException Bad Gateway Error</li>
     *         <li>GatewayTimeoutException Gateway Timeout</li>
     *         <li>TooManyRequestsException Limit exceeded</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaLiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaLiveAsyncClient.ListInputs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medialive-2017-10-14/ListInputs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListInputsResponse> listInputs(ListInputsRequest listInputsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listInputsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaLive");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListInputs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListInputsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListInputsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListInputsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListInputsRequest, ListInputsResponse>().withOperationName("ListInputs")
                            .withMarshaller(new ListInputsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listInputsRequest));
            CompletableFuture<ListInputsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Produces list of inputs that have been created<br/>
     * <p>
     * This is a variant of {@link #listInputs(software.amazon.awssdk.services.medialive.model.ListInputsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.medialive.paginators.ListInputsPublisher publisher = client.listInputsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.medialive.paginators.ListInputsPublisher publisher = client.listInputsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.medialive.model.ListInputsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.medialive.model.ListInputsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listInputs(software.amazon.awssdk.services.medialive.model.ListInputsRequest)} operation.</b>
     * </p>
     *
     * @param listInputsRequest
     *        Placeholder documentation for ListInputsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException This request was invalid</li>
     *         <li>InternalServerErrorException Internal Service Error</li>
     *         <li>ForbiddenException Access was denied</li>
     *         <li>BadGatewayException Bad Gateway Error</li>
     *         <li>GatewayTimeoutException Gateway Timeout</li>
     *         <li>TooManyRequestsException Limit exceeded</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaLiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaLiveAsyncClient.ListInputs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medialive-2017-10-14/ListInputs" target="_top">AWS API
     *      Documentation</a>
     */
    public ListInputsPublisher listInputsPaginator(ListInputsRequest listInputsRequest) {
        return new ListInputsPublisher(this, applyPaginatorUserAgent(listInputsRequest));
    }

    /**
     * List the programs that currently exist for a specific multiplex.
     *
     * @param listMultiplexProgramsRequest
     *        Placeholder documentation for ListMultiplexProgramsRequest
     * @return A Java Future containing the result of the ListMultiplexPrograms operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException MediaLive can't process your request because of a problem in the request. Please
     *         check your request form and syntax.</li>
     *         <li>InternalServerErrorException Unexpected internal service error.</li>
     *         <li>ForbiddenException You do not have permission to list multiplex programs.</li>
     *         <li>BadGatewayException Bad Gateway Error</li>
     *         <li>NotFoundException MediaLive can't provide the list of programs. The multiplex that you specified
     *         doesn’t exist. Check the ID and try again.</li>
     *         <li>GatewayTimeoutException Gateway Timeout Error</li>
     *         <li>TooManyRequestsException Request limit exceeded on list multiplex calls to multiplex service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaLiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaLiveAsyncClient.ListMultiplexPrograms
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medialive-2017-10-14/ListMultiplexPrograms"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListMultiplexProgramsResponse> listMultiplexPrograms(
            ListMultiplexProgramsRequest listMultiplexProgramsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listMultiplexProgramsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaLive");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListMultiplexPrograms");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListMultiplexProgramsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListMultiplexProgramsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListMultiplexProgramsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListMultiplexProgramsRequest, ListMultiplexProgramsResponse>()
                            .withOperationName("ListMultiplexPrograms")
                            .withMarshaller(new ListMultiplexProgramsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listMultiplexProgramsRequest));
            CompletableFuture<ListMultiplexProgramsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * List the programs that currently exist for a specific multiplex.<br/>
     * <p>
     * This is a variant of
     * {@link #listMultiplexPrograms(software.amazon.awssdk.services.medialive.model.ListMultiplexProgramsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.medialive.paginators.ListMultiplexProgramsPublisher publisher = client.listMultiplexProgramsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.medialive.paginators.ListMultiplexProgramsPublisher publisher = client.listMultiplexProgramsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.medialive.model.ListMultiplexProgramsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.medialive.model.ListMultiplexProgramsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listMultiplexPrograms(software.amazon.awssdk.services.medialive.model.ListMultiplexProgramsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listMultiplexProgramsRequest
     *        Placeholder documentation for ListMultiplexProgramsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException MediaLive can't process your request because of a problem in the request. Please
     *         check your request form and syntax.</li>
     *         <li>InternalServerErrorException Unexpected internal service error.</li>
     *         <li>ForbiddenException You do not have permission to list multiplex programs.</li>
     *         <li>BadGatewayException Bad Gateway Error</li>
     *         <li>NotFoundException MediaLive can't provide the list of programs. The multiplex that you specified
     *         doesn’t exist. Check the ID and try again.</li>
     *         <li>GatewayTimeoutException Gateway Timeout Error</li>
     *         <li>TooManyRequestsException Request limit exceeded on list multiplex calls to multiplex service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaLiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaLiveAsyncClient.ListMultiplexPrograms
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medialive-2017-10-14/ListMultiplexPrograms"
     *      target="_top">AWS API Documentation</a>
     */
    public ListMultiplexProgramsPublisher listMultiplexProgramsPaginator(ListMultiplexProgramsRequest listMultiplexProgramsRequest) {
        return new ListMultiplexProgramsPublisher(this, applyPaginatorUserAgent(listMultiplexProgramsRequest));
    }

    /**
     * Retrieve a list of the existing multiplexes.
     *
     * @param listMultiplexesRequest
     *        Placeholder documentation for ListMultiplexesRequest
     * @return A Java Future containing the result of the ListMultiplexes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException MediaLive can't process your request because of a problem in the request. Please
     *         check your request form and syntax.</li>
     *         <li>InternalServerErrorException Unexpected internal service error.</li>
     *         <li>ForbiddenException You don't have permission to list multiplexes.</li>
     *         <li>BadGatewayException Bad Gateway Error</li>
     *         <li>GatewayTimeoutException Gateway Timeout Error</li>
     *         <li>TooManyRequestsException Request limit exceeded on list multiplex calls to multiplex service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaLiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaLiveAsyncClient.ListMultiplexes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medialive-2017-10-14/ListMultiplexes" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListMultiplexesResponse> listMultiplexes(ListMultiplexesRequest listMultiplexesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listMultiplexesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaLive");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListMultiplexes");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListMultiplexesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListMultiplexesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListMultiplexesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListMultiplexesRequest, ListMultiplexesResponse>()
                            .withOperationName("ListMultiplexes")
                            .withMarshaller(new ListMultiplexesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listMultiplexesRequest));
            CompletableFuture<ListMultiplexesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Retrieve a list of the existing multiplexes.<br/>
     * <p>
     * This is a variant of
     * {@link #listMultiplexes(software.amazon.awssdk.services.medialive.model.ListMultiplexesRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.medialive.paginators.ListMultiplexesPublisher publisher = client.listMultiplexesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.medialive.paginators.ListMultiplexesPublisher publisher = client.listMultiplexesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.medialive.model.ListMultiplexesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.medialive.model.ListMultiplexesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listMultiplexes(software.amazon.awssdk.services.medialive.model.ListMultiplexesRequest)} operation.</b>
     * </p>
     *
     * @param listMultiplexesRequest
     *        Placeholder documentation for ListMultiplexesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException MediaLive can't process your request because of a problem in the request. Please
     *         check your request form and syntax.</li>
     *         <li>InternalServerErrorException Unexpected internal service error.</li>
     *         <li>ForbiddenException You don't have permission to list multiplexes.</li>
     *         <li>BadGatewayException Bad Gateway Error</li>
     *         <li>GatewayTimeoutException Gateway Timeout Error</li>
     *         <li>TooManyRequestsException Request limit exceeded on list multiplex calls to multiplex service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaLiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaLiveAsyncClient.ListMultiplexes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medialive-2017-10-14/ListMultiplexes" target="_top">AWS API
     *      Documentation</a>
     */
    public ListMultiplexesPublisher listMultiplexesPaginator(ListMultiplexesRequest listMultiplexesRequest) {
        return new ListMultiplexesPublisher(this, applyPaginatorUserAgent(listMultiplexesRequest));
    }

    /**
     * List offerings available for purchase.
     *
     * @param listOfferingsRequest
     *        Placeholder documentation for ListOfferingsRequest
     * @return A Java Future containing the result of the ListOfferings operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException This request was invalid</li>
     *         <li>InternalServerErrorException Internal service error</li>
     *         <li>ForbiddenException You do not have permission to list offerings</li>
     *         <li>BadGatewayException Bad gateway error</li>
     *         <li>GatewayTimeoutException Gateway Timeout Error</li>
     *         <li>TooManyRequestsException Request limit exceeded on list offerings request</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaLiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaLiveAsyncClient.ListOfferings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medialive-2017-10-14/ListOfferings" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListOfferingsResponse> listOfferings(ListOfferingsRequest listOfferingsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listOfferingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaLive");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListOfferings");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListOfferingsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListOfferingsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListOfferingsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListOfferingsRequest, ListOfferingsResponse>()
                            .withOperationName("ListOfferings")
                            .withMarshaller(new ListOfferingsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listOfferingsRequest));
            CompletableFuture<ListOfferingsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * List offerings available for purchase.<br/>
     * <p>
     * This is a variant of {@link #listOfferings(software.amazon.awssdk.services.medialive.model.ListOfferingsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.medialive.paginators.ListOfferingsPublisher publisher = client.listOfferingsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.medialive.paginators.ListOfferingsPublisher publisher = client.listOfferingsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.medialive.model.ListOfferingsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.medialive.model.ListOfferingsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listOfferings(software.amazon.awssdk.services.medialive.model.ListOfferingsRequest)} operation.</b>
     * </p>
     *
     * @param listOfferingsRequest
     *        Placeholder documentation for ListOfferingsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException This request was invalid</li>
     *         <li>InternalServerErrorException Internal service error</li>
     *         <li>ForbiddenException You do not have permission to list offerings</li>
     *         <li>BadGatewayException Bad gateway error</li>
     *         <li>GatewayTimeoutException Gateway Timeout Error</li>
     *         <li>TooManyRequestsException Request limit exceeded on list offerings request</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaLiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaLiveAsyncClient.ListOfferings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medialive-2017-10-14/ListOfferings" target="_top">AWS API
     *      Documentation</a>
     */
    public ListOfferingsPublisher listOfferingsPaginator(ListOfferingsRequest listOfferingsRequest) {
        return new ListOfferingsPublisher(this, applyPaginatorUserAgent(listOfferingsRequest));
    }

    /**
     * List purchased reservations.
     *
     * @param listReservationsRequest
     *        Placeholder documentation for ListReservationsRequest
     * @return A Java Future containing the result of the ListReservations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException This request was invalid</li>
     *         <li>InternalServerErrorException Internal service error</li>
     *         <li>ForbiddenException You do not have permission to list reservations</li>
     *         <li>BadGatewayException Bad gateway error</li>
     *         <li>GatewayTimeoutException Gateway Timeout Error</li>
     *         <li>TooManyRequestsException Request limit exceeded on list reservations request</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaLiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaLiveAsyncClient.ListReservations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medialive-2017-10-14/ListReservations" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListReservationsResponse> listReservations(ListReservationsRequest listReservationsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listReservationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaLive");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListReservations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListReservationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListReservationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListReservationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListReservationsRequest, ListReservationsResponse>()
                            .withOperationName("ListReservations")
                            .withMarshaller(new ListReservationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listReservationsRequest));
            CompletableFuture<ListReservationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * List purchased reservations.<br/>
     * <p>
     * This is a variant of
     * {@link #listReservations(software.amazon.awssdk.services.medialive.model.ListReservationsRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.medialive.paginators.ListReservationsPublisher publisher = client.listReservationsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.medialive.paginators.ListReservationsPublisher publisher = client.listReservationsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.medialive.model.ListReservationsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.medialive.model.ListReservationsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listReservations(software.amazon.awssdk.services.medialive.model.ListReservationsRequest)} operation.</b>
     * </p>
     *
     * @param listReservationsRequest
     *        Placeholder documentation for ListReservationsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException This request was invalid</li>
     *         <li>InternalServerErrorException Internal service error</li>
     *         <li>ForbiddenException You do not have permission to list reservations</li>
     *         <li>BadGatewayException Bad gateway error</li>
     *         <li>GatewayTimeoutException Gateway Timeout Error</li>
     *         <li>TooManyRequestsException Request limit exceeded on list reservations request</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaLiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaLiveAsyncClient.ListReservations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medialive-2017-10-14/ListReservations" target="_top">AWS
     *      API Documentation</a>
     */
    public ListReservationsPublisher listReservationsPaginator(ListReservationsRequest listReservationsRequest) {
        return new ListReservationsPublisher(this, applyPaginatorUserAgent(listReservationsRequest));
    }

    /**
     * Produces list of tags that have been created for a resource
     *
     * @param listTagsForResourceRequest
     *        Placeholder documentation for ListTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException The arn was not found</li>
     *         <li>BadRequestException This request was invalid</li>
     *         <li>InternalServerErrorException Internal Service Error</li>
     *         <li>ForbiddenException Access was denied</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaLiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaLiveAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medialive-2017-10-14/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaLive");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Purchase an offering and create a reservation.
     *
     * @param purchaseOfferingRequest
     *        Placeholder documentation for PurchaseOfferingRequest
     * @return A Java Future containing the result of the PurchaseOffering operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException This request was invalid</li>
     *         <li>InternalServerErrorException Internal service error</li>
     *         <li>ForbiddenException You do not have permission to purchase the offering</li>
     *         <li>BadGatewayException Bad gateway error</li>
     *         <li>NotFoundException Offering you're attempting to purchase does not exist</li>
     *         <li>GatewayTimeoutException Gateway Timeout Error</li>
     *         <li>TooManyRequestsException Request limit exceeded on purchase offering request</li>
     *         <li>ConflictException Offering purchase prevented by service resource issue</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaLiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaLiveAsyncClient.PurchaseOffering
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medialive-2017-10-14/PurchaseOffering" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<PurchaseOfferingResponse> purchaseOffering(PurchaseOfferingRequest purchaseOfferingRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, purchaseOfferingRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaLive");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PurchaseOffering");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PurchaseOfferingResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PurchaseOfferingResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PurchaseOfferingResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PurchaseOfferingRequest, PurchaseOfferingResponse>()
                            .withOperationName("PurchaseOffering")
                            .withMarshaller(new PurchaseOfferingRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(purchaseOfferingRequest));
            CompletableFuture<PurchaseOfferingResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Send a reboot command to the specified input device. The device will begin rebooting within a few seconds of
     * sending the command. When the reboot is complete, the device’s connection status will change to connected.
     *
     * @param rebootInputDeviceRequest
     *        A request to reboot an AWS Elemental device.
     * @return A Java Future containing the result of the RebootInputDevice operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException This request was invalid.</li>
     *         <li>UnprocessableEntityException Reboot operation failed, input could not be validated.</li>
     *         <li>InternalServerErrorException Unexpected internal service error.</li>
     *         <li>ForbiddenException You do not have permission to reboot input device.</li>
     *         <li>BadGatewayException Bad gateway error.</li>
     *         <li>NotFoundException Input device not found.</li>
     *         <li>GatewayTimeoutException Gateway Timeout Error</li>
     *         <li>TooManyRequestsException Request limit exceeded on reboot device calls to the input device service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaLiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaLiveAsyncClient.RebootInputDevice
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medialive-2017-10-14/RebootInputDevice" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<RebootInputDeviceResponse> rebootInputDevice(RebootInputDeviceRequest rebootInputDeviceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, rebootInputDeviceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaLive");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RebootInputDevice");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RebootInputDeviceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RebootInputDeviceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RebootInputDeviceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RebootInputDeviceRequest, RebootInputDeviceResponse>()
                            .withOperationName("RebootInputDevice")
                            .withMarshaller(new RebootInputDeviceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(rebootInputDeviceRequest));
            CompletableFuture<RebootInputDeviceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Reject the transfer of the specified input device to your AWS account.
     *
     * @param rejectInputDeviceTransferRequest
     *        Placeholder documentation for RejectInputDeviceTransferRequest
     * @return A Java Future containing the result of the RejectInputDeviceTransfer operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException This request was invalid.</li>
     *         <li>UnprocessableEntityException Transfer operation failed, input could not be validated.</li>
     *         <li>InternalServerErrorException Unexpected internal service error.</li>
     *         <li>ForbiddenException You do not have permission to reject input device transfers.</li>
     *         <li>BadGatewayException Bad gateway error.</li>
     *         <li>NotFoundException Input device not found.</li>
     *         <li>GatewayTimeoutException Gateway Timeout Error</li>
     *         <li>TooManyRequestsException Request limit exceeded on transfer device calls to the input device service.
     *         </li>
     *         <li>ConflictException Input device transfer could not be rejected.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaLiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaLiveAsyncClient.RejectInputDeviceTransfer
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medialive-2017-10-14/RejectInputDeviceTransfer"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<RejectInputDeviceTransferResponse> rejectInputDeviceTransfer(
            RejectInputDeviceTransferRequest rejectInputDeviceTransferRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, rejectInputDeviceTransferRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaLive");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RejectInputDeviceTransfer");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RejectInputDeviceTransferResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RejectInputDeviceTransferResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RejectInputDeviceTransferResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RejectInputDeviceTransferRequest, RejectInputDeviceTransferResponse>()
                            .withOperationName("RejectInputDeviceTransfer")
                            .withMarshaller(new RejectInputDeviceTransferRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(rejectInputDeviceTransferRequest));
            CompletableFuture<RejectInputDeviceTransferResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Starts an existing channel
     *
     * @param startChannelRequest
     *        Placeholder documentation for StartChannelRequest
     * @return A Java Future containing the result of the StartChannel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException This request was invalid.</li>
     *         <li>InternalServerErrorException Unexpected internal service error.</li>
     *         <li>ForbiddenException You do not have permission to start the channel.</li>
     *         <li>BadGatewayException Bad Gateway Error</li>
     *         <li>NotFoundException The channel you're requesting to start does not exist.</li>
     *         <li>GatewayTimeoutException Gateway Timeout Error</li>
     *         <li>TooManyRequestsException Request limit exceeded on start channel calls to channel service.</li>
     *         <li>ConflictException The channel is unable to start due to an issue with channel resources.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaLiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaLiveAsyncClient.StartChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medialive-2017-10-14/StartChannel" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StartChannelResponse> startChannel(StartChannelRequest startChannelRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaLive");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartChannel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartChannelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    StartChannelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartChannelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartChannelRequest, StartChannelResponse>()
                            .withOperationName("StartChannel").withMarshaller(new StartChannelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(startChannelRequest));
            CompletableFuture<StartChannelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Start a maintenance window for the specified input device. Starting a maintenance window will give the device up
     * to two hours to install software. If the device was streaming prior to the maintenance, it will resume streaming
     * when the software is fully installed. Devices automatically install updates while they are powered on and their
     * MediaLive channels are stopped. A maintenance window allows you to update a device without having to stop
     * MediaLive channels that use the device. The device must remain powered on and connected to the internet for the
     * duration of the maintenance.
     *
     * @param startInputDeviceMaintenanceWindowRequest
     *        Placeholder documentation for StartInputDeviceMaintenanceWindowRequest
     * @return A Java Future containing the result of the StartInputDeviceMaintenanceWindow operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException This request was invalid.</li>
     *         <li>UnprocessableEntityException Start maintenance window operation failed, input could not be validated.
     *         </li>
     *         <li>InternalServerErrorException Unexpected internal service error.</li>
     *         <li>ForbiddenException You do not have permission to start a maintenance window for this input device.</li>
     *         <li>BadGatewayException Bad gateway error.</li>
     *         <li>NotFoundException Input device not found.</li>
     *         <li>GatewayTimeoutException Gateway Timeout Error</li>
     *         <li>TooManyRequestsException Request limit exceeded on start maintenance window calls to the input device
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaLiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaLiveAsyncClient.StartInputDeviceMaintenanceWindow
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medialive-2017-10-14/StartInputDeviceMaintenanceWindow"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartInputDeviceMaintenanceWindowResponse> startInputDeviceMaintenanceWindow(
            StartInputDeviceMaintenanceWindowRequest startInputDeviceMaintenanceWindowRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                startInputDeviceMaintenanceWindowRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaLive");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartInputDeviceMaintenanceWindow");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartInputDeviceMaintenanceWindowResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, StartInputDeviceMaintenanceWindowResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartInputDeviceMaintenanceWindowResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartInputDeviceMaintenanceWindowRequest, StartInputDeviceMaintenanceWindowResponse>()
                            .withOperationName("StartInputDeviceMaintenanceWindow")
                            .withMarshaller(new StartInputDeviceMaintenanceWindowRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(startInputDeviceMaintenanceWindowRequest));
            CompletableFuture<StartInputDeviceMaintenanceWindowResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Start (run) the multiplex. Starting the multiplex does not start the channels. You must explicitly start each
     * channel.
     *
     * @param startMultiplexRequest
     *        Placeholder documentation for StartMultiplexRequest
     * @return A Java Future containing the result of the StartMultiplex operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException MediaLive can't process your request because of a problem in the request. Please
     *         check your request form and syntax.</li>
     *         <li>InternalServerErrorException Unexpected internal service error.</li>
     *         <li>ForbiddenException You don't have permission to start the multiplex.</li>
     *         <li>BadGatewayException Bad Gateway Error</li>
     *         <li>NotFoundException The multiplex that you are trying to start doesn’t exist. Check the ID and try
     *         again.</li>
     *         <li>GatewayTimeoutException Gateway Timeout Error</li>
     *         <li>TooManyRequestsException Request limit exceeded on start multiplex calls to multiplex service.</li>
     *         <li>ConflictException The multiplex is unable to start due to an issue with multiplex resources.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaLiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaLiveAsyncClient.StartMultiplex
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medialive-2017-10-14/StartMultiplex" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StartMultiplexResponse> startMultiplex(StartMultiplexRequest startMultiplexRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startMultiplexRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaLive");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartMultiplex");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartMultiplexResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartMultiplexResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartMultiplexResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartMultiplexRequest, StartMultiplexResponse>()
                            .withOperationName("StartMultiplex")
                            .withMarshaller(new StartMultiplexRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(startMultiplexRequest));
            CompletableFuture<StartMultiplexResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Stops a running channel
     *
     * @param stopChannelRequest
     *        Placeholder documentation for StopChannelRequest
     * @return A Java Future containing the result of the StopChannel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException This request was invalid.</li>
     *         <li>InternalServerErrorException Unexpected internal service error.</li>
     *         <li>ForbiddenException You do not have permission to stop the channel.</li>
     *         <li>BadGatewayException Bad Gateway Error</li>
     *         <li>NotFoundException The channel you're requesting to stop does not exist.</li>
     *         <li>GatewayTimeoutException Gateway Timeout Error</li>
     *         <li>TooManyRequestsException Request limit exceeded on stop channel calls to channel service.</li>
     *         <li>ConflictException The channel is unable to stop due to an issue with channel resources.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaLiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaLiveAsyncClient.StopChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medialive-2017-10-14/StopChannel" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StopChannelResponse> stopChannel(StopChannelRequest stopChannelRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaLive");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopChannel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StopChannelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    StopChannelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StopChannelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StopChannelRequest, StopChannelResponse>()
                            .withOperationName("StopChannel").withMarshaller(new StopChannelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(stopChannelRequest));
            CompletableFuture<StopChannelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Stops a running multiplex. If the multiplex isn't running, this action has no effect.
     *
     * @param stopMultiplexRequest
     *        Placeholder documentation for StopMultiplexRequest
     * @return A Java Future containing the result of the StopMultiplex operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException MediaLive can't process your request because of a problem in the request. Please
     *         check your request form and syntax.</li>
     *         <li>InternalServerErrorException Unexpected internal service error.</li>
     *         <li>ForbiddenException You don't have permission to stop the multiplex.</li>
     *         <li>BadGatewayException Bad Gateway Error</li>
     *         <li>NotFoundException The multiplex that you are trying to stop doesn’t exist. Check the ID and try
     *         again.</li>
     *         <li>GatewayTimeoutException Gateway Timeout Error</li>
     *         <li>TooManyRequestsException Request limit exceeded on stop multiplex calls to multiplex service.</li>
     *         <li>ConflictException The multiplex is unable to stop due to an issue with multiplex resources.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaLiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaLiveAsyncClient.StopMultiplex
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medialive-2017-10-14/StopMultiplex" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StopMultiplexResponse> stopMultiplex(StopMultiplexRequest stopMultiplexRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopMultiplexRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaLive");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopMultiplex");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StopMultiplexResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    StopMultiplexResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StopMultiplexResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StopMultiplexRequest, StopMultiplexResponse>()
                            .withOperationName("StopMultiplex")
                            .withMarshaller(new StopMultiplexRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(stopMultiplexRequest));
            CompletableFuture<StopMultiplexResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Start an input device transfer to another AWS account. After you make the request, the other account must accept
     * or reject the transfer.
     *
     * @param transferInputDeviceRequest
     *        A request to transfer an input device.
     * @return A Java Future containing the result of the TransferInputDevice operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException This request was invalid.</li>
     *         <li>UnprocessableEntityException Transfer operation failed, input could not be validated.</li>
     *         <li>InternalServerErrorException Unexpected internal service error.</li>
     *         <li>ForbiddenException You do not have permission to transfer input devices.</li>
     *         <li>BadGatewayException Bad gateway error.</li>
     *         <li>NotFoundException Input device not found.</li>
     *         <li>GatewayTimeoutException Gateway Timeout Error</li>
     *         <li>TooManyRequestsException Request limit exceeded on transfer device calls to the input device service.
     *         </li>
     *         <li>ConflictException Input device could not be transferred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaLiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaLiveAsyncClient.TransferInputDevice
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medialive-2017-10-14/TransferInputDevice" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<TransferInputDeviceResponse> transferInputDevice(
            TransferInputDeviceRequest transferInputDeviceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, transferInputDeviceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaLive");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TransferInputDevice");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TransferInputDeviceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, TransferInputDeviceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TransferInputDeviceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TransferInputDeviceRequest, TransferInputDeviceResponse>()
                            .withOperationName("TransferInputDevice")
                            .withMarshaller(new TransferInputDeviceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(transferInputDeviceRequest));
            CompletableFuture<TransferInputDeviceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Updates a channel.
     *
     * @param updateChannelRequest
     *        A request to update a channel.
     * @return A Java Future containing the result of the UpdateChannel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException This request was invalid.</li>
     *         <li>UnprocessableEntityException The channel configuration failed validation and could not be updated.</li>
     *         <li>InternalServerErrorException Unexpected internal service error.</li>
     *         <li>ForbiddenException You do not have permission to update the channel.</li>
     *         <li>BadGatewayException Bad Gateway Error</li>
     *         <li>GatewayTimeoutException Gateway Timeout Error</li>
     *         <li>ConflictException The channel is unable to update due to an issue with channel resources.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaLiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaLiveAsyncClient.UpdateChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medialive-2017-10-14/UpdateChannel" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateChannelResponse> updateChannel(UpdateChannelRequest updateChannelRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaLive");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateChannel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateChannelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateChannelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateChannelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateChannelRequest, UpdateChannelResponse>()
                            .withOperationName("UpdateChannel")
                            .withMarshaller(new UpdateChannelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateChannelRequest));
            CompletableFuture<UpdateChannelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Changes the class of the channel.
     *
     * @param updateChannelClassRequest
     *        Channel class that the channel should be updated to.
     * @return A Java Future containing the result of the UpdateChannelClass operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException This request to update the channel class was invalid.</li>
     *         <li>UnprocessableEntityException The channel configuration failed validation when attempting to update
     *         the channel class.</li>
     *         <li>InternalServerErrorException Unexpected internal service error.</li>
     *         <li>ForbiddenException You do not have permission to update the class of this channel.</li>
     *         <li>BadGatewayException Bad Gateway Error</li>
     *         <li>NotFoundException The channel you're trying to update the class on does not exist.</li>
     *         <li>GatewayTimeoutException Gateway Timeout Error</li>
     *         <li>TooManyRequestsException Request limit exceeded on update channel class calls.</li>
     *         <li>ConflictException The channel class cannot be updated due to an issue with channel resources.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaLiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaLiveAsyncClient.UpdateChannelClass
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medialive-2017-10-14/UpdateChannelClass" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateChannelClassResponse> updateChannelClass(UpdateChannelClassRequest updateChannelClassRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateChannelClassRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaLive");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateChannelClass");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateChannelClassResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateChannelClassResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateChannelClassResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateChannelClassRequest, UpdateChannelClassResponse>()
                            .withOperationName("UpdateChannelClass")
                            .withMarshaller(new UpdateChannelClassRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateChannelClassRequest));
            CompletableFuture<UpdateChannelClassResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Updates an input.
     *
     * @param updateInputRequest
     *        A request to update an input.
     * @return A Java Future containing the result of the UpdateInput operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException This request to update the input was invalid.</li>
     *         <li>InternalServerErrorException Internal Service Error</li>
     *         <li>ForbiddenException The requester does not have permission to update an input.</li>
     *         <li>BadGatewayException Bad Gateway Error</li>
     *         <li>NotFoundException The input was not found.</li>
     *         <li>GatewayTimeoutException Gateway Timeout Error</li>
     *         <li>ConflictException The input was unable to be updated at this time due to an issue with input
     *         resources.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaLiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaLiveAsyncClient.UpdateInput
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medialive-2017-10-14/UpdateInput" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateInputResponse> updateInput(UpdateInputRequest updateInputRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateInputRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaLive");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateInput");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateInputResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateInputResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateInputResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateInputRequest, UpdateInputResponse>()
                            .withOperationName("UpdateInput").withMarshaller(new UpdateInputRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateInputRequest));
            CompletableFuture<UpdateInputResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Updates the parameters for the input device.
     *
     * @param updateInputDeviceRequest
     *        A request to update an input device.
     * @return A Java Future containing the result of the UpdateInputDevice operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException This request was invalid.</li>
     *         <li>UnprocessableEntityException Input device failed validation and could not be created.</li>
     *         <li>InternalServerErrorException Unexpected internal service error.</li>
     *         <li>ForbiddenException You do not have permission to update the input device.</li>
     *         <li>BadGatewayException Bad gateway error.</li>
     *         <li>NotFoundException The input device you're requesting to does not exist. Check the ID.</li>
     *         <li>GatewayTimeoutException Gateway Timeout Error</li>
     *         <li>TooManyRequestsException Request limit exceeded on update calls to the input device service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaLiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaLiveAsyncClient.UpdateInputDevice
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medialive-2017-10-14/UpdateInputDevice" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateInputDeviceResponse> updateInputDevice(UpdateInputDeviceRequest updateInputDeviceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateInputDeviceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaLive");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateInputDevice");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateInputDeviceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateInputDeviceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateInputDeviceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateInputDeviceRequest, UpdateInputDeviceResponse>()
                            .withOperationName("UpdateInputDevice")
                            .withMarshaller(new UpdateInputDeviceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateInputDeviceRequest));
            CompletableFuture<UpdateInputDeviceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Update an Input Security Group's Whilelists.
     *
     * @param updateInputSecurityGroupRequest
     *        The request to update some combination of the Input Security Group name and the IPv4 CIDRs the Input
     *        Security Group should allow.
     * @return A Java Future containing the result of the UpdateInputSecurityGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request to update the Input Security Group was invalid</li>
     *         <li>InternalServerErrorException Internal Server Error</li>
     *         <li>ForbiddenException The requester does not have permission to update an Input Security Group</li>
     *         <li>BadGatewayException Bad Gateway Error</li>
     *         <li>NotFoundException The Input Security Group was not found.</li>
     *         <li>GatewayTimeoutException Gateway Timeout Error</li>
     *         <li>ConflictException The Input Security Group was unable to be updated due to an issue with input
     *         security group resources.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaLiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaLiveAsyncClient.UpdateInputSecurityGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medialive-2017-10-14/UpdateInputSecurityGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateInputSecurityGroupResponse> updateInputSecurityGroup(
            UpdateInputSecurityGroupRequest updateInputSecurityGroupRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateInputSecurityGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaLive");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateInputSecurityGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateInputSecurityGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateInputSecurityGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateInputSecurityGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateInputSecurityGroupRequest, UpdateInputSecurityGroupResponse>()
                            .withOperationName("UpdateInputSecurityGroup")
                            .withMarshaller(new UpdateInputSecurityGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateInputSecurityGroupRequest));
            CompletableFuture<UpdateInputSecurityGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Updates a multiplex.
     *
     * @param updateMultiplexRequest
     *        A request to update a multiplex.
     * @return A Java Future containing the result of the UpdateMultiplex operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException MediaLive can't process your request because of a problem in the request. Please
     *         check your request form and syntax.</li>
     *         <li>UnprocessableEntityException The multiplex configuration failed validation and could not be updated.</li>
     *         <li>InternalServerErrorException Unexpected internal service error.</li>
     *         <li>ForbiddenException You don't have permission to update the multiplex.</li>
     *         <li>BadGatewayException Bad Gateway Error</li>
     *         <li>NotFoundException The multiplex that you are trying to update doesn’t exist. Check the ID and try
     *         again.</li>
     *         <li>GatewayTimeoutException Gateway Timeout Error</li>
     *         <li>ConflictException The multiplex is unable to update due to an issue with multiplex resources.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaLiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaLiveAsyncClient.UpdateMultiplex
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medialive-2017-10-14/UpdateMultiplex" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateMultiplexResponse> updateMultiplex(UpdateMultiplexRequest updateMultiplexRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateMultiplexRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaLive");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateMultiplex");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateMultiplexResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateMultiplexResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateMultiplexResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateMultiplexRequest, UpdateMultiplexResponse>()
                            .withOperationName("UpdateMultiplex")
                            .withMarshaller(new UpdateMultiplexRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateMultiplexRequest));
            CompletableFuture<UpdateMultiplexResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Update a program in a multiplex.
     *
     * @param updateMultiplexProgramRequest
     *        A request to update a program in a multiplex.
     * @return A Java Future containing the result of the UpdateMultiplexProgram operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException MediaLive can't process your request because of a problem in the request. Please
     *         check your request form and syntax.</li>
     *         <li>UnprocessableEntityException The multiplex program failed validation and could not be updated.</li>
     *         <li>InternalServerErrorException Unexpected internal service error.</li>
     *         <li>ForbiddenException You don't have permission to update the multiplex program.</li>
     *         <li>BadGatewayException Bad Gateway Error</li>
     *         <li>NotFoundException MediaLive can't update the program. The multiplex or the program that you specified
     *         doesn’t exist. Check the IDs and try again.</li>
     *         <li>GatewayTimeoutException Gateway Timeout Error</li>
     *         <li>ConflictException The multiplex program is unable to update due to an issue with multiplex resources.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaLiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaLiveAsyncClient.UpdateMultiplexProgram
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medialive-2017-10-14/UpdateMultiplexProgram"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateMultiplexProgramResponse> updateMultiplexProgram(
            UpdateMultiplexProgramRequest updateMultiplexProgramRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateMultiplexProgramRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaLive");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateMultiplexProgram");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateMultiplexProgramResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateMultiplexProgramResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateMultiplexProgramResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateMultiplexProgramRequest, UpdateMultiplexProgramResponse>()
                            .withOperationName("UpdateMultiplexProgram")
                            .withMarshaller(new UpdateMultiplexProgramRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateMultiplexProgramRequest));
            CompletableFuture<UpdateMultiplexProgramResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Update reservation.
     *
     * @param updateReservationRequest
     *        Request to update a reservation
     * @return A Java Future containing the result of the UpdateReservation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException This request was invalid</li>
     *         <li>InternalServerErrorException Internal service error</li>
     *         <li>ForbiddenException You do not have permission to update reservation</li>
     *         <li>BadGatewayException Bad gateway error</li>
     *         <li>NotFoundException Reservation not found</li>
     *         <li>GatewayTimeoutException Gateway Timeout Error</li>
     *         <li>TooManyRequestsException Request limit exceeded</li>
     *         <li>ConflictException The reservation could not be updated</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaLiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaLiveAsyncClient.UpdateReservation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medialive-2017-10-14/UpdateReservation" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateReservationResponse> updateReservation(UpdateReservationRequest updateReservationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateReservationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaLive");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateReservation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateReservationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateReservationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateReservationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateReservationRequest, UpdateReservationResponse>()
                            .withOperationName("UpdateReservation")
                            .withMarshaller(new UpdateReservationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateReservationRequest));
            CompletableFuture<UpdateReservationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(MediaLiveException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ForbiddenException")
                                .exceptionBuilderSupplier(ForbiddenException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotFoundException")
                                .exceptionBuilderSupplier(NotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnprocessableEntityException")
                                .exceptionBuilderSupplier(UnprocessableEntityException::builder).httpStatusCode(422).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyRequestsException")
                                .exceptionBuilderSupplier(TooManyRequestsException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequestException")
                                .exceptionBuilderSupplier(BadRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadGatewayException")
                                .exceptionBuilderSupplier(BadGatewayException::builder).httpStatusCode(502).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("GatewayTimeoutException")
                                .exceptionBuilderSupplier(GatewayTimeoutException::builder).httpStatusCode(504).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerErrorException")
                                .exceptionBuilderSupplier(InternalServerErrorException::builder).httpStatusCode(500).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends MediaLiveRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public MediaLiveAsyncWaiter waiter() {
        return MediaLiveAsyncWaiter.builder().client(this).scheduledExecutorService(executorService).build();
    }
}
