/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mediapackagevod;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.mediapackagevod.internal.MediaPackageVodServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.mediapackagevod.model.ConfigureLogsRequest;
import software.amazon.awssdk.services.mediapackagevod.model.ConfigureLogsResponse;
import software.amazon.awssdk.services.mediapackagevod.model.CreateAssetRequest;
import software.amazon.awssdk.services.mediapackagevod.model.CreateAssetResponse;
import software.amazon.awssdk.services.mediapackagevod.model.CreatePackagingConfigurationRequest;
import software.amazon.awssdk.services.mediapackagevod.model.CreatePackagingConfigurationResponse;
import software.amazon.awssdk.services.mediapackagevod.model.CreatePackagingGroupRequest;
import software.amazon.awssdk.services.mediapackagevod.model.CreatePackagingGroupResponse;
import software.amazon.awssdk.services.mediapackagevod.model.DeleteAssetRequest;
import software.amazon.awssdk.services.mediapackagevod.model.DeleteAssetResponse;
import software.amazon.awssdk.services.mediapackagevod.model.DeletePackagingConfigurationRequest;
import software.amazon.awssdk.services.mediapackagevod.model.DeletePackagingConfigurationResponse;
import software.amazon.awssdk.services.mediapackagevod.model.DeletePackagingGroupRequest;
import software.amazon.awssdk.services.mediapackagevod.model.DeletePackagingGroupResponse;
import software.amazon.awssdk.services.mediapackagevod.model.DescribeAssetRequest;
import software.amazon.awssdk.services.mediapackagevod.model.DescribeAssetResponse;
import software.amazon.awssdk.services.mediapackagevod.model.DescribePackagingConfigurationRequest;
import software.amazon.awssdk.services.mediapackagevod.model.DescribePackagingConfigurationResponse;
import software.amazon.awssdk.services.mediapackagevod.model.DescribePackagingGroupRequest;
import software.amazon.awssdk.services.mediapackagevod.model.DescribePackagingGroupResponse;
import software.amazon.awssdk.services.mediapackagevod.model.ForbiddenException;
import software.amazon.awssdk.services.mediapackagevod.model.InternalServerErrorException;
import software.amazon.awssdk.services.mediapackagevod.model.ListAssetsRequest;
import software.amazon.awssdk.services.mediapackagevod.model.ListAssetsResponse;
import software.amazon.awssdk.services.mediapackagevod.model.ListPackagingConfigurationsRequest;
import software.amazon.awssdk.services.mediapackagevod.model.ListPackagingConfigurationsResponse;
import software.amazon.awssdk.services.mediapackagevod.model.ListPackagingGroupsRequest;
import software.amazon.awssdk.services.mediapackagevod.model.ListPackagingGroupsResponse;
import software.amazon.awssdk.services.mediapackagevod.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.mediapackagevod.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.mediapackagevod.model.MediaPackageVodException;
import software.amazon.awssdk.services.mediapackagevod.model.NotFoundException;
import software.amazon.awssdk.services.mediapackagevod.model.ServiceUnavailableException;
import software.amazon.awssdk.services.mediapackagevod.model.TagResourceRequest;
import software.amazon.awssdk.services.mediapackagevod.model.TagResourceResponse;
import software.amazon.awssdk.services.mediapackagevod.model.TooManyRequestsException;
import software.amazon.awssdk.services.mediapackagevod.model.UnprocessableEntityException;
import software.amazon.awssdk.services.mediapackagevod.model.UntagResourceRequest;
import software.amazon.awssdk.services.mediapackagevod.model.UntagResourceResponse;
import software.amazon.awssdk.services.mediapackagevod.model.UpdatePackagingGroupRequest;
import software.amazon.awssdk.services.mediapackagevod.model.UpdatePackagingGroupResponse;
import software.amazon.awssdk.services.mediapackagevod.transform.ConfigureLogsRequestMarshaller;
import software.amazon.awssdk.services.mediapackagevod.transform.CreateAssetRequestMarshaller;
import software.amazon.awssdk.services.mediapackagevod.transform.CreatePackagingConfigurationRequestMarshaller;
import software.amazon.awssdk.services.mediapackagevod.transform.CreatePackagingGroupRequestMarshaller;
import software.amazon.awssdk.services.mediapackagevod.transform.DeleteAssetRequestMarshaller;
import software.amazon.awssdk.services.mediapackagevod.transform.DeletePackagingConfigurationRequestMarshaller;
import software.amazon.awssdk.services.mediapackagevod.transform.DeletePackagingGroupRequestMarshaller;
import software.amazon.awssdk.services.mediapackagevod.transform.DescribeAssetRequestMarshaller;
import software.amazon.awssdk.services.mediapackagevod.transform.DescribePackagingConfigurationRequestMarshaller;
import software.amazon.awssdk.services.mediapackagevod.transform.DescribePackagingGroupRequestMarshaller;
import software.amazon.awssdk.services.mediapackagevod.transform.ListAssetsRequestMarshaller;
import software.amazon.awssdk.services.mediapackagevod.transform.ListPackagingConfigurationsRequestMarshaller;
import software.amazon.awssdk.services.mediapackagevod.transform.ListPackagingGroupsRequestMarshaller;
import software.amazon.awssdk.services.mediapackagevod.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.mediapackagevod.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.mediapackagevod.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.mediapackagevod.transform.UpdatePackagingGroupRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link MediaPackageVodAsyncClient}.
 *
 * @see MediaPackageVodAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultMediaPackageVodAsyncClient implements MediaPackageVodAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultMediaPackageVodAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultMediaPackageVodAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * Changes the packaging group's properities to configure log subscription
     *
     * @param configureLogsRequest
     *        The option to configure log subscription.
     * @return A Java Future containing the result of the ConfigureLogs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnprocessableEntityException The parameters sent in the request are not valid.</li>
     *         <li>InternalServerErrorException An unexpected error occurred.</li>
     *         <li>ForbiddenException The client is not authorized to access the requested resource.</li>
     *         <li>NotFoundException The requested resource does not exist.</li>
     *         <li>ServiceUnavailableException An unexpected error occurred.</li>
     *         <li>TooManyRequestsException The client has exceeded their resource or throttling limits.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageVodException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MediaPackageVodAsyncClient.ConfigureLogs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackage-vod-2018-11-07/ConfigureLogs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ConfigureLogsResponse> configureLogs(ConfigureLogsRequest configureLogsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(configureLogsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, configureLogsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackage Vod");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ConfigureLogs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ConfigureLogsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ConfigureLogsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ConfigureLogsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ConfigureLogsRequest, ConfigureLogsResponse>()
                            .withOperationName("ConfigureLogs").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ConfigureLogsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(configureLogsRequest));
            CompletableFuture<ConfigureLogsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Creates a new MediaPackage VOD Asset resource.
     *
     * @param createAssetRequest
     *        A new MediaPackage VOD Asset configuration.
     * @return A Java Future containing the result of the CreateAsset operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnprocessableEntityException The parameters sent in the request are not valid.</li>
     *         <li>InternalServerErrorException An unexpected error occurred.</li>
     *         <li>ForbiddenException The client is not authorized to access the requested resource.</li>
     *         <li>NotFoundException The requested resource does not exist.</li>
     *         <li>ServiceUnavailableException An unexpected error occurred.</li>
     *         <li>TooManyRequestsException The client has exceeded their resource or throttling limits.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageVodException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MediaPackageVodAsyncClient.CreateAsset
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackage-vod-2018-11-07/CreateAsset" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateAssetResponse> createAsset(CreateAssetRequest createAssetRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createAssetRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createAssetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackage Vod");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateAsset");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateAssetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateAssetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateAssetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateAssetRequest, CreateAssetResponse>()
                            .withOperationName("CreateAsset").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateAssetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createAssetRequest));
            CompletableFuture<CreateAssetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Creates a new MediaPackage VOD PackagingConfiguration resource.
     *
     * @param createPackagingConfigurationRequest
     *        A new MediaPackage VOD PackagingConfiguration resource configuration.
     * @return A Java Future containing the result of the CreatePackagingConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnprocessableEntityException The parameters sent in the request are not valid.</li>
     *         <li>InternalServerErrorException An unexpected error occurred.</li>
     *         <li>ForbiddenException The client is not authorized to access the requested resource.</li>
     *         <li>NotFoundException The requested resource does not exist.</li>
     *         <li>ServiceUnavailableException An unexpected error occurred.</li>
     *         <li>TooManyRequestsException The client has exceeded their resource or throttling limits.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageVodException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MediaPackageVodAsyncClient.CreatePackagingConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackage-vod-2018-11-07/CreatePackagingConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreatePackagingConfigurationResponse> createPackagingConfiguration(
            CreatePackagingConfigurationRequest createPackagingConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createPackagingConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createPackagingConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackage Vod");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreatePackagingConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreatePackagingConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreatePackagingConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreatePackagingConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreatePackagingConfigurationRequest, CreatePackagingConfigurationResponse>()
                            .withOperationName("CreatePackagingConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreatePackagingConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createPackagingConfigurationRequest));
            CompletableFuture<CreatePackagingConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Creates a new MediaPackage VOD PackagingGroup resource.
     *
     * @param createPackagingGroupRequest
     *        A new MediaPackage VOD PackagingGroup resource configuration.
     * @return A Java Future containing the result of the CreatePackagingGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnprocessableEntityException The parameters sent in the request are not valid.</li>
     *         <li>InternalServerErrorException An unexpected error occurred.</li>
     *         <li>ForbiddenException The client is not authorized to access the requested resource.</li>
     *         <li>NotFoundException The requested resource does not exist.</li>
     *         <li>ServiceUnavailableException An unexpected error occurred.</li>
     *         <li>TooManyRequestsException The client has exceeded their resource or throttling limits.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageVodException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MediaPackageVodAsyncClient.CreatePackagingGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackage-vod-2018-11-07/CreatePackagingGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreatePackagingGroupResponse> createPackagingGroup(
            CreatePackagingGroupRequest createPackagingGroupRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createPackagingGroupRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createPackagingGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackage Vod");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreatePackagingGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreatePackagingGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreatePackagingGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreatePackagingGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreatePackagingGroupRequest, CreatePackagingGroupResponse>()
                            .withOperationName("CreatePackagingGroup").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreatePackagingGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createPackagingGroupRequest));
            CompletableFuture<CreatePackagingGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Deletes an existing MediaPackage VOD Asset resource.
     *
     * @param deleteAssetRequest
     * @return A Java Future containing the result of the DeleteAsset operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnprocessableEntityException The parameters sent in the request are not valid.</li>
     *         <li>InternalServerErrorException An unexpected error occurred.</li>
     *         <li>ForbiddenException The client is not authorized to access the requested resource.</li>
     *         <li>NotFoundException The requested resource does not exist.</li>
     *         <li>ServiceUnavailableException An unexpected error occurred.</li>
     *         <li>TooManyRequestsException The client has exceeded their resource or throttling limits.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageVodException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MediaPackageVodAsyncClient.DeleteAsset
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackage-vod-2018-11-07/DeleteAsset" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteAssetResponse> deleteAsset(DeleteAssetRequest deleteAssetRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteAssetRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAssetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackage Vod");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAsset");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteAssetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteAssetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteAssetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteAssetRequest, DeleteAssetResponse>()
                            .withOperationName("DeleteAsset").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteAssetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteAssetRequest));
            CompletableFuture<DeleteAssetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Deletes a MediaPackage VOD PackagingConfiguration resource.
     *
     * @param deletePackagingConfigurationRequest
     * @return A Java Future containing the result of the DeletePackagingConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnprocessableEntityException The parameters sent in the request are not valid.</li>
     *         <li>InternalServerErrorException An unexpected error occurred.</li>
     *         <li>ForbiddenException The client is not authorized to access the requested resource.</li>
     *         <li>NotFoundException The requested resource does not exist.</li>
     *         <li>ServiceUnavailableException An unexpected error occurred.</li>
     *         <li>TooManyRequestsException The client has exceeded their resource or throttling limits.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageVodException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MediaPackageVodAsyncClient.DeletePackagingConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackage-vod-2018-11-07/DeletePackagingConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeletePackagingConfigurationResponse> deletePackagingConfiguration(
            DeletePackagingConfigurationRequest deletePackagingConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deletePackagingConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deletePackagingConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackage Vod");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeletePackagingConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeletePackagingConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeletePackagingConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeletePackagingConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeletePackagingConfigurationRequest, DeletePackagingConfigurationResponse>()
                            .withOperationName("DeletePackagingConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeletePackagingConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deletePackagingConfigurationRequest));
            CompletableFuture<DeletePackagingConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Deletes a MediaPackage VOD PackagingGroup resource.
     *
     * @param deletePackagingGroupRequest
     * @return A Java Future containing the result of the DeletePackagingGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnprocessableEntityException The parameters sent in the request are not valid.</li>
     *         <li>InternalServerErrorException An unexpected error occurred.</li>
     *         <li>ForbiddenException The client is not authorized to access the requested resource.</li>
     *         <li>NotFoundException The requested resource does not exist.</li>
     *         <li>ServiceUnavailableException An unexpected error occurred.</li>
     *         <li>TooManyRequestsException The client has exceeded their resource or throttling limits.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageVodException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MediaPackageVodAsyncClient.DeletePackagingGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackage-vod-2018-11-07/DeletePackagingGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeletePackagingGroupResponse> deletePackagingGroup(
            DeletePackagingGroupRequest deletePackagingGroupRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deletePackagingGroupRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deletePackagingGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackage Vod");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeletePackagingGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeletePackagingGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeletePackagingGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeletePackagingGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeletePackagingGroupRequest, DeletePackagingGroupResponse>()
                            .withOperationName("DeletePackagingGroup").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeletePackagingGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deletePackagingGroupRequest));
            CompletableFuture<DeletePackagingGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Returns a description of a MediaPackage VOD Asset resource.
     *
     * @param describeAssetRequest
     * @return A Java Future containing the result of the DescribeAsset operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnprocessableEntityException The parameters sent in the request are not valid.</li>
     *         <li>InternalServerErrorException An unexpected error occurred.</li>
     *         <li>ForbiddenException The client is not authorized to access the requested resource.</li>
     *         <li>NotFoundException The requested resource does not exist.</li>
     *         <li>ServiceUnavailableException An unexpected error occurred.</li>
     *         <li>TooManyRequestsException The client has exceeded their resource or throttling limits.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageVodException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MediaPackageVodAsyncClient.DescribeAsset
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackage-vod-2018-11-07/DescribeAsset"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeAssetResponse> describeAsset(DescribeAssetRequest describeAssetRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeAssetRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAssetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackage Vod");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAsset");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeAssetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DescribeAssetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeAssetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeAssetRequest, DescribeAssetResponse>()
                            .withOperationName("DescribeAsset").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeAssetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeAssetRequest));
            CompletableFuture<DescribeAssetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Returns a description of a MediaPackage VOD PackagingConfiguration resource.
     *
     * @param describePackagingConfigurationRequest
     * @return A Java Future containing the result of the DescribePackagingConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnprocessableEntityException The parameters sent in the request are not valid.</li>
     *         <li>InternalServerErrorException An unexpected error occurred.</li>
     *         <li>ForbiddenException The client is not authorized to access the requested resource.</li>
     *         <li>NotFoundException The requested resource does not exist.</li>
     *         <li>ServiceUnavailableException An unexpected error occurred.</li>
     *         <li>TooManyRequestsException The client has exceeded their resource or throttling limits.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageVodException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MediaPackageVodAsyncClient.DescribePackagingConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackage-vod-2018-11-07/DescribePackagingConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribePackagingConfigurationResponse> describePackagingConfiguration(
            DescribePackagingConfigurationRequest describePackagingConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describePackagingConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describePackagingConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackage Vod");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribePackagingConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribePackagingConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribePackagingConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribePackagingConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribePackagingConfigurationRequest, DescribePackagingConfigurationResponse>()
                            .withOperationName("DescribePackagingConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribePackagingConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describePackagingConfigurationRequest));
            CompletableFuture<DescribePackagingConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Returns a description of a MediaPackage VOD PackagingGroup resource.
     *
     * @param describePackagingGroupRequest
     * @return A Java Future containing the result of the DescribePackagingGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnprocessableEntityException The parameters sent in the request are not valid.</li>
     *         <li>InternalServerErrorException An unexpected error occurred.</li>
     *         <li>ForbiddenException The client is not authorized to access the requested resource.</li>
     *         <li>NotFoundException The requested resource does not exist.</li>
     *         <li>ServiceUnavailableException An unexpected error occurred.</li>
     *         <li>TooManyRequestsException The client has exceeded their resource or throttling limits.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageVodException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MediaPackageVodAsyncClient.DescribePackagingGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackage-vod-2018-11-07/DescribePackagingGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribePackagingGroupResponse> describePackagingGroup(
            DescribePackagingGroupRequest describePackagingGroupRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describePackagingGroupRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describePackagingGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackage Vod");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribePackagingGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribePackagingGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribePackagingGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribePackagingGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribePackagingGroupRequest, DescribePackagingGroupResponse>()
                            .withOperationName("DescribePackagingGroup").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribePackagingGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describePackagingGroupRequest));
            CompletableFuture<DescribePackagingGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Returns a collection of MediaPackage VOD Asset resources.
     *
     * @param listAssetsRequest
     * @return A Java Future containing the result of the ListAssets operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnprocessableEntityException The parameters sent in the request are not valid.</li>
     *         <li>InternalServerErrorException An unexpected error occurred.</li>
     *         <li>ForbiddenException The client is not authorized to access the requested resource.</li>
     *         <li>NotFoundException The requested resource does not exist.</li>
     *         <li>ServiceUnavailableException An unexpected error occurred.</li>
     *         <li>TooManyRequestsException The client has exceeded their resource or throttling limits.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageVodException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MediaPackageVodAsyncClient.ListAssets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackage-vod-2018-11-07/ListAssets" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListAssetsResponse> listAssets(ListAssetsRequest listAssetsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listAssetsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAssetsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackage Vod");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAssets");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAssetsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListAssetsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListAssetsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAssetsRequest, ListAssetsResponse>().withOperationName("ListAssets")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListAssetsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listAssetsRequest));
            CompletableFuture<ListAssetsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Returns a collection of MediaPackage VOD PackagingConfiguration resources.
     *
     * @param listPackagingConfigurationsRequest
     * @return A Java Future containing the result of the ListPackagingConfigurations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnprocessableEntityException The parameters sent in the request are not valid.</li>
     *         <li>InternalServerErrorException An unexpected error occurred.</li>
     *         <li>ForbiddenException The client is not authorized to access the requested resource.</li>
     *         <li>NotFoundException The requested resource does not exist.</li>
     *         <li>ServiceUnavailableException An unexpected error occurred.</li>
     *         <li>TooManyRequestsException The client has exceeded their resource or throttling limits.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageVodException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MediaPackageVodAsyncClient.ListPackagingConfigurations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackage-vod-2018-11-07/ListPackagingConfigurations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListPackagingConfigurationsResponse> listPackagingConfigurations(
            ListPackagingConfigurationsRequest listPackagingConfigurationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listPackagingConfigurationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPackagingConfigurationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackage Vod");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPackagingConfigurations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListPackagingConfigurationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListPackagingConfigurationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListPackagingConfigurationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPackagingConfigurationsRequest, ListPackagingConfigurationsResponse>()
                            .withOperationName("ListPackagingConfigurations").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListPackagingConfigurationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listPackagingConfigurationsRequest));
            CompletableFuture<ListPackagingConfigurationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Returns a collection of MediaPackage VOD PackagingGroup resources.
     *
     * @param listPackagingGroupsRequest
     * @return A Java Future containing the result of the ListPackagingGroups operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnprocessableEntityException The parameters sent in the request are not valid.</li>
     *         <li>InternalServerErrorException An unexpected error occurred.</li>
     *         <li>ForbiddenException The client is not authorized to access the requested resource.</li>
     *         <li>NotFoundException The requested resource does not exist.</li>
     *         <li>ServiceUnavailableException An unexpected error occurred.</li>
     *         <li>TooManyRequestsException The client has exceeded their resource or throttling limits.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageVodException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MediaPackageVodAsyncClient.ListPackagingGroups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackage-vod-2018-11-07/ListPackagingGroups"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListPackagingGroupsResponse> listPackagingGroups(
            ListPackagingGroupsRequest listPackagingGroupsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listPackagingGroupsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPackagingGroupsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackage Vod");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPackagingGroups");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListPackagingGroupsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListPackagingGroupsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListPackagingGroupsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPackagingGroupsRequest, ListPackagingGroupsResponse>()
                            .withOperationName("ListPackagingGroups").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListPackagingGroupsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listPackagingGroupsRequest));
            CompletableFuture<ListPackagingGroupsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Returns a list of the tags assigned to the specified resource.
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageVodException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MediaPackageVodAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackage-vod-2018-11-07/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackage Vod");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Adds tags to the specified resource. You can specify one or more tags to add.
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageVodException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MediaPackageVodAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackage-vod-2018-11-07/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackage Vod");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Removes tags from the specified resource. You can specify one or more tags to remove.
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageVodException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MediaPackageVodAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackage-vod-2018-11-07/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackage Vod");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Updates a specific packaging group. You can't change the id attribute or any other system-generated attributes.
     *
     * @param updatePackagingGroupRequest
     *        A MediaPackage VOD PackagingGroup resource configuration.
     * @return A Java Future containing the result of the UpdatePackagingGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnprocessableEntityException The parameters sent in the request are not valid.</li>
     *         <li>InternalServerErrorException An unexpected error occurred.</li>
     *         <li>ForbiddenException The client is not authorized to access the requested resource.</li>
     *         <li>NotFoundException The requested resource does not exist.</li>
     *         <li>ServiceUnavailableException An unexpected error occurred.</li>
     *         <li>TooManyRequestsException The client has exceeded their resource or throttling limits.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageVodException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MediaPackageVodAsyncClient.UpdatePackagingGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackage-vod-2018-11-07/UpdatePackagingGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdatePackagingGroupResponse> updatePackagingGroup(
            UpdatePackagingGroupRequest updatePackagingGroupRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updatePackagingGroupRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updatePackagingGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackage Vod");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdatePackagingGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdatePackagingGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdatePackagingGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdatePackagingGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdatePackagingGroupRequest, UpdatePackagingGroupResponse>()
                            .withOperationName("UpdatePackagingGroup").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdatePackagingGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updatePackagingGroupRequest));
            CompletableFuture<UpdatePackagingGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final MediaPackageVodServiceClientConfiguration serviceClientConfiguration() {
        return new MediaPackageVodServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(MediaPackageVodException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnprocessableEntityException")
                                .exceptionBuilderSupplier(UnprocessableEntityException::builder).httpStatusCode(422).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotFoundException")
                                .exceptionBuilderSupplier(NotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).httpStatusCode(503).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ForbiddenException")
                                .exceptionBuilderSupplier(ForbiddenException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyRequestsException")
                                .exceptionBuilderSupplier(TooManyRequestsException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerErrorException")
                                .exceptionBuilderSupplier(InternalServerErrorException::builder).httpStatusCode(500).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        MediaPackageVodServiceClientConfigurationBuilder serviceConfigBuilder = new MediaPackageVodServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
