/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.migrationhubconfig;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.migrationhubconfig.internal.MigrationHubConfigServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.migrationhubconfig.model.AccessDeniedException;
import software.amazon.awssdk.services.migrationhubconfig.model.CreateHomeRegionControlRequest;
import software.amazon.awssdk.services.migrationhubconfig.model.CreateHomeRegionControlResponse;
import software.amazon.awssdk.services.migrationhubconfig.model.DeleteHomeRegionControlRequest;
import software.amazon.awssdk.services.migrationhubconfig.model.DeleteHomeRegionControlResponse;
import software.amazon.awssdk.services.migrationhubconfig.model.DescribeHomeRegionControlsRequest;
import software.amazon.awssdk.services.migrationhubconfig.model.DescribeHomeRegionControlsResponse;
import software.amazon.awssdk.services.migrationhubconfig.model.DryRunOperationException;
import software.amazon.awssdk.services.migrationhubconfig.model.GetHomeRegionRequest;
import software.amazon.awssdk.services.migrationhubconfig.model.GetHomeRegionResponse;
import software.amazon.awssdk.services.migrationhubconfig.model.InternalServerErrorException;
import software.amazon.awssdk.services.migrationhubconfig.model.InvalidInputException;
import software.amazon.awssdk.services.migrationhubconfig.model.MigrationHubConfigException;
import software.amazon.awssdk.services.migrationhubconfig.model.ServiceUnavailableException;
import software.amazon.awssdk.services.migrationhubconfig.model.ThrottlingException;
import software.amazon.awssdk.services.migrationhubconfig.transform.CreateHomeRegionControlRequestMarshaller;
import software.amazon.awssdk.services.migrationhubconfig.transform.DeleteHomeRegionControlRequestMarshaller;
import software.amazon.awssdk.services.migrationhubconfig.transform.DescribeHomeRegionControlsRequestMarshaller;
import software.amazon.awssdk.services.migrationhubconfig.transform.GetHomeRegionRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link MigrationHubConfigClient}.
 *
 * @see MigrationHubConfigClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultMigrationHubConfigClient implements MigrationHubConfigClient {
    private static final Logger log = Logger.loggerFor(DefaultMigrationHubConfigClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.AWS_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultMigrationHubConfigClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * This API sets up the home region for the calling account only.
     * </p>
     *
     * @param createHomeRegionControlRequest
     * @return Result of the CreateHomeRegionControl operation returned by the service.
     * @throws InternalServerErrorException
     *         Exception raised when an internal, configuration, or dependency error is encountered.
     * @throws ServiceUnavailableException
     *         Exception raised when a request fails due to temporary unavailability of the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws DryRunOperationException
     *         Exception raised to indicate that authorization of an action was successful, when the <code>DryRun</code>
     *         flag is set to true.
     * @throws InvalidInputException
     *         Exception raised when the provided input violates a policy constraint or is entered in the wrong format
     *         or data type.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MigrationHubConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MigrationHubConfigClient.CreateHomeRegionControl
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/migrationhub-config-2019-06-30/CreateHomeRegionControl"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateHomeRegionControlResponse createHomeRegionControl(CreateHomeRegionControlRequest createHomeRegionControlRequest)
            throws InternalServerErrorException, ServiceUnavailableException, AccessDeniedException, ThrottlingException,
            DryRunOperationException, InvalidInputException, AwsServiceException, SdkClientException, MigrationHubConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateHomeRegionControlResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateHomeRegionControlResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createHomeRegionControlRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createHomeRegionControlRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MigrationHub Config");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateHomeRegionControl");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateHomeRegionControlRequest, CreateHomeRegionControlResponse>()
                            .withOperationName("CreateHomeRegionControl").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(createHomeRegionControlRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateHomeRegionControlRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation deletes the home region configuration for the calling account. The operation does not delete
     * discovery or migration tracking data in the home region.
     * </p>
     *
     * @param deleteHomeRegionControlRequest
     * @return Result of the DeleteHomeRegionControl operation returned by the service.
     * @throws InternalServerErrorException
     *         Exception raised when an internal, configuration, or dependency error is encountered.
     * @throws ServiceUnavailableException
     *         Exception raised when a request fails due to temporary unavailability of the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InvalidInputException
     *         Exception raised when the provided input violates a policy constraint or is entered in the wrong format
     *         or data type.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MigrationHubConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MigrationHubConfigClient.DeleteHomeRegionControl
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/migrationhub-config-2019-06-30/DeleteHomeRegionControl"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteHomeRegionControlResponse deleteHomeRegionControl(DeleteHomeRegionControlRequest deleteHomeRegionControlRequest)
            throws InternalServerErrorException, ServiceUnavailableException, AccessDeniedException, ThrottlingException,
            InvalidInputException, AwsServiceException, SdkClientException, MigrationHubConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteHomeRegionControlResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteHomeRegionControlResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteHomeRegionControlRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteHomeRegionControlRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MigrationHub Config");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteHomeRegionControl");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteHomeRegionControlRequest, DeleteHomeRegionControlResponse>()
                            .withOperationName("DeleteHomeRegionControl").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(deleteHomeRegionControlRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteHomeRegionControlRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This API permits filtering on the <code>ControlId</code> and <code>HomeRegion</code> fields.
     * </p>
     *
     * @param describeHomeRegionControlsRequest
     * @return Result of the DescribeHomeRegionControls operation returned by the service.
     * @throws InternalServerErrorException
     *         Exception raised when an internal, configuration, or dependency error is encountered.
     * @throws ServiceUnavailableException
     *         Exception raised when a request fails due to temporary unavailability of the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InvalidInputException
     *         Exception raised when the provided input violates a policy constraint or is entered in the wrong format
     *         or data type.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MigrationHubConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MigrationHubConfigClient.DescribeHomeRegionControls
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/migrationhub-config-2019-06-30/DescribeHomeRegionControls"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeHomeRegionControlsResponse describeHomeRegionControls(
            DescribeHomeRegionControlsRequest describeHomeRegionControlsRequest) throws InternalServerErrorException,
            ServiceUnavailableException, AccessDeniedException, ThrottlingException, InvalidInputException, AwsServiceException,
            SdkClientException, MigrationHubConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeHomeRegionControlsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeHomeRegionControlsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeHomeRegionControlsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeHomeRegionControlsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MigrationHub Config");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeHomeRegionControls");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeHomeRegionControlsRequest, DescribeHomeRegionControlsResponse>()
                            .withOperationName("DescribeHomeRegionControls").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(describeHomeRegionControlsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeHomeRegionControlsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the calling account’s home region, if configured. This API is used by other AWS services to determine the
     * regional endpoint for calling AWS Application Discovery Service and Migration Hub. You must call
     * <code>GetHomeRegion</code> at least once before you call any other AWS Application Discovery Service and AWS
     * Migration Hub APIs, to obtain the account's Migration Hub home region.
     * </p>
     *
     * @param getHomeRegionRequest
     * @return Result of the GetHomeRegion operation returned by the service.
     * @throws InternalServerErrorException
     *         Exception raised when an internal, configuration, or dependency error is encountered.
     * @throws ServiceUnavailableException
     *         Exception raised when a request fails due to temporary unavailability of the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InvalidInputException
     *         Exception raised when the provided input violates a policy constraint or is entered in the wrong format
     *         or data type.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MigrationHubConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MigrationHubConfigClient.GetHomeRegion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/migrationhub-config-2019-06-30/GetHomeRegion"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetHomeRegionResponse getHomeRegion(GetHomeRegionRequest getHomeRegionRequest) throws InternalServerErrorException,
            ServiceUnavailableException, AccessDeniedException, ThrottlingException, InvalidInputException, AwsServiceException,
            SdkClientException, MigrationHubConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetHomeRegionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetHomeRegionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getHomeRegionRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getHomeRegionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MigrationHub Config");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetHomeRegion");

            return clientHandler.execute(new ClientExecutionParams<GetHomeRegionRequest, GetHomeRegionResponse>()
                    .withOperationName("GetHomeRegion").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getHomeRegionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetHomeRegionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        MigrationHubConfigServiceClientConfigurationBuilder serviceConfigBuilder = new MigrationHubConfigServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(MigrationHubConfigException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidInputException")
                                .exceptionBuilderSupplier(InvalidInputException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerError")
                                .exceptionBuilderSupplier(InternalServerErrorException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DryRunOperation")
                                .exceptionBuilderSupplier(DryRunOperationException::builder).httpStatusCode(400).build());
    }

    @Override
    public final MigrationHubConfigServiceClientConfiguration serviceClientConfiguration() {
        return new MigrationHubConfigServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
