/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.migrationhubconfig.paginators;

import java.util.Iterator;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.pagination.sync.PaginatedResponsesIterator;
import software.amazon.awssdk.core.pagination.sync.SdkIterable;
import software.amazon.awssdk.core.pagination.sync.SyncPageFetcher;
import software.amazon.awssdk.core.util.PaginatorUtils;
import software.amazon.awssdk.services.migrationhubconfig.MigrationHubConfigClient;
import software.amazon.awssdk.services.migrationhubconfig.internal.UserAgentUtils;
import software.amazon.awssdk.services.migrationhubconfig.model.DescribeHomeRegionControlsRequest;
import software.amazon.awssdk.services.migrationhubconfig.model.DescribeHomeRegionControlsResponse;

/**
 * <p>
 * Represents the output for the
 * {@link software.amazon.awssdk.services.migrationhubconfig.MigrationHubConfigClient#describeHomeRegionControlsPaginator(software.amazon.awssdk.services.migrationhubconfig.model.DescribeHomeRegionControlsRequest)}
 * operation which is a paginated operation. This class is an iterable of
 * {@link software.amazon.awssdk.services.migrationhubconfig.model.DescribeHomeRegionControlsResponse} that can be used
 * to iterate through all the response pages of the operation.
 * </p>
 * <p>
 * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet and
 * so there is no guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily
 * loading response pages by making service calls until there are no pages left or your iteration stops. If there are
 * errors in your request, you will see the failures only after you start iterating through the iterable.
 * </p>
 *
 * <p>
 * The following are few ways to iterate through the response pages:
 * </p>
 * 1) Using a Stream
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.migrationhubconfig.paginators.DescribeHomeRegionControlsIterable responses = client.describeHomeRegionControlsPaginator(request);
 * responses.stream().forEach(....);
 * }
 * </pre>
 *
 * 2) Using For loop
 * 
 * <pre>
 * {
 *     &#064;code
 *     software.amazon.awssdk.services.migrationhubconfig.paginators.DescribeHomeRegionControlsIterable responses = client
 *             .describeHomeRegionControlsPaginator(request);
 *     for (software.amazon.awssdk.services.migrationhubconfig.model.DescribeHomeRegionControlsResponse response : responses) {
 *         // do something;
 *     }
 * }
 * </pre>
 *
 * 3) Use iterator directly
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.migrationhubconfig.paginators.DescribeHomeRegionControlsIterable responses = client.describeHomeRegionControlsPaginator(request);
 * responses.iterator().forEachRemaining(....);
 * }
 * </pre>
 * <p>
 * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the paginator.
 * It only limits the number of results in each page.</b>
 * </p>
 * <p>
 * <b>Note: If you prefer to have control on service calls, use the
 * {@link #describeHomeRegionControls(software.amazon.awssdk.services.migrationhubconfig.model.DescribeHomeRegionControlsRequest)}
 * operation.</b>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class DescribeHomeRegionControlsIterable implements SdkIterable<DescribeHomeRegionControlsResponse> {
    private final MigrationHubConfigClient client;

    private final DescribeHomeRegionControlsRequest firstRequest;

    private final SyncPageFetcher nextPageFetcher;

    public DescribeHomeRegionControlsIterable(MigrationHubConfigClient client, DescribeHomeRegionControlsRequest firstRequest) {
        this.client = client;
        this.firstRequest = UserAgentUtils.applyPaginatorUserAgent(firstRequest);
        this.nextPageFetcher = new DescribeHomeRegionControlsResponseFetcher();
    }

    @Override
    public Iterator<DescribeHomeRegionControlsResponse> iterator() {
        return PaginatedResponsesIterator.builder().nextPageFetcher(nextPageFetcher).build();
    }

    private class DescribeHomeRegionControlsResponseFetcher implements SyncPageFetcher<DescribeHomeRegionControlsResponse> {
        @Override
        public boolean hasNextPage(DescribeHomeRegionControlsResponse previousPage) {
            return PaginatorUtils.isOutputTokenAvailable(previousPage.nextToken());
        }

        @Override
        public DescribeHomeRegionControlsResponse nextPage(DescribeHomeRegionControlsResponse previousPage) {
            if (previousPage == null) {
                return client.describeHomeRegionControls(firstRequest);
            }
            return client.describeHomeRegionControls(firstRequest.toBuilder().nextToken(previousPage.nextToken()).build());
        }
    }
}
