/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.migrationhubstrategy;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.migrationhubstrategy.model.AccessDeniedException;
import software.amazon.awssdk.services.migrationhubstrategy.model.ConflictException;
import software.amazon.awssdk.services.migrationhubstrategy.model.GetApplicationComponentDetailsRequest;
import software.amazon.awssdk.services.migrationhubstrategy.model.GetApplicationComponentDetailsResponse;
import software.amazon.awssdk.services.migrationhubstrategy.model.GetApplicationComponentStrategiesRequest;
import software.amazon.awssdk.services.migrationhubstrategy.model.GetApplicationComponentStrategiesResponse;
import software.amazon.awssdk.services.migrationhubstrategy.model.GetAssessmentRequest;
import software.amazon.awssdk.services.migrationhubstrategy.model.GetAssessmentResponse;
import software.amazon.awssdk.services.migrationhubstrategy.model.GetImportFileTaskRequest;
import software.amazon.awssdk.services.migrationhubstrategy.model.GetImportFileTaskResponse;
import software.amazon.awssdk.services.migrationhubstrategy.model.GetPortfolioPreferencesRequest;
import software.amazon.awssdk.services.migrationhubstrategy.model.GetPortfolioPreferencesResponse;
import software.amazon.awssdk.services.migrationhubstrategy.model.GetPortfolioSummaryRequest;
import software.amazon.awssdk.services.migrationhubstrategy.model.GetPortfolioSummaryResponse;
import software.amazon.awssdk.services.migrationhubstrategy.model.GetRecommendationReportDetailsRequest;
import software.amazon.awssdk.services.migrationhubstrategy.model.GetRecommendationReportDetailsResponse;
import software.amazon.awssdk.services.migrationhubstrategy.model.GetServerDetailsRequest;
import software.amazon.awssdk.services.migrationhubstrategy.model.GetServerDetailsResponse;
import software.amazon.awssdk.services.migrationhubstrategy.model.GetServerStrategiesRequest;
import software.amazon.awssdk.services.migrationhubstrategy.model.GetServerStrategiesResponse;
import software.amazon.awssdk.services.migrationhubstrategy.model.InternalServerException;
import software.amazon.awssdk.services.migrationhubstrategy.model.ListApplicationComponentsRequest;
import software.amazon.awssdk.services.migrationhubstrategy.model.ListApplicationComponentsResponse;
import software.amazon.awssdk.services.migrationhubstrategy.model.ListCollectorsRequest;
import software.amazon.awssdk.services.migrationhubstrategy.model.ListCollectorsResponse;
import software.amazon.awssdk.services.migrationhubstrategy.model.ListImportFileTaskRequest;
import software.amazon.awssdk.services.migrationhubstrategy.model.ListImportFileTaskResponse;
import software.amazon.awssdk.services.migrationhubstrategy.model.ListServersRequest;
import software.amazon.awssdk.services.migrationhubstrategy.model.ListServersResponse;
import software.amazon.awssdk.services.migrationhubstrategy.model.MigrationHubStrategyException;
import software.amazon.awssdk.services.migrationhubstrategy.model.MigrationHubStrategyRequest;
import software.amazon.awssdk.services.migrationhubstrategy.model.PutPortfolioPreferencesRequest;
import software.amazon.awssdk.services.migrationhubstrategy.model.PutPortfolioPreferencesResponse;
import software.amazon.awssdk.services.migrationhubstrategy.model.ResourceNotFoundException;
import software.amazon.awssdk.services.migrationhubstrategy.model.ServiceLinkedRoleLockClientException;
import software.amazon.awssdk.services.migrationhubstrategy.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.migrationhubstrategy.model.StartAssessmentRequest;
import software.amazon.awssdk.services.migrationhubstrategy.model.StartAssessmentResponse;
import software.amazon.awssdk.services.migrationhubstrategy.model.StartImportFileTaskRequest;
import software.amazon.awssdk.services.migrationhubstrategy.model.StartImportFileTaskResponse;
import software.amazon.awssdk.services.migrationhubstrategy.model.StartRecommendationReportGenerationRequest;
import software.amazon.awssdk.services.migrationhubstrategy.model.StartRecommendationReportGenerationResponse;
import software.amazon.awssdk.services.migrationhubstrategy.model.StopAssessmentRequest;
import software.amazon.awssdk.services.migrationhubstrategy.model.StopAssessmentResponse;
import software.amazon.awssdk.services.migrationhubstrategy.model.ThrottlingException;
import software.amazon.awssdk.services.migrationhubstrategy.model.UpdateApplicationComponentConfigRequest;
import software.amazon.awssdk.services.migrationhubstrategy.model.UpdateApplicationComponentConfigResponse;
import software.amazon.awssdk.services.migrationhubstrategy.model.UpdateServerConfigRequest;
import software.amazon.awssdk.services.migrationhubstrategy.model.UpdateServerConfigResponse;
import software.amazon.awssdk.services.migrationhubstrategy.model.ValidationException;
import software.amazon.awssdk.services.migrationhubstrategy.paginators.GetServerDetailsIterable;
import software.amazon.awssdk.services.migrationhubstrategy.paginators.ListApplicationComponentsIterable;
import software.amazon.awssdk.services.migrationhubstrategy.paginators.ListCollectorsIterable;
import software.amazon.awssdk.services.migrationhubstrategy.paginators.ListImportFileTaskIterable;
import software.amazon.awssdk.services.migrationhubstrategy.paginators.ListServersIterable;
import software.amazon.awssdk.services.migrationhubstrategy.transform.GetApplicationComponentDetailsRequestMarshaller;
import software.amazon.awssdk.services.migrationhubstrategy.transform.GetApplicationComponentStrategiesRequestMarshaller;
import software.amazon.awssdk.services.migrationhubstrategy.transform.GetAssessmentRequestMarshaller;
import software.amazon.awssdk.services.migrationhubstrategy.transform.GetImportFileTaskRequestMarshaller;
import software.amazon.awssdk.services.migrationhubstrategy.transform.GetPortfolioPreferencesRequestMarshaller;
import software.amazon.awssdk.services.migrationhubstrategy.transform.GetPortfolioSummaryRequestMarshaller;
import software.amazon.awssdk.services.migrationhubstrategy.transform.GetRecommendationReportDetailsRequestMarshaller;
import software.amazon.awssdk.services.migrationhubstrategy.transform.GetServerDetailsRequestMarshaller;
import software.amazon.awssdk.services.migrationhubstrategy.transform.GetServerStrategiesRequestMarshaller;
import software.amazon.awssdk.services.migrationhubstrategy.transform.ListApplicationComponentsRequestMarshaller;
import software.amazon.awssdk.services.migrationhubstrategy.transform.ListCollectorsRequestMarshaller;
import software.amazon.awssdk.services.migrationhubstrategy.transform.ListImportFileTaskRequestMarshaller;
import software.amazon.awssdk.services.migrationhubstrategy.transform.ListServersRequestMarshaller;
import software.amazon.awssdk.services.migrationhubstrategy.transform.PutPortfolioPreferencesRequestMarshaller;
import software.amazon.awssdk.services.migrationhubstrategy.transform.StartAssessmentRequestMarshaller;
import software.amazon.awssdk.services.migrationhubstrategy.transform.StartImportFileTaskRequestMarshaller;
import software.amazon.awssdk.services.migrationhubstrategy.transform.StartRecommendationReportGenerationRequestMarshaller;
import software.amazon.awssdk.services.migrationhubstrategy.transform.StopAssessmentRequestMarshaller;
import software.amazon.awssdk.services.migrationhubstrategy.transform.UpdateApplicationComponentConfigRequestMarshaller;
import software.amazon.awssdk.services.migrationhubstrategy.transform.UpdateServerConfigRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link MigrationHubStrategyClient}.
 *
 * @see MigrationHubStrategyClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultMigrationHubStrategyClient implements MigrationHubStrategyClient {
    private static final Logger log = Logger.loggerFor(DefaultMigrationHubStrategyClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultMigrationHubStrategyClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Retrieves details about an application component.
     * </p>
     *
     * @param getApplicationComponentDetailsRequest
     * @return Result of the GetApplicationComponentDetails operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified ID in the request is not found.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerException
     *         The server experienced an internal error. Try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MigrationHubStrategyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MigrationHubStrategyClient.GetApplicationComponentDetails
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/migrationhubstrategy-2020-02-19/GetApplicationComponentDetails"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetApplicationComponentDetailsResponse getApplicationComponentDetails(
            GetApplicationComponentDetailsRequest getApplicationComponentDetailsRequest) throws ResourceNotFoundException,
            ThrottlingException, InternalServerException, AwsServiceException, SdkClientException, MigrationHubStrategyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetApplicationComponentDetailsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetApplicationComponentDetailsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getApplicationComponentDetailsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MigrationHubStrategy");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetApplicationComponentDetails");

            return clientHandler
                    .execute(new ClientExecutionParams<GetApplicationComponentDetailsRequest, GetApplicationComponentDetailsResponse>()
                            .withOperationName("GetApplicationComponentDetails").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(getApplicationComponentDetailsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetApplicationComponentDetailsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves a list of all the recommended strategies and tools for an application component running on a server.
     * </p>
     *
     * @param getApplicationComponentStrategiesRequest
     * @return Result of the GetApplicationComponentStrategies operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified ID in the request is not found.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerException
     *         The server experienced an internal error. Try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MigrationHubStrategyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MigrationHubStrategyClient.GetApplicationComponentStrategies
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/migrationhubstrategy-2020-02-19/GetApplicationComponentStrategies"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetApplicationComponentStrategiesResponse getApplicationComponentStrategies(
            GetApplicationComponentStrategiesRequest getApplicationComponentStrategiesRequest) throws ResourceNotFoundException,
            ThrottlingException, InternalServerException, AwsServiceException, SdkClientException, MigrationHubStrategyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetApplicationComponentStrategiesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetApplicationComponentStrategiesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getApplicationComponentStrategiesRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MigrationHubStrategy");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetApplicationComponentStrategies");

            return clientHandler
                    .execute(new ClientExecutionParams<GetApplicationComponentStrategiesRequest, GetApplicationComponentStrategiesResponse>()
                            .withOperationName("GetApplicationComponentStrategies").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(getApplicationComponentStrategiesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetApplicationComponentStrategiesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the status of an on-going assessment.
     * </p>
     *
     * @param getAssessmentRequest
     * @return Result of the GetAssessment operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified ID in the request is not found.
     * @throws AccessDeniedException
     *         The AWS user account does not have permission to perform the action. Check the AWS Identity and Access
     *         Management (IAM) policy associated with this account.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerException
     *         The server experienced an internal error. Try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MigrationHubStrategyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MigrationHubStrategyClient.GetAssessment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/migrationhubstrategy-2020-02-19/GetAssessment"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetAssessmentResponse getAssessment(GetAssessmentRequest getAssessmentRequest) throws ResourceNotFoundException,
            AccessDeniedException, ThrottlingException, InternalServerException, AwsServiceException, SdkClientException,
            MigrationHubStrategyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetAssessmentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetAssessmentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAssessmentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MigrationHubStrategy");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAssessment");

            return clientHandler.execute(new ClientExecutionParams<GetAssessmentRequest, GetAssessmentResponse>()
                    .withOperationName("GetAssessment").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getAssessmentRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetAssessmentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the details about a specific import task.
     * </p>
     *
     * @param getImportFileTaskRequest
     * @return Result of the GetImportFileTask operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified ID in the request is not found.
     * @throws AccessDeniedException
     *         The AWS user account does not have permission to perform the action. Check the AWS Identity and Access
     *         Management (IAM) policy associated with this account.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The request body isn't valid.
     * @throws InternalServerException
     *         The server experienced an internal error. Try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MigrationHubStrategyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MigrationHubStrategyClient.GetImportFileTask
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/migrationhubstrategy-2020-02-19/GetImportFileTask"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetImportFileTaskResponse getImportFileTask(GetImportFileTaskRequest getImportFileTaskRequest)
            throws ResourceNotFoundException, AccessDeniedException, ThrottlingException, ValidationException,
            InternalServerException, AwsServiceException, SdkClientException, MigrationHubStrategyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetImportFileTaskResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetImportFileTaskResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getImportFileTaskRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MigrationHubStrategy");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetImportFileTask");

            return clientHandler.execute(new ClientExecutionParams<GetImportFileTaskRequest, GetImportFileTaskResponse>()
                    .withOperationName("GetImportFileTask").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getImportFileTaskRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetImportFileTaskRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves your migration and modernization preferences.
     * </p>
     *
     * @param getPortfolioPreferencesRequest
     * @return Result of the GetPortfolioPreferences operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified ID in the request is not found.
     * @throws AccessDeniedException
     *         The AWS user account does not have permission to perform the action. Check the AWS Identity and Access
     *         Management (IAM) policy associated with this account.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerException
     *         The server experienced an internal error. Try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MigrationHubStrategyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MigrationHubStrategyClient.GetPortfolioPreferences
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/migrationhubstrategy-2020-02-19/GetPortfolioPreferences"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetPortfolioPreferencesResponse getPortfolioPreferences(GetPortfolioPreferencesRequest getPortfolioPreferencesRequest)
            throws ResourceNotFoundException, AccessDeniedException, ThrottlingException, InternalServerException,
            AwsServiceException, SdkClientException, MigrationHubStrategyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetPortfolioPreferencesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetPortfolioPreferencesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPortfolioPreferencesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MigrationHubStrategy");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPortfolioPreferences");

            return clientHandler
                    .execute(new ClientExecutionParams<GetPortfolioPreferencesRequest, GetPortfolioPreferencesResponse>()
                            .withOperationName("GetPortfolioPreferences").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(getPortfolioPreferencesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetPortfolioPreferencesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves overall summary including the number of servers to rehost and the overall number of anti-patterns.
     * </p>
     *
     * @param getPortfolioSummaryRequest
     * @return Result of the GetPortfolioSummary operation returned by the service.
     * @throws AccessDeniedException
     *         The AWS user account does not have permission to perform the action. Check the AWS Identity and Access
     *         Management (IAM) policy associated with this account.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerException
     *         The server experienced an internal error. Try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MigrationHubStrategyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MigrationHubStrategyClient.GetPortfolioSummary
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/migrationhubstrategy-2020-02-19/GetPortfolioSummary"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetPortfolioSummaryResponse getPortfolioSummary(GetPortfolioSummaryRequest getPortfolioSummaryRequest)
            throws AccessDeniedException, ThrottlingException, InternalServerException, AwsServiceException, SdkClientException,
            MigrationHubStrategyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetPortfolioSummaryResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetPortfolioSummaryResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPortfolioSummaryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MigrationHubStrategy");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPortfolioSummary");

            return clientHandler.execute(new ClientExecutionParams<GetPortfolioSummaryRequest, GetPortfolioSummaryResponse>()
                    .withOperationName("GetPortfolioSummary").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getPortfolioSummaryRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetPortfolioSummaryRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves detailed information about the specified recommendation report.
     * </p>
     *
     * @param getRecommendationReportDetailsRequest
     * @return Result of the GetRecommendationReportDetails operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified ID in the request is not found.
     * @throws AccessDeniedException
     *         The AWS user account does not have permission to perform the action. Check the AWS Identity and Access
     *         Management (IAM) policy associated with this account.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The request body isn't valid.
     * @throws InternalServerException
     *         The server experienced an internal error. Try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MigrationHubStrategyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MigrationHubStrategyClient.GetRecommendationReportDetails
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/migrationhubstrategy-2020-02-19/GetRecommendationReportDetails"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetRecommendationReportDetailsResponse getRecommendationReportDetails(
            GetRecommendationReportDetailsRequest getRecommendationReportDetailsRequest) throws ResourceNotFoundException,
            AccessDeniedException, ThrottlingException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, MigrationHubStrategyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetRecommendationReportDetailsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetRecommendationReportDetailsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getRecommendationReportDetailsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MigrationHubStrategy");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRecommendationReportDetails");

            return clientHandler
                    .execute(new ClientExecutionParams<GetRecommendationReportDetailsRequest, GetRecommendationReportDetailsResponse>()
                            .withOperationName("GetRecommendationReportDetails").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(getRecommendationReportDetailsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetRecommendationReportDetailsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves detailed information about a specified server.
     * </p>
     *
     * @param getServerDetailsRequest
     * @return Result of the GetServerDetails operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified ID in the request is not found.
     * @throws AccessDeniedException
     *         The AWS user account does not have permission to perform the action. Check the AWS Identity and Access
     *         Management (IAM) policy associated with this account.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The request body isn't valid.
     * @throws InternalServerException
     *         The server experienced an internal error. Try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MigrationHubStrategyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MigrationHubStrategyClient.GetServerDetails
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/migrationhubstrategy-2020-02-19/GetServerDetails"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetServerDetailsResponse getServerDetails(GetServerDetailsRequest getServerDetailsRequest)
            throws ResourceNotFoundException, AccessDeniedException, ThrottlingException, ValidationException,
            InternalServerException, AwsServiceException, SdkClientException, MigrationHubStrategyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetServerDetailsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetServerDetailsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getServerDetailsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MigrationHubStrategy");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetServerDetails");

            return clientHandler.execute(new ClientExecutionParams<GetServerDetailsRequest, GetServerDetailsResponse>()
                    .withOperationName("GetServerDetails").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getServerDetailsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetServerDetailsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves detailed information about a specified server.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #getServerDetails(software.amazon.awssdk.services.migrationhubstrategy.model.GetServerDetailsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.migrationhubstrategy.paginators.GetServerDetailsIterable responses = client.getServerDetailsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.migrationhubstrategy.paginators.GetServerDetailsIterable responses = client
     *             .getServerDetailsPaginator(request);
     *     for (software.amazon.awssdk.services.migrationhubstrategy.model.GetServerDetailsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.migrationhubstrategy.paginators.GetServerDetailsIterable responses = client.getServerDetailsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getServerDetails(software.amazon.awssdk.services.migrationhubstrategy.model.GetServerDetailsRequest)}
     * operation.</b>
     * </p>
     *
     * @param getServerDetailsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceNotFoundException
     *         The specified ID in the request is not found.
     * @throws AccessDeniedException
     *         The AWS user account does not have permission to perform the action. Check the AWS Identity and Access
     *         Management (IAM) policy associated with this account.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The request body isn't valid.
     * @throws InternalServerException
     *         The server experienced an internal error. Try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MigrationHubStrategyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MigrationHubStrategyClient.GetServerDetails
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/migrationhubstrategy-2020-02-19/GetServerDetails"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetServerDetailsIterable getServerDetailsPaginator(GetServerDetailsRequest getServerDetailsRequest)
            throws ResourceNotFoundException, AccessDeniedException, ThrottlingException, ValidationException,
            InternalServerException, AwsServiceException, SdkClientException, MigrationHubStrategyException {
        return new GetServerDetailsIterable(this, applyPaginatorUserAgent(getServerDetailsRequest));
    }

    /**
     * <p>
     * Retrieves recommended strategies and tools for the specified server.
     * </p>
     *
     * @param getServerStrategiesRequest
     * @return Result of the GetServerStrategies operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified ID in the request is not found.
     * @throws AccessDeniedException
     *         The AWS user account does not have permission to perform the action. Check the AWS Identity and Access
     *         Management (IAM) policy associated with this account.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The request body isn't valid.
     * @throws InternalServerException
     *         The server experienced an internal error. Try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MigrationHubStrategyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MigrationHubStrategyClient.GetServerStrategies
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/migrationhubstrategy-2020-02-19/GetServerStrategies"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetServerStrategiesResponse getServerStrategies(GetServerStrategiesRequest getServerStrategiesRequest)
            throws ResourceNotFoundException, AccessDeniedException, ThrottlingException, ValidationException,
            InternalServerException, AwsServiceException, SdkClientException, MigrationHubStrategyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetServerStrategiesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetServerStrategiesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getServerStrategiesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MigrationHubStrategy");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetServerStrategies");

            return clientHandler.execute(new ClientExecutionParams<GetServerStrategiesRequest, GetServerStrategiesResponse>()
                    .withOperationName("GetServerStrategies").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getServerStrategiesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetServerStrategiesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves a list of all the application components (processes).
     * </p>
     *
     * @param listApplicationComponentsRequest
     * @return Result of the ListApplicationComponents operation returned by the service.
     * @throws AccessDeniedException
     *         The AWS user account does not have permission to perform the action. Check the AWS Identity and Access
     *         Management (IAM) policy associated with this account.
     * @throws ServiceLinkedRoleLockClientException
     *         Exception to indicate that the service-linked role (SLR) is locked.
     * @throws ValidationException
     *         The request body isn't valid.
     * @throws InternalServerException
     *         The server experienced an internal error. Try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MigrationHubStrategyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MigrationHubStrategyClient.ListApplicationComponents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/migrationhubstrategy-2020-02-19/ListApplicationComponents"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListApplicationComponentsResponse listApplicationComponents(
            ListApplicationComponentsRequest listApplicationComponentsRequest) throws AccessDeniedException,
            ServiceLinkedRoleLockClientException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, MigrationHubStrategyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListApplicationComponentsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListApplicationComponentsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listApplicationComponentsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MigrationHubStrategy");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListApplicationComponents");

            return clientHandler
                    .execute(new ClientExecutionParams<ListApplicationComponentsRequest, ListApplicationComponentsResponse>()
                            .withOperationName("ListApplicationComponents").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(listApplicationComponentsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListApplicationComponentsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves a list of all the application components (processes).
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listApplicationComponents(software.amazon.awssdk.services.migrationhubstrategy.model.ListApplicationComponentsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.migrationhubstrategy.paginators.ListApplicationComponentsIterable responses = client.listApplicationComponentsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.migrationhubstrategy.paginators.ListApplicationComponentsIterable responses = client
     *             .listApplicationComponentsPaginator(request);
     *     for (software.amazon.awssdk.services.migrationhubstrategy.model.ListApplicationComponentsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.migrationhubstrategy.paginators.ListApplicationComponentsIterable responses = client.listApplicationComponentsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listApplicationComponents(software.amazon.awssdk.services.migrationhubstrategy.model.ListApplicationComponentsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listApplicationComponentsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws AccessDeniedException
     *         The AWS user account does not have permission to perform the action. Check the AWS Identity and Access
     *         Management (IAM) policy associated with this account.
     * @throws ServiceLinkedRoleLockClientException
     *         Exception to indicate that the service-linked role (SLR) is locked.
     * @throws ValidationException
     *         The request body isn't valid.
     * @throws InternalServerException
     *         The server experienced an internal error. Try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MigrationHubStrategyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MigrationHubStrategyClient.ListApplicationComponents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/migrationhubstrategy-2020-02-19/ListApplicationComponents"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListApplicationComponentsIterable listApplicationComponentsPaginator(
            ListApplicationComponentsRequest listApplicationComponentsRequest) throws AccessDeniedException,
            ServiceLinkedRoleLockClientException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, MigrationHubStrategyException {
        return new ListApplicationComponentsIterable(this, applyPaginatorUserAgent(listApplicationComponentsRequest));
    }

    /**
     * <p>
     * Retrieves a list of all the installed collectors.
     * </p>
     *
     * @param listCollectorsRequest
     * @return Result of the ListCollectors operation returned by the service.
     * @throws AccessDeniedException
     *         The AWS user account does not have permission to perform the action. Check the AWS Identity and Access
     *         Management (IAM) policy associated with this account.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The request body isn't valid.
     * @throws InternalServerException
     *         The server experienced an internal error. Try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MigrationHubStrategyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MigrationHubStrategyClient.ListCollectors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/migrationhubstrategy-2020-02-19/ListCollectors"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListCollectorsResponse listCollectors(ListCollectorsRequest listCollectorsRequest) throws AccessDeniedException,
            ThrottlingException, ValidationException, InternalServerException, AwsServiceException, SdkClientException,
            MigrationHubStrategyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListCollectorsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListCollectorsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listCollectorsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MigrationHubStrategy");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListCollectors");

            return clientHandler.execute(new ClientExecutionParams<ListCollectorsRequest, ListCollectorsResponse>()
                    .withOperationName("ListCollectors").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listCollectorsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListCollectorsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves a list of all the installed collectors.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listCollectors(software.amazon.awssdk.services.migrationhubstrategy.model.ListCollectorsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.migrationhubstrategy.paginators.ListCollectorsIterable responses = client.listCollectorsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.migrationhubstrategy.paginators.ListCollectorsIterable responses = client
     *             .listCollectorsPaginator(request);
     *     for (software.amazon.awssdk.services.migrationhubstrategy.model.ListCollectorsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.migrationhubstrategy.paginators.ListCollectorsIterable responses = client.listCollectorsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listCollectors(software.amazon.awssdk.services.migrationhubstrategy.model.ListCollectorsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listCollectorsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws AccessDeniedException
     *         The AWS user account does not have permission to perform the action. Check the AWS Identity and Access
     *         Management (IAM) policy associated with this account.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The request body isn't valid.
     * @throws InternalServerException
     *         The server experienced an internal error. Try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MigrationHubStrategyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MigrationHubStrategyClient.ListCollectors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/migrationhubstrategy-2020-02-19/ListCollectors"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListCollectorsIterable listCollectorsPaginator(ListCollectorsRequest listCollectorsRequest)
            throws AccessDeniedException, ThrottlingException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, MigrationHubStrategyException {
        return new ListCollectorsIterable(this, applyPaginatorUserAgent(listCollectorsRequest));
    }

    /**
     * <p>
     * Retrieves a list of all the imports performed.
     * </p>
     *
     * @param listImportFileTaskRequest
     * @return Result of the ListImportFileTask operation returned by the service.
     * @throws AccessDeniedException
     *         The AWS user account does not have permission to perform the action. Check the AWS Identity and Access
     *         Management (IAM) policy associated with this account.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The request body isn't valid.
     * @throws InternalServerException
     *         The server experienced an internal error. Try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MigrationHubStrategyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MigrationHubStrategyClient.ListImportFileTask
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/migrationhubstrategy-2020-02-19/ListImportFileTask"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListImportFileTaskResponse listImportFileTask(ListImportFileTaskRequest listImportFileTaskRequest)
            throws AccessDeniedException, ThrottlingException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, MigrationHubStrategyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListImportFileTaskResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListImportFileTaskResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listImportFileTaskRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MigrationHubStrategy");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListImportFileTask");

            return clientHandler.execute(new ClientExecutionParams<ListImportFileTaskRequest, ListImportFileTaskResponse>()
                    .withOperationName("ListImportFileTask").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listImportFileTaskRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListImportFileTaskRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves a list of all the imports performed.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listImportFileTask(software.amazon.awssdk.services.migrationhubstrategy.model.ListImportFileTaskRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.migrationhubstrategy.paginators.ListImportFileTaskIterable responses = client.listImportFileTaskPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.migrationhubstrategy.paginators.ListImportFileTaskIterable responses = client
     *             .listImportFileTaskPaginator(request);
     *     for (software.amazon.awssdk.services.migrationhubstrategy.model.ListImportFileTaskResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.migrationhubstrategy.paginators.ListImportFileTaskIterable responses = client.listImportFileTaskPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listImportFileTask(software.amazon.awssdk.services.migrationhubstrategy.model.ListImportFileTaskRequest)}
     * operation.</b>
     * </p>
     *
     * @param listImportFileTaskRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws AccessDeniedException
     *         The AWS user account does not have permission to perform the action. Check the AWS Identity and Access
     *         Management (IAM) policy associated with this account.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The request body isn't valid.
     * @throws InternalServerException
     *         The server experienced an internal error. Try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MigrationHubStrategyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MigrationHubStrategyClient.ListImportFileTask
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/migrationhubstrategy-2020-02-19/ListImportFileTask"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListImportFileTaskIterable listImportFileTaskPaginator(ListImportFileTaskRequest listImportFileTaskRequest)
            throws AccessDeniedException, ThrottlingException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, MigrationHubStrategyException {
        return new ListImportFileTaskIterable(this, applyPaginatorUserAgent(listImportFileTaskRequest));
    }

    /**
     * <p>
     * Returns a list of all the servers.
     * </p>
     *
     * @param listServersRequest
     * @return Result of the ListServers operation returned by the service.
     * @throws AccessDeniedException
     *         The AWS user account does not have permission to perform the action. Check the AWS Identity and Access
     *         Management (IAM) policy associated with this account.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The request body isn't valid.
     * @throws InternalServerException
     *         The server experienced an internal error. Try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MigrationHubStrategyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MigrationHubStrategyClient.ListServers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/migrationhubstrategy-2020-02-19/ListServers"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListServersResponse listServers(ListServersRequest listServersRequest) throws AccessDeniedException,
            ThrottlingException, ValidationException, InternalServerException, AwsServiceException, SdkClientException,
            MigrationHubStrategyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListServersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListServersResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listServersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MigrationHubStrategy");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListServers");

            return clientHandler.execute(new ClientExecutionParams<ListServersRequest, ListServersResponse>()
                    .withOperationName("ListServers").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listServersRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListServersRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of all the servers.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listServers(software.amazon.awssdk.services.migrationhubstrategy.model.ListServersRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.migrationhubstrategy.paginators.ListServersIterable responses = client.listServersPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.migrationhubstrategy.paginators.ListServersIterable responses = client
     *             .listServersPaginator(request);
     *     for (software.amazon.awssdk.services.migrationhubstrategy.model.ListServersResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.migrationhubstrategy.paginators.ListServersIterable responses = client.listServersPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listServers(software.amazon.awssdk.services.migrationhubstrategy.model.ListServersRequest)}
     * operation.</b>
     * </p>
     *
     * @param listServersRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws AccessDeniedException
     *         The AWS user account does not have permission to perform the action. Check the AWS Identity and Access
     *         Management (IAM) policy associated with this account.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The request body isn't valid.
     * @throws InternalServerException
     *         The server experienced an internal error. Try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MigrationHubStrategyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MigrationHubStrategyClient.ListServers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/migrationhubstrategy-2020-02-19/ListServers"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListServersIterable listServersPaginator(ListServersRequest listServersRequest) throws AccessDeniedException,
            ThrottlingException, ValidationException, InternalServerException, AwsServiceException, SdkClientException,
            MigrationHubStrategyException {
        return new ListServersIterable(this, applyPaginatorUserAgent(listServersRequest));
    }

    /**
     * <p>
     * Saves the specified migration and modernization preferences.
     * </p>
     *
     * @param putPortfolioPreferencesRequest
     * @return Result of the PutPortfolioPreferences operation returned by the service.
     * @throws AccessDeniedException
     *         The AWS user account does not have permission to perform the action. Check the AWS Identity and Access
     *         Management (IAM) policy associated with this account.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The request body isn't valid.
     * @throws ConflictException
     *         Exception to indicate that there is an ongoing task when a new task is created. Return when once the
     *         existing tasks are complete.
     * @throws InternalServerException
     *         The server experienced an internal error. Try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MigrationHubStrategyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MigrationHubStrategyClient.PutPortfolioPreferences
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/migrationhubstrategy-2020-02-19/PutPortfolioPreferences"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public PutPortfolioPreferencesResponse putPortfolioPreferences(PutPortfolioPreferencesRequest putPortfolioPreferencesRequest)
            throws AccessDeniedException, ThrottlingException, ValidationException, ConflictException, InternalServerException,
            AwsServiceException, SdkClientException, MigrationHubStrategyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutPortfolioPreferencesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, PutPortfolioPreferencesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putPortfolioPreferencesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MigrationHubStrategy");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutPortfolioPreferences");

            return clientHandler
                    .execute(new ClientExecutionParams<PutPortfolioPreferencesRequest, PutPortfolioPreferencesResponse>()
                            .withOperationName("PutPortfolioPreferences").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(putPortfolioPreferencesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new PutPortfolioPreferencesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Starts the assessment of an on-premises environment.
     * </p>
     *
     * @param startAssessmentRequest
     * @return Result of the StartAssessment operation returned by the service.
     * @throws AccessDeniedException
     *         The AWS user account does not have permission to perform the action. Check the AWS Identity and Access
     *         Management (IAM) policy associated with this account.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ServiceQuotaExceededException
     *         The AWS account has reached its quota of imports. Contact AWS Support to increase the quota for this
     *         account.
     * @throws InternalServerException
     *         The server experienced an internal error. Try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MigrationHubStrategyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MigrationHubStrategyClient.StartAssessment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/migrationhubstrategy-2020-02-19/StartAssessment"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StartAssessmentResponse startAssessment(StartAssessmentRequest startAssessmentRequest) throws AccessDeniedException,
            ThrottlingException, ServiceQuotaExceededException, InternalServerException, AwsServiceException, SdkClientException,
            MigrationHubStrategyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartAssessmentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StartAssessmentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startAssessmentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MigrationHubStrategy");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartAssessment");

            return clientHandler.execute(new ClientExecutionParams<StartAssessmentRequest, StartAssessmentResponse>()
                    .withOperationName("StartAssessment").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(startAssessmentRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartAssessmentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Starts a file import.
     * </p>
     *
     * @param startImportFileTaskRequest
     * @return Result of the StartImportFileTask operation returned by the service.
     * @throws AccessDeniedException
     *         The AWS user account does not have permission to perform the action. Check the AWS Identity and Access
     *         Management (IAM) policy associated with this account.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ServiceQuotaExceededException
     *         The AWS account has reached its quota of imports. Contact AWS Support to increase the quota for this
     *         account.
     * @throws ValidationException
     *         The request body isn't valid.
     * @throws InternalServerException
     *         The server experienced an internal error. Try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MigrationHubStrategyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MigrationHubStrategyClient.StartImportFileTask
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/migrationhubstrategy-2020-02-19/StartImportFileTask"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StartImportFileTaskResponse startImportFileTask(StartImportFileTaskRequest startImportFileTaskRequest)
            throws AccessDeniedException, ThrottlingException, ServiceQuotaExceededException, ValidationException,
            InternalServerException, AwsServiceException, SdkClientException, MigrationHubStrategyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartImportFileTaskResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StartImportFileTaskResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startImportFileTaskRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MigrationHubStrategy");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartImportFileTask");

            return clientHandler.execute(new ClientExecutionParams<StartImportFileTaskRequest, StartImportFileTaskResponse>()
                    .withOperationName("StartImportFileTask").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(startImportFileTaskRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartImportFileTaskRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Starts generating a recommendation report.
     * </p>
     *
     * @param startRecommendationReportGenerationRequest
     * @return Result of the StartRecommendationReportGeneration operation returned by the service.
     * @throws AccessDeniedException
     *         The AWS user account does not have permission to perform the action. Check the AWS Identity and Access
     *         Management (IAM) policy associated with this account.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The request body isn't valid.
     * @throws ConflictException
     *         Exception to indicate that there is an ongoing task when a new task is created. Return when once the
     *         existing tasks are complete.
     * @throws InternalServerException
     *         The server experienced an internal error. Try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MigrationHubStrategyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MigrationHubStrategyClient.StartRecommendationReportGeneration
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/migrationhubstrategy-2020-02-19/StartRecommendationReportGeneration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StartRecommendationReportGenerationResponse startRecommendationReportGeneration(
            StartRecommendationReportGenerationRequest startRecommendationReportGenerationRequest) throws AccessDeniedException,
            ThrottlingException, ValidationException, ConflictException, InternalServerException, AwsServiceException,
            SdkClientException, MigrationHubStrategyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartRecommendationReportGenerationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StartRecommendationReportGenerationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                startRecommendationReportGenerationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MigrationHubStrategy");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartRecommendationReportGeneration");

            return clientHandler
                    .execute(new ClientExecutionParams<StartRecommendationReportGenerationRequest, StartRecommendationReportGenerationResponse>()
                            .withOperationName("StartRecommendationReportGeneration").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(startRecommendationReportGenerationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new StartRecommendationReportGenerationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Stops the assessment of an on-premises environment.
     * </p>
     *
     * @param stopAssessmentRequest
     * @return Result of the StopAssessment operation returned by the service.
     * @throws AccessDeniedException
     *         The AWS user account does not have permission to perform the action. Check the AWS Identity and Access
     *         Management (IAM) policy associated with this account.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The request body isn't valid.
     * @throws InternalServerException
     *         The server experienced an internal error. Try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MigrationHubStrategyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MigrationHubStrategyClient.StopAssessment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/migrationhubstrategy-2020-02-19/StopAssessment"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StopAssessmentResponse stopAssessment(StopAssessmentRequest stopAssessmentRequest) throws AccessDeniedException,
            ThrottlingException, ValidationException, InternalServerException, AwsServiceException, SdkClientException,
            MigrationHubStrategyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StopAssessmentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StopAssessmentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopAssessmentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MigrationHubStrategy");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopAssessment");

            return clientHandler.execute(new ClientExecutionParams<StopAssessmentRequest, StopAssessmentResponse>()
                    .withOperationName("StopAssessment").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(stopAssessmentRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StopAssessmentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the configuration of an application component.
     * </p>
     *
     * @param updateApplicationComponentConfigRequest
     * @return Result of the UpdateApplicationComponentConfig operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified ID in the request is not found.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The request body isn't valid.
     * @throws InternalServerException
     *         The server experienced an internal error. Try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MigrationHubStrategyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MigrationHubStrategyClient.UpdateApplicationComponentConfig
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/migrationhubstrategy-2020-02-19/UpdateApplicationComponentConfig"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateApplicationComponentConfigResponse updateApplicationComponentConfig(
            UpdateApplicationComponentConfigRequest updateApplicationComponentConfigRequest) throws ResourceNotFoundException,
            ThrottlingException, ValidationException, InternalServerException, AwsServiceException, SdkClientException,
            MigrationHubStrategyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateApplicationComponentConfigResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateApplicationComponentConfigResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateApplicationComponentConfigRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MigrationHubStrategy");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateApplicationComponentConfig");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateApplicationComponentConfigRequest, UpdateApplicationComponentConfigResponse>()
                            .withOperationName("UpdateApplicationComponentConfig").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(updateApplicationComponentConfigRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateApplicationComponentConfigRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the configuration of the specified server.
     * </p>
     *
     * @param updateServerConfigRequest
     * @return Result of the UpdateServerConfig operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified ID in the request is not found.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The request body isn't valid.
     * @throws InternalServerException
     *         The server experienced an internal error. Try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MigrationHubStrategyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MigrationHubStrategyClient.UpdateServerConfig
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/migrationhubstrategy-2020-02-19/UpdateServerConfig"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateServerConfigResponse updateServerConfig(UpdateServerConfigRequest updateServerConfigRequest)
            throws ResourceNotFoundException, ThrottlingException, ValidationException, InternalServerException,
            AwsServiceException, SdkClientException, MigrationHubStrategyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateServerConfigResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateServerConfigResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateServerConfigRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MigrationHubStrategy");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateServerConfig");

            return clientHandler.execute(new ClientExecutionParams<UpdateServerConfigRequest, UpdateServerConfigResponse>()
                    .withOperationName("UpdateServerConfig").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateServerConfigRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateServerConfigRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(MigrationHubStrategyException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceLinkedRoleLockClientException")
                                .exceptionBuilderSupplier(ServiceLinkedRoleLockClientException::builder).httpStatusCode(400)
                                .build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends MigrationHubStrategyRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
