/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.nimble.paginators;

import java.util.Collections;
import java.util.Iterator;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.pagination.sync.PaginatedItemsIterable;
import software.amazon.awssdk.core.pagination.sync.PaginatedResponsesIterator;
import software.amazon.awssdk.core.pagination.sync.SdkIterable;
import software.amazon.awssdk.core.pagination.sync.SyncPageFetcher;
import software.amazon.awssdk.core.util.PaginatorUtils;
import software.amazon.awssdk.services.nimble.NimbleClient;
import software.amazon.awssdk.services.nimble.model.ListStudiosRequest;
import software.amazon.awssdk.services.nimble.model.ListStudiosResponse;
import software.amazon.awssdk.services.nimble.model.Studio;

/**
 * <p>
 * Represents the output for the
 * {@link software.amazon.awssdk.services.nimble.NimbleClient#listStudiosPaginator(software.amazon.awssdk.services.nimble.model.ListStudiosRequest)}
 * operation which is a paginated operation. This class is an iterable of
 * {@link software.amazon.awssdk.services.nimble.model.ListStudiosResponse} that can be used to iterate through all the
 * response pages of the operation.
 * </p>
 * <p>
 * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet and
 * so there is no guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily
 * loading response pages by making service calls until there are no pages left or your iteration stops. If there are
 * errors in your request, you will see the failures only after you start iterating through the iterable.
 * </p>
 *
 * <p>
 * The following are few ways to iterate through the response pages:
 * </p>
 * 1) Using a Stream
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.nimble.paginators.ListStudiosIterable responses = client.listStudiosPaginator(request);
 * responses.stream().forEach(....);
 * }
 * </pre>
 *
 * 2) Using For loop
 * 
 * <pre>
 * {
 *     &#064;code
 *     software.amazon.awssdk.services.nimble.paginators.ListStudiosIterable responses = client.listStudiosPaginator(request);
 *     for (software.amazon.awssdk.services.nimble.model.ListStudiosResponse response : responses) {
 *         // do something;
 *     }
 * }
 * </pre>
 *
 * 3) Use iterator directly
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.nimble.paginators.ListStudiosIterable responses = client.listStudiosPaginator(request);
 * responses.iterator().forEachRemaining(....);
 * }
 * </pre>
 * <p>
 * <b>Please notice that the configuration of null won't limit the number of results you get with the paginator. It only
 * limits the number of results in each page.</b>
 * </p>
 * <p>
 * <b>Note: If you prefer to have control on service calls, use the
 * {@link #listStudios(software.amazon.awssdk.services.nimble.model.ListStudiosRequest)} operation.</b>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class ListStudiosIterable implements SdkIterable<ListStudiosResponse> {
    private final NimbleClient client;

    private final ListStudiosRequest firstRequest;

    private final SyncPageFetcher nextPageFetcher;

    public ListStudiosIterable(NimbleClient client, ListStudiosRequest firstRequest) {
        this.client = client;
        this.firstRequest = firstRequest;
        this.nextPageFetcher = new ListStudiosResponseFetcher();
    }

    @Override
    public Iterator<ListStudiosResponse> iterator() {
        return PaginatedResponsesIterator.builder().nextPageFetcher(nextPageFetcher).build();
    }

    /**
     * Returns an iterable to iterate through the paginated {@link ListStudiosResponse#studios()} member. The returned
     * iterable is used to iterate through the results across all response pages and not a single page.
     *
     * This method is useful if you are interested in iterating over the paginated member in the response pages instead
     * of the top level pages. Similar to iteration over pages, this method internally makes service calls to get the
     * next list of results until the iteration stops or there are no more results.
     */
    public final SdkIterable<Studio> studios() {
        Function<ListStudiosResponse, Iterator<Studio>> getIterator = response -> {
            if (response != null && response.studios() != null) {
                return response.studios().iterator();
            }
            return Collections.emptyIterator();
        };
        return PaginatedItemsIterable.<ListStudiosResponse, Studio> builder().pagesIterable(this)
                .itemIteratorFunction(getIterator).build();
    }

    private class ListStudiosResponseFetcher implements SyncPageFetcher<ListStudiosResponse> {
        @Override
        public boolean hasNextPage(ListStudiosResponse previousPage) {
            return PaginatorUtils.isOutputTokenAvailable(previousPage.nextToken());
        }

        @Override
        public ListStudiosResponse nextPage(ListStudiosResponse previousPage) {
            if (previousPage == null) {
                return client.listStudios(firstRequest);
            }
            return client.listStudios(firstRequest.toBuilder().nextToken(previousPage.nextToken()).build());
        }
    }
}
