/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.opensearch;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.opensearch.model.AcceptInboundConnectionRequest;
import software.amazon.awssdk.services.opensearch.model.AcceptInboundConnectionResponse;
import software.amazon.awssdk.services.opensearch.model.AccessDeniedException;
import software.amazon.awssdk.services.opensearch.model.AddTagsRequest;
import software.amazon.awssdk.services.opensearch.model.AddTagsResponse;
import software.amazon.awssdk.services.opensearch.model.AssociatePackageRequest;
import software.amazon.awssdk.services.opensearch.model.AssociatePackageResponse;
import software.amazon.awssdk.services.opensearch.model.AuthorizeVpcEndpointAccessRequest;
import software.amazon.awssdk.services.opensearch.model.AuthorizeVpcEndpointAccessResponse;
import software.amazon.awssdk.services.opensearch.model.BaseException;
import software.amazon.awssdk.services.opensearch.model.CancelServiceSoftwareUpdateRequest;
import software.amazon.awssdk.services.opensearch.model.CancelServiceSoftwareUpdateResponse;
import software.amazon.awssdk.services.opensearch.model.ConflictException;
import software.amazon.awssdk.services.opensearch.model.CreateDomainRequest;
import software.amazon.awssdk.services.opensearch.model.CreateDomainResponse;
import software.amazon.awssdk.services.opensearch.model.CreateOutboundConnectionRequest;
import software.amazon.awssdk.services.opensearch.model.CreateOutboundConnectionResponse;
import software.amazon.awssdk.services.opensearch.model.CreatePackageRequest;
import software.amazon.awssdk.services.opensearch.model.CreatePackageResponse;
import software.amazon.awssdk.services.opensearch.model.CreateVpcEndpointRequest;
import software.amazon.awssdk.services.opensearch.model.CreateVpcEndpointResponse;
import software.amazon.awssdk.services.opensearch.model.DeleteDomainRequest;
import software.amazon.awssdk.services.opensearch.model.DeleteDomainResponse;
import software.amazon.awssdk.services.opensearch.model.DeleteInboundConnectionRequest;
import software.amazon.awssdk.services.opensearch.model.DeleteInboundConnectionResponse;
import software.amazon.awssdk.services.opensearch.model.DeleteOutboundConnectionRequest;
import software.amazon.awssdk.services.opensearch.model.DeleteOutboundConnectionResponse;
import software.amazon.awssdk.services.opensearch.model.DeletePackageRequest;
import software.amazon.awssdk.services.opensearch.model.DeletePackageResponse;
import software.amazon.awssdk.services.opensearch.model.DeleteVpcEndpointRequest;
import software.amazon.awssdk.services.opensearch.model.DeleteVpcEndpointResponse;
import software.amazon.awssdk.services.opensearch.model.DescribeDomainAutoTunesRequest;
import software.amazon.awssdk.services.opensearch.model.DescribeDomainAutoTunesResponse;
import software.amazon.awssdk.services.opensearch.model.DescribeDomainChangeProgressRequest;
import software.amazon.awssdk.services.opensearch.model.DescribeDomainChangeProgressResponse;
import software.amazon.awssdk.services.opensearch.model.DescribeDomainConfigRequest;
import software.amazon.awssdk.services.opensearch.model.DescribeDomainConfigResponse;
import software.amazon.awssdk.services.opensearch.model.DescribeDomainRequest;
import software.amazon.awssdk.services.opensearch.model.DescribeDomainResponse;
import software.amazon.awssdk.services.opensearch.model.DescribeDomainsRequest;
import software.amazon.awssdk.services.opensearch.model.DescribeDomainsResponse;
import software.amazon.awssdk.services.opensearch.model.DescribeInboundConnectionsRequest;
import software.amazon.awssdk.services.opensearch.model.DescribeInboundConnectionsResponse;
import software.amazon.awssdk.services.opensearch.model.DescribeInstanceTypeLimitsRequest;
import software.amazon.awssdk.services.opensearch.model.DescribeInstanceTypeLimitsResponse;
import software.amazon.awssdk.services.opensearch.model.DescribeOutboundConnectionsRequest;
import software.amazon.awssdk.services.opensearch.model.DescribeOutboundConnectionsResponse;
import software.amazon.awssdk.services.opensearch.model.DescribePackagesRequest;
import software.amazon.awssdk.services.opensearch.model.DescribePackagesResponse;
import software.amazon.awssdk.services.opensearch.model.DescribeReservedInstanceOfferingsRequest;
import software.amazon.awssdk.services.opensearch.model.DescribeReservedInstanceOfferingsResponse;
import software.amazon.awssdk.services.opensearch.model.DescribeReservedInstancesRequest;
import software.amazon.awssdk.services.opensearch.model.DescribeReservedInstancesResponse;
import software.amazon.awssdk.services.opensearch.model.DescribeVpcEndpointsRequest;
import software.amazon.awssdk.services.opensearch.model.DescribeVpcEndpointsResponse;
import software.amazon.awssdk.services.opensearch.model.DisabledOperationException;
import software.amazon.awssdk.services.opensearch.model.DissociatePackageRequest;
import software.amazon.awssdk.services.opensearch.model.DissociatePackageResponse;
import software.amazon.awssdk.services.opensearch.model.GetCompatibleVersionsRequest;
import software.amazon.awssdk.services.opensearch.model.GetCompatibleVersionsResponse;
import software.amazon.awssdk.services.opensearch.model.GetPackageVersionHistoryRequest;
import software.amazon.awssdk.services.opensearch.model.GetPackageVersionHistoryResponse;
import software.amazon.awssdk.services.opensearch.model.GetUpgradeHistoryRequest;
import software.amazon.awssdk.services.opensearch.model.GetUpgradeHistoryResponse;
import software.amazon.awssdk.services.opensearch.model.GetUpgradeStatusRequest;
import software.amazon.awssdk.services.opensearch.model.GetUpgradeStatusResponse;
import software.amazon.awssdk.services.opensearch.model.InternalException;
import software.amazon.awssdk.services.opensearch.model.InvalidPaginationTokenException;
import software.amazon.awssdk.services.opensearch.model.InvalidTypeException;
import software.amazon.awssdk.services.opensearch.model.LimitExceededException;
import software.amazon.awssdk.services.opensearch.model.ListDomainNamesRequest;
import software.amazon.awssdk.services.opensearch.model.ListDomainNamesResponse;
import software.amazon.awssdk.services.opensearch.model.ListDomainsForPackageRequest;
import software.amazon.awssdk.services.opensearch.model.ListDomainsForPackageResponse;
import software.amazon.awssdk.services.opensearch.model.ListInstanceTypeDetailsRequest;
import software.amazon.awssdk.services.opensearch.model.ListInstanceTypeDetailsResponse;
import software.amazon.awssdk.services.opensearch.model.ListPackagesForDomainRequest;
import software.amazon.awssdk.services.opensearch.model.ListPackagesForDomainResponse;
import software.amazon.awssdk.services.opensearch.model.ListTagsRequest;
import software.amazon.awssdk.services.opensearch.model.ListTagsResponse;
import software.amazon.awssdk.services.opensearch.model.ListVersionsRequest;
import software.amazon.awssdk.services.opensearch.model.ListVersionsResponse;
import software.amazon.awssdk.services.opensearch.model.ListVpcEndpointAccessRequest;
import software.amazon.awssdk.services.opensearch.model.ListVpcEndpointAccessResponse;
import software.amazon.awssdk.services.opensearch.model.ListVpcEndpointsForDomainRequest;
import software.amazon.awssdk.services.opensearch.model.ListVpcEndpointsForDomainResponse;
import software.amazon.awssdk.services.opensearch.model.ListVpcEndpointsRequest;
import software.amazon.awssdk.services.opensearch.model.ListVpcEndpointsResponse;
import software.amazon.awssdk.services.opensearch.model.OpenSearchException;
import software.amazon.awssdk.services.opensearch.model.OpenSearchRequest;
import software.amazon.awssdk.services.opensearch.model.PurchaseReservedInstanceOfferingRequest;
import software.amazon.awssdk.services.opensearch.model.PurchaseReservedInstanceOfferingResponse;
import software.amazon.awssdk.services.opensearch.model.RejectInboundConnectionRequest;
import software.amazon.awssdk.services.opensearch.model.RejectInboundConnectionResponse;
import software.amazon.awssdk.services.opensearch.model.RemoveTagsRequest;
import software.amazon.awssdk.services.opensearch.model.RemoveTagsResponse;
import software.amazon.awssdk.services.opensearch.model.ResourceAlreadyExistsException;
import software.amazon.awssdk.services.opensearch.model.ResourceNotFoundException;
import software.amazon.awssdk.services.opensearch.model.RevokeVpcEndpointAccessRequest;
import software.amazon.awssdk.services.opensearch.model.RevokeVpcEndpointAccessResponse;
import software.amazon.awssdk.services.opensearch.model.StartServiceSoftwareUpdateRequest;
import software.amazon.awssdk.services.opensearch.model.StartServiceSoftwareUpdateResponse;
import software.amazon.awssdk.services.opensearch.model.UpdateDomainConfigRequest;
import software.amazon.awssdk.services.opensearch.model.UpdateDomainConfigResponse;
import software.amazon.awssdk.services.opensearch.model.UpdatePackageRequest;
import software.amazon.awssdk.services.opensearch.model.UpdatePackageResponse;
import software.amazon.awssdk.services.opensearch.model.UpdateVpcEndpointRequest;
import software.amazon.awssdk.services.opensearch.model.UpdateVpcEndpointResponse;
import software.amazon.awssdk.services.opensearch.model.UpgradeDomainRequest;
import software.amazon.awssdk.services.opensearch.model.UpgradeDomainResponse;
import software.amazon.awssdk.services.opensearch.model.ValidationException;
import software.amazon.awssdk.services.opensearch.paginators.DescribeDomainAutoTunesPublisher;
import software.amazon.awssdk.services.opensearch.paginators.DescribeInboundConnectionsPublisher;
import software.amazon.awssdk.services.opensearch.paginators.DescribeOutboundConnectionsPublisher;
import software.amazon.awssdk.services.opensearch.paginators.DescribePackagesPublisher;
import software.amazon.awssdk.services.opensearch.paginators.DescribeReservedInstanceOfferingsPublisher;
import software.amazon.awssdk.services.opensearch.paginators.DescribeReservedInstancesPublisher;
import software.amazon.awssdk.services.opensearch.paginators.GetPackageVersionHistoryPublisher;
import software.amazon.awssdk.services.opensearch.paginators.GetUpgradeHistoryPublisher;
import software.amazon.awssdk.services.opensearch.paginators.ListDomainsForPackagePublisher;
import software.amazon.awssdk.services.opensearch.paginators.ListInstanceTypeDetailsPublisher;
import software.amazon.awssdk.services.opensearch.paginators.ListPackagesForDomainPublisher;
import software.amazon.awssdk.services.opensearch.paginators.ListVersionsPublisher;
import software.amazon.awssdk.services.opensearch.transform.AcceptInboundConnectionRequestMarshaller;
import software.amazon.awssdk.services.opensearch.transform.AddTagsRequestMarshaller;
import software.amazon.awssdk.services.opensearch.transform.AssociatePackageRequestMarshaller;
import software.amazon.awssdk.services.opensearch.transform.AuthorizeVpcEndpointAccessRequestMarshaller;
import software.amazon.awssdk.services.opensearch.transform.CancelServiceSoftwareUpdateRequestMarshaller;
import software.amazon.awssdk.services.opensearch.transform.CreateDomainRequestMarshaller;
import software.amazon.awssdk.services.opensearch.transform.CreateOutboundConnectionRequestMarshaller;
import software.amazon.awssdk.services.opensearch.transform.CreatePackageRequestMarshaller;
import software.amazon.awssdk.services.opensearch.transform.CreateVpcEndpointRequestMarshaller;
import software.amazon.awssdk.services.opensearch.transform.DeleteDomainRequestMarshaller;
import software.amazon.awssdk.services.opensearch.transform.DeleteInboundConnectionRequestMarshaller;
import software.amazon.awssdk.services.opensearch.transform.DeleteOutboundConnectionRequestMarshaller;
import software.amazon.awssdk.services.opensearch.transform.DeletePackageRequestMarshaller;
import software.amazon.awssdk.services.opensearch.transform.DeleteVpcEndpointRequestMarshaller;
import software.amazon.awssdk.services.opensearch.transform.DescribeDomainAutoTunesRequestMarshaller;
import software.amazon.awssdk.services.opensearch.transform.DescribeDomainChangeProgressRequestMarshaller;
import software.amazon.awssdk.services.opensearch.transform.DescribeDomainConfigRequestMarshaller;
import software.amazon.awssdk.services.opensearch.transform.DescribeDomainRequestMarshaller;
import software.amazon.awssdk.services.opensearch.transform.DescribeDomainsRequestMarshaller;
import software.amazon.awssdk.services.opensearch.transform.DescribeInboundConnectionsRequestMarshaller;
import software.amazon.awssdk.services.opensearch.transform.DescribeInstanceTypeLimitsRequestMarshaller;
import software.amazon.awssdk.services.opensearch.transform.DescribeOutboundConnectionsRequestMarshaller;
import software.amazon.awssdk.services.opensearch.transform.DescribePackagesRequestMarshaller;
import software.amazon.awssdk.services.opensearch.transform.DescribeReservedInstanceOfferingsRequestMarshaller;
import software.amazon.awssdk.services.opensearch.transform.DescribeReservedInstancesRequestMarshaller;
import software.amazon.awssdk.services.opensearch.transform.DescribeVpcEndpointsRequestMarshaller;
import software.amazon.awssdk.services.opensearch.transform.DissociatePackageRequestMarshaller;
import software.amazon.awssdk.services.opensearch.transform.GetCompatibleVersionsRequestMarshaller;
import software.amazon.awssdk.services.opensearch.transform.GetPackageVersionHistoryRequestMarshaller;
import software.amazon.awssdk.services.opensearch.transform.GetUpgradeHistoryRequestMarshaller;
import software.amazon.awssdk.services.opensearch.transform.GetUpgradeStatusRequestMarshaller;
import software.amazon.awssdk.services.opensearch.transform.ListDomainNamesRequestMarshaller;
import software.amazon.awssdk.services.opensearch.transform.ListDomainsForPackageRequestMarshaller;
import software.amazon.awssdk.services.opensearch.transform.ListInstanceTypeDetailsRequestMarshaller;
import software.amazon.awssdk.services.opensearch.transform.ListPackagesForDomainRequestMarshaller;
import software.amazon.awssdk.services.opensearch.transform.ListTagsRequestMarshaller;
import software.amazon.awssdk.services.opensearch.transform.ListVersionsRequestMarshaller;
import software.amazon.awssdk.services.opensearch.transform.ListVpcEndpointAccessRequestMarshaller;
import software.amazon.awssdk.services.opensearch.transform.ListVpcEndpointsForDomainRequestMarshaller;
import software.amazon.awssdk.services.opensearch.transform.ListVpcEndpointsRequestMarshaller;
import software.amazon.awssdk.services.opensearch.transform.PurchaseReservedInstanceOfferingRequestMarshaller;
import software.amazon.awssdk.services.opensearch.transform.RejectInboundConnectionRequestMarshaller;
import software.amazon.awssdk.services.opensearch.transform.RemoveTagsRequestMarshaller;
import software.amazon.awssdk.services.opensearch.transform.RevokeVpcEndpointAccessRequestMarshaller;
import software.amazon.awssdk.services.opensearch.transform.StartServiceSoftwareUpdateRequestMarshaller;
import software.amazon.awssdk.services.opensearch.transform.UpdateDomainConfigRequestMarshaller;
import software.amazon.awssdk.services.opensearch.transform.UpdatePackageRequestMarshaller;
import software.amazon.awssdk.services.opensearch.transform.UpdateVpcEndpointRequestMarshaller;
import software.amazon.awssdk.services.opensearch.transform.UpgradeDomainRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link OpenSearchAsyncClient}.
 *
 * @see OpenSearchAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultOpenSearchAsyncClient implements OpenSearchAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultOpenSearchAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultOpenSearchAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Allows the destination Amazon OpenSearch Service domain owner to accept an inbound cross-cluster search
     * connection request. For more information, see <a
     * href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/cross-cluster-search.html"
     * >Cross-cluster search for Amazon OpenSearch Service</a>.
     * </p>
     *
     * @param acceptInboundConnectionRequest
     *        Container for the parameters to the <code>AcceptInboundConnection</code> operation.
     * @return A Java Future containing the result of the AcceptInboundConnection operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException An exception for accessing or deleting a resource that does not exist.
     *         Gives http status code of 400.</li>
     *         <li>LimitExceededException An exception for trying to create more than allowed resources or
     *         sub-resources. Gives http status code of 409.</li>
     *         <li>DisabledOperationException An error occured because the client wanted to access a not supported
     *         operation. Gives http status code of 409.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OpenSearchAsyncClient.AcceptInboundConnection
     */
    @Override
    public CompletableFuture<AcceptInboundConnectionResponse> acceptInboundConnection(
            AcceptInboundConnectionRequest acceptInboundConnectionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, acceptInboundConnectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AcceptInboundConnection");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AcceptInboundConnectionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AcceptInboundConnectionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AcceptInboundConnectionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AcceptInboundConnectionRequest, AcceptInboundConnectionResponse>()
                            .withOperationName("AcceptInboundConnection")
                            .withMarshaller(new AcceptInboundConnectionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(acceptInboundConnectionRequest));
            CompletableFuture<AcceptInboundConnectionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Attaches tags to an existing Amazon OpenSearch Service domain. Tags are a set of case-sensitive key-value pairs.
     * An domain can have up to 10 tags. For more information, see <a href=
     * "https://docs.aws.amazon.com/opensearch-service/latest/developerguide/managedomains.html#managedomains-awsresorcetagging"
     * > Tagging Amazon OpenSearch Service domains</a>.
     * </p>
     *
     * @param addTagsRequest
     *        Container for the parameters to the <code>AddTags</code> operation. Specifies the tags to attach to the
     *        domain.
     * @return A Java Future containing the result of the AddTags operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BaseException An error occurred while processing the request.</li>
     *         <li>LimitExceededException An exception for trying to create more than allowed resources or
     *         sub-resources. Gives http status code of 409.</li>
     *         <li>ValidationException An exception for missing / invalid input fields. Gives http status code of 400.</li>
     *         <li>InternalException The request processing has failed because of an unknown error, exception or failure
     *         (the failure is internal to the service) . Gives http status code of 500.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OpenSearchAsyncClient.AddTags
     */
    @Override
    public CompletableFuture<AddTagsResponse> addTags(AddTagsRequest addTagsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, addTagsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AddTags");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AddTagsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    AddTagsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AddTagsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AddTagsRequest, AddTagsResponse>().withOperationName("AddTags")
                            .withMarshaller(new AddTagsRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(addTagsRequest));
            CompletableFuture<AddTagsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Associates a package with an Amazon OpenSearch Service domain. For more information, see <a
     * href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/custom-packages.html">Custom packages
     * for Amazon OpenSearch Service</a>.
     * </p>
     *
     * @param associatePackageRequest
     *        Container for the request parameters to the <code>AssociatePackage</code> operation.
     * @return A Java Future containing the result of the AssociatePackage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BaseException An error occurred while processing the request.</li>
     *         <li>InternalException The request processing has failed because of an unknown error, exception or failure
     *         (the failure is internal to the service) . Gives http status code of 500.</li>
     *         <li>ResourceNotFoundException An exception for accessing or deleting a resource that does not exist.
     *         Gives http status code of 400.</li>
     *         <li>AccessDeniedException An error occurred because user does not have permissions to access the
     *         resource. Returns HTTP status code 403.</li>
     *         <li>ValidationException An exception for missing / invalid input fields. Gives http status code of 400.</li>
     *         <li>ConflictException An error occurred because the client attempts to remove a resource that is
     *         currently in use. Returns HTTP status code 409.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OpenSearchAsyncClient.AssociatePackage
     */
    @Override
    public CompletableFuture<AssociatePackageResponse> associatePackage(AssociatePackageRequest associatePackageRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, associatePackageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociatePackage");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AssociatePackageResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AssociatePackageResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AssociatePackageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AssociatePackageRequest, AssociatePackageResponse>()
                            .withOperationName("AssociatePackage")
                            .withMarshaller(new AssociatePackageRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(associatePackageRequest));
            CompletableFuture<AssociatePackageResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides access to an Amazon OpenSearch Service domain through the use of an interface VPC endpoint.
     * </p>
     *
     * @param authorizeVpcEndpointAccessRequest
     * @return A Java Future containing the result of the AuthorizeVpcEndpointAccess operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException An exception for accessing or deleting a resource that does not exist.
     *         Gives http status code of 400.</li>
     *         <li>DisabledOperationException An error occured because the client wanted to access a not supported
     *         operation. Gives http status code of 409.</li>
     *         <li>LimitExceededException An exception for trying to create more than allowed resources or
     *         sub-resources. Gives http status code of 409.</li>
     *         <li>ValidationException An exception for missing / invalid input fields. Gives http status code of 400.</li>
     *         <li>InternalException The request processing has failed because of an unknown error, exception or failure
     *         (the failure is internal to the service) . Gives http status code of 500.</li>
     *         <li>BaseException An error occurred while processing the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OpenSearchAsyncClient.AuthorizeVpcEndpointAccess
     */
    @Override
    public CompletableFuture<AuthorizeVpcEndpointAccessResponse> authorizeVpcEndpointAccess(
            AuthorizeVpcEndpointAccessRequest authorizeVpcEndpointAccessRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, authorizeVpcEndpointAccessRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AuthorizeVpcEndpointAccess");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AuthorizeVpcEndpointAccessResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AuthorizeVpcEndpointAccessResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AuthorizeVpcEndpointAccessResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AuthorizeVpcEndpointAccessRequest, AuthorizeVpcEndpointAccessResponse>()
                            .withOperationName("AuthorizeVpcEndpointAccess")
                            .withMarshaller(new AuthorizeVpcEndpointAccessRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(authorizeVpcEndpointAccessRequest));
            CompletableFuture<AuthorizeVpcEndpointAccessResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Cancels a scheduled service software update for an Amazon OpenSearch Service domain. You can only perform this
     * operation before the <code>AutomatedUpdateDate</code> and when the domain's <code>UpdateStatus</code> is
     * <code>PENDING_UPDATE</code>. For more information, see <a
     * href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/service-software.html">Service
     * software updates in Amazon OpenSearch Service</a>.
     * </p>
     *
     * @param cancelServiceSoftwareUpdateRequest
     *        Container for the request parameters to cancel a service software update.
     * @return A Java Future containing the result of the CancelServiceSoftwareUpdate operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BaseException An error occurred while processing the request.</li>
     *         <li>InternalException The request processing has failed because of an unknown error, exception or failure
     *         (the failure is internal to the service) . Gives http status code of 500.</li>
     *         <li>ResourceNotFoundException An exception for accessing or deleting a resource that does not exist.
     *         Gives http status code of 400.</li>
     *         <li>ValidationException An exception for missing / invalid input fields. Gives http status code of 400.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OpenSearchAsyncClient.CancelServiceSoftwareUpdate
     */
    @Override
    public CompletableFuture<CancelServiceSoftwareUpdateResponse> cancelServiceSoftwareUpdate(
            CancelServiceSoftwareUpdateRequest cancelServiceSoftwareUpdateRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, cancelServiceSoftwareUpdateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CancelServiceSoftwareUpdate");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CancelServiceSoftwareUpdateResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CancelServiceSoftwareUpdateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CancelServiceSoftwareUpdateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CancelServiceSoftwareUpdateRequest, CancelServiceSoftwareUpdateResponse>()
                            .withOperationName("CancelServiceSoftwareUpdate")
                            .withMarshaller(new CancelServiceSoftwareUpdateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(cancelServiceSoftwareUpdateRequest));
            CompletableFuture<CancelServiceSoftwareUpdateResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an Amazon OpenSearch Service domain. For more information, see <a
     * href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/createupdatedomains.html">Creating and
     * managing Amazon OpenSearch Service domains</a>.
     * </p>
     *
     * @param createDomainRequest
     * @return A Java Future containing the result of the CreateDomain operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BaseException An error occurred while processing the request.</li>
     *         <li>DisabledOperationException An error occured because the client wanted to access a not supported
     *         operation. Gives http status code of 409.</li>
     *         <li>InternalException The request processing has failed because of an unknown error, exception or failure
     *         (the failure is internal to the service) . Gives http status code of 500.</li>
     *         <li>InvalidTypeException An exception for trying to create or access sub-resource that is either invalid
     *         or not supported. Gives http status code of 409.</li>
     *         <li>LimitExceededException An exception for trying to create more than allowed resources or
     *         sub-resources. Gives http status code of 409.</li>
     *         <li>ResourceAlreadyExistsException An exception for creating a resource that already exists. Gives http
     *         status code of 400.</li>
     *         <li>ValidationException An exception for missing / invalid input fields. Gives http status code of 400.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OpenSearchAsyncClient.CreateDomain
     */
    @Override
    public CompletableFuture<CreateDomainResponse> createDomain(CreateDomainRequest createDomainRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDomainRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDomain");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateDomainResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateDomainResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateDomainResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateDomainRequest, CreateDomainResponse>()
                            .withOperationName("CreateDomain").withMarshaller(new CreateDomainRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createDomainRequest));
            CompletableFuture<CreateDomainResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new cross-cluster search connection from a source Amazon OpenSearch Service domain to a destination
     * domain. For more information, see <a
     * href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/cross-cluster-search.html"
     * >Cross-cluster search for Amazon OpenSearch Service</a>.
     * </p>
     *
     * @param createOutboundConnectionRequest
     *        Container for the parameters to the <code>CreateOutboundConnection</code> operation.
     * @return A Java Future containing the result of the CreateOutboundConnection operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>LimitExceededException An exception for trying to create more than allowed resources or
     *         sub-resources. Gives http status code of 409.</li>
     *         <li>InternalException The request processing has failed because of an unknown error, exception or failure
     *         (the failure is internal to the service) . Gives http status code of 500.</li>
     *         <li>ResourceAlreadyExistsException An exception for creating a resource that already exists. Gives http
     *         status code of 400.</li>
     *         <li>DisabledOperationException An error occured because the client wanted to access a not supported
     *         operation. Gives http status code of 409.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OpenSearchAsyncClient.CreateOutboundConnection
     */
    @Override
    public CompletableFuture<CreateOutboundConnectionResponse> createOutboundConnection(
            CreateOutboundConnectionRequest createOutboundConnectionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createOutboundConnectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateOutboundConnection");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateOutboundConnectionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateOutboundConnectionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateOutboundConnectionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateOutboundConnectionRequest, CreateOutboundConnectionResponse>()
                            .withOperationName("CreateOutboundConnection")
                            .withMarshaller(new CreateOutboundConnectionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createOutboundConnectionRequest));
            CompletableFuture<CreateOutboundConnectionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a package for use with Amazon OpenSearch Service domains. For more information, see <a
     * href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/custom-packages.html">Custom packages
     * for Amazon OpenSearch Service</a>.
     * </p>
     *
     * @param createPackageRequest
     *        Container for request parameters to the <code>CreatePackage</code> operation.
     * @return A Java Future containing the result of the CreatePackage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BaseException An error occurred while processing the request.</li>
     *         <li>InternalException The request processing has failed because of an unknown error, exception or failure
     *         (the failure is internal to the service) . Gives http status code of 500.</li>
     *         <li>LimitExceededException An exception for trying to create more than allowed resources or
     *         sub-resources. Gives http status code of 409.</li>
     *         <li>InvalidTypeException An exception for trying to create or access sub-resource that is either invalid
     *         or not supported. Gives http status code of 409.</li>
     *         <li>ResourceAlreadyExistsException An exception for creating a resource that already exists. Gives http
     *         status code of 400.</li>
     *         <li>AccessDeniedException An error occurred because user does not have permissions to access the
     *         resource. Returns HTTP status code 403.</li>
     *         <li>ValidationException An exception for missing / invalid input fields. Gives http status code of 400.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OpenSearchAsyncClient.CreatePackage
     */
    @Override
    public CompletableFuture<CreatePackageResponse> createPackage(CreatePackageRequest createPackageRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createPackageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreatePackage");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreatePackageResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreatePackageResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreatePackageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreatePackageRequest, CreatePackageResponse>()
                            .withOperationName("CreatePackage")
                            .withMarshaller(new CreatePackageRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createPackageRequest));
            CompletableFuture<CreatePackageResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an Amazon OpenSearch Service-managed VPC endpoint.
     * </p>
     *
     * @param createVpcEndpointRequest
     * @return A Java Future containing the result of the CreateVpcEndpoint operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException An error occurred because the client attempts to remove a resource that is
     *         currently in use. Returns HTTP status code 409.</li>
     *         <li>ValidationException An exception for missing / invalid input fields. Gives http status code of 400.</li>
     *         <li>LimitExceededException An exception for trying to create more than allowed resources or
     *         sub-resources. Gives http status code of 409.</li>
     *         <li>InternalException The request processing has failed because of an unknown error, exception or failure
     *         (the failure is internal to the service) . Gives http status code of 500.</li>
     *         <li>DisabledOperationException An error occured because the client wanted to access a not supported
     *         operation. Gives http status code of 409.</li>
     *         <li>BaseException An error occurred while processing the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OpenSearchAsyncClient.CreateVpcEndpoint
     */
    @Override
    public CompletableFuture<CreateVpcEndpointResponse> createVpcEndpoint(CreateVpcEndpointRequest createVpcEndpointRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createVpcEndpointRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateVpcEndpoint");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateVpcEndpointResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateVpcEndpointResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateVpcEndpointResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateVpcEndpointRequest, CreateVpcEndpointResponse>()
                            .withOperationName("CreateVpcEndpoint")
                            .withMarshaller(new CreateVpcEndpointRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createVpcEndpointRequest));
            CompletableFuture<CreateVpcEndpointResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an Amazon OpenSearch Service domain and all of its data. You can't recover a domain after you delete it.
     * </p>
     *
     * @param deleteDomainRequest
     *        Container for the parameters to the <code>DeleteDomain</code> operation.
     * @return A Java Future containing the result of the DeleteDomain operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BaseException An error occurred while processing the request.</li>
     *         <li>InternalException The request processing has failed because of an unknown error, exception or failure
     *         (the failure is internal to the service) . Gives http status code of 500.</li>
     *         <li>ResourceNotFoundException An exception for accessing or deleting a resource that does not exist.
     *         Gives http status code of 400.</li>
     *         <li>ValidationException An exception for missing / invalid input fields. Gives http status code of 400.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OpenSearchAsyncClient.DeleteDomain
     */
    @Override
    public CompletableFuture<DeleteDomainResponse> deleteDomain(DeleteDomainRequest deleteDomainRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDomainRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDomain");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteDomainResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteDomainResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteDomainResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteDomainRequest, DeleteDomainResponse>()
                            .withOperationName("DeleteDomain").withMarshaller(new DeleteDomainRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteDomainRequest));
            CompletableFuture<DeleteDomainResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Allows the destination Amazon OpenSearch Service domain owner to delete an existing inbound cross-cluster search
     * connection. For more information, see <a
     * href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/cross-cluster-search.html"
     * >Cross-cluster search for Amazon OpenSearch Service</a>.
     * </p>
     *
     * @param deleteInboundConnectionRequest
     *        Container for the parameters to the <code>DeleteInboundConnection</code> operation.
     * @return A Java Future containing the result of the DeleteInboundConnection operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException An exception for accessing or deleting a resource that does not exist.
     *         Gives http status code of 400.</li>
     *         <li>DisabledOperationException An error occured because the client wanted to access a not supported
     *         operation. Gives http status code of 409.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OpenSearchAsyncClient.DeleteInboundConnection
     */
    @Override
    public CompletableFuture<DeleteInboundConnectionResponse> deleteInboundConnection(
            DeleteInboundConnectionRequest deleteInboundConnectionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteInboundConnectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteInboundConnection");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteInboundConnectionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteInboundConnectionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteInboundConnectionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteInboundConnectionRequest, DeleteInboundConnectionResponse>()
                            .withOperationName("DeleteInboundConnection")
                            .withMarshaller(new DeleteInboundConnectionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteInboundConnectionRequest));
            CompletableFuture<DeleteInboundConnectionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Allows the source Amazon OpenSearch Service domain owner to delete an existing outbound cross-cluster search
     * connection. For more information, see <a
     * href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/cross-cluster-search.html"
     * >Cross-cluster search for Amazon OpenSearch Service</a>.
     * </p>
     *
     * @param deleteOutboundConnectionRequest
     *        Container for the parameters to the <code>DeleteOutboundConnection</code> operation.
     * @return A Java Future containing the result of the DeleteOutboundConnection operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException An exception for accessing or deleting a resource that does not exist.
     *         Gives http status code of 400.</li>
     *         <li>DisabledOperationException An error occured because the client wanted to access a not supported
     *         operation. Gives http status code of 409.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OpenSearchAsyncClient.DeleteOutboundConnection
     */
    @Override
    public CompletableFuture<DeleteOutboundConnectionResponse> deleteOutboundConnection(
            DeleteOutboundConnectionRequest deleteOutboundConnectionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteOutboundConnectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteOutboundConnection");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteOutboundConnectionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteOutboundConnectionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteOutboundConnectionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteOutboundConnectionRequest, DeleteOutboundConnectionResponse>()
                            .withOperationName("DeleteOutboundConnection")
                            .withMarshaller(new DeleteOutboundConnectionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteOutboundConnectionRequest));
            CompletableFuture<DeleteOutboundConnectionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an Amazon OpenSearch Service package. For more information, see <a
     * href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/custom-packages.html">Custom packages
     * for Amazon OpenSearch Service</a>.
     * </p>
     *
     * @param deletePackageRequest
     *        Deletes a package from OpenSearch Service. The package can't be associated with any OpenSearch Service
     *        domain.
     * @return A Java Future containing the result of the DeletePackage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BaseException An error occurred while processing the request.</li>
     *         <li>InternalException The request processing has failed because of an unknown error, exception or failure
     *         (the failure is internal to the service) . Gives http status code of 500.</li>
     *         <li>ResourceNotFoundException An exception for accessing or deleting a resource that does not exist.
     *         Gives http status code of 400.</li>
     *         <li>AccessDeniedException An error occurred because user does not have permissions to access the
     *         resource. Returns HTTP status code 403.</li>
     *         <li>ValidationException An exception for missing / invalid input fields. Gives http status code of 400.</li>
     *         <li>ConflictException An error occurred because the client attempts to remove a resource that is
     *         currently in use. Returns HTTP status code 409.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OpenSearchAsyncClient.DeletePackage
     */
    @Override
    public CompletableFuture<DeletePackageResponse> deletePackage(DeletePackageRequest deletePackageRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deletePackageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeletePackage");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeletePackageResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeletePackageResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeletePackageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeletePackageRequest, DeletePackageResponse>()
                            .withOperationName("DeletePackage")
                            .withMarshaller(new DeletePackageRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deletePackageRequest));
            CompletableFuture<DeletePackageResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an Amazon OpenSearch Service-managed interface VPC endpoint.
     * </p>
     *
     * @param deleteVpcEndpointRequest
     * @return A Java Future containing the result of the DeleteVpcEndpoint operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException An exception for accessing or deleting a resource that does not exist.
     *         Gives http status code of 400.</li>
     *         <li>DisabledOperationException An error occured because the client wanted to access a not supported
     *         operation. Gives http status code of 409.</li>
     *         <li>InternalException The request processing has failed because of an unknown error, exception or failure
     *         (the failure is internal to the service) . Gives http status code of 500.</li>
     *         <li>BaseException An error occurred while processing the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OpenSearchAsyncClient.DeleteVpcEndpoint
     */
    @Override
    public CompletableFuture<DeleteVpcEndpointResponse> deleteVpcEndpoint(DeleteVpcEndpointRequest deleteVpcEndpointRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteVpcEndpointRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteVpcEndpoint");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteVpcEndpointResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteVpcEndpointResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteVpcEndpointResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteVpcEndpointRequest, DeleteVpcEndpointResponse>()
                            .withOperationName("DeleteVpcEndpoint")
                            .withMarshaller(new DeleteVpcEndpointRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteVpcEndpointRequest));
            CompletableFuture<DeleteVpcEndpointResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the domain configuration for the specified Amazon OpenSearch Service domain, including the domain ID,
     * domain service endpoint, and domain ARN.
     * </p>
     *
     * @param describeDomainRequest
     *        Container for the parameters to the <code>DescribeDomain</code> operation.
     * @return A Java Future containing the result of the DescribeDomain operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BaseException An error occurred while processing the request.</li>
     *         <li>InternalException The request processing has failed because of an unknown error, exception or failure
     *         (the failure is internal to the service) . Gives http status code of 500.</li>
     *         <li>ResourceNotFoundException An exception for accessing or deleting a resource that does not exist.
     *         Gives http status code of 400.</li>
     *         <li>ValidationException An exception for missing / invalid input fields. Gives http status code of 400.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OpenSearchAsyncClient.DescribeDomain
     */
    @Override
    public CompletableFuture<DescribeDomainResponse> describeDomain(DescribeDomainRequest describeDomainRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeDomainRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeDomain");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeDomainResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeDomainResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeDomainResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeDomainRequest, DescribeDomainResponse>()
                            .withOperationName("DescribeDomain")
                            .withMarshaller(new DescribeDomainRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeDomainRequest));
            CompletableFuture<DescribeDomainResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the list of optimizations that Auto-Tune has made to an Amazon OpenSearch Service domain. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/auto-tune.html">Auto-Tune for Amazon
     * OpenSearch Service</a>.
     * </p>
     *
     * @param describeDomainAutoTunesRequest
     *        Container for the parameters to the <code>DescribeDomainAutoTunes</code> operation.
     * @return A Java Future containing the result of the DescribeDomainAutoTunes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BaseException An error occurred while processing the request.</li>
     *         <li>InternalException The request processing has failed because of an unknown error, exception or failure
     *         (the failure is internal to the service) . Gives http status code of 500.</li>
     *         <li>ResourceNotFoundException An exception for accessing or deleting a resource that does not exist.
     *         Gives http status code of 400.</li>
     *         <li>ValidationException An exception for missing / invalid input fields. Gives http status code of 400.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OpenSearchAsyncClient.DescribeDomainAutoTunes
     */
    @Override
    public CompletableFuture<DescribeDomainAutoTunesResponse> describeDomainAutoTunes(
            DescribeDomainAutoTunesRequest describeDomainAutoTunesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeDomainAutoTunesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeDomainAutoTunes");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeDomainAutoTunesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeDomainAutoTunesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeDomainAutoTunesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeDomainAutoTunesRequest, DescribeDomainAutoTunesResponse>()
                            .withOperationName("DescribeDomainAutoTunes")
                            .withMarshaller(new DescribeDomainAutoTunesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeDomainAutoTunesRequest));
            CompletableFuture<DescribeDomainAutoTunesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the list of optimizations that Auto-Tune has made to an Amazon OpenSearch Service domain. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/auto-tune.html">Auto-Tune for Amazon
     * OpenSearch Service</a>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeDomainAutoTunes(software.amazon.awssdk.services.opensearch.model.DescribeDomainAutoTunesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.opensearch.paginators.DescribeDomainAutoTunesPublisher publisher = client.describeDomainAutoTunesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.opensearch.paginators.DescribeDomainAutoTunesPublisher publisher = client.describeDomainAutoTunesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.opensearch.model.DescribeDomainAutoTunesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.opensearch.model.DescribeDomainAutoTunesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeDomainAutoTunes(software.amazon.awssdk.services.opensearch.model.DescribeDomainAutoTunesRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeDomainAutoTunesRequest
     *        Container for the parameters to the <code>DescribeDomainAutoTunes</code> operation.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BaseException An error occurred while processing the request.</li>
     *         <li>InternalException The request processing has failed because of an unknown error, exception or failure
     *         (the failure is internal to the service) . Gives http status code of 500.</li>
     *         <li>ResourceNotFoundException An exception for accessing or deleting a resource that does not exist.
     *         Gives http status code of 400.</li>
     *         <li>ValidationException An exception for missing / invalid input fields. Gives http status code of 400.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OpenSearchAsyncClient.DescribeDomainAutoTunes
     */
    public DescribeDomainAutoTunesPublisher describeDomainAutoTunesPaginator(
            DescribeDomainAutoTunesRequest describeDomainAutoTunesRequest) {
        return new DescribeDomainAutoTunesPublisher(this, applyPaginatorUserAgent(describeDomainAutoTunesRequest));
    }

    /**
     * <p>
     * Returns information about the current blue/green deployment happening on an Amazon OpenSearch Service domain. For
     * more information, see <a href=
     * "https://docs.aws.amazon.com/opensearch-service/latest/developerguide/managedomains-configuration-changes.html"
     * >Making configuration changes in Amazon OpenSearch Service</a>.
     * </p>
     *
     * @param describeDomainChangeProgressRequest
     *        Container for the parameters to the <code>DescribeDomainChangeProgress</code> operation.
     * @return A Java Future containing the result of the DescribeDomainChangeProgress operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BaseException An error occurred while processing the request.</li>
     *         <li>InternalException The request processing has failed because of an unknown error, exception or failure
     *         (the failure is internal to the service) . Gives http status code of 500.</li>
     *         <li>ResourceNotFoundException An exception for accessing or deleting a resource that does not exist.
     *         Gives http status code of 400.</li>
     *         <li>ValidationException An exception for missing / invalid input fields. Gives http status code of 400.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OpenSearchAsyncClient.DescribeDomainChangeProgress
     */
    @Override
    public CompletableFuture<DescribeDomainChangeProgressResponse> describeDomainChangeProgress(
            DescribeDomainChangeProgressRequest describeDomainChangeProgressRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeDomainChangeProgressRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeDomainChangeProgress");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeDomainChangeProgressResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeDomainChangeProgressResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeDomainChangeProgressResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeDomainChangeProgressRequest, DescribeDomainChangeProgressResponse>()
                            .withOperationName("DescribeDomainChangeProgress")
                            .withMarshaller(new DescribeDomainChangeProgressRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeDomainChangeProgressRequest));
            CompletableFuture<DescribeDomainChangeProgressResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the configuration of an Amazon OpenSearch Service domain.
     * </p>
     *
     * @param describeDomainConfigRequest
     *        Container for the parameters to the <code>DescribeDomainConfig</code> operation.
     * @return A Java Future containing the result of the DescribeDomainConfig operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BaseException An error occurred while processing the request.</li>
     *         <li>InternalException The request processing has failed because of an unknown error, exception or failure
     *         (the failure is internal to the service) . Gives http status code of 500.</li>
     *         <li>ResourceNotFoundException An exception for accessing or deleting a resource that does not exist.
     *         Gives http status code of 400.</li>
     *         <li>ValidationException An exception for missing / invalid input fields. Gives http status code of 400.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OpenSearchAsyncClient.DescribeDomainConfig
     */
    @Override
    public CompletableFuture<DescribeDomainConfigResponse> describeDomainConfig(
            DescribeDomainConfigRequest describeDomainConfigRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeDomainConfigRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeDomainConfig");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeDomainConfigResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeDomainConfigResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeDomainConfigResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeDomainConfigRequest, DescribeDomainConfigResponse>()
                            .withOperationName("DescribeDomainConfig")
                            .withMarshaller(new DescribeDomainConfigRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeDomainConfigRequest));
            CompletableFuture<DescribeDomainConfigResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns domain configuration information about the specified Amazon OpenSearch Service domains.
     * </p>
     *
     * @param describeDomainsRequest
     *        Container for the parameters to the <code>DescribeDomains</code> operation.
     * @return A Java Future containing the result of the DescribeDomains operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BaseException An error occurred while processing the request.</li>
     *         <li>InternalException The request processing has failed because of an unknown error, exception or failure
     *         (the failure is internal to the service) . Gives http status code of 500.</li>
     *         <li>ValidationException An exception for missing / invalid input fields. Gives http status code of 400.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OpenSearchAsyncClient.DescribeDomains
     */
    @Override
    public CompletableFuture<DescribeDomainsResponse> describeDomains(DescribeDomainsRequest describeDomainsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeDomainsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeDomains");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeDomainsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeDomainsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeDomainsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeDomainsRequest, DescribeDomainsResponse>()
                            .withOperationName("DescribeDomains")
                            .withMarshaller(new DescribeDomainsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeDomainsRequest));
            CompletableFuture<DescribeDomainsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all the inbound cross-cluster search connections for a destination (remote) Amazon OpenSearch Service
     * domain. For more information, see <a
     * href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/cross-cluster-search.html"
     * >Cross-cluster search for Amazon OpenSearch Service</a>.
     * </p>
     *
     * @param describeInboundConnectionsRequest
     *        Container for the parameters to the <code>DescribeInboundConnections</code> operation.
     * @return A Java Future containing the result of the DescribeInboundConnections operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidPaginationTokenException The request processing has failed because of invalid pagination token
     *         provided by customer. Returns an HTTP status code of 400.</li>
     *         <li>DisabledOperationException An error occured because the client wanted to access a not supported
     *         operation. Gives http status code of 409.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OpenSearchAsyncClient.DescribeInboundConnections
     */
    @Override
    public CompletableFuture<DescribeInboundConnectionsResponse> describeInboundConnections(
            DescribeInboundConnectionsRequest describeInboundConnectionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeInboundConnectionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeInboundConnections");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeInboundConnectionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeInboundConnectionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeInboundConnectionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeInboundConnectionsRequest, DescribeInboundConnectionsResponse>()
                            .withOperationName("DescribeInboundConnections")
                            .withMarshaller(new DescribeInboundConnectionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeInboundConnectionsRequest));
            CompletableFuture<DescribeInboundConnectionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all the inbound cross-cluster search connections for a destination (remote) Amazon OpenSearch Service
     * domain. For more information, see <a
     * href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/cross-cluster-search.html"
     * >Cross-cluster search for Amazon OpenSearch Service</a>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeInboundConnections(software.amazon.awssdk.services.opensearch.model.DescribeInboundConnectionsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.opensearch.paginators.DescribeInboundConnectionsPublisher publisher = client.describeInboundConnectionsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.opensearch.paginators.DescribeInboundConnectionsPublisher publisher = client.describeInboundConnectionsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.opensearch.model.DescribeInboundConnectionsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.opensearch.model.DescribeInboundConnectionsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeInboundConnections(software.amazon.awssdk.services.opensearch.model.DescribeInboundConnectionsRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeInboundConnectionsRequest
     *        Container for the parameters to the <code>DescribeInboundConnections</code> operation.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidPaginationTokenException The request processing has failed because of invalid pagination token
     *         provided by customer. Returns an HTTP status code of 400.</li>
     *         <li>DisabledOperationException An error occured because the client wanted to access a not supported
     *         operation. Gives http status code of 409.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OpenSearchAsyncClient.DescribeInboundConnections
     */
    public DescribeInboundConnectionsPublisher describeInboundConnectionsPaginator(
            DescribeInboundConnectionsRequest describeInboundConnectionsRequest) {
        return new DescribeInboundConnectionsPublisher(this, applyPaginatorUserAgent(describeInboundConnectionsRequest));
    }

    /**
     * <p>
     * Describes the instance count, storage, and master node limits for a given OpenSearch or Elasticsearch version and
     * instance type.
     * </p>
     *
     * @param describeInstanceTypeLimitsRequest
     *        Container for the parameters to the <code>DescribeInstanceTypeLimits</code> operation.
     * @return A Java Future containing the result of the DescribeInstanceTypeLimits operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BaseException An error occurred while processing the request.</li>
     *         <li>InternalException The request processing has failed because of an unknown error, exception or failure
     *         (the failure is internal to the service) . Gives http status code of 500.</li>
     *         <li>InvalidTypeException An exception for trying to create or access sub-resource that is either invalid
     *         or not supported. Gives http status code of 409.</li>
     *         <li>LimitExceededException An exception for trying to create more than allowed resources or
     *         sub-resources. Gives http status code of 409.</li>
     *         <li>ResourceNotFoundException An exception for accessing or deleting a resource that does not exist.
     *         Gives http status code of 400.</li>
     *         <li>ValidationException An exception for missing / invalid input fields. Gives http status code of 400.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OpenSearchAsyncClient.DescribeInstanceTypeLimits
     */
    @Override
    public CompletableFuture<DescribeInstanceTypeLimitsResponse> describeInstanceTypeLimits(
            DescribeInstanceTypeLimitsRequest describeInstanceTypeLimitsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeInstanceTypeLimitsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeInstanceTypeLimits");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeInstanceTypeLimitsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeInstanceTypeLimitsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeInstanceTypeLimitsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeInstanceTypeLimitsRequest, DescribeInstanceTypeLimitsResponse>()
                            .withOperationName("DescribeInstanceTypeLimits")
                            .withMarshaller(new DescribeInstanceTypeLimitsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeInstanceTypeLimitsRequest));
            CompletableFuture<DescribeInstanceTypeLimitsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all the outbound cross-cluster connections for a local (source) Amazon OpenSearch Service domain. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/cross-cluster-search.html"
     * >Cross-cluster search for Amazon OpenSearch Service</a>.
     * </p>
     *
     * @param describeOutboundConnectionsRequest
     *        Container for the parameters to the <code>DescribeOutboundConnections</code> operation.
     * @return A Java Future containing the result of the DescribeOutboundConnections operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidPaginationTokenException The request processing has failed because of invalid pagination token
     *         provided by customer. Returns an HTTP status code of 400.</li>
     *         <li>DisabledOperationException An error occured because the client wanted to access a not supported
     *         operation. Gives http status code of 409.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OpenSearchAsyncClient.DescribeOutboundConnections
     */
    @Override
    public CompletableFuture<DescribeOutboundConnectionsResponse> describeOutboundConnections(
            DescribeOutboundConnectionsRequest describeOutboundConnectionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeOutboundConnectionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeOutboundConnections");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeOutboundConnectionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeOutboundConnectionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeOutboundConnectionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeOutboundConnectionsRequest, DescribeOutboundConnectionsResponse>()
                            .withOperationName("DescribeOutboundConnections")
                            .withMarshaller(new DescribeOutboundConnectionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeOutboundConnectionsRequest));
            CompletableFuture<DescribeOutboundConnectionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all the outbound cross-cluster connections for a local (source) Amazon OpenSearch Service domain. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/cross-cluster-search.html"
     * >Cross-cluster search for Amazon OpenSearch Service</a>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeOutboundConnections(software.amazon.awssdk.services.opensearch.model.DescribeOutboundConnectionsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.opensearch.paginators.DescribeOutboundConnectionsPublisher publisher = client.describeOutboundConnectionsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.opensearch.paginators.DescribeOutboundConnectionsPublisher publisher = client.describeOutboundConnectionsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.opensearch.model.DescribeOutboundConnectionsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.opensearch.model.DescribeOutboundConnectionsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeOutboundConnections(software.amazon.awssdk.services.opensearch.model.DescribeOutboundConnectionsRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeOutboundConnectionsRequest
     *        Container for the parameters to the <code>DescribeOutboundConnections</code> operation.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidPaginationTokenException The request processing has failed because of invalid pagination token
     *         provided by customer. Returns an HTTP status code of 400.</li>
     *         <li>DisabledOperationException An error occured because the client wanted to access a not supported
     *         operation. Gives http status code of 409.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OpenSearchAsyncClient.DescribeOutboundConnections
     */
    public DescribeOutboundConnectionsPublisher describeOutboundConnectionsPaginator(
            DescribeOutboundConnectionsRequest describeOutboundConnectionsRequest) {
        return new DescribeOutboundConnectionsPublisher(this, applyPaginatorUserAgent(describeOutboundConnectionsRequest));
    }

    /**
     * <p>
     * Describes all packages available to OpenSearch Service. For more information, see <a
     * href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/custom-packages.html">Custom packages
     * for Amazon OpenSearch Service</a>.
     * </p>
     *
     * @param describePackagesRequest
     *        Container for the request parameters to the <code>DescribePackage</code> operation.
     * @return A Java Future containing the result of the DescribePackages operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BaseException An error occurred while processing the request.</li>
     *         <li>InternalException The request processing has failed because of an unknown error, exception or failure
     *         (the failure is internal to the service) . Gives http status code of 500.</li>
     *         <li>ResourceNotFoundException An exception for accessing or deleting a resource that does not exist.
     *         Gives http status code of 400.</li>
     *         <li>AccessDeniedException An error occurred because user does not have permissions to access the
     *         resource. Returns HTTP status code 403.</li>
     *         <li>ValidationException An exception for missing / invalid input fields. Gives http status code of 400.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OpenSearchAsyncClient.DescribePackages
     */
    @Override
    public CompletableFuture<DescribePackagesResponse> describePackages(DescribePackagesRequest describePackagesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describePackagesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribePackages");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribePackagesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribePackagesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribePackagesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribePackagesRequest, DescribePackagesResponse>()
                            .withOperationName("DescribePackages")
                            .withMarshaller(new DescribePackagesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describePackagesRequest));
            CompletableFuture<DescribePackagesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes all packages available to OpenSearch Service. For more information, see <a
     * href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/custom-packages.html">Custom packages
     * for Amazon OpenSearch Service</a>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describePackages(software.amazon.awssdk.services.opensearch.model.DescribePackagesRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.opensearch.paginators.DescribePackagesPublisher publisher = client.describePackagesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.opensearch.paginators.DescribePackagesPublisher publisher = client.describePackagesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.opensearch.model.DescribePackagesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.opensearch.model.DescribePackagesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describePackages(software.amazon.awssdk.services.opensearch.model.DescribePackagesRequest)}
     * operation.</b>
     * </p>
     *
     * @param describePackagesRequest
     *        Container for the request parameters to the <code>DescribePackage</code> operation.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BaseException An error occurred while processing the request.</li>
     *         <li>InternalException The request processing has failed because of an unknown error, exception or failure
     *         (the failure is internal to the service) . Gives http status code of 500.</li>
     *         <li>ResourceNotFoundException An exception for accessing or deleting a resource that does not exist.
     *         Gives http status code of 400.</li>
     *         <li>AccessDeniedException An error occurred because user does not have permissions to access the
     *         resource. Returns HTTP status code 403.</li>
     *         <li>ValidationException An exception for missing / invalid input fields. Gives http status code of 400.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OpenSearchAsyncClient.DescribePackages
     */
    public DescribePackagesPublisher describePackagesPaginator(DescribePackagesRequest describePackagesRequest) {
        return new DescribePackagesPublisher(this, applyPaginatorUserAgent(describePackagesRequest));
    }

    /**
     * <p>
     * Describes the available Amazon OpenSearch Service Reserved Instance offerings for a given Region. For more
     * information, see <a href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/ri.html">Reserved
     * Instances in Amazon OpenSearch Service</a>.
     * </p>
     *
     * @param describeReservedInstanceOfferingsRequest
     *        Container for the request parameters to a <code>DescribeReservedInstanceOfferings</code> operation.
     * @return A Java Future containing the result of the DescribeReservedInstanceOfferings operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException An exception for accessing or deleting a resource that does not exist.
     *         Gives http status code of 400.</li>
     *         <li>ValidationException An exception for missing / invalid input fields. Gives http status code of 400.</li>
     *         <li>DisabledOperationException An error occured because the client wanted to access a not supported
     *         operation. Gives http status code of 409.</li>
     *         <li>InternalException The request processing has failed because of an unknown error, exception or failure
     *         (the failure is internal to the service) . Gives http status code of 500.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OpenSearchAsyncClient.DescribeReservedInstanceOfferings
     */
    @Override
    public CompletableFuture<DescribeReservedInstanceOfferingsResponse> describeReservedInstanceOfferings(
            DescribeReservedInstanceOfferingsRequest describeReservedInstanceOfferingsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeReservedInstanceOfferingsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeReservedInstanceOfferings");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeReservedInstanceOfferingsResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DescribeReservedInstanceOfferingsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeReservedInstanceOfferingsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeReservedInstanceOfferingsRequest, DescribeReservedInstanceOfferingsResponse>()
                            .withOperationName("DescribeReservedInstanceOfferings")
                            .withMarshaller(new DescribeReservedInstanceOfferingsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeReservedInstanceOfferingsRequest));
            CompletableFuture<DescribeReservedInstanceOfferingsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the available Amazon OpenSearch Service Reserved Instance offerings for a given Region. For more
     * information, see <a href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/ri.html">Reserved
     * Instances in Amazon OpenSearch Service</a>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeReservedInstanceOfferings(software.amazon.awssdk.services.opensearch.model.DescribeReservedInstanceOfferingsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.opensearch.paginators.DescribeReservedInstanceOfferingsPublisher publisher = client.describeReservedInstanceOfferingsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.opensearch.paginators.DescribeReservedInstanceOfferingsPublisher publisher = client.describeReservedInstanceOfferingsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.opensearch.model.DescribeReservedInstanceOfferingsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.opensearch.model.DescribeReservedInstanceOfferingsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeReservedInstanceOfferings(software.amazon.awssdk.services.opensearch.model.DescribeReservedInstanceOfferingsRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeReservedInstanceOfferingsRequest
     *        Container for the request parameters to a <code>DescribeReservedInstanceOfferings</code> operation.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException An exception for accessing or deleting a resource that does not exist.
     *         Gives http status code of 400.</li>
     *         <li>ValidationException An exception for missing / invalid input fields. Gives http status code of 400.</li>
     *         <li>DisabledOperationException An error occured because the client wanted to access a not supported
     *         operation. Gives http status code of 409.</li>
     *         <li>InternalException The request processing has failed because of an unknown error, exception or failure
     *         (the failure is internal to the service) . Gives http status code of 500.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OpenSearchAsyncClient.DescribeReservedInstanceOfferings
     */
    public DescribeReservedInstanceOfferingsPublisher describeReservedInstanceOfferingsPaginator(
            DescribeReservedInstanceOfferingsRequest describeReservedInstanceOfferingsRequest) {
        return new DescribeReservedInstanceOfferingsPublisher(this,
                applyPaginatorUserAgent(describeReservedInstanceOfferingsRequest));
    }

    /**
     * <p>
     * Describes the Amazon OpenSearch Service instances that you have reserved in a given Region. For more information,
     * see <a href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/ri.html">Reserved Instances in
     * Amazon OpenSearch Service</a>.
     * </p>
     *
     * @param describeReservedInstancesRequest
     *        Container for the request parameters to the <code>DescribeReservedInstances</code> operation.
     * @return A Java Future containing the result of the DescribeReservedInstances operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException An exception for accessing or deleting a resource that does not exist.
     *         Gives http status code of 400.</li>
     *         <li>InternalException The request processing has failed because of an unknown error, exception or failure
     *         (the failure is internal to the service) . Gives http status code of 500.</li>
     *         <li>ValidationException An exception for missing / invalid input fields. Gives http status code of 400.</li>
     *         <li>DisabledOperationException An error occured because the client wanted to access a not supported
     *         operation. Gives http status code of 409.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OpenSearchAsyncClient.DescribeReservedInstances
     */
    @Override
    public CompletableFuture<DescribeReservedInstancesResponse> describeReservedInstances(
            DescribeReservedInstancesRequest describeReservedInstancesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeReservedInstancesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeReservedInstances");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeReservedInstancesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeReservedInstancesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeReservedInstancesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeReservedInstancesRequest, DescribeReservedInstancesResponse>()
                            .withOperationName("DescribeReservedInstances")
                            .withMarshaller(new DescribeReservedInstancesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeReservedInstancesRequest));
            CompletableFuture<DescribeReservedInstancesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the Amazon OpenSearch Service instances that you have reserved in a given Region. For more information,
     * see <a href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/ri.html">Reserved Instances in
     * Amazon OpenSearch Service</a>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeReservedInstances(software.amazon.awssdk.services.opensearch.model.DescribeReservedInstancesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.opensearch.paginators.DescribeReservedInstancesPublisher publisher = client.describeReservedInstancesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.opensearch.paginators.DescribeReservedInstancesPublisher publisher = client.describeReservedInstancesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.opensearch.model.DescribeReservedInstancesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.opensearch.model.DescribeReservedInstancesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeReservedInstances(software.amazon.awssdk.services.opensearch.model.DescribeReservedInstancesRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeReservedInstancesRequest
     *        Container for the request parameters to the <code>DescribeReservedInstances</code> operation.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException An exception for accessing or deleting a resource that does not exist.
     *         Gives http status code of 400.</li>
     *         <li>InternalException The request processing has failed because of an unknown error, exception or failure
     *         (the failure is internal to the service) . Gives http status code of 500.</li>
     *         <li>ValidationException An exception for missing / invalid input fields. Gives http status code of 400.</li>
     *         <li>DisabledOperationException An error occured because the client wanted to access a not supported
     *         operation. Gives http status code of 409.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OpenSearchAsyncClient.DescribeReservedInstances
     */
    public DescribeReservedInstancesPublisher describeReservedInstancesPaginator(
            DescribeReservedInstancesRequest describeReservedInstancesRequest) {
        return new DescribeReservedInstancesPublisher(this, applyPaginatorUserAgent(describeReservedInstancesRequest));
    }

    /**
     * <p>
     * Describes one or more Amazon OpenSearch Service-managed VPC endpoints.
     * </p>
     *
     * @param describeVpcEndpointsRequest
     * @return A Java Future containing the result of the DescribeVpcEndpoints operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException An exception for missing / invalid input fields. Gives http status code of 400.</li>
     *         <li>InternalException The request processing has failed because of an unknown error, exception or failure
     *         (the failure is internal to the service) . Gives http status code of 500.</li>
     *         <li>DisabledOperationException An error occured because the client wanted to access a not supported
     *         operation. Gives http status code of 409.</li>
     *         <li>BaseException An error occurred while processing the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OpenSearchAsyncClient.DescribeVpcEndpoints
     */
    @Override
    public CompletableFuture<DescribeVpcEndpointsResponse> describeVpcEndpoints(
            DescribeVpcEndpointsRequest describeVpcEndpointsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeVpcEndpointsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeVpcEndpoints");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeVpcEndpointsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeVpcEndpointsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeVpcEndpointsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeVpcEndpointsRequest, DescribeVpcEndpointsResponse>()
                            .withOperationName("DescribeVpcEndpoints")
                            .withMarshaller(new DescribeVpcEndpointsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeVpcEndpointsRequest));
            CompletableFuture<DescribeVpcEndpointsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes a package from the specified Amazon OpenSearch Service domain. The package can't be in use with any
     * OpenSearch index for the dissociation to succeed. The package is still available in OpenSearch Service for
     * association later. For more information, see <a
     * href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/custom-packages.html">Custom packages
     * for Amazon OpenSearch Service</a>.
     * </p>
     *
     * @param dissociatePackageRequest
     *        Container for the request parameters to the <code>DissociatePackage</code> operation.
     * @return A Java Future containing the result of the DissociatePackage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BaseException An error occurred while processing the request.</li>
     *         <li>InternalException The request processing has failed because of an unknown error, exception or failure
     *         (the failure is internal to the service) . Gives http status code of 500.</li>
     *         <li>ResourceNotFoundException An exception for accessing or deleting a resource that does not exist.
     *         Gives http status code of 400.</li>
     *         <li>AccessDeniedException An error occurred because user does not have permissions to access the
     *         resource. Returns HTTP status code 403.</li>
     *         <li>ValidationException An exception for missing / invalid input fields. Gives http status code of 400.</li>
     *         <li>ConflictException An error occurred because the client attempts to remove a resource that is
     *         currently in use. Returns HTTP status code 409.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OpenSearchAsyncClient.DissociatePackage
     */
    @Override
    public CompletableFuture<DissociatePackageResponse> dissociatePackage(DissociatePackageRequest dissociatePackageRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, dissociatePackageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DissociatePackage");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DissociatePackageResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DissociatePackageResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DissociatePackageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DissociatePackageRequest, DissociatePackageResponse>()
                            .withOperationName("DissociatePackage")
                            .withMarshaller(new DissociatePackageRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(dissociatePackageRequest));
            CompletableFuture<DissociatePackageResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a map of OpenSearch or Elasticsearch versions and the versions you can upgrade them to.
     * </p>
     *
     * @param getCompatibleVersionsRequest
     *        Container for the request parameters to <code>GetCompatibleVersions</code> operation.
     * @return A Java Future containing the result of the GetCompatibleVersions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BaseException An error occurred while processing the request.</li>
     *         <li>ResourceNotFoundException An exception for accessing or deleting a resource that does not exist.
     *         Gives http status code of 400.</li>
     *         <li>DisabledOperationException An error occured because the client wanted to access a not supported
     *         operation. Gives http status code of 409.</li>
     *         <li>ValidationException An exception for missing / invalid input fields. Gives http status code of 400.</li>
     *         <li>InternalException The request processing has failed because of an unknown error, exception or failure
     *         (the failure is internal to the service) . Gives http status code of 500.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OpenSearchAsyncClient.GetCompatibleVersions
     */
    @Override
    public CompletableFuture<GetCompatibleVersionsResponse> getCompatibleVersions(
            GetCompatibleVersionsRequest getCompatibleVersionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getCompatibleVersionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetCompatibleVersions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetCompatibleVersionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetCompatibleVersionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetCompatibleVersionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetCompatibleVersionsRequest, GetCompatibleVersionsResponse>()
                            .withOperationName("GetCompatibleVersions")
                            .withMarshaller(new GetCompatibleVersionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getCompatibleVersionsRequest));
            CompletableFuture<GetCompatibleVersionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of Amazon OpenSearch Service package versions, along with their creation time and commit message.
     * For more information, see <a
     * href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/custom-packages.html">Custom packages
     * for Amazon OpenSearch Service</a>.
     * </p>
     *
     * @param getPackageVersionHistoryRequest
     *        Container for the request parameters to the <code>GetPackageVersionHistory</code> operation.
     * @return A Java Future containing the result of the GetPackageVersionHistory operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BaseException An error occurred while processing the request.</li>
     *         <li>InternalException The request processing has failed because of an unknown error, exception or failure
     *         (the failure is internal to the service) . Gives http status code of 500.</li>
     *         <li>ResourceNotFoundException An exception for accessing or deleting a resource that does not exist.
     *         Gives http status code of 400.</li>
     *         <li>AccessDeniedException An error occurred because user does not have permissions to access the
     *         resource. Returns HTTP status code 403.</li>
     *         <li>ValidationException An exception for missing / invalid input fields. Gives http status code of 400.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OpenSearchAsyncClient.GetPackageVersionHistory
     */
    @Override
    public CompletableFuture<GetPackageVersionHistoryResponse> getPackageVersionHistory(
            GetPackageVersionHistoryRequest getPackageVersionHistoryRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPackageVersionHistoryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPackageVersionHistory");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetPackageVersionHistoryResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetPackageVersionHistoryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetPackageVersionHistoryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetPackageVersionHistoryRequest, GetPackageVersionHistoryResponse>()
                            .withOperationName("GetPackageVersionHistory")
                            .withMarshaller(new GetPackageVersionHistoryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getPackageVersionHistoryRequest));
            CompletableFuture<GetPackageVersionHistoryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of Amazon OpenSearch Service package versions, along with their creation time and commit message.
     * For more information, see <a
     * href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/custom-packages.html">Custom packages
     * for Amazon OpenSearch Service</a>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #getPackageVersionHistory(software.amazon.awssdk.services.opensearch.model.GetPackageVersionHistoryRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.opensearch.paginators.GetPackageVersionHistoryPublisher publisher = client.getPackageVersionHistoryPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.opensearch.paginators.GetPackageVersionHistoryPublisher publisher = client.getPackageVersionHistoryPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.opensearch.model.GetPackageVersionHistoryResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.opensearch.model.GetPackageVersionHistoryResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getPackageVersionHistory(software.amazon.awssdk.services.opensearch.model.GetPackageVersionHistoryRequest)}
     * operation.</b>
     * </p>
     *
     * @param getPackageVersionHistoryRequest
     *        Container for the request parameters to the <code>GetPackageVersionHistory</code> operation.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BaseException An error occurred while processing the request.</li>
     *         <li>InternalException The request processing has failed because of an unknown error, exception or failure
     *         (the failure is internal to the service) . Gives http status code of 500.</li>
     *         <li>ResourceNotFoundException An exception for accessing or deleting a resource that does not exist.
     *         Gives http status code of 400.</li>
     *         <li>AccessDeniedException An error occurred because user does not have permissions to access the
     *         resource. Returns HTTP status code 403.</li>
     *         <li>ValidationException An exception for missing / invalid input fields. Gives http status code of 400.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OpenSearchAsyncClient.GetPackageVersionHistory
     */
    public GetPackageVersionHistoryPublisher getPackageVersionHistoryPaginator(
            GetPackageVersionHistoryRequest getPackageVersionHistoryRequest) {
        return new GetPackageVersionHistoryPublisher(this, applyPaginatorUserAgent(getPackageVersionHistoryRequest));
    }

    /**
     * <p>
     * Retrieves the complete history of the last 10 upgrades performed on an Amazon OpenSearch Service domain.
     * </p>
     *
     * @param getUpgradeHistoryRequest
     *        Container for the request parameters to the <code>GetUpgradeHistory</code> operation.
     * @return A Java Future containing the result of the GetUpgradeHistory operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BaseException An error occurred while processing the request.</li>
     *         <li>ResourceNotFoundException An exception for accessing or deleting a resource that does not exist.
     *         Gives http status code of 400.</li>
     *         <li>DisabledOperationException An error occured because the client wanted to access a not supported
     *         operation. Gives http status code of 409.</li>
     *         <li>ValidationException An exception for missing / invalid input fields. Gives http status code of 400.</li>
     *         <li>InternalException The request processing has failed because of an unknown error, exception or failure
     *         (the failure is internal to the service) . Gives http status code of 500.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OpenSearchAsyncClient.GetUpgradeHistory
     */
    @Override
    public CompletableFuture<GetUpgradeHistoryResponse> getUpgradeHistory(GetUpgradeHistoryRequest getUpgradeHistoryRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getUpgradeHistoryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetUpgradeHistory");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetUpgradeHistoryResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetUpgradeHistoryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetUpgradeHistoryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetUpgradeHistoryRequest, GetUpgradeHistoryResponse>()
                            .withOperationName("GetUpgradeHistory")
                            .withMarshaller(new GetUpgradeHistoryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getUpgradeHistoryRequest));
            CompletableFuture<GetUpgradeHistoryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the complete history of the last 10 upgrades performed on an Amazon OpenSearch Service domain.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #getUpgradeHistory(software.amazon.awssdk.services.opensearch.model.GetUpgradeHistoryRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.opensearch.paginators.GetUpgradeHistoryPublisher publisher = client.getUpgradeHistoryPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.opensearch.paginators.GetUpgradeHistoryPublisher publisher = client.getUpgradeHistoryPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.opensearch.model.GetUpgradeHistoryResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.opensearch.model.GetUpgradeHistoryResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getUpgradeHistory(software.amazon.awssdk.services.opensearch.model.GetUpgradeHistoryRequest)}
     * operation.</b>
     * </p>
     *
     * @param getUpgradeHistoryRequest
     *        Container for the request parameters to the <code>GetUpgradeHistory</code> operation.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BaseException An error occurred while processing the request.</li>
     *         <li>ResourceNotFoundException An exception for accessing or deleting a resource that does not exist.
     *         Gives http status code of 400.</li>
     *         <li>DisabledOperationException An error occured because the client wanted to access a not supported
     *         operation. Gives http status code of 409.</li>
     *         <li>ValidationException An exception for missing / invalid input fields. Gives http status code of 400.</li>
     *         <li>InternalException The request processing has failed because of an unknown error, exception or failure
     *         (the failure is internal to the service) . Gives http status code of 500.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OpenSearchAsyncClient.GetUpgradeHistory
     */
    public GetUpgradeHistoryPublisher getUpgradeHistoryPaginator(GetUpgradeHistoryRequest getUpgradeHistoryRequest) {
        return new GetUpgradeHistoryPublisher(this, applyPaginatorUserAgent(getUpgradeHistoryRequest));
    }

    /**
     * <p>
     * Returns the most recent status of the last upgrade or upgrade eligibility check performed on an Amazon OpenSearch
     * Service domain.
     * </p>
     *
     * @param getUpgradeStatusRequest
     *        Container for the request parameters to the <code>GetUpgradeStatus</code> operation.
     * @return A Java Future containing the result of the GetUpgradeStatus operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BaseException An error occurred while processing the request.</li>
     *         <li>ResourceNotFoundException An exception for accessing or deleting a resource that does not exist.
     *         Gives http status code of 400.</li>
     *         <li>DisabledOperationException An error occured because the client wanted to access a not supported
     *         operation. Gives http status code of 409.</li>
     *         <li>ValidationException An exception for missing / invalid input fields. Gives http status code of 400.</li>
     *         <li>InternalException The request processing has failed because of an unknown error, exception or failure
     *         (the failure is internal to the service) . Gives http status code of 500.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OpenSearchAsyncClient.GetUpgradeStatus
     */
    @Override
    public CompletableFuture<GetUpgradeStatusResponse> getUpgradeStatus(GetUpgradeStatusRequest getUpgradeStatusRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getUpgradeStatusRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetUpgradeStatus");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetUpgradeStatusResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetUpgradeStatusResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetUpgradeStatusResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetUpgradeStatusRequest, GetUpgradeStatusResponse>()
                            .withOperationName("GetUpgradeStatus")
                            .withMarshaller(new GetUpgradeStatusRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getUpgradeStatusRequest));
            CompletableFuture<GetUpgradeStatusResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the names of all Amazon OpenSearch Service domains owned by the current user in the active Region.
     * </p>
     *
     * @param listDomainNamesRequest
     *        Container for the parameters to the <code>ListDomainNames</code> operation.
     * @return A Java Future containing the result of the ListDomainNames operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BaseException An error occurred while processing the request.</li>
     *         <li>ValidationException An exception for missing / invalid input fields. Gives http status code of 400.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OpenSearchAsyncClient.ListDomainNames
     */
    @Override
    public CompletableFuture<ListDomainNamesResponse> listDomainNames(ListDomainNamesRequest listDomainNamesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDomainNamesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDomainNames");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDomainNamesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListDomainNamesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListDomainNamesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDomainNamesRequest, ListDomainNamesResponse>()
                            .withOperationName("ListDomainNames")
                            .withMarshaller(new ListDomainNamesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listDomainNamesRequest));
            CompletableFuture<ListDomainNamesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all Amazon OpenSearch Service domains associated with a given package. For more information, see <a
     * href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/custom-packages.html">Custom packages
     * for Amazon OpenSearch Service</a>.
     * </p>
     *
     * @param listDomainsForPackageRequest
     *        Container for the request parameters to the <code>ListDomainsForPackage</code> operation.
     * @return A Java Future containing the result of the ListDomainsForPackage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BaseException An error occurred while processing the request.</li>
     *         <li>InternalException The request processing has failed because of an unknown error, exception or failure
     *         (the failure is internal to the service) . Gives http status code of 500.</li>
     *         <li>ResourceNotFoundException An exception for accessing or deleting a resource that does not exist.
     *         Gives http status code of 400.</li>
     *         <li>AccessDeniedException An error occurred because user does not have permissions to access the
     *         resource. Returns HTTP status code 403.</li>
     *         <li>ValidationException An exception for missing / invalid input fields. Gives http status code of 400.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OpenSearchAsyncClient.ListDomainsForPackage
     */
    @Override
    public CompletableFuture<ListDomainsForPackageResponse> listDomainsForPackage(
            ListDomainsForPackageRequest listDomainsForPackageRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDomainsForPackageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDomainsForPackage");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDomainsForPackageResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListDomainsForPackageResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListDomainsForPackageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDomainsForPackageRequest, ListDomainsForPackageResponse>()
                            .withOperationName("ListDomainsForPackage")
                            .withMarshaller(new ListDomainsForPackageRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listDomainsForPackageRequest));
            CompletableFuture<ListDomainsForPackageResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all Amazon OpenSearch Service domains associated with a given package. For more information, see <a
     * href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/custom-packages.html">Custom packages
     * for Amazon OpenSearch Service</a>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listDomainsForPackage(software.amazon.awssdk.services.opensearch.model.ListDomainsForPackageRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.opensearch.paginators.ListDomainsForPackagePublisher publisher = client.listDomainsForPackagePaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.opensearch.paginators.ListDomainsForPackagePublisher publisher = client.listDomainsForPackagePaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.opensearch.model.ListDomainsForPackageResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.opensearch.model.ListDomainsForPackageResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listDomainsForPackage(software.amazon.awssdk.services.opensearch.model.ListDomainsForPackageRequest)}
     * operation.</b>
     * </p>
     *
     * @param listDomainsForPackageRequest
     *        Container for the request parameters to the <code>ListDomainsForPackage</code> operation.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BaseException An error occurred while processing the request.</li>
     *         <li>InternalException The request processing has failed because of an unknown error, exception or failure
     *         (the failure is internal to the service) . Gives http status code of 500.</li>
     *         <li>ResourceNotFoundException An exception for accessing or deleting a resource that does not exist.
     *         Gives http status code of 400.</li>
     *         <li>AccessDeniedException An error occurred because user does not have permissions to access the
     *         resource. Returns HTTP status code 403.</li>
     *         <li>ValidationException An exception for missing / invalid input fields. Gives http status code of 400.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OpenSearchAsyncClient.ListDomainsForPackage
     */
    public ListDomainsForPackagePublisher listDomainsForPackagePaginator(ListDomainsForPackageRequest listDomainsForPackageRequest) {
        return new ListDomainsForPackagePublisher(this, applyPaginatorUserAgent(listDomainsForPackageRequest));
    }

    /**
     * <p>
     * Lists all instance types and available features for a given OpenSearch or Elasticsearch version.
     * </p>
     *
     * @param listInstanceTypeDetailsRequest
     * @return A Java Future containing the result of the ListInstanceTypeDetails operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BaseException An error occurred while processing the request.</li>
     *         <li>InternalException The request processing has failed because of an unknown error, exception or failure
     *         (the failure is internal to the service) . Gives http status code of 500.</li>
     *         <li>ResourceNotFoundException An exception for accessing or deleting a resource that does not exist.
     *         Gives http status code of 400.</li>
     *         <li>ValidationException An exception for missing / invalid input fields. Gives http status code of 400.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OpenSearchAsyncClient.ListInstanceTypeDetails
     */
    @Override
    public CompletableFuture<ListInstanceTypeDetailsResponse> listInstanceTypeDetails(
            ListInstanceTypeDetailsRequest listInstanceTypeDetailsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listInstanceTypeDetailsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListInstanceTypeDetails");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListInstanceTypeDetailsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListInstanceTypeDetailsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListInstanceTypeDetailsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListInstanceTypeDetailsRequest, ListInstanceTypeDetailsResponse>()
                            .withOperationName("ListInstanceTypeDetails")
                            .withMarshaller(new ListInstanceTypeDetailsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listInstanceTypeDetailsRequest));
            CompletableFuture<ListInstanceTypeDetailsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all instance types and available features for a given OpenSearch or Elasticsearch version.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listInstanceTypeDetails(software.amazon.awssdk.services.opensearch.model.ListInstanceTypeDetailsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.opensearch.paginators.ListInstanceTypeDetailsPublisher publisher = client.listInstanceTypeDetailsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.opensearch.paginators.ListInstanceTypeDetailsPublisher publisher = client.listInstanceTypeDetailsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.opensearch.model.ListInstanceTypeDetailsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.opensearch.model.ListInstanceTypeDetailsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listInstanceTypeDetails(software.amazon.awssdk.services.opensearch.model.ListInstanceTypeDetailsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listInstanceTypeDetailsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BaseException An error occurred while processing the request.</li>
     *         <li>InternalException The request processing has failed because of an unknown error, exception or failure
     *         (the failure is internal to the service) . Gives http status code of 500.</li>
     *         <li>ResourceNotFoundException An exception for accessing or deleting a resource that does not exist.
     *         Gives http status code of 400.</li>
     *         <li>ValidationException An exception for missing / invalid input fields. Gives http status code of 400.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OpenSearchAsyncClient.ListInstanceTypeDetails
     */
    public ListInstanceTypeDetailsPublisher listInstanceTypeDetailsPaginator(
            ListInstanceTypeDetailsRequest listInstanceTypeDetailsRequest) {
        return new ListInstanceTypeDetailsPublisher(this, applyPaginatorUserAgent(listInstanceTypeDetailsRequest));
    }

    /**
     * <p>
     * Lists all packages associated with an Amazon OpenSearch Service domain. For more information, see <a
     * href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/custom-packages.html">Custom packages
     * for Amazon OpenSearch Service</a>.
     * </p>
     *
     * @param listPackagesForDomainRequest
     *        Container for the request parameters to the <code>ListPackagesForDomain</code> operation.
     * @return A Java Future containing the result of the ListPackagesForDomain operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BaseException An error occurred while processing the request.</li>
     *         <li>InternalException The request processing has failed because of an unknown error, exception or failure
     *         (the failure is internal to the service) . Gives http status code of 500.</li>
     *         <li>ResourceNotFoundException An exception for accessing or deleting a resource that does not exist.
     *         Gives http status code of 400.</li>
     *         <li>AccessDeniedException An error occurred because user does not have permissions to access the
     *         resource. Returns HTTP status code 403.</li>
     *         <li>ValidationException An exception for missing / invalid input fields. Gives http status code of 400.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OpenSearchAsyncClient.ListPackagesForDomain
     */
    @Override
    public CompletableFuture<ListPackagesForDomainResponse> listPackagesForDomain(
            ListPackagesForDomainRequest listPackagesForDomainRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPackagesForDomainRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPackagesForDomain");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListPackagesForDomainResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListPackagesForDomainResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListPackagesForDomainResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPackagesForDomainRequest, ListPackagesForDomainResponse>()
                            .withOperationName("ListPackagesForDomain")
                            .withMarshaller(new ListPackagesForDomainRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listPackagesForDomainRequest));
            CompletableFuture<ListPackagesForDomainResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all packages associated with an Amazon OpenSearch Service domain. For more information, see <a
     * href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/custom-packages.html">Custom packages
     * for Amazon OpenSearch Service</a>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listPackagesForDomain(software.amazon.awssdk.services.opensearch.model.ListPackagesForDomainRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.opensearch.paginators.ListPackagesForDomainPublisher publisher = client.listPackagesForDomainPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.opensearch.paginators.ListPackagesForDomainPublisher publisher = client.listPackagesForDomainPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.opensearch.model.ListPackagesForDomainResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.opensearch.model.ListPackagesForDomainResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listPackagesForDomain(software.amazon.awssdk.services.opensearch.model.ListPackagesForDomainRequest)}
     * operation.</b>
     * </p>
     *
     * @param listPackagesForDomainRequest
     *        Container for the request parameters to the <code>ListPackagesForDomain</code> operation.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BaseException An error occurred while processing the request.</li>
     *         <li>InternalException The request processing has failed because of an unknown error, exception or failure
     *         (the failure is internal to the service) . Gives http status code of 500.</li>
     *         <li>ResourceNotFoundException An exception for accessing or deleting a resource that does not exist.
     *         Gives http status code of 400.</li>
     *         <li>AccessDeniedException An error occurred because user does not have permissions to access the
     *         resource. Returns HTTP status code 403.</li>
     *         <li>ValidationException An exception for missing / invalid input fields. Gives http status code of 400.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OpenSearchAsyncClient.ListPackagesForDomain
     */
    public ListPackagesForDomainPublisher listPackagesForDomainPaginator(ListPackagesForDomainRequest listPackagesForDomainRequest) {
        return new ListPackagesForDomainPublisher(this, applyPaginatorUserAgent(listPackagesForDomainRequest));
    }

    /**
     * <p>
     * Returns all resource tags for an Amazon OpenSearch Service domain. For more information, see <a href=
     * "https://docs.aws.amazon.com/opensearch-service/latest/developerguide/managedomains-awsresourcetagging.html"
     * >Tagging Amazon OpenSearch Service domains</a>.
     * </p>
     *
     * @param listTagsRequest
     *        Container for the parameters to the <code>ListTags</code> operation.
     * @return A Java Future containing the result of the ListTags operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BaseException An error occurred while processing the request.</li>
     *         <li>ResourceNotFoundException An exception for accessing or deleting a resource that does not exist.
     *         Gives http status code of 400.</li>
     *         <li>ValidationException An exception for missing / invalid input fields. Gives http status code of 400.</li>
     *         <li>InternalException The request processing has failed because of an unknown error, exception or failure
     *         (the failure is internal to the service) . Gives http status code of 500.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OpenSearchAsyncClient.ListTags
     */
    @Override
    public CompletableFuture<ListTagsResponse> listTags(ListTagsRequest listTagsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTags");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListTagsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsRequest, ListTagsResponse>().withOperationName("ListTags")
                            .withMarshaller(new ListTagsRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsRequest));
            CompletableFuture<ListTagsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all versions of OpenSearch and Elasticsearch that Amazon OpenSearch Service supports.
     * </p>
     *
     * @param listVersionsRequest
     *        Container for the request parameters to the <code>ListVersions</code> operation.
     * @return A Java Future containing the result of the ListVersions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BaseException An error occurred while processing the request.</li>
     *         <li>InternalException The request processing has failed because of an unknown error, exception or failure
     *         (the failure is internal to the service) . Gives http status code of 500.</li>
     *         <li>ResourceNotFoundException An exception for accessing or deleting a resource that does not exist.
     *         Gives http status code of 400.</li>
     *         <li>ValidationException An exception for missing / invalid input fields. Gives http status code of 400.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OpenSearchAsyncClient.ListVersions
     */
    @Override
    public CompletableFuture<ListVersionsResponse> listVersions(ListVersionsRequest listVersionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listVersionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListVersions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListVersionsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListVersionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListVersionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListVersionsRequest, ListVersionsResponse>()
                            .withOperationName("ListVersions").withMarshaller(new ListVersionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listVersionsRequest));
            CompletableFuture<ListVersionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all versions of OpenSearch and Elasticsearch that Amazon OpenSearch Service supports.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listVersions(software.amazon.awssdk.services.opensearch.model.ListVersionsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.opensearch.paginators.ListVersionsPublisher publisher = client.listVersionsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.opensearch.paginators.ListVersionsPublisher publisher = client.listVersionsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.opensearch.model.ListVersionsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.opensearch.model.ListVersionsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listVersions(software.amazon.awssdk.services.opensearch.model.ListVersionsRequest)} operation.</b>
     * </p>
     *
     * @param listVersionsRequest
     *        Container for the request parameters to the <code>ListVersions</code> operation.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BaseException An error occurred while processing the request.</li>
     *         <li>InternalException The request processing has failed because of an unknown error, exception or failure
     *         (the failure is internal to the service) . Gives http status code of 500.</li>
     *         <li>ResourceNotFoundException An exception for accessing or deleting a resource that does not exist.
     *         Gives http status code of 400.</li>
     *         <li>ValidationException An exception for missing / invalid input fields. Gives http status code of 400.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OpenSearchAsyncClient.ListVersions
     */
    public ListVersionsPublisher listVersionsPaginator(ListVersionsRequest listVersionsRequest) {
        return new ListVersionsPublisher(this, applyPaginatorUserAgent(listVersionsRequest));
    }

    /**
     * <p>
     * Retrieves information about each Amazon Web Services principal that is allowed to access a given Amazon
     * OpenSearch Service domain through the use of an interface VPC endpoint.
     * </p>
     *
     * @param listVpcEndpointAccessRequest
     * @return A Java Future containing the result of the ListVpcEndpointAccess operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException An exception for accessing or deleting a resource that does not exist.
     *         Gives http status code of 400.</li>
     *         <li>DisabledOperationException An error occured because the client wanted to access a not supported
     *         operation. Gives http status code of 409.</li>
     *         <li>InternalException The request processing has failed because of an unknown error, exception or failure
     *         (the failure is internal to the service) . Gives http status code of 500.</li>
     *         <li>BaseException An error occurred while processing the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OpenSearchAsyncClient.ListVpcEndpointAccess
     */
    @Override
    public CompletableFuture<ListVpcEndpointAccessResponse> listVpcEndpointAccess(
            ListVpcEndpointAccessRequest listVpcEndpointAccessRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listVpcEndpointAccessRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListVpcEndpointAccess");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListVpcEndpointAccessResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListVpcEndpointAccessResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListVpcEndpointAccessResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListVpcEndpointAccessRequest, ListVpcEndpointAccessResponse>()
                            .withOperationName("ListVpcEndpointAccess")
                            .withMarshaller(new ListVpcEndpointAccessRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listVpcEndpointAccessRequest));
            CompletableFuture<ListVpcEndpointAccessResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves all Amazon OpenSearch Service-managed VPC endpoints in the current Amazon Web Services account and
     * Region.
     * </p>
     *
     * @param listVpcEndpointsRequest
     * @return A Java Future containing the result of the ListVpcEndpoints operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException The request processing has failed because of an unknown error, exception or failure
     *         (the failure is internal to the service) . Gives http status code of 500.</li>
     *         <li>DisabledOperationException An error occured because the client wanted to access a not supported
     *         operation. Gives http status code of 409.</li>
     *         <li>BaseException An error occurred while processing the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OpenSearchAsyncClient.ListVpcEndpoints
     */
    @Override
    public CompletableFuture<ListVpcEndpointsResponse> listVpcEndpoints(ListVpcEndpointsRequest listVpcEndpointsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listVpcEndpointsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListVpcEndpoints");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListVpcEndpointsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListVpcEndpointsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListVpcEndpointsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListVpcEndpointsRequest, ListVpcEndpointsResponse>()
                            .withOperationName("ListVpcEndpoints")
                            .withMarshaller(new ListVpcEndpointsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listVpcEndpointsRequest));
            CompletableFuture<ListVpcEndpointsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves all Amazon OpenSearch Service-managed VPC endpoints associated with a particular domain.
     * </p>
     *
     * @param listVpcEndpointsForDomainRequest
     * @return A Java Future containing the result of the ListVpcEndpointsForDomain operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException The request processing has failed because of an unknown error, exception or failure
     *         (the failure is internal to the service) . Gives http status code of 500.</li>
     *         <li>DisabledOperationException An error occured because the client wanted to access a not supported
     *         operation. Gives http status code of 409.</li>
     *         <li>ResourceNotFoundException An exception for accessing or deleting a resource that does not exist.
     *         Gives http status code of 400.</li>
     *         <li>BaseException An error occurred while processing the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OpenSearchAsyncClient.ListVpcEndpointsForDomain
     */
    @Override
    public CompletableFuture<ListVpcEndpointsForDomainResponse> listVpcEndpointsForDomain(
            ListVpcEndpointsForDomainRequest listVpcEndpointsForDomainRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listVpcEndpointsForDomainRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListVpcEndpointsForDomain");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListVpcEndpointsForDomainResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListVpcEndpointsForDomainResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListVpcEndpointsForDomainResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListVpcEndpointsForDomainRequest, ListVpcEndpointsForDomainResponse>()
                            .withOperationName("ListVpcEndpointsForDomain")
                            .withMarshaller(new ListVpcEndpointsForDomainRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listVpcEndpointsForDomainRequest));
            CompletableFuture<ListVpcEndpointsForDomainResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Allows you to purchase Amazon OpenSearch Service Reserved Instances.
     * </p>
     *
     * @param purchaseReservedInstanceOfferingRequest
     *        Container for request parameters to the <code>PurchaseReservedInstanceOffering</code> operation.
     * @return A Java Future containing the result of the PurchaseReservedInstanceOffering operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException An exception for accessing or deleting a resource that does not exist.
     *         Gives http status code of 400.</li>
     *         <li>ResourceAlreadyExistsException An exception for creating a resource that already exists. Gives http
     *         status code of 400.</li>
     *         <li>LimitExceededException An exception for trying to create more than allowed resources or
     *         sub-resources. Gives http status code of 409.</li>
     *         <li>DisabledOperationException An error occured because the client wanted to access a not supported
     *         operation. Gives http status code of 409.</li>
     *         <li>ValidationException An exception for missing / invalid input fields. Gives http status code of 400.</li>
     *         <li>InternalException The request processing has failed because of an unknown error, exception or failure
     *         (the failure is internal to the service) . Gives http status code of 500.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OpenSearchAsyncClient.PurchaseReservedInstanceOffering
     */
    @Override
    public CompletableFuture<PurchaseReservedInstanceOfferingResponse> purchaseReservedInstanceOffering(
            PurchaseReservedInstanceOfferingRequest purchaseReservedInstanceOfferingRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                purchaseReservedInstanceOfferingRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PurchaseReservedInstanceOffering");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PurchaseReservedInstanceOfferingResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, PurchaseReservedInstanceOfferingResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PurchaseReservedInstanceOfferingResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PurchaseReservedInstanceOfferingRequest, PurchaseReservedInstanceOfferingResponse>()
                            .withOperationName("PurchaseReservedInstanceOffering")
                            .withMarshaller(new PurchaseReservedInstanceOfferingRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(purchaseReservedInstanceOfferingRequest));
            CompletableFuture<PurchaseReservedInstanceOfferingResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Allows the remote Amazon OpenSearch Service domain owner to reject an inbound cross-cluster connection request.
     * </p>
     *
     * @param rejectInboundConnectionRequest
     *        Container for the request parameters to the <code>RejectInboundConnection</code> operation.
     * @return A Java Future containing the result of the RejectInboundConnection operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException An exception for accessing or deleting a resource that does not exist.
     *         Gives http status code of 400.</li>
     *         <li>DisabledOperationException An error occured because the client wanted to access a not supported
     *         operation. Gives http status code of 409.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OpenSearchAsyncClient.RejectInboundConnection
     */
    @Override
    public CompletableFuture<RejectInboundConnectionResponse> rejectInboundConnection(
            RejectInboundConnectionRequest rejectInboundConnectionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, rejectInboundConnectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RejectInboundConnection");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RejectInboundConnectionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RejectInboundConnectionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RejectInboundConnectionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RejectInboundConnectionRequest, RejectInboundConnectionResponse>()
                            .withOperationName("RejectInboundConnection")
                            .withMarshaller(new RejectInboundConnectionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(rejectInboundConnectionRequest));
            CompletableFuture<RejectInboundConnectionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes the specified set of tags from an Amazon OpenSearch Service domain. For more information, see <a href=
     * "https://docs.aws.amazon.com/opensearch-service/latest/developerguide/managedomains.html#managedomains-awsresorcetagging"
     * > Tagging Amazon OpenSearch Service domains</a>.
     * </p>
     *
     * @param removeTagsRequest
     *        Container for the request parameters to the <code>RemoveTags</code> operation.
     * @return A Java Future containing the result of the RemoveTags operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BaseException An error occurred while processing the request.</li>
     *         <li>ValidationException An exception for missing / invalid input fields. Gives http status code of 400.</li>
     *         <li>InternalException The request processing has failed because of an unknown error, exception or failure
     *         (the failure is internal to the service) . Gives http status code of 500.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OpenSearchAsyncClient.RemoveTags
     */
    @Override
    public CompletableFuture<RemoveTagsResponse> removeTags(RemoveTagsRequest removeTagsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, removeTagsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RemoveTags");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RemoveTagsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    RemoveTagsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RemoveTagsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RemoveTagsRequest, RemoveTagsResponse>().withOperationName("RemoveTags")
                            .withMarshaller(new RemoveTagsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(removeTagsRequest));
            CompletableFuture<RemoveTagsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Revokes access to an Amazon OpenSearch Service domain that was provided through an interface VPC endpoint.
     * </p>
     *
     * @param revokeVpcEndpointAccessRequest
     * @return A Java Future containing the result of the RevokeVpcEndpointAccess operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException An exception for accessing or deleting a resource that does not exist.
     *         Gives http status code of 400.</li>
     *         <li>ValidationException An exception for missing / invalid input fields. Gives http status code of 400.</li>
     *         <li>DisabledOperationException An error occured because the client wanted to access a not supported
     *         operation. Gives http status code of 409.</li>
     *         <li>InternalException The request processing has failed because of an unknown error, exception or failure
     *         (the failure is internal to the service) . Gives http status code of 500.</li>
     *         <li>BaseException An error occurred while processing the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OpenSearchAsyncClient.RevokeVpcEndpointAccess
     */
    @Override
    public CompletableFuture<RevokeVpcEndpointAccessResponse> revokeVpcEndpointAccess(
            RevokeVpcEndpointAccessRequest revokeVpcEndpointAccessRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, revokeVpcEndpointAccessRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RevokeVpcEndpointAccess");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RevokeVpcEndpointAccessResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RevokeVpcEndpointAccessResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RevokeVpcEndpointAccessResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RevokeVpcEndpointAccessRequest, RevokeVpcEndpointAccessResponse>()
                            .withOperationName("RevokeVpcEndpointAccess")
                            .withMarshaller(new RevokeVpcEndpointAccessRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(revokeVpcEndpointAccessRequest));
            CompletableFuture<RevokeVpcEndpointAccessResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Schedules a service software update for an Amazon OpenSearch Service domain. For more information, see <a
     * href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/service-software.html">Service
     * software updates in Amazon OpenSearch Service</a>.
     * </p>
     *
     * @param startServiceSoftwareUpdateRequest
     *        Container for the request parameters to the <code>StartServiceSoftwareUpdate</code> operation.
     * @return A Java Future containing the result of the StartServiceSoftwareUpdate operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BaseException An error occurred while processing the request.</li>
     *         <li>InternalException The request processing has failed because of an unknown error, exception or failure
     *         (the failure is internal to the service) . Gives http status code of 500.</li>
     *         <li>ResourceNotFoundException An exception for accessing or deleting a resource that does not exist.
     *         Gives http status code of 400.</li>
     *         <li>ValidationException An exception for missing / invalid input fields. Gives http status code of 400.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OpenSearchAsyncClient.StartServiceSoftwareUpdate
     */
    @Override
    public CompletableFuture<StartServiceSoftwareUpdateResponse> startServiceSoftwareUpdate(
            StartServiceSoftwareUpdateRequest startServiceSoftwareUpdateRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startServiceSoftwareUpdateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartServiceSoftwareUpdate");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartServiceSoftwareUpdateResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartServiceSoftwareUpdateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartServiceSoftwareUpdateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartServiceSoftwareUpdateRequest, StartServiceSoftwareUpdateResponse>()
                            .withOperationName("StartServiceSoftwareUpdate")
                            .withMarshaller(new StartServiceSoftwareUpdateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(startServiceSoftwareUpdateRequest));
            CompletableFuture<StartServiceSoftwareUpdateResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Modifies the cluster configuration of the specified Amazon OpenSearch Service domain.
     * </p>
     *
     * @param updateDomainConfigRequest
     *        Container for the request parameters to the <code>UpdateDomain</code> operation.
     * @return A Java Future containing the result of the UpdateDomainConfig operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BaseException An error occurred while processing the request.</li>
     *         <li>InternalException The request processing has failed because of an unknown error, exception or failure
     *         (the failure is internal to the service) . Gives http status code of 500.</li>
     *         <li>InvalidTypeException An exception for trying to create or access sub-resource that is either invalid
     *         or not supported. Gives http status code of 409.</li>
     *         <li>LimitExceededException An exception for trying to create more than allowed resources or
     *         sub-resources. Gives http status code of 409.</li>
     *         <li>ResourceNotFoundException An exception for accessing or deleting a resource that does not exist.
     *         Gives http status code of 400.</li>
     *         <li>ValidationException An exception for missing / invalid input fields. Gives http status code of 400.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OpenSearchAsyncClient.UpdateDomainConfig
     */
    @Override
    public CompletableFuture<UpdateDomainConfigResponse> updateDomainConfig(UpdateDomainConfigRequest updateDomainConfigRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateDomainConfigRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateDomainConfig");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateDomainConfigResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateDomainConfigResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateDomainConfigResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateDomainConfigRequest, UpdateDomainConfigResponse>()
                            .withOperationName("UpdateDomainConfig")
                            .withMarshaller(new UpdateDomainConfigRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateDomainConfigRequest));
            CompletableFuture<UpdateDomainConfigResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a package for use with Amazon OpenSearch Service domains. For more information, see <a
     * href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/custom-packages.html">Custom packages
     * for Amazon OpenSearch Service</a>.
     * </p>
     *
     * @param updatePackageRequest
     *        Container for request parameters to the <code>UpdatePackage</code> operation.
     * @return A Java Future containing the result of the UpdatePackage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BaseException An error occurred while processing the request.</li>
     *         <li>InternalException The request processing has failed because of an unknown error, exception or failure
     *         (the failure is internal to the service) . Gives http status code of 500.</li>
     *         <li>LimitExceededException An exception for trying to create more than allowed resources or
     *         sub-resources. Gives http status code of 409.</li>
     *         <li>ResourceNotFoundException An exception for accessing or deleting a resource that does not exist.
     *         Gives http status code of 400.</li>
     *         <li>AccessDeniedException An error occurred because user does not have permissions to access the
     *         resource. Returns HTTP status code 403.</li>
     *         <li>ValidationException An exception for missing / invalid input fields. Gives http status code of 400.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OpenSearchAsyncClient.UpdatePackage
     */
    @Override
    public CompletableFuture<UpdatePackageResponse> updatePackage(UpdatePackageRequest updatePackageRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updatePackageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdatePackage");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdatePackageResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdatePackageResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdatePackageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdatePackageRequest, UpdatePackageResponse>()
                            .withOperationName("UpdatePackage")
                            .withMarshaller(new UpdatePackageRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updatePackageRequest));
            CompletableFuture<UpdatePackageResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Modifies an Amazon OpenSearch Service-managed interface VPC endpoint.
     * </p>
     *
     * @param updateVpcEndpointRequest
     * @return A Java Future containing the result of the UpdateVpcEndpoint operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException An exception for accessing or deleting a resource that does not exist.
     *         Gives http status code of 400.</li>
     *         <li>DisabledOperationException An error occured because the client wanted to access a not supported
     *         operation. Gives http status code of 409.</li>
     *         <li>InternalException The request processing has failed because of an unknown error, exception or failure
     *         (the failure is internal to the service) . Gives http status code of 500.</li>
     *         <li>ValidationException An exception for missing / invalid input fields. Gives http status code of 400.</li>
     *         <li>ConflictException An error occurred because the client attempts to remove a resource that is
     *         currently in use. Returns HTTP status code 409.</li>
     *         <li>BaseException An error occurred while processing the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OpenSearchAsyncClient.UpdateVpcEndpoint
     */
    @Override
    public CompletableFuture<UpdateVpcEndpointResponse> updateVpcEndpoint(UpdateVpcEndpointRequest updateVpcEndpointRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateVpcEndpointRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateVpcEndpoint");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateVpcEndpointResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateVpcEndpointResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateVpcEndpointResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateVpcEndpointRequest, UpdateVpcEndpointResponse>()
                            .withOperationName("UpdateVpcEndpoint")
                            .withMarshaller(new UpdateVpcEndpointRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateVpcEndpointRequest));
            CompletableFuture<UpdateVpcEndpointResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Allows you to either upgrade your Amazon OpenSearch Service domain or perform an upgrade eligibility check to a
     * compatible version of OpenSearch or Elasticsearch.
     * </p>
     *
     * @param upgradeDomainRequest
     *        Container for the request parameters to the <code>UpgradeDomain</code> operation.
     * @return A Java Future containing the result of the UpgradeDomain operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BaseException An error occurred while processing the request.</li>
     *         <li>ResourceNotFoundException An exception for accessing or deleting a resource that does not exist.
     *         Gives http status code of 400.</li>
     *         <li>ResourceAlreadyExistsException An exception for creating a resource that already exists. Gives http
     *         status code of 400.</li>
     *         <li>DisabledOperationException An error occured because the client wanted to access a not supported
     *         operation. Gives http status code of 409.</li>
     *         <li>ValidationException An exception for missing / invalid input fields. Gives http status code of 400.</li>
     *         <li>InternalException The request processing has failed because of an unknown error, exception or failure
     *         (the failure is internal to the service) . Gives http status code of 500.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OpenSearchAsyncClient.UpgradeDomain
     */
    @Override
    public CompletableFuture<UpgradeDomainResponse> upgradeDomain(UpgradeDomainRequest upgradeDomainRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, upgradeDomainRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpgradeDomain");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpgradeDomainResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpgradeDomainResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpgradeDomainResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpgradeDomainRequest, UpgradeDomainResponse>()
                            .withOperationName("UpgradeDomain")
                            .withMarshaller(new UpgradeDomainRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(upgradeDomainRequest));
            CompletableFuture<UpgradeDomainResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(OpenSearchException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DisabledOperationException")
                                .exceptionBuilderSupplier(DisabledOperationException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidTypeException")
                                .exceptionBuilderSupplier(InvalidTypeException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException")
                                .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalException")
                                .exceptionBuilderSupplier(InternalException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidPaginationTokenException")
                                .exceptionBuilderSupplier(InvalidPaginationTokenException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BaseException").exceptionBuilderSupplier(BaseException::builder)
                                .build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends OpenSearchRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
