/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.osis;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.osis.internal.OsisServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.osis.model.AccessDeniedException;
import software.amazon.awssdk.services.osis.model.ConflictException;
import software.amazon.awssdk.services.osis.model.CreatePipelineRequest;
import software.amazon.awssdk.services.osis.model.CreatePipelineResponse;
import software.amazon.awssdk.services.osis.model.DeletePipelineRequest;
import software.amazon.awssdk.services.osis.model.DeletePipelineResponse;
import software.amazon.awssdk.services.osis.model.GetPipelineBlueprintRequest;
import software.amazon.awssdk.services.osis.model.GetPipelineBlueprintResponse;
import software.amazon.awssdk.services.osis.model.GetPipelineChangeProgressRequest;
import software.amazon.awssdk.services.osis.model.GetPipelineChangeProgressResponse;
import software.amazon.awssdk.services.osis.model.GetPipelineRequest;
import software.amazon.awssdk.services.osis.model.GetPipelineResponse;
import software.amazon.awssdk.services.osis.model.InternalException;
import software.amazon.awssdk.services.osis.model.InvalidPaginationTokenException;
import software.amazon.awssdk.services.osis.model.LimitExceededException;
import software.amazon.awssdk.services.osis.model.ListPipelineBlueprintsRequest;
import software.amazon.awssdk.services.osis.model.ListPipelineBlueprintsResponse;
import software.amazon.awssdk.services.osis.model.ListPipelinesRequest;
import software.amazon.awssdk.services.osis.model.ListPipelinesResponse;
import software.amazon.awssdk.services.osis.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.osis.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.osis.model.OsisException;
import software.amazon.awssdk.services.osis.model.ResourceAlreadyExistsException;
import software.amazon.awssdk.services.osis.model.ResourceNotFoundException;
import software.amazon.awssdk.services.osis.model.StartPipelineRequest;
import software.amazon.awssdk.services.osis.model.StartPipelineResponse;
import software.amazon.awssdk.services.osis.model.StopPipelineRequest;
import software.amazon.awssdk.services.osis.model.StopPipelineResponse;
import software.amazon.awssdk.services.osis.model.TagResourceRequest;
import software.amazon.awssdk.services.osis.model.TagResourceResponse;
import software.amazon.awssdk.services.osis.model.UntagResourceRequest;
import software.amazon.awssdk.services.osis.model.UntagResourceResponse;
import software.amazon.awssdk.services.osis.model.UpdatePipelineRequest;
import software.amazon.awssdk.services.osis.model.UpdatePipelineResponse;
import software.amazon.awssdk.services.osis.model.ValidatePipelineRequest;
import software.amazon.awssdk.services.osis.model.ValidatePipelineResponse;
import software.amazon.awssdk.services.osis.model.ValidationException;
import software.amazon.awssdk.services.osis.transform.CreatePipelineRequestMarshaller;
import software.amazon.awssdk.services.osis.transform.DeletePipelineRequestMarshaller;
import software.amazon.awssdk.services.osis.transform.GetPipelineBlueprintRequestMarshaller;
import software.amazon.awssdk.services.osis.transform.GetPipelineChangeProgressRequestMarshaller;
import software.amazon.awssdk.services.osis.transform.GetPipelineRequestMarshaller;
import software.amazon.awssdk.services.osis.transform.ListPipelineBlueprintsRequestMarshaller;
import software.amazon.awssdk.services.osis.transform.ListPipelinesRequestMarshaller;
import software.amazon.awssdk.services.osis.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.osis.transform.StartPipelineRequestMarshaller;
import software.amazon.awssdk.services.osis.transform.StopPipelineRequestMarshaller;
import software.amazon.awssdk.services.osis.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.osis.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.osis.transform.UpdatePipelineRequestMarshaller;
import software.amazon.awssdk.services.osis.transform.ValidatePipelineRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link OsisClient}.
 *
 * @see OsisClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultOsisClient implements OsisClient {
    private static final Logger log = Logger.loggerFor(DefaultOsisClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultOsisClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates an OpenSearch Ingestion pipeline. For more information, see <a
     * href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/creating-pipeline.html">Creating
     * Amazon OpenSearch Ingestion pipelines</a>.
     * </p>
     *
     * @param createPipelineRequest
     * @return Result of the CreatePipeline operation returned by the service.
     * @throws LimitExceededException
     *         You attempted to create more than the allowed number of tags.
     * @throws ValidationException
     *         An exception for missing or invalid input fields.
     * @throws InternalException
     *         The request failed because of an unknown error, exception, or failure (the failure is internal to the
     *         service).
     * @throws AccessDeniedException
     *         You don't have permissions to access the resource.
     * @throws ResourceAlreadyExistsException
     *         You attempted to create a resource that already exists.
     * @throws ResourceNotFoundException
     *         You attempted to access or delete a resource that does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OsisException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OsisClient.CreatePipeline
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/osis-2022-01-01/CreatePipeline" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreatePipelineResponse createPipeline(CreatePipelineRequest createPipelineRequest) throws LimitExceededException,
            ValidationException, InternalException, AccessDeniedException, ResourceAlreadyExistsException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, OsisException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreatePipelineResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreatePipelineResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createPipelineRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createPipelineRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OSIS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreatePipeline");

            return clientHandler.execute(new ClientExecutionParams<CreatePipelineRequest, CreatePipelineResponse>()
                    .withOperationName("CreatePipeline").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createPipelineRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreatePipelineRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes an OpenSearch Ingestion pipeline. For more information, see <a
     * href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/delete-pipeline.html">Deleting Amazon
     * OpenSearch Ingestion pipelines</a>.
     * </p>
     *
     * @param deletePipelineRequest
     * @return Result of the DeletePipeline operation returned by the service.
     * @throws ValidationException
     *         An exception for missing or invalid input fields.
     * @throws InternalException
     *         The request failed because of an unknown error, exception, or failure (the failure is internal to the
     *         service).
     * @throws AccessDeniedException
     *         You don't have permissions to access the resource.
     * @throws ResourceNotFoundException
     *         You attempted to access or delete a resource that does not exist.
     * @throws ConflictException
     *         The client attempted to remove a resource that is currently in use.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OsisException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OsisClient.DeletePipeline
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/osis-2022-01-01/DeletePipeline" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeletePipelineResponse deletePipeline(DeletePipelineRequest deletePipelineRequest) throws ValidationException,
            InternalException, AccessDeniedException, ResourceNotFoundException, ConflictException, AwsServiceException,
            SdkClientException, OsisException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeletePipelineResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeletePipelineResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deletePipelineRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deletePipelineRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OSIS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeletePipeline");

            return clientHandler.execute(new ClientExecutionParams<DeletePipelineRequest, DeletePipelineResponse>()
                    .withOperationName("DeletePipeline").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deletePipelineRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeletePipelineRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves information about an OpenSearch Ingestion pipeline.
     * </p>
     *
     * @param getPipelineRequest
     * @return Result of the GetPipeline operation returned by the service.
     * @throws ValidationException
     *         An exception for missing or invalid input fields.
     * @throws InternalException
     *         The request failed because of an unknown error, exception, or failure (the failure is internal to the
     *         service).
     * @throws AccessDeniedException
     *         You don't have permissions to access the resource.
     * @throws ResourceNotFoundException
     *         You attempted to access or delete a resource that does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OsisException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OsisClient.GetPipeline
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/osis-2022-01-01/GetPipeline" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetPipelineResponse getPipeline(GetPipelineRequest getPipelineRequest) throws ValidationException, InternalException,
            AccessDeniedException, ResourceNotFoundException, AwsServiceException, SdkClientException, OsisException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetPipelineResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetPipelineResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getPipelineRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPipelineRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OSIS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPipeline");

            return clientHandler.execute(new ClientExecutionParams<GetPipelineRequest, GetPipelineResponse>()
                    .withOperationName("GetPipeline").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getPipelineRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetPipelineRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves information about a specific blueprint for OpenSearch Ingestion. Blueprints are templates for the
     * configuration needed for a <code>CreatePipeline</code> request. For more information, see <a href=
     * "https://docs.aws.amazon.com/opensearch-service/latest/developerguide/creating-pipeline.html#pipeline-blueprint"
     * >Using blueprints to create a pipeline</a>.
     * </p>
     *
     * @param getPipelineBlueprintRequest
     * @return Result of the GetPipelineBlueprint operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have permissions to access the resource.
     * @throws InternalException
     *         The request failed because of an unknown error, exception, or failure (the failure is internal to the
     *         service).
     * @throws ValidationException
     *         An exception for missing or invalid input fields.
     * @throws ResourceNotFoundException
     *         You attempted to access or delete a resource that does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OsisException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OsisClient.GetPipelineBlueprint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/osis-2022-01-01/GetPipelineBlueprint" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetPipelineBlueprintResponse getPipelineBlueprint(GetPipelineBlueprintRequest getPipelineBlueprintRequest)
            throws AccessDeniedException, InternalException, ValidationException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, OsisException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetPipelineBlueprintResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetPipelineBlueprintResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getPipelineBlueprintRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPipelineBlueprintRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OSIS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPipelineBlueprint");

            return clientHandler.execute(new ClientExecutionParams<GetPipelineBlueprintRequest, GetPipelineBlueprintResponse>()
                    .withOperationName("GetPipelineBlueprint").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getPipelineBlueprintRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetPipelineBlueprintRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns progress information for the current change happening on an OpenSearch Ingestion pipeline. Currently,
     * this operation only returns information when a pipeline is being created.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/opensearch-service/latest/developerguide/creating-pipeline.html#get-pipeline-progress"
     * >Tracking the status of pipeline creation</a>.
     * </p>
     *
     * @param getPipelineChangeProgressRequest
     * @return Result of the GetPipelineChangeProgress operation returned by the service.
     * @throws ValidationException
     *         An exception for missing or invalid input fields.
     * @throws InternalException
     *         The request failed because of an unknown error, exception, or failure (the failure is internal to the
     *         service).
     * @throws AccessDeniedException
     *         You don't have permissions to access the resource.
     * @throws ResourceNotFoundException
     *         You attempted to access or delete a resource that does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OsisException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OsisClient.GetPipelineChangeProgress
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/osis-2022-01-01/GetPipelineChangeProgress"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetPipelineChangeProgressResponse getPipelineChangeProgress(
            GetPipelineChangeProgressRequest getPipelineChangeProgressRequest) throws ValidationException, InternalException,
            AccessDeniedException, ResourceNotFoundException, AwsServiceException, SdkClientException, OsisException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetPipelineChangeProgressResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetPipelineChangeProgressResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getPipelineChangeProgressRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPipelineChangeProgressRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OSIS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPipelineChangeProgress");

            return clientHandler
                    .execute(new ClientExecutionParams<GetPipelineChangeProgressRequest, GetPipelineChangeProgressResponse>()
                            .withOperationName("GetPipelineChangeProgress").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(getPipelineChangeProgressRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetPipelineChangeProgressRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves a list of all available blueprints for Data Prepper. For more information, see <a href=
     * "https://docs.aws.amazon.com/opensearch-service/latest/developerguide/creating-pipeline.html#pipeline-blueprint"
     * >Using blueprints to create a pipeline</a>.
     * </p>
     *
     * @param listPipelineBlueprintsRequest
     * @return Result of the ListPipelineBlueprints operation returned by the service.
     * @throws ValidationException
     *         An exception for missing or invalid input fields.
     * @throws InternalException
     *         The request failed because of an unknown error, exception, or failure (the failure is internal to the
     *         service).
     * @throws AccessDeniedException
     *         You don't have permissions to access the resource.
     * @throws InvalidPaginationTokenException
     *         An invalid pagination token provided in the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OsisException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OsisClient.ListPipelineBlueprints
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/osis-2022-01-01/ListPipelineBlueprints" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListPipelineBlueprintsResponse listPipelineBlueprints(ListPipelineBlueprintsRequest listPipelineBlueprintsRequest)
            throws ValidationException, InternalException, AccessDeniedException, InvalidPaginationTokenException,
            AwsServiceException, SdkClientException, OsisException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListPipelineBlueprintsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListPipelineBlueprintsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listPipelineBlueprintsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPipelineBlueprintsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OSIS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPipelineBlueprints");

            return clientHandler
                    .execute(new ClientExecutionParams<ListPipelineBlueprintsRequest, ListPipelineBlueprintsResponse>()
                            .withOperationName("ListPipelineBlueprints").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listPipelineBlueprintsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListPipelineBlueprintsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all OpenSearch Ingestion pipelines in the current Amazon Web Services account and Region. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/list-pipeline.html">Viewing Amazon
     * OpenSearch Ingestion pipelines</a>.
     * </p>
     *
     * @param listPipelinesRequest
     * @return Result of the ListPipelines operation returned by the service.
     * @throws ValidationException
     *         An exception for missing or invalid input fields.
     * @throws InternalException
     *         The request failed because of an unknown error, exception, or failure (the failure is internal to the
     *         service).
     * @throws AccessDeniedException
     *         You don't have permissions to access the resource.
     * @throws InvalidPaginationTokenException
     *         An invalid pagination token provided in the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OsisException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OsisClient.ListPipelines
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/osis-2022-01-01/ListPipelines" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListPipelinesResponse listPipelines(ListPipelinesRequest listPipelinesRequest) throws ValidationException,
            InternalException, AccessDeniedException, InvalidPaginationTokenException, AwsServiceException, SdkClientException,
            OsisException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListPipelinesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListPipelinesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listPipelinesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPipelinesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OSIS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPipelines");

            return clientHandler.execute(new ClientExecutionParams<ListPipelinesRequest, ListPipelinesResponse>()
                    .withOperationName("ListPipelines").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listPipelinesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListPipelinesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all resource tags associated with an OpenSearch Ingestion pipeline. For more information, see <a
     * href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/tag-pipeline.html">Tagging Amazon
     * OpenSearch Ingestion pipelines</a>.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ValidationException
     *         An exception for missing or invalid input fields.
     * @throws ResourceNotFoundException
     *         You attempted to access or delete a resource that does not exist.
     * @throws InternalException
     *         The request failed because of an unknown error, exception, or failure (the failure is internal to the
     *         service).
     * @throws AccessDeniedException
     *         You don't have permissions to access the resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OsisException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OsisClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/osis-2022-01-01/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ValidationException, ResourceNotFoundException, InternalException, AccessDeniedException, AwsServiceException,
            SdkClientException, OsisException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OSIS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Starts an OpenSearch Ingestion pipeline. For more information, see <a href=
     * "https://docs.aws.amazon.com/opensearch-service/latest/developerguide/pipeline--stop-start.html#pipeline--start"
     * >Starting an OpenSearch Ingestion pipeline</a>.
     * </p>
     *
     * @param startPipelineRequest
     * @return Result of the StartPipeline operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have permissions to access the resource.
     * @throws ConflictException
     *         The client attempted to remove a resource that is currently in use.
     * @throws InternalException
     *         The request failed because of an unknown error, exception, or failure (the failure is internal to the
     *         service).
     * @throws ResourceNotFoundException
     *         You attempted to access or delete a resource that does not exist.
     * @throws ValidationException
     *         An exception for missing or invalid input fields.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OsisException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OsisClient.StartPipeline
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/osis-2022-01-01/StartPipeline" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public StartPipelineResponse startPipeline(StartPipelineRequest startPipelineRequest) throws AccessDeniedException,
            ConflictException, InternalException, ResourceNotFoundException, ValidationException, AwsServiceException,
            SdkClientException, OsisException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartPipelineResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StartPipelineResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startPipelineRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startPipelineRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OSIS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartPipeline");

            return clientHandler.execute(new ClientExecutionParams<StartPipelineRequest, StartPipelineResponse>()
                    .withOperationName("StartPipeline").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(startPipelineRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartPipelineRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Stops an OpenSearch Ingestion pipeline. For more information, see <a href=
     * "https://docs.aws.amazon.com/opensearch-service/latest/developerguide/pipeline--stop-start.html#pipeline--stop"
     * >Stopping an OpenSearch Ingestion pipeline</a>.
     * </p>
     *
     * @param stopPipelineRequest
     * @return Result of the StopPipeline operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have permissions to access the resource.
     * @throws ConflictException
     *         The client attempted to remove a resource that is currently in use.
     * @throws InternalException
     *         The request failed because of an unknown error, exception, or failure (the failure is internal to the
     *         service).
     * @throws ResourceNotFoundException
     *         You attempted to access or delete a resource that does not exist.
     * @throws ValidationException
     *         An exception for missing or invalid input fields.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OsisException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OsisClient.StopPipeline
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/osis-2022-01-01/StopPipeline" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public StopPipelineResponse stopPipeline(StopPipelineRequest stopPipelineRequest) throws AccessDeniedException,
            ConflictException, InternalException, ResourceNotFoundException, ValidationException, AwsServiceException,
            SdkClientException, OsisException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StopPipelineResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StopPipelineResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(stopPipelineRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopPipelineRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OSIS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopPipeline");

            return clientHandler.execute(new ClientExecutionParams<StopPipelineRequest, StopPipelineResponse>()
                    .withOperationName("StopPipeline").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(stopPipelineRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StopPipelineRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Tags an OpenSearch Ingestion pipeline. For more information, see <a
     * href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/tag-pipeline.html">Tagging Amazon
     * OpenSearch Ingestion pipelines</a>.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws LimitExceededException
     *         You attempted to create more than the allowed number of tags.
     * @throws ValidationException
     *         An exception for missing or invalid input fields.
     * @throws InternalException
     *         The request failed because of an unknown error, exception, or failure (the failure is internal to the
     *         service).
     * @throws AccessDeniedException
     *         You don't have permissions to access the resource.
     * @throws ResourceNotFoundException
     *         You attempted to access or delete a resource that does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OsisException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OsisClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/osis-2022-01-01/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws LimitExceededException,
            ValidationException, InternalException, AccessDeniedException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, OsisException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OSIS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes one or more tags from an OpenSearch Ingestion pipeline. For more information, see <a
     * href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/tag-pipeline.html">Tagging Amazon
     * OpenSearch Ingestion pipelines</a>.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ValidationException
     *         An exception for missing or invalid input fields.
     * @throws ResourceNotFoundException
     *         You attempted to access or delete a resource that does not exist.
     * @throws InternalException
     *         The request failed because of an unknown error, exception, or failure (the failure is internal to the
     *         service).
     * @throws AccessDeniedException
     *         You don't have permissions to access the resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OsisException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OsisClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/osis-2022-01-01/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ValidationException,
            ResourceNotFoundException, InternalException, AccessDeniedException, AwsServiceException, SdkClientException,
            OsisException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OSIS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates an OpenSearch Ingestion pipeline. For more information, see <a
     * href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/update-pipeline.html">Updating Amazon
     * OpenSearch Ingestion pipelines</a>.
     * </p>
     *
     * @param updatePipelineRequest
     * @return Result of the UpdatePipeline operation returned by the service.
     * @throws ValidationException
     *         An exception for missing or invalid input fields.
     * @throws InternalException
     *         The request failed because of an unknown error, exception, or failure (the failure is internal to the
     *         service).
     * @throws AccessDeniedException
     *         You don't have permissions to access the resource.
     * @throws ResourceNotFoundException
     *         You attempted to access or delete a resource that does not exist.
     * @throws ConflictException
     *         The client attempted to remove a resource that is currently in use.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OsisException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OsisClient.UpdatePipeline
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/osis-2022-01-01/UpdatePipeline" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdatePipelineResponse updatePipeline(UpdatePipelineRequest updatePipelineRequest) throws ValidationException,
            InternalException, AccessDeniedException, ResourceNotFoundException, ConflictException, AwsServiceException,
            SdkClientException, OsisException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdatePipelineResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdatePipelineResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updatePipelineRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updatePipelineRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OSIS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdatePipeline");

            return clientHandler.execute(new ClientExecutionParams<UpdatePipelineRequest, UpdatePipelineResponse>()
                    .withOperationName("UpdatePipeline").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updatePipelineRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdatePipelineRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Checks whether an OpenSearch Ingestion pipeline configuration is valid prior to creation. For more information,
     * see <a
     * href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/creating-pipeline.html">Creating
     * Amazon OpenSearch Ingestion pipelines</a>.
     * </p>
     *
     * @param validatePipelineRequest
     * @return Result of the ValidatePipeline operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have permissions to access the resource.
     * @throws InternalException
     *         The request failed because of an unknown error, exception, or failure (the failure is internal to the
     *         service).
     * @throws ValidationException
     *         An exception for missing or invalid input fields.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OsisException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OsisClient.ValidatePipeline
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/osis-2022-01-01/ValidatePipeline" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ValidatePipelineResponse validatePipeline(ValidatePipelineRequest validatePipelineRequest)
            throws AccessDeniedException, InternalException, ValidationException, AwsServiceException, SdkClientException,
            OsisException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ValidatePipelineResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ValidatePipelineResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(validatePipelineRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, validatePipelineRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OSIS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ValidatePipeline");

            return clientHandler.execute(new ClientExecutionParams<ValidatePipelineRequest, ValidatePipelineResponse>()
                    .withOperationName("ValidatePipeline").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(validatePipelineRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ValidatePipelineRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        OsisServiceClientConfigurationBuilder serviceConfigBuilder = new OsisServiceClientConfigurationBuilder(configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(OsisException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException")
                                .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalException")
                                .exceptionBuilderSupplier(InternalException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidPaginationTokenException")
                                .exceptionBuilderSupplier(InvalidPaginationTokenException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(409).build());
    }

    @Override
    public final OsisServiceClientConfiguration serviceClientConfiguration() {
        return new OsisServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
